package Gramene::CDBI::Genes;

use strict;

use base qw( Gramene::CDBI );

use Carp qw( croak );
use Gramene::Config;
use Gramene::Utils qw( table_name_to_gramene_cdbi_class );
use Readonly;

Readonly my $CONFIG_NAME => 'genes';

my ( $DBH ); # global database handle

# ------------------------------------------------
sub db_Main {

=pod

=head2 db_Main

  my $db = $cdbi->db_Main;

This returns a DBI database handle.  If you choose to override this, you
must include "RootClass => 'DBIx::ContextualFetch'" as an option to
DBI->connect.

=cut

    my $class = shift;

    if ( !defined $DBH ) {
        my $config = config();

        $DBH = DBI->connect(
            $config->{'db_dsn'},
            $config->{'db_user'},
            $config->{'db_pass'},
            { __PACKAGE__->_default_attributes },
        );
    }

    return $DBH;
} 

# ------------------------------------------------
sub config {

=pod

=head2 config

  my $config = $cdbi->config;
  my %config = $cdbi->config;

Returns the module's configuration from "gramene.conf."

=cut

    my $gconfig = Gramene::Config->new or croak 'No config file object';
    my $config  = $gconfig->get( $CONFIG_NAME )
                  or croak "Bad configuration section '$CONFIG_NAME'";

    return wantarray ? %$config : $config;
}

# ------------------------------------------------
sub table_name_to_class {

=pod

=head2 table_name_to_class

  my $class = $cdbi->table_name_to_class('foo');

Returns the something like "Gramene::CDBI::ModuleName::Foo."

=cut

    my ( $self, $table ) = @_;

    return table_name_to_gramene_cdbi_class( $CONFIG_NAME, $table );
}

# ------------------------------------------------
sub represented_tables {
    qw/
        gene_allele
        gene_allele_synonym
        gene_dbxref
        gene_dbxref_to_object
        gene_gene
        gene_gene_031407
        gene_gene_synonym
        gene_gene_to_allele
        gene_gene_type
        gene_geographical_location
        gene_germplasm
        gene_image
        gene_map_position
        gene_mutagenesis
        gene_object_to_germplasm
        gene_object_to_image
        gene_object_to_study
        gene_ontology_association
        gene_ontology_association_031407
        gene_species
        gene_study
        gene_study_type

    /
}

# ------------------------------------------------

package Gramene::CDBI::Genes::GeneAlleleSynonym; {

BEGIN { our @ISA = qw( Gramene::CDBI::Genes ); };
# use base qw( Gramene::CDBI::Genes );

Gramene::CDBI::Genes::GeneAlleleSynonym->table('gene_allele_synonym');


sub allele {
    my $self = shift;
    return $self->allele_id;
}
    
our @extras = ();
Gramene::CDBI::Genes::GeneAlleleSynonym->columns( 
    All => qw/ 
        allele_synonym_id
        allele_id
        synonym_name 
    /, @extras
);

Gramene::CDBI::Genes::GeneAlleleSynonym->columns( 
    Essential => qw/ 
        allele_synonym_id
        allele_id
        synonym_name 
    /
);

Gramene::CDBI::Genes::GeneAlleleSynonym->columns( 
    Ordered => qw/ 
        allele_synonym_id
        allele_id
        synonym_name 
    /
);

sub to_string {
    my $self = shift;
    return $self->synonym_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'allele_synonym_id' => 'number',
        'allele_id' => 'number',
        'synonym_name' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Genes::GeneAlleleSynonym->has_a(
    allele_id => 'Gramene::CDBI::Genes::GeneAllele'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_allele => [ 'allele_id', 'allele_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Genes::GeneGeneToAllele; {

BEGIN { our @ISA = qw( Gramene::CDBI::Genes ); };
# use base qw( Gramene::CDBI::Genes );

Gramene::CDBI::Genes::GeneGeneToAllele->table('gene_gene_to_allele');


sub allele {
    my $self = shift;
    return $self->allele_id;
}
    

sub gene {
    my $self = shift;
    return $self->gene_id;
}
    
our @extras = ();
Gramene::CDBI::Genes::GeneGeneToAllele->columns( 
    All => qw/ 
        gene_to_allele_id
        gene_id
        allele_id 
    /, @extras
);

Gramene::CDBI::Genes::GeneGeneToAllele->columns( 
    Essential => qw/ 
        gene_to_allele_id
        gene_id
        allele_id 
    /
);

Gramene::CDBI::Genes::GeneGeneToAllele->columns( 
    Ordered => qw/ 
        gene_to_allele_id
        gene_id
        allele_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->gene_to_allele_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_to_allele_id' => 'number',
        'gene_id' => 'number',
        'allele_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Genes::GeneGeneToAllele->has_a(
    allele_id => 'Gramene::CDBI::Genes::GeneAllele'
);

Gramene::CDBI::Genes::GeneGeneToAllele->has_a(
    gene_id => 'Gramene::CDBI::Genes::GeneGene'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_allele => [ 'allele_id', 'allele_id' ],
            gene_gene => [ 'gene_id', 'gene_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Genes::GeneAllele; {

BEGIN { our @ISA = qw( Gramene::CDBI::Genes ); };
# use base qw( Gramene::CDBI::Genes );

Gramene::CDBI::Genes::GeneAllele->table('gene_allele');

our @extras = ();
Gramene::CDBI::Genes::GeneAllele->columns( 
    All => qw/ 
        allele_id
        accession
        symbol
        name
        description
        allele_interaction_description
        public_curation_comment
        internal_curation_comment 
    /, @extras
);

Gramene::CDBI::Genes::GeneAllele->columns( 
    Essential => qw/ 
        allele_id
        accession
        symbol
        name
        description
        allele_interaction_description
        public_curation_comment
        internal_curation_comment 
    /
);

Gramene::CDBI::Genes::GeneAllele->columns( 
    Ordered => qw/ 
        allele_id
        accession
        symbol
        name
        description
        allele_interaction_description
        public_curation_comment
        internal_curation_comment 
    /
);

sub to_string {
    my $self = shift;
    return $self->accession || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'allele_id' => 'number',
        'accession' => 'string',
        'symbol' => 'string',
        'name' => 'string',
        'description' => 'string',
        'allele_interaction_description' => 'string',
        'public_curation_comment' => 'string',
        'internal_curation_comment' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Genes::GeneAllele->has_many( 
    'gene_allele_synonyms',
    'Gramene::CDBI::Genes::GeneAlleleSynonym',
    'allele_id' 
);

Gramene::CDBI::Genes::GeneAllele->has_many( 
    'gene_gene_to_alleles',
    'Gramene::CDBI::Genes::GeneGeneToAllele',
    'allele_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_allele_synonym => [ 'allele_id', 'allele_id' ],
            gene_gene_to_allele => [ 'allele_id', 'allele_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Genes::GeneDbxrefToObject; {

BEGIN { our @ISA = qw( Gramene::CDBI::Genes ); };
# use base qw( Gramene::CDBI::Genes );

Gramene::CDBI::Genes::GeneDbxrefToObject->table('gene_dbxref_to_object');


sub dbxref {
    my $self = shift;
    return $self->dbxref_id;
}
    
our @extras = ();
Gramene::CDBI::Genes::GeneDbxrefToObject->columns( 
    All => qw/ 
        dbxref_to_object_id
        table_name
        record_id
        dbxref_id
        dbxref_value 
    /, @extras
);

Gramene::CDBI::Genes::GeneDbxrefToObject->columns( 
    Essential => qw/ 
        dbxref_to_object_id
        table_name
        record_id
        dbxref_id
        dbxref_value 
    /
);

Gramene::CDBI::Genes::GeneDbxrefToObject->columns( 
    Ordered => qw/ 
        dbxref_to_object_id
        table_name
        record_id
        dbxref_id
        dbxref_value 
    /
);

sub to_string {
    my $self = shift;
    return $self->table_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'dbxref_to_object_id' => 'number',
        'table_name' => 'string',
        'record_id' => 'number',
        'dbxref_id' => 'number',
        'dbxref_value' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Genes::GeneDbxrefToObject->has_a(
    dbxref_id => 'Gramene::CDBI::Genes::GeneDbxref'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_dbxref => [ 'dbxref_id', 'dbxref_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Genes::GeneDbxref; {

BEGIN { our @ISA = qw( Gramene::CDBI::Genes ); };
# use base qw( Gramene::CDBI::Genes );

Gramene::CDBI::Genes::GeneDbxref->table('gene_dbxref');

our @extras = ();
Gramene::CDBI::Genes::GeneDbxref->columns( 
    All => qw/ 
        dbxref_id
        dbxref_name
        url 
    /, @extras
);

Gramene::CDBI::Genes::GeneDbxref->columns( 
    Essential => qw/ 
        dbxref_id
        dbxref_name
        url 
    /
);

Gramene::CDBI::Genes::GeneDbxref->columns( 
    Ordered => qw/ 
        dbxref_id
        dbxref_name
        url 
    /
);

sub to_string {
    my $self = shift;
    return $self->dbxref_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'dbxref_id' => 'number',
        'dbxref_name' => 'string',
        'url' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Genes::GeneDbxref->has_many( 
    'gene_dbxref_to_objects',
    'Gramene::CDBI::Genes::GeneDbxrefToObject',
    'dbxref_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_dbxref_to_object => [ 'dbxref_id', 'dbxref_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Genes::GeneGeneSynonym; {

BEGIN { our @ISA = qw( Gramene::CDBI::Genes ); };
# use base qw( Gramene::CDBI::Genes );

Gramene::CDBI::Genes::GeneGeneSynonym->table('gene_gene_synonym');


sub gene {
    my $self = shift;
    return $self->gene_id;
}
    
our @extras = ();
Gramene::CDBI::Genes::GeneGeneSynonym->columns( 
    All => qw/ 
        gene_synonym_id
        gene_id
        synonym_name 
    /, @extras
);

Gramene::CDBI::Genes::GeneGeneSynonym->columns( 
    Essential => qw/ 
        gene_synonym_id
        gene_id
        synonym_name 
    /
);

Gramene::CDBI::Genes::GeneGeneSynonym->columns( 
    Ordered => qw/ 
        gene_synonym_id
        gene_id
        synonym_name 
    /
);

sub to_string {
    my $self = shift;
    return $self->synonym_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_synonym_id' => 'number',
        'gene_id' => 'number',
        'synonym_name' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Genes::GeneGeneSynonym->has_a(
    gene_id => 'Gramene::CDBI::Genes::GeneGene'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_gene => [ 'gene_id', 'gene_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Genes::GeneMapPosition; {

BEGIN { our @ISA = qw( Gramene::CDBI::Genes ); };
# use base qw( Gramene::CDBI::Genes );

Gramene::CDBI::Genes::GeneMapPosition->table('gene_map_position');


sub gene {
    my $self = shift;
    return $self->gene_id;
}
    
our @extras = ();
Gramene::CDBI::Genes::GeneMapPosition->columns( 
    All => qw/ 
        map_position_id
        gene_id
        cmap_feature_accession
        cmap_map_set
        cmap_map_accession
        cmap_map_name
        start_position
        stop_position
        map_units 
    /, @extras
);

Gramene::CDBI::Genes::GeneMapPosition->columns( 
    Essential => qw/ 
        map_position_id
        gene_id
        cmap_feature_accession
        cmap_map_set
        cmap_map_accession
        cmap_map_name
        start_position
        stop_position
        map_units 
    /
);

Gramene::CDBI::Genes::GeneMapPosition->columns( 
    Ordered => qw/ 
        map_position_id
        gene_id
        cmap_feature_accession
        cmap_map_set
        cmap_map_accession
        cmap_map_name
        start_position
        stop_position
        map_units 
    /
);

sub to_string {
    my $self = shift;
    return $self->cmap_feature_accession || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'map_position_id' => 'number',
        'gene_id' => 'number',
        'cmap_feature_accession' => 'string',
        'cmap_map_set' => 'string',
        'cmap_map_accession' => 'string',
        'cmap_map_name' => 'string',
        'start_position' => 'number',
        'stop_position' => 'number',
        'map_units' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Genes::GeneMapPosition->has_a(
    gene_id => 'Gramene::CDBI::Genes::GeneGene'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_gene => [ 'gene_id', 'gene_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Genes::GeneGene; {

BEGIN { our @ISA = qw( Gramene::CDBI::Genes ); };
# use base qw( Gramene::CDBI::Genes );

Gramene::CDBI::Genes::GeneGene->table('gene_gene');


sub gene_type {
    my $self = shift;
    return $self->gene_type_id;
}
    

sub species {
    my $self = shift;
    return $self->species_id;
}
    
our @extras = ();
Gramene::CDBI::Genes::GeneGene->columns( 
    All => qw/ 
        gene_id
        accession
        symbol
        name
        species_id
        gene_type_id
        chromosome
        description
        public_curation_comment
        internal_curation_comment
        has_phenotype
        is_obsolete 
    /, @extras
);

Gramene::CDBI::Genes::GeneGene->columns( 
    Essential => qw/ 
        gene_id
        accession
        symbol
        name
        species_id
        gene_type_id
        chromosome
        description
        public_curation_comment
        internal_curation_comment
        has_phenotype
        is_obsolete 
    /
);

Gramene::CDBI::Genes::GeneGene->columns( 
    Ordered => qw/ 
        gene_id
        accession
        symbol
        name
        species_id
        gene_type_id
        chromosome
        description
        public_curation_comment
        internal_curation_comment
        has_phenotype
        is_obsolete 
    /
);

sub to_string {
    my $self = shift;
    return $self->accession || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_id' => 'number',
        'accession' => 'string',
        'symbol' => 'string',
        'name' => 'string',
        'species_id' => 'number',
        'gene_type_id' => 'number',
        'chromosome' => 'string',
        'description' => 'string',
        'public_curation_comment' => 'string',
        'internal_curation_comment' => 'string',
        'has_phenotype' => 'string',
        'is_obsolete' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Genes::GeneGene->has_a(
    gene_type_id => 'Gramene::CDBI::Genes::GeneGeneType'
);

Gramene::CDBI::Genes::GeneGene->has_a(
    species_id => 'Gramene::CDBI::Genes::GeneSpecies'
);

Gramene::CDBI::Genes::GeneGene->has_many( 
    'gene_gene_synonyms',
    'Gramene::CDBI::Genes::GeneGeneSynonym',
    'gene_id' 
);

Gramene::CDBI::Genes::GeneGene->has_many( 
    'gene_gene_to_alleles',
    'Gramene::CDBI::Genes::GeneGeneToAllele',
    'gene_id' 
);

Gramene::CDBI::Genes::GeneGene->has_many( 
    'gene_map_positions',
    'Gramene::CDBI::Genes::GeneMapPosition',
    'gene_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_gene_synonym => [ 'gene_id', 'gene_id' ],
            gene_gene_to_allele => [ 'gene_id', 'gene_id' ],
            gene_gene_type => [ 'gene_type_id', 'gene_type_id' ],
            gene_map_position => [ 'gene_id', 'gene_id' ],
            gene_species => [ 'species_id', 'species_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Genes::GeneGene031407; {

BEGIN { our @ISA = qw( Gramene::CDBI::Genes ); };
# use base qw( Gramene::CDBI::Genes );

Gramene::CDBI::Genes::GeneGene031407->table('gene_gene_031407');

our @extras = ();
Gramene::CDBI::Genes::GeneGene031407->columns( 
    All => qw/ 
        gene_id
        accession
        symbol
        name
        species_id
        gene_type_id
        chromosome
        description
        public_curation_comment
        internal_curation_comment
        has_phenotype
        is_obsolete 
    /, @extras
);

Gramene::CDBI::Genes::GeneGene031407->columns( 
    Essential => qw/ 
        gene_id
        accession
        symbol
        name
        species_id
        gene_type_id
        chromosome
        description
        public_curation_comment
        internal_curation_comment
        has_phenotype
        is_obsolete 
    /
);

Gramene::CDBI::Genes::GeneGene031407->columns( 
    Ordered => qw/ 
        gene_id
        accession
        symbol
        name
        species_id
        gene_type_id
        chromosome
        description
        public_curation_comment
        internal_curation_comment
        has_phenotype
        is_obsolete 
    /
);

sub to_string {
    my $self = shift;
    return $self->accession || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_id' => 'number',
        'accession' => 'string',
        'symbol' => 'string',
        'name' => 'string',
        'species_id' => 'number',
        'gene_type_id' => 'number',
        'chromosome' => 'string',
        'description' => 'string',
        'public_curation_comment' => 'string',
        'internal_curation_comment' => 'string',
        'has_phenotype' => 'string',
        'is_obsolete' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Genes::GeneGeneType; {

BEGIN { our @ISA = qw( Gramene::CDBI::Genes ); };
# use base qw( Gramene::CDBI::Genes );

Gramene::CDBI::Genes::GeneGeneType->table('gene_gene_type');

our @extras = ();
Gramene::CDBI::Genes::GeneGeneType->columns( 
    All => qw/ 
        gene_type_id
        gene_type 
    /, @extras
);

Gramene::CDBI::Genes::GeneGeneType->columns( 
    Essential => qw/ 
        gene_type_id
        gene_type 
    /
);

Gramene::CDBI::Genes::GeneGeneType->columns( 
    Ordered => qw/ 
        gene_type_id
        gene_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->gene_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_type_id' => 'number',
        'gene_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Genes::GeneGeneType->has_many( 
    'gene_genes',
    'Gramene::CDBI::Genes::GeneGene',
    'gene_type_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_gene => [ 'gene_type_id', 'gene_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Genes::GeneObjectToGermplasm; {

BEGIN { our @ISA = qw( Gramene::CDBI::Genes ); };
# use base qw( Gramene::CDBI::Genes );

Gramene::CDBI::Genes::GeneObjectToGermplasm->table('gene_object_to_germplasm');


sub germplasm {
    my $self = shift;
    return $self->germplasm_id;
}
    
our @extras = ();
Gramene::CDBI::Genes::GeneObjectToGermplasm->columns( 
    All => qw/ 
        object_to_germplasm_id
        object_table
        object_id
        germplasm_id 
    /, @extras
);

Gramene::CDBI::Genes::GeneObjectToGermplasm->columns( 
    Essential => qw/ 
        object_to_germplasm_id
        object_table
        object_id
        germplasm_id 
    /
);

Gramene::CDBI::Genes::GeneObjectToGermplasm->columns( 
    Ordered => qw/ 
        object_to_germplasm_id
        object_table
        object_id
        germplasm_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->object_table || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'object_to_germplasm_id' => 'number',
        'object_table' => 'string',
        'object_id' => 'number',
        'germplasm_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Genes::GeneObjectToGermplasm->has_a(
    germplasm_id => 'Gramene::CDBI::Genes::GeneGermplasm'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_germplasm => [ 'germplasm_id', 'germplasm_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Genes::GeneGermplasm; {

BEGIN { our @ISA = qw( Gramene::CDBI::Genes ); };
# use base qw( Gramene::CDBI::Genes );

Gramene::CDBI::Genes::GeneGermplasm->table('gene_germplasm');


sub geographical_location {
    my $self = shift;
    return $self->geographical_location_id;
}
    

sub mutagenesis {
    my $self = shift;
    return $self->mutagenesis_id;
}
    

sub species {
    my $self = shift;
    return $self->species_id;
}
    
our @extras = ();
Gramene::CDBI::Genes::GeneGermplasm->columns( 
    All => qw/ 
        germplasm_id
        accession
        geographical_location_id
        mutagenesis_id
        wild_type
        species_id 
    /, @extras
);

Gramene::CDBI::Genes::GeneGermplasm->columns( 
    Essential => qw/ 
        germplasm_id
        accession
        geographical_location_id
        mutagenesis_id
        wild_type
        species_id 
    /
);

Gramene::CDBI::Genes::GeneGermplasm->columns( 
    Ordered => qw/ 
        germplasm_id
        accession
        geographical_location_id
        mutagenesis_id
        wild_type
        species_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->accession || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'germplasm_id' => 'number',
        'accession' => 'string',
        'geographical_location_id' => 'number',
        'mutagenesis_id' => 'number',
        'wild_type' => 'string',
        'species_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Genes::GeneGermplasm->has_a(
    geographical_location_id => 'Gramene::CDBI::Genes::GeneGeographicalLocation'
);

Gramene::CDBI::Genes::GeneGermplasm->has_a(
    mutagenesis_id => 'Gramene::CDBI::Genes::GeneMutagenesis'
);

Gramene::CDBI::Genes::GeneGermplasm->has_a(
    species_id => 'Gramene::CDBI::Genes::GeneSpecies'
);

Gramene::CDBI::Genes::GeneGermplasm->has_many( 
    'gene_object_to_germplasms',
    'Gramene::CDBI::Genes::GeneObjectToGermplasm',
    'germplasm_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_geographical_location => [ 'geographical_location_id', 'geographical_location_id' ],
            gene_mutagenesis => [ 'mutagenesis_id', 'mutagenesis_id' ],
            gene_object_to_germplasm => [ 'germplasm_id', 'germplasm_id' ],
            gene_species => [ 'species_id', 'species_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Genes::GeneObjectToStudy; {

BEGIN { our @ISA = qw( Gramene::CDBI::Genes ); };
# use base qw( Gramene::CDBI::Genes );

Gramene::CDBI::Genes::GeneObjectToStudy->table('gene_object_to_study');


sub study {
    my $self = shift;
    return $self->study_id;
}
    
our @extras = ();
Gramene::CDBI::Genes::GeneObjectToStudy->columns( 
    All => qw/ 
        object_to_study_id
        object_table
        object_id
        study_id 
    /, @extras
);

Gramene::CDBI::Genes::GeneObjectToStudy->columns( 
    Essential => qw/ 
        object_to_study_id
        object_table
        object_id
        study_id 
    /
);

Gramene::CDBI::Genes::GeneObjectToStudy->columns( 
    Ordered => qw/ 
        object_to_study_id
        object_table
        object_id
        study_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->object_table || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'object_to_study_id' => 'number',
        'object_table' => 'string',
        'object_id' => 'number',
        'study_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Genes::GeneObjectToStudy->has_a(
    study_id => 'Gramene::CDBI::Genes::GeneStudy'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_study => [ 'study_id', 'study_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Genes::GeneStudy; {

BEGIN { our @ISA = qw( Gramene::CDBI::Genes ); };
# use base qw( Gramene::CDBI::Genes );

Gramene::CDBI::Genes::GeneStudy->table('gene_study');


sub geographical_location {
    my $self = shift;
    return $self->geographical_location_id;
}
    

sub study_type {
    my $self = shift;
    return $self->study_type_id;
}
    
our @extras = ();
Gramene::CDBI::Genes::GeneStudy->columns( 
    All => qw/ 
        study_id
        name
        year
        season
        geographical_location_id
        study_type_id
        environmental_factors 
    /, @extras
);

Gramene::CDBI::Genes::GeneStudy->columns( 
    Essential => qw/ 
        study_id
        name
        year
        season
        geographical_location_id
        study_type_id
        environmental_factors 
    /
);

Gramene::CDBI::Genes::GeneStudy->columns( 
    Ordered => qw/ 
        study_id
        name
        year
        season
        geographical_location_id
        study_type_id
        environmental_factors 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'study_id' => 'number',
        'name' => 'string',
        'year' => 'number',
        'season' => 'string',
        'geographical_location_id' => 'number',
        'study_type_id' => 'number',
        'environmental_factors' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Genes::GeneStudy->has_a(
    geographical_location_id => 'Gramene::CDBI::Genes::GeneGeographicalLocation'
);

Gramene::CDBI::Genes::GeneStudy->has_a(
    study_type_id => 'Gramene::CDBI::Genes::GeneStudyType'
);

Gramene::CDBI::Genes::GeneStudy->has_many( 
    'gene_object_to_studies',
    'Gramene::CDBI::Genes::GeneObjectToStudy',
    'study_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_geographical_location => [ 'geographical_location_id', 'geographical_location_id' ],
            gene_object_to_study => [ 'study_id', 'study_id' ],
            gene_study_type => [ 'study_type_id', 'study_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Genes::GeneGeographicalLocation; {

BEGIN { our @ISA = qw( Gramene::CDBI::Genes ); };
# use base qw( Gramene::CDBI::Genes );

Gramene::CDBI::Genes::GeneGeographicalLocation->table('gene_geographical_location');

our @extras = ();
Gramene::CDBI::Genes::GeneGeographicalLocation->columns( 
    All => qw/ 
        geographical_location_id
        location_name
        latitude_degree
        latitude_minute
        is_north
        longitude_degree
        longitude_minute
        is_west
        aptitude 
    /, @extras
);

Gramene::CDBI::Genes::GeneGeographicalLocation->columns( 
    Essential => qw/ 
        geographical_location_id
        location_name
        latitude_degree
        latitude_minute
        is_north
        longitude_degree
        longitude_minute
        is_west
        aptitude 
    /
);

Gramene::CDBI::Genes::GeneGeographicalLocation->columns( 
    Ordered => qw/ 
        geographical_location_id
        location_name
        latitude_degree
        latitude_minute
        is_north
        longitude_degree
        longitude_minute
        is_west
        aptitude 
    /
);

sub to_string {
    my $self = shift;
    return $self->location_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'geographical_location_id' => 'number',
        'location_name' => 'string',
        'latitude_degree' => 'number',
        'latitude_minute' => 'number',
        'is_north' => 'string',
        'longitude_degree' => 'number',
        'longitude_minute' => 'number',
        'is_west' => 'string',
        'aptitude' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Genes::GeneGeographicalLocation->has_many( 
    'gene_germplasms',
    'Gramene::CDBI::Genes::GeneGermplasm',
    'geographical_location_id' 
);

Gramene::CDBI::Genes::GeneGeographicalLocation->has_many( 
    'gene_studies',
    'Gramene::CDBI::Genes::GeneStudy',
    'geographical_location_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_germplasm => [ 'geographical_location_id', 'geographical_location_id' ],
            gene_study => [ 'geographical_location_id', 'geographical_location_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Genes::GeneObjectToImage; {

BEGIN { our @ISA = qw( Gramene::CDBI::Genes ); };
# use base qw( Gramene::CDBI::Genes );

Gramene::CDBI::Genes::GeneObjectToImage->table('gene_object_to_image');


sub image {
    my $self = shift;
    return $self->image_id;
}
    
our @extras = ();
Gramene::CDBI::Genes::GeneObjectToImage->columns( 
    All => qw/ 
        object_to_image_id
        object_table
        object_id
        image_id 
    /, @extras
);

Gramene::CDBI::Genes::GeneObjectToImage->columns( 
    Essential => qw/ 
        object_to_image_id
        object_table
        object_id
        image_id 
    /
);

Gramene::CDBI::Genes::GeneObjectToImage->columns( 
    Ordered => qw/ 
        object_to_image_id
        object_table
        object_id
        image_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->object_table || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'object_to_image_id' => 'number',
        'object_table' => 'string',
        'object_id' => 'number',
        'image_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Genes::GeneObjectToImage->has_a(
    image_id => 'Gramene::CDBI::Genes::GeneImage'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_image => [ 'image_id', 'image_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Genes::GeneImage; {

BEGIN { our @ISA = qw( Gramene::CDBI::Genes ); };
# use base qw( Gramene::CDBI::Genes );

Gramene::CDBI::Genes::GeneImage->table('gene_image');

our @extras = ();
Gramene::CDBI::Genes::GeneImage->columns( 
    All => qw/ 
        image_id
        file_name
        image_comment 
    /, @extras
);

Gramene::CDBI::Genes::GeneImage->columns( 
    Essential => qw/ 
        image_id
        file_name
        image_comment 
    /
);

Gramene::CDBI::Genes::GeneImage->columns( 
    Ordered => qw/ 
        image_id
        file_name
        image_comment 
    /
);

sub to_string {
    my $self = shift;
    return $self->file_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'image_id' => 'number',
        'file_name' => 'string',
        'image_comment' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Genes::GeneImage->has_many( 
    'gene_object_to_images',
    'Gramene::CDBI::Genes::GeneObjectToImage',
    'image_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_object_to_image => [ 'image_id', 'image_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Genes::GeneMutagenesis; {

BEGIN { our @ISA = qw( Gramene::CDBI::Genes ); };
# use base qw( Gramene::CDBI::Genes );

Gramene::CDBI::Genes::GeneMutagenesis->table('gene_mutagenesis');

our @extras = ();
Gramene::CDBI::Genes::GeneMutagenesis->columns( 
    All => qw/ 
        mutagenesis_id
        mutagen
        mutagenesis_method 
    /, @extras
);

Gramene::CDBI::Genes::GeneMutagenesis->columns( 
    Essential => qw/ 
        mutagenesis_id
        mutagen
        mutagenesis_method 
    /
);

Gramene::CDBI::Genes::GeneMutagenesis->columns( 
    Ordered => qw/ 
        mutagenesis_id
        mutagen
        mutagenesis_method 
    /
);

sub to_string {
    my $self = shift;
    return $self->mutagen || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'mutagenesis_id' => 'number',
        'mutagen' => 'string',
        'mutagenesis_method' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Genes::GeneMutagenesis->has_many( 
    'gene_germplasms',
    'Gramene::CDBI::Genes::GeneGermplasm',
    'mutagenesis_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_germplasm => [ 'mutagenesis_id', 'mutagenesis_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Genes::GeneOntologyAssociation; {

BEGIN { our @ISA = qw( Gramene::CDBI::Genes ); };
# use base qw( Gramene::CDBI::Genes );

Gramene::CDBI::Genes::GeneOntologyAssociation->table('gene_ontology_association');

our @extras = ();
Gramene::CDBI::Genes::GeneOntologyAssociation->columns( 
    All => qw/ 
        ontology_association_id
        object_id
        object_table
        term_accession
        term_type 
    /, @extras
);

Gramene::CDBI::Genes::GeneOntologyAssociation->columns( 
    Essential => qw/ 
        ontology_association_id
        object_id
        object_table
        term_accession
        term_type 
    /
);

Gramene::CDBI::Genes::GeneOntologyAssociation->columns( 
    Ordered => qw/ 
        ontology_association_id
        object_id
        object_table
        term_accession
        term_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->object_table || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'ontology_association_id' => 'number',
        'object_id' => 'number',
        'object_table' => 'string',
        'term_accession' => 'string',
        'term_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Genes::GeneOntologyAssociation031407; {

BEGIN { our @ISA = qw( Gramene::CDBI::Genes ); };
# use base qw( Gramene::CDBI::Genes );

Gramene::CDBI::Genes::GeneOntologyAssociation031407->table('gene_ontology_association_031407');

our @extras = ();
Gramene::CDBI::Genes::GeneOntologyAssociation031407->columns( 
    All => qw/ 
        ontology_association_id
        object_id
        object_table
        term_accession
        term_type 
    /, @extras
);

Gramene::CDBI::Genes::GeneOntologyAssociation031407->columns( 
    Essential => qw/ 
        ontology_association_id
        object_id
        object_table
        term_accession
        term_type 
    /
);

Gramene::CDBI::Genes::GeneOntologyAssociation031407->columns( 
    Ordered => qw/ 
        ontology_association_id
        object_id
        object_table
        term_accession
        term_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->object_table || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'ontology_association_id' => 'number',
        'object_id' => 'number',
        'object_table' => 'string',
        'term_accession' => 'string',
        'term_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Genes::GeneSpecies; {

BEGIN { our @ISA = qw( Gramene::CDBI::Genes ); };
# use base qw( Gramene::CDBI::Genes );

Gramene::CDBI::Genes::GeneSpecies->table('gene_species');

our @extras = ();
Gramene::CDBI::Genes::GeneSpecies->columns( 
    All => qw/ 
        species_id
        ncbi_taxa_id
        gramene_taxa_id
        common_name
        genus
        species 
    /, @extras
);

Gramene::CDBI::Genes::GeneSpecies->columns( 
    Essential => qw/ 
        species_id
        ncbi_taxa_id
        gramene_taxa_id
        common_name
        genus
        species 
    /
);

Gramene::CDBI::Genes::GeneSpecies->columns( 
    Ordered => qw/ 
        species_id
        ncbi_taxa_id
        gramene_taxa_id
        common_name
        genus
        species 
    /
);

sub to_string {
    my $self = shift;
    return $self->common_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'species_id' => 'number',
        'ncbi_taxa_id' => 'number',
        'gramene_taxa_id' => 'string',
        'common_name' => 'string',
        'genus' => 'string',
        'species' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Genes::GeneSpecies->has_many( 
    'gene_genes',
    'Gramene::CDBI::Genes::GeneGene',
    'species_id' 
);

Gramene::CDBI::Genes::GeneSpecies->has_many( 
    'gene_germplasms',
    'Gramene::CDBI::Genes::GeneGermplasm',
    'species_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_gene => [ 'species_id', 'species_id' ],
            gene_germplasm => [ 'species_id', 'species_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Genes::GeneStudyType; {

BEGIN { our @ISA = qw( Gramene::CDBI::Genes ); };
# use base qw( Gramene::CDBI::Genes );

Gramene::CDBI::Genes::GeneStudyType->table('gene_study_type');

our @extras = ();
Gramene::CDBI::Genes::GeneStudyType->columns( 
    All => qw/ 
        study_type_id
        study_type 
    /, @extras
);

Gramene::CDBI::Genes::GeneStudyType->columns( 
    Essential => qw/ 
        study_type_id
        study_type 
    /
);

Gramene::CDBI::Genes::GeneStudyType->columns( 
    Ordered => qw/ 
        study_type_id
        study_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->study_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'study_type_id' => 'number',
        'study_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Genes::GeneStudyType->has_many( 
    'gene_studies',
    'Gramene::CDBI::Genes::GeneStudy',
    'study_type_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_study => [ 'study_type_id', 'study_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope



1;

__END__

# ----------------------------------------------------

=pod

=head1 NAME

Gramene::CDBI::Genes - Gramene::CDBI-base classes for Genes

=head1 VERSION

This documentation refers to version $Revision: 1.4 $.

=head1 SYNOPSIS

  use Gramene::CDBI::Genes;

  my @data   = Gramene::CDBI::Genes::GeneGermplasm->retrieve_all;
  my $object = Gramene::CDBI::Genes::GeneGermplasm->retrieve( 42 );

=head1 DESCRIPTION

This module contains Gramene::CDBI-base classes for the Genes 
schema.

=head1 METHODS 

=head2 config_name

  my $config_name = $self->config_name;

Returns the string 'genes'.

=head2 represented_tables

  my @tables = $cdbi->represented_tables;

Returns the names of the tables in the module.  See the CLASSES below.

=cut

=head1 CLASSES

=head2 Gramene::CDBI::Genes::GeneAllele

Corresponds to the table "gene_allele."  

Stringifies (C<to_name>) to "accession".

Fields:

=over 4

=item * allele_id

=item * accession

=item * symbol

=item * name

=item * description

=item * allele_interaction_description

=item * public_curation_comment

=item * internal_curation_comment

=back



Has Many:

=over 4

=item * gene_allele_synonyms => Gramene::CDBI::Genes::GeneAlleleSynonym

=item * gene_gene_to_alleles => Gramene::CDBI::Genes::GeneGeneToAllele

=back


=head2 Gramene::CDBI::Genes::GeneAlleleSynonym

Corresponds to the table "gene_allele_synonym."  

Stringifies (C<to_name>) to "synonym_name".

Fields:

=over 4

=item * allele_synonym_id

=item * allele_id

=item * synonym_name

=back


Has A:

=over 4

=item * allele => Gramene::CDBI::Genes::GeneAllele

=back



=head2 Gramene::CDBI::Genes::GeneDbxref

Corresponds to the table "gene_dbxref."  

Stringifies (C<to_name>) to "dbxref_name".

Fields:

=over 4

=item * dbxref_id

=item * dbxref_name

=item * url

=back



Has Many:

=over 4

=item * gene_dbxref_to_objects => Gramene::CDBI::Genes::GeneDbxrefToObject

=back


=head2 Gramene::CDBI::Genes::GeneDbxrefToObject

Corresponds to the table "gene_dbxref_to_object."  

Stringifies (C<to_name>) to "table_name".

Fields:

=over 4

=item * dbxref_to_object_id

=item * table_name

=item * record_id

=item * dbxref_id

=item * dbxref_value

=back


Has A:

=over 4

=item * dbxref => Gramene::CDBI::Genes::GeneDbxref

=back



=head2 Gramene::CDBI::Genes::GeneGene

Corresponds to the table "gene_gene."  

Stringifies (C<to_name>) to "accession".

Fields:

=over 4

=item * gene_id

=item * accession

=item * symbol

=item * name

=item * species_id

=item * gene_type_id

=item * chromosome

=item * description

=item * public_curation_comment

=item * internal_curation_comment

=item * has_phenotype

=item * is_obsolete

=back


Has A:

=over 4

=item * gene_type => Gramene::CDBI::Genes::GeneGeneType

=item * species => Gramene::CDBI::Genes::GeneSpecies

=back


Has Many:

=over 4

=item * gene_gene_synonyms => Gramene::CDBI::Genes::GeneGeneSynonym

=item * gene_gene_to_alleles => Gramene::CDBI::Genes::GeneGeneToAllele

=item * gene_map_positions => Gramene::CDBI::Genes::GeneMapPosition

=back


=head2 Gramene::CDBI::Genes::GeneGene031407

Corresponds to the table "gene_gene_031407."  

Stringifies (C<to_name>) to "accession".

Fields:

=over 4

=item * gene_id

=item * accession

=item * symbol

=item * name

=item * species_id

=item * gene_type_id

=item * chromosome

=item * description

=item * public_curation_comment

=item * internal_curation_comment

=item * has_phenotype

=item * is_obsolete

=back




=head2 Gramene::CDBI::Genes::GeneGeneSynonym

Corresponds to the table "gene_gene_synonym."  

Stringifies (C<to_name>) to "synonym_name".

Fields:

=over 4

=item * gene_synonym_id

=item * gene_id

=item * synonym_name

=back


Has A:

=over 4

=item * gene => Gramene::CDBI::Genes::GeneGene

=back



=head2 Gramene::CDBI::Genes::GeneGeneToAllele

Corresponds to the table "gene_gene_to_allele."  

Stringifies (C<to_name>) to "gene_to_allele_id".

Fields:

=over 4

=item * gene_to_allele_id

=item * gene_id

=item * allele_id

=back


Has A:

=over 4

=item * allele => Gramene::CDBI::Genes::GeneAllele

=item * gene => Gramene::CDBI::Genes::GeneGene

=back



=head2 Gramene::CDBI::Genes::GeneGeneType

Corresponds to the table "gene_gene_type."  

Stringifies (C<to_name>) to "gene_type".

Fields:

=over 4

=item * gene_type_id

=item * gene_type

=back



Has Many:

=over 4

=item * gene_genes => Gramene::CDBI::Genes::GeneGene

=back


=head2 Gramene::CDBI::Genes::GeneGeographicalLocation

Corresponds to the table "gene_geographical_location."  

Stringifies (C<to_name>) to "location_name".

Fields:

=over 4

=item * geographical_location_id

=item * location_name

=item * latitude_degree

=item * latitude_minute

=item * is_north

=item * longitude_degree

=item * longitude_minute

=item * is_west

=item * aptitude

=back



Has Many:

=over 4

=item * gene_germplasms => Gramene::CDBI::Genes::GeneGermplasm

=item * gene_studies => Gramene::CDBI::Genes::GeneStudy

=back


=head2 Gramene::CDBI::Genes::GeneGermplasm

Corresponds to the table "gene_germplasm."  

Stringifies (C<to_name>) to "accession".

Fields:

=over 4

=item * germplasm_id

=item * accession

=item * geographical_location_id

=item * mutagenesis_id

=item * wild_type

=item * species_id

=back


Has A:

=over 4

=item * geographical_location => Gramene::CDBI::Genes::GeneGeographicalLocation

=item * mutagenesis => Gramene::CDBI::Genes::GeneMutagenesis

=item * species => Gramene::CDBI::Genes::GeneSpecies

=back


Has Many:

=over 4

=item * gene_object_to_germplasms => Gramene::CDBI::Genes::GeneObjectToGermplasm

=back


=head2 Gramene::CDBI::Genes::GeneImage

Corresponds to the table "gene_image."  

Stringifies (C<to_name>) to "file_name".

Fields:

=over 4

=item * image_id

=item * file_name

=item * image_comment

=back



Has Many:

=over 4

=item * gene_object_to_images => Gramene::CDBI::Genes::GeneObjectToImage

=back


=head2 Gramene::CDBI::Genes::GeneMapPosition

Corresponds to the table "gene_map_position."  

Stringifies (C<to_name>) to "cmap_feature_accession".

Fields:

=over 4

=item * map_position_id

=item * gene_id

=item * cmap_feature_accession

=item * cmap_map_set

=item * cmap_map_accession

=item * cmap_map_name

=item * start_position

=item * stop_position

=item * map_units

=back


Has A:

=over 4

=item * gene => Gramene::CDBI::Genes::GeneGene

=back



=head2 Gramene::CDBI::Genes::GeneMutagenesis

Corresponds to the table "gene_mutagenesis."  

Stringifies (C<to_name>) to "mutagen".

Fields:

=over 4

=item * mutagenesis_id

=item * mutagen

=item * mutagenesis_method

=back



Has Many:

=over 4

=item * gene_germplasms => Gramene::CDBI::Genes::GeneGermplasm

=back


=head2 Gramene::CDBI::Genes::GeneObjectToGermplasm

Corresponds to the table "gene_object_to_germplasm."  

Stringifies (C<to_name>) to "object_table".

Fields:

=over 4

=item * object_to_germplasm_id

=item * object_table

=item * object_id

=item * germplasm_id

=back


Has A:

=over 4

=item * germplasm => Gramene::CDBI::Genes::GeneGermplasm

=back



=head2 Gramene::CDBI::Genes::GeneObjectToImage

Corresponds to the table "gene_object_to_image."  

Stringifies (C<to_name>) to "object_table".

Fields:

=over 4

=item * object_to_image_id

=item * object_table

=item * object_id

=item * image_id

=back


Has A:

=over 4

=item * image => Gramene::CDBI::Genes::GeneImage

=back



=head2 Gramene::CDBI::Genes::GeneObjectToStudy

Corresponds to the table "gene_object_to_study."  

Stringifies (C<to_name>) to "object_table".

Fields:

=over 4

=item * object_to_study_id

=item * object_table

=item * object_id

=item * study_id

=back


Has A:

=over 4

=item * study => Gramene::CDBI::Genes::GeneStudy

=back



=head2 Gramene::CDBI::Genes::GeneOntologyAssociation

Corresponds to the table "gene_ontology_association."  

Stringifies (C<to_name>) to "object_table".

Fields:

=over 4

=item * ontology_association_id

=item * object_id

=item * object_table

=item * term_accession

=item * term_type

=back




=head2 Gramene::CDBI::Genes::GeneOntologyAssociation031407

Corresponds to the table "gene_ontology_association_031407."  

Stringifies (C<to_name>) to "object_table".

Fields:

=over 4

=item * ontology_association_id

=item * object_id

=item * object_table

=item * term_accession

=item * term_type

=back




=head2 Gramene::CDBI::Genes::GeneSpecies

Corresponds to the table "gene_species."  

Stringifies (C<to_name>) to "common_name".

Fields:

=over 4

=item * species_id

=item * ncbi_taxa_id

=item * gramene_taxa_id

=item * common_name

=item * genus

=item * species

=back



Has Many:

=over 4

=item * gene_genes => Gramene::CDBI::Genes::GeneGene

=item * gene_germplasms => Gramene::CDBI::Genes::GeneGermplasm

=back


=head2 Gramene::CDBI::Genes::GeneStudy

Corresponds to the table "gene_study."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * study_id

=item * name

=item * year

=item * season

=item * geographical_location_id

=item * study_type_id

=item * environmental_factors

=back


Has A:

=over 4

=item * geographical_location => Gramene::CDBI::Genes::GeneGeographicalLocation

=item * study_type => Gramene::CDBI::Genes::GeneStudyType

=back


Has Many:

=over 4

=item * gene_object_to_studies => Gramene::CDBI::Genes::GeneObjectToStudy

=back


=head2 Gramene::CDBI::Genes::GeneStudyType

Corresponds to the table "gene_study_type."  

Stringifies (C<to_name>) to "study_type".

Fields:

=over 4

=item * study_type_id

=item * study_type

=back



Has Many:

=over 4

=item * gene_studies => Gramene::CDBI::Genes::GeneStudy

=back


=head1 SEE ALSO

Gramene::CDBI.

=head1 AUTHOR

Generated by Ken Youens-Clark on Mon Mar 26 15:22:09 2007.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
