package Gramene::CDBI::Faq;

use strict;

use base qw( Gramene::CDBI );

use Carp qw( croak );
use Gramene::Config;
use Gramene::Utils qw( table_name_to_gramene_cdbi_class );
use Readonly;

Readonly my $CONFIG_NAME => 'faq';

my ( $DBH ); # global database handle

# ------------------------------------------------
sub db_Main {

=pod

=head2 db_Main

  my $db = $cdbi->db_Main;

This returns a DBI database handle.  If you choose to override this, you
must include "RootClass => 'DBIx::ContextualFetch'" as an option to
DBI->connect.

=cut

    my $class = shift;

    if ( !defined $DBH ) {
        my $config = config();

        $DBH = DBI->connect(
            $config->{'db_dsn'},
            $config->{'db_user'},
            $config->{'db_pass'},
            { __PACKAGE__->_default_attributes },
        );
    }

    return $DBH;
} 

# ------------------------------------------------
sub config {

=pod

=head2 config

  my $config = $cdbi->config;
  my %config = $cdbi->config;

Returns the module's configuration from "gramene.conf."

=cut

    my $gconfig = Gramene::Config->new or croak 'No config file object';
    my $config  = $gconfig->get( $CONFIG_NAME )
                  or croak "Bad configuration section '$CONFIG_NAME'";

    return wantarray ? %$config : $config;
}

# ------------------------------------------------
sub table_name_to_class {

=pod

=head2 table_name_to_class

  my $class = $cdbi->table_name_to_class('foo');

Returns the something like "Gramene::CDBI::ModuleName::Foo."

=cut

    my ( $self, $table ) = @_;

    return table_name_to_gramene_cdbi_class( $CONFIG_NAME, $table );
}

# ------------------------------------------------
sub represented_tables {
    qw/
        faq
        faq_question
        session
        user

    /
}

# ------------------------------------------------

package Gramene::CDBI::Faq::FaqQuestion; {

BEGIN { our @ISA = qw( Gramene::CDBI::Faq ); };
# use base qw( Gramene::CDBI::Faq );

Gramene::CDBI::Faq::FaqQuestion->table('faq_question');


sub faq {
    my $self = shift;
    return $self->faq_id;
}
    
our @extras = ();
Gramene::CDBI::Faq::FaqQuestion->columns( 
    All => qw/ 
        faq_question_id
        faq_id
        number
        answer
        question
        created_date
        last_modified 
    /, @extras
);

Gramene::CDBI::Faq::FaqQuestion->columns( 
    Essential => qw/ 
        faq_question_id
        faq_id
        number
        answer
        question
        created_date
        last_modified 
    /
);

Gramene::CDBI::Faq::FaqQuestion->columns( 
    Ordered => qw/ 
        faq_question_id
        faq_id
        number
        answer
        question
        created_date
        last_modified 
    /
);

sub to_string {
    my $self = shift;
    return $self->number || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'faq_question_id' => 'number',
        'faq_id' => 'number',
        'number' => 'number',
        'answer' => 'string',
        'question' => 'string',
        'created_date' => 'string',
        'last_modified' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Faq::FaqQuestion->has_a(
    faq_id => 'Gramene::CDBI::Faq::Faq'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            faq => [ 'faq_id', 'faq_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Faq::Faq; {

BEGIN { our @ISA = qw( Gramene::CDBI::Faq ); };
# use base qw( Gramene::CDBI::Faq );

Gramene::CDBI::Faq::Faq->table('faq');


sub user {
    my $self = shift;
    return $self->user_id;
}
    
our @extras = ();
Gramene::CDBI::Faq::Faq->columns( 
    All => qw/ 
        faq_id
        user_id
        publishdate
        name
        created_date
        last_modified 
    /, @extras
);

Gramene::CDBI::Faq::Faq->columns( 
    Essential => qw/ 
        faq_id
        user_id
        publishdate
        name
        created_date
        last_modified 
    /
);

Gramene::CDBI::Faq::Faq->columns( 
    Ordered => qw/ 
        faq_id
        user_id
        publishdate
        name
        created_date
        last_modified 
    /
);

sub to_string {
    my $self = shift;
    return $self->publishdate || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'faq_id' => 'number',
        'user_id' => 'number',
        'publishdate' => 'string',
        'name' => 'string',
        'created_date' => 'string',
        'last_modified' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Faq::Faq->has_a(
    user_id => 'Gramene::CDBI::Faq::User'
);

Gramene::CDBI::Faq::Faq->has_many( 
    'faq_questions',
    'Gramene::CDBI::Faq::FaqQuestion',
    'faq_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            faq_question => [ 'faq_id', 'faq_id' ],
            user => [ 'user_id', 'user_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Faq::Session; {

BEGIN { our @ISA = qw( Gramene::CDBI::Faq ); };
# use base qw( Gramene::CDBI::Faq );

Gramene::CDBI::Faq::Session->table('session');


sub user {
    my $self = shift;
    return $self->user_id;
}
    
our @extras = ();
Gramene::CDBI::Faq::Session->columns( 
    All => qw/ 
        sessionkey
        permanent
        user_id
        packedip
        expires
        data
        created_date
        last_modified 
    /, @extras
);

Gramene::CDBI::Faq::Session->columns( 
    Essential => qw/ 
        sessionkey
        permanent
        user_id
        packedip
        expires
        data
        created_date
        last_modified 
    /
);

Gramene::CDBI::Faq::Session->columns( 
    Ordered => qw/ 
        sessionkey
        permanent
        user_id
        packedip
        expires
        data
        created_date
        last_modified 
    /
);

sub to_string {
    my $self = shift;
    return $self->permanent || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'sessionkey' => 'string',
        'permanent' => 'string',
        'user_id' => 'number',
        'packedip' => 'number',
        'expires' => 'string',
        'data' => 'string',
        'created_date' => 'string',
        'last_modified' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Faq::Session->has_a(
    user_id => 'Gramene::CDBI::Faq::User'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            user => [ 'user_id', 'user_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::Faq::User; {

BEGIN { our @ISA = qw( Gramene::CDBI::Faq ); };
# use base qw( Gramene::CDBI::Faq );

Gramene::CDBI::Faq::User->table('user');

our @extras = ();
Gramene::CDBI::Faq::User->columns( 
    All => qw/ 
        user_id
        permalogin
        username
        password
        name
        email
        created_date
        last_modified 
    /, @extras
);

Gramene::CDBI::Faq::User->columns( 
    Essential => qw/ 
        user_id
        permalogin
        username
        password
        name
        email
        created_date
        last_modified 
    /
);

Gramene::CDBI::Faq::User->columns( 
    Ordered => qw/ 
        user_id
        permalogin
        username
        password
        name
        email
        created_date
        last_modified 
    /
);

sub to_string {
    my $self = shift;
    return $self->permalogin || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'user_id' => 'number',
        'permalogin' => 'string',
        'username' => 'string',
        'password' => 'string',
        'name' => 'string',
        'email' => 'string',
        'created_date' => 'string',
        'last_modified' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::Faq::User->has_many( 
    'faqs',
    'Gramene::CDBI::Faq::Faq',
    'user_id' 
);

Gramene::CDBI::Faq::User->has_many( 
    'sessions',
    'Gramene::CDBI::Faq::Session',
    'user_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            faq => [ 'user_id', 'user_id' ],
            session => [ 'user_id', 'user_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope



1;

__END__

# ----------------------------------------------------

=pod

=head1 NAME

Gramene::CDBI::Faq - Gramene::CDBI-base classes for Faq

=head1 VERSION

This documentation refers to version $Revision: 1.5 $.

=head1 SYNOPSIS

  use Gramene::CDBI::Faq;

  my @data   = Gramene::CDBI::Faq::FaqQuestion->retrieve_all;
  my $object = Gramene::CDBI::Faq::FaqQuestion->retrieve( 42 );

=head1 DESCRIPTION

This module contains Gramene::CDBI-base classes for the Faq 
schema.

=head1 METHODS 

=head2 config_name

  my $config_name = $self->config_name;

Returns the string 'faq'.

=head2 represented_tables

  my @tables = $cdbi->represented_tables;

Returns the names of the tables in the module.  See the CLASSES below.

=cut

=head1 CLASSES

=head2 Gramene::CDBI::Faq::Faq

Corresponds to the table "faq."  

Stringifies (C<to_name>) to "publishdate".

Fields:

=over 4

=item * faq_id

=item * user_id

=item * publishdate

=item * name

=item * created_date

=item * last_modified

=back


Has A:

=over 4

=item * user => Gramene::CDBI::Faq::User

=back


Has Many:

=over 4

=item * faq_questions => Gramene::CDBI::Faq::FaqQuestion

=back


=head2 Gramene::CDBI::Faq::FaqQuestion

Corresponds to the table "faq_question."  

Stringifies (C<to_name>) to "number".

Fields:

=over 4

=item * faq_question_id

=item * faq_id

=item * number

=item * answer

=item * question

=item * created_date

=item * last_modified

=back


Has A:

=over 4

=item * faq => Gramene::CDBI::Faq::Faq

=back



=head2 Gramene::CDBI::Faq::Session

Corresponds to the table "session."  

Stringifies (C<to_name>) to "permanent".

Fields:

=over 4

=item * sessionkey

=item * permanent

=item * user_id

=item * packedip

=item * expires

=item * data

=item * created_date

=item * last_modified

=back


Has A:

=over 4

=item * user => Gramene::CDBI::Faq::User

=back



=head2 Gramene::CDBI::Faq::User

Corresponds to the table "user."  

Stringifies (C<to_name>) to "permalogin".

Fields:

=over 4

=item * user_id

=item * permalogin

=item * username

=item * password

=item * name

=item * email

=item * created_date

=item * last_modified

=back



Has Many:

=over 4

=item * faqs => Gramene::CDBI::Faq::Faq

=item * sessions => Gramene::CDBI::Faq::Session

=back


=head1 SEE ALSO

Gramene::CDBI.

=head1 AUTHOR

Generated by Ken Youens-Clark on Fri May 11 13:10:49 2007.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
