package Gramene::CDBI::EnsemblRice;

use strict;

use base qw( Gramene::CDBI );

use Carp qw( croak );
use Gramene::Config;
use Gramene::Utils qw( table_name_to_gramene_cdbi_class );
use Readonly;

Readonly my $CONFIG_NAME => 'ensembl_rice';

my ( $DBH ); # global database handle

# ------------------------------------------------
sub db_Main {

=pod

=head2 db_Main

  my $db = $cdbi->db_Main;

This returns a DBI database handle.  If you choose to override this, you
must include "RootClass => 'DBIx::ContextualFetch'" as an option to
DBI->connect.

=cut

    my $class = shift;

    if ( !defined $DBH ) {
        my $config = config();

        $DBH = DBI->connect(
            $config->{'db_dsn'},
            $config->{'db_user'},
            $config->{'db_pass'},
            { __PACKAGE__->_default_attributes },
        );
    }

    return $DBH;
} 

# ------------------------------------------------
sub config {

=pod

=head2 config

  my $config = $cdbi->config;
  my %config = $cdbi->config;

Returns the module's configuration from "gramene.conf."

=cut

    my $gconfig = Gramene::Config->new or croak 'No config file object';
    my $config  = $gconfig->get( $CONFIG_NAME )
                  or croak "Bad configuration section '$CONFIG_NAME'";

    return wantarray ? %$config : $config;
}

# ------------------------------------------------
sub table_name_to_class {

=pod

=head2 table_name_to_class

  my $class = $cdbi->table_name_to_class('foo');

Returns the something like "Gramene::CDBI::ModuleName::Foo."

=cut

    my ( $self, $table ) = @_;

    return table_name_to_gramene_cdbi_class( $CONFIG_NAME, $table );
}

# ------------------------------------------------
sub represented_tables {
    qw/
        alt_allele
        analysis
        analysis_description
        assembly
        assembly_exception
        attrib_type
        bacpac
        coord_system
        density_feature
        density_type
        ditag
        ditag_feature
        dna
        dna_align_feature
        dnac
        ensembl_daf_chr
        ensembl_daf_group
        ensembl_gene_helper
        exon
        exon_stable_id
        exon_transcript
        external_db
        external_synonym
        gene
        gene_archive
        gene_attrib
        gene_stable_id
        go_xref
        gramene_gene_ext_helper
        identity_xref
        interpro
        karyotype
        map
        mapping_session
        marker
        marker_feature
        marker_map_location
        marker_synonym
        meta
        meta_coord
        misc_attrib
        misc_attrib_helper
        misc_feature
        misc_feature_misc_set
        misc_set
        object_xref
        oligo_array
        oligo_feature
        oligo_probe
        peptide_archive
        prediction_exon
        prediction_transcript
        protein_align_feature
        protein_feature
        qtl
        qtl_feature
        qtl_synonym
        regulatory_factor
        regulatory_factor_coding
        regulatory_feature
        regulatory_feature_object
        regulatory_search_region
        repeat_consensus
        repeat_feature
        seq_region
        seq_region_attrib
        seqtable_daf0
        seqtable_daf1
        seqtable_helper
        seqtable_prehelper
        simple_feature
        stable_id_event
        supporting_feature
        transcript
        transcript_attrib
        transcript_stable_id
        transcript_supporting_feature
        translation
        translation_attrib
        translation_stable_id
        unconventional_transcript_association
        unmapped_object
        unmapped_reason
        xref

    /
}

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::AltAllele; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::AltAllele->table('alt_allele');


sub gene {
    my $self = shift;
    return $self->gene_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::AltAllele->columns( 
    All => qw/ 
        alt_allele_id
        gene_id 
    /, @extras
);

Gramene::CDBI::EnsemblRice::AltAllele->columns( 
    Essential => qw/ 
        alt_allele_id
        gene_id 
    /
);

Gramene::CDBI::EnsemblRice::AltAllele->columns( 
    Ordered => qw/ 
        alt_allele_id
        gene_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->alt_allele_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'alt_allele_id' => 'number',
        'gene_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::AltAllele->has_a(
    gene_id => 'Gramene::CDBI::EnsemblRice::Gene'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene => [ 'gene_id', 'gene_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::AnalysisDescription; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::AnalysisDescription->table('analysis_description');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::AnalysisDescription->columns( 
    All => qw/ 
        analysis_id
        description
        display_label
        displayable
        web_data 
    /, @extras
);

Gramene::CDBI::EnsemblRice::AnalysisDescription->columns( 
    Essential => qw/ 
        analysis_id
        description
        display_label
        displayable
        web_data 
    /
);

Gramene::CDBI::EnsemblRice::AnalysisDescription->columns( 
    Ordered => qw/ 
        analysis_id
        description
        display_label
        displayable
        web_data 
    /
);

sub to_string {
    my $self = shift;
    return $self->description || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'analysis_id' => 'number',
        'description' => 'string',
        'display_label' => 'string',
        'displayable' => 'string',
        'web_data' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::AnalysisDescription->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblRice::Analysis'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::DensityFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::DensityFeature->table('density_feature');


sub density_type {
    my $self = shift;
    return $self->density_type_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::DensityFeature->columns( 
    All => qw/ 
        density_feature_id
        density_type_id
        seq_region_id
        seq_region_start
        seq_region_end
        density_value 
    /, @extras
);

Gramene::CDBI::EnsemblRice::DensityFeature->columns( 
    Essential => qw/ 
        density_feature_id
        density_type_id
        seq_region_id
        seq_region_start
        seq_region_end
        density_value 
    /
);

Gramene::CDBI::EnsemblRice::DensityFeature->columns( 
    Ordered => qw/ 
        density_feature_id
        density_type_id
        seq_region_id
        seq_region_start
        seq_region_end
        density_value 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'density_feature_id' => 'number',
        'density_type_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'density_value' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::DensityFeature->has_a(
    density_type_id => 'Gramene::CDBI::EnsemblRice::DensityType'
);

Gramene::CDBI::EnsemblRice::DensityFeature->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            density_type => [ 'density_type_id', 'density_type_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::DensityType; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::DensityType->table('density_type');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::DensityType->columns( 
    All => qw/ 
        density_type_id
        analysis_id
        block_size
        region_features
        value_type 
    /, @extras
);

Gramene::CDBI::EnsemblRice::DensityType->columns( 
    Essential => qw/ 
        density_type_id
        analysis_id
        block_size
        region_features
        value_type 
    /
);

Gramene::CDBI::EnsemblRice::DensityType->columns( 
    Ordered => qw/ 
        density_type_id
        analysis_id
        block_size
        region_features
        value_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->block_size || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'density_type_id' => 'number',
        'analysis_id' => 'number',
        'block_size' => 'number',
        'region_features' => 'number',
        'value_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::DensityType->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblRice::Analysis'
);

Gramene::CDBI::EnsemblRice::DensityType->has_many( 
    'density_features',
    'Gramene::CDBI::EnsemblRice::DensityFeature',
    'density_type_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            density_feature => [ 'density_type_id', 'density_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::DitagFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::DitagFeature->table('ditag_feature');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub ditag {
    my $self = shift;
    return $self->ditag_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::DitagFeature->columns( 
    All => qw/ 
        ditag_feature_id
        ditag_id
        ditag_pair_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        analysis_id
        hit_start
        hit_end
        hit_strand
        cigar_line
        ditag_side 
    /, @extras
);

Gramene::CDBI::EnsemblRice::DitagFeature->columns( 
    Essential => qw/ 
        ditag_feature_id
        ditag_id
        ditag_pair_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        analysis_id
        hit_start
        hit_end
        hit_strand
        cigar_line
        ditag_side 
    /
);

Gramene::CDBI::EnsemblRice::DitagFeature->columns( 
    Ordered => qw/ 
        ditag_feature_id
        ditag_id
        ditag_pair_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        analysis_id
        hit_start
        hit_end
        hit_strand
        cigar_line
        ditag_side 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'ditag_feature_id' => 'number',
        'ditag_id' => 'number',
        'ditag_pair_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'analysis_id' => 'number',
        'hit_start' => 'number',
        'hit_end' => 'number',
        'hit_strand' => 'string',
        'cigar_line' => 'string',
        'ditag_side' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::DitagFeature->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblRice::Analysis'
);

Gramene::CDBI::EnsemblRice::DitagFeature->has_a(
    ditag_id => 'Gramene::CDBI::EnsemblRice::Ditag'
);

Gramene::CDBI::EnsemblRice::DitagFeature->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            ditag => [ 'ditag_id', 'ditag_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::DnaAlignFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::DnaAlignFeature->table('dna_align_feature');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::DnaAlignFeature->columns( 
    All => qw/ 
        dna_align_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        hit_start
        hit_end
        hit_strand
        hit_name
        analysis_id
        score
        evalue
        perc_ident
        cigar_line 
    /, @extras
);

Gramene::CDBI::EnsemblRice::DnaAlignFeature->columns( 
    Essential => qw/ 
        dna_align_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        hit_start
        hit_end
        hit_strand
        hit_name
        analysis_id
        score
        evalue
        perc_ident
        cigar_line 
    /
);

Gramene::CDBI::EnsemblRice::DnaAlignFeature->columns( 
    Ordered => qw/ 
        dna_align_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        hit_start
        hit_end
        hit_strand
        hit_name
        analysis_id
        score
        evalue
        perc_ident
        cigar_line 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'dna_align_feature_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'hit_start' => 'number',
        'hit_end' => 'number',
        'hit_strand' => 'string',
        'hit_name' => 'string',
        'analysis_id' => 'number',
        'score' => 'number',
        'evalue' => 'number',
        'perc_ident' => 'number',
        'cigar_line' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::DnaAlignFeature->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblRice::Analysis'
);

Gramene::CDBI::EnsemblRice::DnaAlignFeature->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::GeneAttrib; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::GeneAttrib->table('gene_attrib');


sub attrib_type {
    my $self = shift;
    return $self->attrib_type_id;
}
    

sub gene {
    my $self = shift;
    return $self->gene_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::GeneAttrib->columns( 
    All => qw/ 
        gene_id
        attrib_type_id
        value 
    /, @extras
);

Gramene::CDBI::EnsemblRice::GeneAttrib->columns( 
    Essential => qw/ 
        gene_id
        attrib_type_id
        value 
    /
);

Gramene::CDBI::EnsemblRice::GeneAttrib->columns( 
    Ordered => qw/ 
        gene_id
        attrib_type_id
        value 
    /
);

sub to_string {
    my $self = shift;
    return $self->value || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_id' => 'number',
        'attrib_type_id' => 'string',
        'value' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::GeneAttrib->has_a(
    attrib_type_id => 'Gramene::CDBI::EnsemblRice::AttribType'
);

Gramene::CDBI::EnsemblRice::GeneAttrib->has_a(
    gene_id => 'Gramene::CDBI::EnsemblRice::Gene'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            attrib_type => [ 'attrib_type_id', 'attrib_type_id' ],
            gene => [ 'gene_id', 'gene_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::GeneStableId; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::GeneStableId->table('gene_stable_id');


sub gene {
    my $self = shift;
    return $self->gene_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::GeneStableId->columns( 
    All => qw/ 
        gene_id
        stable_id
        version
        created_date
        modified_date 
    /, @extras
);

Gramene::CDBI::EnsemblRice::GeneStableId->columns( 
    Essential => qw/ 
        gene_id
        stable_id
        version
        created_date
        modified_date 
    /
);

Gramene::CDBI::EnsemblRice::GeneStableId->columns( 
    Ordered => qw/ 
        gene_id
        stable_id
        version
        created_date
        modified_date 
    /
);

sub to_string {
    my $self = shift;
    return $self->version || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_id' => 'number',
        'stable_id' => 'string',
        'version' => 'number',
        'created_date' => 'string',
        'modified_date' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::GeneStableId->has_a(
    gene_id => 'Gramene::CDBI::EnsemblRice::Gene'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene => [ 'gene_id', 'gene_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::RegulatoryFactorCoding; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::RegulatoryFactorCoding->table('regulatory_factor_coding');


sub gene {
    my $self = shift;
    return $self->gene_id;
}
    

sub regulatory_factor {
    my $self = shift;
    return $self->regulatory_factor_id;
}
    

sub transcript {
    my $self = shift;
    return $self->transcript_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::RegulatoryFactorCoding->columns( 
    All => qw/ 
        regulatory_factor_id
        transcript_id
        gene_id 
    /, @extras
);

Gramene::CDBI::EnsemblRice::RegulatoryFactorCoding->columns( 
    Essential => qw/ 
        regulatory_factor_id
        transcript_id
        gene_id 
    /
);

Gramene::CDBI::EnsemblRice::RegulatoryFactorCoding->columns( 
    Ordered => qw/ 
        regulatory_factor_id
        transcript_id
        gene_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->regulatory_factor_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'regulatory_factor_id' => 'number',
        'transcript_id' => 'number',
        'gene_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::RegulatoryFactorCoding->has_a(
    gene_id => 'Gramene::CDBI::EnsemblRice::Gene'
);

Gramene::CDBI::EnsemblRice::RegulatoryFactorCoding->has_a(
    regulatory_factor_id => 'Gramene::CDBI::EnsemblRice::RegulatoryFactor'
);

Gramene::CDBI::EnsemblRice::RegulatoryFactorCoding->has_a(
    transcript_id => 'Gramene::CDBI::EnsemblRice::Transcript'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene => [ 'gene_id', 'gene_id' ],
            regulatory_factor => [ 'regulatory_factor_id', 'regulatory_factor_id' ],
            transcript => [ 'transcript_id', 'transcript_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::ExonTranscript; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::ExonTranscript->table('exon_transcript');


sub exon {
    my $self = shift;
    return $self->exon_id;
}
    

sub transcript {
    my $self = shift;
    return $self->transcript_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::ExonTranscript->columns( 
    All => qw/ 
        exon_id
        transcript_id
        rank 
    /, @extras
);

Gramene::CDBI::EnsemblRice::ExonTranscript->columns( 
    Essential => qw/ 
        exon_id
        transcript_id
        rank 
    /
);

Gramene::CDBI::EnsemblRice::ExonTranscript->columns( 
    Ordered => qw/ 
        exon_id
        transcript_id
        rank 
    /
);


Gramene::CDBI::EnsemblRice::ExonTranscript->columns( 
    Primary => qw/ 
        exon_id
        transcript_id
        rank 
    / 
);


sub to_string {
    my $self = shift;
    return $self->exon_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'exon_id' => 'number',
        'transcript_id' => 'number',
        'rank' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::ExonTranscript->has_a(
    exon_id => 'Gramene::CDBI::EnsemblRice::Exon'
);

Gramene::CDBI::EnsemblRice::ExonTranscript->has_a(
    transcript_id => 'Gramene::CDBI::EnsemblRice::Transcript'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            exon => [ 'exon_id', 'exon_id' ],
            transcript => [ 'transcript_id', 'transcript_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::TranscriptAttrib; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::TranscriptAttrib->table('transcript_attrib');


sub attrib_type {
    my $self = shift;
    return $self->attrib_type_id;
}
    

sub transcript {
    my $self = shift;
    return $self->transcript_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::TranscriptAttrib->columns( 
    All => qw/ 
        transcript_id
        attrib_type_id
        value 
    /, @extras
);

Gramene::CDBI::EnsemblRice::TranscriptAttrib->columns( 
    Essential => qw/ 
        transcript_id
        attrib_type_id
        value 
    /
);

Gramene::CDBI::EnsemblRice::TranscriptAttrib->columns( 
    Ordered => qw/ 
        transcript_id
        attrib_type_id
        value 
    /
);

sub to_string {
    my $self = shift;
    return $self->value || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'transcript_id' => 'number',
        'attrib_type_id' => 'string',
        'value' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::TranscriptAttrib->has_a(
    attrib_type_id => 'Gramene::CDBI::EnsemblRice::AttribType'
);

Gramene::CDBI::EnsemblRice::TranscriptAttrib->has_a(
    transcript_id => 'Gramene::CDBI::EnsemblRice::Transcript'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            attrib_type => [ 'attrib_type_id', 'attrib_type_id' ],
            transcript => [ 'transcript_id', 'transcript_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::TranscriptStableId; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::TranscriptStableId->table('transcript_stable_id');


sub transcript {
    my $self = shift;
    return $self->transcript_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::TranscriptStableId->columns( 
    All => qw/ 
        transcript_id
        stable_id
        version
        created_date
        modified_date 
    /, @extras
);

Gramene::CDBI::EnsemblRice::TranscriptStableId->columns( 
    Essential => qw/ 
        transcript_id
        stable_id
        version
        created_date
        modified_date 
    /
);

Gramene::CDBI::EnsemblRice::TranscriptStableId->columns( 
    Ordered => qw/ 
        transcript_id
        stable_id
        version
        created_date
        modified_date 
    /
);

sub to_string {
    my $self = shift;
    return $self->version || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'transcript_id' => 'number',
        'stable_id' => 'string',
        'version' => 'number',
        'created_date' => 'string',
        'modified_date' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::TranscriptStableId->has_a(
    transcript_id => 'Gramene::CDBI::EnsemblRice::Transcript'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            transcript => [ 'transcript_id', 'transcript_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::TranscriptSupportingFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::TranscriptSupportingFeature->table('transcript_supporting_feature');


sub transcript {
    my $self = shift;
    return $self->transcript_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::TranscriptSupportingFeature->columns( 
    All => qw/ 
        transcript_id
        feature_type
        feature_id 
    /, @extras
);

Gramene::CDBI::EnsemblRice::TranscriptSupportingFeature->columns( 
    Essential => qw/ 
        transcript_id
        feature_type
        feature_id 
    /
);

Gramene::CDBI::EnsemblRice::TranscriptSupportingFeature->columns( 
    Ordered => qw/ 
        transcript_id
        feature_type
        feature_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->feature_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'transcript_id' => 'number',
        'feature_type' => 'string',
        'feature_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::TranscriptSupportingFeature->has_a(
    transcript_id => 'Gramene::CDBI::EnsemblRice::Transcript'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            transcript => [ 'transcript_id', 'transcript_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::ProteinFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::ProteinFeature->table('protein_feature');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub translation {
    my $self = shift;
    return $self->translation_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::ProteinFeature->columns( 
    All => qw/ 
        protein_feature_id
        translation_id
        seq_start
        seq_end
        hit_start
        hit_end
        hit_id
        analysis_id
        score
        evalue
        perc_ident 
    /, @extras
);

Gramene::CDBI::EnsemblRice::ProteinFeature->columns( 
    Essential => qw/ 
        protein_feature_id
        translation_id
        seq_start
        seq_end
        hit_start
        hit_end
        hit_id
        analysis_id
        score
        evalue
        perc_ident 
    /
);

Gramene::CDBI::EnsemblRice::ProteinFeature->columns( 
    Ordered => qw/ 
        protein_feature_id
        translation_id
        seq_start
        seq_end
        hit_start
        hit_end
        hit_id
        analysis_id
        score
        evalue
        perc_ident 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'protein_feature_id' => 'number',
        'translation_id' => 'number',
        'seq_start' => 'number',
        'seq_end' => 'number',
        'hit_start' => 'number',
        'hit_end' => 'number',
        'hit_id' => 'string',
        'analysis_id' => 'number',
        'score' => 'number',
        'evalue' => 'number',
        'perc_ident' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::ProteinFeature->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblRice::Analysis'
);

Gramene::CDBI::EnsemblRice::ProteinFeature->has_a(
    translation_id => 'Gramene::CDBI::EnsemblRice::Translation'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            translation => [ 'translation_id', 'translation_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::TranslationAttrib; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::TranslationAttrib->table('translation_attrib');


sub attrib_type {
    my $self = shift;
    return $self->attrib_type_id;
}
    

sub translation {
    my $self = shift;
    return $self->translation_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::TranslationAttrib->columns( 
    All => qw/ 
        translation_id
        attrib_type_id
        value 
    /, @extras
);

Gramene::CDBI::EnsemblRice::TranslationAttrib->columns( 
    Essential => qw/ 
        translation_id
        attrib_type_id
        value 
    /
);

Gramene::CDBI::EnsemblRice::TranslationAttrib->columns( 
    Ordered => qw/ 
        translation_id
        attrib_type_id
        value 
    /
);

sub to_string {
    my $self = shift;
    return $self->value || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'translation_id' => 'number',
        'attrib_type_id' => 'string',
        'value' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::TranslationAttrib->has_a(
    attrib_type_id => 'Gramene::CDBI::EnsemblRice::AttribType'
);

Gramene::CDBI::EnsemblRice::TranslationAttrib->has_a(
    translation_id => 'Gramene::CDBI::EnsemblRice::Translation'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            attrib_type => [ 'attrib_type_id', 'attrib_type_id' ],
            translation => [ 'translation_id', 'translation_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::TranslationStableId; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::TranslationStableId->table('translation_stable_id');


sub translation {
    my $self = shift;
    return $self->translation_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::TranslationStableId->columns( 
    All => qw/ 
        translation_id
        stable_id
        version
        created_date
        modified_date 
    /, @extras
);

Gramene::CDBI::EnsemblRice::TranslationStableId->columns( 
    Essential => qw/ 
        translation_id
        stable_id
        version
        created_date
        modified_date 
    /
);

Gramene::CDBI::EnsemblRice::TranslationStableId->columns( 
    Ordered => qw/ 
        translation_id
        stable_id
        version
        created_date
        modified_date 
    /
);

sub to_string {
    my $self = shift;
    return $self->version || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'translation_id' => 'number',
        'stable_id' => 'string',
        'version' => 'number',
        'created_date' => 'string',
        'modified_date' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::TranslationStableId->has_a(
    translation_id => 'Gramene::CDBI::EnsemblRice::Translation'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            translation => [ 'translation_id', 'translation_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::Translation; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::Translation->table('translation');


sub start_exon {
    my $self = shift;
    return $self->start_exon_id;
}
    
sub end_exon {
    my $self = shift;
    return $self->end_exon_id;
}
    

sub transcript {
    my $self = shift;
    return $self->transcript_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::Translation->columns( 
    All => qw/ 
        translation_id
        transcript_id
        seq_start
        start_exon_id
        seq_end
        end_exon_id 
    /, @extras
);

Gramene::CDBI::EnsemblRice::Translation->columns( 
    Essential => qw/ 
        translation_id
        transcript_id
        seq_start
        start_exon_id
        seq_end
        end_exon_id 
    /
);

Gramene::CDBI::EnsemblRice::Translation->columns( 
    Ordered => qw/ 
        translation_id
        transcript_id
        seq_start
        start_exon_id
        seq_end
        end_exon_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'translation_id' => 'number',
        'transcript_id' => 'number',
        'seq_start' => 'number',
        'start_exon_id' => 'number',
        'seq_end' => 'number',
        'end_exon_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::Translation->has_a(
    start_exon_id => 'Gramene::CDBI::EnsemblRice::Exon'
);

Gramene::CDBI::EnsemblRice::Translation->has_a(
    end_exon_id => 'Gramene::CDBI::EnsemblRice::Exon'
);

Gramene::CDBI::EnsemblRice::Translation->has_a(
    transcript_id => 'Gramene::CDBI::EnsemblRice::Transcript'
);

Gramene::CDBI::EnsemblRice::Translation->has_many( 
    'protein_features',
    'Gramene::CDBI::EnsemblRice::ProteinFeature',
    'translation_id' 
);

Gramene::CDBI::EnsemblRice::Translation->has_many( 
    'translation_attribs',
    'Gramene::CDBI::EnsemblRice::TranslationAttrib',
    'translation_id' 
);

Gramene::CDBI::EnsemblRice::Translation->has_many( 
    'translation_stable_ids',
    'Gramene::CDBI::EnsemblRice::TranslationStableId',
    'translation_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            exon => [ 'end_exon_id', 'exon_id' ],
            protein_feature => [ 'translation_id', 'translation_id' ],
            transcript => [ 'transcript_id', 'transcript_id' ],
            translation_attrib => [ 'translation_id', 'translation_id' ],
            translation_stable_id => [ 'translation_id', 'translation_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::Transcript; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::Transcript->table('transcript');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub gene {
    my $self = shift;
    return $self->gene_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    

sub display_xref {
    my $self = shift;
    return $self->display_xref_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::Transcript->columns( 
    All => qw/ 
        transcript_id
        gene_id
        analysis_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        display_xref_id
        biotype
        status
        description
        is_current 
    /, @extras
);

Gramene::CDBI::EnsemblRice::Transcript->columns( 
    Essential => qw/ 
        transcript_id
        gene_id
        analysis_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        display_xref_id
        biotype
        status
        description
        is_current 
    /
);

Gramene::CDBI::EnsemblRice::Transcript->columns( 
    Ordered => qw/ 
        transcript_id
        gene_id
        analysis_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        display_xref_id
        biotype
        status
        description
        is_current 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'transcript_id' => 'number',
        'gene_id' => 'number',
        'analysis_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'display_xref_id' => 'number',
        'biotype' => 'string',
        'status' => 'string',
        'description' => 'string',
        'is_current' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::Transcript->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblRice::Analysis'
);

Gramene::CDBI::EnsemblRice::Transcript->has_a(
    gene_id => 'Gramene::CDBI::EnsemblRice::Gene'
);

Gramene::CDBI::EnsemblRice::Transcript->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);

Gramene::CDBI::EnsemblRice::Transcript->has_a(
    display_xref_id => 'Gramene::CDBI::EnsemblRice::Xref'
);

Gramene::CDBI::EnsemblRice::Transcript->has_many( 
    'exon_transcripts',
    'Gramene::CDBI::EnsemblRice::ExonTranscript',
    'transcript_id' 
);

Gramene::CDBI::EnsemblRice::Transcript->has_many( 
    'regulatory_factor_codings',
    'Gramene::CDBI::EnsemblRice::RegulatoryFactorCoding',
    'transcript_id' 
);

Gramene::CDBI::EnsemblRice::Transcript->has_many( 
    'transcript_attribs',
    'Gramene::CDBI::EnsemblRice::TranscriptAttrib',
    'transcript_id' 
);

Gramene::CDBI::EnsemblRice::Transcript->has_many( 
    'transcript_stable_ids',
    'Gramene::CDBI::EnsemblRice::TranscriptStableId',
    'transcript_id' 
);

Gramene::CDBI::EnsemblRice::Transcript->has_many( 
    'transcript_supporting_features',
    'Gramene::CDBI::EnsemblRice::TranscriptSupportingFeature',
    'transcript_id' 
);

Gramene::CDBI::EnsemblRice::Transcript->has_many( 
    'translations',
    'Gramene::CDBI::EnsemblRice::Translation',
    'transcript_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            exon_transcript => [ 'transcript_id', 'transcript_id' ],
            gene => [ 'gene_id', 'gene_id' ],
            regulatory_factor_coding => [ 'transcript_id', 'transcript_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
            transcript_attrib => [ 'transcript_id', 'transcript_id' ],
            transcript_stable_id => [ 'transcript_id', 'transcript_id' ],
            transcript_supporting_feature => [ 'transcript_id', 'transcript_id' ],
            translation => [ 'transcript_id', 'transcript_id' ],
            xref => [ 'display_xref_id', 'xref_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::Gene; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::Gene->table('gene');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    

sub display_xref {
    my $self = shift;
    return $self->display_xref_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::Gene->columns( 
    All => qw/ 
        gene_id
        biotype
        analysis_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        display_xref_id
        source
        status
        description
        is_current 
    /, @extras
);

Gramene::CDBI::EnsemblRice::Gene->columns( 
    Essential => qw/ 
        gene_id
        biotype
        analysis_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        display_xref_id
        source
        status
        description
        is_current 
    /
);

Gramene::CDBI::EnsemblRice::Gene->columns( 
    Ordered => qw/ 
        gene_id
        biotype
        analysis_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        display_xref_id
        source
        status
        description
        is_current 
    /
);

sub to_string {
    my $self = shift;
    return $self->biotype || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_id' => 'number',
        'biotype' => 'string',
        'analysis_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'display_xref_id' => 'number',
        'source' => 'string',
        'status' => 'string',
        'description' => 'string',
        'is_current' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::Gene->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblRice::Analysis'
);

Gramene::CDBI::EnsemblRice::Gene->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);

Gramene::CDBI::EnsemblRice::Gene->has_a(
    display_xref_id => 'Gramene::CDBI::EnsemblRice::Xref'
);

Gramene::CDBI::EnsemblRice::Gene->has_many( 
    'alt_alleles',
    'Gramene::CDBI::EnsemblRice::AltAllele',
    'gene_id' 
);

Gramene::CDBI::EnsemblRice::Gene->has_many( 
    'gene_attribs',
    'Gramene::CDBI::EnsemblRice::GeneAttrib',
    'gene_id' 
);

Gramene::CDBI::EnsemblRice::Gene->has_many( 
    'gene_stable_ids',
    'Gramene::CDBI::EnsemblRice::GeneStableId',
    'gene_id' 
);

Gramene::CDBI::EnsemblRice::Gene->has_many( 
    'regulatory_factor_codings',
    'Gramene::CDBI::EnsemblRice::RegulatoryFactorCoding',
    'gene_id' 
);

Gramene::CDBI::EnsemblRice::Gene->has_many( 
    'transcripts',
    'Gramene::CDBI::EnsemblRice::Transcript',
    'gene_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            alt_allele => [ 'gene_id', 'gene_id' ],
            analysis => [ 'analysis_id', 'analysis_id' ],
            gene_attrib => [ 'gene_id', 'gene_id' ],
            gene_stable_id => [ 'gene_id', 'gene_id' ],
            regulatory_factor_coding => [ 'gene_id', 'gene_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
            transcript => [ 'gene_id', 'gene_id' ],
            xref => [ 'display_xref_id', 'xref_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::IdentityXref; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::IdentityXref->table('identity_xref');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub object_xref {
    my $self = shift;
    return $self->object_xref_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::IdentityXref->columns( 
    All => qw/ 
        object_xref_id
        query_identity
        target_identity
        hit_start
        hit_end
        translation_start
        translation_end
        cigar_line
        score
        evalue
        analysis_id 
    /, @extras
);

Gramene::CDBI::EnsemblRice::IdentityXref->columns( 
    Essential => qw/ 
        object_xref_id
        query_identity
        target_identity
        hit_start
        hit_end
        translation_start
        translation_end
        cigar_line
        score
        evalue
        analysis_id 
    /
);

Gramene::CDBI::EnsemblRice::IdentityXref->columns( 
    Ordered => qw/ 
        object_xref_id
        query_identity
        target_identity
        hit_start
        hit_end
        translation_start
        translation_end
        cigar_line
        score
        evalue
        analysis_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->query_identity || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'object_xref_id' => 'number',
        'query_identity' => 'number',
        'target_identity' => 'number',
        'hit_start' => 'number',
        'hit_end' => 'number',
        'translation_start' => 'number',
        'translation_end' => 'number',
        'cigar_line' => 'string',
        'score' => 'number',
        'evalue' => 'number',
        'analysis_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::IdentityXref->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblRice::Analysis'
);

Gramene::CDBI::EnsemblRice::IdentityXref->has_a(
    object_xref_id => 'Gramene::CDBI::EnsemblRice::ObjectXref'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            object_xref => [ 'object_xref_id', 'object_xref_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::MarkerFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::MarkerFeature->table('marker_feature');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub marker {
    my $self = shift;
    return $self->marker_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::MarkerFeature->columns( 
    All => qw/ 
        marker_feature_id
        marker_id
        seq_region_id
        seq_region_start
        seq_region_end
        analysis_id
        map_weight 
    /, @extras
);

Gramene::CDBI::EnsemblRice::MarkerFeature->columns( 
    Essential => qw/ 
        marker_feature_id
        marker_id
        seq_region_id
        seq_region_start
        seq_region_end
        analysis_id
        map_weight 
    /
);

Gramene::CDBI::EnsemblRice::MarkerFeature->columns( 
    Ordered => qw/ 
        marker_feature_id
        marker_id
        seq_region_id
        seq_region_start
        seq_region_end
        analysis_id
        map_weight 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_feature_id' => 'number',
        'marker_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'analysis_id' => 'number',
        'map_weight' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::MarkerFeature->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblRice::Analysis'
);

Gramene::CDBI::EnsemblRice::MarkerFeature->has_a(
    marker_id => 'Gramene::CDBI::EnsemblRice::Marker'
);

Gramene::CDBI::EnsemblRice::MarkerFeature->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            marker => [ 'marker_id', 'marker_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::OligoFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::OligoFeature->table('oligo_feature');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub oligo_probe {
    my $self = shift;
    return $self->oligo_probe_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::OligoFeature->columns( 
    All => qw/ 
        oligo_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        mismatches
        oligo_probe_id
        analysis_id 
    /, @extras
);

Gramene::CDBI::EnsemblRice::OligoFeature->columns( 
    Essential => qw/ 
        oligo_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        mismatches
        oligo_probe_id
        analysis_id 
    /
);

Gramene::CDBI::EnsemblRice::OligoFeature->columns( 
    Ordered => qw/ 
        oligo_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        mismatches
        oligo_probe_id
        analysis_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'oligo_feature_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'mismatches' => 'string',
        'oligo_probe_id' => 'number',
        'analysis_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::OligoFeature->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblRice::Analysis'
);

Gramene::CDBI::EnsemblRice::OligoFeature->has_a(
    oligo_probe_id => 'Gramene::CDBI::EnsemblRice::OligoProbe'
);

Gramene::CDBI::EnsemblRice::OligoFeature->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            oligo_probe => [ 'oligo_probe_id', 'oligo_probe_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::PredictionExon; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::PredictionExon->table('prediction_exon');


sub prediction_transcript {
    my $self = shift;
    return $self->prediction_transcript_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::PredictionExon->columns( 
    All => qw/ 
        prediction_exon_id
        prediction_transcript_id
        exon_rank
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        start_phase
        score
        p_value 
    /, @extras
);

Gramene::CDBI::EnsemblRice::PredictionExon->columns( 
    Essential => qw/ 
        prediction_exon_id
        prediction_transcript_id
        exon_rank
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        start_phase
        score
        p_value 
    /
);

Gramene::CDBI::EnsemblRice::PredictionExon->columns( 
    Ordered => qw/ 
        prediction_exon_id
        prediction_transcript_id
        exon_rank
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        start_phase
        score
        p_value 
    /
);

sub to_string {
    my $self = shift;
    return $self->exon_rank || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'prediction_exon_id' => 'number',
        'prediction_transcript_id' => 'number',
        'exon_rank' => 'string',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'start_phase' => 'string',
        'score' => 'number',
        'p_value' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::PredictionExon->has_a(
    prediction_transcript_id => 'Gramene::CDBI::EnsemblRice::PredictionTranscript'
);

Gramene::CDBI::EnsemblRice::PredictionExon->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            prediction_transcript => [ 'prediction_transcript_id', 'prediction_transcript_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::PredictionTranscript; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::PredictionTranscript->table('prediction_transcript');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::PredictionTranscript->columns( 
    All => qw/ 
        prediction_transcript_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        analysis_id
        display_label 
    /, @extras
);

Gramene::CDBI::EnsemblRice::PredictionTranscript->columns( 
    Essential => qw/ 
        prediction_transcript_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        analysis_id
        display_label 
    /
);

Gramene::CDBI::EnsemblRice::PredictionTranscript->columns( 
    Ordered => qw/ 
        prediction_transcript_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        analysis_id
        display_label 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'prediction_transcript_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'analysis_id' => 'number',
        'display_label' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::PredictionTranscript->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblRice::Analysis'
);

Gramene::CDBI::EnsemblRice::PredictionTranscript->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);

Gramene::CDBI::EnsemblRice::PredictionTranscript->has_many( 
    'prediction_exons',
    'Gramene::CDBI::EnsemblRice::PredictionExon',
    'prediction_transcript_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            prediction_exon => [ 'prediction_transcript_id', 'prediction_transcript_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::ProteinAlignFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::ProteinAlignFeature->table('protein_align_feature');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::ProteinAlignFeature->columns( 
    All => qw/ 
        protein_align_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        hit_start
        hit_end
        hit_name
        analysis_id
        score
        evalue
        perc_ident
        cigar_line 
    /, @extras
);

Gramene::CDBI::EnsemblRice::ProteinAlignFeature->columns( 
    Essential => qw/ 
        protein_align_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        hit_start
        hit_end
        hit_name
        analysis_id
        score
        evalue
        perc_ident
        cigar_line 
    /
);

Gramene::CDBI::EnsemblRice::ProteinAlignFeature->columns( 
    Ordered => qw/ 
        protein_align_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        hit_start
        hit_end
        hit_name
        analysis_id
        score
        evalue
        perc_ident
        cigar_line 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'protein_align_feature_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'hit_start' => 'number',
        'hit_end' => 'number',
        'hit_name' => 'string',
        'analysis_id' => 'number',
        'score' => 'number',
        'evalue' => 'number',
        'perc_ident' => 'number',
        'cigar_line' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::ProteinAlignFeature->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblRice::Analysis'
);

Gramene::CDBI::EnsemblRice::ProteinAlignFeature->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::QtlFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::QtlFeature->table('qtl_feature');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub qtl {
    my $self = shift;
    return $self->qtl_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::QtlFeature->columns( 
    All => qw/ 
        seq_region_id
        seq_region_start
        seq_region_end
        qtl_id
        analysis_id 
    /, @extras
);

Gramene::CDBI::EnsemblRice::QtlFeature->columns( 
    Essential => qw/ 
        seq_region_id
        seq_region_start
        seq_region_end
        qtl_id
        analysis_id 
    /
);

Gramene::CDBI::EnsemblRice::QtlFeature->columns( 
    Ordered => qw/ 
        seq_region_id
        seq_region_start
        seq_region_end
        qtl_id
        analysis_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'qtl_id' => 'number',
        'analysis_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::QtlFeature->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblRice::Analysis'
);

Gramene::CDBI::EnsemblRice::QtlFeature->has_a(
    qtl_id => 'Gramene::CDBI::EnsemblRice::Qtl'
);

Gramene::CDBI::EnsemblRice::QtlFeature->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            qtl => [ 'qtl_id', 'qtl_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::RegulatoryFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::RegulatoryFeature->table('regulatory_feature');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub regulatory_factor {
    my $self = shift;
    return $self->regulatory_factor_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::RegulatoryFeature->columns( 
    All => qw/ 
        regulatory_feature_id
        name
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        analysis_id
        regulatory_factor_id 
    /, @extras
);

Gramene::CDBI::EnsemblRice::RegulatoryFeature->columns( 
    Essential => qw/ 
        regulatory_feature_id
        name
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        analysis_id
        regulatory_factor_id 
    /
);

Gramene::CDBI::EnsemblRice::RegulatoryFeature->columns( 
    Ordered => qw/ 
        regulatory_feature_id
        name
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        analysis_id
        regulatory_factor_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'regulatory_feature_id' => 'number',
        'name' => 'string',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'analysis_id' => 'number',
        'regulatory_factor_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::RegulatoryFeature->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblRice::Analysis'
);

Gramene::CDBI::EnsemblRice::RegulatoryFeature->has_a(
    regulatory_factor_id => 'Gramene::CDBI::EnsemblRice::RegulatoryFactor'
);

Gramene::CDBI::EnsemblRice::RegulatoryFeature->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            regulatory_factor => [ 'regulatory_factor_id', 'regulatory_factor_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::RegulatorySearchRegion; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::RegulatorySearchRegion->table('regulatory_search_region');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::RegulatorySearchRegion->columns( 
    All => qw/ 
        regulatory_search_region_id
        name
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        ensembl_object_type
        ensembl_object_id
        analysis_id 
    /, @extras
);

Gramene::CDBI::EnsemblRice::RegulatorySearchRegion->columns( 
    Essential => qw/ 
        regulatory_search_region_id
        name
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        ensembl_object_type
        ensembl_object_id
        analysis_id 
    /
);

Gramene::CDBI::EnsemblRice::RegulatorySearchRegion->columns( 
    Ordered => qw/ 
        regulatory_search_region_id
        name
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        ensembl_object_type
        ensembl_object_id
        analysis_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'regulatory_search_region_id' => 'number',
        'name' => 'string',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'ensembl_object_type' => 'string',
        'ensembl_object_id' => 'number',
        'analysis_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::RegulatorySearchRegion->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblRice::Analysis'
);

Gramene::CDBI::EnsemblRice::RegulatorySearchRegion->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::RepeatFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::RepeatFeature->table('repeat_feature');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub repeat_consensus {
    my $self = shift;
    return $self->repeat_consensus_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::RepeatFeature->columns( 
    All => qw/ 
        repeat_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        repeat_start
        repeat_end
        repeat_consensus_id
        analysis_id
        score 
    /, @extras
);

Gramene::CDBI::EnsemblRice::RepeatFeature->columns( 
    Essential => qw/ 
        repeat_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        repeat_start
        repeat_end
        repeat_consensus_id
        analysis_id
        score 
    /
);

Gramene::CDBI::EnsemblRice::RepeatFeature->columns( 
    Ordered => qw/ 
        repeat_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        repeat_start
        repeat_end
        repeat_consensus_id
        analysis_id
        score 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'repeat_feature_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'repeat_start' => 'number',
        'repeat_end' => 'number',
        'repeat_consensus_id' => 'number',
        'analysis_id' => 'number',
        'score' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::RepeatFeature->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblRice::Analysis'
);

Gramene::CDBI::EnsemblRice::RepeatFeature->has_a(
    repeat_consensus_id => 'Gramene::CDBI::EnsemblRice::RepeatConsensus'
);

Gramene::CDBI::EnsemblRice::RepeatFeature->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            repeat_consensus => [ 'repeat_consensus_id', 'repeat_consensus_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::SimpleFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::SimpleFeature->table('simple_feature');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::SimpleFeature->columns( 
    All => qw/ 
        simple_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        display_label
        analysis_id
        score 
    /, @extras
);

Gramene::CDBI::EnsemblRice::SimpleFeature->columns( 
    Essential => qw/ 
        simple_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        display_label
        analysis_id
        score 
    /
);

Gramene::CDBI::EnsemblRice::SimpleFeature->columns( 
    Ordered => qw/ 
        simple_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        display_label
        analysis_id
        score 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'simple_feature_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'display_label' => 'string',
        'analysis_id' => 'number',
        'score' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::SimpleFeature->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblRice::Analysis'
);

Gramene::CDBI::EnsemblRice::SimpleFeature->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::UnmappedObject; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::UnmappedObject->table('unmapped_object');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub external_db {
    my $self = shift;
    return $self->external_db_id;
}
    

sub unmapped_reason {
    my $self = shift;
    return $self->unmapped_reason_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::UnmappedObject->columns( 
    All => qw/ 
        unmapped_object_id
        type
        analysis_id
        external_db_id
        identifier
        unmapped_reason_id
        query_score
        target_score
        ensembl_id
        ensembl_object_type 
    /, @extras
);

Gramene::CDBI::EnsemblRice::UnmappedObject->columns( 
    Essential => qw/ 
        unmapped_object_id
        type
        analysis_id
        external_db_id
        identifier
        unmapped_reason_id
        query_score
        target_score
        ensembl_id
        ensembl_object_type 
    /
);

Gramene::CDBI::EnsemblRice::UnmappedObject->columns( 
    Ordered => qw/ 
        unmapped_object_id
        type
        analysis_id
        external_db_id
        identifier
        unmapped_reason_id
        query_score
        target_score
        ensembl_id
        ensembl_object_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'unmapped_object_id' => 'number',
        'type' => 'string',
        'analysis_id' => 'number',
        'external_db_id' => 'number',
        'identifier' => 'string',
        'unmapped_reason_id' => 'string',
        'query_score' => 'number',
        'target_score' => 'number',
        'ensembl_id' => 'number',
        'ensembl_object_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::UnmappedObject->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblRice::Analysis'
);

Gramene::CDBI::EnsemblRice::UnmappedObject->has_a(
    external_db_id => 'Gramene::CDBI::EnsemblRice::ExternalDb'
);

Gramene::CDBI::EnsemblRice::UnmappedObject->has_a(
    unmapped_reason_id => 'Gramene::CDBI::EnsemblRice::UnmappedReason'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            external_db => [ 'external_db_id', 'external_db_id' ],
            unmapped_reason => [ 'unmapped_reason_id', 'unmapped_reason_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::Analysis; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::Analysis->table('analysis');

our @extras = ();
Gramene::CDBI::EnsemblRice::Analysis->columns( 
    All => qw/ 
        analysis_id
        created
        logic_name
        db
        db_version
        db_file
        program
        program_version
        program_file
        parameters
        module
        module_version
        gff_source
        gff_feature 
    /, @extras
);

Gramene::CDBI::EnsemblRice::Analysis->columns( 
    Essential => qw/ 
        analysis_id
        created
        logic_name
        db
        db_version
        db_file
        program
        program_version
        program_file
        parameters
        module
        module_version
        gff_source
        gff_feature 
    /
);

Gramene::CDBI::EnsemblRice::Analysis->columns( 
    Ordered => qw/ 
        analysis_id
        created
        logic_name
        db
        db_version
        db_file
        program
        program_version
        program_file
        parameters
        module
        module_version
        gff_source
        gff_feature 
    /
);

sub to_string {
    my $self = shift;
    return $self->created || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'analysis_id' => 'number',
        'created' => 'string',
        'logic_name' => 'string',
        'db' => 'string',
        'db_version' => 'string',
        'db_file' => 'string',
        'program' => 'string',
        'program_version' => 'string',
        'program_file' => 'string',
        'parameters' => 'string',
        'module' => 'string',
        'module_version' => 'string',
        'gff_source' => 'string',
        'gff_feature' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::Analysis->has_many( 
    'analysis_descriptions',
    'Gramene::CDBI::EnsemblRice::AnalysisDescription',
    'analysis_id' 
);

Gramene::CDBI::EnsemblRice::Analysis->has_many( 
    'density_types',
    'Gramene::CDBI::EnsemblRice::DensityType',
    'analysis_id' 
);

Gramene::CDBI::EnsemblRice::Analysis->has_many( 
    'ditag_features',
    'Gramene::CDBI::EnsemblRice::DitagFeature',
    'analysis_id' 
);

Gramene::CDBI::EnsemblRice::Analysis->has_many( 
    'dna_align_features',
    'Gramene::CDBI::EnsemblRice::DnaAlignFeature',
    'analysis_id' 
);

Gramene::CDBI::EnsemblRice::Analysis->has_many( 
    'genes',
    'Gramene::CDBI::EnsemblRice::Gene',
    'analysis_id' 
);

Gramene::CDBI::EnsemblRice::Analysis->has_many( 
    'identity_xrefs',
    'Gramene::CDBI::EnsemblRice::IdentityXref',
    'analysis_id' 
);

Gramene::CDBI::EnsemblRice::Analysis->has_many( 
    'marker_features',
    'Gramene::CDBI::EnsemblRice::MarkerFeature',
    'analysis_id' 
);

Gramene::CDBI::EnsemblRice::Analysis->has_many( 
    'oligo_features',
    'Gramene::CDBI::EnsemblRice::OligoFeature',
    'analysis_id' 
);

Gramene::CDBI::EnsemblRice::Analysis->has_many( 
    'prediction_transcripts',
    'Gramene::CDBI::EnsemblRice::PredictionTranscript',
    'analysis_id' 
);

Gramene::CDBI::EnsemblRice::Analysis->has_many( 
    'protein_align_features',
    'Gramene::CDBI::EnsemblRice::ProteinAlignFeature',
    'analysis_id' 
);

Gramene::CDBI::EnsemblRice::Analysis->has_many( 
    'protein_features',
    'Gramene::CDBI::EnsemblRice::ProteinFeature',
    'analysis_id' 
);

Gramene::CDBI::EnsemblRice::Analysis->has_many( 
    'qtl_features',
    'Gramene::CDBI::EnsemblRice::QtlFeature',
    'analysis_id' 
);

Gramene::CDBI::EnsemblRice::Analysis->has_many( 
    'regulatory_features',
    'Gramene::CDBI::EnsemblRice::RegulatoryFeature',
    'analysis_id' 
);

Gramene::CDBI::EnsemblRice::Analysis->has_many( 
    'regulatory_search_regions',
    'Gramene::CDBI::EnsemblRice::RegulatorySearchRegion',
    'analysis_id' 
);

Gramene::CDBI::EnsemblRice::Analysis->has_many( 
    'repeat_features',
    'Gramene::CDBI::EnsemblRice::RepeatFeature',
    'analysis_id' 
);

Gramene::CDBI::EnsemblRice::Analysis->has_many( 
    'simple_features',
    'Gramene::CDBI::EnsemblRice::SimpleFeature',
    'analysis_id' 
);

Gramene::CDBI::EnsemblRice::Analysis->has_many( 
    'transcripts',
    'Gramene::CDBI::EnsemblRice::Transcript',
    'analysis_id' 
);

Gramene::CDBI::EnsemblRice::Analysis->has_many( 
    'unmapped_objects',
    'Gramene::CDBI::EnsemblRice::UnmappedObject',
    'analysis_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis_description => [ 'analysis_id', 'analysis_id' ],
            density_type => [ 'analysis_id', 'analysis_id' ],
            ditag_feature => [ 'analysis_id', 'analysis_id' ],
            dna_align_feature => [ 'analysis_id', 'analysis_id' ],
            gene => [ 'analysis_id', 'analysis_id' ],
            identity_xref => [ 'analysis_id', 'analysis_id' ],
            marker_feature => [ 'analysis_id', 'analysis_id' ],
            oligo_feature => [ 'analysis_id', 'analysis_id' ],
            prediction_transcript => [ 'analysis_id', 'analysis_id' ],
            protein_align_feature => [ 'analysis_id', 'analysis_id' ],
            protein_feature => [ 'analysis_id', 'analysis_id' ],
            qtl_feature => [ 'analysis_id', 'analysis_id' ],
            regulatory_feature => [ 'analysis_id', 'analysis_id' ],
            regulatory_search_region => [ 'analysis_id', 'analysis_id' ],
            repeat_feature => [ 'analysis_id', 'analysis_id' ],
            simple_feature => [ 'analysis_id', 'analysis_id' ],
            transcript => [ 'analysis_id', 'analysis_id' ],
            unmapped_object => [ 'analysis_id', 'analysis_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::Assembly; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::Assembly->table('assembly');


sub asm_seq_region {
    my $self = shift;
    return $self->asm_seq_region_id;
}
    
sub cmp_seq_region {
    my $self = shift;
    return $self->cmp_seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::Assembly->columns( 
    All => qw/ 
        asm_seq_region_id
        cmp_seq_region_id
        asm_start
        asm_end
        cmp_start
        cmp_end
        ori 
    /, @extras
);

Gramene::CDBI::EnsemblRice::Assembly->columns( 
    Essential => qw/ 
        asm_seq_region_id
        cmp_seq_region_id
        asm_start
        asm_end
        cmp_start
        cmp_end
        ori 
    /
);

Gramene::CDBI::EnsemblRice::Assembly->columns( 
    Ordered => qw/ 
        asm_seq_region_id
        cmp_seq_region_id
        asm_start
        asm_end
        cmp_start
        cmp_end
        ori 
    /
);

sub to_string {
    my $self = shift;
    return $self->asm_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'asm_seq_region_id' => 'number',
        'cmp_seq_region_id' => 'number',
        'asm_start' => 'number',
        'asm_end' => 'number',
        'cmp_start' => 'number',
        'cmp_end' => 'number',
        'ori' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::Assembly->has_a(
    asm_seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);

Gramene::CDBI::EnsemblRice::Assembly->has_a(
    cmp_seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            seq_region => [ 'cmp_seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::AssemblyException; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::AssemblyException->table('assembly_exception');


sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
sub exc_seq_region {
    my $self = shift;
    return $self->exc_seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::AssemblyException->columns( 
    All => qw/ 
        assembly_exception_id
        seq_region_id
        seq_region_start
        seq_region_end
        exc_type
        exc_seq_region_id
        exc_seq_region_start
        exc_seq_region_end
        ori 
    /, @extras
);

Gramene::CDBI::EnsemblRice::AssemblyException->columns( 
    Essential => qw/ 
        assembly_exception_id
        seq_region_id
        seq_region_start
        seq_region_end
        exc_type
        exc_seq_region_id
        exc_seq_region_start
        exc_seq_region_end
        ori 
    /
);

Gramene::CDBI::EnsemblRice::AssemblyException->columns( 
    Ordered => qw/ 
        assembly_exception_id
        seq_region_id
        seq_region_start
        seq_region_end
        exc_type
        exc_seq_region_id
        exc_seq_region_start
        exc_seq_region_end
        ori 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'assembly_exception_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'exc_type' => 'string',
        'exc_seq_region_id' => 'number',
        'exc_seq_region_start' => 'number',
        'exc_seq_region_end' => 'number',
        'ori' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::AssemblyException->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);

Gramene::CDBI::EnsemblRice::AssemblyException->has_a(
    exc_seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            seq_region => [ 'exc_seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::MiscAttrib; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::MiscAttrib->table('misc_attrib');


sub attrib_type {
    my $self = shift;
    return $self->attrib_type_id;
}
    

sub misc_feature {
    my $self = shift;
    return $self->misc_feature_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::MiscAttrib->columns( 
    All => qw/ 
        misc_feature_id
        attrib_type_id
        value 
    /, @extras
);

Gramene::CDBI::EnsemblRice::MiscAttrib->columns( 
    Essential => qw/ 
        misc_feature_id
        attrib_type_id
        value 
    /
);

Gramene::CDBI::EnsemblRice::MiscAttrib->columns( 
    Ordered => qw/ 
        misc_feature_id
        attrib_type_id
        value 
    /
);

sub to_string {
    my $self = shift;
    return $self->value || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'misc_feature_id' => 'number',
        'attrib_type_id' => 'string',
        'value' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::MiscAttrib->has_a(
    attrib_type_id => 'Gramene::CDBI::EnsemblRice::AttribType'
);

Gramene::CDBI::EnsemblRice::MiscAttrib->has_a(
    misc_feature_id => 'Gramene::CDBI::EnsemblRice::MiscFeature'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            attrib_type => [ 'attrib_type_id', 'attrib_type_id' ],
            misc_feature => [ 'misc_feature_id', 'misc_feature_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::SeqRegionAttrib; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::SeqRegionAttrib->table('seq_region_attrib');


sub attrib_type {
    my $self = shift;
    return $self->attrib_type_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::SeqRegionAttrib->columns( 
    All => qw/ 
        seq_region_id
        attrib_type_id
        value 
    /, @extras
);

Gramene::CDBI::EnsemblRice::SeqRegionAttrib->columns( 
    Essential => qw/ 
        seq_region_id
        attrib_type_id
        value 
    /
);

Gramene::CDBI::EnsemblRice::SeqRegionAttrib->columns( 
    Ordered => qw/ 
        seq_region_id
        attrib_type_id
        value 
    /
);

sub to_string {
    my $self = shift;
    return $self->value || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'seq_region_id' => 'number',
        'attrib_type_id' => 'string',
        'value' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::SeqRegionAttrib->has_a(
    attrib_type_id => 'Gramene::CDBI::EnsemblRice::AttribType'
);

Gramene::CDBI::EnsemblRice::SeqRegionAttrib->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            attrib_type => [ 'attrib_type_id', 'attrib_type_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::AttribType; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::AttribType->table('attrib_type');

our @extras = ();
Gramene::CDBI::EnsemblRice::AttribType->columns( 
    All => qw/ 
        attrib_type_id
        code
        name
        description 
    /, @extras
);

Gramene::CDBI::EnsemblRice::AttribType->columns( 
    Essential => qw/ 
        attrib_type_id
        code
        name
        description 
    /
);

Gramene::CDBI::EnsemblRice::AttribType->columns( 
    Ordered => qw/ 
        attrib_type_id
        code
        name
        description 
    /
);

sub to_string {
    my $self = shift;
    return $self->code || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'attrib_type_id' => 'string',
        'code' => 'string',
        'name' => 'string',
        'description' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::AttribType->has_many( 
    'gene_attribs',
    'Gramene::CDBI::EnsemblRice::GeneAttrib',
    'attrib_type_id' 
);

Gramene::CDBI::EnsemblRice::AttribType->has_many( 
    'misc_attribs',
    'Gramene::CDBI::EnsemblRice::MiscAttrib',
    'attrib_type_id' 
);

Gramene::CDBI::EnsemblRice::AttribType->has_many( 
    'seq_region_attribs',
    'Gramene::CDBI::EnsemblRice::SeqRegionAttrib',
    'attrib_type_id' 
);

Gramene::CDBI::EnsemblRice::AttribType->has_many( 
    'transcript_attribs',
    'Gramene::CDBI::EnsemblRice::TranscriptAttrib',
    'attrib_type_id' 
);

Gramene::CDBI::EnsemblRice::AttribType->has_many( 
    'translation_attribs',
    'Gramene::CDBI::EnsemblRice::TranslationAttrib',
    'attrib_type_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_attrib => [ 'attrib_type_id', 'attrib_type_id' ],
            misc_attrib => [ 'attrib_type_id', 'attrib_type_id' ],
            seq_region_attrib => [ 'attrib_type_id', 'attrib_type_id' ],
            transcript_attrib => [ 'attrib_type_id', 'attrib_type_id' ],
            translation_attrib => [ 'attrib_type_id', 'attrib_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::Bacpac; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::Bacpac->table('bacpac');

our @extras = ();
Gramene::CDBI::EnsemblRice::Bacpac->columns( 
    All => qw/ 
        embl_acc
        embl_version
        htg_phase
        bacname
        site
        gi
        chrnum
        length
        seq_region_id 
    /, @extras
);

Gramene::CDBI::EnsemblRice::Bacpac->columns( 
    Essential => qw/ 
        embl_acc
        embl_version
        htg_phase
        bacname
        site
        gi
        chrnum
        length
        seq_region_id 
    /
);

Gramene::CDBI::EnsemblRice::Bacpac->columns( 
    Ordered => qw/ 
        embl_acc
        embl_version
        htg_phase
        bacname
        site
        gi
        chrnum
        length
        seq_region_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->embl_version || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'embl_acc' => 'string',
        'embl_version' => 'number',
        'htg_phase' => 'number',
        'bacname' => 'string',
        'site' => 'string',
        'gi' => 'number',
        'chrnum' => 'number',
        'length' => 'number',
        'seq_region_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::MetaCoord; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::MetaCoord->table('meta_coord');


sub coord_system {
    my $self = shift;
    return $self->coord_system_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::MetaCoord->columns( 
    All => qw/ 
        table_name
        coord_system_id
        max_length 
    /, @extras
);

Gramene::CDBI::EnsemblRice::MetaCoord->columns( 
    Essential => qw/ 
        table_name
        coord_system_id
        max_length 
    /
);

Gramene::CDBI::EnsemblRice::MetaCoord->columns( 
    Ordered => qw/ 
        table_name
        coord_system_id
        max_length 
    /
);

sub to_string {
    my $self = shift;
    return $self->table_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'table_name' => 'string',
        'coord_system_id' => 'number',
        'max_length' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::MetaCoord->has_a(
    coord_system_id => 'Gramene::CDBI::EnsemblRice::CoordSystem'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            coord_system => [ 'coord_system_id', 'coord_system_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::Dna; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::Dna->table('dna');


sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
push @extras, Class::DBI::Column->new( sequence => { 
    accessor => 'seq',
    mutator  => 'seq',
}); 

Gramene::CDBI::EnsemblRice::Dna->columns( 
    All => qw/ 
        seq_region_id 
    /, @extras
);

Gramene::CDBI::EnsemblRice::Dna->columns( 
    Essential => qw/ 
        seq_region_id 
    /
);

Gramene::CDBI::EnsemblRice::Dna->columns( 
    Ordered => qw/ 
        seq_region_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->sequence || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'seq_region_id' => 'number',
        'sequence' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::Dna->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::Dnac; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::Dnac->table('dnac');


sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
push @extras, Class::DBI::Column->new( sequence => { 
    accessor => 'seq',
    mutator  => 'seq',
}); 

Gramene::CDBI::EnsemblRice::Dnac->columns( 
    All => qw/ 
        seq_region_id
        n_line 
    /, @extras
);

Gramene::CDBI::EnsemblRice::Dnac->columns( 
    Essential => qw/ 
        seq_region_id
        n_line 
    /
);

Gramene::CDBI::EnsemblRice::Dnac->columns( 
    Ordered => qw/ 
        seq_region_id
        n_line 
    /
);

sub to_string {
    my $self = shift;
    return $self->sequence || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'seq_region_id' => 'number',
        'sequence' => 'string',
        'n_line' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::Dnac->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::ExonStableId; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::ExonStableId->table('exon_stable_id');


sub exon {
    my $self = shift;
    return $self->exon_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::ExonStableId->columns( 
    All => qw/ 
        exon_id
        stable_id
        version
        created_date
        modified_date 
    /, @extras
);

Gramene::CDBI::EnsemblRice::ExonStableId->columns( 
    Essential => qw/ 
        exon_id
        stable_id
        version
        created_date
        modified_date 
    /
);

Gramene::CDBI::EnsemblRice::ExonStableId->columns( 
    Ordered => qw/ 
        exon_id
        stable_id
        version
        created_date
        modified_date 
    /
);

sub to_string {
    my $self = shift;
    return $self->version || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'exon_id' => 'number',
        'stable_id' => 'string',
        'version' => 'number',
        'created_date' => 'string',
        'modified_date' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::ExonStableId->has_a(
    exon_id => 'Gramene::CDBI::EnsemblRice::Exon'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            exon => [ 'exon_id', 'exon_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::SupportingFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::SupportingFeature->table('supporting_feature');


sub exon {
    my $self = shift;
    return $self->exon_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::SupportingFeature->columns( 
    All => qw/ 
        exon_id
        feature_type
        feature_id 
    /, @extras
);

Gramene::CDBI::EnsemblRice::SupportingFeature->columns( 
    Essential => qw/ 
        exon_id
        feature_type
        feature_id 
    /
);

Gramene::CDBI::EnsemblRice::SupportingFeature->columns( 
    Ordered => qw/ 
        exon_id
        feature_type
        feature_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->feature_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'exon_id' => 'number',
        'feature_type' => 'string',
        'feature_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::SupportingFeature->has_a(
    exon_id => 'Gramene::CDBI::EnsemblRice::Exon'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            exon => [ 'exon_id', 'exon_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::Exon; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::Exon->table('exon');


sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::Exon->columns( 
    All => qw/ 
        exon_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        phase
        end_phase
        is_current 
    /, @extras
);

Gramene::CDBI::EnsemblRice::Exon->columns( 
    Essential => qw/ 
        exon_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        phase
        end_phase
        is_current 
    /
);

Gramene::CDBI::EnsemblRice::Exon->columns( 
    Ordered => qw/ 
        exon_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        phase
        end_phase
        is_current 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'exon_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'phase' => 'string',
        'end_phase' => 'string',
        'is_current' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::Exon->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);

Gramene::CDBI::EnsemblRice::Exon->has_many( 
    'exon_stable_ids',
    'Gramene::CDBI::EnsemblRice::ExonStableId',
    'exon_id' 
);

Gramene::CDBI::EnsemblRice::Exon->has_many( 
    'exon_transcripts',
    'Gramene::CDBI::EnsemblRice::ExonTranscript',
    'exon_id' 
);

Gramene::CDBI::EnsemblRice::Exon->has_many( 
    'supporting_features',
    'Gramene::CDBI::EnsemblRice::SupportingFeature',
    'exon_id' 
);

Gramene::CDBI::EnsemblRice::Exon->has_many( 
    'translations_start_exon',
    'Gramene::CDBI::EnsemblRice::Translation',
    'start_exon_id' 
);

Gramene::CDBI::EnsemblRice::Exon->has_many( 
    'translations_end_exon',
    'Gramene::CDBI::EnsemblRice::Translation',
    'end_exon_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            exon_stable_id => [ 'exon_id', 'exon_id' ],
            exon_transcript => [ 'exon_id', 'exon_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
            supporting_feature => [ 'exon_id', 'exon_id' ],
            translation => [ 'exon_id', 'end_exon_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::Karyotype; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::Karyotype->table('karyotype');


sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::Karyotype->columns( 
    All => qw/ 
        karyotype_id
        seq_region_id
        seq_region_start
        seq_region_end
        band
        stain 
    /, @extras
);

Gramene::CDBI::EnsemblRice::Karyotype->columns( 
    Essential => qw/ 
        karyotype_id
        seq_region_id
        seq_region_start
        seq_region_end
        band
        stain 
    /
);

Gramene::CDBI::EnsemblRice::Karyotype->columns( 
    Ordered => qw/ 
        karyotype_id
        seq_region_id
        seq_region_start
        seq_region_end
        band
        stain 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'karyotype_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'band' => 'string',
        'stain' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::Karyotype->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::MiscFeatureMiscSet; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::MiscFeatureMiscSet->table('misc_feature_misc_set');


sub misc_feature {
    my $self = shift;
    return $self->misc_feature_id;
}
    

sub misc_set {
    my $self = shift;
    return $self->misc_set_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::MiscFeatureMiscSet->columns( 
    All => qw/ 
        misc_feature_id
        misc_set_id 
    /, @extras
);

Gramene::CDBI::EnsemblRice::MiscFeatureMiscSet->columns( 
    Essential => qw/ 
        misc_feature_id
        misc_set_id 
    /
);

Gramene::CDBI::EnsemblRice::MiscFeatureMiscSet->columns( 
    Ordered => qw/ 
        misc_feature_id
        misc_set_id 
    /
);


Gramene::CDBI::EnsemblRice::MiscFeatureMiscSet->columns( 
    Primary => qw/ 
        misc_feature_id
        misc_set_id 
    / 
);


sub to_string {
    my $self = shift;
    return $self->misc_feature_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'misc_feature_id' => 'number',
        'misc_set_id' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::MiscFeatureMiscSet->has_a(
    misc_feature_id => 'Gramene::CDBI::EnsemblRice::MiscFeature'
);

Gramene::CDBI::EnsemblRice::MiscFeatureMiscSet->has_a(
    misc_set_id => 'Gramene::CDBI::EnsemblRice::MiscSet'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            misc_feature => [ 'misc_feature_id', 'misc_feature_id' ],
            misc_set => [ 'misc_set_id', 'misc_set_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::MiscFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::MiscFeature->table('misc_feature');


sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::MiscFeature->columns( 
    All => qw/ 
        misc_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand 
    /, @extras
);

Gramene::CDBI::EnsemblRice::MiscFeature->columns( 
    Essential => qw/ 
        misc_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand 
    /
);

Gramene::CDBI::EnsemblRice::MiscFeature->columns( 
    Ordered => qw/ 
        misc_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'misc_feature_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::MiscFeature->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblRice::SeqRegion'
);

Gramene::CDBI::EnsemblRice::MiscFeature->has_many( 
    'misc_attribs',
    'Gramene::CDBI::EnsemblRice::MiscAttrib',
    'misc_feature_id' 
);

Gramene::CDBI::EnsemblRice::MiscFeature->has_many( 
    'misc_feature_misc_sets',
    'Gramene::CDBI::EnsemblRice::MiscFeatureMiscSet',
    'misc_feature_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            misc_attrib => [ 'misc_feature_id', 'misc_feature_id' ],
            misc_feature_misc_set => [ 'misc_feature_id', 'misc_feature_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::SeqRegion; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::SeqRegion->table('seq_region');


sub coord_system {
    my $self = shift;
    return $self->coord_system_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::SeqRegion->columns( 
    All => qw/ 
        seq_region_id
        name
        coord_system_id
        length 
    /, @extras
);

Gramene::CDBI::EnsemblRice::SeqRegion->columns( 
    Essential => qw/ 
        seq_region_id
        name
        coord_system_id
        length 
    /
);

Gramene::CDBI::EnsemblRice::SeqRegion->columns( 
    Ordered => qw/ 
        seq_region_id
        name
        coord_system_id
        length 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'seq_region_id' => 'number',
        'name' => 'string',
        'coord_system_id' => 'number',
        'length' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::SeqRegion->has_a(
    coord_system_id => 'Gramene::CDBI::EnsemblRice::CoordSystem'
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'assemblies_asm_seq_region',
    'Gramene::CDBI::EnsemblRice::Assembly',
    'asm_seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'assemblies_cmp_seq_region',
    'Gramene::CDBI::EnsemblRice::Assembly',
    'cmp_seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'assembly_exceptions_seq_region',
    'Gramene::CDBI::EnsemblRice::AssemblyException',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'assembly_exceptions_exc_seq_region',
    'Gramene::CDBI::EnsemblRice::AssemblyException',
    'exc_seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'density_features',
    'Gramene::CDBI::EnsemblRice::DensityFeature',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'ditag_features',
    'Gramene::CDBI::EnsemblRice::DitagFeature',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'dnas',
    'Gramene::CDBI::EnsemblRice::Dna',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'dna_align_features',
    'Gramene::CDBI::EnsemblRice::DnaAlignFeature',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'dnacs',
    'Gramene::CDBI::EnsemblRice::Dnac',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'exons',
    'Gramene::CDBI::EnsemblRice::Exon',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'genes',
    'Gramene::CDBI::EnsemblRice::Gene',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'karyotypes',
    'Gramene::CDBI::EnsemblRice::Karyotype',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'marker_features',
    'Gramene::CDBI::EnsemblRice::MarkerFeature',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'misc_features',
    'Gramene::CDBI::EnsemblRice::MiscFeature',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'oligo_features',
    'Gramene::CDBI::EnsemblRice::OligoFeature',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'prediction_exons',
    'Gramene::CDBI::EnsemblRice::PredictionExon',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'prediction_transcripts',
    'Gramene::CDBI::EnsemblRice::PredictionTranscript',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'protein_align_features',
    'Gramene::CDBI::EnsemblRice::ProteinAlignFeature',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'qtl_features',
    'Gramene::CDBI::EnsemblRice::QtlFeature',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'regulatory_features',
    'Gramene::CDBI::EnsemblRice::RegulatoryFeature',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'regulatory_search_regions',
    'Gramene::CDBI::EnsemblRice::RegulatorySearchRegion',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'repeat_features',
    'Gramene::CDBI::EnsemblRice::RepeatFeature',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'seq_region_attribs',
    'Gramene::CDBI::EnsemblRice::SeqRegionAttrib',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'simple_features',
    'Gramene::CDBI::EnsemblRice::SimpleFeature',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblRice::SeqRegion->has_many( 
    'transcripts',
    'Gramene::CDBI::EnsemblRice::Transcript',
    'seq_region_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            assembly => [ 'seq_region_id', 'cmp_seq_region_id' ],
            assembly_exception => [ 'seq_region_id', 'exc_seq_region_id' ],
            coord_system => [ 'coord_system_id', 'coord_system_id' ],
            density_feature => [ 'seq_region_id', 'seq_region_id' ],
            ditag_feature => [ 'seq_region_id', 'seq_region_id' ],
            dna => [ 'seq_region_id', 'seq_region_id' ],
            dna_align_feature => [ 'seq_region_id', 'seq_region_id' ],
            dnac => [ 'seq_region_id', 'seq_region_id' ],
            exon => [ 'seq_region_id', 'seq_region_id' ],
            gene => [ 'seq_region_id', 'seq_region_id' ],
            karyotype => [ 'seq_region_id', 'seq_region_id' ],
            marker_feature => [ 'seq_region_id', 'seq_region_id' ],
            misc_feature => [ 'seq_region_id', 'seq_region_id' ],
            oligo_feature => [ 'seq_region_id', 'seq_region_id' ],
            prediction_exon => [ 'seq_region_id', 'seq_region_id' ],
            prediction_transcript => [ 'seq_region_id', 'seq_region_id' ],
            protein_align_feature => [ 'seq_region_id', 'seq_region_id' ],
            qtl_feature => [ 'seq_region_id', 'seq_region_id' ],
            regulatory_feature => [ 'seq_region_id', 'seq_region_id' ],
            regulatory_search_region => [ 'seq_region_id', 'seq_region_id' ],
            repeat_feature => [ 'seq_region_id', 'seq_region_id' ],
            seq_region_attrib => [ 'seq_region_id', 'seq_region_id' ],
            simple_feature => [ 'seq_region_id', 'seq_region_id' ],
            transcript => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::CoordSystem; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::CoordSystem->table('coord_system');

our @extras = ();
Gramene::CDBI::EnsemblRice::CoordSystem->columns( 
    All => qw/ 
        coord_system_id
        name
        version
        rank
        attrib 
    /, @extras
);

Gramene::CDBI::EnsemblRice::CoordSystem->columns( 
    Essential => qw/ 
        coord_system_id
        name
        version
        rank
        attrib 
    /
);

Gramene::CDBI::EnsemblRice::CoordSystem->columns( 
    Ordered => qw/ 
        coord_system_id
        name
        version
        rank
        attrib 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'coord_system_id' => 'number',
        'name' => 'string',
        'version' => 'string',
        'rank' => 'number',
        'attrib' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::CoordSystem->has_many( 
    'meta_coords',
    'Gramene::CDBI::EnsemblRice::MetaCoord',
    'coord_system_id' 
);

Gramene::CDBI::EnsemblRice::CoordSystem->has_many( 
    'seq_regions',
    'Gramene::CDBI::EnsemblRice::SeqRegion',
    'coord_system_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            meta_coord => [ 'coord_system_id', 'coord_system_id' ],
            seq_region => [ 'coord_system_id', 'coord_system_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::Ditag; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::Ditag->table('ditag');

our @extras = ();
push @extras, Class::DBI::Column->new( sequence => { 
    accessor => 'seq',
    mutator  => 'seq',
}); 

Gramene::CDBI::EnsemblRice::Ditag->columns( 
    All => qw/ 
        ditag_id
        name
        type
        tag_count 
    /, @extras
);

Gramene::CDBI::EnsemblRice::Ditag->columns( 
    Essential => qw/ 
        ditag_id
        name
        type
        tag_count 
    /
);

Gramene::CDBI::EnsemblRice::Ditag->columns( 
    Ordered => qw/ 
        ditag_id
        name
        type
        tag_count 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'ditag_id' => 'number',
        'name' => 'string',
        'type' => 'string',
        'tag_count' => 'string',
        'sequence' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::Ditag->has_many( 
    'ditag_features',
    'Gramene::CDBI::EnsemblRice::DitagFeature',
    'ditag_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            ditag_feature => [ 'ditag_id', 'ditag_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::EnsemblDafChr; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::EnsemblDafChr->table('ensembl_daf_chr');

our @extras = ();
Gramene::CDBI::EnsemblRice::EnsemblDafChr->columns( 
    All => qw/ 
        dna_align_feature_id
        chromosome_id
        chr_start
        chr_end
        chr_strand
        hit_start
        hit_end
        hit_name
        analysis_id 
    /, @extras
);

Gramene::CDBI::EnsemblRice::EnsemblDafChr->columns( 
    Essential => qw/ 
        dna_align_feature_id
        chromosome_id
        chr_start
        chr_end
        chr_strand
        hit_start
        hit_end
        hit_name
        analysis_id 
    /
);

Gramene::CDBI::EnsemblRice::EnsemblDafChr->columns( 
    Ordered => qw/ 
        dna_align_feature_id
        chromosome_id
        chr_start
        chr_end
        chr_strand
        hit_start
        hit_end
        hit_name
        analysis_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->chr_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'dna_align_feature_id' => 'number',
        'chromosome_id' => 'number',
        'chr_start' => 'string',
        'chr_end' => 'string',
        'chr_strand' => 'string',
        'hit_start' => 'number',
        'hit_end' => 'number',
        'hit_name' => 'string',
        'analysis_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::EnsemblDafGroup; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::EnsemblDafGroup->table('ensembl_daf_group');

our @extras = ();
Gramene::CDBI::EnsemblRice::EnsemblDafGroup->columns( 
    All => qw/ 
        daf_group_id
        dna_align_feature_id
        chr_start
        chr_end
        chr_strand 
    /, @extras
);

Gramene::CDBI::EnsemblRice::EnsemblDafGroup->columns( 
    Essential => qw/ 
        daf_group_id
        dna_align_feature_id
        chr_start
        chr_end
        chr_strand 
    /
);

Gramene::CDBI::EnsemblRice::EnsemblDafGroup->columns( 
    Ordered => qw/ 
        daf_group_id
        dna_align_feature_id
        chr_start
        chr_end
        chr_strand 
    /
);


Gramene::CDBI::EnsemblRice::EnsemblDafGroup->columns( 
    Primary => qw/ 
        daf_group_id
        dna_align_feature_id 
    / 
);


sub to_string {
    my $self = shift;
    return $self->chr_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'daf_group_id' => 'number',
        'dna_align_feature_id' => 'number',
        'chr_start' => 'number',
        'chr_end' => 'number',
        'chr_strand' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::EnsemblGeneHelper; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::EnsemblGeneHelper->table('ensembl_gene_helper');

our @extras = ();
Gramene::CDBI::EnsemblRice::EnsemblGeneHelper->columns( 
    All => qw/ 
        gene_id
        description 
    /, @extras
);

Gramene::CDBI::EnsemblRice::EnsemblGeneHelper->columns( 
    Essential => qw/ 
        gene_id
        description 
    /
);

Gramene::CDBI::EnsemblRice::EnsemblGeneHelper->columns( 
    Ordered => qw/ 
        gene_id
        description 
    /
);

sub to_string {
    my $self = shift;
    return $self->description || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_id' => 'number',
        'description' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::ExternalSynonym; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::ExternalSynonym->table('external_synonym');


sub xref {
    my $self = shift;
    return $self->xref_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::ExternalSynonym->columns( 
    All => qw/ 
        xref_id
        synonym 
    /, @extras
);

Gramene::CDBI::EnsemblRice::ExternalSynonym->columns( 
    Essential => qw/ 
        xref_id
        synonym 
    /
);

Gramene::CDBI::EnsemblRice::ExternalSynonym->columns( 
    Ordered => qw/ 
        xref_id
        synonym 
    /
);


Gramene::CDBI::EnsemblRice::ExternalSynonym->columns( 
    Primary => qw/ 
        xref_id
        synonym 
    / 
);


sub to_string {
    my $self = shift;
    return $self->xref_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'xref_id' => 'number',
        'synonym' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::ExternalSynonym->has_a(
    xref_id => 'Gramene::CDBI::EnsemblRice::Xref'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            xref => [ 'xref_id', 'xref_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::GoXref; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::GoXref->table('go_xref');


sub object_xref {
    my $self = shift;
    return $self->object_xref_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::GoXref->columns( 
    All => qw/ 
        object_xref_id
        linkage_type 
    /, @extras
);

Gramene::CDBI::EnsemblRice::GoXref->columns( 
    Essential => qw/ 
        object_xref_id
        linkage_type 
    /
);

Gramene::CDBI::EnsemblRice::GoXref->columns( 
    Ordered => qw/ 
        object_xref_id
        linkage_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->linkage_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'object_xref_id' => 'number',
        'linkage_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::GoXref->has_a(
    object_xref_id => 'Gramene::CDBI::EnsemblRice::ObjectXref'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            object_xref => [ 'object_xref_id', 'object_xref_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::ObjectXref; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::ObjectXref->table('object_xref');


sub xref {
    my $self = shift;
    return $self->xref_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::ObjectXref->columns( 
    All => qw/ 
        object_xref_id
        ensembl_id
        ensembl_object_type
        xref_id 
    /, @extras
);

Gramene::CDBI::EnsemblRice::ObjectXref->columns( 
    Essential => qw/ 
        object_xref_id
        ensembl_id
        ensembl_object_type
        xref_id 
    /
);

Gramene::CDBI::EnsemblRice::ObjectXref->columns( 
    Ordered => qw/ 
        object_xref_id
        ensembl_id
        ensembl_object_type
        xref_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->ensembl_object_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'object_xref_id' => 'number',
        'ensembl_id' => 'number',
        'ensembl_object_type' => 'string',
        'xref_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::ObjectXref->has_a(
    xref_id => 'Gramene::CDBI::EnsemblRice::Xref'
);

Gramene::CDBI::EnsemblRice::ObjectXref->has_many( 
    'go_xrefs',
    'Gramene::CDBI::EnsemblRice::GoXref',
    'object_xref_id' 
);

Gramene::CDBI::EnsemblRice::ObjectXref->has_many( 
    'identity_xrefs',
    'Gramene::CDBI::EnsemblRice::IdentityXref',
    'object_xref_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            go_xref => [ 'object_xref_id', 'object_xref_id' ],
            identity_xref => [ 'object_xref_id', 'object_xref_id' ],
            xref => [ 'xref_id', 'xref_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::Xref; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::Xref->table('xref');


sub external_db {
    my $self = shift;
    return $self->external_db_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::Xref->columns( 
    All => qw/ 
        xref_id
        external_db_id
        dbprimary_acc
        display_label
        version
        description
        info_type
        info_text 
    /, @extras
);

Gramene::CDBI::EnsemblRice::Xref->columns( 
    Essential => qw/ 
        xref_id
        external_db_id
        dbprimary_acc
        display_label
        version
        description
        info_type
        info_text 
    /
);

Gramene::CDBI::EnsemblRice::Xref->columns( 
    Ordered => qw/ 
        xref_id
        external_db_id
        dbprimary_acc
        display_label
        version
        description
        info_type
        info_text 
    /
);

sub to_string {
    my $self = shift;
    return $self->dbprimary_acc || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'xref_id' => 'number',
        'external_db_id' => 'number',
        'dbprimary_acc' => 'string',
        'display_label' => 'string',
        'version' => 'string',
        'description' => 'string',
        'info_type' => 'string',
        'info_text' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::Xref->has_a(
    external_db_id => 'Gramene::CDBI::EnsemblRice::ExternalDb'
);

Gramene::CDBI::EnsemblRice::Xref->has_many( 
    'external_synonyms',
    'Gramene::CDBI::EnsemblRice::ExternalSynonym',
    'xref_id' 
);

Gramene::CDBI::EnsemblRice::Xref->has_many( 
    'genes',
    'Gramene::CDBI::EnsemblRice::Gene',
    'display_xref_id' 
);

Gramene::CDBI::EnsemblRice::Xref->has_many( 
    'object_xrefs',
    'Gramene::CDBI::EnsemblRice::ObjectXref',
    'xref_id' 
);

Gramene::CDBI::EnsemblRice::Xref->has_many( 
    'transcripts',
    'Gramene::CDBI::EnsemblRice::Transcript',
    'display_xref_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            external_db => [ 'external_db_id', 'external_db_id' ],
            external_synonym => [ 'xref_id', 'xref_id' ],
            gene => [ 'xref_id', 'display_xref_id' ],
            object_xref => [ 'xref_id', 'xref_id' ],
            transcript => [ 'xref_id', 'display_xref_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::ExternalDb; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::ExternalDb->table('external_db');

our @extras = ();
Gramene::CDBI::EnsemblRice::ExternalDb->columns( 
    All => qw/ 
        external_db_id
        db_name
        db_release
        status
        dbprimary_acc_linkable
        display_label_linkable
        priority
        db_display_name 
    /, @extras
);

Gramene::CDBI::EnsemblRice::ExternalDb->columns( 
    Essential => qw/ 
        external_db_id
        db_name
        db_release
        status
        dbprimary_acc_linkable
        display_label_linkable
        priority
        db_display_name 
    /
);

Gramene::CDBI::EnsemblRice::ExternalDb->columns( 
    Ordered => qw/ 
        external_db_id
        db_name
        db_release
        status
        dbprimary_acc_linkable
        display_label_linkable
        priority
        db_display_name 
    /
);

sub to_string {
    my $self = shift;
    return $self->db_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'external_db_id' => 'number',
        'db_name' => 'string',
        'db_release' => 'string',
        'status' => 'string',
        'dbprimary_acc_linkable' => 'string',
        'display_label_linkable' => 'string',
        'priority' => 'number',
        'db_display_name' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::ExternalDb->has_many( 
    'unmapped_objects',
    'Gramene::CDBI::EnsemblRice::UnmappedObject',
    'external_db_id' 
);

Gramene::CDBI::EnsemblRice::ExternalDb->has_many( 
    'xrefs',
    'Gramene::CDBI::EnsemblRice::Xref',
    'external_db_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            unmapped_object => [ 'external_db_id', 'external_db_id' ],
            xref => [ 'external_db_id', 'external_db_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::GeneArchive; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::GeneArchive->table('gene_archive');


sub mapping_session {
    my $self = shift;
    return $self->mapping_session_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::GeneArchive->columns( 
    All => qw/ 
        gene_stable_id
        gene_version
        transcript_stable_id
        transcript_version
        translation_stable_id
        translation_version
        peptide_archive_id
        mapping_session_id 
    /, @extras
);

Gramene::CDBI::EnsemblRice::GeneArchive->columns( 
    Essential => qw/ 
        gene_stable_id
        gene_version
        transcript_stable_id
        transcript_version
        translation_stable_id
        translation_version
        peptide_archive_id
        mapping_session_id 
    /
);

Gramene::CDBI::EnsemblRice::GeneArchive->columns( 
    Ordered => qw/ 
        gene_stable_id
        gene_version
        transcript_stable_id
        transcript_version
        translation_stable_id
        translation_version
        peptide_archive_id
        mapping_session_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->gene_version || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_stable_id' => 'string',
        'gene_version' => 'string',
        'transcript_stable_id' => 'string',
        'transcript_version' => 'string',
        'translation_stable_id' => 'string',
        'translation_version' => 'string',
        'peptide_archive_id' => 'number',
        'mapping_session_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::GeneArchive->has_a(
    mapping_session_id => 'Gramene::CDBI::EnsemblRice::MappingSession'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            mapping_session => [ 'mapping_session_id', 'mapping_session_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::GrameneGeneExtHelper; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::GrameneGeneExtHelper->table('gramene_gene_ext_helper');

our @extras = ();
Gramene::CDBI::EnsemblRice::GrameneGeneExtHelper->columns( 
    All => qw/ 
        gene_id
        stable_id
        external_name 
    /, @extras
);

Gramene::CDBI::EnsemblRice::GrameneGeneExtHelper->columns( 
    Essential => qw/ 
        gene_id
        stable_id
        external_name 
    /
);

Gramene::CDBI::EnsemblRice::GrameneGeneExtHelper->columns( 
    Ordered => qw/ 
        gene_id
        stable_id
        external_name 
    /
);

sub to_string {
    my $self = shift;
    return $self->external_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_id' => 'number',
        'stable_id' => 'string',
        'external_name' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::Interpro; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::Interpro->table('interpro');

our @extras = ();
push @extras, Class::DBI::Column->new( id => { 
    accessor => 'id_field',
    mutator  => 'id_field' 
}); 

Gramene::CDBI::EnsemblRice::Interpro->columns( 
    All => qw/ 
        interpro_id
        interpro_ac 
    /, @extras
);

Gramene::CDBI::EnsemblRice::Interpro->columns( 
    Essential => qw/ 
        interpro_id
        interpro_ac 
    /
);

Gramene::CDBI::EnsemblRice::Interpro->columns( 
    Ordered => qw/ 
        interpro_id
        interpro_ac 
    /
);

sub to_string {
    my $self = shift;
    return $self->interpro_ac || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'interpro_id' => 'number',
        'interpro_ac' => 'string',
        'id' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::MarkerMapLocation; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::MarkerMapLocation->table('marker_map_location');


sub map {
    my $self = shift;
    return $self->map_id;
}
    

sub marker {
    my $self = shift;
    return $self->marker_id;
}
    

sub marker_synonym {
    my $self = shift;
    return $self->marker_synonym_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::MarkerMapLocation->columns( 
    All => qw/ 
        marker_id
        map_id
        chromosome_name
        marker_synonym_id
        position
        lod_score 
    /, @extras
);

Gramene::CDBI::EnsemblRice::MarkerMapLocation->columns( 
    Essential => qw/ 
        marker_id
        map_id
        chromosome_name
        marker_synonym_id
        position
        lod_score 
    /
);

Gramene::CDBI::EnsemblRice::MarkerMapLocation->columns( 
    Ordered => qw/ 
        marker_id
        map_id
        chromosome_name
        marker_synonym_id
        position
        lod_score 
    /
);


Gramene::CDBI::EnsemblRice::MarkerMapLocation->columns( 
    Primary => qw/ 
        marker_id
        map_id 
    / 
);


sub to_string {
    my $self = shift;
    return $self->chromosome_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'map_id' => 'number',
        'chromosome_name' => 'string',
        'marker_synonym_id' => 'number',
        'position' => 'string',
        'lod_score' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::MarkerMapLocation->has_a(
    map_id => 'Gramene::CDBI::EnsemblRice::Map'
);

Gramene::CDBI::EnsemblRice::MarkerMapLocation->has_a(
    marker_id => 'Gramene::CDBI::EnsemblRice::Marker'
);

Gramene::CDBI::EnsemblRice::MarkerMapLocation->has_a(
    marker_synonym_id => 'Gramene::CDBI::EnsemblRice::MarkerSynonym'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            map => [ 'map_id', 'map_id' ],
            marker => [ 'marker_id', 'marker_id' ],
            marker_synonym => [ 'marker_synonym_id', 'marker_synonym_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::Map; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::Map->table('map');

our @extras = ();
Gramene::CDBI::EnsemblRice::Map->columns( 
    All => qw/ 
        map_id
        map_name 
    /, @extras
);

Gramene::CDBI::EnsemblRice::Map->columns( 
    Essential => qw/ 
        map_id
        map_name 
    /
);

Gramene::CDBI::EnsemblRice::Map->columns( 
    Ordered => qw/ 
        map_id
        map_name 
    /
);

sub to_string {
    my $self = shift;
    return $self->map_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'map_id' => 'number',
        'map_name' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::Map->has_many( 
    'marker_map_locations',
    'Gramene::CDBI::EnsemblRice::MarkerMapLocation',
    'map_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker_map_location => [ 'map_id', 'map_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::StableIdEvent; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::StableIdEvent->table('stable_id_event');


sub mapping_session {
    my $self = shift;
    return $self->mapping_session_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::StableIdEvent->columns( 
    All => qw/ 
        old_stable_id
        old_version
        new_stable_id
        new_version
        mapping_session_id
        type
        score 
    /, @extras
);

Gramene::CDBI::EnsemblRice::StableIdEvent->columns( 
    Essential => qw/ 
        old_stable_id
        old_version
        new_stable_id
        new_version
        mapping_session_id
        type
        score 
    /
);

Gramene::CDBI::EnsemblRice::StableIdEvent->columns( 
    Ordered => qw/ 
        old_stable_id
        old_version
        new_stable_id
        new_version
        mapping_session_id
        type
        score 
    /
);

sub to_string {
    my $self = shift;
    return $self->old_version || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'old_stable_id' => 'string',
        'old_version' => 'string',
        'new_stable_id' => 'string',
        'new_version' => 'string',
        'mapping_session_id' => 'number',
        'type' => 'string',
        'score' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::StableIdEvent->has_a(
    mapping_session_id => 'Gramene::CDBI::EnsemblRice::MappingSession'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            mapping_session => [ 'mapping_session_id', 'mapping_session_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::MappingSession; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::MappingSession->table('mapping_session');

our @extras = ();
Gramene::CDBI::EnsemblRice::MappingSession->columns( 
    All => qw/ 
        mapping_session_id
        old_db_name
        new_db_name
        old_release
        new_release
        old_assembly
        new_assembly
        created 
    /, @extras
);

Gramene::CDBI::EnsemblRice::MappingSession->columns( 
    Essential => qw/ 
        mapping_session_id
        old_db_name
        new_db_name
        old_release
        new_release
        old_assembly
        new_assembly
        created 
    /
);

Gramene::CDBI::EnsemblRice::MappingSession->columns( 
    Ordered => qw/ 
        mapping_session_id
        old_db_name
        new_db_name
        old_release
        new_release
        old_assembly
        new_assembly
        created 
    /
);

sub to_string {
    my $self = shift;
    return $self->old_db_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'mapping_session_id' => 'number',
        'old_db_name' => 'string',
        'new_db_name' => 'string',
        'old_release' => 'string',
        'new_release' => 'string',
        'old_assembly' => 'string',
        'new_assembly' => 'string',
        'created' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::MappingSession->has_many( 
    'gene_archives',
    'Gramene::CDBI::EnsemblRice::GeneArchive',
    'mapping_session_id' 
);

Gramene::CDBI::EnsemblRice::MappingSession->has_many( 
    'stable_id_events',
    'Gramene::CDBI::EnsemblRice::StableIdEvent',
    'mapping_session_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_archive => [ 'mapping_session_id', 'mapping_session_id' ],
            stable_id_event => [ 'mapping_session_id', 'mapping_session_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::MarkerSynonym; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::MarkerSynonym->table('marker_synonym');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::MarkerSynonym->columns( 
    All => qw/ 
        marker_synonym_id
        marker_id
        source
        name 
    /, @extras
);

Gramene::CDBI::EnsemblRice::MarkerSynonym->columns( 
    Essential => qw/ 
        marker_synonym_id
        marker_id
        source
        name 
    /
);

Gramene::CDBI::EnsemblRice::MarkerSynonym->columns( 
    Ordered => qw/ 
        marker_synonym_id
        marker_id
        source
        name 
    /
);

sub to_string {
    my $self = shift;
    return $self->source || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_synonym_id' => 'number',
        'marker_id' => 'number',
        'source' => 'string',
        'name' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::MarkerSynonym->has_a(
    marker_id => 'Gramene::CDBI::EnsemblRice::Marker'
);

Gramene::CDBI::EnsemblRice::MarkerSynonym->has_many( 
    'markers',
    'Gramene::CDBI::EnsemblRice::Marker',
    'display_marker_synonym_id' 
);

Gramene::CDBI::EnsemblRice::MarkerSynonym->has_many( 
    'marker_map_locations',
    'Gramene::CDBI::EnsemblRice::MarkerMapLocation',
    'marker_synonym_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
            marker_map_location => [ 'marker_synonym_id', 'marker_synonym_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::QtlSynonym; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::QtlSynonym->table('qtl_synonym');


sub qtl {
    my $self = shift;
    return $self->qtl_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::QtlSynonym->columns( 
    All => qw/ 
        qtl_synonym_id
        qtl_id
        source_database
        source_primary_id 
    /, @extras
);

Gramene::CDBI::EnsemblRice::QtlSynonym->columns( 
    Essential => qw/ 
        qtl_synonym_id
        qtl_id
        source_database
        source_primary_id 
    /
);

Gramene::CDBI::EnsemblRice::QtlSynonym->columns( 
    Ordered => qw/ 
        qtl_synonym_id
        qtl_id
        source_database
        source_primary_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->source_database || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'qtl_synonym_id' => 'number',
        'qtl_id' => 'number',
        'source_database' => 'string',
        'source_primary_id' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::QtlSynonym->has_a(
    qtl_id => 'Gramene::CDBI::EnsemblRice::Qtl'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            qtl => [ 'qtl_id', 'qtl_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::Qtl; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::Qtl->table('qtl');


sub peak_marker {
    my $self = shift;
    return $self->peak_marker_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::Qtl->columns( 
    All => qw/ 
        qtl_id
        trait
        lod_score
        flank_marker_id_1
        flank_marker_id_2
        peak_marker_id 
    /, @extras
);

Gramene::CDBI::EnsemblRice::Qtl->columns( 
    Essential => qw/ 
        qtl_id
        trait
        lod_score
        flank_marker_id_1
        flank_marker_id_2
        peak_marker_id 
    /
);

Gramene::CDBI::EnsemblRice::Qtl->columns( 
    Ordered => qw/ 
        qtl_id
        trait
        lod_score
        flank_marker_id_1
        flank_marker_id_2
        peak_marker_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->trait || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'qtl_id' => 'number',
        'trait' => 'string',
        'lod_score' => 'number',
        'flank_marker_id_1' => 'number',
        'flank_marker_id_2' => 'number',
        'peak_marker_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::Qtl->has_a(
    peak_marker_id => 'Gramene::CDBI::EnsemblRice::Marker'
);

Gramene::CDBI::EnsemblRice::Qtl->has_many( 
    'qtl_features',
    'Gramene::CDBI::EnsemblRice::QtlFeature',
    'qtl_id' 
);

Gramene::CDBI::EnsemblRice::Qtl->has_many( 
    'qtl_synonyms',
    'Gramene::CDBI::EnsemblRice::QtlSynonym',
    'qtl_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'peak_marker_id', 'marker_id' ],
            qtl_feature => [ 'qtl_id', 'qtl_id' ],
            qtl_synonym => [ 'qtl_id', 'qtl_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::Marker; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::Marker->table('marker');


sub display_marker_synonym {
    my $self = shift;
    return $self->display_marker_synonym_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::Marker->columns( 
    All => qw/ 
        marker_id
        display_marker_synonym_id
        left_primer
        right_primer
        min_primer_dist
        max_primer_dist
        priority
        type 
    /, @extras
);

Gramene::CDBI::EnsemblRice::Marker->columns( 
    Essential => qw/ 
        marker_id
        display_marker_synonym_id
        left_primer
        right_primer
        min_primer_dist
        max_primer_dist
        priority
        type 
    /
);

Gramene::CDBI::EnsemblRice::Marker->columns( 
    Ordered => qw/ 
        marker_id
        display_marker_synonym_id
        left_primer
        right_primer
        min_primer_dist
        max_primer_dist
        priority
        type 
    /
);

sub to_string {
    my $self = shift;
    return $self->left_primer || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'display_marker_synonym_id' => 'number',
        'left_primer' => 'string',
        'right_primer' => 'string',
        'min_primer_dist' => 'number',
        'max_primer_dist' => 'number',
        'priority' => 'number',
        'type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::Marker->has_a(
    display_marker_synonym_id => 'Gramene::CDBI::EnsemblRice::MarkerSynonym'
);

Gramene::CDBI::EnsemblRice::Marker->has_many( 
    'marker_features',
    'Gramene::CDBI::EnsemblRice::MarkerFeature',
    'marker_id' 
);

Gramene::CDBI::EnsemblRice::Marker->has_many( 
    'marker_map_locations',
    'Gramene::CDBI::EnsemblRice::MarkerMapLocation',
    'marker_id' 
);

Gramene::CDBI::EnsemblRice::Marker->has_many( 
    'marker_synonyms',
    'Gramene::CDBI::EnsemblRice::MarkerSynonym',
    'marker_id' 
);

Gramene::CDBI::EnsemblRice::Marker->has_many( 
    'qtls_flank_marker_1',
    'Gramene::CDBI::EnsemblRice::Qtl',
    'flank_marker_id_1' 
);

Gramene::CDBI::EnsemblRice::Marker->has_many( 
    'qtls_flank_marker_2',
    'Gramene::CDBI::EnsemblRice::Qtl',
    'flank_marker_id_2' 
);

Gramene::CDBI::EnsemblRice::Marker->has_many( 
    'qtls_peak_marker',
    'Gramene::CDBI::EnsemblRice::Qtl',
    'peak_marker_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker_feature => [ 'marker_id', 'marker_id' ],
            marker_map_location => [ 'marker_id', 'marker_id' ],
            marker_synonym => [ 'marker_id', 'marker_id' ],
            qtl => [ 'marker_id', 'peak_marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::Meta; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::Meta->table('meta');

our @extras = ();
Gramene::CDBI::EnsemblRice::Meta->columns( 
    All => qw/ 
        meta_id
        meta_key
        meta_value 
    /, @extras
);

Gramene::CDBI::EnsemblRice::Meta->columns( 
    Essential => qw/ 
        meta_id
        meta_key
        meta_value 
    /
);

Gramene::CDBI::EnsemblRice::Meta->columns( 
    Ordered => qw/ 
        meta_id
        meta_key
        meta_value 
    /
);

sub to_string {
    my $self = shift;
    return $self->meta_key || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'meta_id' => 'number',
        'meta_key' => 'string',
        'meta_value' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::MiscAttribHelper; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::MiscAttribHelper->table('misc_attrib_helper');

our @extras = ();
Gramene::CDBI::EnsemblRice::MiscAttribHelper->columns( 
    All => qw/ 
        name
        attrib_name
        attrib_value 
    /, @extras
);

Gramene::CDBI::EnsemblRice::MiscAttribHelper->columns( 
    Essential => qw/ 
        name
        attrib_name
        attrib_value 
    /
);

Gramene::CDBI::EnsemblRice::MiscAttribHelper->columns( 
    Ordered => qw/ 
        name
        attrib_name
        attrib_value 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'name' => 'string',
        'attrib_name' => 'string',
        'attrib_value' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::MiscSet; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::MiscSet->table('misc_set');

our @extras = ();
Gramene::CDBI::EnsemblRice::MiscSet->columns( 
    All => qw/ 
        misc_set_id
        code
        name
        description
        max_length 
    /, @extras
);

Gramene::CDBI::EnsemblRice::MiscSet->columns( 
    Essential => qw/ 
        misc_set_id
        code
        name
        description
        max_length 
    /
);

Gramene::CDBI::EnsemblRice::MiscSet->columns( 
    Ordered => qw/ 
        misc_set_id
        code
        name
        description
        max_length 
    /
);

sub to_string {
    my $self = shift;
    return $self->code || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'misc_set_id' => 'string',
        'code' => 'string',
        'name' => 'string',
        'description' => 'string',
        'max_length' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::MiscSet->has_many( 
    'misc_feature_misc_sets',
    'Gramene::CDBI::EnsemblRice::MiscFeatureMiscSet',
    'misc_set_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            misc_feature_misc_set => [ 'misc_set_id', 'misc_set_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::OligoProbe; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::OligoProbe->table('oligo_probe');


sub oligo_array {
    my $self = shift;
    return $self->oligo_array_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::OligoProbe->columns( 
    All => qw/ 
        oligo_probe_id
        oligo_array_id
        probeset
        name
        description
        length 
    /, @extras
);

Gramene::CDBI::EnsemblRice::OligoProbe->columns( 
    Essential => qw/ 
        oligo_probe_id
        oligo_array_id
        probeset
        name
        description
        length 
    /
);

Gramene::CDBI::EnsemblRice::OligoProbe->columns( 
    Ordered => qw/ 
        oligo_probe_id
        oligo_array_id
        probeset
        name
        description
        length 
    /
);


Gramene::CDBI::EnsemblRice::OligoProbe->columns( 
    Primary => qw/ 
        oligo_probe_id
        oligo_array_id 
    / 
);


sub to_string {
    my $self = shift;
    return $self->probeset || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'oligo_probe_id' => 'number',
        'oligo_array_id' => 'number',
        'probeset' => 'string',
        'name' => 'string',
        'description' => 'string',
        'length' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::OligoProbe->has_a(
    oligo_array_id => 'Gramene::CDBI::EnsemblRice::OligoArray'
);

Gramene::CDBI::EnsemblRice::OligoProbe->has_many( 
    'oligo_features',
    'Gramene::CDBI::EnsemblRice::OligoFeature',
    'oligo_probe_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            oligo_array => [ 'oligo_array_id', 'oligo_array_id' ],
            oligo_feature => [ 'oligo_probe_id', 'oligo_probe_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::OligoArray; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::OligoArray->table('oligo_array');


sub parent_array {
    my $self = shift;
    return $self->parent_array_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblRice::OligoArray->columns( 
    All => qw/ 
        oligo_array_id
        parent_array_id
        probe_setsize
        name
        type 
    /, @extras
);

Gramene::CDBI::EnsemblRice::OligoArray->columns( 
    Essential => qw/ 
        oligo_array_id
        parent_array_id
        probe_setsize
        name
        type 
    /
);

Gramene::CDBI::EnsemblRice::OligoArray->columns( 
    Ordered => qw/ 
        oligo_array_id
        parent_array_id
        probe_setsize
        name
        type 
    /
);

sub to_string {
    my $self = shift;
    return $self->probe_setsize || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'oligo_array_id' => 'number',
        'parent_array_id' => 'number',
        'probe_setsize' => 'string',
        'name' => 'string',
        'type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::OligoArray->has_a(
    parent_array_id => 'Gramene::CDBI::EnsemblRice::OligoArray'
);

Gramene::CDBI::EnsemblRice::OligoArray->has_many( 
    'oligo_arrays',
    'Gramene::CDBI::EnsemblRice::OligoArray',
    'parent_array_id' 
);

Gramene::CDBI::EnsemblRice::OligoArray->has_many( 
    'oligo_probes',
    'Gramene::CDBI::EnsemblRice::OligoProbe',
    'oligo_array_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            oligo_array => [ 'parent_array_id', 'oligo_array_id' ],
            oligo_probe => [ 'oligo_array_id', 'oligo_array_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::PeptideArchive; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::PeptideArchive->table('peptide_archive');

our @extras = ();
Gramene::CDBI::EnsemblRice::PeptideArchive->columns( 
    All => qw/ 
        peptide_archive_id
        md5_checksum
        peptide_seq 
    /, @extras
);

Gramene::CDBI::EnsemblRice::PeptideArchive->columns( 
    Essential => qw/ 
        peptide_archive_id
        md5_checksum
        peptide_seq 
    /
);

Gramene::CDBI::EnsemblRice::PeptideArchive->columns( 
    Ordered => qw/ 
        peptide_archive_id
        md5_checksum
        peptide_seq 
    /
);

sub to_string {
    my $self = shift;
    return $self->md5_checksum || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'peptide_archive_id' => 'number',
        'md5_checksum' => 'string',
        'peptide_seq' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::RegulatoryFactor; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::RegulatoryFactor->table('regulatory_factor');

our @extras = ();
Gramene::CDBI::EnsemblRice::RegulatoryFactor->columns( 
    All => qw/ 
        regulatory_factor_id
        name
        type 
    /, @extras
);

Gramene::CDBI::EnsemblRice::RegulatoryFactor->columns( 
    Essential => qw/ 
        regulatory_factor_id
        name
        type 
    /
);

Gramene::CDBI::EnsemblRice::RegulatoryFactor->columns( 
    Ordered => qw/ 
        regulatory_factor_id
        name
        type 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'regulatory_factor_id' => 'number',
        'name' => 'string',
        'type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::RegulatoryFactor->has_many( 
    'regulatory_factor_codings',
    'Gramene::CDBI::EnsemblRice::RegulatoryFactorCoding',
    'regulatory_factor_id' 
);

Gramene::CDBI::EnsemblRice::RegulatoryFactor->has_many( 
    'regulatory_features',
    'Gramene::CDBI::EnsemblRice::RegulatoryFeature',
    'regulatory_factor_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            regulatory_factor_coding => [ 'regulatory_factor_id', 'regulatory_factor_id' ],
            regulatory_feature => [ 'regulatory_factor_id', 'regulatory_factor_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::RegulatoryFeatureObject; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::RegulatoryFeatureObject->table('regulatory_feature_object');

our @extras = ();
Gramene::CDBI::EnsemblRice::RegulatoryFeatureObject->columns( 
    All => qw/ 
        regulatory_feature_id
        ensembl_object_type
        ensembl_object_id
        influence
        evidence 
    /, @extras
);

Gramene::CDBI::EnsemblRice::RegulatoryFeatureObject->columns( 
    Essential => qw/ 
        regulatory_feature_id
        ensembl_object_type
        ensembl_object_id
        influence
        evidence 
    /
);

Gramene::CDBI::EnsemblRice::RegulatoryFeatureObject->columns( 
    Ordered => qw/ 
        regulatory_feature_id
        ensembl_object_type
        ensembl_object_id
        influence
        evidence 
    /
);

sub to_string {
    my $self = shift;
    return $self->ensembl_object_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'regulatory_feature_id' => 'number',
        'ensembl_object_type' => 'string',
        'ensembl_object_id' => 'number',
        'influence' => 'string',
        'evidence' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::RepeatConsensus; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::RepeatConsensus->table('repeat_consensus');

our @extras = ();
Gramene::CDBI::EnsemblRice::RepeatConsensus->columns( 
    All => qw/ 
        repeat_consensus_id
        repeat_name
        repeat_class
        repeat_type
        repeat_consensus 
    /, @extras
);

Gramene::CDBI::EnsemblRice::RepeatConsensus->columns( 
    Essential => qw/ 
        repeat_consensus_id
        repeat_name
        repeat_class
        repeat_type
        repeat_consensus 
    /
);

Gramene::CDBI::EnsemblRice::RepeatConsensus->columns( 
    Ordered => qw/ 
        repeat_consensus_id
        repeat_name
        repeat_class
        repeat_type
        repeat_consensus 
    /
);

sub to_string {
    my $self = shift;
    return $self->repeat_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'repeat_consensus_id' => 'number',
        'repeat_name' => 'string',
        'repeat_class' => 'string',
        'repeat_type' => 'string',
        'repeat_consensus' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::RepeatConsensus->has_many( 
    'repeat_features',
    'Gramene::CDBI::EnsemblRice::RepeatFeature',
    'repeat_consensus_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            repeat_feature => [ 'repeat_consensus_id', 'repeat_consensus_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::SeqtableDaf0; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::SeqtableDaf0->table('seqtable_daf0');

our @extras = ();
Gramene::CDBI::EnsemblRice::SeqtableDaf0->columns( 
    All => qw/ 
        embl_acc
        daf_chr_start
        daf_chr_end
        chr_strand
        chromosome
        superctg_end
        superctg_name
        superctg_ori
        superctg_start
        daf_clone_start
        daf_clone_end
        daf_super_start
        daf_super_end 
    /, @extras
);

Gramene::CDBI::EnsemblRice::SeqtableDaf0->columns( 
    Essential => qw/ 
        embl_acc
        daf_chr_start
        daf_chr_end
        chr_strand
        chromosome
        superctg_end
        superctg_name
        superctg_ori
        superctg_start
        daf_clone_start
        daf_clone_end
        daf_super_start
        daf_super_end 
    /
);

Gramene::CDBI::EnsemblRice::SeqtableDaf0->columns( 
    Ordered => qw/ 
        embl_acc
        daf_chr_start
        daf_chr_end
        chr_strand
        chromosome
        superctg_end
        superctg_name
        superctg_ori
        superctg_start
        daf_clone_start
        daf_clone_end
        daf_super_start
        daf_super_end 
    /
);

sub to_string {
    my $self = shift;
    return $self->embl_acc || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'embl_acc' => 'string',
        'daf_chr_start' => 'number',
        'daf_chr_end' => 'number',
        'chr_strand' => 'string',
        'chromosome' => 'string',
        'superctg_end' => 'string',
        'superctg_name' => 'string',
        'superctg_ori' => 'string',
        'superctg_start' => 'string',
        'daf_clone_start' => 'string',
        'daf_clone_end' => 'string',
        'daf_super_start' => 'string',
        'daf_super_end' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::SeqtableDaf1; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::SeqtableDaf1->table('seqtable_daf1');

our @extras = ();
Gramene::CDBI::EnsemblRice::SeqtableDaf1->columns( 
    All => qw/ 
        embl_acc
        chromosome
        chr_start
        chr_end
        superctg_name
        super_start
        super_end 
    /, @extras
);

Gramene::CDBI::EnsemblRice::SeqtableDaf1->columns( 
    Essential => qw/ 
        embl_acc
        chromosome
        chr_start
        chr_end
        superctg_name
        super_start
        super_end 
    /
);

Gramene::CDBI::EnsemblRice::SeqtableDaf1->columns( 
    Ordered => qw/ 
        embl_acc
        chromosome
        chr_start
        chr_end
        superctg_name
        super_start
        super_end 
    /
);

sub to_string {
    my $self = shift;
    return $self->embl_acc || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'embl_acc' => 'string',
        'chromosome' => 'string',
        'chr_start' => 'string',
        'chr_end' => 'string',
        'superctg_name' => 'string',
        'super_start' => 'string',
        'super_end' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::SeqtableHelper; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::SeqtableHelper->table('seqtable_helper');

our @extras = ();
Gramene::CDBI::EnsemblRice::SeqtableHelper->columns( 
    All => qw/ 
        clone_id
        bacname
        chromosome
        embl_acc
        embl_version
        htg_phase
        super_name
        super_chr_start
        super_chr_end
        clone_super_start
        clone_super_end
        clone_chr_start
        clone_chr_end
        marker_name
        marker_type
        jrgp_rflp_cm
        jrgp_rflp_map
        jrgp_rflp_feature
        cornell_rflp_cm
        cornell_rflp_map
        cornell_rflp_feature
        irmi_ssr_cm
        irmi_ssr_map
        irmi_ssr_feature 
    /, @extras
);

Gramene::CDBI::EnsemblRice::SeqtableHelper->columns( 
    Essential => qw/ 
        clone_id
        bacname
        chromosome
        embl_acc
        embl_version
        htg_phase
        super_name
        super_chr_start
        super_chr_end
        clone_super_start
        clone_super_end
        clone_chr_start
        clone_chr_end
        marker_name
        marker_type
        jrgp_rflp_cm
        jrgp_rflp_map
        jrgp_rflp_feature
        cornell_rflp_cm
        cornell_rflp_map
        cornell_rflp_feature
        irmi_ssr_cm
        irmi_ssr_map
        irmi_ssr_feature 
    /
);

Gramene::CDBI::EnsemblRice::SeqtableHelper->columns( 
    Ordered => qw/ 
        clone_id
        bacname
        chromosome
        embl_acc
        embl_version
        htg_phase
        super_name
        super_chr_start
        super_chr_end
        clone_super_start
        clone_super_end
        clone_chr_start
        clone_chr_end
        marker_name
        marker_type
        jrgp_rflp_cm
        jrgp_rflp_map
        jrgp_rflp_feature
        cornell_rflp_cm
        cornell_rflp_map
        cornell_rflp_feature
        irmi_ssr_cm
        irmi_ssr_map
        irmi_ssr_feature 
    /
);

sub to_string {
    my $self = shift;
    return $self->bacname || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'clone_id' => 'number',
        'bacname' => 'string',
        'chromosome' => 'string',
        'embl_acc' => 'string',
        'embl_version' => 'number',
        'htg_phase' => 'number',
        'super_name' => 'string',
        'super_chr_start' => 'number',
        'super_chr_end' => 'number',
        'clone_super_start' => 'number',
        'clone_super_end' => 'number',
        'clone_chr_start' => 'number',
        'clone_chr_end' => 'number',
        'marker_name' => 'string',
        'marker_type' => 'string',
        'jrgp_rflp_cm' => 'number',
        'jrgp_rflp_map' => 'string',
        'jrgp_rflp_feature' => 'string',
        'cornell_rflp_cm' => 'number',
        'cornell_rflp_map' => 'string',
        'cornell_rflp_feature' => 'string',
        'irmi_ssr_cm' => 'number',
        'irmi_ssr_map' => 'string',
        'irmi_ssr_feature' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::SeqtablePrehelper; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::SeqtablePrehelper->table('seqtable_prehelper');

our @extras = ();
Gramene::CDBI::EnsemblRice::SeqtablePrehelper->columns( 
    All => qw/ 
        clone_id
        chr_id
        embl_acc
        embl_version
        htg_phase
        chromosome
        super_name
        super_chr_start
        super_chr_end
        clone_super_start
        clone_super_end
        clone_chr_start
        clone_chr_end
        contig_clone_start
        contig_clone_end
        bacname
        bac_length
        source 
    /, @extras
);

Gramene::CDBI::EnsemblRice::SeqtablePrehelper->columns( 
    Essential => qw/ 
        clone_id
        chr_id
        embl_acc
        embl_version
        htg_phase
        chromosome
        super_name
        super_chr_start
        super_chr_end
        clone_super_start
        clone_super_end
        clone_chr_start
        clone_chr_end
        contig_clone_start
        contig_clone_end
        bacname
        bac_length
        source 
    /
);

Gramene::CDBI::EnsemblRice::SeqtablePrehelper->columns( 
    Ordered => qw/ 
        clone_id
        chr_id
        embl_acc
        embl_version
        htg_phase
        chromosome
        super_name
        super_chr_start
        super_chr_end
        clone_super_start
        clone_super_end
        clone_chr_start
        clone_chr_end
        contig_clone_start
        contig_clone_end
        bacname
        bac_length
        source 
    /
);

sub to_string {
    my $self = shift;
    return $self->embl_acc || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'clone_id' => 'number',
        'chr_id' => 'number',
        'embl_acc' => 'string',
        'embl_version' => 'number',
        'htg_phase' => 'number',
        'chromosome' => 'string',
        'super_name' => 'string',
        'super_chr_start' => 'number',
        'super_chr_end' => 'number',
        'clone_super_start' => 'number',
        'clone_super_end' => 'number',
        'clone_chr_start' => 'number',
        'clone_chr_end' => 'number',
        'contig_clone_start' => 'number',
        'contig_clone_end' => 'number',
        'bacname' => 'string',
        'bac_length' => 'number',
        'source' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::UnconventionalTranscriptAssociation; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::UnconventionalTranscriptAssociation->table('unconventional_transcript_association');

our @extras = ();
Gramene::CDBI::EnsemblRice::UnconventionalTranscriptAssociation->columns( 
    All => qw/ 
        transcript_id
        gene_id
        interaction_type 
    /, @extras
);

Gramene::CDBI::EnsemblRice::UnconventionalTranscriptAssociation->columns( 
    Essential => qw/ 
        transcript_id
        gene_id
        interaction_type 
    /
);

Gramene::CDBI::EnsemblRice::UnconventionalTranscriptAssociation->columns( 
    Ordered => qw/ 
        transcript_id
        gene_id
        interaction_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->interaction_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'transcript_id' => 'number',
        'gene_id' => 'number',
        'interaction_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblRice::UnmappedReason; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblRice ); };
# use base qw( Gramene::CDBI::EnsemblRice );

Gramene::CDBI::EnsemblRice::UnmappedReason->table('unmapped_reason');

our @extras = ();
Gramene::CDBI::EnsemblRice::UnmappedReason->columns( 
    All => qw/ 
        unmapped_reason_id
        summary_description
        full_description 
    /, @extras
);

Gramene::CDBI::EnsemblRice::UnmappedReason->columns( 
    Essential => qw/ 
        unmapped_reason_id
        summary_description
        full_description 
    /
);

Gramene::CDBI::EnsemblRice::UnmappedReason->columns( 
    Ordered => qw/ 
        unmapped_reason_id
        summary_description
        full_description 
    /
);

sub to_string {
    my $self = shift;
    return $self->summary_description || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'unmapped_reason_id' => 'string',
        'summary_description' => 'string',
        'full_description' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblRice::UnmappedReason->has_many( 
    'unmapped_objects',
    'Gramene::CDBI::EnsemblRice::UnmappedObject',
    'unmapped_reason_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            unmapped_object => [ 'unmapped_reason_id', 'unmapped_reason_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope



1;

__END__

# ----------------------------------------------------

=pod

=head1 NAME

Gramene::CDBI::EnsemblRice - Gramene::CDBI-base classes for EnsemblRice

=head1 VERSION

This documentation refers to version $Revision: 1.5 $.

=head1 SYNOPSIS

  use Gramene::CDBI::EnsemblRice;

  my @data   = Gramene::CDBI::EnsemblRice::RegulatoryFactor->retrieve_all;
  my $object = Gramene::CDBI::EnsemblRice::RegulatoryFactor->retrieve( 42 );

=head1 DESCRIPTION

This module contains Gramene::CDBI-base classes for the EnsemblRice 
schema.

=head1 METHODS 

=head2 config_name

  my $config_name = $self->config_name;

Returns the string 'ensembl_rice'.

=head2 represented_tables

  my @tables = $cdbi->represented_tables;

Returns the names of the tables in the module.  See the CLASSES below.

=cut

=head1 CLASSES

=head2 Gramene::CDBI::EnsemblRice::AltAllele

Corresponds to the table "alt_allele."  

Stringifies (C<to_name>) to "alt_allele_id".

Fields:

=over 4

=item * alt_allele_id

=item * gene_id

=back


Has A:

=over 4

=item * gene => Gramene::CDBI::EnsemblRice::Gene

=back



=head2 Gramene::CDBI::EnsemblRice::Analysis

Corresponds to the table "analysis."  

Stringifies (C<to_name>) to "created".

Fields:

=over 4

=item * analysis_id

=item * created

=item * logic_name

=item * db

=item * db_version

=item * db_file

=item * program

=item * program_version

=item * program_file

=item * parameters

=item * module

=item * module_version

=item * gff_source

=item * gff_feature

=back



Has Many:

=over 4

=item * analysis_descriptions => Gramene::CDBI::EnsemblRice::AnalysisDescription

=item * density_types => Gramene::CDBI::EnsemblRice::DensityType

=item * ditag_features => Gramene::CDBI::EnsemblRice::DitagFeature

=item * dna_align_features => Gramene::CDBI::EnsemblRice::DnaAlignFeature

=item * genes => Gramene::CDBI::EnsemblRice::Gene

=item * identity_xrefs => Gramene::CDBI::EnsemblRice::IdentityXref

=item * marker_features => Gramene::CDBI::EnsemblRice::MarkerFeature

=item * oligo_features => Gramene::CDBI::EnsemblRice::OligoFeature

=item * prediction_transcripts => Gramene::CDBI::EnsemblRice::PredictionTranscript

=item * protein_align_features => Gramene::CDBI::EnsemblRice::ProteinAlignFeature

=item * protein_features => Gramene::CDBI::EnsemblRice::ProteinFeature

=item * qtl_features => Gramene::CDBI::EnsemblRice::QtlFeature

=item * regulatory_features => Gramene::CDBI::EnsemblRice::RegulatoryFeature

=item * regulatory_search_regions => Gramene::CDBI::EnsemblRice::RegulatorySearchRegion

=item * repeat_features => Gramene::CDBI::EnsemblRice::RepeatFeature

=item * simple_features => Gramene::CDBI::EnsemblRice::SimpleFeature

=item * transcripts => Gramene::CDBI::EnsemblRice::Transcript

=item * unmapped_objects => Gramene::CDBI::EnsemblRice::UnmappedObject

=back


=head2 Gramene::CDBI::EnsemblRice::AnalysisDescription

Corresponds to the table "analysis_description."  

Stringifies (C<to_name>) to "description".

Fields:

=over 4

=item * analysis_id

=item * description

=item * display_label

=item * displayable

=item * web_data

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblRice::Analysis

=back



=head2 Gramene::CDBI::EnsemblRice::Assembly

Corresponds to the table "assembly."  

Stringifies (C<to_name>) to "asm_start".

Fields:

=over 4

=item * asm_seq_region_id

=item * cmp_seq_region_id

=item * asm_start

=item * asm_end

=item * cmp_start

=item * cmp_end

=item * ori

=back


Has A:

=over 4

=item * asm_seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=item * cmp_seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblRice::AssemblyException

Corresponds to the table "assembly_exception."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * assembly_exception_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * exc_type

=item * exc_seq_region_id

=item * exc_seq_region_start

=item * exc_seq_region_end

=item * ori

=back


Has A:

=over 4

=item * seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=item * exc_seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblRice::AttribType

Corresponds to the table "attrib_type."  

Stringifies (C<to_name>) to "code".

Fields:

=over 4

=item * attrib_type_id

=item * code

=item * name

=item * description

=back



Has Many:

=over 4

=item * gene_attribs => Gramene::CDBI::EnsemblRice::GeneAttrib

=item * misc_attribs => Gramene::CDBI::EnsemblRice::MiscAttrib

=item * seq_region_attribs => Gramene::CDBI::EnsemblRice::SeqRegionAttrib

=item * transcript_attribs => Gramene::CDBI::EnsemblRice::TranscriptAttrib

=item * translation_attribs => Gramene::CDBI::EnsemblRice::TranslationAttrib

=back


=head2 Gramene::CDBI::EnsemblRice::Bacpac

Corresponds to the table "bacpac."  

Stringifies (C<to_name>) to "embl_version".

Fields:

=over 4

=item * embl_acc

=item * embl_version

=item * htg_phase

=item * bacname

=item * site

=item * gi

=item * chrnum

=item * length

=item * seq_region_id

=back




=head2 Gramene::CDBI::EnsemblRice::CoordSystem

Corresponds to the table "coord_system."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * coord_system_id

=item * name

=item * version

=item * rank

=item * attrib

=back



Has Many:

=over 4

=item * meta_coords => Gramene::CDBI::EnsemblRice::MetaCoord

=item * seq_regions => Gramene::CDBI::EnsemblRice::SeqRegion

=back


=head2 Gramene::CDBI::EnsemblRice::DensityFeature

Corresponds to the table "density_feature."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * density_feature_id

=item * density_type_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * density_value

=back


Has A:

=over 4

=item * density_type => Gramene::CDBI::EnsemblRice::DensityType

=item * seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblRice::DensityType

Corresponds to the table "density_type."  

Stringifies (C<to_name>) to "block_size".

Fields:

=over 4

=item * density_type_id

=item * analysis_id

=item * block_size

=item * region_features

=item * value_type

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblRice::Analysis

=back


Has Many:

=over 4

=item * density_features => Gramene::CDBI::EnsemblRice::DensityFeature

=back


=head2 Gramene::CDBI::EnsemblRice::Ditag

Corresponds to the table "ditag."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * ditag_id

=item * name

=item * type

=item * tag_count

=back



Has Many:

=over 4

=item * ditag_features => Gramene::CDBI::EnsemblRice::DitagFeature

=back


=head2 Gramene::CDBI::EnsemblRice::DitagFeature

Corresponds to the table "ditag_feature."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * ditag_feature_id

=item * ditag_id

=item * ditag_pair_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * analysis_id

=item * hit_start

=item * hit_end

=item * hit_strand

=item * cigar_line

=item * ditag_side

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblRice::Analysis

=item * ditag => Gramene::CDBI::EnsemblRice::Ditag

=item * seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblRice::Dna

Corresponds to the table "dna."  

Stringifies (C<to_name>) to "sequence".

Fields:

=over 4

=item * seq_region_id

=back


Has A:

=over 4

=item * seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblRice::DnaAlignFeature

Corresponds to the table "dna_align_feature."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * dna_align_feature_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * hit_start

=item * hit_end

=item * hit_strand

=item * hit_name

=item * analysis_id

=item * score

=item * evalue

=item * perc_ident

=item * cigar_line

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblRice::Analysis

=item * seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblRice::Dnac

Corresponds to the table "dnac."  

Stringifies (C<to_name>) to "sequence".

Fields:

=over 4

=item * seq_region_id

=item * n_line

=back


Has A:

=over 4

=item * seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblRice::EnsemblDafChr

Corresponds to the table "ensembl_daf_chr."  

Stringifies (C<to_name>) to "chr_start".

Fields:

=over 4

=item * dna_align_feature_id

=item * chromosome_id

=item * chr_start

=item * chr_end

=item * chr_strand

=item * hit_start

=item * hit_end

=item * hit_name

=item * analysis_id

=back




=head2 Gramene::CDBI::EnsemblRice::EnsemblDafGroup

Corresponds to the table "ensembl_daf_group."  

Stringifies (C<to_name>) to "chr_start".

Fields:

=over 4

=item * daf_group_id

=item * dna_align_feature_id

=item * chr_start

=item * chr_end

=item * chr_strand

=back




=head2 Gramene::CDBI::EnsemblRice::EnsemblGeneHelper

Corresponds to the table "ensembl_gene_helper."  

Stringifies (C<to_name>) to "description".

Fields:

=over 4

=item * gene_id

=item * description

=back




=head2 Gramene::CDBI::EnsemblRice::Exon

Corresponds to the table "exon."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * exon_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * phase

=item * end_phase

=item * is_current

=back


Has A:

=over 4

=item * seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=back


Has Many:

=over 4

=item * exon_stable_ids => Gramene::CDBI::EnsemblRice::ExonStableId

=item * exon_transcripts => Gramene::CDBI::EnsemblRice::ExonTranscript

=item * supporting_features => Gramene::CDBI::EnsemblRice::SupportingFeature

=item * translations_start_exon => Gramene::CDBI::EnsemblRice::Translation

=item * translations_end_exon => Gramene::CDBI::EnsemblRice::Translation

=back


=head2 Gramene::CDBI::EnsemblRice::ExonStableId

Corresponds to the table "exon_stable_id."  

Stringifies (C<to_name>) to "version".

Fields:

=over 4

=item * exon_id

=item * stable_id

=item * version

=item * created_date

=item * modified_date

=back


Has A:

=over 4

=item * exon => Gramene::CDBI::EnsemblRice::Exon

=back



=head2 Gramene::CDBI::EnsemblRice::ExonTranscript

Corresponds to the table "exon_transcript."  

Stringifies (C<to_name>) to "exon_id".

Fields:

=over 4

=item * exon_id

=item * transcript_id

=item * rank

=back


Has A:

=over 4

=item * exon => Gramene::CDBI::EnsemblRice::Exon

=item * transcript => Gramene::CDBI::EnsemblRice::Transcript

=back



=head2 Gramene::CDBI::EnsemblRice::ExternalDb

Corresponds to the table "external_db."  

Stringifies (C<to_name>) to "db_name".

Fields:

=over 4

=item * external_db_id

=item * db_name

=item * db_release

=item * status

=item * dbprimary_acc_linkable

=item * display_label_linkable

=item * priority

=item * db_display_name

=back



Has Many:

=over 4

=item * unmapped_objects => Gramene::CDBI::EnsemblRice::UnmappedObject

=item * xrefs => Gramene::CDBI::EnsemblRice::Xref

=back


=head2 Gramene::CDBI::EnsemblRice::ExternalSynonym

Corresponds to the table "external_synonym."  

Stringifies (C<to_name>) to "xref_id".

Fields:

=over 4

=item * xref_id

=item * synonym

=back


Has A:

=over 4

=item * xref => Gramene::CDBI::EnsemblRice::Xref

=back



=head2 Gramene::CDBI::EnsemblRice::Gene

Corresponds to the table "gene."  

Stringifies (C<to_name>) to "biotype".

Fields:

=over 4

=item * gene_id

=item * biotype

=item * analysis_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * display_xref_id

=item * source

=item * status

=item * description

=item * is_current

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblRice::Analysis

=item * seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=item * display_xref => Gramene::CDBI::EnsemblRice::Xref

=back


Has Many:

=over 4

=item * alt_alleles => Gramene::CDBI::EnsemblRice::AltAllele

=item * gene_attribs => Gramene::CDBI::EnsemblRice::GeneAttrib

=item * gene_stable_ids => Gramene::CDBI::EnsemblRice::GeneStableId

=item * regulatory_factor_codings => Gramene::CDBI::EnsemblRice::RegulatoryFactorCoding

=item * transcripts => Gramene::CDBI::EnsemblRice::Transcript

=back


=head2 Gramene::CDBI::EnsemblRice::GeneArchive

Corresponds to the table "gene_archive."  

Stringifies (C<to_name>) to "gene_version".

Fields:

=over 4

=item * gene_stable_id

=item * gene_version

=item * transcript_stable_id

=item * transcript_version

=item * translation_stable_id

=item * translation_version

=item * peptide_archive_id

=item * mapping_session_id

=back


Has A:

=over 4

=item * mapping_session => Gramene::CDBI::EnsemblRice::MappingSession

=back



=head2 Gramene::CDBI::EnsemblRice::GeneAttrib

Corresponds to the table "gene_attrib."  

Stringifies (C<to_name>) to "value".

Fields:

=over 4

=item * gene_id

=item * attrib_type_id

=item * value

=back


Has A:

=over 4

=item * attrib_type => Gramene::CDBI::EnsemblRice::AttribType

=item * gene => Gramene::CDBI::EnsemblRice::Gene

=back



=head2 Gramene::CDBI::EnsemblRice::GeneStableId

Corresponds to the table "gene_stable_id."  

Stringifies (C<to_name>) to "version".

Fields:

=over 4

=item * gene_id

=item * stable_id

=item * version

=item * created_date

=item * modified_date

=back


Has A:

=over 4

=item * gene => Gramene::CDBI::EnsemblRice::Gene

=back



=head2 Gramene::CDBI::EnsemblRice::GoXref

Corresponds to the table "go_xref."  

Stringifies (C<to_name>) to "linkage_type".

Fields:

=over 4

=item * object_xref_id

=item * linkage_type

=back


Has A:

=over 4

=item * object_xref => Gramene::CDBI::EnsemblRice::ObjectXref

=back



=head2 Gramene::CDBI::EnsemblRice::GrameneGeneExtHelper

Corresponds to the table "gramene_gene_ext_helper."  

Stringifies (C<to_name>) to "external_name".

Fields:

=over 4

=item * gene_id

=item * stable_id

=item * external_name

=back




=head2 Gramene::CDBI::EnsemblRice::IdentityXref

Corresponds to the table "identity_xref."  

Stringifies (C<to_name>) to "query_identity".

Fields:

=over 4

=item * object_xref_id

=item * query_identity

=item * target_identity

=item * hit_start

=item * hit_end

=item * translation_start

=item * translation_end

=item * cigar_line

=item * score

=item * evalue

=item * analysis_id

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblRice::Analysis

=item * object_xref => Gramene::CDBI::EnsemblRice::ObjectXref

=back



=head2 Gramene::CDBI::EnsemblRice::Interpro

Corresponds to the table "interpro."  

Stringifies (C<to_name>) to "interpro_ac".

Fields:

=over 4

=item * interpro_id

=item * interpro_ac

=back




=head2 Gramene::CDBI::EnsemblRice::Karyotype

Corresponds to the table "karyotype."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * karyotype_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * band

=item * stain

=back


Has A:

=over 4

=item * seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblRice::Map

Corresponds to the table "map."  

Stringifies (C<to_name>) to "map_name".

Fields:

=over 4

=item * map_id

=item * map_name

=back



Has Many:

=over 4

=item * marker_map_locations => Gramene::CDBI::EnsemblRice::MarkerMapLocation

=back


=head2 Gramene::CDBI::EnsemblRice::MappingSession

Corresponds to the table "mapping_session."  

Stringifies (C<to_name>) to "old_db_name".

Fields:

=over 4

=item * mapping_session_id

=item * old_db_name

=item * new_db_name

=item * old_release

=item * new_release

=item * old_assembly

=item * new_assembly

=item * created

=back



Has Many:

=over 4

=item * gene_archives => Gramene::CDBI::EnsemblRice::GeneArchive

=item * stable_id_events => Gramene::CDBI::EnsemblRice::StableIdEvent

=back


=head2 Gramene::CDBI::EnsemblRice::Marker

Corresponds to the table "marker."  

Stringifies (C<to_name>) to "left_primer".

Fields:

=over 4

=item * marker_id

=item * display_marker_synonym_id

=item * left_primer

=item * right_primer

=item * min_primer_dist

=item * max_primer_dist

=item * priority

=item * type

=back


Has A:

=over 4

=item * display_marker_synonym => Gramene::CDBI::EnsemblRice::MarkerSynonym

=back


Has Many:

=over 4

=item * marker_features => Gramene::CDBI::EnsemblRice::MarkerFeature

=item * marker_map_locations => Gramene::CDBI::EnsemblRice::MarkerMapLocation

=item * marker_synonyms => Gramene::CDBI::EnsemblRice::MarkerSynonym

=item * qtls_flank_marker_1 => Gramene::CDBI::EnsemblRice::Qtl

=item * qtls_flank_marker_2 => Gramene::CDBI::EnsemblRice::Qtl

=item * qtls_peak_marker => Gramene::CDBI::EnsemblRice::Qtl

=back


=head2 Gramene::CDBI::EnsemblRice::MarkerFeature

Corresponds to the table "marker_feature."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * marker_feature_id

=item * marker_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * analysis_id

=item * map_weight

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblRice::Analysis

=item * marker => Gramene::CDBI::EnsemblRice::Marker

=item * seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblRice::MarkerMapLocation

Corresponds to the table "marker_map_location."  

Stringifies (C<to_name>) to "chromosome_name".

Fields:

=over 4

=item * marker_id

=item * map_id

=item * chromosome_name

=item * marker_synonym_id

=item * position

=item * lod_score

=back


Has A:

=over 4

=item * map => Gramene::CDBI::EnsemblRice::Map

=item * marker => Gramene::CDBI::EnsemblRice::Marker

=item * marker_synonym => Gramene::CDBI::EnsemblRice::MarkerSynonym

=back



=head2 Gramene::CDBI::EnsemblRice::MarkerSynonym

Corresponds to the table "marker_synonym."  

Stringifies (C<to_name>) to "source".

Fields:

=over 4

=item * marker_synonym_id

=item * marker_id

=item * source

=item * name

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::EnsemblRice::Marker

=back


Has Many:

=over 4

=item * markers => Gramene::CDBI::EnsemblRice::Marker

=item * marker_map_locations => Gramene::CDBI::EnsemblRice::MarkerMapLocation

=back


=head2 Gramene::CDBI::EnsemblRice::Meta

Corresponds to the table "meta."  

Stringifies (C<to_name>) to "meta_key".

Fields:

=over 4

=item * meta_id

=item * meta_key

=item * meta_value

=back




=head2 Gramene::CDBI::EnsemblRice::MetaCoord

Corresponds to the table "meta_coord."  

Stringifies (C<to_name>) to "table_name".

Fields:

=over 4

=item * table_name

=item * coord_system_id

=item * max_length

=back


Has A:

=over 4

=item * coord_system => Gramene::CDBI::EnsemblRice::CoordSystem

=back



=head2 Gramene::CDBI::EnsemblRice::MiscAttrib

Corresponds to the table "misc_attrib."  

Stringifies (C<to_name>) to "value".

Fields:

=over 4

=item * misc_feature_id

=item * attrib_type_id

=item * value

=back


Has A:

=over 4

=item * attrib_type => Gramene::CDBI::EnsemblRice::AttribType

=item * misc_feature => Gramene::CDBI::EnsemblRice::MiscFeature

=back



=head2 Gramene::CDBI::EnsemblRice::MiscAttribHelper

Corresponds to the table "misc_attrib_helper."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * name

=item * attrib_name

=item * attrib_value

=back




=head2 Gramene::CDBI::EnsemblRice::MiscFeature

Corresponds to the table "misc_feature."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * misc_feature_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=back


Has A:

=over 4

=item * seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=back


Has Many:

=over 4

=item * misc_attribs => Gramene::CDBI::EnsemblRice::MiscAttrib

=item * misc_feature_misc_sets => Gramene::CDBI::EnsemblRice::MiscFeatureMiscSet

=back


=head2 Gramene::CDBI::EnsemblRice::MiscFeatureMiscSet

Corresponds to the table "misc_feature_misc_set."  

Stringifies (C<to_name>) to "misc_feature_id".

Fields:

=over 4

=item * misc_feature_id

=item * misc_set_id

=back


Has A:

=over 4

=item * misc_feature => Gramene::CDBI::EnsemblRice::MiscFeature

=item * misc_set => Gramene::CDBI::EnsemblRice::MiscSet

=back



=head2 Gramene::CDBI::EnsemblRice::MiscSet

Corresponds to the table "misc_set."  

Stringifies (C<to_name>) to "code".

Fields:

=over 4

=item * misc_set_id

=item * code

=item * name

=item * description

=item * max_length

=back



Has Many:

=over 4

=item * misc_feature_misc_sets => Gramene::CDBI::EnsemblRice::MiscFeatureMiscSet

=back


=head2 Gramene::CDBI::EnsemblRice::ObjectXref

Corresponds to the table "object_xref."  

Stringifies (C<to_name>) to "ensembl_object_type".

Fields:

=over 4

=item * object_xref_id

=item * ensembl_id

=item * ensembl_object_type

=item * xref_id

=back


Has A:

=over 4

=item * xref => Gramene::CDBI::EnsemblRice::Xref

=back


Has Many:

=over 4

=item * go_xrefs => Gramene::CDBI::EnsemblRice::GoXref

=item * identity_xrefs => Gramene::CDBI::EnsemblRice::IdentityXref

=back


=head2 Gramene::CDBI::EnsemblRice::OligoArray

Corresponds to the table "oligo_array."  

Stringifies (C<to_name>) to "probe_setsize".

Fields:

=over 4

=item * oligo_array_id

=item * parent_array_id

=item * probe_setsize

=item * name

=item * type

=back


Has A:

=over 4

=item * parent_array => Gramene::CDBI::EnsemblRice::OligoArray

=back


Has Many:

=over 4

=item * oligo_arrays => Gramene::CDBI::EnsemblRice::OligoArray

=item * oligo_probes => Gramene::CDBI::EnsemblRice::OligoProbe

=back


=head2 Gramene::CDBI::EnsemblRice::OligoFeature

Corresponds to the table "oligo_feature."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * oligo_feature_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * mismatches

=item * oligo_probe_id

=item * analysis_id

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblRice::Analysis

=item * oligo_probe => Gramene::CDBI::EnsemblRice::OligoProbe

=item * seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblRice::OligoProbe

Corresponds to the table "oligo_probe."  

Stringifies (C<to_name>) to "probeset".

Fields:

=over 4

=item * oligo_probe_id

=item * oligo_array_id

=item * probeset

=item * name

=item * description

=item * length

=back


Has A:

=over 4

=item * oligo_array => Gramene::CDBI::EnsemblRice::OligoArray

=back


Has Many:

=over 4

=item * oligo_features => Gramene::CDBI::EnsemblRice::OligoFeature

=back


=head2 Gramene::CDBI::EnsemblRice::PeptideArchive

Corresponds to the table "peptide_archive."  

Stringifies (C<to_name>) to "md5_checksum".

Fields:

=over 4

=item * peptide_archive_id

=item * md5_checksum

=item * peptide_seq

=back




=head2 Gramene::CDBI::EnsemblRice::PredictionExon

Corresponds to the table "prediction_exon."  

Stringifies (C<to_name>) to "exon_rank".

Fields:

=over 4

=item * prediction_exon_id

=item * prediction_transcript_id

=item * exon_rank

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * start_phase

=item * score

=item * p_value

=back


Has A:

=over 4

=item * prediction_transcript => Gramene::CDBI::EnsemblRice::PredictionTranscript

=item * seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblRice::PredictionTranscript

Corresponds to the table "prediction_transcript."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * prediction_transcript_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * analysis_id

=item * display_label

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblRice::Analysis

=item * seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=back


Has Many:

=over 4

=item * prediction_exons => Gramene::CDBI::EnsemblRice::PredictionExon

=back


=head2 Gramene::CDBI::EnsemblRice::ProteinAlignFeature

Corresponds to the table "protein_align_feature."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * protein_align_feature_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * hit_start

=item * hit_end

=item * hit_name

=item * analysis_id

=item * score

=item * evalue

=item * perc_ident

=item * cigar_line

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblRice::Analysis

=item * seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblRice::ProteinFeature

Corresponds to the table "protein_feature."  

Stringifies (C<to_name>) to "seq_start".

Fields:

=over 4

=item * protein_feature_id

=item * translation_id

=item * seq_start

=item * seq_end

=item * hit_start

=item * hit_end

=item * hit_id

=item * analysis_id

=item * score

=item * evalue

=item * perc_ident

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblRice::Analysis

=item * translation => Gramene::CDBI::EnsemblRice::Translation

=back



=head2 Gramene::CDBI::EnsemblRice::Qtl

Corresponds to the table "qtl."  

Stringifies (C<to_name>) to "trait".

Fields:

=over 4

=item * qtl_id

=item * trait

=item * lod_score

=item * flank_marker_id_1

=item * flank_marker_id_2

=item * peak_marker_id

=back


Has A:

=over 4

=item * peak_marker => Gramene::CDBI::EnsemblRice::Marker

=back


Has Many:

=over 4

=item * qtl_features => Gramene::CDBI::EnsemblRice::QtlFeature

=item * qtl_synonyms => Gramene::CDBI::EnsemblRice::QtlSynonym

=back


=head2 Gramene::CDBI::EnsemblRice::QtlFeature

Corresponds to the table "qtl_feature."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * qtl_id

=item * analysis_id

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblRice::Analysis

=item * qtl => Gramene::CDBI::EnsemblRice::Qtl

=item * seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblRice::QtlSynonym

Corresponds to the table "qtl_synonym."  

Stringifies (C<to_name>) to "source_database".

Fields:

=over 4

=item * qtl_synonym_id

=item * qtl_id

=item * source_database

=item * source_primary_id

=back


Has A:

=over 4

=item * qtl => Gramene::CDBI::EnsemblRice::Qtl

=back



=head2 Gramene::CDBI::EnsemblRice::RegulatoryFactor

Corresponds to the table "regulatory_factor."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * regulatory_factor_id

=item * name

=item * type

=back



Has Many:

=over 4

=item * regulatory_factor_codings => Gramene::CDBI::EnsemblRice::RegulatoryFactorCoding

=item * regulatory_features => Gramene::CDBI::EnsemblRice::RegulatoryFeature

=back


=head2 Gramene::CDBI::EnsemblRice::RegulatoryFactorCoding

Corresponds to the table "regulatory_factor_coding."  

Stringifies (C<to_name>) to "regulatory_factor_id".

Fields:

=over 4

=item * regulatory_factor_id

=item * transcript_id

=item * gene_id

=back


Has A:

=over 4

=item * gene => Gramene::CDBI::EnsemblRice::Gene

=item * regulatory_factor => Gramene::CDBI::EnsemblRice::RegulatoryFactor

=item * transcript => Gramene::CDBI::EnsemblRice::Transcript

=back



=head2 Gramene::CDBI::EnsemblRice::RegulatoryFeature

Corresponds to the table "regulatory_feature."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * regulatory_feature_id

=item * name

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * analysis_id

=item * regulatory_factor_id

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblRice::Analysis

=item * regulatory_factor => Gramene::CDBI::EnsemblRice::RegulatoryFactor

=item * seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblRice::RegulatoryFeatureObject

Corresponds to the table "regulatory_feature_object."  

Stringifies (C<to_name>) to "ensembl_object_type".

Fields:

=over 4

=item * regulatory_feature_id

=item * ensembl_object_type

=item * ensembl_object_id

=item * influence

=item * evidence

=back




=head2 Gramene::CDBI::EnsemblRice::RegulatorySearchRegion

Corresponds to the table "regulatory_search_region."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * regulatory_search_region_id

=item * name

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * ensembl_object_type

=item * ensembl_object_id

=item * analysis_id

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblRice::Analysis

=item * seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblRice::RepeatConsensus

Corresponds to the table "repeat_consensus."  

Stringifies (C<to_name>) to "repeat_name".

Fields:

=over 4

=item * repeat_consensus_id

=item * repeat_name

=item * repeat_class

=item * repeat_type

=item * repeat_consensus

=back



Has Many:

=over 4

=item * repeat_features => Gramene::CDBI::EnsemblRice::RepeatFeature

=back


=head2 Gramene::CDBI::EnsemblRice::RepeatFeature

Corresponds to the table "repeat_feature."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * repeat_feature_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * repeat_start

=item * repeat_end

=item * repeat_consensus_id

=item * analysis_id

=item * score

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblRice::Analysis

=item * repeat_consensus => Gramene::CDBI::EnsemblRice::RepeatConsensus

=item * seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblRice::SeqRegion

Corresponds to the table "seq_region."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * seq_region_id

=item * name

=item * coord_system_id

=item * length

=back


Has A:

=over 4

=item * coord_system => Gramene::CDBI::EnsemblRice::CoordSystem

=back


Has Many:

=over 4

=item * assemblies_asm_seq_region => Gramene::CDBI::EnsemblRice::Assembly

=item * assemblies_cmp_seq_region => Gramene::CDBI::EnsemblRice::Assembly

=item * assembly_exceptions_seq_region => Gramene::CDBI::EnsemblRice::AssemblyException

=item * assembly_exceptions_exc_seq_region => Gramene::CDBI::EnsemblRice::AssemblyException

=item * density_features => Gramene::CDBI::EnsemblRice::DensityFeature

=item * ditag_features => Gramene::CDBI::EnsemblRice::DitagFeature

=item * dnas => Gramene::CDBI::EnsemblRice::Dna

=item * dna_align_features => Gramene::CDBI::EnsemblRice::DnaAlignFeature

=item * dnacs => Gramene::CDBI::EnsemblRice::Dnac

=item * exons => Gramene::CDBI::EnsemblRice::Exon

=item * genes => Gramene::CDBI::EnsemblRice::Gene

=item * karyotypes => Gramene::CDBI::EnsemblRice::Karyotype

=item * marker_features => Gramene::CDBI::EnsemblRice::MarkerFeature

=item * misc_features => Gramene::CDBI::EnsemblRice::MiscFeature

=item * oligo_features => Gramene::CDBI::EnsemblRice::OligoFeature

=item * prediction_exons => Gramene::CDBI::EnsemblRice::PredictionExon

=item * prediction_transcripts => Gramene::CDBI::EnsemblRice::PredictionTranscript

=item * protein_align_features => Gramene::CDBI::EnsemblRice::ProteinAlignFeature

=item * qtl_features => Gramene::CDBI::EnsemblRice::QtlFeature

=item * regulatory_features => Gramene::CDBI::EnsemblRice::RegulatoryFeature

=item * regulatory_search_regions => Gramene::CDBI::EnsemblRice::RegulatorySearchRegion

=item * repeat_features => Gramene::CDBI::EnsemblRice::RepeatFeature

=item * seq_region_attribs => Gramene::CDBI::EnsemblRice::SeqRegionAttrib

=item * simple_features => Gramene::CDBI::EnsemblRice::SimpleFeature

=item * transcripts => Gramene::CDBI::EnsemblRice::Transcript

=back


=head2 Gramene::CDBI::EnsemblRice::SeqRegionAttrib

Corresponds to the table "seq_region_attrib."  

Stringifies (C<to_name>) to "value".

Fields:

=over 4

=item * seq_region_id

=item * attrib_type_id

=item * value

=back


Has A:

=over 4

=item * attrib_type => Gramene::CDBI::EnsemblRice::AttribType

=item * seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblRice::SeqtableDaf0

Corresponds to the table "seqtable_daf0."  

Stringifies (C<to_name>) to "embl_acc".

Fields:

=over 4

=item * embl_acc

=item * daf_chr_start

=item * daf_chr_end

=item * chr_strand

=item * chromosome

=item * superctg_end

=item * superctg_name

=item * superctg_ori

=item * superctg_start

=item * daf_clone_start

=item * daf_clone_end

=item * daf_super_start

=item * daf_super_end

=back




=head2 Gramene::CDBI::EnsemblRice::SeqtableDaf1

Corresponds to the table "seqtable_daf1."  

Stringifies (C<to_name>) to "embl_acc".

Fields:

=over 4

=item * embl_acc

=item * chromosome

=item * chr_start

=item * chr_end

=item * superctg_name

=item * super_start

=item * super_end

=back




=head2 Gramene::CDBI::EnsemblRice::SeqtableHelper

Corresponds to the table "seqtable_helper."  

Stringifies (C<to_name>) to "bacname".

Fields:

=over 4

=item * clone_id

=item * bacname

=item * chromosome

=item * embl_acc

=item * embl_version

=item * htg_phase

=item * super_name

=item * super_chr_start

=item * super_chr_end

=item * clone_super_start

=item * clone_super_end

=item * clone_chr_start

=item * clone_chr_end

=item * marker_name

=item * marker_type

=item * jrgp_rflp_cm

=item * jrgp_rflp_map

=item * jrgp_rflp_feature

=item * cornell_rflp_cm

=item * cornell_rflp_map

=item * cornell_rflp_feature

=item * irmi_ssr_cm

=item * irmi_ssr_map

=item * irmi_ssr_feature

=back




=head2 Gramene::CDBI::EnsemblRice::SeqtablePrehelper

Corresponds to the table "seqtable_prehelper."  

Stringifies (C<to_name>) to "embl_acc".

Fields:

=over 4

=item * clone_id

=item * chr_id

=item * embl_acc

=item * embl_version

=item * htg_phase

=item * chromosome

=item * super_name

=item * super_chr_start

=item * super_chr_end

=item * clone_super_start

=item * clone_super_end

=item * clone_chr_start

=item * clone_chr_end

=item * contig_clone_start

=item * contig_clone_end

=item * bacname

=item * bac_length

=item * source

=back




=head2 Gramene::CDBI::EnsemblRice::SimpleFeature

Corresponds to the table "simple_feature."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * simple_feature_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * display_label

=item * analysis_id

=item * score

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblRice::Analysis

=item * seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblRice::StableIdEvent

Corresponds to the table "stable_id_event."  

Stringifies (C<to_name>) to "old_version".

Fields:

=over 4

=item * old_stable_id

=item * old_version

=item * new_stable_id

=item * new_version

=item * mapping_session_id

=item * type

=item * score

=back


Has A:

=over 4

=item * mapping_session => Gramene::CDBI::EnsemblRice::MappingSession

=back



=head2 Gramene::CDBI::EnsemblRice::SupportingFeature

Corresponds to the table "supporting_feature."  

Stringifies (C<to_name>) to "feature_type".

Fields:

=over 4

=item * exon_id

=item * feature_type

=item * feature_id

=back


Has A:

=over 4

=item * exon => Gramene::CDBI::EnsemblRice::Exon

=back



=head2 Gramene::CDBI::EnsemblRice::Transcript

Corresponds to the table "transcript."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * transcript_id

=item * gene_id

=item * analysis_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * display_xref_id

=item * biotype

=item * status

=item * description

=item * is_current

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblRice::Analysis

=item * gene => Gramene::CDBI::EnsemblRice::Gene

=item * seq_region => Gramene::CDBI::EnsemblRice::SeqRegion

=item * display_xref => Gramene::CDBI::EnsemblRice::Xref

=back


Has Many:

=over 4

=item * exon_transcripts => Gramene::CDBI::EnsemblRice::ExonTranscript

=item * regulatory_factor_codings => Gramene::CDBI::EnsemblRice::RegulatoryFactorCoding

=item * transcript_attribs => Gramene::CDBI::EnsemblRice::TranscriptAttrib

=item * transcript_stable_ids => Gramene::CDBI::EnsemblRice::TranscriptStableId

=item * transcript_supporting_features => Gramene::CDBI::EnsemblRice::TranscriptSupportingFeature

=item * translations => Gramene::CDBI::EnsemblRice::Translation

=back


=head2 Gramene::CDBI::EnsemblRice::TranscriptAttrib

Corresponds to the table "transcript_attrib."  

Stringifies (C<to_name>) to "value".

Fields:

=over 4

=item * transcript_id

=item * attrib_type_id

=item * value

=back


Has A:

=over 4

=item * attrib_type => Gramene::CDBI::EnsemblRice::AttribType

=item * transcript => Gramene::CDBI::EnsemblRice::Transcript

=back



=head2 Gramene::CDBI::EnsemblRice::TranscriptStableId

Corresponds to the table "transcript_stable_id."  

Stringifies (C<to_name>) to "version".

Fields:

=over 4

=item * transcript_id

=item * stable_id

=item * version

=item * created_date

=item * modified_date

=back


Has A:

=over 4

=item * transcript => Gramene::CDBI::EnsemblRice::Transcript

=back



=head2 Gramene::CDBI::EnsemblRice::TranscriptSupportingFeature

Corresponds to the table "transcript_supporting_feature."  

Stringifies (C<to_name>) to "feature_type".

Fields:

=over 4

=item * transcript_id

=item * feature_type

=item * feature_id

=back


Has A:

=over 4

=item * transcript => Gramene::CDBI::EnsemblRice::Transcript

=back



=head2 Gramene::CDBI::EnsemblRice::Translation

Corresponds to the table "translation."  

Stringifies (C<to_name>) to "seq_start".

Fields:

=over 4

=item * translation_id

=item * transcript_id

=item * seq_start

=item * start_exon_id

=item * seq_end

=item * end_exon_id

=back


Has A:

=over 4

=item * start_exon => Gramene::CDBI::EnsemblRice::Exon

=item * end_exon => Gramene::CDBI::EnsemblRice::Exon

=item * transcript => Gramene::CDBI::EnsemblRice::Transcript

=back


Has Many:

=over 4

=item * protein_features => Gramene::CDBI::EnsemblRice::ProteinFeature

=item * translation_attribs => Gramene::CDBI::EnsemblRice::TranslationAttrib

=item * translation_stable_ids => Gramene::CDBI::EnsemblRice::TranslationStableId

=back


=head2 Gramene::CDBI::EnsemblRice::TranslationAttrib

Corresponds to the table "translation_attrib."  

Stringifies (C<to_name>) to "value".

Fields:

=over 4

=item * translation_id

=item * attrib_type_id

=item * value

=back


Has A:

=over 4

=item * attrib_type => Gramene::CDBI::EnsemblRice::AttribType

=item * translation => Gramene::CDBI::EnsemblRice::Translation

=back



=head2 Gramene::CDBI::EnsemblRice::TranslationStableId

Corresponds to the table "translation_stable_id."  

Stringifies (C<to_name>) to "version".

Fields:

=over 4

=item * translation_id

=item * stable_id

=item * version

=item * created_date

=item * modified_date

=back


Has A:

=over 4

=item * translation => Gramene::CDBI::EnsemblRice::Translation

=back



=head2 Gramene::CDBI::EnsemblRice::UnconventionalTranscriptAssociation

Corresponds to the table "unconventional_transcript_association."  

Stringifies (C<to_name>) to "interaction_type".

Fields:

=over 4

=item * transcript_id

=item * gene_id

=item * interaction_type

=back




=head2 Gramene::CDBI::EnsemblRice::UnmappedObject

Corresponds to the table "unmapped_object."  

Stringifies (C<to_name>) to "type".

Fields:

=over 4

=item * unmapped_object_id

=item * type

=item * analysis_id

=item * external_db_id

=item * identifier

=item * unmapped_reason_id

=item * query_score

=item * target_score

=item * ensembl_id

=item * ensembl_object_type

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblRice::Analysis

=item * external_db => Gramene::CDBI::EnsemblRice::ExternalDb

=item * unmapped_reason => Gramene::CDBI::EnsemblRice::UnmappedReason

=back



=head2 Gramene::CDBI::EnsemblRice::UnmappedReason

Corresponds to the table "unmapped_reason."  

Stringifies (C<to_name>) to "summary_description".

Fields:

=over 4

=item * unmapped_reason_id

=item * summary_description

=item * full_description

=back



Has Many:

=over 4

=item * unmapped_objects => Gramene::CDBI::EnsemblRice::UnmappedObject

=back


=head2 Gramene::CDBI::EnsemblRice::Xref

Corresponds to the table "xref."  

Stringifies (C<to_name>) to "dbprimary_acc".

Fields:

=over 4

=item * xref_id

=item * external_db_id

=item * dbprimary_acc

=item * display_label

=item * version

=item * description

=item * info_type

=item * info_text

=back


Has A:

=over 4

=item * external_db => Gramene::CDBI::EnsemblRice::ExternalDb

=back


Has Many:

=over 4

=item * external_synonyms => Gramene::CDBI::EnsemblRice::ExternalSynonym

=item * genes => Gramene::CDBI::EnsemblRice::Gene

=item * object_xrefs => Gramene::CDBI::EnsemblRice::ObjectXref

=item * transcripts => Gramene::CDBI::EnsemblRice::Transcript

=back


=head1 SEE ALSO

Gramene::CDBI.

=head1 AUTHOR

Generated by Ken Youens-Clark on Mon Mar 26 15:23:32 2007.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
