package Gramene::CDBI::EnsemblArabidopsis;

use strict;

use base qw( Gramene::CDBI );

use Carp qw( croak );
use Gramene::Config;
use Gramene::Utils qw( table_name_to_gramene_cdbi_class );
use Readonly;

Readonly my $CONFIG_NAME => 'ensembl_arabidopsis';

my ( $DBH ); # global database handle

# ------------------------------------------------
sub db_Main {

=pod

=head2 db_Main

  my $db = $cdbi->db_Main;

This returns a DBI database handle.  If you choose to override this, you
must include "RootClass => 'DBIx::ContextualFetch'" as an option to
DBI->connect.

=cut

    my $class = shift;

    if ( !defined $DBH ) {
        my $config = config();

        $DBH = DBI->connect(
            $config->{'db_dsn'},
            $config->{'db_user'},
            $config->{'db_pass'},
            { __PACKAGE__->_default_attributes },
        );
    }

    return $DBH;
} 

# ------------------------------------------------
sub config {

=pod

=head2 config

  my $config = $cdbi->config;
  my %config = $cdbi->config;

Returns the module's configuration from "gramene.conf."

=cut

    my $gconfig = Gramene::Config->new or croak 'No config file object';
    my $config  = $gconfig->get( $CONFIG_NAME )
                  or croak "Bad configuration section '$CONFIG_NAME'";

    return wantarray ? %$config : $config;
}

# ------------------------------------------------
sub table_name_to_class {

=pod

=head2 table_name_to_class

  my $class = $cdbi->table_name_to_class('foo');

Returns the something like "Gramene::CDBI::ModuleName::Foo."

=cut

    my ( $self, $table ) = @_;

    return table_name_to_gramene_cdbi_class( $CONFIG_NAME, $table );
}

# ------------------------------------------------
sub represented_tables {
    qw/
        alt_allele
        analysis
        analysis_description
        assembly
        assembly_exception
        attrib_type
        coord_system
        density_feature
        density_type
        ditag
        ditag_feature
        dna
        dna_align_feature
        dnac
        ensembl_gene_helper
        exon
        exon_stable_id
        exon_transcript
        external_db
        external_synonym
        gene
        gene_archive
        gene_attrib
        gene_stable_id
        go_xref
        gramene_gene_ext_helper
        gramene_gene_x_tmp0
        gramene_gene_x_tmp1
        identity_xref
        input_id_analysis
        input_id_type_analysis
        insert_coords
        interpro
        job
        job_status
        karyotype
        map
        mapping_session
        marker
        marker_feature
        marker_map_location
        marker_synonym
        meta
        meta_coord
        misc_attrib
        misc_attrib_helper
        misc_feature
        misc_feature_misc_set
        misc_set
        object_xref
        oligo_array
        oligo_feature
        oligo_probe
        peptide_archive
        prediction_exon
        prediction_transcript
        protein_align_feature
        protein_feature
        qtl
        qtl_feature
        qtl_synonym
        regulatory_factor
        regulatory_factor_coding
        regulatory_feature
        regulatory_feature_object
        regulatory_search_region
        repeat_consensus
        repeat_feature
        rule_conditions
        rule_goal
        seq_region
        seq_region_attrib
        seqtable_cmap_tmp
        seqtable_cmap_tmp0
        seqtable_daf0
        seqtable_daf1
        seqtable_helper
        seqtable_marker_tmp
        seqtable_prehelper
        simple_feature
        stable_id_event
        supporting_feature
        transcript
        transcript_attrib
        transcript_stable_id
        transcript_supporting_feature
        translation
        translation_attrib
        translation_stable_id
        unconventional_transcript_association
        unmapped_object
        unmapped_reason
        xref

    /
}

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::AltAllele; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::AltAllele->table('alt_allele');


sub gene {
    my $self = shift;
    return $self->gene_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::AltAllele->columns( 
    All => qw/ 
        alt_allele_id
        gene_id 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::AltAllele->columns( 
    Essential => qw/ 
        alt_allele_id
        gene_id 
    /
);

Gramene::CDBI::EnsemblArabidopsis::AltAllele->columns( 
    Ordered => qw/ 
        alt_allele_id
        gene_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->alt_allele_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'alt_allele_id' => 'number',
        'gene_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::AltAllele->has_a(
    gene_id => 'Gramene::CDBI::EnsemblArabidopsis::Gene'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene => [ 'gene_id', 'gene_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::AnalysisDescription; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::AnalysisDescription->table('analysis_description');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::AnalysisDescription->columns( 
    All => qw/ 
        analysis_id
        description
        display_label
        displayable
        web_data 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::AnalysisDescription->columns( 
    Essential => qw/ 
        analysis_id
        description
        display_label
        displayable
        web_data 
    /
);

Gramene::CDBI::EnsemblArabidopsis::AnalysisDescription->columns( 
    Ordered => qw/ 
        analysis_id
        description
        display_label
        displayable
        web_data 
    /
);

sub to_string {
    my $self = shift;
    return $self->description || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'analysis_id' => 'number',
        'description' => 'string',
        'display_label' => 'string',
        'displayable' => 'string',
        'web_data' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::AnalysisDescription->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblArabidopsis::Analysis'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::DensityFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::DensityFeature->table('density_feature');


sub density_type {
    my $self = shift;
    return $self->density_type_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::DensityFeature->columns( 
    All => qw/ 
        density_feature_id
        density_type_id
        seq_region_id
        seq_region_start
        seq_region_end
        density_value 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::DensityFeature->columns( 
    Essential => qw/ 
        density_feature_id
        density_type_id
        seq_region_id
        seq_region_start
        seq_region_end
        density_value 
    /
);

Gramene::CDBI::EnsemblArabidopsis::DensityFeature->columns( 
    Ordered => qw/ 
        density_feature_id
        density_type_id
        seq_region_id
        seq_region_start
        seq_region_end
        density_value 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'density_feature_id' => 'number',
        'density_type_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'density_value' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::DensityFeature->has_a(
    density_type_id => 'Gramene::CDBI::EnsemblArabidopsis::DensityType'
);

Gramene::CDBI::EnsemblArabidopsis::DensityFeature->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            density_type => [ 'density_type_id', 'density_type_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::DensityType; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::DensityType->table('density_type');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::DensityType->columns( 
    All => qw/ 
        density_type_id
        analysis_id
        block_size
        region_features
        value_type 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::DensityType->columns( 
    Essential => qw/ 
        density_type_id
        analysis_id
        block_size
        region_features
        value_type 
    /
);

Gramene::CDBI::EnsemblArabidopsis::DensityType->columns( 
    Ordered => qw/ 
        density_type_id
        analysis_id
        block_size
        region_features
        value_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->block_size || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'density_type_id' => 'number',
        'analysis_id' => 'number',
        'block_size' => 'number',
        'region_features' => 'number',
        'value_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::DensityType->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblArabidopsis::Analysis'
);

Gramene::CDBI::EnsemblArabidopsis::DensityType->has_many( 
    'density_features',
    'Gramene::CDBI::EnsemblArabidopsis::DensityFeature',
    'density_type_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            density_feature => [ 'density_type_id', 'density_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::DitagFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::DitagFeature->table('ditag_feature');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub ditag {
    my $self = shift;
    return $self->ditag_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::DitagFeature->columns( 
    All => qw/ 
        ditag_feature_id
        ditag_id
        ditag_pair_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        analysis_id
        hit_start
        hit_end
        hit_strand
        cigar_line
        ditag_side 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::DitagFeature->columns( 
    Essential => qw/ 
        ditag_feature_id
        ditag_id
        ditag_pair_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        analysis_id
        hit_start
        hit_end
        hit_strand
        cigar_line
        ditag_side 
    /
);

Gramene::CDBI::EnsemblArabidopsis::DitagFeature->columns( 
    Ordered => qw/ 
        ditag_feature_id
        ditag_id
        ditag_pair_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        analysis_id
        hit_start
        hit_end
        hit_strand
        cigar_line
        ditag_side 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'ditag_feature_id' => 'number',
        'ditag_id' => 'number',
        'ditag_pair_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'analysis_id' => 'number',
        'hit_start' => 'number',
        'hit_end' => 'number',
        'hit_strand' => 'string',
        'cigar_line' => 'string',
        'ditag_side' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::DitagFeature->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblArabidopsis::Analysis'
);

Gramene::CDBI::EnsemblArabidopsis::DitagFeature->has_a(
    ditag_id => 'Gramene::CDBI::EnsemblArabidopsis::Ditag'
);

Gramene::CDBI::EnsemblArabidopsis::DitagFeature->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            ditag => [ 'ditag_id', 'ditag_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::DnaAlignFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::DnaAlignFeature->table('dna_align_feature');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::DnaAlignFeature->columns( 
    All => qw/ 
        dna_align_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        hit_start
        hit_end
        hit_strand
        hit_name
        analysis_id
        score
        evalue
        perc_ident
        cigar_line 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::DnaAlignFeature->columns( 
    Essential => qw/ 
        dna_align_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        hit_start
        hit_end
        hit_strand
        hit_name
        analysis_id
        score
        evalue
        perc_ident
        cigar_line 
    /
);

Gramene::CDBI::EnsemblArabidopsis::DnaAlignFeature->columns( 
    Ordered => qw/ 
        dna_align_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        hit_start
        hit_end
        hit_strand
        hit_name
        analysis_id
        score
        evalue
        perc_ident
        cigar_line 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'dna_align_feature_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'hit_start' => 'number',
        'hit_end' => 'number',
        'hit_strand' => 'string',
        'hit_name' => 'string',
        'analysis_id' => 'number',
        'score' => 'number',
        'evalue' => 'number',
        'perc_ident' => 'number',
        'cigar_line' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::DnaAlignFeature->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblArabidopsis::Analysis'
);

Gramene::CDBI::EnsemblArabidopsis::DnaAlignFeature->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::GeneAttrib; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::GeneAttrib->table('gene_attrib');


sub attrib_type {
    my $self = shift;
    return $self->attrib_type_id;
}
    

sub gene {
    my $self = shift;
    return $self->gene_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::GeneAttrib->columns( 
    All => qw/ 
        gene_id
        attrib_type_id
        value 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::GeneAttrib->columns( 
    Essential => qw/ 
        gene_id
        attrib_type_id
        value 
    /
);

Gramene::CDBI::EnsemblArabidopsis::GeneAttrib->columns( 
    Ordered => qw/ 
        gene_id
        attrib_type_id
        value 
    /
);

sub to_string {
    my $self = shift;
    return $self->value || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_id' => 'number',
        'attrib_type_id' => 'string',
        'value' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::GeneAttrib->has_a(
    attrib_type_id => 'Gramene::CDBI::EnsemblArabidopsis::AttribType'
);

Gramene::CDBI::EnsemblArabidopsis::GeneAttrib->has_a(
    gene_id => 'Gramene::CDBI::EnsemblArabidopsis::Gene'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            attrib_type => [ 'attrib_type_id', 'attrib_type_id' ],
            gene => [ 'gene_id', 'gene_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::GeneStableId; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::GeneStableId->table('gene_stable_id');


sub gene {
    my $self = shift;
    return $self->gene_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::GeneStableId->columns( 
    All => qw/ 
        gene_id
        stable_id
        version
        created_date
        modified_date 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::GeneStableId->columns( 
    Essential => qw/ 
        gene_id
        stable_id
        version
        created_date
        modified_date 
    /
);

Gramene::CDBI::EnsemblArabidopsis::GeneStableId->columns( 
    Ordered => qw/ 
        gene_id
        stable_id
        version
        created_date
        modified_date 
    /
);

sub to_string {
    my $self = shift;
    return $self->version || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_id' => 'number',
        'stable_id' => 'string',
        'version' => 'number',
        'created_date' => 'string',
        'modified_date' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::GeneStableId->has_a(
    gene_id => 'Gramene::CDBI::EnsemblArabidopsis::Gene'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene => [ 'gene_id', 'gene_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactorCoding; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactorCoding->table('regulatory_factor_coding');


sub gene {
    my $self = shift;
    return $self->gene_id;
}
    

sub regulatory_factor {
    my $self = shift;
    return $self->regulatory_factor_id;
}
    

sub transcript {
    my $self = shift;
    return $self->transcript_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactorCoding->columns( 
    All => qw/ 
        regulatory_factor_id
        transcript_id
        gene_id 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactorCoding->columns( 
    Essential => qw/ 
        regulatory_factor_id
        transcript_id
        gene_id 
    /
);

Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactorCoding->columns( 
    Ordered => qw/ 
        regulatory_factor_id
        transcript_id
        gene_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->regulatory_factor_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'regulatory_factor_id' => 'number',
        'transcript_id' => 'number',
        'gene_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactorCoding->has_a(
    gene_id => 'Gramene::CDBI::EnsemblArabidopsis::Gene'
);

Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactorCoding->has_a(
    regulatory_factor_id => 'Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactor'
);

Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactorCoding->has_a(
    transcript_id => 'Gramene::CDBI::EnsemblArabidopsis::Transcript'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene => [ 'gene_id', 'gene_id' ],
            regulatory_factor => [ 'regulatory_factor_id', 'regulatory_factor_id' ],
            transcript => [ 'transcript_id', 'transcript_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::ExonTranscript; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::ExonTranscript->table('exon_transcript');


sub exon {
    my $self = shift;
    return $self->exon_id;
}
    

sub transcript {
    my $self = shift;
    return $self->transcript_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::ExonTranscript->columns( 
    All => qw/ 
        exon_id
        transcript_id
        rank 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::ExonTranscript->columns( 
    Essential => qw/ 
        exon_id
        transcript_id
        rank 
    /
);

Gramene::CDBI::EnsemblArabidopsis::ExonTranscript->columns( 
    Ordered => qw/ 
        exon_id
        transcript_id
        rank 
    /
);


Gramene::CDBI::EnsemblArabidopsis::ExonTranscript->columns( 
    Primary => qw/ 
        exon_id
        transcript_id
        rank 
    / 
);


sub to_string {
    my $self = shift;
    return $self->exon_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'exon_id' => 'number',
        'transcript_id' => 'number',
        'rank' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::ExonTranscript->has_a(
    exon_id => 'Gramene::CDBI::EnsemblArabidopsis::Exon'
);

Gramene::CDBI::EnsemblArabidopsis::ExonTranscript->has_a(
    transcript_id => 'Gramene::CDBI::EnsemblArabidopsis::Transcript'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            exon => [ 'exon_id', 'exon_id' ],
            transcript => [ 'transcript_id', 'transcript_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::TranscriptAttrib; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::TranscriptAttrib->table('transcript_attrib');


sub attrib_type {
    my $self = shift;
    return $self->attrib_type_id;
}
    

sub transcript {
    my $self = shift;
    return $self->transcript_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::TranscriptAttrib->columns( 
    All => qw/ 
        transcript_id
        attrib_type_id
        value 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::TranscriptAttrib->columns( 
    Essential => qw/ 
        transcript_id
        attrib_type_id
        value 
    /
);

Gramene::CDBI::EnsemblArabidopsis::TranscriptAttrib->columns( 
    Ordered => qw/ 
        transcript_id
        attrib_type_id
        value 
    /
);

sub to_string {
    my $self = shift;
    return $self->value || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'transcript_id' => 'number',
        'attrib_type_id' => 'string',
        'value' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::TranscriptAttrib->has_a(
    attrib_type_id => 'Gramene::CDBI::EnsemblArabidopsis::AttribType'
);

Gramene::CDBI::EnsemblArabidopsis::TranscriptAttrib->has_a(
    transcript_id => 'Gramene::CDBI::EnsemblArabidopsis::Transcript'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            attrib_type => [ 'attrib_type_id', 'attrib_type_id' ],
            transcript => [ 'transcript_id', 'transcript_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::TranscriptStableId; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::TranscriptStableId->table('transcript_stable_id');


sub transcript {
    my $self = shift;
    return $self->transcript_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::TranscriptStableId->columns( 
    All => qw/ 
        transcript_id
        stable_id
        version
        created_date
        modified_date 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::TranscriptStableId->columns( 
    Essential => qw/ 
        transcript_id
        stable_id
        version
        created_date
        modified_date 
    /
);

Gramene::CDBI::EnsemblArabidopsis::TranscriptStableId->columns( 
    Ordered => qw/ 
        transcript_id
        stable_id
        version
        created_date
        modified_date 
    /
);

sub to_string {
    my $self = shift;
    return $self->version || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'transcript_id' => 'number',
        'stable_id' => 'string',
        'version' => 'number',
        'created_date' => 'string',
        'modified_date' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::TranscriptStableId->has_a(
    transcript_id => 'Gramene::CDBI::EnsemblArabidopsis::Transcript'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            transcript => [ 'transcript_id', 'transcript_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::TranscriptSupportingFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::TranscriptSupportingFeature->table('transcript_supporting_feature');


sub transcript {
    my $self = shift;
    return $self->transcript_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::TranscriptSupportingFeature->columns( 
    All => qw/ 
        transcript_id
        feature_type
        feature_id 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::TranscriptSupportingFeature->columns( 
    Essential => qw/ 
        transcript_id
        feature_type
        feature_id 
    /
);

Gramene::CDBI::EnsemblArabidopsis::TranscriptSupportingFeature->columns( 
    Ordered => qw/ 
        transcript_id
        feature_type
        feature_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->feature_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'transcript_id' => 'number',
        'feature_type' => 'string',
        'feature_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::TranscriptSupportingFeature->has_a(
    transcript_id => 'Gramene::CDBI::EnsemblArabidopsis::Transcript'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            transcript => [ 'transcript_id', 'transcript_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::ProteinFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::ProteinFeature->table('protein_feature');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub translation {
    my $self = shift;
    return $self->translation_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::ProteinFeature->columns( 
    All => qw/ 
        protein_feature_id
        translation_id
        seq_start
        seq_end
        hit_start
        hit_end
        hit_id
        analysis_id
        score
        evalue
        perc_ident 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::ProteinFeature->columns( 
    Essential => qw/ 
        protein_feature_id
        translation_id
        seq_start
        seq_end
        hit_start
        hit_end
        hit_id
        analysis_id
        score
        evalue
        perc_ident 
    /
);

Gramene::CDBI::EnsemblArabidopsis::ProteinFeature->columns( 
    Ordered => qw/ 
        protein_feature_id
        translation_id
        seq_start
        seq_end
        hit_start
        hit_end
        hit_id
        analysis_id
        score
        evalue
        perc_ident 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'protein_feature_id' => 'number',
        'translation_id' => 'number',
        'seq_start' => 'number',
        'seq_end' => 'number',
        'hit_start' => 'number',
        'hit_end' => 'number',
        'hit_id' => 'string',
        'analysis_id' => 'number',
        'score' => 'number',
        'evalue' => 'number',
        'perc_ident' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::ProteinFeature->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblArabidopsis::Analysis'
);

Gramene::CDBI::EnsemblArabidopsis::ProteinFeature->has_a(
    translation_id => 'Gramene::CDBI::EnsemblArabidopsis::Translation'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            translation => [ 'translation_id', 'translation_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::TranslationAttrib; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::TranslationAttrib->table('translation_attrib');


sub attrib_type {
    my $self = shift;
    return $self->attrib_type_id;
}
    

sub translation {
    my $self = shift;
    return $self->translation_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::TranslationAttrib->columns( 
    All => qw/ 
        translation_id
        attrib_type_id
        value 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::TranslationAttrib->columns( 
    Essential => qw/ 
        translation_id
        attrib_type_id
        value 
    /
);

Gramene::CDBI::EnsemblArabidopsis::TranslationAttrib->columns( 
    Ordered => qw/ 
        translation_id
        attrib_type_id
        value 
    /
);

sub to_string {
    my $self = shift;
    return $self->value || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'translation_id' => 'number',
        'attrib_type_id' => 'string',
        'value' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::TranslationAttrib->has_a(
    attrib_type_id => 'Gramene::CDBI::EnsemblArabidopsis::AttribType'
);

Gramene::CDBI::EnsemblArabidopsis::TranslationAttrib->has_a(
    translation_id => 'Gramene::CDBI::EnsemblArabidopsis::Translation'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            attrib_type => [ 'attrib_type_id', 'attrib_type_id' ],
            translation => [ 'translation_id', 'translation_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::TranslationStableId; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::TranslationStableId->table('translation_stable_id');


sub translation {
    my $self = shift;
    return $self->translation_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::TranslationStableId->columns( 
    All => qw/ 
        translation_id
        stable_id
        version
        created_date
        modified_date 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::TranslationStableId->columns( 
    Essential => qw/ 
        translation_id
        stable_id
        version
        created_date
        modified_date 
    /
);

Gramene::CDBI::EnsemblArabidopsis::TranslationStableId->columns( 
    Ordered => qw/ 
        translation_id
        stable_id
        version
        created_date
        modified_date 
    /
);

sub to_string {
    my $self = shift;
    return $self->version || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'translation_id' => 'number',
        'stable_id' => 'string',
        'version' => 'number',
        'created_date' => 'string',
        'modified_date' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::TranslationStableId->has_a(
    translation_id => 'Gramene::CDBI::EnsemblArabidopsis::Translation'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            translation => [ 'translation_id', 'translation_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::Translation; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::Translation->table('translation');


sub start_exon {
    my $self = shift;
    return $self->start_exon_id;
}
    
sub end_exon {
    my $self = shift;
    return $self->end_exon_id;
}
    

sub transcript {
    my $self = shift;
    return $self->transcript_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::Translation->columns( 
    All => qw/ 
        translation_id
        transcript_id
        seq_start
        start_exon_id
        seq_end
        end_exon_id 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::Translation->columns( 
    Essential => qw/ 
        translation_id
        transcript_id
        seq_start
        start_exon_id
        seq_end
        end_exon_id 
    /
);

Gramene::CDBI::EnsemblArabidopsis::Translation->columns( 
    Ordered => qw/ 
        translation_id
        transcript_id
        seq_start
        start_exon_id
        seq_end
        end_exon_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'translation_id' => 'number',
        'transcript_id' => 'number',
        'seq_start' => 'number',
        'start_exon_id' => 'number',
        'seq_end' => 'number',
        'end_exon_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::Translation->has_a(
    start_exon_id => 'Gramene::CDBI::EnsemblArabidopsis::Exon'
);

Gramene::CDBI::EnsemblArabidopsis::Translation->has_a(
    end_exon_id => 'Gramene::CDBI::EnsemblArabidopsis::Exon'
);

Gramene::CDBI::EnsemblArabidopsis::Translation->has_a(
    transcript_id => 'Gramene::CDBI::EnsemblArabidopsis::Transcript'
);

Gramene::CDBI::EnsemblArabidopsis::Translation->has_many( 
    'protein_features',
    'Gramene::CDBI::EnsemblArabidopsis::ProteinFeature',
    'translation_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Translation->has_many( 
    'translation_attribs',
    'Gramene::CDBI::EnsemblArabidopsis::TranslationAttrib',
    'translation_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Translation->has_many( 
    'translation_stable_ids',
    'Gramene::CDBI::EnsemblArabidopsis::TranslationStableId',
    'translation_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            exon => [ 'end_exon_id', 'exon_id' ],
            protein_feature => [ 'translation_id', 'translation_id' ],
            transcript => [ 'transcript_id', 'transcript_id' ],
            translation_attrib => [ 'translation_id', 'translation_id' ],
            translation_stable_id => [ 'translation_id', 'translation_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::Transcript; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::Transcript->table('transcript');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub gene {
    my $self = shift;
    return $self->gene_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    

sub display_xref {
    my $self = shift;
    return $self->display_xref_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::Transcript->columns( 
    All => qw/ 
        transcript_id
        gene_id
        analysis_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        display_xref_id
        biotype
        status
        description
        is_current 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::Transcript->columns( 
    Essential => qw/ 
        transcript_id
        gene_id
        analysis_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        display_xref_id
        biotype
        status
        description
        is_current 
    /
);

Gramene::CDBI::EnsemblArabidopsis::Transcript->columns( 
    Ordered => qw/ 
        transcript_id
        gene_id
        analysis_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        display_xref_id
        biotype
        status
        description
        is_current 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'transcript_id' => 'number',
        'gene_id' => 'number',
        'analysis_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'display_xref_id' => 'number',
        'biotype' => 'string',
        'status' => 'string',
        'description' => 'string',
        'is_current' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::Transcript->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblArabidopsis::Analysis'
);

Gramene::CDBI::EnsemblArabidopsis::Transcript->has_a(
    gene_id => 'Gramene::CDBI::EnsemblArabidopsis::Gene'
);

Gramene::CDBI::EnsemblArabidopsis::Transcript->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);

Gramene::CDBI::EnsemblArabidopsis::Transcript->has_a(
    display_xref_id => 'Gramene::CDBI::EnsemblArabidopsis::Xref'
);

Gramene::CDBI::EnsemblArabidopsis::Transcript->has_many( 
    'exon_transcripts',
    'Gramene::CDBI::EnsemblArabidopsis::ExonTranscript',
    'transcript_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Transcript->has_many( 
    'regulatory_factor_codings',
    'Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactorCoding',
    'transcript_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Transcript->has_many( 
    'transcript_attribs',
    'Gramene::CDBI::EnsemblArabidopsis::TranscriptAttrib',
    'transcript_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Transcript->has_many( 
    'transcript_stable_ids',
    'Gramene::CDBI::EnsemblArabidopsis::TranscriptStableId',
    'transcript_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Transcript->has_many( 
    'transcript_supporting_features',
    'Gramene::CDBI::EnsemblArabidopsis::TranscriptSupportingFeature',
    'transcript_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Transcript->has_many( 
    'translations',
    'Gramene::CDBI::EnsemblArabidopsis::Translation',
    'transcript_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            exon_transcript => [ 'transcript_id', 'transcript_id' ],
            gene => [ 'gene_id', 'gene_id' ],
            regulatory_factor_coding => [ 'transcript_id', 'transcript_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
            transcript_attrib => [ 'transcript_id', 'transcript_id' ],
            transcript_stable_id => [ 'transcript_id', 'transcript_id' ],
            transcript_supporting_feature => [ 'transcript_id', 'transcript_id' ],
            translation => [ 'transcript_id', 'transcript_id' ],
            xref => [ 'display_xref_id', 'xref_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::Gene; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::Gene->table('gene');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    

sub display_xref {
    my $self = shift;
    return $self->display_xref_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::Gene->columns( 
    All => qw/ 
        gene_id
        biotype
        analysis_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        display_xref_id
        source
        status
        description
        is_current 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::Gene->columns( 
    Essential => qw/ 
        gene_id
        biotype
        analysis_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        display_xref_id
        source
        status
        description
        is_current 
    /
);

Gramene::CDBI::EnsemblArabidopsis::Gene->columns( 
    Ordered => qw/ 
        gene_id
        biotype
        analysis_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        display_xref_id
        source
        status
        description
        is_current 
    /
);

sub to_string {
    my $self = shift;
    return $self->biotype || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_id' => 'number',
        'biotype' => 'string',
        'analysis_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'display_xref_id' => 'number',
        'source' => 'string',
        'status' => 'string',
        'description' => 'string',
        'is_current' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::Gene->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblArabidopsis::Analysis'
);

Gramene::CDBI::EnsemblArabidopsis::Gene->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);

Gramene::CDBI::EnsemblArabidopsis::Gene->has_a(
    display_xref_id => 'Gramene::CDBI::EnsemblArabidopsis::Xref'
);

Gramene::CDBI::EnsemblArabidopsis::Gene->has_many( 
    'alt_alleles',
    'Gramene::CDBI::EnsemblArabidopsis::AltAllele',
    'gene_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Gene->has_many( 
    'gene_attribs',
    'Gramene::CDBI::EnsemblArabidopsis::GeneAttrib',
    'gene_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Gene->has_many( 
    'gene_stable_ids',
    'Gramene::CDBI::EnsemblArabidopsis::GeneStableId',
    'gene_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Gene->has_many( 
    'regulatory_factor_codings',
    'Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactorCoding',
    'gene_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Gene->has_many( 
    'transcripts',
    'Gramene::CDBI::EnsemblArabidopsis::Transcript',
    'gene_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            alt_allele => [ 'gene_id', 'gene_id' ],
            analysis => [ 'analysis_id', 'analysis_id' ],
            gene_attrib => [ 'gene_id', 'gene_id' ],
            gene_stable_id => [ 'gene_id', 'gene_id' ],
            regulatory_factor_coding => [ 'gene_id', 'gene_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
            transcript => [ 'gene_id', 'gene_id' ],
            xref => [ 'display_xref_id', 'xref_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::IdentityXref; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::IdentityXref->table('identity_xref');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub object_xref {
    my $self = shift;
    return $self->object_xref_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::IdentityXref->columns( 
    All => qw/ 
        object_xref_id
        query_identity
        target_identity
        hit_start
        hit_end
        translation_start
        translation_end
        cigar_line
        score
        evalue
        analysis_id 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::IdentityXref->columns( 
    Essential => qw/ 
        object_xref_id
        query_identity
        target_identity
        hit_start
        hit_end
        translation_start
        translation_end
        cigar_line
        score
        evalue
        analysis_id 
    /
);

Gramene::CDBI::EnsemblArabidopsis::IdentityXref->columns( 
    Ordered => qw/ 
        object_xref_id
        query_identity
        target_identity
        hit_start
        hit_end
        translation_start
        translation_end
        cigar_line
        score
        evalue
        analysis_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->query_identity || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'object_xref_id' => 'number',
        'query_identity' => 'number',
        'target_identity' => 'number',
        'hit_start' => 'number',
        'hit_end' => 'number',
        'translation_start' => 'number',
        'translation_end' => 'number',
        'cigar_line' => 'string',
        'score' => 'number',
        'evalue' => 'number',
        'analysis_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::IdentityXref->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblArabidopsis::Analysis'
);

Gramene::CDBI::EnsemblArabidopsis::IdentityXref->has_a(
    object_xref_id => 'Gramene::CDBI::EnsemblArabidopsis::ObjectXref'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            object_xref => [ 'object_xref_id', 'object_xref_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::MarkerFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::MarkerFeature->table('marker_feature');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub marker {
    my $self = shift;
    return $self->marker_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::MarkerFeature->columns( 
    All => qw/ 
        marker_feature_id
        marker_id
        seq_region_id
        seq_region_start
        seq_region_end
        analysis_id
        map_weight 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::MarkerFeature->columns( 
    Essential => qw/ 
        marker_feature_id
        marker_id
        seq_region_id
        seq_region_start
        seq_region_end
        analysis_id
        map_weight 
    /
);

Gramene::CDBI::EnsemblArabidopsis::MarkerFeature->columns( 
    Ordered => qw/ 
        marker_feature_id
        marker_id
        seq_region_id
        seq_region_start
        seq_region_end
        analysis_id
        map_weight 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_feature_id' => 'number',
        'marker_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'analysis_id' => 'number',
        'map_weight' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::MarkerFeature->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblArabidopsis::Analysis'
);

Gramene::CDBI::EnsemblArabidopsis::MarkerFeature->has_a(
    marker_id => 'Gramene::CDBI::EnsemblArabidopsis::Marker'
);

Gramene::CDBI::EnsemblArabidopsis::MarkerFeature->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            marker => [ 'marker_id', 'marker_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::OligoFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::OligoFeature->table('oligo_feature');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub oligo_probe {
    my $self = shift;
    return $self->oligo_probe_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::OligoFeature->columns( 
    All => qw/ 
        oligo_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        mismatches
        oligo_probe_id
        analysis_id 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::OligoFeature->columns( 
    Essential => qw/ 
        oligo_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        mismatches
        oligo_probe_id
        analysis_id 
    /
);

Gramene::CDBI::EnsemblArabidopsis::OligoFeature->columns( 
    Ordered => qw/ 
        oligo_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        mismatches
        oligo_probe_id
        analysis_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'oligo_feature_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'mismatches' => 'string',
        'oligo_probe_id' => 'number',
        'analysis_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::OligoFeature->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblArabidopsis::Analysis'
);

Gramene::CDBI::EnsemblArabidopsis::OligoFeature->has_a(
    oligo_probe_id => 'Gramene::CDBI::EnsemblArabidopsis::OligoProbe'
);

Gramene::CDBI::EnsemblArabidopsis::OligoFeature->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            oligo_probe => [ 'oligo_probe_id', 'oligo_probe_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::PredictionExon; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::PredictionExon->table('prediction_exon');


sub prediction_transcript {
    my $self = shift;
    return $self->prediction_transcript_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::PredictionExon->columns( 
    All => qw/ 
        prediction_exon_id
        prediction_transcript_id
        exon_rank
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        start_phase
        score
        p_value 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::PredictionExon->columns( 
    Essential => qw/ 
        prediction_exon_id
        prediction_transcript_id
        exon_rank
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        start_phase
        score
        p_value 
    /
);

Gramene::CDBI::EnsemblArabidopsis::PredictionExon->columns( 
    Ordered => qw/ 
        prediction_exon_id
        prediction_transcript_id
        exon_rank
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        start_phase
        score
        p_value 
    /
);

sub to_string {
    my $self = shift;
    return $self->exon_rank || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'prediction_exon_id' => 'number',
        'prediction_transcript_id' => 'number',
        'exon_rank' => 'string',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'start_phase' => 'string',
        'score' => 'number',
        'p_value' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::PredictionExon->has_a(
    prediction_transcript_id => 'Gramene::CDBI::EnsemblArabidopsis::PredictionTranscript'
);

Gramene::CDBI::EnsemblArabidopsis::PredictionExon->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            prediction_transcript => [ 'prediction_transcript_id', 'prediction_transcript_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::PredictionTranscript; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::PredictionTranscript->table('prediction_transcript');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::PredictionTranscript->columns( 
    All => qw/ 
        prediction_transcript_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        analysis_id
        display_label 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::PredictionTranscript->columns( 
    Essential => qw/ 
        prediction_transcript_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        analysis_id
        display_label 
    /
);

Gramene::CDBI::EnsemblArabidopsis::PredictionTranscript->columns( 
    Ordered => qw/ 
        prediction_transcript_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        analysis_id
        display_label 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'prediction_transcript_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'analysis_id' => 'number',
        'display_label' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::PredictionTranscript->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblArabidopsis::Analysis'
);

Gramene::CDBI::EnsemblArabidopsis::PredictionTranscript->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);

Gramene::CDBI::EnsemblArabidopsis::PredictionTranscript->has_many( 
    'prediction_exons',
    'Gramene::CDBI::EnsemblArabidopsis::PredictionExon',
    'prediction_transcript_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            prediction_exon => [ 'prediction_transcript_id', 'prediction_transcript_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::ProteinAlignFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::ProteinAlignFeature->table('protein_align_feature');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::ProteinAlignFeature->columns( 
    All => qw/ 
        protein_align_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        hit_start
        hit_end
        hit_name
        analysis_id
        score
        evalue
        perc_ident
        cigar_line 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::ProteinAlignFeature->columns( 
    Essential => qw/ 
        protein_align_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        hit_start
        hit_end
        hit_name
        analysis_id
        score
        evalue
        perc_ident
        cigar_line 
    /
);

Gramene::CDBI::EnsemblArabidopsis::ProteinAlignFeature->columns( 
    Ordered => qw/ 
        protein_align_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        hit_start
        hit_end
        hit_name
        analysis_id
        score
        evalue
        perc_ident
        cigar_line 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'protein_align_feature_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'hit_start' => 'number',
        'hit_end' => 'number',
        'hit_name' => 'string',
        'analysis_id' => 'number',
        'score' => 'number',
        'evalue' => 'number',
        'perc_ident' => 'number',
        'cigar_line' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::ProteinAlignFeature->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblArabidopsis::Analysis'
);

Gramene::CDBI::EnsemblArabidopsis::ProteinAlignFeature->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::QtlFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::QtlFeature->table('qtl_feature');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub qtl {
    my $self = shift;
    return $self->qtl_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::QtlFeature->columns( 
    All => qw/ 
        seq_region_id
        seq_region_start
        seq_region_end
        qtl_id
        analysis_id 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::QtlFeature->columns( 
    Essential => qw/ 
        seq_region_id
        seq_region_start
        seq_region_end
        qtl_id
        analysis_id 
    /
);

Gramene::CDBI::EnsemblArabidopsis::QtlFeature->columns( 
    Ordered => qw/ 
        seq_region_id
        seq_region_start
        seq_region_end
        qtl_id
        analysis_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'qtl_id' => 'number',
        'analysis_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::QtlFeature->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblArabidopsis::Analysis'
);

Gramene::CDBI::EnsemblArabidopsis::QtlFeature->has_a(
    qtl_id => 'Gramene::CDBI::EnsemblArabidopsis::Qtl'
);

Gramene::CDBI::EnsemblArabidopsis::QtlFeature->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            qtl => [ 'qtl_id', 'qtl_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::RegulatoryFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::RegulatoryFeature->table('regulatory_feature');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub regulatory_factor {
    my $self = shift;
    return $self->regulatory_factor_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::RegulatoryFeature->columns( 
    All => qw/ 
        regulatory_feature_id
        name
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        analysis_id
        regulatory_factor_id 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::RegulatoryFeature->columns( 
    Essential => qw/ 
        regulatory_feature_id
        name
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        analysis_id
        regulatory_factor_id 
    /
);

Gramene::CDBI::EnsemblArabidopsis::RegulatoryFeature->columns( 
    Ordered => qw/ 
        regulatory_feature_id
        name
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        analysis_id
        regulatory_factor_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'regulatory_feature_id' => 'number',
        'name' => 'string',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'analysis_id' => 'number',
        'regulatory_factor_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::RegulatoryFeature->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblArabidopsis::Analysis'
);

Gramene::CDBI::EnsemblArabidopsis::RegulatoryFeature->has_a(
    regulatory_factor_id => 'Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactor'
);

Gramene::CDBI::EnsemblArabidopsis::RegulatoryFeature->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            regulatory_factor => [ 'regulatory_factor_id', 'regulatory_factor_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::RegulatorySearchRegion; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::RegulatorySearchRegion->table('regulatory_search_region');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::RegulatorySearchRegion->columns( 
    All => qw/ 
        regulatory_search_region_id
        name
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        ensembl_object_type
        ensembl_object_id
        analysis_id 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::RegulatorySearchRegion->columns( 
    Essential => qw/ 
        regulatory_search_region_id
        name
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        ensembl_object_type
        ensembl_object_id
        analysis_id 
    /
);

Gramene::CDBI::EnsemblArabidopsis::RegulatorySearchRegion->columns( 
    Ordered => qw/ 
        regulatory_search_region_id
        name
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        ensembl_object_type
        ensembl_object_id
        analysis_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'regulatory_search_region_id' => 'number',
        'name' => 'string',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'ensembl_object_type' => 'string',
        'ensembl_object_id' => 'number',
        'analysis_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::RegulatorySearchRegion->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblArabidopsis::Analysis'
);

Gramene::CDBI::EnsemblArabidopsis::RegulatorySearchRegion->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::RepeatFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::RepeatFeature->table('repeat_feature');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub repeat_consensus {
    my $self = shift;
    return $self->repeat_consensus_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::RepeatFeature->columns( 
    All => qw/ 
        repeat_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        repeat_start
        repeat_end
        repeat_consensus_id
        analysis_id
        score 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::RepeatFeature->columns( 
    Essential => qw/ 
        repeat_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        repeat_start
        repeat_end
        repeat_consensus_id
        analysis_id
        score 
    /
);

Gramene::CDBI::EnsemblArabidopsis::RepeatFeature->columns( 
    Ordered => qw/ 
        repeat_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        repeat_start
        repeat_end
        repeat_consensus_id
        analysis_id
        score 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'repeat_feature_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'repeat_start' => 'number',
        'repeat_end' => 'number',
        'repeat_consensus_id' => 'number',
        'analysis_id' => 'number',
        'score' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::RepeatFeature->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblArabidopsis::Analysis'
);

Gramene::CDBI::EnsemblArabidopsis::RepeatFeature->has_a(
    repeat_consensus_id => 'Gramene::CDBI::EnsemblArabidopsis::RepeatConsensus'
);

Gramene::CDBI::EnsemblArabidopsis::RepeatFeature->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            repeat_consensus => [ 'repeat_consensus_id', 'repeat_consensus_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::SimpleFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::SimpleFeature->table('simple_feature');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::SimpleFeature->columns( 
    All => qw/ 
        simple_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        display_label
        analysis_id
        score 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::SimpleFeature->columns( 
    Essential => qw/ 
        simple_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        display_label
        analysis_id
        score 
    /
);

Gramene::CDBI::EnsemblArabidopsis::SimpleFeature->columns( 
    Ordered => qw/ 
        simple_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        display_label
        analysis_id
        score 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'simple_feature_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'display_label' => 'string',
        'analysis_id' => 'number',
        'score' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::SimpleFeature->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblArabidopsis::Analysis'
);

Gramene::CDBI::EnsemblArabidopsis::SimpleFeature->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::UnmappedObject; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::UnmappedObject->table('unmapped_object');


sub analysis {
    my $self = shift;
    return $self->analysis_id;
}
    

sub external_db {
    my $self = shift;
    return $self->external_db_id;
}
    

sub unmapped_reason {
    my $self = shift;
    return $self->unmapped_reason_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::UnmappedObject->columns( 
    All => qw/ 
        unmapped_object_id
        type
        analysis_id
        external_db_id
        identifier
        unmapped_reason_id
        query_score
        target_score
        ensembl_id
        ensembl_object_type 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::UnmappedObject->columns( 
    Essential => qw/ 
        unmapped_object_id
        type
        analysis_id
        external_db_id
        identifier
        unmapped_reason_id
        query_score
        target_score
        ensembl_id
        ensembl_object_type 
    /
);

Gramene::CDBI::EnsemblArabidopsis::UnmappedObject->columns( 
    Ordered => qw/ 
        unmapped_object_id
        type
        analysis_id
        external_db_id
        identifier
        unmapped_reason_id
        query_score
        target_score
        ensembl_id
        ensembl_object_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'unmapped_object_id' => 'number',
        'type' => 'string',
        'analysis_id' => 'number',
        'external_db_id' => 'number',
        'identifier' => 'string',
        'unmapped_reason_id' => 'string',
        'query_score' => 'number',
        'target_score' => 'number',
        'ensembl_id' => 'number',
        'ensembl_object_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::UnmappedObject->has_a(
    analysis_id => 'Gramene::CDBI::EnsemblArabidopsis::Analysis'
);

Gramene::CDBI::EnsemblArabidopsis::UnmappedObject->has_a(
    external_db_id => 'Gramene::CDBI::EnsemblArabidopsis::ExternalDb'
);

Gramene::CDBI::EnsemblArabidopsis::UnmappedObject->has_a(
    unmapped_reason_id => 'Gramene::CDBI::EnsemblArabidopsis::UnmappedReason'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis => [ 'analysis_id', 'analysis_id' ],
            external_db => [ 'external_db_id', 'external_db_id' ],
            unmapped_reason => [ 'unmapped_reason_id', 'unmapped_reason_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::Analysis; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::Analysis->table('analysis');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::Analysis->columns( 
    All => qw/ 
        analysis_id
        created
        logic_name
        db
        db_version
        db_file
        program
        program_version
        program_file
        parameters
        module
        module_version
        gff_source
        gff_feature 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::Analysis->columns( 
    Essential => qw/ 
        analysis_id
        created
        logic_name
        db
        db_version
        db_file
        program
        program_version
        program_file
        parameters
        module
        module_version
        gff_source
        gff_feature 
    /
);

Gramene::CDBI::EnsemblArabidopsis::Analysis->columns( 
    Ordered => qw/ 
        analysis_id
        created
        logic_name
        db
        db_version
        db_file
        program
        program_version
        program_file
        parameters
        module
        module_version
        gff_source
        gff_feature 
    /
);

sub to_string {
    my $self = shift;
    return $self->created || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'analysis_id' => 'number',
        'created' => 'string',
        'logic_name' => 'string',
        'db' => 'string',
        'db_version' => 'string',
        'db_file' => 'string',
        'program' => 'string',
        'program_version' => 'string',
        'program_file' => 'string',
        'parameters' => 'string',
        'module' => 'string',
        'module_version' => 'string',
        'gff_source' => 'string',
        'gff_feature' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::Analysis->has_many( 
    'analysis_descriptions',
    'Gramene::CDBI::EnsemblArabidopsis::AnalysisDescription',
    'analysis_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Analysis->has_many( 
    'density_types',
    'Gramene::CDBI::EnsemblArabidopsis::DensityType',
    'analysis_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Analysis->has_many( 
    'ditag_features',
    'Gramene::CDBI::EnsemblArabidopsis::DitagFeature',
    'analysis_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Analysis->has_many( 
    'dna_align_features',
    'Gramene::CDBI::EnsemblArabidopsis::DnaAlignFeature',
    'analysis_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Analysis->has_many( 
    'genes',
    'Gramene::CDBI::EnsemblArabidopsis::Gene',
    'analysis_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Analysis->has_many( 
    'identity_xrefs',
    'Gramene::CDBI::EnsemblArabidopsis::IdentityXref',
    'analysis_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Analysis->has_many( 
    'marker_features',
    'Gramene::CDBI::EnsemblArabidopsis::MarkerFeature',
    'analysis_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Analysis->has_many( 
    'oligo_features',
    'Gramene::CDBI::EnsemblArabidopsis::OligoFeature',
    'analysis_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Analysis->has_many( 
    'prediction_transcripts',
    'Gramene::CDBI::EnsemblArabidopsis::PredictionTranscript',
    'analysis_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Analysis->has_many( 
    'protein_align_features',
    'Gramene::CDBI::EnsemblArabidopsis::ProteinAlignFeature',
    'analysis_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Analysis->has_many( 
    'protein_features',
    'Gramene::CDBI::EnsemblArabidopsis::ProteinFeature',
    'analysis_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Analysis->has_many( 
    'qtl_features',
    'Gramene::CDBI::EnsemblArabidopsis::QtlFeature',
    'analysis_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Analysis->has_many( 
    'regulatory_features',
    'Gramene::CDBI::EnsemblArabidopsis::RegulatoryFeature',
    'analysis_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Analysis->has_many( 
    'regulatory_search_regions',
    'Gramene::CDBI::EnsemblArabidopsis::RegulatorySearchRegion',
    'analysis_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Analysis->has_many( 
    'repeat_features',
    'Gramene::CDBI::EnsemblArabidopsis::RepeatFeature',
    'analysis_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Analysis->has_many( 
    'simple_features',
    'Gramene::CDBI::EnsemblArabidopsis::SimpleFeature',
    'analysis_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Analysis->has_many( 
    'transcripts',
    'Gramene::CDBI::EnsemblArabidopsis::Transcript',
    'analysis_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Analysis->has_many( 
    'unmapped_objects',
    'Gramene::CDBI::EnsemblArabidopsis::UnmappedObject',
    'analysis_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            analysis_description => [ 'analysis_id', 'analysis_id' ],
            density_type => [ 'analysis_id', 'analysis_id' ],
            ditag_feature => [ 'analysis_id', 'analysis_id' ],
            dna_align_feature => [ 'analysis_id', 'analysis_id' ],
            gene => [ 'analysis_id', 'analysis_id' ],
            identity_xref => [ 'analysis_id', 'analysis_id' ],
            marker_feature => [ 'analysis_id', 'analysis_id' ],
            oligo_feature => [ 'analysis_id', 'analysis_id' ],
            prediction_transcript => [ 'analysis_id', 'analysis_id' ],
            protein_align_feature => [ 'analysis_id', 'analysis_id' ],
            protein_feature => [ 'analysis_id', 'analysis_id' ],
            qtl_feature => [ 'analysis_id', 'analysis_id' ],
            regulatory_feature => [ 'analysis_id', 'analysis_id' ],
            regulatory_search_region => [ 'analysis_id', 'analysis_id' ],
            repeat_feature => [ 'analysis_id', 'analysis_id' ],
            simple_feature => [ 'analysis_id', 'analysis_id' ],
            transcript => [ 'analysis_id', 'analysis_id' ],
            unmapped_object => [ 'analysis_id', 'analysis_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::Assembly; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::Assembly->table('assembly');


sub asm_seq_region {
    my $self = shift;
    return $self->asm_seq_region_id;
}
    
sub cmp_seq_region {
    my $self = shift;
    return $self->cmp_seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::Assembly->columns( 
    All => qw/ 
        asm_seq_region_id
        cmp_seq_region_id
        asm_start
        asm_end
        cmp_start
        cmp_end
        ori 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::Assembly->columns( 
    Essential => qw/ 
        asm_seq_region_id
        cmp_seq_region_id
        asm_start
        asm_end
        cmp_start
        cmp_end
        ori 
    /
);

Gramene::CDBI::EnsemblArabidopsis::Assembly->columns( 
    Ordered => qw/ 
        asm_seq_region_id
        cmp_seq_region_id
        asm_start
        asm_end
        cmp_start
        cmp_end
        ori 
    /
);

sub to_string {
    my $self = shift;
    return $self->asm_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'asm_seq_region_id' => 'number',
        'cmp_seq_region_id' => 'number',
        'asm_start' => 'number',
        'asm_end' => 'number',
        'cmp_start' => 'number',
        'cmp_end' => 'number',
        'ori' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::Assembly->has_a(
    asm_seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);

Gramene::CDBI::EnsemblArabidopsis::Assembly->has_a(
    cmp_seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            seq_region => [ 'cmp_seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::AssemblyException; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::AssemblyException->table('assembly_exception');


sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
sub exc_seq_region {
    my $self = shift;
    return $self->exc_seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::AssemblyException->columns( 
    All => qw/ 
        assembly_exception_id
        seq_region_id
        seq_region_start
        seq_region_end
        exc_type
        exc_seq_region_id
        exc_seq_region_start
        exc_seq_region_end
        ori 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::AssemblyException->columns( 
    Essential => qw/ 
        assembly_exception_id
        seq_region_id
        seq_region_start
        seq_region_end
        exc_type
        exc_seq_region_id
        exc_seq_region_start
        exc_seq_region_end
        ori 
    /
);

Gramene::CDBI::EnsemblArabidopsis::AssemblyException->columns( 
    Ordered => qw/ 
        assembly_exception_id
        seq_region_id
        seq_region_start
        seq_region_end
        exc_type
        exc_seq_region_id
        exc_seq_region_start
        exc_seq_region_end
        ori 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'assembly_exception_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'exc_type' => 'string',
        'exc_seq_region_id' => 'number',
        'exc_seq_region_start' => 'number',
        'exc_seq_region_end' => 'number',
        'ori' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::AssemblyException->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);

Gramene::CDBI::EnsemblArabidopsis::AssemblyException->has_a(
    exc_seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            seq_region => [ 'exc_seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::MiscAttrib; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::MiscAttrib->table('misc_attrib');


sub attrib_type {
    my $self = shift;
    return $self->attrib_type_id;
}
    

sub misc_feature {
    my $self = shift;
    return $self->misc_feature_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::MiscAttrib->columns( 
    All => qw/ 
        misc_feature_id
        attrib_type_id
        value 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::MiscAttrib->columns( 
    Essential => qw/ 
        misc_feature_id
        attrib_type_id
        value 
    /
);

Gramene::CDBI::EnsemblArabidopsis::MiscAttrib->columns( 
    Ordered => qw/ 
        misc_feature_id
        attrib_type_id
        value 
    /
);

sub to_string {
    my $self = shift;
    return $self->value || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'misc_feature_id' => 'number',
        'attrib_type_id' => 'string',
        'value' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::MiscAttrib->has_a(
    attrib_type_id => 'Gramene::CDBI::EnsemblArabidopsis::AttribType'
);

Gramene::CDBI::EnsemblArabidopsis::MiscAttrib->has_a(
    misc_feature_id => 'Gramene::CDBI::EnsemblArabidopsis::MiscFeature'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            attrib_type => [ 'attrib_type_id', 'attrib_type_id' ],
            misc_feature => [ 'misc_feature_id', 'misc_feature_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::SeqRegionAttrib; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::SeqRegionAttrib->table('seq_region_attrib');


sub attrib_type {
    my $self = shift;
    return $self->attrib_type_id;
}
    

sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::SeqRegionAttrib->columns( 
    All => qw/ 
        seq_region_id
        attrib_type_id
        value 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegionAttrib->columns( 
    Essential => qw/ 
        seq_region_id
        attrib_type_id
        value 
    /
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegionAttrib->columns( 
    Ordered => qw/ 
        seq_region_id
        attrib_type_id
        value 
    /
);

sub to_string {
    my $self = shift;
    return $self->value || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'seq_region_id' => 'number',
        'attrib_type_id' => 'string',
        'value' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::SeqRegionAttrib->has_a(
    attrib_type_id => 'Gramene::CDBI::EnsemblArabidopsis::AttribType'
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegionAttrib->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            attrib_type => [ 'attrib_type_id', 'attrib_type_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::AttribType; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::AttribType->table('attrib_type');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::AttribType->columns( 
    All => qw/ 
        attrib_type_id
        code
        name
        description 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::AttribType->columns( 
    Essential => qw/ 
        attrib_type_id
        code
        name
        description 
    /
);

Gramene::CDBI::EnsemblArabidopsis::AttribType->columns( 
    Ordered => qw/ 
        attrib_type_id
        code
        name
        description 
    /
);

sub to_string {
    my $self = shift;
    return $self->code || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'attrib_type_id' => 'string',
        'code' => 'string',
        'name' => 'string',
        'description' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::AttribType->has_many( 
    'gene_attribs',
    'Gramene::CDBI::EnsemblArabidopsis::GeneAttrib',
    'attrib_type_id' 
);

Gramene::CDBI::EnsemblArabidopsis::AttribType->has_many( 
    'misc_attribs',
    'Gramene::CDBI::EnsemblArabidopsis::MiscAttrib',
    'attrib_type_id' 
);

Gramene::CDBI::EnsemblArabidopsis::AttribType->has_many( 
    'seq_region_attribs',
    'Gramene::CDBI::EnsemblArabidopsis::SeqRegionAttrib',
    'attrib_type_id' 
);

Gramene::CDBI::EnsemblArabidopsis::AttribType->has_many( 
    'transcript_attribs',
    'Gramene::CDBI::EnsemblArabidopsis::TranscriptAttrib',
    'attrib_type_id' 
);

Gramene::CDBI::EnsemblArabidopsis::AttribType->has_many( 
    'translation_attribs',
    'Gramene::CDBI::EnsemblArabidopsis::TranslationAttrib',
    'attrib_type_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_attrib => [ 'attrib_type_id', 'attrib_type_id' ],
            misc_attrib => [ 'attrib_type_id', 'attrib_type_id' ],
            seq_region_attrib => [ 'attrib_type_id', 'attrib_type_id' ],
            transcript_attrib => [ 'attrib_type_id', 'attrib_type_id' ],
            translation_attrib => [ 'attrib_type_id', 'attrib_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::MetaCoord; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::MetaCoord->table('meta_coord');


sub coord_system {
    my $self = shift;
    return $self->coord_system_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::MetaCoord->columns( 
    All => qw/ 
        table_name
        coord_system_id
        max_length 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::MetaCoord->columns( 
    Essential => qw/ 
        table_name
        coord_system_id
        max_length 
    /
);

Gramene::CDBI::EnsemblArabidopsis::MetaCoord->columns( 
    Ordered => qw/ 
        table_name
        coord_system_id
        max_length 
    /
);

sub to_string {
    my $self = shift;
    return $self->table_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'table_name' => 'string',
        'coord_system_id' => 'number',
        'max_length' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::MetaCoord->has_a(
    coord_system_id => 'Gramene::CDBI::EnsemblArabidopsis::CoordSystem'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            coord_system => [ 'coord_system_id', 'coord_system_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::Dna; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::Dna->table('dna');


sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
push @extras, Class::DBI::Column->new( sequence => { 
    accessor => 'seq',
    mutator  => 'seq',
}); 

Gramene::CDBI::EnsemblArabidopsis::Dna->columns( 
    All => qw/ 
        seq_region_id 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::Dna->columns( 
    Essential => qw/ 
        seq_region_id 
    /
);

Gramene::CDBI::EnsemblArabidopsis::Dna->columns( 
    Ordered => qw/ 
        seq_region_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->sequence || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'seq_region_id' => 'number',
        'sequence' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::Dna->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::Dnac; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::Dnac->table('dnac');


sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
push @extras, Class::DBI::Column->new( sequence => { 
    accessor => 'seq',
    mutator  => 'seq',
}); 

Gramene::CDBI::EnsemblArabidopsis::Dnac->columns( 
    All => qw/ 
        seq_region_id
        n_line 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::Dnac->columns( 
    Essential => qw/ 
        seq_region_id
        n_line 
    /
);

Gramene::CDBI::EnsemblArabidopsis::Dnac->columns( 
    Ordered => qw/ 
        seq_region_id
        n_line 
    /
);

sub to_string {
    my $self = shift;
    return $self->sequence || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'seq_region_id' => 'number',
        'sequence' => 'string',
        'n_line' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::Dnac->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::ExonStableId; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::ExonStableId->table('exon_stable_id');


sub exon {
    my $self = shift;
    return $self->exon_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::ExonStableId->columns( 
    All => qw/ 
        exon_id
        stable_id
        version
        created_date
        modified_date 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::ExonStableId->columns( 
    Essential => qw/ 
        exon_id
        stable_id
        version
        created_date
        modified_date 
    /
);

Gramene::CDBI::EnsemblArabidopsis::ExonStableId->columns( 
    Ordered => qw/ 
        exon_id
        stable_id
        version
        created_date
        modified_date 
    /
);

sub to_string {
    my $self = shift;
    return $self->version || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'exon_id' => 'number',
        'stable_id' => 'string',
        'version' => 'number',
        'created_date' => 'string',
        'modified_date' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::ExonStableId->has_a(
    exon_id => 'Gramene::CDBI::EnsemblArabidopsis::Exon'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            exon => [ 'exon_id', 'exon_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::SupportingFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::SupportingFeature->table('supporting_feature');


sub exon {
    my $self = shift;
    return $self->exon_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::SupportingFeature->columns( 
    All => qw/ 
        exon_id
        feature_type
        feature_id 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::SupportingFeature->columns( 
    Essential => qw/ 
        exon_id
        feature_type
        feature_id 
    /
);

Gramene::CDBI::EnsemblArabidopsis::SupportingFeature->columns( 
    Ordered => qw/ 
        exon_id
        feature_type
        feature_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->feature_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'exon_id' => 'number',
        'feature_type' => 'string',
        'feature_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::SupportingFeature->has_a(
    exon_id => 'Gramene::CDBI::EnsemblArabidopsis::Exon'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            exon => [ 'exon_id', 'exon_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::Exon; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::Exon->table('exon');


sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::Exon->columns( 
    All => qw/ 
        exon_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        phase
        end_phase
        is_current 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::Exon->columns( 
    Essential => qw/ 
        exon_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        phase
        end_phase
        is_current 
    /
);

Gramene::CDBI::EnsemblArabidopsis::Exon->columns( 
    Ordered => qw/ 
        exon_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand
        phase
        end_phase
        is_current 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'exon_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
        'phase' => 'string',
        'end_phase' => 'string',
        'is_current' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::Exon->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);

Gramene::CDBI::EnsemblArabidopsis::Exon->has_many( 
    'exon_stable_ids',
    'Gramene::CDBI::EnsemblArabidopsis::ExonStableId',
    'exon_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Exon->has_many( 
    'exon_transcripts',
    'Gramene::CDBI::EnsemblArabidopsis::ExonTranscript',
    'exon_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Exon->has_many( 
    'supporting_features',
    'Gramene::CDBI::EnsemblArabidopsis::SupportingFeature',
    'exon_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Exon->has_many( 
    'translations_start_exon',
    'Gramene::CDBI::EnsemblArabidopsis::Translation',
    'start_exon_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Exon->has_many( 
    'translations_end_exon',
    'Gramene::CDBI::EnsemblArabidopsis::Translation',
    'end_exon_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            exon_stable_id => [ 'exon_id', 'exon_id' ],
            exon_transcript => [ 'exon_id', 'exon_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
            supporting_feature => [ 'exon_id', 'exon_id' ],
            translation => [ 'exon_id', 'end_exon_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::Karyotype; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::Karyotype->table('karyotype');


sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::Karyotype->columns( 
    All => qw/ 
        karyotype_id
        seq_region_id
        seq_region_start
        seq_region_end
        band
        stain 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::Karyotype->columns( 
    Essential => qw/ 
        karyotype_id
        seq_region_id
        seq_region_start
        seq_region_end
        band
        stain 
    /
);

Gramene::CDBI::EnsemblArabidopsis::Karyotype->columns( 
    Ordered => qw/ 
        karyotype_id
        seq_region_id
        seq_region_start
        seq_region_end
        band
        stain 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'karyotype_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'band' => 'string',
        'stain' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::Karyotype->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::MiscFeatureMiscSet; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::MiscFeatureMiscSet->table('misc_feature_misc_set');


sub misc_feature {
    my $self = shift;
    return $self->misc_feature_id;
}
    

sub misc_set {
    my $self = shift;
    return $self->misc_set_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::MiscFeatureMiscSet->columns( 
    All => qw/ 
        misc_feature_id
        misc_set_id 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::MiscFeatureMiscSet->columns( 
    Essential => qw/ 
        misc_feature_id
        misc_set_id 
    /
);

Gramene::CDBI::EnsemblArabidopsis::MiscFeatureMiscSet->columns( 
    Ordered => qw/ 
        misc_feature_id
        misc_set_id 
    /
);


Gramene::CDBI::EnsemblArabidopsis::MiscFeatureMiscSet->columns( 
    Primary => qw/ 
        misc_feature_id
        misc_set_id 
    / 
);


sub to_string {
    my $self = shift;
    return $self->misc_feature_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'misc_feature_id' => 'number',
        'misc_set_id' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::MiscFeatureMiscSet->has_a(
    misc_feature_id => 'Gramene::CDBI::EnsemblArabidopsis::MiscFeature'
);

Gramene::CDBI::EnsemblArabidopsis::MiscFeatureMiscSet->has_a(
    misc_set_id => 'Gramene::CDBI::EnsemblArabidopsis::MiscSet'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            misc_feature => [ 'misc_feature_id', 'misc_feature_id' ],
            misc_set => [ 'misc_set_id', 'misc_set_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::MiscFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::MiscFeature->table('misc_feature');


sub seq_region {
    my $self = shift;
    return $self->seq_region_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::MiscFeature->columns( 
    All => qw/ 
        misc_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::MiscFeature->columns( 
    Essential => qw/ 
        misc_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand 
    /
);

Gramene::CDBI::EnsemblArabidopsis::MiscFeature->columns( 
    Ordered => qw/ 
        misc_feature_id
        seq_region_id
        seq_region_start
        seq_region_end
        seq_region_strand 
    /
);

sub to_string {
    my $self = shift;
    return $self->seq_region_start || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'misc_feature_id' => 'number',
        'seq_region_id' => 'number',
        'seq_region_start' => 'number',
        'seq_region_end' => 'number',
        'seq_region_strand' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::MiscFeature->has_a(
    seq_region_id => 'Gramene::CDBI::EnsemblArabidopsis::SeqRegion'
);

Gramene::CDBI::EnsemblArabidopsis::MiscFeature->has_many( 
    'misc_attribs',
    'Gramene::CDBI::EnsemblArabidopsis::MiscAttrib',
    'misc_feature_id' 
);

Gramene::CDBI::EnsemblArabidopsis::MiscFeature->has_many( 
    'misc_feature_misc_sets',
    'Gramene::CDBI::EnsemblArabidopsis::MiscFeatureMiscSet',
    'misc_feature_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            misc_attrib => [ 'misc_feature_id', 'misc_feature_id' ],
            misc_feature_misc_set => [ 'misc_feature_id', 'misc_feature_id' ],
            seq_region => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::SeqRegion; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->table('seq_region');


sub coord_system {
    my $self = shift;
    return $self->coord_system_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::SeqRegion->columns( 
    All => qw/ 
        seq_region_id
        name
        coord_system_id
        length 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->columns( 
    Essential => qw/ 
        seq_region_id
        name
        coord_system_id
        length 
    /
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->columns( 
    Ordered => qw/ 
        seq_region_id
        name
        coord_system_id
        length 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'seq_region_id' => 'number',
        'name' => 'string',
        'coord_system_id' => 'number',
        'length' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_a(
    coord_system_id => 'Gramene::CDBI::EnsemblArabidopsis::CoordSystem'
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'assemblies_asm_seq_region',
    'Gramene::CDBI::EnsemblArabidopsis::Assembly',
    'asm_seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'assemblies_cmp_seq_region',
    'Gramene::CDBI::EnsemblArabidopsis::Assembly',
    'cmp_seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'assembly_exceptions_seq_region',
    'Gramene::CDBI::EnsemblArabidopsis::AssemblyException',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'assembly_exceptions_exc_seq_region',
    'Gramene::CDBI::EnsemblArabidopsis::AssemblyException',
    'exc_seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'density_features',
    'Gramene::CDBI::EnsemblArabidopsis::DensityFeature',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'ditag_features',
    'Gramene::CDBI::EnsemblArabidopsis::DitagFeature',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'dnas',
    'Gramene::CDBI::EnsemblArabidopsis::Dna',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'dna_align_features',
    'Gramene::CDBI::EnsemblArabidopsis::DnaAlignFeature',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'dnacs',
    'Gramene::CDBI::EnsemblArabidopsis::Dnac',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'exons',
    'Gramene::CDBI::EnsemblArabidopsis::Exon',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'genes',
    'Gramene::CDBI::EnsemblArabidopsis::Gene',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'karyotypes',
    'Gramene::CDBI::EnsemblArabidopsis::Karyotype',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'marker_features',
    'Gramene::CDBI::EnsemblArabidopsis::MarkerFeature',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'misc_features',
    'Gramene::CDBI::EnsemblArabidopsis::MiscFeature',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'oligo_features',
    'Gramene::CDBI::EnsemblArabidopsis::OligoFeature',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'prediction_exons',
    'Gramene::CDBI::EnsemblArabidopsis::PredictionExon',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'prediction_transcripts',
    'Gramene::CDBI::EnsemblArabidopsis::PredictionTranscript',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'protein_align_features',
    'Gramene::CDBI::EnsemblArabidopsis::ProteinAlignFeature',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'qtl_features',
    'Gramene::CDBI::EnsemblArabidopsis::QtlFeature',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'regulatory_features',
    'Gramene::CDBI::EnsemblArabidopsis::RegulatoryFeature',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'regulatory_search_regions',
    'Gramene::CDBI::EnsemblArabidopsis::RegulatorySearchRegion',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'repeat_features',
    'Gramene::CDBI::EnsemblArabidopsis::RepeatFeature',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'seq_region_attribs',
    'Gramene::CDBI::EnsemblArabidopsis::SeqRegionAttrib',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'simple_features',
    'Gramene::CDBI::EnsemblArabidopsis::SimpleFeature',
    'seq_region_id' 
);

Gramene::CDBI::EnsemblArabidopsis::SeqRegion->has_many( 
    'transcripts',
    'Gramene::CDBI::EnsemblArabidopsis::Transcript',
    'seq_region_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            assembly => [ 'seq_region_id', 'cmp_seq_region_id' ],
            assembly_exception => [ 'seq_region_id', 'exc_seq_region_id' ],
            coord_system => [ 'coord_system_id', 'coord_system_id' ],
            density_feature => [ 'seq_region_id', 'seq_region_id' ],
            ditag_feature => [ 'seq_region_id', 'seq_region_id' ],
            dna => [ 'seq_region_id', 'seq_region_id' ],
            dna_align_feature => [ 'seq_region_id', 'seq_region_id' ],
            dnac => [ 'seq_region_id', 'seq_region_id' ],
            exon => [ 'seq_region_id', 'seq_region_id' ],
            gene => [ 'seq_region_id', 'seq_region_id' ],
            karyotype => [ 'seq_region_id', 'seq_region_id' ],
            marker_feature => [ 'seq_region_id', 'seq_region_id' ],
            misc_feature => [ 'seq_region_id', 'seq_region_id' ],
            oligo_feature => [ 'seq_region_id', 'seq_region_id' ],
            prediction_exon => [ 'seq_region_id', 'seq_region_id' ],
            prediction_transcript => [ 'seq_region_id', 'seq_region_id' ],
            protein_align_feature => [ 'seq_region_id', 'seq_region_id' ],
            qtl_feature => [ 'seq_region_id', 'seq_region_id' ],
            regulatory_feature => [ 'seq_region_id', 'seq_region_id' ],
            regulatory_search_region => [ 'seq_region_id', 'seq_region_id' ],
            repeat_feature => [ 'seq_region_id', 'seq_region_id' ],
            seq_region_attrib => [ 'seq_region_id', 'seq_region_id' ],
            simple_feature => [ 'seq_region_id', 'seq_region_id' ],
            transcript => [ 'seq_region_id', 'seq_region_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::CoordSystem; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::CoordSystem->table('coord_system');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::CoordSystem->columns( 
    All => qw/ 
        coord_system_id
        name
        version
        rank
        attrib 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::CoordSystem->columns( 
    Essential => qw/ 
        coord_system_id
        name
        version
        rank
        attrib 
    /
);

Gramene::CDBI::EnsemblArabidopsis::CoordSystem->columns( 
    Ordered => qw/ 
        coord_system_id
        name
        version
        rank
        attrib 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'coord_system_id' => 'number',
        'name' => 'string',
        'version' => 'string',
        'rank' => 'number',
        'attrib' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::CoordSystem->has_many( 
    'meta_coords',
    'Gramene::CDBI::EnsemblArabidopsis::MetaCoord',
    'coord_system_id' 
);

Gramene::CDBI::EnsemblArabidopsis::CoordSystem->has_many( 
    'seq_regions',
    'Gramene::CDBI::EnsemblArabidopsis::SeqRegion',
    'coord_system_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            meta_coord => [ 'coord_system_id', 'coord_system_id' ],
            seq_region => [ 'coord_system_id', 'coord_system_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::Ditag; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::Ditag->table('ditag');

our @extras = ();
push @extras, Class::DBI::Column->new( sequence => { 
    accessor => 'seq',
    mutator  => 'seq',
}); 

Gramene::CDBI::EnsemblArabidopsis::Ditag->columns( 
    All => qw/ 
        ditag_id
        name
        type
        tag_count 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::Ditag->columns( 
    Essential => qw/ 
        ditag_id
        name
        type
        tag_count 
    /
);

Gramene::CDBI::EnsemblArabidopsis::Ditag->columns( 
    Ordered => qw/ 
        ditag_id
        name
        type
        tag_count 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'ditag_id' => 'number',
        'name' => 'string',
        'type' => 'string',
        'tag_count' => 'string',
        'sequence' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::Ditag->has_many( 
    'ditag_features',
    'Gramene::CDBI::EnsemblArabidopsis::DitagFeature',
    'ditag_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            ditag_feature => [ 'ditag_id', 'ditag_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::EnsemblGeneHelper; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::EnsemblGeneHelper->table('ensembl_gene_helper');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::EnsemblGeneHelper->columns( 
    All => qw/ 
        gene_id
        description 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::EnsemblGeneHelper->columns( 
    Essential => qw/ 
        gene_id
        description 
    /
);

Gramene::CDBI::EnsemblArabidopsis::EnsemblGeneHelper->columns( 
    Ordered => qw/ 
        gene_id
        description 
    /
);

sub to_string {
    my $self = shift;
    return $self->description || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_id' => 'number',
        'description' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::ExternalSynonym; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::ExternalSynonym->table('external_synonym');


sub xref {
    my $self = shift;
    return $self->xref_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::ExternalSynonym->columns( 
    All => qw/ 
        xref_id
        synonym 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::ExternalSynonym->columns( 
    Essential => qw/ 
        xref_id
        synonym 
    /
);

Gramene::CDBI::EnsemblArabidopsis::ExternalSynonym->columns( 
    Ordered => qw/ 
        xref_id
        synonym 
    /
);


Gramene::CDBI::EnsemblArabidopsis::ExternalSynonym->columns( 
    Primary => qw/ 
        xref_id
        synonym 
    / 
);


sub to_string {
    my $self = shift;
    return $self->xref_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'xref_id' => 'number',
        'synonym' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::ExternalSynonym->has_a(
    xref_id => 'Gramene::CDBI::EnsemblArabidopsis::Xref'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            xref => [ 'xref_id', 'xref_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::GoXref; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::GoXref->table('go_xref');


sub object_xref {
    my $self = shift;
    return $self->object_xref_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::GoXref->columns( 
    All => qw/ 
        object_xref_id
        linkage_type 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::GoXref->columns( 
    Essential => qw/ 
        object_xref_id
        linkage_type 
    /
);

Gramene::CDBI::EnsemblArabidopsis::GoXref->columns( 
    Ordered => qw/ 
        object_xref_id
        linkage_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->linkage_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'object_xref_id' => 'number',
        'linkage_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::GoXref->has_a(
    object_xref_id => 'Gramene::CDBI::EnsemblArabidopsis::ObjectXref'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            object_xref => [ 'object_xref_id', 'object_xref_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::ObjectXref; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::ObjectXref->table('object_xref');


sub xref {
    my $self = shift;
    return $self->xref_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::ObjectXref->columns( 
    All => qw/ 
        object_xref_id
        ensembl_id
        ensembl_object_type
        xref_id 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::ObjectXref->columns( 
    Essential => qw/ 
        object_xref_id
        ensembl_id
        ensembl_object_type
        xref_id 
    /
);

Gramene::CDBI::EnsemblArabidopsis::ObjectXref->columns( 
    Ordered => qw/ 
        object_xref_id
        ensembl_id
        ensembl_object_type
        xref_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->ensembl_object_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'object_xref_id' => 'number',
        'ensembl_id' => 'number',
        'ensembl_object_type' => 'string',
        'xref_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::ObjectXref->has_a(
    xref_id => 'Gramene::CDBI::EnsemblArabidopsis::Xref'
);

Gramene::CDBI::EnsemblArabidopsis::ObjectXref->has_many( 
    'go_xrefs',
    'Gramene::CDBI::EnsemblArabidopsis::GoXref',
    'object_xref_id' 
);

Gramene::CDBI::EnsemblArabidopsis::ObjectXref->has_many( 
    'identity_xrefs',
    'Gramene::CDBI::EnsemblArabidopsis::IdentityXref',
    'object_xref_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            go_xref => [ 'object_xref_id', 'object_xref_id' ],
            identity_xref => [ 'object_xref_id', 'object_xref_id' ],
            xref => [ 'xref_id', 'xref_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::Xref; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::Xref->table('xref');


sub external_db {
    my $self = shift;
    return $self->external_db_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::Xref->columns( 
    All => qw/ 
        xref_id
        external_db_id
        dbprimary_acc
        display_label
        version
        description
        info_type
        info_text 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::Xref->columns( 
    Essential => qw/ 
        xref_id
        external_db_id
        dbprimary_acc
        display_label
        version
        description
        info_type
        info_text 
    /
);

Gramene::CDBI::EnsemblArabidopsis::Xref->columns( 
    Ordered => qw/ 
        xref_id
        external_db_id
        dbprimary_acc
        display_label
        version
        description
        info_type
        info_text 
    /
);

sub to_string {
    my $self = shift;
    return $self->dbprimary_acc || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'xref_id' => 'number',
        'external_db_id' => 'number',
        'dbprimary_acc' => 'string',
        'display_label' => 'string',
        'version' => 'string',
        'description' => 'string',
        'info_type' => 'string',
        'info_text' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::Xref->has_a(
    external_db_id => 'Gramene::CDBI::EnsemblArabidopsis::ExternalDb'
);

Gramene::CDBI::EnsemblArabidopsis::Xref->has_many( 
    'external_synonyms',
    'Gramene::CDBI::EnsemblArabidopsis::ExternalSynonym',
    'xref_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Xref->has_many( 
    'genes',
    'Gramene::CDBI::EnsemblArabidopsis::Gene',
    'display_xref_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Xref->has_many( 
    'object_xrefs',
    'Gramene::CDBI::EnsemblArabidopsis::ObjectXref',
    'xref_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Xref->has_many( 
    'transcripts',
    'Gramene::CDBI::EnsemblArabidopsis::Transcript',
    'display_xref_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            external_db => [ 'external_db_id', 'external_db_id' ],
            external_synonym => [ 'xref_id', 'xref_id' ],
            gene => [ 'xref_id', 'display_xref_id' ],
            object_xref => [ 'xref_id', 'xref_id' ],
            transcript => [ 'xref_id', 'display_xref_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::ExternalDb; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::ExternalDb->table('external_db');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::ExternalDb->columns( 
    All => qw/ 
        external_db_id
        db_name
        db_release
        status
        dbprimary_acc_linkable
        display_label_linkable
        priority
        db_display_name 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::ExternalDb->columns( 
    Essential => qw/ 
        external_db_id
        db_name
        db_release
        status
        dbprimary_acc_linkable
        display_label_linkable
        priority
        db_display_name 
    /
);

Gramene::CDBI::EnsemblArabidopsis::ExternalDb->columns( 
    Ordered => qw/ 
        external_db_id
        db_name
        db_release
        status
        dbprimary_acc_linkable
        display_label_linkable
        priority
        db_display_name 
    /
);

sub to_string {
    my $self = shift;
    return $self->db_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'external_db_id' => 'number',
        'db_name' => 'string',
        'db_release' => 'string',
        'status' => 'string',
        'dbprimary_acc_linkable' => 'string',
        'display_label_linkable' => 'string',
        'priority' => 'number',
        'db_display_name' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::ExternalDb->has_many( 
    'unmapped_objects',
    'Gramene::CDBI::EnsemblArabidopsis::UnmappedObject',
    'external_db_id' 
);

Gramene::CDBI::EnsemblArabidopsis::ExternalDb->has_many( 
    'xrefs',
    'Gramene::CDBI::EnsemblArabidopsis::Xref',
    'external_db_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            unmapped_object => [ 'external_db_id', 'external_db_id' ],
            xref => [ 'external_db_id', 'external_db_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::GeneArchive; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::GeneArchive->table('gene_archive');


sub mapping_session {
    my $self = shift;
    return $self->mapping_session_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::GeneArchive->columns( 
    All => qw/ 
        gene_stable_id
        gene_version
        transcript_stable_id
        transcript_version
        translation_stable_id
        translation_version
        peptide_archive_id
        mapping_session_id 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::GeneArchive->columns( 
    Essential => qw/ 
        gene_stable_id
        gene_version
        transcript_stable_id
        transcript_version
        translation_stable_id
        translation_version
        peptide_archive_id
        mapping_session_id 
    /
);

Gramene::CDBI::EnsemblArabidopsis::GeneArchive->columns( 
    Ordered => qw/ 
        gene_stable_id
        gene_version
        transcript_stable_id
        transcript_version
        translation_stable_id
        translation_version
        peptide_archive_id
        mapping_session_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->gene_version || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_stable_id' => 'string',
        'gene_version' => 'string',
        'transcript_stable_id' => 'string',
        'transcript_version' => 'string',
        'translation_stable_id' => 'string',
        'translation_version' => 'string',
        'peptide_archive_id' => 'number',
        'mapping_session_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::GeneArchive->has_a(
    mapping_session_id => 'Gramene::CDBI::EnsemblArabidopsis::MappingSession'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            mapping_session => [ 'mapping_session_id', 'mapping_session_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::GrameneGeneExtHelper; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::GrameneGeneExtHelper->table('gramene_gene_ext_helper');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::GrameneGeneExtHelper->columns( 
    All => qw/ 
        gene_id
        stable_id
        external_name 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::GrameneGeneExtHelper->columns( 
    Essential => qw/ 
        gene_id
        stable_id
        external_name 
    /
);

Gramene::CDBI::EnsemblArabidopsis::GrameneGeneExtHelper->columns( 
    Ordered => qw/ 
        gene_id
        stable_id
        external_name 
    /
);

sub to_string {
    my $self = shift;
    return $self->external_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_id' => 'number',
        'stable_id' => 'string',
        'external_name' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::GrameneGeneXTmp0; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::GrameneGeneXTmp0->table('gramene_gene_x_tmp0');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::GrameneGeneXTmp0->columns( 
    All => qw/ 
        gene_id
        display_label
        dbprimary_acc 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::GrameneGeneXTmp0->columns( 
    Essential => qw/ 
        gene_id
        display_label
        dbprimary_acc 
    /
);

Gramene::CDBI::EnsemblArabidopsis::GrameneGeneXTmp0->columns( 
    Ordered => qw/ 
        gene_id
        display_label
        dbprimary_acc 
    /
);

sub to_string {
    my $self = shift;
    return $self->display_label || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_id' => 'number',
        'display_label' => 'string',
        'dbprimary_acc' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::GrameneGeneXTmp1; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::GrameneGeneXTmp1->table('gramene_gene_x_tmp1');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::GrameneGeneXTmp1->columns( 
    All => qw/ 
        gene_id
        external_name 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::GrameneGeneXTmp1->columns( 
    Essential => qw/ 
        gene_id
        external_name 
    /
);

Gramene::CDBI::EnsemblArabidopsis::GrameneGeneXTmp1->columns( 
    Ordered => qw/ 
        gene_id
        external_name 
    /
);

sub to_string {
    my $self = shift;
    return $self->external_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'gene_id' => 'number',
        'external_name' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::InputIdAnalysis; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::InputIdAnalysis->table('input_id_analysis');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::InputIdAnalysis->columns( 
    All => qw/ 
        analysis_id
        input_id
        input_id_type
        created
        runhost
        db_version
        result 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::InputIdAnalysis->columns( 
    Essential => qw/ 
        analysis_id
        input_id
        input_id_type
        created
        runhost
        db_version
        result 
    /
);

Gramene::CDBI::EnsemblArabidopsis::InputIdAnalysis->columns( 
    Ordered => qw/ 
        analysis_id
        input_id
        input_id_type
        created
        runhost
        db_version
        result 
    /
);


Gramene::CDBI::EnsemblArabidopsis::InputIdAnalysis->columns( 
    Primary => qw/ 
        analysis_id
        input_id 
    / 
);


sub to_string {
    my $self = shift;
    return $self->input_id_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'input_id' => 'string',
        'input_id_type' => 'string',
        'analysis_id' => 'string',
        'created' => 'string',
        'runhost' => 'string',
        'db_version' => 'string',
        'result' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::InputIdTypeAnalysis; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::InputIdTypeAnalysis->table('input_id_type_analysis');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::InputIdTypeAnalysis->columns( 
    All => qw/ 
        analysis_id
        input_id_type 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::InputIdTypeAnalysis->columns( 
    Essential => qw/ 
        analysis_id
        input_id_type 
    /
);

Gramene::CDBI::EnsemblArabidopsis::InputIdTypeAnalysis->columns( 
    Ordered => qw/ 
        analysis_id
        input_id_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->input_id_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'analysis_id' => 'string',
        'input_id_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::InsertCoords; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::InsertCoords->table('insert_coords');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::InsertCoords->columns( 
    All => qw/ 
        coord_name
        e_value
        start_coord
        end_coord
        EMBL_ID
        EMBL_name
        coord_name_short
        chr
        ori
        NASC_code
        NASC_name 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::InsertCoords->columns( 
    Essential => qw/ 
        coord_name
        e_value
        start_coord
        end_coord
        EMBL_ID
        EMBL_name
        coord_name_short
        chr
        ori
        NASC_code
        NASC_name 
    /
);

Gramene::CDBI::EnsemblArabidopsis::InsertCoords->columns( 
    Ordered => qw/ 
        coord_name
        e_value
        start_coord
        end_coord
        EMBL_ID
        EMBL_name
        coord_name_short
        chr
        ori
        NASC_code
        NASC_name 
    /
);


Gramene::CDBI::EnsemblArabidopsis::InsertCoords->columns( 
    Primary => qw/ 
        coord_name
        e_value
        start_coord
        end_coord 
    / 
);


sub to_string {
    my $self = shift;
    return $self->EMBL_ID || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'EMBL_ID' => 'string',
        'EMBL_name' => 'string',
        'coord_name' => 'string',
        'coord_name_short' => 'string',
        'chr' => 'number',
        'start_coord' => 'number',
        'end_coord' => 'number',
        'ori' => 'number',
        'e_value' => 'number',
        'NASC_code' => 'string',
        'NASC_name' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::Interpro; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::Interpro->table('interpro');

our @extras = ();
push @extras, Class::DBI::Column->new( id => { 
    accessor => 'id_field',
    mutator  => 'id_field' 
}); 

Gramene::CDBI::EnsemblArabidopsis::Interpro->columns( 
    All => qw/ 
        interpro_ac 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::Interpro->columns( 
    Essential => qw/ 
        interpro_ac 
    /
);

Gramene::CDBI::EnsemblArabidopsis::Interpro->columns( 
    Ordered => qw/ 
        interpro_ac 
    /
);

sub to_string {
    my $self = shift;
    return $self->interpro_ac || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'interpro_ac' => 'string',
        'id' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::Job; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::Job->table('job');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::Job->columns( 
    All => qw/ 
        job_id
        input_id
        analysis_id
        submission_id
        stdout_file
        stderr_file
        retry_count
        temp_dir
        exec_host 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::Job->columns( 
    Essential => qw/ 
        job_id
        input_id
        analysis_id
        submission_id
        stdout_file
        stderr_file
        retry_count
        temp_dir
        exec_host 
    /
);

Gramene::CDBI::EnsemblArabidopsis::Job->columns( 
    Ordered => qw/ 
        job_id
        input_id
        analysis_id
        submission_id
        stdout_file
        stderr_file
        retry_count
        temp_dir
        exec_host 
    /
);

sub to_string {
    my $self = shift;
    return $self->stdout_file || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'job_id' => 'number',
        'input_id' => 'string',
        'analysis_id' => 'string',
        'submission_id' => 'string',
        'stdout_file' => 'string',
        'stderr_file' => 'string',
        'retry_count' => 'string',
        'temp_dir' => 'string',
        'exec_host' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::JobStatus; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::JobStatus->table('job_status');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::JobStatus->columns( 
    All => qw/ 
        job_id
        status
        time
        is_current 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::JobStatus->columns( 
    Essential => qw/ 
        job_id
        status
        time
        is_current 
    /
);

Gramene::CDBI::EnsemblArabidopsis::JobStatus->columns( 
    Ordered => qw/ 
        job_id
        status
        time
        is_current 
    /
);

sub to_string {
    my $self = shift;
    return $self->status || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'job_id' => 'number',
        'status' => 'string',
        'time' => 'string',
        'is_current' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::MarkerMapLocation; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::MarkerMapLocation->table('marker_map_location');


sub map {
    my $self = shift;
    return $self->map_id;
}
    

sub marker {
    my $self = shift;
    return $self->marker_id;
}
    

sub marker_synonym {
    my $self = shift;
    return $self->marker_synonym_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::MarkerMapLocation->columns( 
    All => qw/ 
        marker_id
        map_id
        chromosome_name
        marker_synonym_id
        position
        lod_score 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::MarkerMapLocation->columns( 
    Essential => qw/ 
        marker_id
        map_id
        chromosome_name
        marker_synonym_id
        position
        lod_score 
    /
);

Gramene::CDBI::EnsemblArabidopsis::MarkerMapLocation->columns( 
    Ordered => qw/ 
        marker_id
        map_id
        chromosome_name
        marker_synonym_id
        position
        lod_score 
    /
);


Gramene::CDBI::EnsemblArabidopsis::MarkerMapLocation->columns( 
    Primary => qw/ 
        marker_id
        map_id 
    / 
);


sub to_string {
    my $self = shift;
    return $self->chromosome_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'map_id' => 'number',
        'chromosome_name' => 'string',
        'marker_synonym_id' => 'number',
        'position' => 'string',
        'lod_score' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::MarkerMapLocation->has_a(
    map_id => 'Gramene::CDBI::EnsemblArabidopsis::Map'
);

Gramene::CDBI::EnsemblArabidopsis::MarkerMapLocation->has_a(
    marker_id => 'Gramene::CDBI::EnsemblArabidopsis::Marker'
);

Gramene::CDBI::EnsemblArabidopsis::MarkerMapLocation->has_a(
    marker_synonym_id => 'Gramene::CDBI::EnsemblArabidopsis::MarkerSynonym'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            map => [ 'map_id', 'map_id' ],
            marker => [ 'marker_id', 'marker_id' ],
            marker_synonym => [ 'marker_synonym_id', 'marker_synonym_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::Map; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::Map->table('map');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::Map->columns( 
    All => qw/ 
        map_id
        map_name 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::Map->columns( 
    Essential => qw/ 
        map_id
        map_name 
    /
);

Gramene::CDBI::EnsemblArabidopsis::Map->columns( 
    Ordered => qw/ 
        map_id
        map_name 
    /
);

sub to_string {
    my $self = shift;
    return $self->map_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'map_id' => 'number',
        'map_name' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::Map->has_many( 
    'marker_map_locations',
    'Gramene::CDBI::EnsemblArabidopsis::MarkerMapLocation',
    'map_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker_map_location => [ 'map_id', 'map_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::StableIdEvent; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::StableIdEvent->table('stable_id_event');


sub mapping_session {
    my $self = shift;
    return $self->mapping_session_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::StableIdEvent->columns( 
    All => qw/ 
        old_stable_id
        old_version
        new_stable_id
        new_version
        mapping_session_id
        type
        score 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::StableIdEvent->columns( 
    Essential => qw/ 
        old_stable_id
        old_version
        new_stable_id
        new_version
        mapping_session_id
        type
        score 
    /
);

Gramene::CDBI::EnsemblArabidopsis::StableIdEvent->columns( 
    Ordered => qw/ 
        old_stable_id
        old_version
        new_stable_id
        new_version
        mapping_session_id
        type
        score 
    /
);

sub to_string {
    my $self = shift;
    return $self->old_version || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'old_stable_id' => 'string',
        'old_version' => 'string',
        'new_stable_id' => 'string',
        'new_version' => 'string',
        'mapping_session_id' => 'number',
        'type' => 'string',
        'score' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::StableIdEvent->has_a(
    mapping_session_id => 'Gramene::CDBI::EnsemblArabidopsis::MappingSession'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            mapping_session => [ 'mapping_session_id', 'mapping_session_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::MappingSession; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::MappingSession->table('mapping_session');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::MappingSession->columns( 
    All => qw/ 
        mapping_session_id
        old_db_name
        new_db_name
        old_release
        new_release
        old_assembly
        new_assembly
        created 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::MappingSession->columns( 
    Essential => qw/ 
        mapping_session_id
        old_db_name
        new_db_name
        old_release
        new_release
        old_assembly
        new_assembly
        created 
    /
);

Gramene::CDBI::EnsemblArabidopsis::MappingSession->columns( 
    Ordered => qw/ 
        mapping_session_id
        old_db_name
        new_db_name
        old_release
        new_release
        old_assembly
        new_assembly
        created 
    /
);

sub to_string {
    my $self = shift;
    return $self->old_db_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'mapping_session_id' => 'number',
        'old_db_name' => 'string',
        'new_db_name' => 'string',
        'old_release' => 'string',
        'new_release' => 'string',
        'old_assembly' => 'string',
        'new_assembly' => 'string',
        'created' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::MappingSession->has_many( 
    'gene_archives',
    'Gramene::CDBI::EnsemblArabidopsis::GeneArchive',
    'mapping_session_id' 
);

Gramene::CDBI::EnsemblArabidopsis::MappingSession->has_many( 
    'stable_id_events',
    'Gramene::CDBI::EnsemblArabidopsis::StableIdEvent',
    'mapping_session_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            gene_archive => [ 'mapping_session_id', 'mapping_session_id' ],
            stable_id_event => [ 'mapping_session_id', 'mapping_session_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::MarkerSynonym; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::MarkerSynonym->table('marker_synonym');


sub marker {
    my $self = shift;
    return $self->marker_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::MarkerSynonym->columns( 
    All => qw/ 
        marker_synonym_id
        marker_id
        source
        name 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::MarkerSynonym->columns( 
    Essential => qw/ 
        marker_synonym_id
        marker_id
        source
        name 
    /
);

Gramene::CDBI::EnsemblArabidopsis::MarkerSynonym->columns( 
    Ordered => qw/ 
        marker_synonym_id
        marker_id
        source
        name 
    /
);

sub to_string {
    my $self = shift;
    return $self->source || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_synonym_id' => 'number',
        'marker_id' => 'number',
        'source' => 'string',
        'name' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::MarkerSynonym->has_a(
    marker_id => 'Gramene::CDBI::EnsemblArabidopsis::Marker'
);

Gramene::CDBI::EnsemblArabidopsis::MarkerSynonym->has_many( 
    'markers',
    'Gramene::CDBI::EnsemblArabidopsis::Marker',
    'display_marker_synonym_id' 
);

Gramene::CDBI::EnsemblArabidopsis::MarkerSynonym->has_many( 
    'marker_map_locations',
    'Gramene::CDBI::EnsemblArabidopsis::MarkerMapLocation',
    'marker_synonym_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'marker_id', 'marker_id' ],
            marker_map_location => [ 'marker_synonym_id', 'marker_synonym_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::QtlSynonym; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::QtlSynonym->table('qtl_synonym');


sub qtl {
    my $self = shift;
    return $self->qtl_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::QtlSynonym->columns( 
    All => qw/ 
        qtl_synonym_id
        qtl_id
        source_database
        source_primary_id 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::QtlSynonym->columns( 
    Essential => qw/ 
        qtl_synonym_id
        qtl_id
        source_database
        source_primary_id 
    /
);

Gramene::CDBI::EnsemblArabidopsis::QtlSynonym->columns( 
    Ordered => qw/ 
        qtl_synonym_id
        qtl_id
        source_database
        source_primary_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->source_database || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'qtl_synonym_id' => 'number',
        'qtl_id' => 'number',
        'source_database' => 'string',
        'source_primary_id' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::QtlSynonym->has_a(
    qtl_id => 'Gramene::CDBI::EnsemblArabidopsis::Qtl'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            qtl => [ 'qtl_id', 'qtl_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::Qtl; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::Qtl->table('qtl');


sub peak_marker {
    my $self = shift;
    return $self->peak_marker_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::Qtl->columns( 
    All => qw/ 
        qtl_id
        trait
        lod_score
        flank_marker_id_1
        flank_marker_id_2
        peak_marker_id 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::Qtl->columns( 
    Essential => qw/ 
        qtl_id
        trait
        lod_score
        flank_marker_id_1
        flank_marker_id_2
        peak_marker_id 
    /
);

Gramene::CDBI::EnsemblArabidopsis::Qtl->columns( 
    Ordered => qw/ 
        qtl_id
        trait
        lod_score
        flank_marker_id_1
        flank_marker_id_2
        peak_marker_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->trait || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'qtl_id' => 'number',
        'trait' => 'string',
        'lod_score' => 'number',
        'flank_marker_id_1' => 'number',
        'flank_marker_id_2' => 'number',
        'peak_marker_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::Qtl->has_a(
    peak_marker_id => 'Gramene::CDBI::EnsemblArabidopsis::Marker'
);

Gramene::CDBI::EnsemblArabidopsis::Qtl->has_many( 
    'qtl_features',
    'Gramene::CDBI::EnsemblArabidopsis::QtlFeature',
    'qtl_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Qtl->has_many( 
    'qtl_synonyms',
    'Gramene::CDBI::EnsemblArabidopsis::QtlSynonym',
    'qtl_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker => [ 'peak_marker_id', 'marker_id' ],
            qtl_feature => [ 'qtl_id', 'qtl_id' ],
            qtl_synonym => [ 'qtl_id', 'qtl_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::Marker; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::Marker->table('marker');


sub display_marker_synonym {
    my $self = shift;
    return $self->display_marker_synonym_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::Marker->columns( 
    All => qw/ 
        marker_id
        display_marker_synonym_id
        left_primer
        right_primer
        min_primer_dist
        max_primer_dist
        priority
        type 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::Marker->columns( 
    Essential => qw/ 
        marker_id
        display_marker_synonym_id
        left_primer
        right_primer
        min_primer_dist
        max_primer_dist
        priority
        type 
    /
);

Gramene::CDBI::EnsemblArabidopsis::Marker->columns( 
    Ordered => qw/ 
        marker_id
        display_marker_synonym_id
        left_primer
        right_primer
        min_primer_dist
        max_primer_dist
        priority
        type 
    /
);

sub to_string {
    my $self = shift;
    return $self->left_primer || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_id' => 'number',
        'display_marker_synonym_id' => 'number',
        'left_primer' => 'string',
        'right_primer' => 'string',
        'min_primer_dist' => 'number',
        'max_primer_dist' => 'number',
        'priority' => 'number',
        'type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::Marker->has_a(
    display_marker_synonym_id => 'Gramene::CDBI::EnsemblArabidopsis::MarkerSynonym'
);

Gramene::CDBI::EnsemblArabidopsis::Marker->has_many( 
    'marker_features',
    'Gramene::CDBI::EnsemblArabidopsis::MarkerFeature',
    'marker_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Marker->has_many( 
    'marker_map_locations',
    'Gramene::CDBI::EnsemblArabidopsis::MarkerMapLocation',
    'marker_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Marker->has_many( 
    'marker_synonyms',
    'Gramene::CDBI::EnsemblArabidopsis::MarkerSynonym',
    'marker_id' 
);

Gramene::CDBI::EnsemblArabidopsis::Marker->has_many( 
    'qtls_flank_marker_1',
    'Gramene::CDBI::EnsemblArabidopsis::Qtl',
    'flank_marker_id_1' 
);

Gramene::CDBI::EnsemblArabidopsis::Marker->has_many( 
    'qtls_flank_marker_2',
    'Gramene::CDBI::EnsemblArabidopsis::Qtl',
    'flank_marker_id_2' 
);

Gramene::CDBI::EnsemblArabidopsis::Marker->has_many( 
    'qtls_peak_marker',
    'Gramene::CDBI::EnsemblArabidopsis::Qtl',
    'peak_marker_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            marker_feature => [ 'marker_id', 'marker_id' ],
            marker_map_location => [ 'marker_id', 'marker_id' ],
            marker_synonym => [ 'marker_id', 'marker_id' ],
            qtl => [ 'marker_id', 'peak_marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::Meta; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::Meta->table('meta');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::Meta->columns( 
    All => qw/ 
        meta_id
        meta_key
        meta_value 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::Meta->columns( 
    Essential => qw/ 
        meta_id
        meta_key
        meta_value 
    /
);

Gramene::CDBI::EnsemblArabidopsis::Meta->columns( 
    Ordered => qw/ 
        meta_id
        meta_key
        meta_value 
    /
);

sub to_string {
    my $self = shift;
    return $self->meta_key || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'meta_id' => 'number',
        'meta_key' => 'string',
        'meta_value' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::MiscAttribHelper; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::MiscAttribHelper->table('misc_attrib_helper');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::MiscAttribHelper->columns( 
    All => qw/ 
        name
        attrib_name
        attrib_value 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::MiscAttribHelper->columns( 
    Essential => qw/ 
        name
        attrib_name
        attrib_value 
    /
);

Gramene::CDBI::EnsemblArabidopsis::MiscAttribHelper->columns( 
    Ordered => qw/ 
        name
        attrib_name
        attrib_value 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'name' => 'string',
        'attrib_name' => 'string',
        'attrib_value' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::MiscSet; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::MiscSet->table('misc_set');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::MiscSet->columns( 
    All => qw/ 
        misc_set_id
        code
        name
        description
        max_length 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::MiscSet->columns( 
    Essential => qw/ 
        misc_set_id
        code
        name
        description
        max_length 
    /
);

Gramene::CDBI::EnsemblArabidopsis::MiscSet->columns( 
    Ordered => qw/ 
        misc_set_id
        code
        name
        description
        max_length 
    /
);

sub to_string {
    my $self = shift;
    return $self->code || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'misc_set_id' => 'string',
        'code' => 'string',
        'name' => 'string',
        'description' => 'string',
        'max_length' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::MiscSet->has_many( 
    'misc_feature_misc_sets',
    'Gramene::CDBI::EnsemblArabidopsis::MiscFeatureMiscSet',
    'misc_set_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            misc_feature_misc_set => [ 'misc_set_id', 'misc_set_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::OligoProbe; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::OligoProbe->table('oligo_probe');


sub oligo_array {
    my $self = shift;
    return $self->oligo_array_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::OligoProbe->columns( 
    All => qw/ 
        oligo_probe_id
        oligo_array_id
        probeset
        name
        description
        length 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::OligoProbe->columns( 
    Essential => qw/ 
        oligo_probe_id
        oligo_array_id
        probeset
        name
        description
        length 
    /
);

Gramene::CDBI::EnsemblArabidopsis::OligoProbe->columns( 
    Ordered => qw/ 
        oligo_probe_id
        oligo_array_id
        probeset
        name
        description
        length 
    /
);


Gramene::CDBI::EnsemblArabidopsis::OligoProbe->columns( 
    Primary => qw/ 
        oligo_probe_id
        oligo_array_id 
    / 
);


sub to_string {
    my $self = shift;
    return $self->probeset || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'oligo_probe_id' => 'number',
        'oligo_array_id' => 'number',
        'probeset' => 'string',
        'name' => 'string',
        'description' => 'string',
        'length' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::OligoProbe->has_a(
    oligo_array_id => 'Gramene::CDBI::EnsemblArabidopsis::OligoArray'
);

Gramene::CDBI::EnsemblArabidopsis::OligoProbe->has_many( 
    'oligo_features',
    'Gramene::CDBI::EnsemblArabidopsis::OligoFeature',
    'oligo_probe_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            oligo_array => [ 'oligo_array_id', 'oligo_array_id' ],
            oligo_feature => [ 'oligo_probe_id', 'oligo_probe_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::OligoArray; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::OligoArray->table('oligo_array');


sub parent_array {
    my $self = shift;
    return $self->parent_array_id;
}
    
our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::OligoArray->columns( 
    All => qw/ 
        oligo_array_id
        parent_array_id
        probe_setsize
        name
        type 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::OligoArray->columns( 
    Essential => qw/ 
        oligo_array_id
        parent_array_id
        probe_setsize
        name
        type 
    /
);

Gramene::CDBI::EnsemblArabidopsis::OligoArray->columns( 
    Ordered => qw/ 
        oligo_array_id
        parent_array_id
        probe_setsize
        name
        type 
    /
);

sub to_string {
    my $self = shift;
    return $self->probe_setsize || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'oligo_array_id' => 'number',
        'parent_array_id' => 'number',
        'probe_setsize' => 'string',
        'name' => 'string',
        'type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::OligoArray->has_a(
    parent_array_id => 'Gramene::CDBI::EnsemblArabidopsis::OligoArray'
);

Gramene::CDBI::EnsemblArabidopsis::OligoArray->has_many( 
    'oligo_arrays',
    'Gramene::CDBI::EnsemblArabidopsis::OligoArray',
    'parent_array_id' 
);

Gramene::CDBI::EnsemblArabidopsis::OligoArray->has_many( 
    'oligo_probes',
    'Gramene::CDBI::EnsemblArabidopsis::OligoProbe',
    'oligo_array_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            oligo_array => [ 'parent_array_id', 'oligo_array_id' ],
            oligo_probe => [ 'oligo_array_id', 'oligo_array_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::PeptideArchive; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::PeptideArchive->table('peptide_archive');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::PeptideArchive->columns( 
    All => qw/ 
        peptide_archive_id
        md5_checksum
        peptide_seq 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::PeptideArchive->columns( 
    Essential => qw/ 
        peptide_archive_id
        md5_checksum
        peptide_seq 
    /
);

Gramene::CDBI::EnsemblArabidopsis::PeptideArchive->columns( 
    Ordered => qw/ 
        peptide_archive_id
        md5_checksum
        peptide_seq 
    /
);

sub to_string {
    my $self = shift;
    return $self->md5_checksum || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'peptide_archive_id' => 'number',
        'md5_checksum' => 'string',
        'peptide_seq' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactor; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactor->table('regulatory_factor');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactor->columns( 
    All => qw/ 
        regulatory_factor_id
        name
        type 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactor->columns( 
    Essential => qw/ 
        regulatory_factor_id
        name
        type 
    /
);

Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactor->columns( 
    Ordered => qw/ 
        regulatory_factor_id
        name
        type 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'regulatory_factor_id' => 'number',
        'name' => 'string',
        'type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactor->has_many( 
    'regulatory_factor_codings',
    'Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactorCoding',
    'regulatory_factor_id' 
);

Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactor->has_many( 
    'regulatory_features',
    'Gramene::CDBI::EnsemblArabidopsis::RegulatoryFeature',
    'regulatory_factor_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            regulatory_factor_coding => [ 'regulatory_factor_id', 'regulatory_factor_id' ],
            regulatory_feature => [ 'regulatory_factor_id', 'regulatory_factor_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::RegulatoryFeatureObject; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::RegulatoryFeatureObject->table('regulatory_feature_object');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::RegulatoryFeatureObject->columns( 
    All => qw/ 
        regulatory_feature_id
        ensembl_object_type
        ensembl_object_id
        influence
        evidence 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::RegulatoryFeatureObject->columns( 
    Essential => qw/ 
        regulatory_feature_id
        ensembl_object_type
        ensembl_object_id
        influence
        evidence 
    /
);

Gramene::CDBI::EnsemblArabidopsis::RegulatoryFeatureObject->columns( 
    Ordered => qw/ 
        regulatory_feature_id
        ensembl_object_type
        ensembl_object_id
        influence
        evidence 
    /
);

sub to_string {
    my $self = shift;
    return $self->ensembl_object_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'regulatory_feature_id' => 'number',
        'ensembl_object_type' => 'string',
        'ensembl_object_id' => 'number',
        'influence' => 'string',
        'evidence' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::RepeatConsensus; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::RepeatConsensus->table('repeat_consensus');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::RepeatConsensus->columns( 
    All => qw/ 
        repeat_consensus_id
        repeat_name
        repeat_class
        repeat_type
        repeat_consensus 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::RepeatConsensus->columns( 
    Essential => qw/ 
        repeat_consensus_id
        repeat_name
        repeat_class
        repeat_type
        repeat_consensus 
    /
);

Gramene::CDBI::EnsemblArabidopsis::RepeatConsensus->columns( 
    Ordered => qw/ 
        repeat_consensus_id
        repeat_name
        repeat_class
        repeat_type
        repeat_consensus 
    /
);

sub to_string {
    my $self = shift;
    return $self->repeat_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'repeat_consensus_id' => 'number',
        'repeat_name' => 'string',
        'repeat_class' => 'string',
        'repeat_type' => 'string',
        'repeat_consensus' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::RepeatConsensus->has_many( 
    'repeat_features',
    'Gramene::CDBI::EnsemblArabidopsis::RepeatFeature',
    'repeat_consensus_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            repeat_feature => [ 'repeat_consensus_id', 'repeat_consensus_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::RuleConditions; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::RuleConditions->table('rule_conditions');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::RuleConditions->columns( 
    All => qw/ 
        rule_id
        condition 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::RuleConditions->columns( 
    Essential => qw/ 
        rule_id
        condition 
    /
);

Gramene::CDBI::EnsemblArabidopsis::RuleConditions->columns( 
    Ordered => qw/ 
        rule_id
        condition 
    /
);

sub to_string {
    my $self = shift;
    return $self->condition || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'rule_id' => 'string',
        'condition' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::RuleGoal; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::RuleGoal->table('rule_goal');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::RuleGoal->columns( 
    All => qw/ 
        rule_id
        goal 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::RuleGoal->columns( 
    Essential => qw/ 
        rule_id
        goal 
    /
);

Gramene::CDBI::EnsemblArabidopsis::RuleGoal->columns( 
    Ordered => qw/ 
        rule_id
        goal 
    /
);

sub to_string {
    my $self = shift;
    return $self->goal || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'rule_id' => 'string',
        'goal' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::SeqtableCmapTmp; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::SeqtableCmapTmp->table('seqtable_cmap_tmp');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::SeqtableCmapTmp->columns( 
    All => qw/ 
        marker_name
        cms_aid
        cf_aid
        map_name
        cm_aid
        position 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::SeqtableCmapTmp->columns( 
    Essential => qw/ 
        marker_name
        cms_aid
        cf_aid
        map_name
        cm_aid
        position 
    /
);

Gramene::CDBI::EnsemblArabidopsis::SeqtableCmapTmp->columns( 
    Ordered => qw/ 
        marker_name
        cms_aid
        cf_aid
        map_name
        cm_aid
        position 
    /
);

sub to_string {
    my $self = shift;
    return $self->marker_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'marker_name' => 'string',
        'cms_aid' => 'string',
        'cf_aid' => 'string',
        'map_name' => 'string',
        'cm_aid' => 'string',
        'position' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::SeqtableCmapTmp0; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::SeqtableCmapTmp0->table('seqtable_cmap_tmp0');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::SeqtableCmapTmp0->columns( 
    All => qw/ 
        cname
        cms_aid
        map_name
        cf_aid
        cm_aid
        position 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::SeqtableCmapTmp0->columns( 
    Essential => qw/ 
        cname
        cms_aid
        map_name
        cf_aid
        cm_aid
        position 
    /
);

Gramene::CDBI::EnsemblArabidopsis::SeqtableCmapTmp0->columns( 
    Ordered => qw/ 
        cname
        cms_aid
        map_name
        cf_aid
        cm_aid
        position 
    /
);

sub to_string {
    my $self = shift;
    return $self->cname || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'cname' => 'string',
        'cms_aid' => 'string',
        'map_name' => 'string',
        'cf_aid' => 'string',
        'cm_aid' => 'string',
        'position' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::SeqtableDaf0; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::SeqtableDaf0->table('seqtable_daf0');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::SeqtableDaf0->columns( 
    All => qw/ 
        embl_acc
        daf_chr_start
        daf_chr_end
        chr_strand
        chromosome
        superctg_end
        superctg_name
        superctg_ori
        superctg_start
        daf_clone_start
        daf_clone_end
        daf_super_start
        daf_super_end 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::SeqtableDaf0->columns( 
    Essential => qw/ 
        embl_acc
        daf_chr_start
        daf_chr_end
        chr_strand
        chromosome
        superctg_end
        superctg_name
        superctg_ori
        superctg_start
        daf_clone_start
        daf_clone_end
        daf_super_start
        daf_super_end 
    /
);

Gramene::CDBI::EnsemblArabidopsis::SeqtableDaf0->columns( 
    Ordered => qw/ 
        embl_acc
        daf_chr_start
        daf_chr_end
        chr_strand
        chromosome
        superctg_end
        superctg_name
        superctg_ori
        superctg_start
        daf_clone_start
        daf_clone_end
        daf_super_start
        daf_super_end 
    /
);

sub to_string {
    my $self = shift;
    return $self->embl_acc || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'embl_acc' => 'string',
        'daf_chr_start' => 'number',
        'daf_chr_end' => 'number',
        'chr_strand' => 'string',
        'chromosome' => 'string',
        'superctg_end' => 'string',
        'superctg_name' => 'string',
        'superctg_ori' => 'string',
        'superctg_start' => 'string',
        'daf_clone_start' => 'string',
        'daf_clone_end' => 'string',
        'daf_super_start' => 'string',
        'daf_super_end' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::SeqtableDaf1; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::SeqtableDaf1->table('seqtable_daf1');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::SeqtableDaf1->columns( 
    All => qw/ 
        embl_acc
        chromosome
        chr_start
        chr_end
        superctg_name
        super_start
        super_end 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::SeqtableDaf1->columns( 
    Essential => qw/ 
        embl_acc
        chromosome
        chr_start
        chr_end
        superctg_name
        super_start
        super_end 
    /
);

Gramene::CDBI::EnsemblArabidopsis::SeqtableDaf1->columns( 
    Ordered => qw/ 
        embl_acc
        chromosome
        chr_start
        chr_end
        superctg_name
        super_start
        super_end 
    /
);

sub to_string {
    my $self = shift;
    return $self->embl_acc || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'embl_acc' => 'string',
        'chromosome' => 'string',
        'chr_start' => 'string',
        'chr_end' => 'string',
        'superctg_name' => 'string',
        'super_start' => 'string',
        'super_end' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::SeqtableHelper; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::SeqtableHelper->table('seqtable_helper');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::SeqtableHelper->columns( 
    All => qw/ 
        clone_id
        bacname
        chromosome
        embl_acc
        embl_version
        htg_phase
        super_name
        super_chr_start
        super_chr_end
        clone_super_start
        clone_super_end
        clone_chr_start
        clone_chr_end
        marker_name
        marker_type
        jrgp_rflp_cm
        jrgp_rflp_map
        jrgp_rflp_feature
        cornell_rflp_cm
        cornell_rflp_map
        cornell_rflp_feature
        irmi_ssr_cm
        irmi_ssr_map
        irmi_ssr_feature 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::SeqtableHelper->columns( 
    Essential => qw/ 
        clone_id
        bacname
        chromosome
        embl_acc
        embl_version
        htg_phase
        super_name
        super_chr_start
        super_chr_end
        clone_super_start
        clone_super_end
        clone_chr_start
        clone_chr_end
        marker_name
        marker_type
        jrgp_rflp_cm
        jrgp_rflp_map
        jrgp_rflp_feature
        cornell_rflp_cm
        cornell_rflp_map
        cornell_rflp_feature
        irmi_ssr_cm
        irmi_ssr_map
        irmi_ssr_feature 
    /
);

Gramene::CDBI::EnsemblArabidopsis::SeqtableHelper->columns( 
    Ordered => qw/ 
        clone_id
        bacname
        chromosome
        embl_acc
        embl_version
        htg_phase
        super_name
        super_chr_start
        super_chr_end
        clone_super_start
        clone_super_end
        clone_chr_start
        clone_chr_end
        marker_name
        marker_type
        jrgp_rflp_cm
        jrgp_rflp_map
        jrgp_rflp_feature
        cornell_rflp_cm
        cornell_rflp_map
        cornell_rflp_feature
        irmi_ssr_cm
        irmi_ssr_map
        irmi_ssr_feature 
    /
);

sub to_string {
    my $self = shift;
    return $self->bacname || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'clone_id' => 'number',
        'bacname' => 'string',
        'chromosome' => 'string',
        'embl_acc' => 'string',
        'embl_version' => 'number',
        'htg_phase' => 'number',
        'super_name' => 'string',
        'super_chr_start' => 'number',
        'super_chr_end' => 'number',
        'clone_super_start' => 'number',
        'clone_super_end' => 'number',
        'clone_chr_start' => 'number',
        'clone_chr_end' => 'number',
        'marker_name' => 'string',
        'marker_type' => 'string',
        'jrgp_rflp_cm' => 'number',
        'jrgp_rflp_map' => 'string',
        'jrgp_rflp_feature' => 'string',
        'cornell_rflp_cm' => 'number',
        'cornell_rflp_map' => 'string',
        'cornell_rflp_feature' => 'string',
        'irmi_ssr_cm' => 'number',
        'irmi_ssr_map' => 'string',
        'irmi_ssr_feature' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::SeqtableMarkerTmp; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::SeqtableMarkerTmp->table('seqtable_marker_tmp');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::SeqtableMarkerTmp->columns( 
    All => qw/ 
        clone_id
        mname
        mtype 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::SeqtableMarkerTmp->columns( 
    Essential => qw/ 
        clone_id
        mname
        mtype 
    /
);

Gramene::CDBI::EnsemblArabidopsis::SeqtableMarkerTmp->columns( 
    Ordered => qw/ 
        clone_id
        mname
        mtype 
    /
);

sub to_string {
    my $self = shift;
    return $self->mname || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'clone_id' => 'number',
        'mname' => 'string',
        'mtype' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::SeqtablePrehelper; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::SeqtablePrehelper->table('seqtable_prehelper');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::SeqtablePrehelper->columns( 
    All => qw/ 
        clone_id
        chr_id
        embl_acc
        embl_version
        htg_phase
        chromosome
        super_name
        super_chr_start
        super_chr_end
        clone_super_start
        clone_super_end
        clone_chr_start
        clone_chr_end
        contig_clone_start
        contig_clone_end
        bacname
        bac_length
        source 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::SeqtablePrehelper->columns( 
    Essential => qw/ 
        clone_id
        chr_id
        embl_acc
        embl_version
        htg_phase
        chromosome
        super_name
        super_chr_start
        super_chr_end
        clone_super_start
        clone_super_end
        clone_chr_start
        clone_chr_end
        contig_clone_start
        contig_clone_end
        bacname
        bac_length
        source 
    /
);

Gramene::CDBI::EnsemblArabidopsis::SeqtablePrehelper->columns( 
    Ordered => qw/ 
        clone_id
        chr_id
        embl_acc
        embl_version
        htg_phase
        chromosome
        super_name
        super_chr_start
        super_chr_end
        clone_super_start
        clone_super_end
        clone_chr_start
        clone_chr_end
        contig_clone_start
        contig_clone_end
        bacname
        bac_length
        source 
    /
);

sub to_string {
    my $self = shift;
    return $self->embl_acc || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'clone_id' => 'number',
        'chr_id' => 'number',
        'embl_acc' => 'string',
        'embl_version' => 'number',
        'htg_phase' => 'number',
        'chromosome' => 'string',
        'super_name' => 'string',
        'super_chr_start' => 'number',
        'super_chr_end' => 'number',
        'clone_super_start' => 'number',
        'clone_super_end' => 'number',
        'clone_chr_start' => 'number',
        'clone_chr_end' => 'number',
        'contig_clone_start' => 'number',
        'contig_clone_end' => 'number',
        'bacname' => 'string',
        'bac_length' => 'number',
        'source' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::UnconventionalTranscriptAssociation; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::UnconventionalTranscriptAssociation->table('unconventional_transcript_association');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::UnconventionalTranscriptAssociation->columns( 
    All => qw/ 
        transcript_id
        gene_id
        interaction_type 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::UnconventionalTranscriptAssociation->columns( 
    Essential => qw/ 
        transcript_id
        gene_id
        interaction_type 
    /
);

Gramene::CDBI::EnsemblArabidopsis::UnconventionalTranscriptAssociation->columns( 
    Ordered => qw/ 
        transcript_id
        gene_id
        interaction_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->interaction_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'transcript_id' => 'number',
        'gene_id' => 'number',
        'interaction_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::EnsemblArabidopsis::UnmappedReason; {

BEGIN { our @ISA = qw( Gramene::CDBI::EnsemblArabidopsis ); };
# use base qw( Gramene::CDBI::EnsemblArabidopsis );

Gramene::CDBI::EnsemblArabidopsis::UnmappedReason->table('unmapped_reason');

our @extras = ();
Gramene::CDBI::EnsemblArabidopsis::UnmappedReason->columns( 
    All => qw/ 
        unmapped_reason_id
        summary_description
        full_description 
    /, @extras
);

Gramene::CDBI::EnsemblArabidopsis::UnmappedReason->columns( 
    Essential => qw/ 
        unmapped_reason_id
        summary_description
        full_description 
    /
);

Gramene::CDBI::EnsemblArabidopsis::UnmappedReason->columns( 
    Ordered => qw/ 
        unmapped_reason_id
        summary_description
        full_description 
    /
);

sub to_string {
    my $self = shift;
    return $self->summary_description || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'unmapped_reason_id' => 'string',
        'summary_description' => 'string',
        'full_description' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::EnsemblArabidopsis::UnmappedReason->has_many( 
    'unmapped_objects',
    'Gramene::CDBI::EnsemblArabidopsis::UnmappedObject',
    'unmapped_reason_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            unmapped_object => [ 'unmapped_reason_id', 'unmapped_reason_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope



1;

__END__

# ----------------------------------------------------

=pod

=head1 NAME

Gramene::CDBI::EnsemblArabidopsis - Gramene::CDBI-base classes for EnsemblArabidopsis

=head1 VERSION

This documentation refers to version $Revision: 1.2.2.1 $.

=head1 SYNOPSIS

  use Gramene::CDBI::EnsemblArabidopsis;

  my @data   = Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactor->retrieve_all;
  my $object = Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactor->retrieve( 42 );

=head1 DESCRIPTION

This module contains Gramene::CDBI-base classes for the EnsemblArabidopsis 
schema.

=head1 METHODS 

=head2 config_name

  my $config_name = $self->config_name;

Returns the string 'ensembl_arabidopsis'.

=head2 represented_tables

  my @tables = $cdbi->represented_tables;

Returns the names of the tables in the module.  See the CLASSES below.

=cut

=head1 CLASSES

=head2 Gramene::CDBI::EnsemblArabidopsis::AltAllele

Corresponds to the table "alt_allele."  

Stringifies (C<to_name>) to "alt_allele_id".

Fields:

=over 4

=item * alt_allele_id

=item * gene_id

=back


Has A:

=over 4

=item * gene => Gramene::CDBI::EnsemblArabidopsis::Gene

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::Analysis

Corresponds to the table "analysis."  

Stringifies (C<to_name>) to "created".

Fields:

=over 4

=item * analysis_id

=item * created

=item * logic_name

=item * db

=item * db_version

=item * db_file

=item * program

=item * program_version

=item * program_file

=item * parameters

=item * module

=item * module_version

=item * gff_source

=item * gff_feature

=back



Has Many:

=over 4

=item * analysis_descriptions => Gramene::CDBI::EnsemblArabidopsis::AnalysisDescription

=item * density_types => Gramene::CDBI::EnsemblArabidopsis::DensityType

=item * ditag_features => Gramene::CDBI::EnsemblArabidopsis::DitagFeature

=item * dna_align_features => Gramene::CDBI::EnsemblArabidopsis::DnaAlignFeature

=item * genes => Gramene::CDBI::EnsemblArabidopsis::Gene

=item * identity_xrefs => Gramene::CDBI::EnsemblArabidopsis::IdentityXref

=item * marker_features => Gramene::CDBI::EnsemblArabidopsis::MarkerFeature

=item * oligo_features => Gramene::CDBI::EnsemblArabidopsis::OligoFeature

=item * prediction_transcripts => Gramene::CDBI::EnsemblArabidopsis::PredictionTranscript

=item * protein_align_features => Gramene::CDBI::EnsemblArabidopsis::ProteinAlignFeature

=item * protein_features => Gramene::CDBI::EnsemblArabidopsis::ProteinFeature

=item * qtl_features => Gramene::CDBI::EnsemblArabidopsis::QtlFeature

=item * regulatory_features => Gramene::CDBI::EnsemblArabidopsis::RegulatoryFeature

=item * regulatory_search_regions => Gramene::CDBI::EnsemblArabidopsis::RegulatorySearchRegion

=item * repeat_features => Gramene::CDBI::EnsemblArabidopsis::RepeatFeature

=item * simple_features => Gramene::CDBI::EnsemblArabidopsis::SimpleFeature

=item * transcripts => Gramene::CDBI::EnsemblArabidopsis::Transcript

=item * unmapped_objects => Gramene::CDBI::EnsemblArabidopsis::UnmappedObject

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::AnalysisDescription

Corresponds to the table "analysis_description."  

Stringifies (C<to_name>) to "description".

Fields:

=over 4

=item * analysis_id

=item * description

=item * display_label

=item * displayable

=item * web_data

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblArabidopsis::Analysis

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::Assembly

Corresponds to the table "assembly."  

Stringifies (C<to_name>) to "asm_start".

Fields:

=over 4

=item * asm_seq_region_id

=item * cmp_seq_region_id

=item * asm_start

=item * asm_end

=item * cmp_start

=item * cmp_end

=item * ori

=back


Has A:

=over 4

=item * asm_seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=item * cmp_seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::AssemblyException

Corresponds to the table "assembly_exception."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * assembly_exception_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * exc_type

=item * exc_seq_region_id

=item * exc_seq_region_start

=item * exc_seq_region_end

=item * ori

=back


Has A:

=over 4

=item * seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=item * exc_seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::AttribType

Corresponds to the table "attrib_type."  

Stringifies (C<to_name>) to "code".

Fields:

=over 4

=item * attrib_type_id

=item * code

=item * name

=item * description

=back



Has Many:

=over 4

=item * gene_attribs => Gramene::CDBI::EnsemblArabidopsis::GeneAttrib

=item * misc_attribs => Gramene::CDBI::EnsemblArabidopsis::MiscAttrib

=item * seq_region_attribs => Gramene::CDBI::EnsemblArabidopsis::SeqRegionAttrib

=item * transcript_attribs => Gramene::CDBI::EnsemblArabidopsis::TranscriptAttrib

=item * translation_attribs => Gramene::CDBI::EnsemblArabidopsis::TranslationAttrib

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::CoordSystem

Corresponds to the table "coord_system."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * coord_system_id

=item * name

=item * version

=item * rank

=item * attrib

=back



Has Many:

=over 4

=item * meta_coords => Gramene::CDBI::EnsemblArabidopsis::MetaCoord

=item * seq_regions => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::DensityFeature

Corresponds to the table "density_feature."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * density_feature_id

=item * density_type_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * density_value

=back


Has A:

=over 4

=item * density_type => Gramene::CDBI::EnsemblArabidopsis::DensityType

=item * seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::DensityType

Corresponds to the table "density_type."  

Stringifies (C<to_name>) to "block_size".

Fields:

=over 4

=item * density_type_id

=item * analysis_id

=item * block_size

=item * region_features

=item * value_type

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblArabidopsis::Analysis

=back


Has Many:

=over 4

=item * density_features => Gramene::CDBI::EnsemblArabidopsis::DensityFeature

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::Ditag

Corresponds to the table "ditag."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * ditag_id

=item * name

=item * type

=item * tag_count

=back



Has Many:

=over 4

=item * ditag_features => Gramene::CDBI::EnsemblArabidopsis::DitagFeature

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::DitagFeature

Corresponds to the table "ditag_feature."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * ditag_feature_id

=item * ditag_id

=item * ditag_pair_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * analysis_id

=item * hit_start

=item * hit_end

=item * hit_strand

=item * cigar_line

=item * ditag_side

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblArabidopsis::Analysis

=item * ditag => Gramene::CDBI::EnsemblArabidopsis::Ditag

=item * seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::Dna

Corresponds to the table "dna."  

Stringifies (C<to_name>) to "sequence".

Fields:

=over 4

=item * seq_region_id

=back


Has A:

=over 4

=item * seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::DnaAlignFeature

Corresponds to the table "dna_align_feature."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * dna_align_feature_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * hit_start

=item * hit_end

=item * hit_strand

=item * hit_name

=item * analysis_id

=item * score

=item * evalue

=item * perc_ident

=item * cigar_line

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblArabidopsis::Analysis

=item * seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::Dnac

Corresponds to the table "dnac."  

Stringifies (C<to_name>) to "sequence".

Fields:

=over 4

=item * seq_region_id

=item * n_line

=back


Has A:

=over 4

=item * seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::EnsemblGeneHelper

Corresponds to the table "ensembl_gene_helper."  

Stringifies (C<to_name>) to "description".

Fields:

=over 4

=item * gene_id

=item * description

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::Exon

Corresponds to the table "exon."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * exon_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * phase

=item * end_phase

=item * is_current

=back


Has A:

=over 4

=item * seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=back


Has Many:

=over 4

=item * exon_stable_ids => Gramene::CDBI::EnsemblArabidopsis::ExonStableId

=item * exon_transcripts => Gramene::CDBI::EnsemblArabidopsis::ExonTranscript

=item * supporting_features => Gramene::CDBI::EnsemblArabidopsis::SupportingFeature

=item * translations_start_exon => Gramene::CDBI::EnsemblArabidopsis::Translation

=item * translations_end_exon => Gramene::CDBI::EnsemblArabidopsis::Translation

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::ExonStableId

Corresponds to the table "exon_stable_id."  

Stringifies (C<to_name>) to "version".

Fields:

=over 4

=item * exon_id

=item * stable_id

=item * version

=item * created_date

=item * modified_date

=back


Has A:

=over 4

=item * exon => Gramene::CDBI::EnsemblArabidopsis::Exon

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::ExonTranscript

Corresponds to the table "exon_transcript."  

Stringifies (C<to_name>) to "exon_id".

Fields:

=over 4

=item * exon_id

=item * transcript_id

=item * rank

=back


Has A:

=over 4

=item * exon => Gramene::CDBI::EnsemblArabidopsis::Exon

=item * transcript => Gramene::CDBI::EnsemblArabidopsis::Transcript

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::ExternalDb

Corresponds to the table "external_db."  

Stringifies (C<to_name>) to "db_name".

Fields:

=over 4

=item * external_db_id

=item * db_name

=item * db_release

=item * status

=item * dbprimary_acc_linkable

=item * display_label_linkable

=item * priority

=item * db_display_name

=back



Has Many:

=over 4

=item * unmapped_objects => Gramene::CDBI::EnsemblArabidopsis::UnmappedObject

=item * xrefs => Gramene::CDBI::EnsemblArabidopsis::Xref

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::ExternalSynonym

Corresponds to the table "external_synonym."  

Stringifies (C<to_name>) to "xref_id".

Fields:

=over 4

=item * xref_id

=item * synonym

=back


Has A:

=over 4

=item * xref => Gramene::CDBI::EnsemblArabidopsis::Xref

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::Gene

Corresponds to the table "gene."  

Stringifies (C<to_name>) to "biotype".

Fields:

=over 4

=item * gene_id

=item * biotype

=item * analysis_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * display_xref_id

=item * source

=item * status

=item * description

=item * is_current

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblArabidopsis::Analysis

=item * seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=item * display_xref => Gramene::CDBI::EnsemblArabidopsis::Xref

=back


Has Many:

=over 4

=item * alt_alleles => Gramene::CDBI::EnsemblArabidopsis::AltAllele

=item * gene_attribs => Gramene::CDBI::EnsemblArabidopsis::GeneAttrib

=item * gene_stable_ids => Gramene::CDBI::EnsemblArabidopsis::GeneStableId

=item * regulatory_factor_codings => Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactorCoding

=item * transcripts => Gramene::CDBI::EnsemblArabidopsis::Transcript

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::GeneArchive

Corresponds to the table "gene_archive."  

Stringifies (C<to_name>) to "gene_version".

Fields:

=over 4

=item * gene_stable_id

=item * gene_version

=item * transcript_stable_id

=item * transcript_version

=item * translation_stable_id

=item * translation_version

=item * peptide_archive_id

=item * mapping_session_id

=back


Has A:

=over 4

=item * mapping_session => Gramene::CDBI::EnsemblArabidopsis::MappingSession

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::GeneAttrib

Corresponds to the table "gene_attrib."  

Stringifies (C<to_name>) to "value".

Fields:

=over 4

=item * gene_id

=item * attrib_type_id

=item * value

=back


Has A:

=over 4

=item * attrib_type => Gramene::CDBI::EnsemblArabidopsis::AttribType

=item * gene => Gramene::CDBI::EnsemblArabidopsis::Gene

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::GeneStableId

Corresponds to the table "gene_stable_id."  

Stringifies (C<to_name>) to "version".

Fields:

=over 4

=item * gene_id

=item * stable_id

=item * version

=item * created_date

=item * modified_date

=back


Has A:

=over 4

=item * gene => Gramene::CDBI::EnsemblArabidopsis::Gene

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::GoXref

Corresponds to the table "go_xref."  

Stringifies (C<to_name>) to "linkage_type".

Fields:

=over 4

=item * object_xref_id

=item * linkage_type

=back


Has A:

=over 4

=item * object_xref => Gramene::CDBI::EnsemblArabidopsis::ObjectXref

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::GrameneGeneExtHelper

Corresponds to the table "gramene_gene_ext_helper."  

Stringifies (C<to_name>) to "external_name".

Fields:

=over 4

=item * gene_id

=item * stable_id

=item * external_name

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::GrameneGeneXTmp0

Corresponds to the table "gramene_gene_x_tmp0."  

Stringifies (C<to_name>) to "display_label".

Fields:

=over 4

=item * gene_id

=item * display_label

=item * dbprimary_acc

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::GrameneGeneXTmp1

Corresponds to the table "gramene_gene_x_tmp1."  

Stringifies (C<to_name>) to "external_name".

Fields:

=over 4

=item * gene_id

=item * external_name

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::IdentityXref

Corresponds to the table "identity_xref."  

Stringifies (C<to_name>) to "query_identity".

Fields:

=over 4

=item * object_xref_id

=item * query_identity

=item * target_identity

=item * hit_start

=item * hit_end

=item * translation_start

=item * translation_end

=item * cigar_line

=item * score

=item * evalue

=item * analysis_id

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblArabidopsis::Analysis

=item * object_xref => Gramene::CDBI::EnsemblArabidopsis::ObjectXref

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::InputIdAnalysis

Corresponds to the table "input_id_analysis."  

Stringifies (C<to_name>) to "input_id_type".

Fields:

=over 4

=item * analysis_id

=item * input_id

=item * input_id_type

=item * created

=item * runhost

=item * db_version

=item * result

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::InputIdTypeAnalysis

Corresponds to the table "input_id_type_analysis."  

Stringifies (C<to_name>) to "input_id_type".

Fields:

=over 4

=item * analysis_id

=item * input_id_type

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::InsertCoords

Corresponds to the table "insert_coords."  

Stringifies (C<to_name>) to "EMBL_ID".

Fields:

=over 4

=item * coord_name

=item * e_value

=item * start_coord

=item * end_coord

=item * EMBL_ID

=item * EMBL_name

=item * coord_name_short

=item * chr

=item * ori

=item * NASC_code

=item * NASC_name

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::Interpro

Corresponds to the table "interpro."  

Stringifies (C<to_name>) to "interpro_ac".

Fields:

=over 4

=item * interpro_ac

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::Job

Corresponds to the table "job."  

Stringifies (C<to_name>) to "stdout_file".

Fields:

=over 4

=item * job_id

=item * input_id

=item * analysis_id

=item * submission_id

=item * stdout_file

=item * stderr_file

=item * retry_count

=item * temp_dir

=item * exec_host

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::JobStatus

Corresponds to the table "job_status."  

Stringifies (C<to_name>) to "status".

Fields:

=over 4

=item * job_id

=item * status

=item * time

=item * is_current

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::Karyotype

Corresponds to the table "karyotype."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * karyotype_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * band

=item * stain

=back


Has A:

=over 4

=item * seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::Map

Corresponds to the table "map."  

Stringifies (C<to_name>) to "map_name".

Fields:

=over 4

=item * map_id

=item * map_name

=back



Has Many:

=over 4

=item * marker_map_locations => Gramene::CDBI::EnsemblArabidopsis::MarkerMapLocation

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::MappingSession

Corresponds to the table "mapping_session."  

Stringifies (C<to_name>) to "old_db_name".

Fields:

=over 4

=item * mapping_session_id

=item * old_db_name

=item * new_db_name

=item * old_release

=item * new_release

=item * old_assembly

=item * new_assembly

=item * created

=back



Has Many:

=over 4

=item * gene_archives => Gramene::CDBI::EnsemblArabidopsis::GeneArchive

=item * stable_id_events => Gramene::CDBI::EnsemblArabidopsis::StableIdEvent

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::Marker

Corresponds to the table "marker."  

Stringifies (C<to_name>) to "left_primer".

Fields:

=over 4

=item * marker_id

=item * display_marker_synonym_id

=item * left_primer

=item * right_primer

=item * min_primer_dist

=item * max_primer_dist

=item * priority

=item * type

=back


Has A:

=over 4

=item * display_marker_synonym => Gramene::CDBI::EnsemblArabidopsis::MarkerSynonym

=back


Has Many:

=over 4

=item * marker_features => Gramene::CDBI::EnsemblArabidopsis::MarkerFeature

=item * marker_map_locations => Gramene::CDBI::EnsemblArabidopsis::MarkerMapLocation

=item * marker_synonyms => Gramene::CDBI::EnsemblArabidopsis::MarkerSynonym

=item * qtls_flank_marker_1 => Gramene::CDBI::EnsemblArabidopsis::Qtl

=item * qtls_flank_marker_2 => Gramene::CDBI::EnsemblArabidopsis::Qtl

=item * qtls_peak_marker => Gramene::CDBI::EnsemblArabidopsis::Qtl

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::MarkerFeature

Corresponds to the table "marker_feature."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * marker_feature_id

=item * marker_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * analysis_id

=item * map_weight

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblArabidopsis::Analysis

=item * marker => Gramene::CDBI::EnsemblArabidopsis::Marker

=item * seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::MarkerMapLocation

Corresponds to the table "marker_map_location."  

Stringifies (C<to_name>) to "chromosome_name".

Fields:

=over 4

=item * marker_id

=item * map_id

=item * chromosome_name

=item * marker_synonym_id

=item * position

=item * lod_score

=back


Has A:

=over 4

=item * map => Gramene::CDBI::EnsemblArabidopsis::Map

=item * marker => Gramene::CDBI::EnsemblArabidopsis::Marker

=item * marker_synonym => Gramene::CDBI::EnsemblArabidopsis::MarkerSynonym

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::MarkerSynonym

Corresponds to the table "marker_synonym."  

Stringifies (C<to_name>) to "source".

Fields:

=over 4

=item * marker_synonym_id

=item * marker_id

=item * source

=item * name

=back


Has A:

=over 4

=item * marker => Gramene::CDBI::EnsemblArabidopsis::Marker

=back


Has Many:

=over 4

=item * markers => Gramene::CDBI::EnsemblArabidopsis::Marker

=item * marker_map_locations => Gramene::CDBI::EnsemblArabidopsis::MarkerMapLocation

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::Meta

Corresponds to the table "meta."  

Stringifies (C<to_name>) to "meta_key".

Fields:

=over 4

=item * meta_id

=item * meta_key

=item * meta_value

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::MetaCoord

Corresponds to the table "meta_coord."  

Stringifies (C<to_name>) to "table_name".

Fields:

=over 4

=item * table_name

=item * coord_system_id

=item * max_length

=back


Has A:

=over 4

=item * coord_system => Gramene::CDBI::EnsemblArabidopsis::CoordSystem

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::MiscAttrib

Corresponds to the table "misc_attrib."  

Stringifies (C<to_name>) to "value".

Fields:

=over 4

=item * misc_feature_id

=item * attrib_type_id

=item * value

=back


Has A:

=over 4

=item * attrib_type => Gramene::CDBI::EnsemblArabidopsis::AttribType

=item * misc_feature => Gramene::CDBI::EnsemblArabidopsis::MiscFeature

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::MiscAttribHelper

Corresponds to the table "misc_attrib_helper."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * name

=item * attrib_name

=item * attrib_value

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::MiscFeature

Corresponds to the table "misc_feature."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * misc_feature_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=back


Has A:

=over 4

=item * seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=back


Has Many:

=over 4

=item * misc_attribs => Gramene::CDBI::EnsemblArabidopsis::MiscAttrib

=item * misc_feature_misc_sets => Gramene::CDBI::EnsemblArabidopsis::MiscFeatureMiscSet

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::MiscFeatureMiscSet

Corresponds to the table "misc_feature_misc_set."  

Stringifies (C<to_name>) to "misc_feature_id".

Fields:

=over 4

=item * misc_feature_id

=item * misc_set_id

=back


Has A:

=over 4

=item * misc_feature => Gramene::CDBI::EnsemblArabidopsis::MiscFeature

=item * misc_set => Gramene::CDBI::EnsemblArabidopsis::MiscSet

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::MiscSet

Corresponds to the table "misc_set."  

Stringifies (C<to_name>) to "code".

Fields:

=over 4

=item * misc_set_id

=item * code

=item * name

=item * description

=item * max_length

=back



Has Many:

=over 4

=item * misc_feature_misc_sets => Gramene::CDBI::EnsemblArabidopsis::MiscFeatureMiscSet

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::ObjectXref

Corresponds to the table "object_xref."  

Stringifies (C<to_name>) to "ensembl_object_type".

Fields:

=over 4

=item * object_xref_id

=item * ensembl_id

=item * ensembl_object_type

=item * xref_id

=back


Has A:

=over 4

=item * xref => Gramene::CDBI::EnsemblArabidopsis::Xref

=back


Has Many:

=over 4

=item * go_xrefs => Gramene::CDBI::EnsemblArabidopsis::GoXref

=item * identity_xrefs => Gramene::CDBI::EnsemblArabidopsis::IdentityXref

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::OligoArray

Corresponds to the table "oligo_array."  

Stringifies (C<to_name>) to "probe_setsize".

Fields:

=over 4

=item * oligo_array_id

=item * parent_array_id

=item * probe_setsize

=item * name

=item * type

=back


Has A:

=over 4

=item * parent_array => Gramene::CDBI::EnsemblArabidopsis::OligoArray

=back


Has Many:

=over 4

=item * oligo_arrays => Gramene::CDBI::EnsemblArabidopsis::OligoArray

=item * oligo_probes => Gramene::CDBI::EnsemblArabidopsis::OligoProbe

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::OligoFeature

Corresponds to the table "oligo_feature."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * oligo_feature_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * mismatches

=item * oligo_probe_id

=item * analysis_id

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblArabidopsis::Analysis

=item * oligo_probe => Gramene::CDBI::EnsemblArabidopsis::OligoProbe

=item * seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::OligoProbe

Corresponds to the table "oligo_probe."  

Stringifies (C<to_name>) to "probeset".

Fields:

=over 4

=item * oligo_probe_id

=item * oligo_array_id

=item * probeset

=item * name

=item * description

=item * length

=back


Has A:

=over 4

=item * oligo_array => Gramene::CDBI::EnsemblArabidopsis::OligoArray

=back


Has Many:

=over 4

=item * oligo_features => Gramene::CDBI::EnsemblArabidopsis::OligoFeature

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::PeptideArchive

Corresponds to the table "peptide_archive."  

Stringifies (C<to_name>) to "md5_checksum".

Fields:

=over 4

=item * peptide_archive_id

=item * md5_checksum

=item * peptide_seq

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::PredictionExon

Corresponds to the table "prediction_exon."  

Stringifies (C<to_name>) to "exon_rank".

Fields:

=over 4

=item * prediction_exon_id

=item * prediction_transcript_id

=item * exon_rank

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * start_phase

=item * score

=item * p_value

=back


Has A:

=over 4

=item * prediction_transcript => Gramene::CDBI::EnsemblArabidopsis::PredictionTranscript

=item * seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::PredictionTranscript

Corresponds to the table "prediction_transcript."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * prediction_transcript_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * analysis_id

=item * display_label

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblArabidopsis::Analysis

=item * seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=back


Has Many:

=over 4

=item * prediction_exons => Gramene::CDBI::EnsemblArabidopsis::PredictionExon

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::ProteinAlignFeature

Corresponds to the table "protein_align_feature."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * protein_align_feature_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * hit_start

=item * hit_end

=item * hit_name

=item * analysis_id

=item * score

=item * evalue

=item * perc_ident

=item * cigar_line

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblArabidopsis::Analysis

=item * seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::ProteinFeature

Corresponds to the table "protein_feature."  

Stringifies (C<to_name>) to "seq_start".

Fields:

=over 4

=item * protein_feature_id

=item * translation_id

=item * seq_start

=item * seq_end

=item * hit_start

=item * hit_end

=item * hit_id

=item * analysis_id

=item * score

=item * evalue

=item * perc_ident

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblArabidopsis::Analysis

=item * translation => Gramene::CDBI::EnsemblArabidopsis::Translation

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::Qtl

Corresponds to the table "qtl."  

Stringifies (C<to_name>) to "trait".

Fields:

=over 4

=item * qtl_id

=item * trait

=item * lod_score

=item * flank_marker_id_1

=item * flank_marker_id_2

=item * peak_marker_id

=back


Has A:

=over 4

=item * peak_marker => Gramene::CDBI::EnsemblArabidopsis::Marker

=back


Has Many:

=over 4

=item * qtl_features => Gramene::CDBI::EnsemblArabidopsis::QtlFeature

=item * qtl_synonyms => Gramene::CDBI::EnsemblArabidopsis::QtlSynonym

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::QtlFeature

Corresponds to the table "qtl_feature."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * qtl_id

=item * analysis_id

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblArabidopsis::Analysis

=item * qtl => Gramene::CDBI::EnsemblArabidopsis::Qtl

=item * seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::QtlSynonym

Corresponds to the table "qtl_synonym."  

Stringifies (C<to_name>) to "source_database".

Fields:

=over 4

=item * qtl_synonym_id

=item * qtl_id

=item * source_database

=item * source_primary_id

=back


Has A:

=over 4

=item * qtl => Gramene::CDBI::EnsemblArabidopsis::Qtl

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactor

Corresponds to the table "regulatory_factor."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * regulatory_factor_id

=item * name

=item * type

=back



Has Many:

=over 4

=item * regulatory_factor_codings => Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactorCoding

=item * regulatory_features => Gramene::CDBI::EnsemblArabidopsis::RegulatoryFeature

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactorCoding

Corresponds to the table "regulatory_factor_coding."  

Stringifies (C<to_name>) to "regulatory_factor_id".

Fields:

=over 4

=item * regulatory_factor_id

=item * transcript_id

=item * gene_id

=back


Has A:

=over 4

=item * gene => Gramene::CDBI::EnsemblArabidopsis::Gene

=item * regulatory_factor => Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactor

=item * transcript => Gramene::CDBI::EnsemblArabidopsis::Transcript

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::RegulatoryFeature

Corresponds to the table "regulatory_feature."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * regulatory_feature_id

=item * name

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * analysis_id

=item * regulatory_factor_id

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblArabidopsis::Analysis

=item * regulatory_factor => Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactor

=item * seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::RegulatoryFeatureObject

Corresponds to the table "regulatory_feature_object."  

Stringifies (C<to_name>) to "ensembl_object_type".

Fields:

=over 4

=item * regulatory_feature_id

=item * ensembl_object_type

=item * ensembl_object_id

=item * influence

=item * evidence

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::RegulatorySearchRegion

Corresponds to the table "regulatory_search_region."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * regulatory_search_region_id

=item * name

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * ensembl_object_type

=item * ensembl_object_id

=item * analysis_id

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblArabidopsis::Analysis

=item * seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::RepeatConsensus

Corresponds to the table "repeat_consensus."  

Stringifies (C<to_name>) to "repeat_name".

Fields:

=over 4

=item * repeat_consensus_id

=item * repeat_name

=item * repeat_class

=item * repeat_type

=item * repeat_consensus

=back



Has Many:

=over 4

=item * repeat_features => Gramene::CDBI::EnsemblArabidopsis::RepeatFeature

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::RepeatFeature

Corresponds to the table "repeat_feature."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * repeat_feature_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * repeat_start

=item * repeat_end

=item * repeat_consensus_id

=item * analysis_id

=item * score

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblArabidopsis::Analysis

=item * repeat_consensus => Gramene::CDBI::EnsemblArabidopsis::RepeatConsensus

=item * seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::RuleConditions

Corresponds to the table "rule_conditions."  

Stringifies (C<to_name>) to "condition".

Fields:

=over 4

=item * rule_id

=item * condition

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::RuleGoal

Corresponds to the table "rule_goal."  

Stringifies (C<to_name>) to "goal".

Fields:

=over 4

=item * rule_id

=item * goal

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::SeqRegion

Corresponds to the table "seq_region."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * seq_region_id

=item * name

=item * coord_system_id

=item * length

=back


Has A:

=over 4

=item * coord_system => Gramene::CDBI::EnsemblArabidopsis::CoordSystem

=back


Has Many:

=over 4

=item * assemblies_asm_seq_region => Gramene::CDBI::EnsemblArabidopsis::Assembly

=item * assemblies_cmp_seq_region => Gramene::CDBI::EnsemblArabidopsis::Assembly

=item * assembly_exceptions_seq_region => Gramene::CDBI::EnsemblArabidopsis::AssemblyException

=item * assembly_exceptions_exc_seq_region => Gramene::CDBI::EnsemblArabidopsis::AssemblyException

=item * density_features => Gramene::CDBI::EnsemblArabidopsis::DensityFeature

=item * ditag_features => Gramene::CDBI::EnsemblArabidopsis::DitagFeature

=item * dnas => Gramene::CDBI::EnsemblArabidopsis::Dna

=item * dna_align_features => Gramene::CDBI::EnsemblArabidopsis::DnaAlignFeature

=item * dnacs => Gramene::CDBI::EnsemblArabidopsis::Dnac

=item * exons => Gramene::CDBI::EnsemblArabidopsis::Exon

=item * genes => Gramene::CDBI::EnsemblArabidopsis::Gene

=item * karyotypes => Gramene::CDBI::EnsemblArabidopsis::Karyotype

=item * marker_features => Gramene::CDBI::EnsemblArabidopsis::MarkerFeature

=item * misc_features => Gramene::CDBI::EnsemblArabidopsis::MiscFeature

=item * oligo_features => Gramene::CDBI::EnsemblArabidopsis::OligoFeature

=item * prediction_exons => Gramene::CDBI::EnsemblArabidopsis::PredictionExon

=item * prediction_transcripts => Gramene::CDBI::EnsemblArabidopsis::PredictionTranscript

=item * protein_align_features => Gramene::CDBI::EnsemblArabidopsis::ProteinAlignFeature

=item * qtl_features => Gramene::CDBI::EnsemblArabidopsis::QtlFeature

=item * regulatory_features => Gramene::CDBI::EnsemblArabidopsis::RegulatoryFeature

=item * regulatory_search_regions => Gramene::CDBI::EnsemblArabidopsis::RegulatorySearchRegion

=item * repeat_features => Gramene::CDBI::EnsemblArabidopsis::RepeatFeature

=item * seq_region_attribs => Gramene::CDBI::EnsemblArabidopsis::SeqRegionAttrib

=item * simple_features => Gramene::CDBI::EnsemblArabidopsis::SimpleFeature

=item * transcripts => Gramene::CDBI::EnsemblArabidopsis::Transcript

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::SeqRegionAttrib

Corresponds to the table "seq_region_attrib."  

Stringifies (C<to_name>) to "value".

Fields:

=over 4

=item * seq_region_id

=item * attrib_type_id

=item * value

=back


Has A:

=over 4

=item * attrib_type => Gramene::CDBI::EnsemblArabidopsis::AttribType

=item * seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::SeqtableCmapTmp

Corresponds to the table "seqtable_cmap_tmp."  

Stringifies (C<to_name>) to "marker_name".

Fields:

=over 4

=item * marker_name

=item * cms_aid

=item * cf_aid

=item * map_name

=item * cm_aid

=item * position

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::SeqtableCmapTmp0

Corresponds to the table "seqtable_cmap_tmp0."  

Stringifies (C<to_name>) to "cname".

Fields:

=over 4

=item * cname

=item * cms_aid

=item * map_name

=item * cf_aid

=item * cm_aid

=item * position

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::SeqtableDaf0

Corresponds to the table "seqtable_daf0."  

Stringifies (C<to_name>) to "embl_acc".

Fields:

=over 4

=item * embl_acc

=item * daf_chr_start

=item * daf_chr_end

=item * chr_strand

=item * chromosome

=item * superctg_end

=item * superctg_name

=item * superctg_ori

=item * superctg_start

=item * daf_clone_start

=item * daf_clone_end

=item * daf_super_start

=item * daf_super_end

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::SeqtableDaf1

Corresponds to the table "seqtable_daf1."  

Stringifies (C<to_name>) to "embl_acc".

Fields:

=over 4

=item * embl_acc

=item * chromosome

=item * chr_start

=item * chr_end

=item * superctg_name

=item * super_start

=item * super_end

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::SeqtableHelper

Corresponds to the table "seqtable_helper."  

Stringifies (C<to_name>) to "bacname".

Fields:

=over 4

=item * clone_id

=item * bacname

=item * chromosome

=item * embl_acc

=item * embl_version

=item * htg_phase

=item * super_name

=item * super_chr_start

=item * super_chr_end

=item * clone_super_start

=item * clone_super_end

=item * clone_chr_start

=item * clone_chr_end

=item * marker_name

=item * marker_type

=item * jrgp_rflp_cm

=item * jrgp_rflp_map

=item * jrgp_rflp_feature

=item * cornell_rflp_cm

=item * cornell_rflp_map

=item * cornell_rflp_feature

=item * irmi_ssr_cm

=item * irmi_ssr_map

=item * irmi_ssr_feature

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::SeqtableMarkerTmp

Corresponds to the table "seqtable_marker_tmp."  

Stringifies (C<to_name>) to "mname".

Fields:

=over 4

=item * clone_id

=item * mname

=item * mtype

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::SeqtablePrehelper

Corresponds to the table "seqtable_prehelper."  

Stringifies (C<to_name>) to "embl_acc".

Fields:

=over 4

=item * clone_id

=item * chr_id

=item * embl_acc

=item * embl_version

=item * htg_phase

=item * chromosome

=item * super_name

=item * super_chr_start

=item * super_chr_end

=item * clone_super_start

=item * clone_super_end

=item * clone_chr_start

=item * clone_chr_end

=item * contig_clone_start

=item * contig_clone_end

=item * bacname

=item * bac_length

=item * source

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::SimpleFeature

Corresponds to the table "simple_feature."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * simple_feature_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * display_label

=item * analysis_id

=item * score

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblArabidopsis::Analysis

=item * seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::StableIdEvent

Corresponds to the table "stable_id_event."  

Stringifies (C<to_name>) to "old_version".

Fields:

=over 4

=item * old_stable_id

=item * old_version

=item * new_stable_id

=item * new_version

=item * mapping_session_id

=item * type

=item * score

=back


Has A:

=over 4

=item * mapping_session => Gramene::CDBI::EnsemblArabidopsis::MappingSession

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::SupportingFeature

Corresponds to the table "supporting_feature."  

Stringifies (C<to_name>) to "feature_type".

Fields:

=over 4

=item * exon_id

=item * feature_type

=item * feature_id

=back


Has A:

=over 4

=item * exon => Gramene::CDBI::EnsemblArabidopsis::Exon

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::Transcript

Corresponds to the table "transcript."  

Stringifies (C<to_name>) to "seq_region_start".

Fields:

=over 4

=item * transcript_id

=item * gene_id

=item * analysis_id

=item * seq_region_id

=item * seq_region_start

=item * seq_region_end

=item * seq_region_strand

=item * display_xref_id

=item * biotype

=item * status

=item * description

=item * is_current

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblArabidopsis::Analysis

=item * gene => Gramene::CDBI::EnsemblArabidopsis::Gene

=item * seq_region => Gramene::CDBI::EnsemblArabidopsis::SeqRegion

=item * display_xref => Gramene::CDBI::EnsemblArabidopsis::Xref

=back


Has Many:

=over 4

=item * exon_transcripts => Gramene::CDBI::EnsemblArabidopsis::ExonTranscript

=item * regulatory_factor_codings => Gramene::CDBI::EnsemblArabidopsis::RegulatoryFactorCoding

=item * transcript_attribs => Gramene::CDBI::EnsemblArabidopsis::TranscriptAttrib

=item * transcript_stable_ids => Gramene::CDBI::EnsemblArabidopsis::TranscriptStableId

=item * transcript_supporting_features => Gramene::CDBI::EnsemblArabidopsis::TranscriptSupportingFeature

=item * translations => Gramene::CDBI::EnsemblArabidopsis::Translation

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::TranscriptAttrib

Corresponds to the table "transcript_attrib."  

Stringifies (C<to_name>) to "value".

Fields:

=over 4

=item * transcript_id

=item * attrib_type_id

=item * value

=back


Has A:

=over 4

=item * attrib_type => Gramene::CDBI::EnsemblArabidopsis::AttribType

=item * transcript => Gramene::CDBI::EnsemblArabidopsis::Transcript

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::TranscriptStableId

Corresponds to the table "transcript_stable_id."  

Stringifies (C<to_name>) to "version".

Fields:

=over 4

=item * transcript_id

=item * stable_id

=item * version

=item * created_date

=item * modified_date

=back


Has A:

=over 4

=item * transcript => Gramene::CDBI::EnsemblArabidopsis::Transcript

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::TranscriptSupportingFeature

Corresponds to the table "transcript_supporting_feature."  

Stringifies (C<to_name>) to "feature_type".

Fields:

=over 4

=item * transcript_id

=item * feature_type

=item * feature_id

=back


Has A:

=over 4

=item * transcript => Gramene::CDBI::EnsemblArabidopsis::Transcript

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::Translation

Corresponds to the table "translation."  

Stringifies (C<to_name>) to "seq_start".

Fields:

=over 4

=item * translation_id

=item * transcript_id

=item * seq_start

=item * start_exon_id

=item * seq_end

=item * end_exon_id

=back


Has A:

=over 4

=item * start_exon => Gramene::CDBI::EnsemblArabidopsis::Exon

=item * end_exon => Gramene::CDBI::EnsemblArabidopsis::Exon

=item * transcript => Gramene::CDBI::EnsemblArabidopsis::Transcript

=back


Has Many:

=over 4

=item * protein_features => Gramene::CDBI::EnsemblArabidopsis::ProteinFeature

=item * translation_attribs => Gramene::CDBI::EnsemblArabidopsis::TranslationAttrib

=item * translation_stable_ids => Gramene::CDBI::EnsemblArabidopsis::TranslationStableId

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::TranslationAttrib

Corresponds to the table "translation_attrib."  

Stringifies (C<to_name>) to "value".

Fields:

=over 4

=item * translation_id

=item * attrib_type_id

=item * value

=back


Has A:

=over 4

=item * attrib_type => Gramene::CDBI::EnsemblArabidopsis::AttribType

=item * translation => Gramene::CDBI::EnsemblArabidopsis::Translation

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::TranslationStableId

Corresponds to the table "translation_stable_id."  

Stringifies (C<to_name>) to "version".

Fields:

=over 4

=item * translation_id

=item * stable_id

=item * version

=item * created_date

=item * modified_date

=back


Has A:

=over 4

=item * translation => Gramene::CDBI::EnsemblArabidopsis::Translation

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::UnconventionalTranscriptAssociation

Corresponds to the table "unconventional_transcript_association."  

Stringifies (C<to_name>) to "interaction_type".

Fields:

=over 4

=item * transcript_id

=item * gene_id

=item * interaction_type

=back




=head2 Gramene::CDBI::EnsemblArabidopsis::UnmappedObject

Corresponds to the table "unmapped_object."  

Stringifies (C<to_name>) to "type".

Fields:

=over 4

=item * unmapped_object_id

=item * type

=item * analysis_id

=item * external_db_id

=item * identifier

=item * unmapped_reason_id

=item * query_score

=item * target_score

=item * ensembl_id

=item * ensembl_object_type

=back


Has A:

=over 4

=item * analysis => Gramene::CDBI::EnsemblArabidopsis::Analysis

=item * external_db => Gramene::CDBI::EnsemblArabidopsis::ExternalDb

=item * unmapped_reason => Gramene::CDBI::EnsemblArabidopsis::UnmappedReason

=back



=head2 Gramene::CDBI::EnsemblArabidopsis::UnmappedReason

Corresponds to the table "unmapped_reason."  

Stringifies (C<to_name>) to "summary_description".

Fields:

=over 4

=item * unmapped_reason_id

=item * summary_description

=item * full_description

=back



Has Many:

=over 4

=item * unmapped_objects => Gramene::CDBI::EnsemblArabidopsis::UnmappedObject

=back


=head2 Gramene::CDBI::EnsemblArabidopsis::Xref

Corresponds to the table "xref."  

Stringifies (C<to_name>) to "dbprimary_acc".

Fields:

=over 4

=item * xref_id

=item * external_db_id

=item * dbprimary_acc

=item * display_label

=item * version

=item * description

=item * info_type

=item * info_text

=back


Has A:

=over 4

=item * external_db => Gramene::CDBI::EnsemblArabidopsis::ExternalDb

=back


Has Many:

=over 4

=item * external_synonyms => Gramene::CDBI::EnsemblArabidopsis::ExternalSynonym

=item * genes => Gramene::CDBI::EnsemblArabidopsis::Gene

=item * object_xrefs => Gramene::CDBI::EnsemblArabidopsis::ObjectXref

=item * transcripts => Gramene::CDBI::EnsemblArabidopsis::Transcript

=back


=head1 SEE ALSO

Gramene::CDBI.

=head1 AUTHOR

Generated by Ken Youens-Clark on Tue Jun 12 10:49:19 2007.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
