package Gramene::CDBI::DiversityMaize;

use strict;

use base qw( Gramene::CDBI );

use Carp qw( croak );
use Gramene::Config;
use Gramene::Utils qw( table_name_to_gramene_cdbi_class );
use Readonly;

Readonly my $CONFIG_NAME => 'diversity_maize';

my ( $DBH ); # global database handle

# ------------------------------------------------
sub db_Main {

=pod

=head2 db_Main

  my $db = $cdbi->db_Main;

This returns a DBI database handle.  If you choose to override this, you
must include "RootClass => 'DBIx::ContextualFetch'" as an option to
DBI->connect.

=cut

    my $class = shift;

    if ( !defined $DBH ) {
        my $config = config();

        $DBH = DBI->connect(
            $config->{'db_dsn'},
            $config->{'db_user'},
            $config->{'db_pass'},
            { __PACKAGE__->_default_attributes },
        );
    }

    return $DBH;
} 

# ------------------------------------------------
sub config {

=pod

=head2 config

  my $config = $cdbi->config;
  my %config = $cdbi->config;

Returns the module's configuration from "gramene.conf."

=cut

    my $gconfig = Gramene::Config->new or croak 'No config file object';
    my $config  = $gconfig->get( $CONFIG_NAME )
                  or croak "Bad configuration section '$CONFIG_NAME'";

    return wantarray ? %$config : $config;
}

# ------------------------------------------------
sub table_name_to_class {

=pod

=head2 table_name_to_class

  my $class = $cdbi->table_name_to_class('foo');

Returns the something like "Gramene::CDBI::ModuleName::Foo."

=cut

    my ( $self, $table ) = @_;

    return table_name_to_gramene_cdbi_class( $CONFIG_NAME, $table );
}

# ------------------------------------------------
sub represented_tables {
    qw/
        cdv_allele_curated_allele
        cdv_curated_allele
        cdv_curation
        cdv_map_feature
        cdv_map_feature_annotation
        cdv_map_feature_annotation_type
        cdv_marker
        cdv_marker_annotation
        cdv_marker_annotation_type
        cdv_passport_group
        cdv_passport_set
        cdv_reason
        cdv_source
        div_aa_annotation
        div_accession_collecting
        div_allele
        div_allele_assay
        div_annotation_type
        div_experiment
        div_generation
        div_locality
        div_obs_unit
        div_obs_unit_sample
        div_passport
        div_poly_type
        div_scoring_tech_type
        div_statistic_type
        div_stock
        div_stock_parent
        div_synonym
        div_taxonomy
        div_trait
        div_trait_uom
        div_treatment
        div_treatment_uom
        div_unit_of_measure
        gdpdm_search
        qtl_analysis_method
        qtl_analysis_result
        qtl_marker
        qtl_obs_unit
        qtl_obs_unit_set
        qtl_qtl
        qtl_study
        qtl_trait
        xref
        xref_type

    /
}

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::CdvAlleleCuratedAllele; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::CdvAlleleCuratedAllele->table('cdv_allele_curated_allele');


sub cdv_curated_allele {
    my $self = shift;
    return $self->cdv_curated_allele_id;
}
    

sub cdv_curation {
    my $self = shift;
    return $self->cdv_curation_id;
}
    

sub div_allele {
    my $self = shift;
    return $self->div_allele_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::CdvAlleleCuratedAllele->columns( 
    All => qw/ 
        cdv_allele_curated_allele_id
        cdv_curated_allele_id
        div_allele_id
        cdv_curation_id 
    /, @extras
);

Gramene::CDBI::DiversityMaize::CdvAlleleCuratedAllele->columns( 
    Essential => qw/ 
        cdv_allele_curated_allele_id
        cdv_curated_allele_id
        div_allele_id
        cdv_curation_id 
    /
);

Gramene::CDBI::DiversityMaize::CdvAlleleCuratedAllele->columns( 
    Ordered => qw/ 
        cdv_allele_curated_allele_id
        cdv_curated_allele_id
        div_allele_id
        cdv_curation_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->cdv_allele_curated_allele_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'cdv_allele_curated_allele_id' => 'number',
        'cdv_curated_allele_id' => 'number',
        'div_allele_id' => 'number',
        'cdv_curation_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::CdvAlleleCuratedAllele->has_a(
    cdv_curated_allele_id => 'Gramene::CDBI::DiversityMaize::CdvCuratedAllele'
);

Gramene::CDBI::DiversityMaize::CdvAlleleCuratedAllele->has_a(
    cdv_curation_id => 'Gramene::CDBI::DiversityMaize::CdvCuration'
);

Gramene::CDBI::DiversityMaize::CdvAlleleCuratedAllele->has_a(
    div_allele_id => 'Gramene::CDBI::DiversityMaize::DivAllele'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_curated_allele => [ 'cdv_curated_allele_id', 'cdv_curated_allele_id' ],
            cdv_curation => [ 'cdv_curation_id', 'cdv_curation_id' ],
            div_allele => [ 'div_allele_id', 'div_allele_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::CdvCuratedAllele; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::CdvCuratedAllele->table('cdv_curated_allele');

our @extras = ();
Gramene::CDBI::DiversityMaize::CdvCuratedAllele->columns( 
    All => qw/ 
        cdv_curated_allele_id
        name 
    /, @extras
);

Gramene::CDBI::DiversityMaize::CdvCuratedAllele->columns( 
    Essential => qw/ 
        cdv_curated_allele_id
        name 
    /
);

Gramene::CDBI::DiversityMaize::CdvCuratedAllele->columns( 
    Ordered => qw/ 
        cdv_curated_allele_id
        name 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'cdv_curated_allele_id' => 'number',
        'name' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::CdvCuratedAllele->has_many( 
    'cdv_allele_curated_alleles',
    'Gramene::CDBI::DiversityMaize::CdvAlleleCuratedAllele',
    'cdv_curated_allele_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_allele_curated_allele => [ 'cdv_curated_allele_id', 'cdv_curated_allele_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::CdvCuration; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::CdvCuration->table('cdv_curation');


sub cdv_reason {
    my $self = shift;
    return $self->cdv_reason_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::CdvCuration->columns( 
    All => qw/ 
        cdv_curation_id
        cdv_reason_id
        curator
        curation_date
        comments 
    /, @extras
);

Gramene::CDBI::DiversityMaize::CdvCuration->columns( 
    Essential => qw/ 
        cdv_curation_id
        cdv_reason_id
        curator
        curation_date 
    /
);

Gramene::CDBI::DiversityMaize::CdvCuration->columns( 
    Ordered => qw/ 
        cdv_curation_id
        cdv_reason_id
        curator
        curation_date
        comments 
    /
);

sub to_string {
    my $self = shift;
    return $self->curator || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'cdv_curation_id' => 'number',
        'cdv_reason_id' => 'number',
        'curator' => 'string',
        'curation_date' => 'string',
        'comments' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::CdvCuration->has_a(
    cdv_reason_id => 'Gramene::CDBI::DiversityMaize::CdvReason'
);

Gramene::CDBI::DiversityMaize::CdvCuration->has_many( 
    'cdv_allele_curated_alleles',
    'Gramene::CDBI::DiversityMaize::CdvAlleleCuratedAllele',
    'cdv_curation_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_allele_curated_allele => [ 'cdv_curation_id', 'cdv_curation_id' ],
            cdv_reason => [ 'cdv_reason_id', 'cdv_reason_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::CdvMapFeatureAnnotation; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::CdvMapFeatureAnnotation->table('cdv_map_feature_annotation');


sub cdv_map_feature {
    my $self = shift;
    return $self->cdv_map_feature_id;
}
    

sub cdv_map_feature_annotation_type {
    my $self = shift;
    return $self->cdv_map_feature_annotation_type_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::CdvMapFeatureAnnotation->columns( 
    All => qw/ 
        cdv_map_feature_annotation_id
        cdv_map_feature_id
        cdv_map_feature_annotation_type_id
        annotation_value 
    /, @extras
);

Gramene::CDBI::DiversityMaize::CdvMapFeatureAnnotation->columns( 
    Essential => qw/ 
        cdv_map_feature_annotation_id
        cdv_map_feature_id
        cdv_map_feature_annotation_type_id
        annotation_value 
    /
);

Gramene::CDBI::DiversityMaize::CdvMapFeatureAnnotation->columns( 
    Ordered => qw/ 
        cdv_map_feature_annotation_id
        cdv_map_feature_id
        cdv_map_feature_annotation_type_id
        annotation_value 
    /
);

sub to_string {
    my $self = shift;
    return $self->annotation_value || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'cdv_map_feature_annotation_id' => 'number',
        'cdv_map_feature_id' => 'number',
        'cdv_map_feature_annotation_type_id' => 'number',
        'annotation_value' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::CdvMapFeatureAnnotation->has_a(
    cdv_map_feature_id => 'Gramene::CDBI::DiversityMaize::CdvMapFeature'
);

Gramene::CDBI::DiversityMaize::CdvMapFeatureAnnotation->has_a(
    cdv_map_feature_annotation_type_id => 'Gramene::CDBI::DiversityMaize::CdvMapFeatureAnnotationType'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_map_feature => [ 'cdv_map_feature_id', 'cdv_map_feature_id' ],
            cdv_map_feature_annotation_type => [ 'cdv_map_feature_annotation_type_id', 'cdv_map_feature_annotation_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::CdvMarkerAnnotation; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::CdvMarkerAnnotation->table('cdv_marker_annotation');


sub cdv_marker {
    my $self = shift;
    return $self->cdv_marker_id;
}
    

sub cdv_marker_annotation_type {
    my $self = shift;
    return $self->cdv_marker_annotation_type_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::CdvMarkerAnnotation->columns( 
    All => qw/ 
        cdv_marker_annotation_id
        cdv_marker_id
        cdv_marker_annotation_type_id
        annotation_value 
    /, @extras
);

Gramene::CDBI::DiversityMaize::CdvMarkerAnnotation->columns( 
    Essential => qw/ 
        cdv_marker_annotation_id
        cdv_marker_id
        cdv_marker_annotation_type_id
        annotation_value 
    /
);

Gramene::CDBI::DiversityMaize::CdvMarkerAnnotation->columns( 
    Ordered => qw/ 
        cdv_marker_annotation_id
        cdv_marker_id
        cdv_marker_annotation_type_id
        annotation_value 
    /
);

sub to_string {
    my $self = shift;
    return $self->annotation_value || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'cdv_marker_annotation_id' => 'number',
        'cdv_marker_id' => 'number',
        'cdv_marker_annotation_type_id' => 'number',
        'annotation_value' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::CdvMarkerAnnotation->has_a(
    cdv_marker_id => 'Gramene::CDBI::DiversityMaize::CdvMarker'
);

Gramene::CDBI::DiversityMaize::CdvMarkerAnnotation->has_a(
    cdv_marker_annotation_type_id => 'Gramene::CDBI::DiversityMaize::CdvMarkerAnnotationType'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_marker => [ 'cdv_marker_id', 'cdv_marker_id' ],
            cdv_marker_annotation_type => [ 'cdv_marker_annotation_type_id', 'cdv_marker_annotation_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivAaAnnotation; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivAaAnnotation->table('div_aa_annotation');


sub div_allele_assay {
    my $self = shift;
    return $self->div_allele_assay_id;
}
    

sub div_annotation_type {
    my $self = shift;
    return $self->div_annotation_type_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::DivAaAnnotation->columns( 
    All => qw/ 
        div_aa_annotation_id
        div_annotation_type_id
        div_allele_assay_id
        annotation_value 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivAaAnnotation->columns( 
    Essential => qw/ 
        div_aa_annotation_id
        div_annotation_type_id
        div_allele_assay_id
        annotation_value 
    /
);

Gramene::CDBI::DiversityMaize::DivAaAnnotation->columns( 
    Ordered => qw/ 
        div_aa_annotation_id
        div_annotation_type_id
        div_allele_assay_id
        annotation_value 
    /
);

sub to_string {
    my $self = shift;
    return $self->annotation_value || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_aa_annotation_id' => 'number',
        'div_annotation_type_id' => 'number',
        'div_allele_assay_id' => 'number',
        'annotation_value' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivAaAnnotation->has_a(
    div_allele_assay_id => 'Gramene::CDBI::DiversityMaize::DivAlleleAssay'
);

Gramene::CDBI::DiversityMaize::DivAaAnnotation->has_a(
    div_annotation_type_id => 'Gramene::CDBI::DiversityMaize::DivAnnotationType'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_allele_assay => [ 'div_allele_assay_id', 'div_allele_assay_id' ],
            div_annotation_type => [ 'div_annotation_type_id', 'div_annotation_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivAllele; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivAllele->table('div_allele');


sub div_allele_assay {
    my $self = shift;
    return $self->div_allele_assay_id;
}
    

sub div_obs_unit_sample {
    my $self = shift;
    return $self->div_obs_unit_sample_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::DivAllele->columns( 
    All => qw/ 
        div_allele_id
        div_obs_unit_sample_id
        div_allele_assay_id
        accession
        referencedb
        allele_num
        quality
        value
        proportion
        total_n 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivAllele->columns( 
    Essential => qw/ 
        div_allele_id
        div_obs_unit_sample_id
        div_allele_assay_id
        accession
        referencedb
        allele_num
        quality
        value
        proportion
        total_n 
    /
);

Gramene::CDBI::DiversityMaize::DivAllele->columns( 
    Ordered => qw/ 
        div_allele_id
        div_obs_unit_sample_id
        div_allele_assay_id
        accession
        referencedb
        allele_num
        quality
        value
        proportion
        total_n 
    /
);

sub to_string {
    my $self = shift;
    return $self->accession || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_allele_id' => 'number',
        'div_obs_unit_sample_id' => 'number',
        'div_allele_assay_id' => 'number',
        'accession' => 'string',
        'referencedb' => 'string',
        'allele_num' => 'number',
        'quality' => 'string',
        'value' => 'string',
        'proportion' => 'number',
        'total_n' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivAllele->has_a(
    div_allele_assay_id => 'Gramene::CDBI::DiversityMaize::DivAlleleAssay'
);

Gramene::CDBI::DiversityMaize::DivAllele->has_a(
    div_obs_unit_sample_id => 'Gramene::CDBI::DiversityMaize::DivObsUnitSample'
);

Gramene::CDBI::DiversityMaize::DivAllele->has_many( 
    'cdv_allele_curated_alleles',
    'Gramene::CDBI::DiversityMaize::CdvAlleleCuratedAllele',
    'div_allele_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_allele_curated_allele => [ 'div_allele_id', 'div_allele_id' ],
            div_allele_assay => [ 'div_allele_assay_id', 'div_allele_assay_id' ],
            div_obs_unit_sample => [ 'div_obs_unit_sample_id', 'div_obs_unit_sample_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivAlleleAssay; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivAlleleAssay->table('div_allele_assay');


sub cdv_marker {
    my $self = shift;
    return $self->cdv_marker_id;
}
    

sub div_source_assay {
    my $self = shift;
    return $self->div_source_assay_id;
}
    

sub div_poly_type {
    my $self = shift;
    return $self->div_poly_type_id;
}
    

sub div_scoring_tech_type {
    my $self = shift;
    return $self->div_scoring_tech_type_id;
}
    

sub div_ref_stock {
    my $self = shift;
    return $self->div_ref_stock_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::DivAlleleAssay->columns( 
    All => qw/ 
        div_allele_assay_id
        div_source_assay_id
        div_poly_type_id
        div_scoring_tech_type_id
        div_ref_stock_id
        cdv_marker_id
        name
        assay_date
        producer
        position
        ref_seq
        length
        comments 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivAlleleAssay->columns( 
    Essential => qw/ 
        div_allele_assay_id
        div_source_assay_id
        div_poly_type_id
        div_scoring_tech_type_id
        div_ref_stock_id
        cdv_marker_id
        name
        assay_date
        producer
        position
        length 
    /
);

Gramene::CDBI::DiversityMaize::DivAlleleAssay->columns( 
    Ordered => qw/ 
        div_allele_assay_id
        div_source_assay_id
        div_poly_type_id
        div_scoring_tech_type_id
        div_ref_stock_id
        cdv_marker_id
        name
        assay_date
        producer
        position
        ref_seq
        length
        comments 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_allele_assay_id' => 'number',
        'div_source_assay_id' => 'number',
        'div_poly_type_id' => 'number',
        'div_scoring_tech_type_id' => 'number',
        'div_ref_stock_id' => 'number',
        'cdv_marker_id' => 'number',
        'name' => 'string',
        'assay_date' => 'string',
        'producer' => 'string',
        'position' => 'string',
        'ref_seq' => 'string',
        'length' => 'number',
        'comments' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivAlleleAssay->has_a(
    cdv_marker_id => 'Gramene::CDBI::DiversityMaize::CdvMarker'
);

Gramene::CDBI::DiversityMaize::DivAlleleAssay->has_a(
    div_source_assay_id => 'Gramene::CDBI::DiversityMaize::DivAlleleAssay'
);

Gramene::CDBI::DiversityMaize::DivAlleleAssay->has_a(
    div_poly_type_id => 'Gramene::CDBI::DiversityMaize::DivPolyType'
);

Gramene::CDBI::DiversityMaize::DivAlleleAssay->has_a(
    div_scoring_tech_type_id => 'Gramene::CDBI::DiversityMaize::DivScoringTechType'
);

Gramene::CDBI::DiversityMaize::DivAlleleAssay->has_a(
    div_ref_stock_id => 'Gramene::CDBI::DiversityMaize::DivStock'
);

Gramene::CDBI::DiversityMaize::DivAlleleAssay->has_many( 
    'div_aa_annotations',
    'Gramene::CDBI::DiversityMaize::DivAaAnnotation',
    'div_allele_assay_id' 
);

Gramene::CDBI::DiversityMaize::DivAlleleAssay->has_many( 
    'div_alleles',
    'Gramene::CDBI::DiversityMaize::DivAllele',
    'div_allele_assay_id' 
);

Gramene::CDBI::DiversityMaize::DivAlleleAssay->has_many( 
    'div_allele_assays',
    'Gramene::CDBI::DiversityMaize::DivAlleleAssay',
    'div_source_assay_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_marker => [ 'cdv_marker_id', 'cdv_marker_id' ],
            div_aa_annotation => [ 'div_allele_assay_id', 'div_allele_assay_id' ],
            div_allele => [ 'div_allele_assay_id', 'div_allele_assay_id' ],
            div_allele_assay => [ 'div_source_assay_id', 'div_allele_assay_id' ],
            div_poly_type => [ 'div_poly_type_id', 'div_poly_type_id' ],
            div_scoring_tech_type => [ 'div_scoring_tech_type_id', 'div_scoring_tech_type_id' ],
            div_stock => [ 'div_ref_stock_id', 'div_stock_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::QtlMarker; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::QtlMarker->table('qtl_marker');


sub cdv_marker {
    my $self = shift;
    return $self->cdv_marker_id;
}
    

sub qtl_qtl {
    my $self = shift;
    return $self->qtl_qtl_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::QtlMarker->columns( 
    All => qw/ 
        qtl_marker_id
        qtl_qtl_id
        cdv_marker_id
        genetic_relationship
        comments 
    /, @extras
);

Gramene::CDBI::DiversityMaize::QtlMarker->columns( 
    Essential => qw/ 
        qtl_marker_id
        qtl_qtl_id
        cdv_marker_id
        genetic_relationship 
    /
);

Gramene::CDBI::DiversityMaize::QtlMarker->columns( 
    Ordered => qw/ 
        qtl_marker_id
        qtl_qtl_id
        cdv_marker_id
        genetic_relationship
        comments 
    /
);

sub to_string {
    my $self = shift;
    return $self->genetic_relationship || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'qtl_marker_id' => 'number',
        'qtl_qtl_id' => 'number',
        'cdv_marker_id' => 'number',
        'genetic_relationship' => 'string',
        'comments' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::QtlMarker->has_a(
    cdv_marker_id => 'Gramene::CDBI::DiversityMaize::CdvMarker'
);

Gramene::CDBI::DiversityMaize::QtlMarker->has_a(
    qtl_qtl_id => 'Gramene::CDBI::DiversityMaize::QtlQtl'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_marker => [ 'cdv_marker_id', 'cdv_marker_id' ],
            qtl_qtl => [ 'qtl_qtl_id', 'qtl_qtl_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::CdvMarker; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::CdvMarker->table('cdv_marker');


sub cdv_map_feature {
    my $self = shift;
    return $self->cdv_map_feature_id;
}
    

sub div_ref_stock {
    my $self = shift;
    return $self->div_ref_stock_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::CdvMarker->columns( 
    All => qw/ 
        cdv_marker_id
        cdv_map_feature_id
        div_ref_stock_id
        name
        position
        length
        ref_seq
        marker_aid 
    /, @extras
);

Gramene::CDBI::DiversityMaize::CdvMarker->columns( 
    Essential => qw/ 
        cdv_marker_id
        cdv_map_feature_id
        div_ref_stock_id
        name
        position
        length
        marker_aid 
    /
);

Gramene::CDBI::DiversityMaize::CdvMarker->columns( 
    Ordered => qw/ 
        cdv_marker_id
        cdv_map_feature_id
        div_ref_stock_id
        name
        position
        length
        ref_seq
        marker_aid 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'cdv_marker_id' => 'number',
        'cdv_map_feature_id' => 'number',
        'div_ref_stock_id' => 'number',
        'name' => 'string',
        'position' => 'number',
        'length' => 'number',
        'ref_seq' => 'string',
        'marker_aid' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::CdvMarker->has_a(
    cdv_map_feature_id => 'Gramene::CDBI::DiversityMaize::CdvMapFeature'
);

Gramene::CDBI::DiversityMaize::CdvMarker->has_a(
    div_ref_stock_id => 'Gramene::CDBI::DiversityMaize::DivStock'
);

Gramene::CDBI::DiversityMaize::CdvMarker->has_many( 
    'cdv_marker_annotations',
    'Gramene::CDBI::DiversityMaize::CdvMarkerAnnotation',
    'cdv_marker_id' 
);

Gramene::CDBI::DiversityMaize::CdvMarker->has_many( 
    'div_allele_assays',
    'Gramene::CDBI::DiversityMaize::DivAlleleAssay',
    'cdv_marker_id' 
);

Gramene::CDBI::DiversityMaize::CdvMarker->has_many( 
    'qtl_markers',
    'Gramene::CDBI::DiversityMaize::QtlMarker',
    'cdv_marker_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_map_feature => [ 'cdv_map_feature_id', 'cdv_map_feature_id' ],
            cdv_marker_annotation => [ 'cdv_marker_id', 'cdv_marker_id' ],
            div_allele_assay => [ 'cdv_marker_id', 'cdv_marker_id' ],
            div_stock => [ 'div_ref_stock_id', 'div_stock_id' ],
            qtl_marker => [ 'cdv_marker_id', 'cdv_marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::CdvMapFeature; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::CdvMapFeature->table('cdv_map_feature');

our @extras = ();
Gramene::CDBI::DiversityMaize::CdvMapFeature->columns( 
    All => qw/ 
        cdv_map_feature_id
        name
        chromosome_name
        genetic_bin
        xref_map_id
        genetic_map
        genetic_position
        locus_type
        physical_position
        comments 
    /, @extras
);

Gramene::CDBI::DiversityMaize::CdvMapFeature->columns( 
    Essential => qw/ 
        cdv_map_feature_id
        name
        chromosome_name
        genetic_bin
        xref_map_id
        genetic_map
        genetic_position
        locus_type
        physical_position 
    /
);

Gramene::CDBI::DiversityMaize::CdvMapFeature->columns( 
    Ordered => qw/ 
        cdv_map_feature_id
        name
        chromosome_name
        genetic_bin
        xref_map_id
        genetic_map
        genetic_position
        locus_type
        physical_position
        comments 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'cdv_map_feature_id' => 'number',
        'name' => 'string',
        'chromosome_name' => 'string',
        'genetic_bin' => 'string',
        'xref_map_id' => 'string',
        'genetic_map' => 'string',
        'genetic_position' => 'number',
        'locus_type' => 'string',
        'physical_position' => 'number',
        'comments' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::CdvMapFeature->has_many( 
    'cdv_map_feature_annotations',
    'Gramene::CDBI::DiversityMaize::CdvMapFeatureAnnotation',
    'cdv_map_feature_id' 
);

Gramene::CDBI::DiversityMaize::CdvMapFeature->has_many( 
    'cdv_markers',
    'Gramene::CDBI::DiversityMaize::CdvMarker',
    'cdv_map_feature_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_map_feature_annotation => [ 'cdv_map_feature_id', 'cdv_map_feature_id' ],
            cdv_marker => [ 'cdv_map_feature_id', 'cdv_map_feature_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::CdvMapFeatureAnnotationType; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::CdvMapFeatureAnnotationType->table('cdv_map_feature_annotation_type');

our @extras = ();
Gramene::CDBI::DiversityMaize::CdvMapFeatureAnnotationType->columns( 
    All => qw/ 
        cdv_map_feature_annotation_type_id
        anno_type 
    /, @extras
);

Gramene::CDBI::DiversityMaize::CdvMapFeatureAnnotationType->columns( 
    Essential => qw/ 
        cdv_map_feature_annotation_type_id
        anno_type 
    /
);

Gramene::CDBI::DiversityMaize::CdvMapFeatureAnnotationType->columns( 
    Ordered => qw/ 
        cdv_map_feature_annotation_type_id
        anno_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->anno_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'cdv_map_feature_annotation_type_id' => 'number',
        'anno_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::CdvMapFeatureAnnotationType->has_many( 
    'cdv_map_feature_annotations',
    'Gramene::CDBI::DiversityMaize::CdvMapFeatureAnnotation',
    'cdv_map_feature_annotation_type_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_map_feature_annotation => [ 'cdv_map_feature_annotation_type_id', 'cdv_map_feature_annotation_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::CdvMarkerAnnotationType; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::CdvMarkerAnnotationType->table('cdv_marker_annotation_type');

our @extras = ();
Gramene::CDBI::DiversityMaize::CdvMarkerAnnotationType->columns( 
    All => qw/ 
        cdv_marker_annotation_type_id
        anno_type 
    /, @extras
);

Gramene::CDBI::DiversityMaize::CdvMarkerAnnotationType->columns( 
    Essential => qw/ 
        cdv_marker_annotation_type_id
        anno_type 
    /
);

Gramene::CDBI::DiversityMaize::CdvMarkerAnnotationType->columns( 
    Ordered => qw/ 
        cdv_marker_annotation_type_id
        anno_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->anno_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'cdv_marker_annotation_type_id' => 'number',
        'anno_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::CdvMarkerAnnotationType->has_many( 
    'cdv_marker_annotations',
    'Gramene::CDBI::DiversityMaize::CdvMarkerAnnotation',
    'cdv_marker_annotation_type_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_marker_annotation => [ 'cdv_marker_annotation_type_id', 'cdv_marker_annotation_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::CdvPassportSet; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::CdvPassportSet->table('cdv_passport_set');


sub cdv_passport_group {
    my $self = shift;
    return $self->cdv_passport_group_id;
}
    

sub div_passport {
    my $self = shift;
    return $self->div_passport_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::CdvPassportSet->columns( 
    All => qw/ 
        cdv_passport_set_id
        div_passport_id
        cdv_passport_group_id 
    /, @extras
);

Gramene::CDBI::DiversityMaize::CdvPassportSet->columns( 
    Essential => qw/ 
        cdv_passport_set_id
        div_passport_id
        cdv_passport_group_id 
    /
);

Gramene::CDBI::DiversityMaize::CdvPassportSet->columns( 
    Ordered => qw/ 
        cdv_passport_set_id
        div_passport_id
        cdv_passport_group_id 
    /
);

sub to_string {
    my $self = shift;
    return $self->cdv_passport_set_id || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'cdv_passport_set_id' => 'number',
        'div_passport_id' => 'number',
        'cdv_passport_group_id' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::CdvPassportSet->has_a(
    cdv_passport_group_id => 'Gramene::CDBI::DiversityMaize::CdvPassportGroup'
);

Gramene::CDBI::DiversityMaize::CdvPassportSet->has_a(
    div_passport_id => 'Gramene::CDBI::DiversityMaize::DivPassport'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_passport_group => [ 'cdv_passport_group_id', 'cdv_passport_group_id' ],
            div_passport => [ 'div_passport_id', 'div_passport_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::CdvPassportGroup; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::CdvPassportGroup->table('cdv_passport_group');

our @extras = ();
Gramene::CDBI::DiversityMaize::CdvPassportGroup->columns( 
    All => qw/ 
        cdv_passport_group_id
        group_name 
    /, @extras
);

Gramene::CDBI::DiversityMaize::CdvPassportGroup->columns( 
    Essential => qw/ 
        cdv_passport_group_id
        group_name 
    /
);

Gramene::CDBI::DiversityMaize::CdvPassportGroup->columns( 
    Ordered => qw/ 
        cdv_passport_group_id
        group_name 
    /
);

sub to_string {
    my $self = shift;
    return $self->group_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'cdv_passport_group_id' => 'number',
        'group_name' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::CdvPassportGroup->has_many( 
    'cdv_passport_sets',
    'Gramene::CDBI::DiversityMaize::CdvPassportSet',
    'cdv_passport_group_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_passport_set => [ 'cdv_passport_group_id', 'cdv_passport_group_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::CdvReason; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::CdvReason->table('cdv_reason');

our @extras = ();
Gramene::CDBI::DiversityMaize::CdvReason->columns( 
    All => qw/ 
        cdv_reason_id
        description 
    /, @extras
);

Gramene::CDBI::DiversityMaize::CdvReason->columns( 
    Essential => qw/ 
        cdv_reason_id
        description 
    /
);

Gramene::CDBI::DiversityMaize::CdvReason->columns( 
    Ordered => qw/ 
        cdv_reason_id
        description 
    /
);

sub to_string {
    my $self = shift;
    return $self->description || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'cdv_reason_id' => 'number',
        'description' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::CdvReason->has_many( 
    'cdv_curations',
    'Gramene::CDBI::DiversityMaize::CdvCuration',
    'cdv_reason_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_curation => [ 'cdv_reason_id', 'cdv_reason_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivObsUnitSample; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivObsUnitSample->table('div_obs_unit_sample');


sub div_obs_unit {
    my $self = shift;
    return $self->div_obs_unit_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::DivObsUnitSample->columns( 
    All => qw/ 
        div_obs_unit_sample_id
        div_obs_unit_id
        name
        sample_date
        producer
        comments 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivObsUnitSample->columns( 
    Essential => qw/ 
        div_obs_unit_sample_id
        div_obs_unit_id
        name
        sample_date
        producer 
    /
);

Gramene::CDBI::DiversityMaize::DivObsUnitSample->columns( 
    Ordered => qw/ 
        div_obs_unit_sample_id
        div_obs_unit_id
        name
        sample_date
        producer
        comments 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_obs_unit_sample_id' => 'number',
        'div_obs_unit_id' => 'number',
        'name' => 'string',
        'sample_date' => 'string',
        'producer' => 'string',
        'comments' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivObsUnitSample->has_a(
    div_obs_unit_id => 'Gramene::CDBI::DiversityMaize::DivObsUnit'
);

Gramene::CDBI::DiversityMaize::DivObsUnitSample->has_many( 
    'div_alleles',
    'Gramene::CDBI::DiversityMaize::DivAllele',
    'div_obs_unit_sample_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_allele => [ 'div_obs_unit_sample_id', 'div_obs_unit_sample_id' ],
            div_obs_unit => [ 'div_obs_unit_id', 'div_obs_unit_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivTrait; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivTrait->table('div_trait');


sub div_obs_unit {
    my $self = shift;
    return $self->div_obs_unit_id;
}
    

sub div_statistic_type {
    my $self = shift;
    return $self->div_statistic_type_id;
}
    

sub div_trait_uom {
    my $self = shift;
    return $self->div_trait_uom_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::DivTrait->columns( 
    All => qw/ 
        div_trait_id
        div_trait_uom_id
        div_statistic_type_id
        div_obs_unit_id
        value
        date_measured 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivTrait->columns( 
    Essential => qw/ 
        div_trait_id
        div_trait_uom_id
        div_statistic_type_id
        div_obs_unit_id
        value
        date_measured 
    /
);

Gramene::CDBI::DiversityMaize::DivTrait->columns( 
    Ordered => qw/ 
        div_trait_id
        div_trait_uom_id
        div_statistic_type_id
        div_obs_unit_id
        value
        date_measured 
    /
);

sub to_string {
    my $self = shift;
    return $self->value || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_trait_id' => 'number',
        'div_trait_uom_id' => 'number',
        'div_statistic_type_id' => 'number',
        'div_obs_unit_id' => 'number',
        'value' => 'string',
        'date_measured' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivTrait->has_a(
    div_obs_unit_id => 'Gramene::CDBI::DiversityMaize::DivObsUnit'
);

Gramene::CDBI::DiversityMaize::DivTrait->has_a(
    div_statistic_type_id => 'Gramene::CDBI::DiversityMaize::DivStatisticType'
);

Gramene::CDBI::DiversityMaize::DivTrait->has_a(
    div_trait_uom_id => 'Gramene::CDBI::DiversityMaize::DivTraitUom'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_obs_unit => [ 'div_obs_unit_id', 'div_obs_unit_id' ],
            div_statistic_type => [ 'div_statistic_type_id', 'div_statistic_type_id' ],
            div_trait_uom => [ 'div_trait_uom_id', 'div_trait_uom_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivTreatment; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivTreatment->table('div_treatment');


sub div_obs_unit {
    my $self = shift;
    return $self->div_obs_unit_id;
}
    

sub div_treatment_uom {
    my $self = shift;
    return $self->div_treatment_uom_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::DivTreatment->columns( 
    All => qw/ 
        div_treatment_id
        div_treatment_uom_id
        div_obs_unit_id
        value 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivTreatment->columns( 
    Essential => qw/ 
        div_treatment_id
        div_treatment_uom_id
        div_obs_unit_id
        value 
    /
);

Gramene::CDBI::DiversityMaize::DivTreatment->columns( 
    Ordered => qw/ 
        div_treatment_id
        div_treatment_uom_id
        div_obs_unit_id
        value 
    /
);

sub to_string {
    my $self = shift;
    return $self->value || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_treatment_id' => 'number',
        'div_treatment_uom_id' => 'number',
        'div_obs_unit_id' => 'number',
        'value' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivTreatment->has_a(
    div_obs_unit_id => 'Gramene::CDBI::DiversityMaize::DivObsUnit'
);

Gramene::CDBI::DiversityMaize::DivTreatment->has_a(
    div_treatment_uom_id => 'Gramene::CDBI::DiversityMaize::DivTreatmentUom'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_obs_unit => [ 'div_obs_unit_id', 'div_obs_unit_id' ],
            div_treatment_uom => [ 'div_treatment_uom_id', 'div_treatment_uom_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivObsUnit; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivObsUnit->table('div_obs_unit');


sub div_experiment {
    my $self = shift;
    return $self->div_experiment_id;
}
    

sub div_locality {
    my $self = shift;
    return $self->div_locality_id;
}
    

sub div_stock {
    my $self = shift;
    return $self->div_stock_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::DivObsUnit->columns( 
    All => qw/ 
        div_obs_unit_id
        div_experiment_id
        div_stock_id
        div_locality_id
        name
        coord_x
        coord_y
        rep
        block
        plot
        season
        plant
        planting_date
        harvest_date
        comments 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivObsUnit->columns( 
    Essential => qw/ 
        div_obs_unit_id
        div_experiment_id
        div_stock_id
        div_locality_id
        name
        coord_x
        coord_y
        rep
        block
        plot
        season
        plant
        planting_date
        harvest_date 
    /
);

Gramene::CDBI::DiversityMaize::DivObsUnit->columns( 
    Ordered => qw/ 
        div_obs_unit_id
        div_experiment_id
        div_stock_id
        div_locality_id
        name
        coord_x
        coord_y
        rep
        block
        plot
        season
        plant
        planting_date
        harvest_date
        comments 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_obs_unit_id' => 'number',
        'div_experiment_id' => 'number',
        'div_stock_id' => 'number',
        'div_locality_id' => 'number',
        'name' => 'string',
        'coord_x' => 'number',
        'coord_y' => 'number',
        'rep' => 'string',
        'block' => 'string',
        'plot' => 'string',
        'season' => 'string',
        'plant' => 'string',
        'planting_date' => 'string',
        'harvest_date' => 'string',
        'comments' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivObsUnit->has_a(
    div_experiment_id => 'Gramene::CDBI::DiversityMaize::DivExperiment'
);

Gramene::CDBI::DiversityMaize::DivObsUnit->has_a(
    div_locality_id => 'Gramene::CDBI::DiversityMaize::DivLocality'
);

Gramene::CDBI::DiversityMaize::DivObsUnit->has_a(
    div_stock_id => 'Gramene::CDBI::DiversityMaize::DivStock'
);

Gramene::CDBI::DiversityMaize::DivObsUnit->has_many( 
    'div_obs_unit_samples',
    'Gramene::CDBI::DiversityMaize::DivObsUnitSample',
    'div_obs_unit_id' 
);

Gramene::CDBI::DiversityMaize::DivObsUnit->has_many( 
    'div_traits',
    'Gramene::CDBI::DiversityMaize::DivTrait',
    'div_obs_unit_id' 
);

Gramene::CDBI::DiversityMaize::DivObsUnit->has_many( 
    'div_treatments',
    'Gramene::CDBI::DiversityMaize::DivTreatment',
    'div_obs_unit_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_experiment => [ 'div_experiment_id', 'div_experiment_id' ],
            div_locality => [ 'div_locality_id', 'div_locality_id' ],
            div_obs_unit_sample => [ 'div_obs_unit_id', 'div_obs_unit_id' ],
            div_stock => [ 'div_stock_id', 'div_stock_id' ],
            div_trait => [ 'div_obs_unit_id', 'div_obs_unit_id' ],
            div_treatment => [ 'div_obs_unit_id', 'div_obs_unit_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::QtlQtl; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::QtlQtl->table('qtl_qtl');


sub qtl_analysis_result {
    my $self = shift;
    return $self->qtl_analysis_result_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::QtlQtl->columns( 
    All => qw/ 
        qtl_qtl_id
        qtl_analysis_result_id
        qtl_accession_id
        published_symbol
        linkage_group
        start_position
        peak_position
        stop_position
        additive
        dominance
        lod
        f_value
        t_value
        p_value
        r_square
        comments 
    /, @extras
);

Gramene::CDBI::DiversityMaize::QtlQtl->columns( 
    Essential => qw/ 
        qtl_qtl_id
        qtl_analysis_result_id
        qtl_accession_id
        published_symbol
        linkage_group
        start_position
        peak_position
        stop_position
        additive
        dominance
        lod
        f_value
        t_value
        p_value
        r_square 
    /
);

Gramene::CDBI::DiversityMaize::QtlQtl->columns( 
    Ordered => qw/ 
        qtl_qtl_id
        qtl_analysis_result_id
        qtl_accession_id
        published_symbol
        linkage_group
        start_position
        peak_position
        stop_position
        additive
        dominance
        lod
        f_value
        t_value
        p_value
        r_square
        comments 
    /
);

sub to_string {
    my $self = shift;
    return $self->published_symbol || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'qtl_qtl_id' => 'number',
        'qtl_analysis_result_id' => 'number',
        'qtl_accession_id' => 'string',
        'published_symbol' => 'string',
        'linkage_group' => 'string',
        'start_position' => 'number',
        'peak_position' => 'number',
        'stop_position' => 'number',
        'additive' => 'number',
        'dominance' => 'number',
        'lod' => 'number',
        'f_value' => 'number',
        't_value' => 'number',
        'p_value' => 'number',
        'r_square' => 'number',
        'comments' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::QtlQtl->has_a(
    qtl_analysis_result_id => 'Gramene::CDBI::DiversityMaize::QtlAnalysisResult'
);

Gramene::CDBI::DiversityMaize::QtlQtl->has_many( 
    'qtl_markers',
    'Gramene::CDBI::DiversityMaize::QtlMarker',
    'qtl_qtl_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            qtl_analysis_result => [ 'qtl_analysis_result_id', 'qtl_analysis_result_id' ],
            qtl_marker => [ 'qtl_qtl_id', 'qtl_qtl_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::QtlAnalysisResult; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::QtlAnalysisResult->table('qtl_analysis_result');


sub div_trait_uom {
    my $self = shift;
    return $self->div_trait_uom_id;
}
    

sub qtl_analysis_method {
    my $self = shift;
    return $self->qtl_analysis_method_id;
}
    

sub qtl_obs_unit {
    my $self = shift;
    return $self->qtl_obs_unit_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::QtlAnalysisResult->columns( 
    All => qw/ 
        qtl_analysis_result_id
        qtl_obs_unit_id
        qtl_analysis_method_id
        div_trait_uom_id
        r_square
        genetic_var
        gxe_var
        environment_var 
    /, @extras
);

Gramene::CDBI::DiversityMaize::QtlAnalysisResult->columns( 
    Essential => qw/ 
        qtl_analysis_result_id
        qtl_obs_unit_id
        qtl_analysis_method_id
        div_trait_uom_id
        r_square
        genetic_var
        gxe_var
        environment_var 
    /
);

Gramene::CDBI::DiversityMaize::QtlAnalysisResult->columns( 
    Ordered => qw/ 
        qtl_analysis_result_id
        qtl_obs_unit_id
        qtl_analysis_method_id
        div_trait_uom_id
        r_square
        genetic_var
        gxe_var
        environment_var 
    /
);

sub to_string {
    my $self = shift;
    return $self->r_square || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'qtl_analysis_result_id' => 'number',
        'qtl_obs_unit_id' => 'number',
        'qtl_analysis_method_id' => 'number',
        'div_trait_uom_id' => 'number',
        'r_square' => 'number',
        'genetic_var' => 'number',
        'gxe_var' => 'number',
        'environment_var' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::QtlAnalysisResult->has_a(
    div_trait_uom_id => 'Gramene::CDBI::DiversityMaize::DivTraitUom'
);

Gramene::CDBI::DiversityMaize::QtlAnalysisResult->has_a(
    qtl_analysis_method_id => 'Gramene::CDBI::DiversityMaize::QtlAnalysisMethod'
);

Gramene::CDBI::DiversityMaize::QtlAnalysisResult->has_a(
    qtl_obs_unit_id => 'Gramene::CDBI::DiversityMaize::QtlObsUnit'
);

Gramene::CDBI::DiversityMaize::QtlAnalysisResult->has_many( 
    'qtl_qtls',
    'Gramene::CDBI::DiversityMaize::QtlQtl',
    'qtl_analysis_result_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_trait_uom => [ 'div_trait_uom_id', 'div_trait_uom_id' ],
            qtl_analysis_method => [ 'qtl_analysis_method_id', 'qtl_analysis_method_id' ],
            qtl_obs_unit => [ 'qtl_obs_unit_id', 'qtl_obs_unit_id' ],
            qtl_qtl => [ 'qtl_analysis_result_id', 'qtl_analysis_result_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::QtlAnalysisMethod; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::QtlAnalysisMethod->table('qtl_analysis_method');


sub positive_parent_stock {
    my $self = shift;
    return $self->positive_parent_stock_id;
}
    

sub qtl_study {
    my $self = shift;
    return $self->qtl_study_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::QtlAnalysisMethod->columns( 
    All => qw/ 
        qtl_analysis_method_id
        qtl_study_id
        threshold_method
        permutation_number
        statistical_model
        positive_parent_stock_id
        comments 
    /, @extras
);

Gramene::CDBI::DiversityMaize::QtlAnalysisMethod->columns( 
    Essential => qw/ 
        qtl_analysis_method_id
        qtl_study_id
        threshold_method
        permutation_number
        statistical_model
        positive_parent_stock_id 
    /
);

Gramene::CDBI::DiversityMaize::QtlAnalysisMethod->columns( 
    Ordered => qw/ 
        qtl_analysis_method_id
        qtl_study_id
        threshold_method
        permutation_number
        statistical_model
        positive_parent_stock_id
        comments 
    /
);

sub to_string {
    my $self = shift;
    return $self->threshold_method || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'qtl_analysis_method_id' => 'number',
        'qtl_study_id' => 'number',
        'threshold_method' => 'string',
        'permutation_number' => 'number',
        'statistical_model' => 'string',
        'positive_parent_stock_id' => 'number',
        'comments' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::QtlAnalysisMethod->has_a(
    positive_parent_stock_id => 'Gramene::CDBI::DiversityMaize::DivStockParent'
);

Gramene::CDBI::DiversityMaize::QtlAnalysisMethod->has_a(
    qtl_study_id => 'Gramene::CDBI::DiversityMaize::QtlStudy'
);

Gramene::CDBI::DiversityMaize::QtlAnalysisMethod->has_many( 
    'qtl_analysis_results',
    'Gramene::CDBI::DiversityMaize::QtlAnalysisResult',
    'qtl_analysis_method_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_stock_parent => [ 'positive_parent_stock_id', 'div_stock_parent_id' ],
            qtl_analysis_result => [ 'qtl_analysis_method_id', 'qtl_analysis_method_id' ],
            qtl_study => [ 'qtl_study_id', 'qtl_study_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivStockParent; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivStockParent->table('div_stock_parent');


sub div_stock {
    my $self = shift;
    return $self->div_stock_id;
}
    
sub div_parent {
    my $self = shift;
    return $self->div_parent_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::DivStockParent->columns( 
    All => qw/ 
        div_stock_parent_id
        div_stock_id
        div_parent_id
        role
        recurrent 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivStockParent->columns( 
    Essential => qw/ 
        div_stock_parent_id
        div_stock_id
        div_parent_id
        role
        recurrent 
    /
);

Gramene::CDBI::DiversityMaize::DivStockParent->columns( 
    Ordered => qw/ 
        div_stock_parent_id
        div_stock_id
        div_parent_id
        role
        recurrent 
    /
);

sub to_string {
    my $self = shift;
    return $self->role || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_stock_parent_id' => 'number',
        'div_stock_id' => 'number',
        'div_parent_id' => 'number',
        'role' => 'string',
        'recurrent' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivStockParent->has_a(
    div_stock_id => 'Gramene::CDBI::DiversityMaize::DivStock'
);

Gramene::CDBI::DiversityMaize::DivStockParent->has_a(
    div_parent_id => 'Gramene::CDBI::DiversityMaize::DivStock'
);

Gramene::CDBI::DiversityMaize::DivStockParent->has_many( 
    'qtl_analysis_methods',
    'Gramene::CDBI::DiversityMaize::QtlAnalysisMethod',
    'positive_parent_stock_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_stock => [ 'div_parent_id', 'div_stock_id' ],
            qtl_analysis_method => [ 'div_stock_parent_id', 'positive_parent_stock_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivStock; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivStock->table('div_stock');


sub div_generation {
    my $self = shift;
    return $self->div_generation_id;
}
    

sub div_passport {
    my $self = shift;
    return $self->div_passport_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::DivStock->columns( 
    All => qw/ 
        div_stock_id
        div_generation_id
        div_passport_id
        seed_lot
        stock_source
        comments 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivStock->columns( 
    Essential => qw/ 
        div_stock_id
        div_generation_id
        div_passport_id
        seed_lot
        stock_source 
    /
);

Gramene::CDBI::DiversityMaize::DivStock->columns( 
    Ordered => qw/ 
        div_stock_id
        div_generation_id
        div_passport_id
        seed_lot
        stock_source
        comments 
    /
);

sub to_string {
    my $self = shift;
    return $self->seed_lot || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_stock_id' => 'number',
        'div_generation_id' => 'number',
        'div_passport_id' => 'number',
        'seed_lot' => 'string',
        'stock_source' => 'string',
        'comments' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivStock->has_a(
    div_generation_id => 'Gramene::CDBI::DiversityMaize::DivGeneration'
);

Gramene::CDBI::DiversityMaize::DivStock->has_a(
    div_passport_id => 'Gramene::CDBI::DiversityMaize::DivPassport'
);

Gramene::CDBI::DiversityMaize::DivStock->has_many( 
    'cdv_markers',
    'Gramene::CDBI::DiversityMaize::CdvMarker',
    'div_ref_stock_id' 
);

Gramene::CDBI::DiversityMaize::DivStock->has_many( 
    'div_allele_assays',
    'Gramene::CDBI::DiversityMaize::DivAlleleAssay',
    'div_ref_stock_id' 
);

Gramene::CDBI::DiversityMaize::DivStock->has_many( 
    'div_obs_units',
    'Gramene::CDBI::DiversityMaize::DivObsUnit',
    'div_stock_id' 
);

Gramene::CDBI::DiversityMaize::DivStock->has_many( 
    'div_stock_parents_div_stock',
    'Gramene::CDBI::DiversityMaize::DivStockParent',
    'div_stock_id' 
);

Gramene::CDBI::DiversityMaize::DivStock->has_many( 
    'div_stock_parents_div_parent',
    'Gramene::CDBI::DiversityMaize::DivStockParent',
    'div_parent_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_marker => [ 'div_stock_id', 'div_ref_stock_id' ],
            div_allele_assay => [ 'div_stock_id', 'div_ref_stock_id' ],
            div_generation => [ 'div_generation_id', 'div_generation_id' ],
            div_obs_unit => [ 'div_stock_id', 'div_stock_id' ],
            div_passport => [ 'div_passport_id', 'div_passport_id' ],
            div_stock_parent => [ 'div_stock_id', 'div_parent_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivSynonym; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivSynonym->table('div_synonym');


sub div_passport {
    my $self = shift;
    return $self->div_passport_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::DivSynonym->columns( 
    All => qw/ 
        div_synonym_id
        div_passport_id
        synonym
        comments 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivSynonym->columns( 
    Essential => qw/ 
        div_synonym_id
        div_passport_id
        synonym 
    /
);

Gramene::CDBI::DiversityMaize::DivSynonym->columns( 
    Ordered => qw/ 
        div_synonym_id
        div_passport_id
        synonym
        comments 
    /
);

sub to_string {
    my $self = shift;
    return $self->synonym || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_synonym_id' => 'number',
        'div_passport_id' => 'number',
        'synonym' => 'string',
        'comments' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivSynonym->has_a(
    div_passport_id => 'Gramene::CDBI::DiversityMaize::DivPassport'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_passport => [ 'div_passport_id', 'div_passport_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivPassport; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivPassport->table('div_passport');


sub cdv_source {
    my $self = shift;
    return $self->cdv_source_id;
}
    

sub div_accession_collecting {
    my $self = shift;
    return $self->div_accession_collecting_id;
}
    

sub div_taxonomy {
    my $self = shift;
    return $self->div_taxonomy_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::DivPassport->columns( 
    All => qw/ 
        div_passport_id
        div_taxonomy_id
        div_accession_collecting_id
        cdv_source_id
        accename
        germplasm_type
        source
        accenumb
        sampstat
        comments 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivPassport->columns( 
    Essential => qw/ 
        div_passport_id
        div_taxonomy_id
        div_accession_collecting_id
        cdv_source_id
        accename
        germplasm_type
        source
        accenumb
        sampstat 
    /
);

Gramene::CDBI::DiversityMaize::DivPassport->columns( 
    Ordered => qw/ 
        div_passport_id
        div_taxonomy_id
        div_accession_collecting_id
        cdv_source_id
        accename
        germplasm_type
        source
        accenumb
        sampstat
        comments 
    /
);

sub to_string {
    my $self = shift;
    return $self->accename || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_passport_id' => 'number',
        'div_taxonomy_id' => 'number',
        'div_accession_collecting_id' => 'number',
        'cdv_source_id' => 'number',
        'accename' => 'string',
        'germplasm_type' => 'string',
        'source' => 'string',
        'accenumb' => 'string',
        'sampstat' => 'string',
        'comments' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivPassport->has_a(
    cdv_source_id => 'Gramene::CDBI::DiversityMaize::CdvSource'
);

Gramene::CDBI::DiversityMaize::DivPassport->has_a(
    div_accession_collecting_id => 'Gramene::CDBI::DiversityMaize::DivAccessionCollecting'
);

Gramene::CDBI::DiversityMaize::DivPassport->has_a(
    div_taxonomy_id => 'Gramene::CDBI::DiversityMaize::DivTaxonomy'
);

Gramene::CDBI::DiversityMaize::DivPassport->has_many( 
    'cdv_passport_sets',
    'Gramene::CDBI::DiversityMaize::CdvPassportSet',
    'div_passport_id' 
);

Gramene::CDBI::DiversityMaize::DivPassport->has_many( 
    'div_stocks',
    'Gramene::CDBI::DiversityMaize::DivStock',
    'div_passport_id' 
);

Gramene::CDBI::DiversityMaize::DivPassport->has_many( 
    'div_synonyms',
    'Gramene::CDBI::DiversityMaize::DivSynonym',
    'div_passport_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_passport_set => [ 'div_passport_id', 'div_passport_id' ],
            cdv_source => [ 'cdv_source_id', 'cdv_source_id' ],
            div_accession_collecting => [ 'div_accession_collecting_id', 'div_accession_collecting_id' ],
            div_stock => [ 'div_passport_id', 'div_passport_id' ],
            div_synonym => [ 'div_passport_id', 'div_passport_id' ],
            div_taxonomy => [ 'div_taxonomy_id', 'div_taxonomy_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::CdvSource; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::CdvSource->table('cdv_source');

our @extras = ();
Gramene::CDBI::DiversityMaize::CdvSource->columns( 
    All => qw/ 
        cdv_source_id
        source
        contact
        institute
        department
        address
        city
        state_province
        country
        phone
        fax
        email
        url
        comments 
    /, @extras
);

Gramene::CDBI::DiversityMaize::CdvSource->columns( 
    Essential => qw/ 
        cdv_source_id
        source
        contact
        institute
        department
        address
        city
        state_province
        country
        phone
        fax
        email
        url 
    /
);

Gramene::CDBI::DiversityMaize::CdvSource->columns( 
    Ordered => qw/ 
        cdv_source_id
        source
        contact
        institute
        department
        address
        city
        state_province
        country
        phone
        fax
        email
        url
        comments 
    /
);

sub to_string {
    my $self = shift;
    return $self->source || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'cdv_source_id' => 'number',
        'source' => 'string',
        'contact' => 'string',
        'institute' => 'string',
        'department' => 'string',
        'address' => 'string',
        'city' => 'string',
        'state_province' => 'string',
        'country' => 'string',
        'phone' => 'string',
        'fax' => 'string',
        'email' => 'string',
        'url' => 'string',
        'comments' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::CdvSource->has_many( 
    'div_passports',
    'Gramene::CDBI::DiversityMaize::DivPassport',
    'cdv_source_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_passport => [ 'cdv_source_id', 'cdv_source_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivAccessionCollecting; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivAccessionCollecting->table('div_accession_collecting');


sub div_locality {
    my $self = shift;
    return $self->div_locality_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::DivAccessionCollecting->columns( 
    All => qw/ 
        div_accession_collecting_id
        div_locality_id
        collector
        collnumb
        collsrc
        collcode
        col_date 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivAccessionCollecting->columns( 
    Essential => qw/ 
        div_accession_collecting_id
        div_locality_id
        collector
        collnumb
        collsrc
        collcode
        col_date 
    /
);

Gramene::CDBI::DiversityMaize::DivAccessionCollecting->columns( 
    Ordered => qw/ 
        div_accession_collecting_id
        div_locality_id
        collector
        collnumb
        collsrc
        collcode
        col_date 
    /
);

sub to_string {
    my $self = shift;
    return $self->collector || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_accession_collecting_id' => 'number',
        'div_locality_id' => 'number',
        'collector' => 'string',
        'collnumb' => 'string',
        'collsrc' => 'string',
        'collcode' => 'string',
        'col_date' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivAccessionCollecting->has_a(
    div_locality_id => 'Gramene::CDBI::DiversityMaize::DivLocality'
);

Gramene::CDBI::DiversityMaize::DivAccessionCollecting->has_many( 
    'div_passports',
    'Gramene::CDBI::DiversityMaize::DivPassport',
    'div_accession_collecting_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_locality => [ 'div_locality_id', 'div_locality_id' ],
            div_passport => [ 'div_accession_collecting_id', 'div_accession_collecting_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivAnnotationType; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivAnnotationType->table('div_annotation_type');

our @extras = ();
Gramene::CDBI::DiversityMaize::DivAnnotationType->columns( 
    All => qw/ 
        div_annotation_type_id
        anno_type 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivAnnotationType->columns( 
    Essential => qw/ 
        div_annotation_type_id
        anno_type 
    /
);

Gramene::CDBI::DiversityMaize::DivAnnotationType->columns( 
    Ordered => qw/ 
        div_annotation_type_id
        anno_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->anno_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_annotation_type_id' => 'number',
        'anno_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivAnnotationType->has_many( 
    'div_aa_annotations',
    'Gramene::CDBI::DiversityMaize::DivAaAnnotation',
    'div_annotation_type_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_aa_annotation => [ 'div_annotation_type_id', 'div_annotation_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::QtlObsUnitSet; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::QtlObsUnitSet->table('qtl_obs_unit_set');


sub qtl_obs_unit {
    my $self = shift;
    return $self->qtl_obs_unit_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::QtlObsUnitSet->columns( 
    All => qw/ 
        qtl_obs_unit_set_id
        qtl_obs_unit_id
        name
        comments 
    /, @extras
);

Gramene::CDBI::DiversityMaize::QtlObsUnitSet->columns( 
    Essential => qw/ 
        qtl_obs_unit_set_id
        qtl_obs_unit_id
        name 
    /
);

Gramene::CDBI::DiversityMaize::QtlObsUnitSet->columns( 
    Ordered => qw/ 
        qtl_obs_unit_set_id
        qtl_obs_unit_id
        name
        comments 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'qtl_obs_unit_set_id' => 'number',
        'qtl_obs_unit_id' => 'number',
        'name' => 'string',
        'comments' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::QtlObsUnitSet->has_a(
    qtl_obs_unit_id => 'Gramene::CDBI::DiversityMaize::QtlObsUnit'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            qtl_obs_unit => [ 'qtl_obs_unit_id', 'qtl_obs_unit_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::QtlTrait; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::QtlTrait->table('qtl_trait');


sub div_statistic_type {
    my $self = shift;
    return $self->div_statistic_type_id;
}
    

sub div_trait_uom {
    my $self = shift;
    return $self->div_trait_uom_id;
}
    

sub qtl_obs_unit {
    my $self = shift;
    return $self->qtl_obs_unit_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::QtlTrait->columns( 
    All => qw/ 
        qtl_trait_id
        div_trait_uom_id
        div_statistic_type_id
        qtl_obs_unit_id
        date_measured
        value 
    /, @extras
);

Gramene::CDBI::DiversityMaize::QtlTrait->columns( 
    Essential => qw/ 
        qtl_trait_id
        div_trait_uom_id
        div_statistic_type_id
        qtl_obs_unit_id
        date_measured
        value 
    /
);

Gramene::CDBI::DiversityMaize::QtlTrait->columns( 
    Ordered => qw/ 
        qtl_trait_id
        div_trait_uom_id
        div_statistic_type_id
        qtl_obs_unit_id
        date_measured
        value 
    /
);

sub to_string {
    my $self = shift;
    return $self->date_measured || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'qtl_trait_id' => 'number',
        'div_trait_uom_id' => 'number',
        'div_statistic_type_id' => 'number',
        'qtl_obs_unit_id' => 'number',
        'date_measured' => 'string',
        'value' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::QtlTrait->has_a(
    div_statistic_type_id => 'Gramene::CDBI::DiversityMaize::DivStatisticType'
);

Gramene::CDBI::DiversityMaize::QtlTrait->has_a(
    div_trait_uom_id => 'Gramene::CDBI::DiversityMaize::DivTraitUom'
);

Gramene::CDBI::DiversityMaize::QtlTrait->has_a(
    qtl_obs_unit_id => 'Gramene::CDBI::DiversityMaize::QtlObsUnit'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_statistic_type => [ 'div_statistic_type_id', 'div_statistic_type_id' ],
            div_trait_uom => [ 'div_trait_uom_id', 'div_trait_uom_id' ],
            qtl_obs_unit => [ 'qtl_obs_unit_id', 'qtl_obs_unit_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::QtlObsUnit; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::QtlObsUnit->table('qtl_obs_unit');


sub qtl_study {
    my $self = shift;
    return $self->qtl_study_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::QtlObsUnit->columns( 
    All => qw/ 
        qtl_obs_unit_id
        qtl_study_id
        name
        comments 
    /, @extras
);

Gramene::CDBI::DiversityMaize::QtlObsUnit->columns( 
    Essential => qw/ 
        qtl_obs_unit_id
        qtl_study_id
        name 
    /
);

Gramene::CDBI::DiversityMaize::QtlObsUnit->columns( 
    Ordered => qw/ 
        qtl_obs_unit_id
        qtl_study_id
        name
        comments 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'qtl_obs_unit_id' => 'number',
        'qtl_study_id' => 'number',
        'name' => 'string',
        'comments' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::QtlObsUnit->has_a(
    qtl_study_id => 'Gramene::CDBI::DiversityMaize::QtlStudy'
);

Gramene::CDBI::DiversityMaize::QtlObsUnit->has_many( 
    'qtl_analysis_results',
    'Gramene::CDBI::DiversityMaize::QtlAnalysisResult',
    'qtl_obs_unit_id' 
);

Gramene::CDBI::DiversityMaize::QtlObsUnit->has_many( 
    'qtl_obs_unit_sets',
    'Gramene::CDBI::DiversityMaize::QtlObsUnitSet',
    'qtl_obs_unit_id' 
);

Gramene::CDBI::DiversityMaize::QtlObsUnit->has_many( 
    'qtl_traits',
    'Gramene::CDBI::DiversityMaize::QtlTrait',
    'qtl_obs_unit_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            qtl_analysis_result => [ 'qtl_obs_unit_id', 'qtl_obs_unit_id' ],
            qtl_obs_unit_set => [ 'qtl_obs_unit_id', 'qtl_obs_unit_id' ],
            qtl_study => [ 'qtl_study_id', 'qtl_study_id' ],
            qtl_trait => [ 'qtl_obs_unit_id', 'qtl_obs_unit_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::QtlStudy; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::QtlStudy->table('qtl_study');


sub div_experiment {
    my $self = shift;
    return $self->div_experiment_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::QtlStudy->columns( 
    All => qw/ 
        qtl_study_id
        name
        reference_id
        div_experiment_id
        cmap_map_set_acc 
    /, @extras
);

Gramene::CDBI::DiversityMaize::QtlStudy->columns( 
    Essential => qw/ 
        qtl_study_id
        name
        reference_id
        div_experiment_id
        cmap_map_set_acc 
    /
);

Gramene::CDBI::DiversityMaize::QtlStudy->columns( 
    Ordered => qw/ 
        qtl_study_id
        name
        reference_id
        div_experiment_id
        cmap_map_set_acc 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'qtl_study_id' => 'number',
        'name' => 'string',
        'reference_id' => 'number',
        'div_experiment_id' => 'number',
        'cmap_map_set_acc' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::QtlStudy->has_a(
    div_experiment_id => 'Gramene::CDBI::DiversityMaize::DivExperiment'
);

Gramene::CDBI::DiversityMaize::QtlStudy->has_many( 
    'qtl_analysis_methods',
    'Gramene::CDBI::DiversityMaize::QtlAnalysisMethod',
    'qtl_study_id' 
);

Gramene::CDBI::DiversityMaize::QtlStudy->has_many( 
    'qtl_obs_units',
    'Gramene::CDBI::DiversityMaize::QtlObsUnit',
    'qtl_study_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_experiment => [ 'div_experiment_id', 'div_experiment_id' ],
            qtl_analysis_method => [ 'qtl_study_id', 'qtl_study_id' ],
            qtl_obs_unit => [ 'qtl_study_id', 'qtl_study_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivExperiment; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivExperiment->table('div_experiment');

our @extras = ();
Gramene::CDBI::DiversityMaize::DivExperiment->columns( 
    All => qw/ 
        div_experiment_id
        name
        design
        originator
        comments 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivExperiment->columns( 
    Essential => qw/ 
        div_experiment_id
        name
        design
        originator 
    /
);

Gramene::CDBI::DiversityMaize::DivExperiment->columns( 
    Ordered => qw/ 
        div_experiment_id
        name
        design
        originator
        comments 
    /
);

sub to_string {
    my $self = shift;
    return $self->name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_experiment_id' => 'number',
        'name' => 'string',
        'design' => 'string',
        'originator' => 'string',
        'comments' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivExperiment->has_many( 
    'div_obs_units',
    'Gramene::CDBI::DiversityMaize::DivObsUnit',
    'div_experiment_id' 
);

Gramene::CDBI::DiversityMaize::DivExperiment->has_many( 
    'qtl_studies',
    'Gramene::CDBI::DiversityMaize::QtlStudy',
    'div_experiment_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_obs_unit => [ 'div_experiment_id', 'div_experiment_id' ],
            qtl_study => [ 'div_experiment_id', 'div_experiment_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivGeneration; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivGeneration->table('div_generation');

our @extras = ();
Gramene::CDBI::DiversityMaize::DivGeneration->columns( 
    All => qw/ 
        div_generation_id
        value
        icis_id
        comments
        selfing_number
        sibbing_number 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivGeneration->columns( 
    Essential => qw/ 
        div_generation_id
        value
        icis_id
        selfing_number
        sibbing_number 
    /
);

Gramene::CDBI::DiversityMaize::DivGeneration->columns( 
    Ordered => qw/ 
        div_generation_id
        value
        icis_id
        comments
        selfing_number
        sibbing_number 
    /
);

sub to_string {
    my $self = shift;
    return $self->value || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_generation_id' => 'number',
        'value' => 'string',
        'icis_id' => 'string',
        'comments' => 'string',
        'selfing_number' => 'number',
        'sibbing_number' => 'number',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivGeneration->has_many( 
    'div_stocks',
    'Gramene::CDBI::DiversityMaize::DivStock',
    'div_generation_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_stock => [ 'div_generation_id', 'div_generation_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivLocality; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivLocality->table('div_locality');

our @extras = ();
Gramene::CDBI::DiversityMaize::DivLocality->columns( 
    All => qw/ 
        div_locality_id
        elevation
        city
        country
        origcty
        latitude
        longitude
        locality_name
        state_province 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivLocality->columns( 
    Essential => qw/ 
        div_locality_id
        elevation
        city
        country
        origcty
        latitude
        longitude
        locality_name
        state_province 
    /
);

Gramene::CDBI::DiversityMaize::DivLocality->columns( 
    Ordered => qw/ 
        div_locality_id
        elevation
        city
        country
        origcty
        latitude
        longitude
        locality_name
        state_province 
    /
);

sub to_string {
    my $self = shift;
    return $self->elevation || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_locality_id' => 'number',
        'elevation' => 'number',
        'city' => 'string',
        'country' => 'string',
        'origcty' => 'string',
        'latitude' => 'number',
        'longitude' => 'number',
        'locality_name' => 'string',
        'state_province' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivLocality->has_many( 
    'div_accession_collectings',
    'Gramene::CDBI::DiversityMaize::DivAccessionCollecting',
    'div_locality_id' 
);

Gramene::CDBI::DiversityMaize::DivLocality->has_many( 
    'div_obs_units',
    'Gramene::CDBI::DiversityMaize::DivObsUnit',
    'div_locality_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_accession_collecting => [ 'div_locality_id', 'div_locality_id' ],
            div_obs_unit => [ 'div_locality_id', 'div_locality_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivPolyType; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivPolyType->table('div_poly_type');

our @extras = ();
Gramene::CDBI::DiversityMaize::DivPolyType->columns( 
    All => qw/ 
        div_poly_type_id
        poly_type 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivPolyType->columns( 
    Essential => qw/ 
        div_poly_type_id
        poly_type 
    /
);

Gramene::CDBI::DiversityMaize::DivPolyType->columns( 
    Ordered => qw/ 
        div_poly_type_id
        poly_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->poly_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_poly_type_id' => 'number',
        'poly_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivPolyType->has_many( 
    'div_allele_assays',
    'Gramene::CDBI::DiversityMaize::DivAlleleAssay',
    'div_poly_type_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_allele_assay => [ 'div_poly_type_id', 'div_poly_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivScoringTechType; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivScoringTechType->table('div_scoring_tech_type');

our @extras = ();
Gramene::CDBI::DiversityMaize::DivScoringTechType->columns( 
    All => qw/ 
        div_scoring_tech_type_id
        scoring_tech_group
        scoring_tech_type 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivScoringTechType->columns( 
    Essential => qw/ 
        div_scoring_tech_type_id
        scoring_tech_group
        scoring_tech_type 
    /
);

Gramene::CDBI::DiversityMaize::DivScoringTechType->columns( 
    Ordered => qw/ 
        div_scoring_tech_type_id
        scoring_tech_group
        scoring_tech_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->scoring_tech_group || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_scoring_tech_type_id' => 'number',
        'scoring_tech_group' => 'string',
        'scoring_tech_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivScoringTechType->has_many( 
    'div_allele_assays',
    'Gramene::CDBI::DiversityMaize::DivAlleleAssay',
    'div_scoring_tech_type_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_allele_assay => [ 'div_scoring_tech_type_id', 'div_scoring_tech_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivStatisticType; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivStatisticType->table('div_statistic_type');

our @extras = ();
Gramene::CDBI::DiversityMaize::DivStatisticType->columns( 
    All => qw/ 
        div_statistic_type_id
        stat_type 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivStatisticType->columns( 
    Essential => qw/ 
        div_statistic_type_id
        stat_type 
    /
);

Gramene::CDBI::DiversityMaize::DivStatisticType->columns( 
    Ordered => qw/ 
        div_statistic_type_id
        stat_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->stat_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_statistic_type_id' => 'number',
        'stat_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivStatisticType->has_many( 
    'div_traits',
    'Gramene::CDBI::DiversityMaize::DivTrait',
    'div_statistic_type_id' 
);

Gramene::CDBI::DiversityMaize::DivStatisticType->has_many( 
    'qtl_traits',
    'Gramene::CDBI::DiversityMaize::QtlTrait',
    'div_statistic_type_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_trait => [ 'div_statistic_type_id', 'div_statistic_type_id' ],
            qtl_trait => [ 'div_statistic_type_id', 'div_statistic_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivTaxonomy; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivTaxonomy->table('div_taxonomy');

our @extras = ();
Gramene::CDBI::DiversityMaize::DivTaxonomy->columns( 
    All => qw/ 
        div_taxonomy_id
        genus
        species
        subspecies
        subtaxa
        race
        population
        common_name
        term_accession 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivTaxonomy->columns( 
    Essential => qw/ 
        div_taxonomy_id
        genus
        species
        subspecies
        subtaxa
        race
        population
        common_name
        term_accession 
    /
);

Gramene::CDBI::DiversityMaize::DivTaxonomy->columns( 
    Ordered => qw/ 
        div_taxonomy_id
        genus
        species
        subspecies
        subtaxa
        race
        population
        common_name
        term_accession 
    /
);

sub to_string {
    my $self = shift;
    return $self->genus || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_taxonomy_id' => 'number',
        'genus' => 'string',
        'species' => 'string',
        'subspecies' => 'string',
        'subtaxa' => 'string',
        'race' => 'string',
        'population' => 'string',
        'common_name' => 'string',
        'term_accession' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivTaxonomy->has_many( 
    'div_passports',
    'Gramene::CDBI::DiversityMaize::DivPassport',
    'div_taxonomy_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_passport => [ 'div_taxonomy_id', 'div_taxonomy_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivTraitUom; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivTraitUom->table('div_trait_uom');


sub div_unit_of_measure {
    my $self = shift;
    return $self->div_unit_of_measure_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::DivTraitUom->columns( 
    All => qw/ 
        div_trait_uom_id
        div_unit_of_measure_id
        local_trait_name
        trait_protocol
        to_accession
        eo_accession 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivTraitUom->columns( 
    Essential => qw/ 
        div_trait_uom_id
        div_unit_of_measure_id
        local_trait_name
        trait_protocol
        to_accession
        eo_accession 
    /
);

Gramene::CDBI::DiversityMaize::DivTraitUom->columns( 
    Ordered => qw/ 
        div_trait_uom_id
        div_unit_of_measure_id
        local_trait_name
        trait_protocol
        to_accession
        eo_accession 
    /
);

sub to_string {
    my $self = shift;
    return $self->local_trait_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_trait_uom_id' => 'number',
        'div_unit_of_measure_id' => 'number',
        'local_trait_name' => 'string',
        'trait_protocol' => 'string',
        'to_accession' => 'string',
        'eo_accession' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivTraitUom->has_a(
    div_unit_of_measure_id => 'Gramene::CDBI::DiversityMaize::DivUnitOfMeasure'
);

Gramene::CDBI::DiversityMaize::DivTraitUom->has_many( 
    'div_traits',
    'Gramene::CDBI::DiversityMaize::DivTrait',
    'div_trait_uom_id' 
);

Gramene::CDBI::DiversityMaize::DivTraitUom->has_many( 
    'qtl_analysis_results',
    'Gramene::CDBI::DiversityMaize::QtlAnalysisResult',
    'div_trait_uom_id' 
);

Gramene::CDBI::DiversityMaize::DivTraitUom->has_many( 
    'qtl_traits',
    'Gramene::CDBI::DiversityMaize::QtlTrait',
    'div_trait_uom_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_trait => [ 'div_trait_uom_id', 'div_trait_uom_id' ],
            div_unit_of_measure => [ 'div_unit_of_measure_id', 'div_unit_of_measure_id' ],
            qtl_analysis_result => [ 'div_trait_uom_id', 'div_trait_uom_id' ],
            qtl_trait => [ 'div_trait_uom_id', 'div_trait_uom_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivTreatmentUom; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivTreatmentUom->table('div_treatment_uom');


sub div_unit_of_measure {
    my $self = shift;
    return $self->div_unit_of_measure_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::DivTreatmentUom->columns( 
    All => qw/ 
        div_treatment_uom_id
        div_unit_of_measure_id
        local_treatment_name
        treatment_protocol
        eo_accession 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivTreatmentUom->columns( 
    Essential => qw/ 
        div_treatment_uom_id
        div_unit_of_measure_id
        local_treatment_name
        treatment_protocol
        eo_accession 
    /
);

Gramene::CDBI::DiversityMaize::DivTreatmentUom->columns( 
    Ordered => qw/ 
        div_treatment_uom_id
        div_unit_of_measure_id
        local_treatment_name
        treatment_protocol
        eo_accession 
    /
);

sub to_string {
    my $self = shift;
    return $self->local_treatment_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_treatment_uom_id' => 'number',
        'div_unit_of_measure_id' => 'number',
        'local_treatment_name' => 'string',
        'treatment_protocol' => 'string',
        'eo_accession' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivTreatmentUom->has_a(
    div_unit_of_measure_id => 'Gramene::CDBI::DiversityMaize::DivUnitOfMeasure'
);

Gramene::CDBI::DiversityMaize::DivTreatmentUom->has_many( 
    'div_treatments',
    'Gramene::CDBI::DiversityMaize::DivTreatment',
    'div_treatment_uom_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_treatment => [ 'div_treatment_uom_id', 'div_treatment_uom_id' ],
            div_unit_of_measure => [ 'div_unit_of_measure_id', 'div_unit_of_measure_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::DivUnitOfMeasure; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::DivUnitOfMeasure->table('div_unit_of_measure');

our @extras = ();
Gramene::CDBI::DiversityMaize::DivUnitOfMeasure->columns( 
    All => qw/ 
        div_unit_of_measure_id
        unit_type 
    /, @extras
);

Gramene::CDBI::DiversityMaize::DivUnitOfMeasure->columns( 
    Essential => qw/ 
        div_unit_of_measure_id
        unit_type 
    /
);

Gramene::CDBI::DiversityMaize::DivUnitOfMeasure->columns( 
    Ordered => qw/ 
        div_unit_of_measure_id
        unit_type 
    /
);

sub to_string {
    my $self = shift;
    return $self->unit_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'div_unit_of_measure_id' => 'number',
        'unit_type' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::DivUnitOfMeasure->has_many( 
    'div_trait_uoms',
    'Gramene::CDBI::DiversityMaize::DivTraitUom',
    'div_unit_of_measure_id' 
);

Gramene::CDBI::DiversityMaize::DivUnitOfMeasure->has_many( 
    'div_treatment_uoms',
    'Gramene::CDBI::DiversityMaize::DivTreatmentUom',
    'div_unit_of_measure_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_trait_uom => [ 'div_unit_of_measure_id', 'div_unit_of_measure_id' ],
            div_treatment_uom => [ 'div_unit_of_measure_id', 'div_unit_of_measure_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::GdpdmSearch; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::GdpdmSearch->table('gdpdm_search');

our @extras = ();
push @extras, Class::DBI::Column->new( id => { 
    accessor => 'id_field',
    mutator  => 'id_field' 
}); 

Gramene::CDBI::DiversityMaize::GdpdmSearch->columns( 
    All => qw/ 
        id
        table_name
        record_id
        record_text 
    /, @extras
);

Gramene::CDBI::DiversityMaize::GdpdmSearch->columns( 
    Essential => qw/ 
        id
        table_name
        record_id
        record_text 
    /
);

Gramene::CDBI::DiversityMaize::GdpdmSearch->columns( 
    Ordered => qw/ 
        id
        table_name
        record_id
        record_text 
    /
);

sub to_string {
    my $self = shift;
    return $self->table_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'id' => 'number',
        'table_name' => 'string',
        'record_id' => 'number',
        'record_text' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}



sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::Xref; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::Xref->table('xref');


sub xref_type {
    my $self = shift;
    return $self->xref_type_id;
}
    
our @extras = ();
Gramene::CDBI::DiversityMaize::Xref->columns( 
    All => qw/ 
        xref_id
        table_name
        record_id
        xref_type_id
        xref_value
        comments 
    /, @extras
);

Gramene::CDBI::DiversityMaize::Xref->columns( 
    Essential => qw/ 
        xref_id
        table_name
        record_id
        xref_type_id
        xref_value 
    /
);

Gramene::CDBI::DiversityMaize::Xref->columns( 
    Ordered => qw/ 
        xref_id
        table_name
        record_id
        xref_type_id
        xref_value
        comments 
    /
);

sub to_string {
    my $self = shift;
    return $self->table_name || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'xref_id' => 'number',
        'table_name' => 'string',
        'record_id' => 'number',
        'xref_type_id' => 'number',
        'xref_value' => 'string',
        'comments' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::Xref->has_a(
    xref_type_id => 'Gramene::CDBI::DiversityMaize::XrefType'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            xref_type => [ 'xref_type_id', 'xref_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope

# ------------------------------------------------

package Gramene::CDBI::DiversityMaize::XrefType; {

BEGIN { our @ISA = qw( Gramene::CDBI::DiversityMaize ); };
# use base qw( Gramene::CDBI::DiversityMaize );

Gramene::CDBI::DiversityMaize::XrefType->table('xref_type');

our @extras = ();
Gramene::CDBI::DiversityMaize::XrefType->columns( 
    All => qw/ 
        xref_type_id
        xref_type
        url_template 
    /, @extras
);

Gramene::CDBI::DiversityMaize::XrefType->columns( 
    Essential => qw/ 
        xref_type_id
        xref_type
        url_template 
    /
);

Gramene::CDBI::DiversityMaize::XrefType->columns( 
    Ordered => qw/ 
        xref_type_id
        xref_type
        url_template 
    /
);

sub to_string {
    my $self = shift;
    return $self->xref_type || '';
}

sub data_type {
    my $self   = shift;
    my $column = shift or croak('No field name');
    my %type   = {
        'xref_type_id' => 'number',
        'xref_type' => 'string',
        'url_template' => 'string',
    };

    my $type = $type{ $column } or croak("Invalid column ($column)");

    return $type;
}


Gramene::CDBI::DiversityMaize::XrefType->has_many( 
    'xrefs',
    'Gramene::CDBI::DiversityMaize::Xref',
    'xref_type_id' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            xref => [ 'xref_type_id', 'xref_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}

} # end package scope



1;

__END__

# ----------------------------------------------------

=pod

=head1 NAME

Gramene::CDBI::DiversityMaize - Gramene::CDBI-base classes for DiversityMaize

=head1 VERSION

This documentation refers to version $Revision: 1.5 $.

=head1 SYNOPSIS

  use Gramene::CDBI::DiversityMaize;

  my @data   = Gramene::CDBI::DiversityMaize::DivStockParent->retrieve_all;
  my $object = Gramene::CDBI::DiversityMaize::DivStockParent->retrieve( 42 );

=head1 DESCRIPTION

This module contains Gramene::CDBI-base classes for the DiversityMaize 
schema.

=head1 METHODS 

=head2 config_name

  my $config_name = $self->config_name;

Returns the string 'diversity_maize'.

=head2 represented_tables

  my @tables = $cdbi->represented_tables;

Returns the names of the tables in the module.  See the CLASSES below.

=cut

=head1 CLASSES

=head2 Gramene::CDBI::DiversityMaize::CdvAlleleCuratedAllele

Corresponds to the table "cdv_allele_curated_allele."  

Stringifies (C<to_name>) to "cdv_allele_curated_allele_id".

Fields:

=over 4

=item * cdv_allele_curated_allele_id

=item * cdv_curated_allele_id

=item * div_allele_id

=item * cdv_curation_id

=back


Has A:

=over 4

=item * cdv_curated_allele => Gramene::CDBI::DiversityMaize::CdvCuratedAllele

=item * cdv_curation => Gramene::CDBI::DiversityMaize::CdvCuration

=item * div_allele => Gramene::CDBI::DiversityMaize::DivAllele

=back



=head2 Gramene::CDBI::DiversityMaize::CdvCuratedAllele

Corresponds to the table "cdv_curated_allele."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * cdv_curated_allele_id

=item * name

=back



Has Many:

=over 4

=item * cdv_allele_curated_alleles => Gramene::CDBI::DiversityMaize::CdvAlleleCuratedAllele

=back


=head2 Gramene::CDBI::DiversityMaize::CdvCuration

Corresponds to the table "cdv_curation."  

Stringifies (C<to_name>) to "curator".

Fields:

=over 4

=item * cdv_curation_id

=item * cdv_reason_id

=item * curator

=item * curation_date

=item * comments

=back


Has A:

=over 4

=item * cdv_reason => Gramene::CDBI::DiversityMaize::CdvReason

=back


Has Many:

=over 4

=item * cdv_allele_curated_alleles => Gramene::CDBI::DiversityMaize::CdvAlleleCuratedAllele

=back


=head2 Gramene::CDBI::DiversityMaize::CdvMapFeature

Corresponds to the table "cdv_map_feature."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * cdv_map_feature_id

=item * name

=item * chromosome_name

=item * genetic_bin

=item * xref_map_id

=item * genetic_map

=item * genetic_position

=item * locus_type

=item * physical_position

=item * comments

=back



Has Many:

=over 4

=item * cdv_map_feature_annotations => Gramene::CDBI::DiversityMaize::CdvMapFeatureAnnotation

=item * cdv_markers => Gramene::CDBI::DiversityMaize::CdvMarker

=back


=head2 Gramene::CDBI::DiversityMaize::CdvMapFeatureAnnotation

Corresponds to the table "cdv_map_feature_annotation."  

Stringifies (C<to_name>) to "annotation_value".

Fields:

=over 4

=item * cdv_map_feature_annotation_id

=item * cdv_map_feature_id

=item * cdv_map_feature_annotation_type_id

=item * annotation_value

=back


Has A:

=over 4

=item * cdv_map_feature => Gramene::CDBI::DiversityMaize::CdvMapFeature

=item * cdv_map_feature_annotation_type => Gramene::CDBI::DiversityMaize::CdvMapFeatureAnnotationType

=back



=head2 Gramene::CDBI::DiversityMaize::CdvMapFeatureAnnotationType

Corresponds to the table "cdv_map_feature_annotation_type."  

Stringifies (C<to_name>) to "anno_type".

Fields:

=over 4

=item * cdv_map_feature_annotation_type_id

=item * anno_type

=back



Has Many:

=over 4

=item * cdv_map_feature_annotations => Gramene::CDBI::DiversityMaize::CdvMapFeatureAnnotation

=back


=head2 Gramene::CDBI::DiversityMaize::CdvMarker

Corresponds to the table "cdv_marker."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * cdv_marker_id

=item * cdv_map_feature_id

=item * div_ref_stock_id

=item * name

=item * position

=item * length

=item * ref_seq

=item * marker_aid

=back


Has A:

=over 4

=item * cdv_map_feature => Gramene::CDBI::DiversityMaize::CdvMapFeature

=item * div_ref_stock => Gramene::CDBI::DiversityMaize::DivStock

=back


Has Many:

=over 4

=item * cdv_marker_annotations => Gramene::CDBI::DiversityMaize::CdvMarkerAnnotation

=item * div_allele_assays => Gramene::CDBI::DiversityMaize::DivAlleleAssay

=item * qtl_markers => Gramene::CDBI::DiversityMaize::QtlMarker

=back


=head2 Gramene::CDBI::DiversityMaize::CdvMarkerAnnotation

Corresponds to the table "cdv_marker_annotation."  

Stringifies (C<to_name>) to "annotation_value".

Fields:

=over 4

=item * cdv_marker_annotation_id

=item * cdv_marker_id

=item * cdv_marker_annotation_type_id

=item * annotation_value

=back


Has A:

=over 4

=item * cdv_marker => Gramene::CDBI::DiversityMaize::CdvMarker

=item * cdv_marker_annotation_type => Gramene::CDBI::DiversityMaize::CdvMarkerAnnotationType

=back



=head2 Gramene::CDBI::DiversityMaize::CdvMarkerAnnotationType

Corresponds to the table "cdv_marker_annotation_type."  

Stringifies (C<to_name>) to "anno_type".

Fields:

=over 4

=item * cdv_marker_annotation_type_id

=item * anno_type

=back



Has Many:

=over 4

=item * cdv_marker_annotations => Gramene::CDBI::DiversityMaize::CdvMarkerAnnotation

=back


=head2 Gramene::CDBI::DiversityMaize::CdvPassportGroup

Corresponds to the table "cdv_passport_group."  

Stringifies (C<to_name>) to "group_name".

Fields:

=over 4

=item * cdv_passport_group_id

=item * group_name

=back



Has Many:

=over 4

=item * cdv_passport_sets => Gramene::CDBI::DiversityMaize::CdvPassportSet

=back


=head2 Gramene::CDBI::DiversityMaize::CdvPassportSet

Corresponds to the table "cdv_passport_set."  

Stringifies (C<to_name>) to "cdv_passport_set_id".

Fields:

=over 4

=item * cdv_passport_set_id

=item * div_passport_id

=item * cdv_passport_group_id

=back


Has A:

=over 4

=item * cdv_passport_group => Gramene::CDBI::DiversityMaize::CdvPassportGroup

=item * div_passport => Gramene::CDBI::DiversityMaize::DivPassport

=back



=head2 Gramene::CDBI::DiversityMaize::CdvReason

Corresponds to the table "cdv_reason."  

Stringifies (C<to_name>) to "description".

Fields:

=over 4

=item * cdv_reason_id

=item * description

=back



Has Many:

=over 4

=item * cdv_curations => Gramene::CDBI::DiversityMaize::CdvCuration

=back


=head2 Gramene::CDBI::DiversityMaize::CdvSource

Corresponds to the table "cdv_source."  

Stringifies (C<to_name>) to "source".

Fields:

=over 4

=item * cdv_source_id

=item * source

=item * contact

=item * institute

=item * department

=item * address

=item * city

=item * state_province

=item * country

=item * phone

=item * fax

=item * email

=item * url

=item * comments

=back



Has Many:

=over 4

=item * div_passports => Gramene::CDBI::DiversityMaize::DivPassport

=back


=head2 Gramene::CDBI::DiversityMaize::DivAaAnnotation

Corresponds to the table "div_aa_annotation."  

Stringifies (C<to_name>) to "annotation_value".

Fields:

=over 4

=item * div_aa_annotation_id

=item * div_annotation_type_id

=item * div_allele_assay_id

=item * annotation_value

=back


Has A:

=over 4

=item * div_allele_assay => Gramene::CDBI::DiversityMaize::DivAlleleAssay

=item * div_annotation_type => Gramene::CDBI::DiversityMaize::DivAnnotationType

=back



=head2 Gramene::CDBI::DiversityMaize::DivAccessionCollecting

Corresponds to the table "div_accession_collecting."  

Stringifies (C<to_name>) to "collector".

Fields:

=over 4

=item * div_accession_collecting_id

=item * div_locality_id

=item * collector

=item * collnumb

=item * collsrc

=item * collcode

=item * col_date

=back


Has A:

=over 4

=item * div_locality => Gramene::CDBI::DiversityMaize::DivLocality

=back


Has Many:

=over 4

=item * div_passports => Gramene::CDBI::DiversityMaize::DivPassport

=back


=head2 Gramene::CDBI::DiversityMaize::DivAllele

Corresponds to the table "div_allele."  

Stringifies (C<to_name>) to "accession".

Fields:

=over 4

=item * div_allele_id

=item * div_obs_unit_sample_id

=item * div_allele_assay_id

=item * accession

=item * referencedb

=item * allele_num

=item * quality

=item * value

=item * proportion

=item * total_n

=back


Has A:

=over 4

=item * div_allele_assay => Gramene::CDBI::DiversityMaize::DivAlleleAssay

=item * div_obs_unit_sample => Gramene::CDBI::DiversityMaize::DivObsUnitSample

=back


Has Many:

=over 4

=item * cdv_allele_curated_alleles => Gramene::CDBI::DiversityMaize::CdvAlleleCuratedAllele

=back


=head2 Gramene::CDBI::DiversityMaize::DivAlleleAssay

Corresponds to the table "div_allele_assay."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * div_allele_assay_id

=item * div_source_assay_id

=item * div_poly_type_id

=item * div_scoring_tech_type_id

=item * div_ref_stock_id

=item * cdv_marker_id

=item * name

=item * assay_date

=item * producer

=item * position

=item * ref_seq

=item * length

=item * comments

=back


Has A:

=over 4

=item * cdv_marker => Gramene::CDBI::DiversityMaize::CdvMarker

=item * div_source_assay => Gramene::CDBI::DiversityMaize::DivAlleleAssay

=item * div_poly_type => Gramene::CDBI::DiversityMaize::DivPolyType

=item * div_scoring_tech_type => Gramene::CDBI::DiversityMaize::DivScoringTechType

=item * div_ref_stock => Gramene::CDBI::DiversityMaize::DivStock

=back


Has Many:

=over 4

=item * div_aa_annotations => Gramene::CDBI::DiversityMaize::DivAaAnnotation

=item * div_alleles => Gramene::CDBI::DiversityMaize::DivAllele

=item * div_allele_assays => Gramene::CDBI::DiversityMaize::DivAlleleAssay

=back


=head2 Gramene::CDBI::DiversityMaize::DivAnnotationType

Corresponds to the table "div_annotation_type."  

Stringifies (C<to_name>) to "anno_type".

Fields:

=over 4

=item * div_annotation_type_id

=item * anno_type

=back



Has Many:

=over 4

=item * div_aa_annotations => Gramene::CDBI::DiversityMaize::DivAaAnnotation

=back


=head2 Gramene::CDBI::DiversityMaize::DivExperiment

Corresponds to the table "div_experiment."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * div_experiment_id

=item * name

=item * design

=item * originator

=item * comments

=back



Has Many:

=over 4

=item * div_obs_units => Gramene::CDBI::DiversityMaize::DivObsUnit

=item * qtl_studies => Gramene::CDBI::DiversityMaize::QtlStudy

=back


=head2 Gramene::CDBI::DiversityMaize::DivGeneration

Corresponds to the table "div_generation."  

Stringifies (C<to_name>) to "value".

Fields:

=over 4

=item * div_generation_id

=item * value

=item * icis_id

=item * comments

=item * selfing_number

=item * sibbing_number

=back



Has Many:

=over 4

=item * div_stocks => Gramene::CDBI::DiversityMaize::DivStock

=back


=head2 Gramene::CDBI::DiversityMaize::DivLocality

Corresponds to the table "div_locality."  

Stringifies (C<to_name>) to "elevation".

Fields:

=over 4

=item * div_locality_id

=item * elevation

=item * city

=item * country

=item * origcty

=item * latitude

=item * longitude

=item * locality_name

=item * state_province

=back



Has Many:

=over 4

=item * div_accession_collectings => Gramene::CDBI::DiversityMaize::DivAccessionCollecting

=item * div_obs_units => Gramene::CDBI::DiversityMaize::DivObsUnit

=back


=head2 Gramene::CDBI::DiversityMaize::DivObsUnit

Corresponds to the table "div_obs_unit."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * div_obs_unit_id

=item * div_experiment_id

=item * div_stock_id

=item * div_locality_id

=item * name

=item * coord_x

=item * coord_y

=item * rep

=item * block

=item * plot

=item * season

=item * plant

=item * planting_date

=item * harvest_date

=item * comments

=back


Has A:

=over 4

=item * div_experiment => Gramene::CDBI::DiversityMaize::DivExperiment

=item * div_locality => Gramene::CDBI::DiversityMaize::DivLocality

=item * div_stock => Gramene::CDBI::DiversityMaize::DivStock

=back


Has Many:

=over 4

=item * div_obs_unit_samples => Gramene::CDBI::DiversityMaize::DivObsUnitSample

=item * div_traits => Gramene::CDBI::DiversityMaize::DivTrait

=item * div_treatments => Gramene::CDBI::DiversityMaize::DivTreatment

=back


=head2 Gramene::CDBI::DiversityMaize::DivObsUnitSample

Corresponds to the table "div_obs_unit_sample."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * div_obs_unit_sample_id

=item * div_obs_unit_id

=item * name

=item * sample_date

=item * producer

=item * comments

=back


Has A:

=over 4

=item * div_obs_unit => Gramene::CDBI::DiversityMaize::DivObsUnit

=back


Has Many:

=over 4

=item * div_alleles => Gramene::CDBI::DiversityMaize::DivAllele

=back


=head2 Gramene::CDBI::DiversityMaize::DivPassport

Corresponds to the table "div_passport."  

Stringifies (C<to_name>) to "accename".

Fields:

=over 4

=item * div_passport_id

=item * div_taxonomy_id

=item * div_accession_collecting_id

=item * cdv_source_id

=item * accename

=item * germplasm_type

=item * source

=item * accenumb

=item * sampstat

=item * comments

=back


Has A:

=over 4

=item * cdv_source => Gramene::CDBI::DiversityMaize::CdvSource

=item * div_accession_collecting => Gramene::CDBI::DiversityMaize::DivAccessionCollecting

=item * div_taxonomy => Gramene::CDBI::DiversityMaize::DivTaxonomy

=back


Has Many:

=over 4

=item * cdv_passport_sets => Gramene::CDBI::DiversityMaize::CdvPassportSet

=item * div_stocks => Gramene::CDBI::DiversityMaize::DivStock

=item * div_synonyms => Gramene::CDBI::DiversityMaize::DivSynonym

=back


=head2 Gramene::CDBI::DiversityMaize::DivPolyType

Corresponds to the table "div_poly_type."  

Stringifies (C<to_name>) to "poly_type".

Fields:

=over 4

=item * div_poly_type_id

=item * poly_type

=back



Has Many:

=over 4

=item * div_allele_assays => Gramene::CDBI::DiversityMaize::DivAlleleAssay

=back


=head2 Gramene::CDBI::DiversityMaize::DivScoringTechType

Corresponds to the table "div_scoring_tech_type."  

Stringifies (C<to_name>) to "scoring_tech_group".

Fields:

=over 4

=item * div_scoring_tech_type_id

=item * scoring_tech_group

=item * scoring_tech_type

=back



Has Many:

=over 4

=item * div_allele_assays => Gramene::CDBI::DiversityMaize::DivAlleleAssay

=back


=head2 Gramene::CDBI::DiversityMaize::DivStatisticType

Corresponds to the table "div_statistic_type."  

Stringifies (C<to_name>) to "stat_type".

Fields:

=over 4

=item * div_statistic_type_id

=item * stat_type

=back



Has Many:

=over 4

=item * div_traits => Gramene::CDBI::DiversityMaize::DivTrait

=item * qtl_traits => Gramene::CDBI::DiversityMaize::QtlTrait

=back


=head2 Gramene::CDBI::DiversityMaize::DivStock

Corresponds to the table "div_stock."  

Stringifies (C<to_name>) to "seed_lot".

Fields:

=over 4

=item * div_stock_id

=item * div_generation_id

=item * div_passport_id

=item * seed_lot

=item * stock_source

=item * comments

=back


Has A:

=over 4

=item * div_generation => Gramene::CDBI::DiversityMaize::DivGeneration

=item * div_passport => Gramene::CDBI::DiversityMaize::DivPassport

=back


Has Many:

=over 4

=item * cdv_markers => Gramene::CDBI::DiversityMaize::CdvMarker

=item * div_allele_assays => Gramene::CDBI::DiversityMaize::DivAlleleAssay

=item * div_obs_units => Gramene::CDBI::DiversityMaize::DivObsUnit

=item * div_stock_parents_div_stock => Gramene::CDBI::DiversityMaize::DivStockParent

=item * div_stock_parents_div_parent => Gramene::CDBI::DiversityMaize::DivStockParent

=back


=head2 Gramene::CDBI::DiversityMaize::DivStockParent

Corresponds to the table "div_stock_parent."  

Stringifies (C<to_name>) to "role".

Fields:

=over 4

=item * div_stock_parent_id

=item * div_stock_id

=item * div_parent_id

=item * role

=item * recurrent

=back


Has A:

=over 4

=item * div_stock => Gramene::CDBI::DiversityMaize::DivStock

=item * div_parent => Gramene::CDBI::DiversityMaize::DivStock

=back


Has Many:

=over 4

=item * qtl_analysis_methods => Gramene::CDBI::DiversityMaize::QtlAnalysisMethod

=back


=head2 Gramene::CDBI::DiversityMaize::DivSynonym

Corresponds to the table "div_synonym."  

Stringifies (C<to_name>) to "synonym".

Fields:

=over 4

=item * div_synonym_id

=item * div_passport_id

=item * synonym

=item * comments

=back


Has A:

=over 4

=item * div_passport => Gramene::CDBI::DiversityMaize::DivPassport

=back



=head2 Gramene::CDBI::DiversityMaize::DivTaxonomy

Corresponds to the table "div_taxonomy."  

Stringifies (C<to_name>) to "genus".

Fields:

=over 4

=item * div_taxonomy_id

=item * genus

=item * species

=item * subspecies

=item * subtaxa

=item * race

=item * population

=item * common_name

=item * term_accession

=back



Has Many:

=over 4

=item * div_passports => Gramene::CDBI::DiversityMaize::DivPassport

=back


=head2 Gramene::CDBI::DiversityMaize::DivTrait

Corresponds to the table "div_trait."  

Stringifies (C<to_name>) to "value".

Fields:

=over 4

=item * div_trait_id

=item * div_trait_uom_id

=item * div_statistic_type_id

=item * div_obs_unit_id

=item * value

=item * date_measured

=back


Has A:

=over 4

=item * div_obs_unit => Gramene::CDBI::DiversityMaize::DivObsUnit

=item * div_statistic_type => Gramene::CDBI::DiversityMaize::DivStatisticType

=item * div_trait_uom => Gramene::CDBI::DiversityMaize::DivTraitUom

=back



=head2 Gramene::CDBI::DiversityMaize::DivTraitUom

Corresponds to the table "div_trait_uom."  

Stringifies (C<to_name>) to "local_trait_name".

Fields:

=over 4

=item * div_trait_uom_id

=item * div_unit_of_measure_id

=item * local_trait_name

=item * trait_protocol

=item * to_accession

=item * eo_accession

=back


Has A:

=over 4

=item * div_unit_of_measure => Gramene::CDBI::DiversityMaize::DivUnitOfMeasure

=back


Has Many:

=over 4

=item * div_traits => Gramene::CDBI::DiversityMaize::DivTrait

=item * qtl_analysis_results => Gramene::CDBI::DiversityMaize::QtlAnalysisResult

=item * qtl_traits => Gramene::CDBI::DiversityMaize::QtlTrait

=back


=head2 Gramene::CDBI::DiversityMaize::DivTreatment

Corresponds to the table "div_treatment."  

Stringifies (C<to_name>) to "value".

Fields:

=over 4

=item * div_treatment_id

=item * div_treatment_uom_id

=item * div_obs_unit_id

=item * value

=back


Has A:

=over 4

=item * div_obs_unit => Gramene::CDBI::DiversityMaize::DivObsUnit

=item * div_treatment_uom => Gramene::CDBI::DiversityMaize::DivTreatmentUom

=back



=head2 Gramene::CDBI::DiversityMaize::DivTreatmentUom

Corresponds to the table "div_treatment_uom."  

Stringifies (C<to_name>) to "local_treatment_name".

Fields:

=over 4

=item * div_treatment_uom_id

=item * div_unit_of_measure_id

=item * local_treatment_name

=item * treatment_protocol

=item * eo_accession

=back


Has A:

=over 4

=item * div_unit_of_measure => Gramene::CDBI::DiversityMaize::DivUnitOfMeasure

=back


Has Many:

=over 4

=item * div_treatments => Gramene::CDBI::DiversityMaize::DivTreatment

=back


=head2 Gramene::CDBI::DiversityMaize::DivUnitOfMeasure

Corresponds to the table "div_unit_of_measure."  

Stringifies (C<to_name>) to "unit_type".

Fields:

=over 4

=item * div_unit_of_measure_id

=item * unit_type

=back



Has Many:

=over 4

=item * div_trait_uoms => Gramene::CDBI::DiversityMaize::DivTraitUom

=item * div_treatment_uoms => Gramene::CDBI::DiversityMaize::DivTreatmentUom

=back


=head2 Gramene::CDBI::DiversityMaize::GdpdmSearch

Corresponds to the table "gdpdm_search."  

Stringifies (C<to_name>) to "table_name".

Fields:

=over 4

=item * id

=item * table_name

=item * record_id

=item * record_text

=back




=head2 Gramene::CDBI::DiversityMaize::QtlAnalysisMethod

Corresponds to the table "qtl_analysis_method."  

Stringifies (C<to_name>) to "threshold_method".

Fields:

=over 4

=item * qtl_analysis_method_id

=item * qtl_study_id

=item * threshold_method

=item * permutation_number

=item * statistical_model

=item * positive_parent_stock_id

=item * comments

=back


Has A:

=over 4

=item * positive_parent_stock => Gramene::CDBI::DiversityMaize::DivStockParent

=item * qtl_study => Gramene::CDBI::DiversityMaize::QtlStudy

=back


Has Many:

=over 4

=item * qtl_analysis_results => Gramene::CDBI::DiversityMaize::QtlAnalysisResult

=back


=head2 Gramene::CDBI::DiversityMaize::QtlAnalysisResult

Corresponds to the table "qtl_analysis_result."  

Stringifies (C<to_name>) to "r_square".

Fields:

=over 4

=item * qtl_analysis_result_id

=item * qtl_obs_unit_id

=item * qtl_analysis_method_id

=item * div_trait_uom_id

=item * r_square

=item * genetic_var

=item * gxe_var

=item * environment_var

=back


Has A:

=over 4

=item * div_trait_uom => Gramene::CDBI::DiversityMaize::DivTraitUom

=item * qtl_analysis_method => Gramene::CDBI::DiversityMaize::QtlAnalysisMethod

=item * qtl_obs_unit => Gramene::CDBI::DiversityMaize::QtlObsUnit

=back


Has Many:

=over 4

=item * qtl_qtls => Gramene::CDBI::DiversityMaize::QtlQtl

=back


=head2 Gramene::CDBI::DiversityMaize::QtlMarker

Corresponds to the table "qtl_marker."  

Stringifies (C<to_name>) to "genetic_relationship".

Fields:

=over 4

=item * qtl_marker_id

=item * qtl_qtl_id

=item * cdv_marker_id

=item * genetic_relationship

=item * comments

=back


Has A:

=over 4

=item * cdv_marker => Gramene::CDBI::DiversityMaize::CdvMarker

=item * qtl_qtl => Gramene::CDBI::DiversityMaize::QtlQtl

=back



=head2 Gramene::CDBI::DiversityMaize::QtlObsUnit

Corresponds to the table "qtl_obs_unit."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * qtl_obs_unit_id

=item * qtl_study_id

=item * name

=item * comments

=back


Has A:

=over 4

=item * qtl_study => Gramene::CDBI::DiversityMaize::QtlStudy

=back


Has Many:

=over 4

=item * qtl_analysis_results => Gramene::CDBI::DiversityMaize::QtlAnalysisResult

=item * qtl_obs_unit_sets => Gramene::CDBI::DiversityMaize::QtlObsUnitSet

=item * qtl_traits => Gramene::CDBI::DiversityMaize::QtlTrait

=back


=head2 Gramene::CDBI::DiversityMaize::QtlObsUnitSet

Corresponds to the table "qtl_obs_unit_set."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * qtl_obs_unit_set_id

=item * qtl_obs_unit_id

=item * name

=item * comments

=back


Has A:

=over 4

=item * qtl_obs_unit => Gramene::CDBI::DiversityMaize::QtlObsUnit

=back



=head2 Gramene::CDBI::DiversityMaize::QtlQtl

Corresponds to the table "qtl_qtl."  

Stringifies (C<to_name>) to "published_symbol".

Fields:

=over 4

=item * qtl_qtl_id

=item * qtl_analysis_result_id

=item * qtl_accession_id

=item * published_symbol

=item * linkage_group

=item * start_position

=item * peak_position

=item * stop_position

=item * additive

=item * dominance

=item * lod

=item * f_value

=item * t_value

=item * p_value

=item * r_square

=item * comments

=back


Has A:

=over 4

=item * qtl_analysis_result => Gramene::CDBI::DiversityMaize::QtlAnalysisResult

=back


Has Many:

=over 4

=item * qtl_markers => Gramene::CDBI::DiversityMaize::QtlMarker

=back


=head2 Gramene::CDBI::DiversityMaize::QtlStudy

Corresponds to the table "qtl_study."  

Stringifies (C<to_name>) to "name".

Fields:

=over 4

=item * qtl_study_id

=item * name

=item * reference_id

=item * div_experiment_id

=item * cmap_map_set_acc

=back


Has A:

=over 4

=item * div_experiment => Gramene::CDBI::DiversityMaize::DivExperiment

=back


Has Many:

=over 4

=item * qtl_analysis_methods => Gramene::CDBI::DiversityMaize::QtlAnalysisMethod

=item * qtl_obs_units => Gramene::CDBI::DiversityMaize::QtlObsUnit

=back


=head2 Gramene::CDBI::DiversityMaize::QtlTrait

Corresponds to the table "qtl_trait."  

Stringifies (C<to_name>) to "date_measured".

Fields:

=over 4

=item * qtl_trait_id

=item * div_trait_uom_id

=item * div_statistic_type_id

=item * qtl_obs_unit_id

=item * date_measured

=item * value

=back


Has A:

=over 4

=item * div_statistic_type => Gramene::CDBI::DiversityMaize::DivStatisticType

=item * div_trait_uom => Gramene::CDBI::DiversityMaize::DivTraitUom

=item * qtl_obs_unit => Gramene::CDBI::DiversityMaize::QtlObsUnit

=back



=head2 Gramene::CDBI::DiversityMaize::Xref

Corresponds to the table "xref."  

Stringifies (C<to_name>) to "table_name".

Fields:

=over 4

=item * xref_id

=item * table_name

=item * record_id

=item * xref_type_id

=item * xref_value

=item * comments

=back


Has A:

=over 4

=item * xref_type => Gramene::CDBI::DiversityMaize::XrefType

=back



=head2 Gramene::CDBI::DiversityMaize::XrefType

Corresponds to the table "xref_type."  

Stringifies (C<to_name>) to "xref_type".

Fields:

=over 4

=item * xref_type_id

=item * xref_type

=item * url_template

=back



Has Many:

=over 4

=item * xrefs => Gramene::CDBI::DiversityMaize::Xref

=back


=head1 SEE ALSO

Gramene::CDBI.

=head1 AUTHOR

Generated by Ken Youens-Clark on Mon Mar 26 15:21:32 2007.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
