package Gramene::CDBI;

# vim: set ts=4 sw=4 et:

# $Id: CDBI.pm,v 1.4 2007/03/20 21:48:40 kclark Exp $

=head1 NAME

Gramene::CDBI - Base class for Gramene::CDBI classes

=head1 SYNOPSIS

  package Gramene::CDBI::Foo;

  use base( Gramene::CDBI );

=head1 DESCRIPTION

This is a convenient base class for Class::DBI-based objects for
the various Gramene modules (e.g., markers, QTL).

=head1 METHODS

=cut

# ----------------------------------------------------

use strict;

use base qw( Class::DBI::Sweet );

use Carp qw( croak confess );
use Class::DBI;
use Class::DBI::AsForm;
use Class::DBI::FromForm;
use Class::DBI::Plugin::AbstractCount;
use Class::DBI::Plugin::RetrieveAll;
use Class::DBI::Pager;
use Graph::Directed;
use Readonly;

our $VERSION = sprintf '%d.%02d', qq$Revision: 1.4 $ =~ /(\d+)\.(\d+)/;

Readonly my $COLON => q{:};
Readonly my $COMMA => q{,};

# ------------------------------------------------
sub represented_tables {

=pod

=head2 represented_tables

  my @tables = $cdbi->represented_tables;

Derived classes should implement this method.  It should return an
array of the table names present in the schema.

=cut

    my $self  = shift;
    my $class = ref $self;
    confess "The class '$class' hasn't implemented 'represented_tables'!";
}


# ------------------------------------------------
sub as_graph {

=pod

=head2 as_graph

Returns the schema represented as a Graph::Directed object where the
tables are vertices and their relationships are the edges.  The tables
must be listed in the L<represented_tables> method.

=cut

    my $self = shift;
    my $g    = Graph::Directed->new;
    
    for my $table ( $self->represented_tables ) {
        my $class = $self->table_name_to_class( $table );
        my $has_a = $class->meta_info('has_a') || {};
        
        # Check foreign key references.
        if ( %$has_a ) {
            while ( my ( $fk, $fk_info ) = each %$has_a ) {
                my $fk_class = $fk_info->{'foreign_class'};
                $g->add_edge( $table, $fk_class->table );
            }
        }
        
        if ( !$g->has_vertex( $table ) ) {
            $g->add_vertex( $table );
        }
    }   
        
    return $g;
}       

# ------------------------------------------------
sub get_path {
    my ( $self, $from, $to ) = @_;

    if ( $from && $to ) {
        if ( $from eq $to ) {
            croak "get_path 'from' and 'to' tables are the same ($from)!";
        }

        my $graph = $self->as_graph;

        for my $table ( $from, $to ) {
            if ( !$graph->has_vertex( $table ) ) {
                croak "Invalid table ($table)";
            }
        }

        # make edges bidirectional
        my @edges = $graph->edges;
        for my $edge ( @edges ) {
            $graph->add_edge( reverse @$edge );
        }

        my $these_tables = join( $COLON, sort $from, $to );

        my %defined_path;
        my $config = $self->config;
        while ( my ( $key, $val ) = each %{ $config->{'path'} } ) {
            my ( $t1, $t2 ) = sort split $COLON, $key;
            $defined_path{ join $COLON, $t1, $t2 } = [ split $COMMA, $val ];
        }

        my @path;
        if ( my $path = $defined_path{ $these_tables } ) {
            if ( $path->[0] eq $from ) {
                @path = @{ $path };
            }
            else {
                @path = reverse @{ $path };
            }
        }
        else {
            @path = $graph->SP_Dijkstra( $from, $to );
        }

        return @path;
    }
    else {
        croak 'Need both "to" and "from" tables for "get_path"';
    }
}

# ------------------------------------------------
sub get_related {
    my $self       = shift;
    my $dest_table = shift or croak 'No destination table';
    my $opts_ref   = shift || {};
    my $this_table = $self->table;
    my @path       = $self->get_path( $this_table, $dest_table );

    if ( @path ) {
        my $start_table    = $path[0];
        my $start_class    = $self->table_name_to_class( $start_table );
        my $dest_class     = $self->table_name_to_class( $dest_table );
        my $start_pk_field = $start_class->columns('Primary');
        my $dest_pk_field  = $dest_class->columns('Primary');
        my $sql
            = "select distinct $dest_table.$dest_pk_field from "
            . join( ', ', @path )
            . " where $start_table.$start_pk_field="
            . $self->id;

        my @joins;
        for my $table_num ( 0 .. $#path - 1 ) {
            my $t1                  = $path[ $table_num ];
            my $t2                  = $path[ $table_num + 1 ];
            my $class               = $self->table_name_to_class( $t1 );
            my ( $t1_fld, $t2_fld ) = $class->path_to( $t2 );
            push @joins, "$t1.$t1_fld=$t2.$t2_fld";
        }

        if ( @joins ) {
            $sql .= join(' ', map { " and $_ " } @joins);
        }

        my $db  = $self->db_Main;
        my $ids = $db->selectcol_arrayref( $sql );
        my @data;
        for my $id ( @$ids ) {
            push @data, $dest_class->retrieve( $id );
        }

        if ( @data ) {
            return wantarray ? @data : $data[0];
        }
        else {
            return wantarray ? () : undef;
        }

        if ( @data ) {
            return wantarray ? @data : $data[0];
        }
        else {
            return wantarray ? () : undef;
        }
    }
    else {
        my $this_table = $self->table;
        croak "No path from $this_table to $dest_table";
    }

    return;
}

# ------------------------------------------------
sub _uniq {
# Internal subroutine for uniquing objects returned by _extract

    my %h;
    map { $h{ $_->id() }++ == 0 ? $_ : () } @_;
}

# ------------------------------------------------
sub _extract {
# Internal subroutine called recursively by "get_related" to 
# drill into the schema to get objects

    my ( $from, $path ) = @_;

    return if !defined $from;

    my $next = shift @$path or return;

    return if !$from->can( $next );

    my @return;
    for my $object ( $from->$next() ) {
        next if !defined $object;

        if ( @$path ) {
            push @return, _extract( $object, $path );
        }
        else {
            push @return, $object;
        };
    }

    return @return;
}

# ------------------------------------------------
sub object_type {

=pod

=head2 object_type

  my $type = $cdbi->object_type;

Prints a prettied up version of the table name.  Splits on underscores
and then uppercases the first letter of each resulting word, then
joins them on spaces, so "marker_type" would print as "Marker Type."

=cut

    my $self  = shift;
    my $table = $self->table;
    return join ' ', map { ucfirst $_ } split /_/, $table;
}

1;

# --------------------------------------------------------

=pod

=head1 SEE ALSO

Class::DBI, Graph.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
