package Gramene::Apache::TipGenerator;

# $Id: TipGenerator.pm,v 1.5 2005/12/02 20:24:44 kclark Exp $

=head1 NAME

Gramene::Apache::TipGenerator - Generate tips for web pages

=head1 SYNOPSIS

  Gramene::Apache::TipGenerator::random_tip();

=head1 DESCRIPTION

This is a simple module for working with a "tips" database.  Currently
only a "random_tip" function is defined, but another could be added to
show all the tips, a subset of tips, etc.

The tip database itself is a plain text file formatted like the
popular "fortune" file (records delimited by "%\n").  The location of
this file is in a "<tips>" section of the "gramene.conf" file.

To remove a tip from the rotation but to keep it available for the 
archive ("all_tips"), place a hash ("#") at the beginning of the tip
to comment it out.

=head1 SEE ALSO

Gramene::Config.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2004 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut

# ----------------------------------------------------

use strict;
use Gramene::Config;
use List::Util qw( shuffle );
use Data::Dumper;

use vars qw( $VERSION @Tips $Last_Tip_Num );

$VERSION = sprintf "%d.%02d", q$Revision: 1.5 $ =~ /(\d+)\.(\d+)/;

BEGIN {
    $Last_Tip_Num = 0;
};

# ----------------------------------------------------
sub load_tips {
    return 1 if @Tips;

    my $cfile     = Gramene::Config->new;
    my $conf      = $cfile->get('tips');
    my $tips_file = $conf->{'file'} or 
                    die "No tips file defined\n";

    if ( -e $tips_file ) {
        open my $fh, $tips_file or die "Can't read '$tips_file': $!\n";
        local $/ = "%\n";
        TIP:
        while ( my $line = <$fh> ) {
            chomp $line;
            next TIP if $line =~ /^\s*#/; # commented out
            push @Tips, $line;
        }
        chomp @Tips;
        close $fh;
        @Tips = shuffle @Tips;
    }
    else {
        die "Tips file '$tips_file' doesn't exist.\n";
    }

    die "No tips in '$tips_file'?\n" if scalar @Tips == 0;

    return @Tips;
}

# ----------------------------------------------------
sub random_tip {
    eval { load_tips(); };

    if ( @$ ) {
        return "Error loading tips: $@\n";
    }

    if ( $Last_Tip_Num > $#Tips ) {
        $Last_Tip_Num = 0;
    }

    return $Tips[ $Last_Tip_Num++ ];
}

# ----------------------------------------------------
sub all_tips {
    load_tips();
    return join "\n", map { "<p>$_</p>" } @Tips;
}

1;
