# POD documentation - main docs before the code

=head1 NAME

Bio::SeqIO::ncbitrace - Creates SeqIO stream from 'unigene' files

=head1 SYNOPSIS

Do not use this module directly.  Use it via the Bio::SeqIO class.

=head1 SYNOPSYS

  use Bio::SeqIO;
  use Bio::FeatureIO;
  use Bio::ClusterIO;

  my $seq_io = Bio::SeqIO->new( -format => 'fasta',
                                -file   => 'Os.seq.uniq' );

  my $cluster_io = Bio::ClusterIO->new( -format => 'unigene',
                                        -file   => 'Os.data.sp' );

  my $unigene_io = Bio::SeqIO->new( -format    => 'unigene',
                                    -seqio     => $seq_io,
                                    -clusterio => $cluster_io );

  while( my $seq = $unigene_io->next_seq{
    ...
  }

=head1 DESCRIPTION

This object can create Bio::Seq objects from files downloaded from the
NCBI UniGene FTP site. The data is split across the 'data' file
containing the cluster members, and the 'seq.uniq' file, containing
the consensus sequence for the cluster. Each returned sequence
represents a merge of data from the two files.

The source data files for UniGene clusters can be downloaded from here;
ftp://ftp.ncbi.nih.gov/repository/UniGene

=head1 AUTHORS - William Spooner

Email: whs@ebi.ac.uk

=head1 APPENDIX

The rest of the documentation details each of the object
methods. Internal methods are usually preceded with a _

=cut

# Let the code begin...

package Bio::SeqIO::ncbitrace;
use vars qw(@ISA);
use strict;
use Bio::SeqIO;
use Bio::ClusterIO;
use Bio::Species;

# Object preamble - inherits from Bio::SeqIO
@ISA = qw(Bio::SeqIO);

# Chained constructor
sub _initialize {
  my($self,@args) = @_;

  $self->SUPER::_initialize(@args);

  my %params = @args;
  @params{ map { lc $_ } keys %params } = values %params; # lowercase keys

  my @streams = qw( seq cluster );

  # Validate streams
  my %io_streams;
  foreach my $stream( @streams ){
    my $io = $params{"-".$stream} || next;
    unless( UNIVERSAL::isa($io, "Bio::Root::IO") ){
      $self->throw( "-$stream must be Bio::Root::IO compliant" );
    }
    $io_streams{$stream} = $io;
  }
  $self->{_streams} = {%io_streams};
}

=head2 next_seq

 Title   : next_seq
 Usage   : $seq = $stream->next_seq()
 Function: returns the next sequence in the stream
 Returns : Bio::Seq object
 Args    : NONE

=cut

sub next_seq {
  my $self = shift;
  my $seq;
  if( $self->has_stream('seqio') ){
    $seq = $self->next_obj('seqio');
  }

  my $cluster;
  if( $self->has_stream('clusterio') ){
    $cluster = $self->next_obj('clusterio');
  }

  return $seq;
}
