# POD documentation - main docs before the code

=head1 NAME

Bio::SeqIO::traceinfo - Creates FeatureIO stream from 'trace info' files

=head1 SYNOPSIS

Do not use this module directly.  Use it via the Bio::FeatureIO class.

=head1 DESCRIPTION

This object can transform Bio::Feature objects to and from 'info' files
like those downloaded from the NCBI Trace Archive.

=head1 AUTHORS - William Spooner

Email: whs@ebi.ac.uk

=head1 APPENDIX

The rest of the documentation details each of the object
methods. Internal methods are usually preceded with a _

=cut

# Let the code begin...

package Bio::FeatureIO::traceinfo;
use vars qw(@ISA);
use strict;
use Bio::FeatureIO;
use Bio::SeqFeature::Generic;

# Object preamble - inherits from Bio::FeatureIO

@ISA = qw(Bio::FeatureIO);

BEGIN{
  # See the following page for a description of trace annotaton fields;
  # http://www.ncbi.nlm.nih.gov/Traces/trace.cgi?cmd=show&f=rfc_b&m=doc&s=rfc_b
  # NCBI supply a matrix that specifies which fields are 
  # required/optional/nonsense for each trace_type/strategy combination,
  # but that is a task for later...
  use vars qw( @REQUIRED_FIELDS @CONDITIONAL_REQUIRED_FIELDS @OPTIONAL_FILEDS 
               @INTERNAL_FIELDS @SUBMISSION_FIELDS %ALL_FIELDS );

  my @REQUIRED_FIELDS = qw
      (
       CENTER_NAME           CENTER_PROJECT        PROGRAM_ID 
       SOURCE_TYPE           SPECIES_CODE          STRATEGY       
       SUBMISSION_TYPE       TRACE_FORMAT          TRACE_NAME     
       TRACE_TYPE_CODE
       );
  my @CONDITIONAL_REQUIRED_FIELDS = qw
      (
       AMPLIFICATION_FORWARD AMPLIFICATION_REVERSE CLIP_VECTOR_LEFT      
       CLIP_VECTOR_RIGHT     CLONE_ID              CLONE_ID_LIST 
       COLLECTION_DATE       ENVIRONMENT_TYPE      HOST_ID      
       HOST_LOCATION         HOST_SPECIES          INSERT_FLANK_LEFT 
       INSERT_FLANK_RIGHT    INSERT_SIZE           INSERT_STDEV 
       LATITUDE              LIBRARY_ID            LONGITUDE     
       PRIMER_LIST           REFERENCE_ACCESSION   REFERENCE_ACC_MAX
       REFERENCE_ACC_MIN     REFERENCE_OFFSET      SEQ_LIB_ID       
       STRAIN                TRANSPOSON_ACC        TRANSPOSON_CODE
       );
  my @OPTIONAL_FIELDS = qw
      (
       ACCESSION             AMPLIFICATION_SIZE    ATTEMPT 
       BASE_FILE CHEMISTRY   CHEMISTRY_TYPE        CHROMOSOME 
       CLIP_QUALITY_LEFT     CLIP_QUALITY_RIGHT    CVECTOR_ACCESSION
       CVECTOR_CODE          DEPTH                 ELEVATION 
       GENE_NAME 
       HI_FILTER_SIZE        HOST_CONDITION        INDIVIDUAL_ID 
       LO_FILTER_SIZE        PEAK_FILE             PH
       PICK_GROUP_ID         PLACE_NAME            PLATE_ID 
       POPULATION_ID         PREP_GROUP_ID         PRIMER 
       PRIMER_CODE           PROJECT_NAME          
       RUN_DATE              RUN_GROUP_ID          RUN_LANE 
       RUN_MACHINE_ID        RUN_MACHINE_TYPE      SALINITY 
       SVECTOR_ACCESSION     SVECTOR_CODE          TEMPERATURE 
       TEMPLATE_ID           TRACE_END             WELL_ID 
       );
  my @INTERNAL_FIELDS = qw
      (
       BASES_20              BASES_40              BASES_60
       LOAD_DATE             TAXID                 TI
       MATE
       );
  @SUBMISSION_FIELDS = qw
      (
       FEATURE_ID_FILE       FEATURE_ID_FILE_NAME  FEATURE_SIGNAL_FILE
       FEATURE_SIGNAL_FILE_NAME                    QUAL_FILE
       TRACE_FILE                              
       );

  my %ALL_FIELDS = map{$_=>1} ( @REQUIRED_FIELDS,
                                @CONDITIONAL_REQUIRED_FIELDS,
                                @OPTIONAL_FIELDS );
}

=pod

=cut

sub _initialize {
  my($self,@args) = @_;

  $self->SUPER::_initialize(@args);
}


=pod

=cut

sub next_feature {
  my $self = shift;
  
  my $feature;
  while( my $str = $self->_readline ){
    chomp($str);
    my( $key, $value ) = split( /\:\s+/, $str, 2 );
    $key || last;
    $feature ||= Bio::SeqFeature::Generic->new();
    if( $key eq 'trace_name' ){
      $feature->display_name($value);
    }
    $feature->add_tag_value( $key, $value );
  }
  return $feature;
}


=pod

=cut

sub write_feature{
  my $self = shift;
  my $fetaure = shift || $self->throw( "Need a feature" );
  unless( UNIVERSAL::isa('Bio::SeqFeatureI') ){
    $self->throw( "Feature must be a Bio::SeqFeatureI" );
  }
  my $line_t = '%s: %s\n';
  my $str = '';

  # TODO: validate that all required tags are populated,
  # and that no spurious tags are present
  foreach my $tag( $self->get_all_tags ){
    $str .= sprintf( $line_t, $_, $self->get_tag_values($_) );
  }
  $self->print("$str\n");
}

1;
