package Basset::Encryption::WeakKey;

#Basset::Encryption::WeakKey, copyright and (c) 2003 James A Thomason III
#Basset::Encryption::WeakKey is distributed under the terms of the Perl Artistic License.

=pod

=head1 NAME

Basset::Encryption::WeakKey - encryption, but not too terribly secure. Okay, screw that, I don't know
how secure the thing is, since I'm not a cryptographer. I have a wonderful, interesting, huge book
on codes and cyphers on my bookshelf collecting dust. I've barely even glanced in it. I really should.
Ho hum.

=head1 AUTHOR

Jim Thomason, jim@jimandkoka.com

=head1 DESCRIPTION

Anyway, in my ever going quest to re-invent my own version of things, I came up with this. I'd
previously written Basset::Encryption::Weak, but that was very very weak. And non-keyed. It was 
just designed to make something tough to read, but not actually really truely hide something.
If you had the code, after all, you could just run decrypt on it and be done.

So I decided to write WeakKey here. Beefier encryption (though I don't know how beefy) and keyed,
so you actually need to know how to open it up.

Obviously still easily victim to a brute force attack, so choose your keys wisely.

=cut


$VERSION = '1.00';

use Basset::Object;
our @ISA = Basset::Object->pkg_for_type('object');

use strict;
use warnings;

use Basset::Encryption::ProgressiveRotational;

my @character_map = ('a'..'z', ' ', 'A'..'Z', '0'..'9', qw[? !]);
my %character_map;
@character_map{@character_map} = (0..$#character_map);

__PACKAGE__->add_class_attr('default_key');

#does what it sounds like, takes a number, and returns its factors.
#Basset::Encryption::WeakKey->_factor(18); #returns 1, 2, 3, 6, 9, 18

=pod

=head1 METHODS

=cut

=pod

=over

=cut

#=pod
#
#=item _factor
#
#does what it sounds like, takes a number, and returns its factors. Used internally
#
# Basset::Encryption::WeakKey->_factor(18); #returns 1, 2, 3, 6, 9, 18
#
#=cut

sub _factor {
	my $class	= shift;
	my $number	= shift;
	
	return $class->error("Cannot _factor w/o number", "BEWk-01") unless defined $number;
	
	return $class->error("Can only _factor positive numbers", "BEWk-02") unless $number > 0;
	
	my @factors = ();
	
	foreach my $i (1 .. $number){
		push @factors, $i if $number % $i == 0;
	};
	
	return @factors;
};

=pod

=begin btest _factor

$test->is(scalar(__PACKAGE__->_factor), undef, "Cannot factor w/o number");
$test->is(__PACKAGE__->errcode, "BEWk-01", "proper error code");
$test->is(scalar(__PACKAGE__->_factor(-1)), undef, "Cannot factor negative numbers");
$test->is(__PACKAGE__->errcode, "BEWk-02", "proper error code");
$test->is(scalar(__PACKAGE__->_factor(0)), undef, "Cannot factor zero");
$test->is(__PACKAGE__->errcode, "BEWk-02", "proper error code");

my @one_factors = __PACKAGE__->_factor(1);
$test->is(scalar(@one_factors), 1, "one has 1 factor");
$test->is($one_factors[0], 1, 'single factor is 1');

my @two_factors = __PACKAGE__->_factor(2);
$test->is(scalar(@two_factors), 2, "two has 2 factors");
$test->is($two_factors[0], 1, 'first factor is 1');
$test->is($two_factors[1], 2, 'second factor is 2');

my @ten_factors = __PACKAGE__->_factor(10);
$test->is(scalar(@ten_factors), 4, "ten has 4 factors");
$test->is($ten_factors[0], 1, 'first factor is 1');
$test->is($ten_factors[1], 2, 'second factor is 2');
$test->is($ten_factors[2], 5, 'third factor is 5');
$test->is($ten_factors[3], 10, 'fourth factor is 10');

=end btest

=cut

=item encrypt

Class method. Given a string, encrypts it.

 my $encrypted = Basset::Encryption::WeakKey->encrypt("Hello, world", "password");
 print $encrypted; #prints M5cLUCcZKbGtFEMzve?1
 
Note that there is a random component to the encryption, so your encrypted value may be
different.

You may encrypt without a key, if you choose. But I don't konw why you'd want to.

=cut

sub encrypt {
	my $class	= shift;
	my $string	= shift;
	my $key		= shift || $class->default_key || 'keyless';

	# first, we'll convert the key to binary
	my $binary_key	  = reverse unpack("B*", $key);

	# our binary sum is the sum of all digits in the binary key	
	my $binary_sum;
	$binary_sum += $_ for split(//, $binary_key);

	# our rotational sum is the sum of all digits in the binary key, where
	# each digits is multiplied by 2 and by its position.
	# so the first digit is * 2 * 1, the second is * 2 * 2, the third is * 2 * 3, etc.
	my $n = 1;
	my $rotational_sum = 0;
	foreach my $idx (split(//, $binary_key)) {
		$rotational_sum += $idx * 2 * $n++;
	};

	# Now we'll convert the rotational sum to binary
	my $binary_rotational_sum = unpack("B*", $rotational_sum);
	
	# and factor the rotational sum
	my @f = Basset::Encryption::WeakKey->_factor($rotational_sum);

	# We now encrypt the string using ProgressiveRotational, starting with a rotation
	# at the binary sum, and increasing at points defined by the factors of the
	# rotational sum
	$string = Basset::Encryption::ProgressiveRotational->encrypt($string, $binary_sum, [@f]);

	#convert the string to binary
	my $binary_string = unpack("B*", $string);
	
	#factor the binary_sum, and toss those numbers into a hash
	my %key_factors = map {$_, 1} Basset::Encryption::WeakKey->_factor($binary_sum);

	# now, if our binary string is longer than our binary key, then increase the
	# size of our binary key so we're sure it's greater than the length of our
	# binary_string
	if (length $binary_string > length $binary_key) {
		$binary_key .= $binary_key x (length($binary_string) - length($binary_key));
	};

	#and cut it back down to the size of the binary_string
	$binary_key = substr($binary_key, 0, length $binary_string);
	
	#split up the binary_string and binary_key into arrays.
	my @binary_string	= split(//, $binary_string);
	my @binary_key		= split(//, $binary_key);
	
	#set up an array to store our encrypted string
	my @new_string = ();
	
	# now iterate through our binary string
	foreach my $idx (0..$#binary_string) {

		# if we're at position of one of the factors of the binary_sum, then insert
		# a random number
		if ($key_factors{$idx % ($binary_sum + 1)}) {
			push @new_string, int rand(2);
		};
		#xor the string and the key, and toss it into the new string
		my $val = ($binary_string[$idx] + 0) ^ ($binary_key[$idx] + 0);
		push @new_string, $val;

	};
	# tack on a set of 01+ to bring the length of the binary string up to 
	# a multiple of 8
	push @new_string, 0, (1) x (@new_string % 8 + 7);

	# set the binary_string to the xor'ed string
	$binary_string = join('', @new_string);
	
	#set up another array to hold our encrypted string
	my @encrypted_string = ();
	
	#The binary bits is an array of 6 bit blocks from the binary_string
	my @binary_bits = $binary_string =~ /(.{1,6})/g;
	
	# iterate through the binary_bits, and convert them into integers
	# then replace that integer with the equivalent value from the character map
	# and toss that character onto the encrypted string
	foreach my $bit (@binary_bits){
		$bit .= "0" x (6 - length $bit);
		my $num = ord(pack("B*", '00' . $bit));

		push @encrypted_string, $character_map[$num];
	};

	# finally, return that encrypted string through another progressiverotational encryption with the
	# binary_sum over the @f array.
	return Basset::Encryption::ProgressiveRotational->encrypt(join('', @encrypted_string), $binary_sum, [@f]);

};

=pod

=begin btest encrypt

my $encrypted = __PACKAGE__->encrypt('string');
$test->ok($encrypted, "Got keyless encrypted string");
my $decrypted = __PACKAGE__->decrypt($encrypted);
$test->is($decrypted, 'string', 'decrypted string properly');

my $encrypted2 = __PACKAGE__->encrypt('string2', 'password');
$test->ok($encrypted2, "Got encrypted string with password");
my $decrypted2 = __PACKAGE__->decrypt($encrypted2, "password");
$test->is($decrypted2, 'string2', 'decrypted string properly with password');

my $encrypted3 = __PACKAGE__->encrypt('very long string' x 50, 'a');
$test->ok($encrypted3, "Got very long encrypted string with short password");
my $decrypted3 = __PACKAGE__->decrypt($encrypted3, 'a');
$test->is($decrypted3, 'very long string' x 50, "decrypted very long string with short password");

my $encrypted4 = __PACKAGE__->encrypt('a', 'very long string');
$test->ok($encrypted4, "Got very long encrypted string with short password");
my $decrypted4 = __PACKAGE__->decrypt($encrypted4, 'very long string');
$test->is($decrypted4, 'a', "decrypted very long string with short password");

=end btest

=cut

=pod

=item decrypt

Class method. Given a string, decrypts it.

 my $decrypted = Basset::Encryption::WeakKey->decrypt("M5cLUCuZAbGtFvTzve?1", "password");
 print $decrypted; #prints Hello, world

If you encrypted without a key, then you should decrypt without a key as well.

=cut

sub decrypt {

	my $class				= shift;
	my $encrypted_string	= shift;
	my $key					= shift || $class->default_key || 'keyless';

	# first, we'll convert the key to binary
	my $binary_key	  = reverse unpack("B*", $key);

	# our binary sum is the sum of all digits in the binary key	
	my $binary_sum;
	$binary_sum += $_ for split(//, $binary_key);

	# our rotational sum is the sum of all digits in the binary key, where
	# each digits is multiplied by 2 and by its position.
	# so the first digit is * 2 * 1, the second is * 2 * 2, the third is * 2 * 3, etc.
	my $n = 1;
	my $rotational_sum = 0;
	foreach my $idx (split(//, $binary_key)) {
		$rotational_sum += $idx * 2 * $n++;
	};
	
	# Now we'll convert the rotational sum to binary
	my $binary_rotational_sum = unpack("B*", $rotational_sum);
	
	# and factor the rotational sum
	my @f = Basset::Encryption::WeakKey->_factor($rotational_sum);
	
	#factor the binary_sum, and toss those numbers into a hash
	my %key_factors = map {$_, 1} Basset::Encryption::WeakKey->_factor($binary_sum);
	
	# run the passed string through a progressiverotational decrypt, starting at the binary_sum, and increasing
	# at points defined by the factors of the rotational sum
	$encrypted_string = Basset::Encryption::ProgressiveRotational->decrypt($encrypted_string, $binary_sum, [@f]);

	#set up an array to store the binary version of the encrypted_string
	my @binary_string = ();
	
	# iterate through each character in the encrypted string. Use the character map to convert
	# the character into a number, convert that number to binary, and then take the lower 6 bits.
	foreach my $char (split(//, $encrypted_string)) {
		my $num = $character_map{$char};
		my $bin = unpack("B*", pack("N", $num || 0 ));
		$bin = $1 if $bin =~ /(......)$/;
		push @binary_string, $bin;
	};
	
	#our binary_string is all the binary_bits in the binary_string array
	my $binary_string = join('', @binary_string);

	# We padded the end of the string with 01+, so get rid of it.
	# the decimal -> binary conversion sometimes tacks on extra 0s at the end, so get rid of those too.
	$binary_string =~ s/01+0*$//;

	# now, if our binary string is longer than our binary key, then increase the
	# size of our binary key so we're sure it's greater than the length of our
	# binary_string
	if (length $binary_string > length $binary_key) {
		$binary_key .= $binary_key x (length($binary_string) - length($binary_key));
	};

	#and cut it back down to the size of the binary_string
	$binary_key = substr($binary_key, 0, length $binary_string);
	
	#set up our character arrays for the string and the key
	@binary_string	= split(//, $binary_string);
	my @binary_key		= split(//, $binary_key);

	# set up an array to hold our new string
	my @new_string = ();

	# ahh, the tricky part. walk through the binary string and strip out the random
	# values. Then, with the remaining legitimate values, xor the string and the key and
	# push the result onto @new_string.
	my $num_hit		= 0;
	my $last_idx	= 0;
	foreach my $idx (0..$#binary_string) {
		if ($idx - $num_hit  > $last_idx && $key_factors{($idx - $num_hit) % ($binary_sum + 1)}) {
			$last_idx = $idx - $num_hit;		
			$num_hit++;
			next;
		};

		push @new_string, ($binary_string[$idx] + 0) ^ ($binary_key[$idx - $num_hit] + 0);
	};

	#convert the new string back from decimal
	my $string = pack("B*", join('', @new_string));

	# finally, run the  string through a progressiverotational decrypt, starting at the binary_sum, and increasing
	# at points defined by the factors of the rotational sum. and we're done.
	return Basset::Encryption::ProgressiveRotational->decrypt($string, $binary_sum, [@f]);

};

=pod

=begin btest decrypt

my $encrypted = __PACKAGE__->encrypt('string');
$test->ok($encrypted, "Got keyless encrypted string");
my $decrypted = __PACKAGE__->decrypt($encrypted);
$test->is($decrypted, 'string', 'decrypted string properly');

my $encrypted2 = __PACKAGE__->encrypt('string2', 'password');
$test->ok($encrypted2, "Got encrypted string with password");
my $decrypted2 = __PACKAGE__->decrypt($encrypted2, "password");
$test->is($decrypted2, 'string2', 'decrypted string properly with password');

my $encrypted3 = __PACKAGE__->encrypt('very long string' x 50, 'a');
$test->ok($encrypted3, "Got very long encrypted string with short password");
my $decrypted3 = __PACKAGE__->decrypt($encrypted3, 'a');
$test->is($decrypted3, 'very long string' x 50, "decrypted very long string with short password");

my $encrypted4 = __PACKAGE__->encrypt('a', 'very long string');
$test->ok($encrypted4, "Got very long encrypted string with short password");
my $decrypted4 = __PACKAGE__->decrypt($encrypted4, 'very long string');
$test->is($decrypted4, 'a', "decrypted very long string with short password");

=end btest

=cut

1;

=pod

=back

=cut
