package Basset::Encryption::ProgressiveRotational;

#Basset::Encryption::ProgressiveRotational, copyright and (c) 2003 James A Thomason III
#Basset::Encryption::ProgressiveRotational is distributed under the terms of the Perl Artistic License.

=pod

=head1 NAME

Basset::Encryption::ProgressiveRotational - fairly powerful encryption options. Assuming that the only
options you're considering are ones based around a rotational algorithm. In the trivial case, this
is trivial to crack. However, in the more extreme cases, you can come up with some reasonably intense
encrypted strings. But for christ's sake, if it's important, use a better tool!

=head1 AUTHOR

Jim Thomason, jim@jimandkoka.com

=head1 DESCRIPTION

I've written another, more powerful (B<much> more powerful) encryption module, but it was lacking
a little bit. Turned out that under certain conditions, if you guessed close to the correct key, a
substantial portion of the encrypted data would be decrypted. Not a good thing.

So I decided to put a little more oomph behind that module by adding a rotational component. I figured,
"Heck, it's not gonna do much, but it'll slow someone down a little bit." But I thought about it more
and more and decided that it just wasn't terribly secure to do that. I mean, someone's gonna figure
out right quick that it's a simple rotational cypher as the last step.

But I like rotational cyphers. They're fun. It's the first cypher I learned (probably the first one that
most people learned) so it has a certain soft spot for me. So I opted to keep the rotational base and
just beef it up a bit.

To that end, I added the option to make the rotation progressive. So you can specify where you want
to start rotating, and how much you want to increase the rotation on each character. So the first character
is rot 1, the second is rot 2, the third is rot 3, etc.  'Course you have a few more options than that,
it's not like it only encrypts that one way.

Secure? Naah, probably not. But it's fun.

=cut


$VERSION = '1.00';

use Basset::Object;
our @ISA = Basset::Object->pkg_for_type('object');

use strict;
use warnings;

=pod

=over

=item encrypt

encrypt is a method that accepts at least one and up to 3 arguments.

The first argument is the string to encrypt, the second is the value of the first rotation, and the third 
is the amount to increase the rotation. The second and third arguments each default to 1.

It does rotational encryption on four character sets, A-Z, a-z, 0-9, 0x20-0x2F (ascii).

The simplest thing you can do is a standard rot-1, non-incrementing encryption

 my $encrypted = Basset::Encryption::ProgressiveRotational->encrypt('abcABC123', 1, 0);
 print $encrypted; #prints bcdBCD234

well, you could also do encrypt('abcABC123', 0, 0) for no encryption, but that'd be silly. Note
that you need to pass it a 0 for the third arg, or it defaults to 1.
 
Not exactly secure or anything. It gets interesting when you give it the second argument, the
amount to increase the rotation.

 my $encrypted = Basset::Encryption::ProgressiveRotational->encrypt('abcABC123', 1, 1);
 print $encrypted; #prints bdf$FHJ913

Note that the first character is rot1, the second is rot2, the third is rot3, etc.

Alternatively, the third argument may be an array, to specify adding varying amounts on each pass

 my $encrypted = Basset::Encryption::ProgressiveRotational->encrypt('abcABC123', 1, [1, 2, 3]);
 print $encrypted; #prints bdg'ILP582

Here, the first character is rot 1, the second is rot 2, the third is rot 4, the fourth is rot 7,
the fifth is rot 8, and so on.

And that's it.

=cut

sub encrypt {
	my $class	= shift;
	my $string	= shift;
	
	my $rot		= @_ ? shift : 1;
	my $inc		= @_ ? shift : 1;
	
	$inc = [$inc] unless ref $inc;
	my $inc_idx = 0;
	
	my @new_string = ();
	
	foreach my $char (split(//, $string)) {
		if ($char =~ /[a-zA-Z0-9\x20-\x2F]/){ # !"#\$%&'()*+,\-.\/
			if ($char =~ /[a-z]/) {
				my $val = ord($char) + $rot;
				while ($val > 122) {
					$val -= 26;
				};
				$char = chr($val);
			}
			elsif ($char =~ /[A-Z]/) {
				my $val = ord($char) + $rot;
				while ($val > 90) {
					$val -= 26;
				};
				$char = chr($val);
			}
			elsif ($char =~ /[0-9]/) {
				my $val = ord($char) + $rot;
				while ($val > 57) {
					$val -= 10;
				};
				$char = chr($val);
			}
			else {# if ($char =~ /[\x20-\x2F]/) { is implied
				my $val = ord($char) + $rot;
				while ($val > 47) {
					$val -= 16;
				};
				$char = chr($val);
			}
			
			$rot += $inc->[$inc_idx++ % @$inc] % 26;
		};
		push @new_string, $char;
	};
	
	return join('', @new_string);
	
};

=pod

=begin btest encrypt

$test->is(__PACKAGE__->encrypt('a', 0, 0), 'a', "encrypted single letter w/o rotation");
$test->is(__PACKAGE__->encrypt('a', 1, 0), 'b', "encrypted single letter w/single rotation");
$test->is(__PACKAGE__->encrypt('a', 1, [0]), 'b', "encrypted single letter w/single rotation, arrayref");
$test->is(__PACKAGE__->encrypt('a'), 'b', "encrypted single letter w/default params");
$test->is(__PACKAGE__->encrypt('a', 2, 0), 'c', "encrypted single letter w/double rotation");
$test->is(__PACKAGE__->encrypt('a', 2, 1), 'c', "encrypted single letter w/double rotation increasing");
$test->is(__PACKAGE__->encrypt('abcdefghijklmnopqrstuvwxyz', 1, 0), 'bcdefghijklmnopqrstuvwxyza', "encrypted lower case, 1, 0");
$test->is(__PACKAGE__->encrypt('ABCDEFGHIJKLMNOPQRSTUVWXYZ', 1, 0), 'BCDEFGHIJKLMNOPQRSTUVWXYZA', "encrypted upper case, 1, 0");
$test->is(__PACKAGE__->encrypt('0123456789', 1, 0), '1234567890', "encrypted numeric, 1, 0");
$test->is(__PACKAGE__->encrypt('!"#\$%&\'()*+,-./\\ ', 1, 0), '"#$\%&\'()*+,-./ \\!', "encrypted punctuation, 1, 0");
$test->is(__PACKAGE__->encrypt('abcde', 1, 1), 'bdfhj', "encrypted string w/single rotation, linear increase");
$test->is(__PACKAGE__->encrypt('abcde', 1, [1, 2]), 'bdgil', "encrypted string w/single rotation, staggered increase");
$test->is(__PACKAGE__->encrypt('abcde', 1, [1, 2, 3]), 'bdgkm', "encrypted string w/single rotation, staggered increase");
$test->is(__PACKAGE__->encrypt("\x3F", 1, 0), "\x3F", "encrypting binary character is binary character");

=end btest

=cut



=pod

=item decrypt

operates the same way as encrypt, but takes an encrypted string and decrypts it. Technically, it
can also be used to encrypt an unencrypted string, which encrypt would then decrypt. i.e., they're
just inverse functions.

Give it the same arguments you used to encrypt, and it'll decrypt.

 my $decrypted = Basset::Encryption::ProgressiveRotational->decrypt("bdg'ILP582", 1, [1, 2, 3]);
 print $decrypted; # prints abcABC123

=back

=cut

sub decrypt {
	my $class	= shift;
	my $string	= shift;
	
	my $rot		= @_ ? shift : 1;
	my $inc		= @_ ? shift : 1;
	
	$inc = [$inc] unless ref $inc;
	my $inc_idx = 0;

	my @new_string = ();
	
	foreach my $char (split(//, $string)) {
		if ($char =~ /[a-zA-Z0-9\x20-\x2F]/){ # !"#\$%&'()*+,\-.\/
			if ($char =~ /[a-z]/) {
				my $val = ord($char) - $rot;
				while ($val < 97) {
					$val += 26;
				};
				$char = chr($val);
			}
			elsif ($char =~ /[A-Z]/) {
				my $val = ord($char) - $rot;
				while ($val < 65) {
					$val += 26;
				};
				$char = chr($val);
			}
			elsif ($char =~ /[0-9]/) {
				my $val = ord($char) - $rot;
				while ($val < 48) {
					$val += 10;
				};
				$char = chr($val);
			}
			else { #if ($char =~ /[\x20-\x2F]/) { is implied
				my $val = ord($char) - $rot;
				while ($val < 32) {
					$val += 16;
				};
				$char = chr($val);
			}
			
			$rot += $inc->[$inc_idx++ % @$inc] % 26;
		};
		push @new_string, $char;
	};
	
	return join('', @new_string);
	
};

=pod

=begin btest decrypt

$test->is(__PACKAGE__->decrypt('a', 0, 0), 'a', "decrypted single letter w/o rotation");
$test->is(__PACKAGE__->decrypt('b', 1, 0), 'a', "decrypted single letter w/single rotation");
$test->is(__PACKAGE__->decrypt('b', 1, [0]), 'a', "decrypted single letter w/single rotation, arrayref");
$test->is(__PACKAGE__->decrypt('b'), 'a', "decrypted single letter w/default params");
$test->is(__PACKAGE__->decrypt('c', 2, 0), 'a', "decrypted single letter w/double rotation");
$test->is(__PACKAGE__->decrypt('c', 2, 1), 'a', "decrypted single letter w/double rotation increasing");
$test->is(__PACKAGE__->decrypt('bcdefghijklmnopqrstuvwxyza', 1, 0), 'abcdefghijklmnopqrstuvwxyz', "decrypted lower case, 1, 0");
$test->is(__PACKAGE__->decrypt('BCDEFGHIJKLMNOPQRSTUVWXYZA', 1, 0), 'ABCDEFGHIJKLMNOPQRSTUVWXYZ', "decrypted upper case, 1, 0");
$test->is(__PACKAGE__->decrypt('1234567890', 1, 0), '0123456789', "decrypted numeric, 1, 0");
$test->is(__PACKAGE__->decrypt('"#$\%&\'()*+,-./ \\!', 1, 0), '!"#\$%&\'()*+,-./\\ ', "decrypted punctuation, 1, 0");
$test->is(__PACKAGE__->decrypt('bdfhj', 1, 1), 'abcde', "decrypted string w/single rotation, linear increase");
$test->is(__PACKAGE__->decrypt('bdgil', 1, [1, 2]), 'abcde', "decrypted string w/single rotation, staggered increase");
$test->is(__PACKAGE__->decrypt('bdgkm', 1, [1, 2, 3]), 'abcde', "decrypted string w/single rotation, staggered increase");
$test->is(__PACKAGE__->decrypt("\x3F", 1, 0), "\x3F", "decrypting binary character is binary character");

=end btest

=cut



1;

__END__

=pod

ta dah!

That's all there is to it. Useful? Ehh, maybe. With proper choosing of your starting value
and judicious use of seeming random rotational increases, it'd be pretty tough to crack.

I think.

'course I'm not really a cryptographer, so this could probably all be cracked by a retarded infant.

Have fun, and don't store your credit cards in here.

=cut
