package Basset::CGI;

#Basset::CGI, copyright and (c) 2003 James A Thomason III
#Basset::CGI is distributed under the terms of the Perl Artistic License.

=pod

=head1 NAME

Basset::CGI - used to handle web requests

=head1 AUTHOR

Jim Thomason, jim@jimandkoka.com

=head1 DESCRIPTION

Basset::CGI currently does little more than serve as a gateway between CGI and the rest of the Basset framework.
Why does it do this, you ask? For a couple of reasons.

First of all, it makes CGI's interface consistent with everything else. You get the convenient error methods, 
syntax, conf file access, you name it.

Secondly, it's easy to subclass and replace (swapping out the cgi entry in your conf file) if you want to alter
functionality. You know that Basset::* objects won't be using functionality other than what's listed in here, so
if you want to subclass it and override param to use your own method, you may feel free. Much easier than
subclassing the entire CGI package.

Basset::CGI may not be based upon CGI.pm in the future.

B<Please note>: Basset::CGI is considered a Tier 1 module. Therefore, you shouldn't directly use it
in code. Use Basset::Object's class_for_type method instead.

 my $cgi = Basset::Object->pkg_for_type('cgi');
 
 my $foo = $cgi->param('foo');

This isn't a requirement, but using the abstract factory capabilities will save you a lot of time
and trouble if you ever want to move away from using Basset::CGI to do your form handling. Then,
you just need to toss an equivalent class into your conf file, not re-write all your code.

=head1 REQUIREMENTS

CGI, CGI::Cookie, Date::Calc

=cut

$VERSION = '1.00';

use Basset::Extractor;
use Basset::Encryption::WeakKey;
use Gramene::Config;

@ISA = qw(Basset::Extractor);

use CGI ();
use CGI::Cookie;
use Date::Calc qw(Date_to_Text);

use strict;
use warnings;

=pod

=head1 ATTRIBUTES

=over

=item headers

An arrayref containing the headers to go to your browser. Rarely accessed directly.

=back

=cut

__PACKAGE__->add_class_attr('headers', []);

=pod

=head1 METHODS

=over

=item param

Accepts a list of arguments and then returns their values as passed in, either via POST or GET.
Currently does not distinguish between the two.

 my ($name, $password) = Basset::CGI->param('name', 'password');

=cut

sub param {
	my $class = shift;
	
	my @return = ();
	
	foreach my $param (@_) {
		my @values = CGI::param($param);
		push @return, map {defined($_) && length($_) ? $_ : undef} @values;
	};
	return wantarray ? @return : $return[0];
};

sub extract {
	return shift->param(@_);
}

=pod

=item all_params

Returns a list of all arguments passed to your CGI, either via POST or GET.

 my @all = Basset::CGI->all_params();

=cut

sub all_params {
	my $class = shift;
	
	return CGI::Vars();
};

sub extract_all {
	return shift->all_params(@_);
};

=pod

=item add_header

Adds a header to your eventual CGI output.

 Basset::CGI->add_header('Content-type:text/plain');

=cut

sub add_header {
	my $class = shift;
	my $header = shift;
	
	push @{$class->headers}, $header;
	
	return $header;
};

=pod

=item content

Returns a valid Content-type header for your use.

 my $content_header = Basset::CGI->content();
 
 print $content_header, $page_contents;

Includes all headers you've specified. May be passed headers as well.

 my $content_header = Basset::CGI->content('foo:bar');

=cut

sub content {
	my $class	= shift;
	my $type	= shift;
	my $headers	= shift || [];
	
	$headers = [@$headers, @{$class->headers}];
	
	my $status_header = "Status: 200 OK";
	
	foreach my $header (@$headers) {
		if ($header =~ /^Status/) {
			$status_header = $header;
			last;
		}
	}

	my $h = $status_header . "\n";
	
	foreach (@$headers){
		$h .= "$_\n";
	};
	
	$h .= "Content-type: $type\n";
	
	foreach (@$headers){
		$h .= "$_\n";
	};
	
	$h .= "\n";
	
	$class->headers([]);	#wipe out the class headers, they've been used
	
	return $h;
};

=pod

=item encode_cookie

This is a wrapper around Basset::Encryption::WeakKey->encrypt to give you an encrypted cookie string.

=cut

sub encode_cookie {
	my $class = shift;
	
	return Basset::Encryption::WeakKey->encrypt(@_);
};

=pod

=item encode_cookie

This is a wrapper around Basset::Encryption::WeakKey->decrypt to decrypt a cookie string encrypted
with encode_cookie

=cut

sub decode_cookie {
	my $class = shift;
	
	return Basset::Encryption::WeakKey->decrypt(@_);
};

=pod

=item escape

URL escapes the key/value pair passed.

 my $escaped = Basset::CGi->escape('foo', 'this&that'); #$escape is foo=this%26that

Also, you may pass an arrayref of values

 my $escaped = Basset::CGi->escape('foo', ['this&that', 'me', '***'); #$escape is foo=this%26that&foo=me&foo=%2A%2A%2A

=cut

sub escape {
	my $class	= shift;
	my $key		= shift;
	my $value	= shift || '';
	if (ref $value eq 'ARRAY'){
		my @q = undef;
		foreach my $v (@$value){
			$v = '' unless defined $v;
			 $v =~ s/([^a-zA-Z0-9_.-])/uc sprintf("%%%02x",ord($1))/eg;
			 push @q, "$key=$v";
		};
		return join("&", @q);
	}
	elsif (! ref $value){
		$value =~ s/([^a-zA-Z0-9_.-])/uc sprintf("%%%02x",ord($1))/eg;
		return "$key=$value";	
	}
	else {
		return $class->error("Cannot escape, don't know what a $value is", "BCGI-01");
	};
};

=pod

=item redirect

Redirects you to the url specified and B<exits the program>. May accept optional args to URL encode
and include, and optional headers.

 Basset::CGI->redirect(
 	'http://www.bassetsoftware.com',
 	{
 		'username' => 'jim'
 	},
 	['Additional-header: foo']
 );

args and headers are optional.

=cut

sub redirect {
	my $class	= shift;
	
	my $url	 	= shift;
	my $args	= shift;
	my $headers	= shift || [];
	
	my $q = '';

	if ($args){
		foreach my $key (keys %$args) {
			if (ref $args->{$key} eq 'ARRAY'){
				$q .= join('', map{"&" . $class->escape($key, $_)} @{$args->{$key}});
			}
			else {
				$q .= "&" if $q;
				$q .= $class->escape($key, $args->{$key});
			};
		};
	};
#	my $q = join('&', map {$class->escape($_, $args->{$_})} keys %$args) if $args;
	$url .= "?$q" if $q;
	
	$headers = [@$headers, @{$class->headers}];
	
	print "Status: 302 Moved\n";

	foreach (@$headers){
		print $_, "\n";
	};

	print "Location: $url\n";

	foreach (@$headers){
		print $_, "\n";
	};

	print "\n";
	
	$class->headers([]);	#wipe out the class headers, they've been used

	exit;
};

=pod

=item set_cookie

Creates and returns a CGI cookie object for you. You'll need to add it as a header yourself.

 my $c = Basset::CGI->cookie(
 	'cookie_name',
 	'cookie_value', 	#this will be encoded internally by this method
 	'2004-12-22 00:00:00',	#expiration time, or undef for life of browser
 	'/'					#website accessible string
 );
 
 Basset::CGI->add_header('Set-Cookie:' . $c);

=cut

sub set_cookie {
	my $class	= shift;
	my $cookie	= shift;
	my $value	= shift;
	my $expires	= shift || undef;
	my $path	= shift || '/';

	if ($expires) {
		$expires = Basset::CGI->format_date_for_cookie($expires);
	};

	my $c = CGI::Cookie->new(
		'-name'		=> $cookie,
		'-value'	=> $class->encode_cookie($value),
		'-expires'	=> $expires,
		'-path' 	=> $path
	);

	return $c;
};

=pod

=item drop_cookie

Deletes the specified cookie.

 my $c = Basset::CGI->drop_cookie(
 	'cookie_name',
 	'/'		#optional path
 );
 
 Basset::CGI->add_header('Set-Cookie:' . $c);

=cut

sub drop_cookie {
	my $class	= shift;
	my $cookie	= shift;
	my $path	= shift || '/';
	
	my $c = CGI::Cookie->new(
		'-name'		=> $cookie,
		'-value'	=> 0,
		'-path'		=> $path,
		'-expires'	=> '-1M'
	);

	return $c;
};

=pod

=item read_cookie

Reads the specified cookie.

 my $value = Basset::CGI->read_cookie('cookie_name');

Will automatically decode the cookie value.

=cut

sub read_cookie {
	my $class = shift;
	my $cookie = shift;

	my %cookies		= CGI::Cookie->fetch;
	my $c			= $cookies{$cookie};
	if ($c) {
		return $class->decode_cookie($c->value);
	}
	else {
		return;
	};

};

=pod

=item format_date_for_cookie

Used internally. Alters a "YYYY-MM-DD HH:MM:SS" timestamp to be a valid cookie timestamp format.

=cut

sub format_date_for_cookie {
	my $class = shift;
	my $date = shift;
	
	if ($date =~ /(\d\d\d\d)-(\d\d)-(\d\d)( \d\d:\d\d:\d\d)?/) {
		my ($y, $m, $d, $time) = ($1, $2, $3, $4);
		my $dstring = Date_to_Text($y, $m, $d);
		$dstring =~ s/^(...)/$1,/;
		$time ||= "00:00:00";
		
		$dstring .= $time . " GMT";
		
		return $dstring;
	}
	else {
		return $date;
	};
};

=pod

=item read_session

Reads and returns the current session object. This expects a 'session' type to be set in your conf
file which is compatible with Basset::Session.

=cut

sub read_session {
	my $class = shift;
	
	my $session_class = $class->pkg_for_type('session') || return;
	my %cookies		= CGI::Cookie->fetch;
	my $login		= $cookies{$session_class->cookiename()};
	if ($login) {
		my $sessionkey	= Basset::CGI->decode_cookie($login->value);
		
		my $session	= $session_class->load($sessionkey)
			|| return $class->error($session_class->errvals);
		
		my $config = Gramene::Config->new();
		my $faq_config = $config->get('faq_edit');
	
		Basset::Object->conf->{'Basset::DB'}->{'-dsn'} = $faq_config->{'db_dsn'};
		Basset::Object->conf->{'Basset::DB'}->{'-user'} = $faq_config->{'db_user'};
		Basset::Object->conf->{'Basset::DB'}->{'-pass'} = $faq_config->{'db_pass'};
		Basset::Object->pkg_for_type('persistentobject')->driver(undef);
			
		if ($session->expired){
			$session->delete();
			return;
		}
		else {
			return $session;
		};
	}
	else {
		return;
	};
};

1;

=pod

=back

=cut
