###############################################################################
#
#   Name:           ExportView::GFF3Exporter;
#
#   Description:    Object for dumping data in GFF
#
#   History:        2001-08-07  jws - original version
#                   2004-10-17  scs - GFF3 version
#
#   Warnings:   
#                   Need to handle transcripts without translations
#                   Need to handle exons belonging to multiple transcripts
#                   Stupid transform of wrong dummy evalues
#                   DNA align features are omited if no TYPE_MAPPING instead
#                       of defaulting to 'nucleotide_match'
#                   Need to do Dbxref for Similarity features [ info approx in
#                               contigviewbottom.pm ]
#                   GR is really reference to protein_search.  Need a ref
#                       to /db/searches/browser
#
#
###############################################################################
package ExportView::GFF3Exporter;

use strict;

use Data::Dumper;       #for debugging

use ExportView::DBAdaptor;
use ExportView::Helper;
use ExportView::Out;
use Bio::EnsEMBL::SeqFeature;
use Bio::EnsEMBL::DBSQL::DBEntryAdaptor;

use constant MEMORY_TRACE => 
        warn( Maize::Util::StatM::change() || 'mem=' );

# external_db.db_name  to GO
use constant DB_MAPPING => {
   EGAD          => 'TIGR_EGAD',
   ENTREZPRO     => 'protein_id',
   PF            => 'Pfam',
   PIR           => 'PIR',
   SWISSPROT     => 'Swiss-Prot',
   #TIGR references may not be correct
   TIGR_FN       => 'TIGR_Osa',
   TIGR_MODEL    => 'TIGR_Osa',
   TIGR_TU       => 'TIGR_Osa',
   stable_id     => 'GR',
};

# Sequence Ontology:
# should be in database - analysis?
use constant TYPE_MAPPING => {
   #For DNALC:
   Rice_est                  => 'EST_match',
   Rice_jap_cDNA_KOME        => 'cDNA_match',
   Swissprot_Trembl_Proteins => 'protein_match',
   Maize_est                 => 'EST_match',
   Maize_meth_filt_hi_cot_cluster => 'cross_genome_match',

   #what Chris Wants:
   Rice_ESTCluster_TGI          => 'EST_match',
   Barley_ESTCluster_TGI        => 'EST_match',
   Maize_ESTCluster_TGI         => 'EST_match',
   Sorghum_ESTCluster_TGI       => 'EST_match',
   Wheat_ESTCluster_TGI         => 'EST_match',
   Rice_GI              => 'EST_match',
   Barley_GI    => 'EST_match',
   Maize_GI             => 'EST_match',
   Sorghum_GI   => 'EST_match',
   Wheat_GI             => 'EST_match',

   #etc.
   Rice_BACend                  => 'nucleotide_match',
   Barley_est                   => 'EST_match',
   Millet_est                   => 'EST_match',
   Sorghum_est                  => 'EST_match',
   Maize_BACend                 => 'cross_genome_match',
   Maize_hi_cot_Bennetzen       => 'cross_genome_match',
   Rice_Brachyantha_BACend      => 'cross_genome_match',
   Rice_Nivara_BACend           => 'cross_genome_match',
   Rice_Rufipogon_BACend        => 'cross_genome_match',

   #what Joshua Wants:
   Ryegrass_MethylFilterCluster_Orion => 'cross_genome_match',#track name
   Ryegrass_MethylFilter_Orion  => 'cross_genome_match',#track name
   Ryegrass_Assembly            => 'cross_genome_match',#logic name
   Ryegrass_Sequence            => 'cross_genome_match',#logic name
   Sorghum_CDNA                 => 'EST_match',
   Sorghum_EST                  => 'EST_match',#track name
   Sorghum_est                  => 'EST_match',#logic_name
   Sorghum_ESTCluster_Pratt     => 'EST_match',#track name
   Sorghum_ESTCluster_TUG       => 'EST_match',#track name
   Sorghum_cluster_Pratt        => 'EST_match',#logic_name
   Sorghum_tug                  => 'EST_match',#logic_name
   'Sorghum_gss-read_Klein'     => 'cross_genome_match',#logic_name
   Sorghum_GSS_Klein            => 'cross_genome_match',#track name
   Sorghum_Markers              => 'cross_genome_match',
   Sorghum_MethylFilter_Orion   => 'cross_genome_match',#track name
   Sorghum_orion                => 'cross_genome_match',#logic_name



   EST_match => 'EST_match',
   cDNA_match => 'cDNA_match',
   cross_genome_match => 'cross_genome_match',
   expressed_sequence_match => 'expressed_sequence_match',
   nucleotide_match => 'nucleotide_match',
   protein_match => 'protein_match',
   translated_nucleotide_match => 'translated_nucleotide_match',

};

use constant ALIGNMENT_DB => {
   Rice_est                  => 'EMBL',
   Rice_jap_cDNA_KOME        => 'EMBL',
   Swissprot_Trembl_Proteins => 'SPTR', # or is it just Swiss-Prot??
   Maize_est                 => 'EMBL',
   Maize_hi_cot_Bennetzen    => 'EMBL',
   Maize_meth_filt_hi_cot_cluster => 'TIGR_Maize',

       #need to quote colons in db names (which really should be in ids)...
   Rice_ESTCluster_TGI          => 'TIGR_TGI%3aRice',
   Barley_ESTCluster_TGI        => 'TIGR_TGI%3aBarley',
   Maize_ESTCluster_TGI         => 'TIGR_TGI%3aMaize',
   Sorghum_ESTCluster_TGI       => 'TIGR_TGI%3aSorghum',
   Wheat_ESTCluster_TGI         => 'TIGR_TGI%3aWheat',

   Maize_hi_cot_Bennetzen       => 'EMBL',
   Rice_Brachyantha_BACend      => 'EMBL',
   Rice_Nivara_BACend           => 'EMBL',
   Rice_Rufipogon_BACend        => 'EMBL',
   Barley_est                   => 'EMBL',
   Millet_est                   => 'EMBL',
   Sorghum_est                  => 'EMBL',
   Maize_BACend                 => 'EMBL',
};

use constant MARKER_TYPE => {
   est            => 'EST',
   microsatellite => 'microsatellite',
   tos17          => 'transposable_element_insertion_site',
   rflp           => 'RFLP_fragment',
   tag            => 'tag',
};

sub new {
    my ($class,$cgi,$databases) = @_;
    my $self = {};
    bless $self, $class;
    $self->cgi($cgi);
    $self->{databases} = $databases;
    return $self;
}

#######################################################
# export_data - main subroutine called from ExportView
#######################################################
sub export_data {
    my $self = shift;
    my $seqout;
    my $cgi = $self->cgi;
{
my @p=$cgi->param;
for my $p(@p) {
    my @v=$cgi->param($p);
    print STDERR "$p=",join(",",@v),"\n";
}
}
    my $dbadaptor = ExportView::DBAdaptor->new($cgi,$self->{databases});
    my $dbea= $self->{databases}->get_DBAdaptor('core')->get_DBEntryAdaptor;
    my $helper = ExportView::Helper->new;

    my $output_format = $cgi->param('out') || 'text';
    my $delim = $cgi->param('gff_format') || 'gff';

    unless ($cgi->param('gff_similarity')   ||
            $cgi->param('gff_repeat')       ||
            $cgi->param('gff_genscan')   ||
            $cgi->param('gff_gene')         ||
            $cgi->param('gff_external')     ||
            $cgi->param('gff_marker')       ||
            $cgi->param('gff_variation')
            ){
      die "Please select at least one type of feature for export in the 'Select export options' section.";
    }

    my %delim_lookup = (
                        gff =>  "\t",
                        tab =>  "\t",
                        csv =>  ",",
                    );

    my $glue = $delim_lookup{$delim} || "\t";

    my $vc;
    eval { $vc = $dbadaptor->fetch_vc; };
    $@ && die $@;
    die $helper->missing_field_error if !$vc;

    ##################################################
    # Export requested features in appropriate format
    ##################################################
    my @out;
    tie @out, 'ExportView::Out' if $output_format eq 'direct';

    push @out
            , '##gff-version 3'
            ,join(" ",'##sequence-region',$vc->name,1,$vc->length)  #using ->name implies start=1
                                        #could use ->chr_name, ->chr_start, ->chr_end instead
                                        #but then would have to use ->chr_name throughout
            ,'#'
            ,'# Alignments:'
            ,'#  Rice_est by blat (http://www.gramene.org/documentation/Alignment_docs/doc_rice_est.html)'
            ,'#  Rice_jap_cDNA_KOME by blat (http://www.gramene.org/documentation/Alignment_docs/doc_rice_cdna.html)'
            ,'#  Swissprot_Trembl_Proteins by blast. Preliminary version.'
            ,'#  Maize_est by blat (http://www.gramene.org/documentation/Alignment_docs/doc_maize_est.html)'
            ,'#  Maize_meth_filt_hi_cot_cluster [genomic] by blat '
            ,'#     (http://www.gramene.org/documentation/Alignment_docs/doc_maize_methyl_tigr_hi_cot_cluster.html)'
            ,'#'
            ,'# Dbxrefs:'
            ,'#  TIGR_Osa is  http://www.tigr.org/tigr-scripts/e2k1/euk_display.dbi?db=osa1&orf='
            ,'#  TIGR_Maize is http://www.tigr.org/tigr-scripts/tgi/report.pl?species=Maize&id='
            ,'#  GR url should be http://www.gramene.org/db/searches/browser?query='
            ;


    my @common_fields = qw( seqname
                            source
                            feature
                            start
                            end
                            score
                            strand
                            phase
                         );
                        
    # build list of additional fields
    my @other_fields=qw(ID Dbxref Name Alias);
    if ($cgi->param('gff_similarity') || $cgi->param('gff_repeat') ){
        push @other_fields, qw(Target Gap);
    }
    if ($cgi->param('gff_genscan') || $cgi->param('gff_gene')){
        push @other_fields, qw(Parent);
    }

    # accumulate data
    MEMORY_TRACE;
    if ($cgi->param('gff_similarity')){
        my %oksource= map {($_,1)} $cgi->param('gff_similarity_source');
        print STDERR map { "$_ ok=$oksource{$_}\n" } keys %oksource;
        my %parentseen={};
        my %traced={};
        foreach my $feature (@{$vc->get_all_SimilarityFeatures()}){
            if(%oksource) {
                next unless $oksource{ $feature->analysis->gff_source};
            } else {
                next if $feature->analysis->gff_source eq 'Proprietary';
            }
            my $what=(split ":",ref($feature))[-1];
            my $id=$feature->dbID;
            my $type=TYPE_MAPPING->{$feature->analysis->gff_feature};
            { my $tracekey="$what\t$type\t".$feature->analysis->gff_source
                            ."\t".$feature->analysis->logic_name ;
              warn "$tracekey\n" unless $traced{"$tracekey"}++;
            }
            next unless $type;
            my $quoted_name=quote($feature->hseqname);
            my $ref = { 'Target'  => join(" ",$quoted_name
                                           ,$feature->hstart
                                           ,$feature->hend
                               ,$feature->hstrand?( $feature->hstrand>=0?'+'
                                                                        :'-' )
                                                 :''
                                           )
                       ,'ID' => $what.$id
                      };
            {
                my $cigar;
                eval {  #Sometimes a cigar isn't even a cigar
                    $cigar =cigar2gff($feature->cigar_string);
                };
                $ref->{ 'Gap'} = $cigar if $cigar && ! $@;
            }
            { my $db= $feature->analysis->db || $feature->analysis->gff_feature;
              $db=ALIGNMENT_DB->{$db} if ALIGNMENT_DB->{$db};
            #$ref->{Dbxref}=[ '"'.join(":",$db,$quoted_name).'"' ]
            $ref->{Dbxref}=[ join(":",$db,$quoted_name) ]
                            if $db;
            #$ref->{Dbxref}=[ '"'.join(":",ALIGNMENT_DB->{$feature->primary_tag},$quoted_name).'"' ]
            #               if ALIGNMENT_DB->{$feature->primary_tag} ;
            }
            if( $what =~ /DnaDna/ ) {  #Link together grouped hits
                my $group=$self->daf_group($id); # daf_group_id,chr_start,chr_end,chr_strand
                if($group) {
                    my $parent="DafGroup".$group->[0];
                    if(! $parentseen{$parent}++) {
                        my $fields = join($glue,
                              @{$self->common_fields($feature,{ score => "."
                                                         , type  =>$type
                                                         , start =>$group->[1]
                                                         , end   =>$group->[2]
                                                         , strand=>$group->[3]
                                                         , source =>
                                                 $feature->analysis->logic_name
                                                 })});
                        $fields .= $glue .
                           $self->other_fields($feature,\@other_fields, $delim,
                                    { ID => $parent ,Dbxref => $ref->{Dbxref} }
                                        );
                        push @out, $fields;
                    }
                    $ref->{Parent}=$parent;
                }
            }
            my $special={score => $feature->p_value, type => $type
                       , source => $feature->analysis->logic_name};
                        #NB: ->p_value is evalue in the database
            #############
            ## stupid: ##
            $special->{score}=power10(-$special->{score}) if $special->{score}>1;
            #############
            my $fields = join($glue, @{$self->common_fields($feature,$special)});
            $fields .= $glue . $self->other_fields($feature,\@other_fields, $delim, $ref);
            push @out, $fields;
        }
        push @out, "###";
    }
    MEMORY_TRACE;
    if ($cgi->param('gff_marker')){
        my $first=1;
        foreach my $feature (@{$vc->get_all_MarkerFeatures()}){
            my $syn = $feature->marker->display_MarkerSynonym ;
            my ($name,$source);
            if($syn) {
                $name=$syn->name;
                $source=$syn->source;
            } else {
                print STDERR "marker feature id=",$feature->dbID,", marker id=",$feature->marker->dbID," no synonym\n";
                $name=":".$feature->marker->dbID.":";
                undef $source;
            }
            my $ref = { ID => "marker".$feature->dbID,  'Name' => $name };
            my $fields = join($glue, @{$self->common_fields($feature,{ source=>$source
                                               ,type=>(MARKER_TYPE->{$feature->marker->type||'tag'} || 'tag')})});
            $fields .= $glue . $self->other_fields($feature,\@other_fields, $delim, $ref);
            push @out, $fields;
        }
        push @out, "###";
    }
    MEMORY_TRACE;
    if ($cgi->param('gff_repeat')){
        print STDERR "Warning: do not handle repeats in GFF3 yet\n";
        if(0) {
        foreach my $feature (@{$vc->get_all_RepeatFeatures()}){
            my $ref = { 'hid'   => $feature->repeat_consensus()->name(),
                        'hstart'=> $feature->hstart,
                        'hend'  => $feature->hend,
                        };
            my $fields = join($glue, @{$self->common_fields($feature, {type=>'repeat_region'})});
            $fields .= $glue . $self->other_fields($feature,\@other_fields, $delim, $ref);
            push @out, $fields;
        }
        }
    }
    MEMORY_TRACE;
    if ($cgi->param('gff_external')){
        foreach my $feature (@{$vc->get_all_ExternalFeatures()}){
            my $fields = join($glue, @{$self->common_fields($feature)});
            $fields .= $glue . $self->other_fields($feature,\@other_fields, $delim);
            push @out, $fields;
        }
    }
    MEMORY_TRACE;
    if ($cgi->param('gff_variation')){
        foreach my $feature (@{$vc->get_all_SNPs()}){
            my $fields = join($glue, @{$self->common_fields($feature)});
            $fields .= $glue . $self->other_fields($feature,\@other_fields, $delim);
            push @out, $fields;
        }
    }
    my $seqid=$vc->name;

=head2 'gff_genscan' = Prediction Transcripts

    Assume all CDS

=cut

    MEMORY_TRACE;
    if ($cgi->param('gff_genscan')){
        foreach my $transcript (@{$vc->get_all_PredictionTranscripts()}){
            my $strand=$transcript->get_all_Exons->[0]->strand; #strand of 1st exon
            my $phase=0;
            #Need to save and trash ->stable_id which is not valid for an xref
            # Won't set if false value, so filter out 'none' later
            my $transcript_id=$transcript->stable_id;
            if($transcript->can('display_label')) {     #Newer Ensembl
                $transcript->display_label('none');
            } else {    #Ensembl 13
                $transcript->stable_id('none');
            }
            {
            my $tscriptref={ ID =>$transcript_id, Name => $transcript_id };
            push @out, join($glue, @{$self->common_fields($transcript
                                                ,{ seqid=>$seqid
                                                   ,type=>'mRNA'
                                                   ,strand=>$strand})}
                            , $self->other_fields($transcript,\@other_fields,
                                $delim, $tscriptref) );
            }
            foreach my $exon (@{$transcript->get_all_Exons()}){
                my $fields = join($glue, @{$self->common_fields($exon
                                            ,{type=>'exon'
                               , source=>$transcript->analysis->gff_source})});
                my $ref = {'Parent' => $transcript_id };
                $fields .= $glue . $self->other_fields($exon,\@other_fields
                , $delim, $ref);
                push @out, $fields;
            }
            push @out, "###";
        }
    }
    MEMORY_TRACE;
    if ($cgi->param('gff_gene')){
        foreach my $gene (@{$vc->get_all_Genes()}){
            #NB: $gene->analysis->gff_source comes out empty
            my $strand=$gene->strand;
            my %gene_exon_dbID_to_exon_transcripts=();
                #key is exon->dbID. value is [ exon, \@transcript_stable_ids ]
            push @out, join($glue, @{$self->common_fields($gene,{ source=>$gene->type
                                                                 ,seqid=>$seqid
                                                                 ,type=>'gene'})}
                            , $self->other_fields($gene,\@other_fields, $delim, {}));

            foreach my $transcript(@{$gene->get_all_Transcripts()}){
                my($cds_start,$cds_end,$phase,$entrezpro)=();
                if($transcript->translation) {
                    ($cds_start,$cds_end)=rev2neg($strand
                                            ,$transcript->coding_region_start
                                            ,$transcript->coding_region_end);
                    $phase=0;
                    $entrezpro = map { $_->primary_id }
                                 grep { $_->dbname eq 'ENTREZPRO' }
                          @{$dbea->fetch_all_by_Translation(
                                                 $transcript->translation)};
                } else {
                    $phase='.';
                }
                {
                my $tscriptref={ Parent=>$gene->stable_id };
                #$tscriptref->{ Dbxref}=['"'.DB_MAPPING->{ENTREZPRO}.qq(:$entrezpro")] if $entrezpro;
                $tscriptref->{ Dbxref}=[DB_MAPPING->{ENTREZPRO}.qq(:$entrezpro)] if $entrezpro;
                my $type = $transcript->stable_id =~ /tRNA/ ? 'tRNA'
                                                            : 'mRNA';
                push @out, join($glue, 
                   @{$self->common_fields($transcript,{ source=>$gene->type
                                                           ,seqid=>$seqid
                                                           ,type=>$type
                                                           ,strand=>$strand})}
                            , $self->other_fields($transcript,\@other_fields, 
                                               $delim, $tscriptref) );
                }
                
                foreach my $exon(@{$transcript->get_all_Exons()}){
                    $gene_exon_dbID_to_exon_transcripts{$exon->dbID}
                       ||= [ $exon, []];
                    push
                      @{$gene_exon_dbID_to_exon_transcripts{$exon->dbID}->[1]}
                      ,$transcript->stable_id;

                    my ($estart,$eend)=rev2neg($strand,$exon->start,$exon->end);
                    if(defined $cds_start && $estart<$cds_start) {  # Some of 
                                                                # 5' UTR here
                        my ($piece_start,$piece_end)=
                                       rev2neg($strand, $estart,$cds_start-1);
                        my $piece=Bio::EnsEMBL::SeqFeature->new(
                             -seqname     => $seqid
                            ,-start       => $piece_start
                            ,-end         => $piece_end
                            ,-strand      => $strand
                            #,-source_tag  => $gene->type
                        );
                        push @out, join($glue, 
                             @{$self->common_fields($piece,
                                               { type => "five_prime_UTR"
                                                 ,source=>$gene->type})}
                            , $self->other_fields($piece,\@other_fields, $delim,
                                {        Parent  => $transcript->stable_id 
                                        ,ID      => "5pU".$transcript->dbID
                                       #id in case of multiple pieces
                                }
                            ) );
                        $estart=$cds_start;
                    }
                    if(defined $cds_end && $eend>$cds_end) {    #some of 3' 
                                                                #UTR here
                        my ($piece_start,$piece_end)=
                                        rev2neg($strand, $cds_end+1,$eend) ;
                        my $piece=Bio::EnsEMBL::SeqFeature->new(
                             -seqname     => $seqid
                            ,-start       => $piece_start
                            ,-end         => $piece_end
                            ,-strand      => $strand
                            #,-source_tag  => $gene->type
                            ,-analysis    => $gene->analysis
                        );
                        push @out, join($glue, 
                             @{$self->common_fields($piece,
                                     { type => "three_prime_UTR"
                                          ,source=>$gene->type})}
                            , $self->other_fields($piece,\@other_fields, $delim,
                                    {    Parent  => $transcript->stable_id 
                                        ,ID      => "3pU".$transcript->dbID
                                           #id in case of multiple pieces
                                    }
                            ) );
                        $eend=$cds_end
                    }
                    if(defined($cds_start) && $estart<=$eend) {   #some of CDS 
                                                                  # here
                        my ($piece_start,$piece_end)=
                                            rev2neg($strand, $estart,$eend) ;
                        my $piece=Bio::EnsEMBL::SeqFeature->new(
                             -phase => $phase
                            ,-seqname     => $seqid
                            ,-start       => $piece_start
                            ,-end         => $piece_end
                            ,-strand      => $strand
                        #    ,-source_tag  => $gene->type supposed to
                        #         come from analysis, so don't set here
                            ,-analysis    => $gene->analysis
                        );
                        push @out, join($glue, 
                                @{$self->common_fields($piece,{ type => "CDS"
                                                  ,source=>$gene->type})}
                            , $self->other_fields($piece,\@other_fields, $delim,
                                    {    Parent  => $transcript->stable_id 
                                        ,ID      => "cds".$transcript->dbID
                                           #id in case of multiple pieces
                                    }
                            ) );
                        $phase=($phase+ $eend-$estart+1)%3;
                    }
                }
            } #end for each transcript of $gene
#           {
#           $Data::Dumper::Terse=1;
#           print STDERR "\nexon_dbID->[exon,[scripts]] "
#                      ,Dumper( \%gene_exon_dbID_to_exon_transcripts ),"\n";
#           }
            #Now output the exons:
            while (my ($eid,$et)=each %gene_exon_dbID_to_exon_transcripts) {
                my ($exon,$tsi)=@$et;
                my $fields = join($glue,
                             @{$self->common_fields($exon,
                                        {type=>'exon'
                                        ,source=>$gene->type})});
                $fields .= $glue .
                       $self->other_fields($exon,\@other_fields, $delim
                                        , { Parent=>$tsi
                                           , ID => "exon".$eid});
                           # otherwise it gets an ID based on stable_id
                           # which can be confused with transcript, etc.,
                           # ids
                push @out, $fields;
            } #end for each exon
        push @out, "###";
        } # end for each gene
    }

    MEMORY_TRACE;
    push @out,"##FASTA",">".$vc->name;
    push @out,($vc->seq=~ /(.{1,60})/g);

    unless ($output_format eq 'direct' or scalar (@out) > 0){
        die ("This query produces no results.  Try exporting a different region.");
    }

    # Add the field names to the output, if required
    unless ($delim eq 'gff'){
        unshift @out, join($glue, @common_fields, @other_fields);
    }


    if ($output_format eq 'zip'){
        my ($zip_fh,$zip_url) =  $helper->get_new_zip_filename;
        open (ZIPOUT, ">$zip_fh") or die ("Cannot create new zip file.  Please try again later");

        foreach my $line(@out){
            print ZIPOUT "$line\n";
        }
        close ZIPOUT;
        my $status = system(" gzip $zip_fh");
        die("Cannot create zipped export.  Please try again later.") unless $status == 0;
        $helper->send_redirect("$zip_url.gz");
    }
    else {
        $helper->send_header($output_format);
        
        if ($output_format eq 'html'){
            print "<h3>".$vc->id."</h3>";
            print "<pre>";
        }
        foreach my $line(@out){
            print $line."\n";
        }
        
        print "</pre>" if $output_format eq 'html';

        $helper->send_footer($output_format);
    }
    MEMORY_TRACE;
    untie @out if $output_format eq 'direct';
    MEMORY_TRACE;
}


sub cgi{
   my $self = shift;
   if( @_ ) {
      my $value = shift;
      $self->{'cgi'} = $value;
    }
    return $self->{'cgi'};
}


sub seqname{
   my $self = shift;
   if( @_ ) {
      my $value = shift;
      $self->{'seqname'} = $value;
    }
    return $self->{'seqname'};
}

sub common_fields{
    my ($self, $feature, $special) = @_;
    my ($source, $type, $start, $end, $str,$score,$phase,$seqid,$strand)=
         @$special{qw(source type start end str score phase seqid strand)};

    if (!defined($score)) {
        if($feature->can('score') && $feature->score()) {
            $score =  $feature->score();
        } else {
            $score = '.' ;
        }
    }
    $score = sprintf("%.5g", $score) unless $score eq '.';

    if (!defined($strand) && $feature->can('strand')) {
        $strand = $feature->strand();
    }
    if (! $strand) {
        $strand = ".";
    } elsif( $strand == 1 ) {
        $strand = '+';
    } elsif ( $strand == -1 ) {
        $strand = '-';
    }

    if(!defined($seqid)) {
        if (defined $self->seqname) {
            $seqid=$self->seqname;
        } else {
            if($feature->can('entire_seq') && $feature->entire_seq()) {
              $seqid = $feature->entire_seq()->name();
            }
            if(!$seqid && $feature->can('seqname')) {
              $seqid = $feature->seqname();
              $seqid =~ s/\s/_/g;
            }
            $seqid || 'SEQ';
        }
    }

    if(!defined($source)) {
        if ($feature->can('source_tag')){
            $source = $feature->source_tag;
            #warn "$source from source_tag\n";
        }
        if( ! $source && $feature->can('analysis')
                                && $feature->analysis){
            $source = $feature->analysis->gff_source;
            #warn "$source from analysis gff_source\n";
        }
        #warn "source already = ensembl\n" if $source eq 'ensembl';
        #warn "no source\n" unless $source;
        $source ||=  'ensembl';
    }
    $source =~ s/\s/_/g;
    $source =~ s/:/_/g;

    if(!defined($type)) {
        if ($feature->can('primary_tag')){
            $type = $feature->primary_tag;
            $type =~ s/\s/_/g;
        } elsif ($feature->can('analysis')){
            $type = $feature->analysis->gff_feature;
        }
        $type ||= '.';
    }

    if(!defined($start)) {
        if ($feature->can('start')){
            $start = $feature->start;
        }
    }

    if(!defined($end)) {
        if ($feature->can('end')){
            $end = $feature->end;
        }
    }

    if (!defined($phase)) {
        if ( $feature->can('phase')) {
            $phase = $feature->phase();
        }
        $phase = '.' unless defined $phase;
    }
    $phase = '.' if $phase<0 || $type eq 'exon'; #only CDSs have phase

    my @results = ( $seqid,
                    $source,
                    $type,
                    $start,
                    $end,
                    $score,
                    $strand,
                    $phase,
                  );
    return \@results;

}


sub other_fields{
    my ($self, $feature, $fieldlist, $delim, $dataref) = @_;
    my %fields;
    @fields{@$fieldlist} = ();    # build hash of fields from fieldlist

    if ($dataref){  # if passed some field data, load it into the hash
        %fields = (%fields, %{$dataref});
    }

    if($feature->can('get_all_DBEntries')) {
#?      for my $dbe in ( @{$feature->get_all_DBEntries} ) {
#?          print join(" ^ ",$dbe->dbname,DB_MAPPING->{$dbe->dbname},$dbe->primary_id)
#?                      ,"\n"
#?      }
        push @{$fields{Dbxref}},
                    map { join(":", @$_ ) }  #acc Scott Cain
                    #map { '"'.join(":", @$_ ).'"' }  # acc spec
                    grep { $_->[0] }
                    map {[DB_MAPPING->{$_->dbname},$_->primary_id]}
                    @{$feature->get_all_DBEntries}
                    ;
    }
    if($feature->can('external_name') and my $en=$feature->external_name) {
        if($fields{Name}) {
            push @{$fields{Alias}},$en;
        } else {
            $fields{Name}=$en;
        }
    }
    { my $si;
    if($feature->can('stable_id') and $si=$feature->stable_id and $si ne 'none') {
        push @{$fields{Dbxref}}, DB_MAPPING->{stable_id}.qq(:$si);
        #push @{$fields{Dbxref}}, '"'.DB_MAPPING->{stable_id}.qq(:$si");
        $fields{ID}||=$si;
        if($fields{Name}) {
            push @{$fields{Alias}},$si;
        } else {
            $fields{Name}=$si;
        }
    }
    }

    # Munge the fields together appropriately for the delimiter type
    my $munged;
    if ($delim eq 'tab'){
        $munged = join ("\t", @fields{@$fieldlist});
    }
    elsif ($delim eq 'csv'){
        $munged = join (",", @fields{@$fieldlist});
    }
    elsif ($delim eq 'gff'){
        #$munged = join (";",map { "$_=".
#                                   (ref($fields{$_} )
#                                          ? join(",",@{$fields{$_}} )
#                                          : $fields{$_} )
#                                  }
#               grep {defined $fields{$_}} @$fieldlist);
        foreach my $field(@$fieldlist){
            if (defined $fields{$field}){
                $munged .= "$field=".
                        (ref($fields{$field} )
                               ? join(",",@{$fields{$field}} )
                               : $fields{$field} )
                        #do join here so can build a field up
                        # by pushing
                        .";";
            }
        }
        $munged=substr($munged,0,-1) if $munged;
    }
    return $munged;
}

#Following should be method?:

=head2 cigar2gff

convert EnsEMBL CIGAR string (numbers before letters, 1 optional, no spaces)
to GFF3 CIGAR string (numbers after letters, pieces separated by space)

=cut
sub cigar2gff {
    my ($cigar)=@_;
    $cigar =~ tr/\n//d;
    return $cigar unless $cigar;
#    print STDERR "c($cigar";
    $cigar=~ s/^(\D)/1$1/;
    $cigar=~ s/(\D)(\D)/$12$2/g;
    $cigar=~ s/(\d+)(\D+)/$2$1/g;
    $cigar=~ s/(\d)(\D)/$1 $2/g;
#    print STDERR ",$cigar)\n";
    return $cigar;
}

=head2 rev2neg

Negate and interchange coordinates if reverse strand
It may look stupid, but it keeps me sane (SCS)

=cut

sub rev2neg {
    my $strand=shift;
    if($strand>=0) {
        return @_;
    } else {
        return map { -$_ } reverse @_;
    }
}

sub power10 {
    return exp((shift)*log(10));
}
sub log10 {
    return log(shift)/log(10);
}

#The table involved may not exist;
sub _daf_group_sth {
    my $self=shift;
    unless(defined $self->{'_daf_group_sth'}) {
        my $sth;
        eval{
            $sth=$self->{databases}->get_DBAdaptor('core')->dbc->prepare("
                select daf_group_id,chr_start,chr_end,chr_strand from ensembl_daf_group
                where dna_align_feature_id=?");
        };
        warn "daf_group prepare: $DBI::errstr, $@" unless $sth;
        $self->{'_daf_group_sth'}=$sth||0;
    }
    return $self->{'_daf_group_sth'};
}

sub daf_group {
    my ($self,$id)=@_;
    my $sth=$self->_daf_group_sth;
    return $sth &&
        $sth->{Database}->selectrow_arrayref($sth,{},$id);

}

sub quote {
    my ($text)=@_;
    $text =~ s/([^a-zA-Z0-9. :^*\$\@!+_?-])/sprintf("%%%02x",ord($1))/eg;
    return $text;

}

1;


