package EnsEMBL::Maize::Component::Export;

use EnsEMBL::Web::Component;
use EnsEMBL::Web::Form;
use EnsEMBL::Web::Document::SpreadSheet;

use Readonly;

use strict;
use warnings;
no warnings "uninitialized";

our @ISA = qw(EnsEMBL::Web::Component);

Readonly my %FORMATS => (
    'gff' => {
        'delim'  => "\t",
        'fields' => [
            qw(seqname source feature start end score strand frame attributes)],
    },
    'csv' => {
        'delim'  => ',',
        'fields' => [qw(seqname refseq feature subtype start end)],
    },
    'tab' => {
        'delim'  => "\t",
        'fields' => [qw(seqname refseq feature subtype start end)],
    },
);

sub stage1 {
    my ($panel, $object) = @_;
    $panel->print(@{ [ $panel->form('stage1_form')->render() ] });
    return 1;
}

sub stage2 {
    my ($panel, $object) = @_;
    $panel->print(@{ [ $panel->form('stage2_form')->render() ] });
    return 1;
}

sub stage1_form {
    my ($panel, $object) = @_;

    my @errors = (
        '',
        qq(Sorry, there was a problem locating the requested DNA sequence. Please check your choices - including chromosome name - and try again.),
        qq(Sorry, your chosen anchor points appear to be on different sequence regions. Please check your choices and try again.)
    );

    my $form = EnsEMBL::Web::Form->new('stage1_form',
        "/@{[$object->species]}/exportview", 'get');
    my $sitetype = ucfirst(lc($object->species_defs->ENSEMBL_SITETYPE));
    $form->add_element(
        'type'  => 'Information',
        'value' => qq(<p>
      Choose one or two features from the same chromosome as anchor points
      and display the region between them.
      Both features must be mapped to the current $sitetype golden
      tiling path.  If you select "None" for the second feature,
      the display will be based around the first feature.
    </p>
    )
    );

    if ($object->param('error')) {
        my $error_text = $errors[ $object->param('error') ];
        $form->add_element(
            'type'  => 'Information',
            'value' => '<p class="error">'
                . $error_text
                . ' If you continue to have a problem, please contact <a href="mailto:helpdesk@ensembl.org">helpdesk@ensembl.org</a>.</strong></p>'
        );
    }

    $form->add_element('type' => 'SubHeader', 'value' => 'Region');

    if ($object->param('l') =~ /(\w+):(-?[\.\w]+)-([\.\w]+)/) {
        my ($sr, $s, $e) = ($1, $2, $3);
        $object->input_param('seq_region_name', $sr);
        $object->input_param('type1',           'bp');
        $object->input_param('anchor1',         $s);
        $object->input_param('type2',           'bp');
        $object->input_param('anchor2',         $e);
    }
    $form->add_element(
        'type'  => 'String',
        'label' => 'Chromosome name/fragment',
        'style' => 'small',
        'name'  => 'seq_region_name',
        'id'    => 'seq_region_name',
        'value' => $object->param('seq_region_name')
    );

    my @types = (
        { 'value' => 'band',       'name' => 'Band' },
        { 'value' => 'region',     'name' => 'Region' },
        { 'value' => 'marker',     'name' => 'Marker' },
        { 'value' => 'bp',         'name' => 'Base pair' },
        { 'value' => 'gene',       'name' => 'Gene' },
        { 'value' => 'peptide',    'name' => 'Peptide' },
        { 'value' => 'transcript', 'name' => 'Transcript' },
    );
    my @types_1 = @types;
    $form->add_element(
        'type'         => 'DropDownAndString',
        'select'       => 'select',
        'name'         => 'type1',
        'label'        => 'From (type):',
        'values'       => \@types_1,
        'value'        => $object->param('type1') || 'bp',
        'string_name'  => 'anchor1',
        'string_label' => 'From (value)',
        'string_value' => $object->param('anchor1'),
        'style'        => 'medium',
        'required'     => 'yes'
    );
    unshift(@types, { 'value' => 'none', 'name' => 'None' });
    $form->add_element(
        'type'         => 'DropDownAndString',
        'select'       => 'select',
        'name'         => 'type2',
        'label'        => 'To (type):',
        'values'       => \@types,
        'value'        => $object->param('type2') || 'bp',
        'string_name'  => 'anchor2',
        'string_label' => 'To (value)',
        'style'        => 'medium',
        'string_value' => $object->param('anchor2')
    );
    $form->add_element('type' => 'SubHeader', 'value' => 'Context');
    $form->add_element(
        'type'     => 'String',
        'style'    => 'short',
        'required' => 'no',
        'value'    => '',
        'name'     => 'upstream',
        'label'    => 'Bp upstream (to the left)'
    );

    $form->add_element(
        'type'     => 'String',
        'required' => 'no',
        'value'    => '',
        'style'    => 'short',
        'name'     => 'downstream',
        'label'    => 'Bp downstream (to the right)'
    );
    $form->add_element('type' => 'SubHeader', 'value' => 'Output format');

    my $formats = $object->__data->{'formats'};
    my @formats = ();
    foreach my $super (
        sort { $formats->{$a}{'name'} cmp $formats->{$b}{'name'} }
        keys %$formats
        )
    {
        foreach my $format (
            sort {
                $formats->{$super}{'sub'}{$a} cmp $formats->{$super}{'sub'}{$b}
            } keys %{ $formats->{$super}{'sub'} }
            )
        {
            push @formats,
                {
                'group' => $formats->{$super}{'name'},
                'value' => $format,
                'name'  => $formats->{$super}{'sub'}{$format}
                };
        }
    }
    $form->add_element(
        'type'   => 'DropDown',
        'select' => 'select',
        'name'   => 'format',
        'label'  => 'Output Format',
        'values' => \@formats,
        'value'  => $object->param('format')
    );
    $form->add_element(
        'type'  => 'Hidden',
        'name'  => 'action',
        'value' => 'select'
    );
    $form->add_element(
        'type'     => 'Submit',
        'value'    => 'Continue >>',
        'spanning' => 'center'
    );
    return $form;
}

sub add_hidden_fields {
    my ($form, $object) = @_;
    my @fields
        = qw(seq_region_name type1 anchor1 type2 anchor2 downstream upstream format);
    my $flag = 1;
    foreach (@fields) {
        next unless defined $object->param($_);
        $form->add_element(
            'type'  => 'Hidden',
            'name'  => $_,
            'value' => $object->param($_)
        );
        next if $_ eq 'format';
        $flag = 0;
    }
    if ($flag) {
        $form->add_element(
            'type'  => 'Hidden',
            'name'  => 'l',
            'value' => $object->seq_region_name . ':'
                . $object->seq_region_start . '-'
                . $object->seq_region_end
        );
    }
    my $text = "<p>You are exporting @{[$object->seq_region_type_and_name]} 
                @{[ $object->thousandify( $object->seq_region_start ) ]} - 
                @{[ $object->thousandify( $object->seq_region_end ) ] }.</p>\n";
    my @O = ();
    if (   $object->param('seq_region_name')
        && $object->seq_region_name eq $object->param('seq_region_name'))
    {
        push @O, $object->seq_region_type_and_name;
    }
    push @O, ucfirst($object->param('type1')) . ' ' . $object->param('anchor1')
        if $object->param('anchor1');
    push @O, ucfirst($object->param('type2')) . ' ' . $object->param('anchor2')
        if $object->param('anchor2') && $object->param('type2') ne 'none';
    push @O, " plus " . $object->param('downstream') . " basepairs downstream"
        if $object->param('downstream');
    push @O, " plus " . $object->param('upstream') . " basepairs upstream"
        if $object->param('upstream');

    if (@O) {
        $text .= "<blockquote>This region is defined by: "
            . join(", ", @O)
            . "</blockquote>\n";
    }
    $form->add_element('type' => 'Information', 'value' => $text);
    $form->add_element(
        'type'  => 'Hidden',
        'name'  => 'action',
        'value' => 'export'
    );
}

sub add_HTML_select {
    my ($form, $object) = @_;
    my @options = (
        [ 'html' => 'HTML' ],
        [ 'txt'  => 'Text' ],
        [ 'gz'   => 'Compressed text (.gz)' ]
    );
    $form->add_element(
        'type'   => 'DropDown',
        'name'   => 'output',
        'label'  => 'Output format',
        'value'  => $object->param('output') || 'html',
        'values' =>
            [ map { { 'value' => $_->[0], 'name' => $_->[1] } } @options ]
    );
    $form->add_attribute('onSubmit',
        qq(this.elements['_format'].value='HTML';this.target='_self';flag='';for(var i=0;i<this.elements['output'].length;i++){if(this.elements['output'][i].checked){flag=this.elements['output'][i].value;}}if(flag=='txt'){this.elements['_format'].value='Text';this.target='_blank'}if(flag=='gz'){this.elements['_format'].value='TextGz';})
    );
}

sub export_form {

    # TODO: consolidate into one form
}

sub fpc_features_form {
    my ($panel, $object) = @_;
    my $form = EnsEMBL::Web::Form->new('stage2_form',
        "/@{[$object->species]}/exportview", 'get');
    $form->add_element(
        'type'  => 'Hidden',
        'name'  => '_format',
        'value' => 'HTML'
    );
    add_hidden_fields($form, $object);
    $form->add_element(
        'type'  => 'SubHeader',
        'value' => 'Feature format options'
    );
    my @options = (
        [ 'clones'         => 'All FPC Clones' ],
        [ 'acc_clones'     => 'Sequenced FPC Clones' ],
        [ 'contigs'        => 'FPC Contigs' ],
        [ 'corebinmarkers' => 'Core-Bin Markers' ],
        [ 'ssr_markers'    => 'Electronic SSR Markers' ],
        [ 'overgos'        => 'Overgo Markers' ],
        [ 'other_markers'  => 'Other Markers' ],
    );
    my %checked = map { $_->[0] => 'yes' } @options;
    $form->add_element(
        'type'   => 'MultiSelect',
        'class'  => 'radiocheck1col',
        'name'   => 'options',
        'label'  => 'Features to export',
        'values' => [
            map {
                +{  'value'   => $_->[0],
                    'name'    => $_->[1],
                    'checked' => $checked{ $_->[0] }
                    }
                } @options
        ]
    );
    add_HTML_select($form, $object);
    $form->add_element(
        'type'     => 'Submit',
        'value'    => 'Continue >>',
        'spanning' => 'center'
    );
    return $form;
}

use EnsEMBL::Web::SeqDumper;
use CGI qw(escapeHTML);

sub fpc_features {
    my ($panel, $object) = @_;
    my %checked = map { $_ => 'yes' } $object->param('options');
    $panel->print("<pre>");

    my $format = $object->param('format');
    $panel->print(
        join($FORMATS{$format}{'delim'}, @{ $FORMATS{$format}{'fields'} }),
        "\n");

    my @features = ();
    if ($checked{'ssr_markers'}) {
        _add_features(
            {   'object'   => $object,
                'list'     => \@features,
                'function' => 'get_all_MarkerFeatures',
                'analyses' => [qw(ssr_marker)],
                'type'     => 'Marker'
            }
        );
    }
    if ($checked{'overgos'}) {
        _add_features(
            {   'object'   => $object,
                'list'     => \@features,
                'function' => 'get_all_MarkerFeatures',
                'analyses' => [qw(cornsensus overgo-ap overgo-dupont)],
                'type'     => 'Marker'
            }
        );
    }
    if ($checked{'other_markers'}) {
        _add_features(
            {   'object'   => $object,
                'list'     => \@features,
                'function' => 'get_all_MarkerFeatures',
                'analyses' => [qw(Knob Telo Ribo Chloro Maize_marker)],
                'type'     => 'Marker'
            }
        );
    }
    if ($checked{'corebinmarkers'}) {
        _add_features(
            {   'object'   => $object,
                'list'     => \@features,
                'function' => 'get_all_MarkerFeatures',
                'analyses' => [qw(core_bin_marker)],
                'type'     => 'Marker'
            }
        );
    }
    if ($checked{'clones'}) {
        _add_features(
            {   'object'   => $object,
                'list'     => \@features,
                'function' => 'get_all_MiscFeatures',
                'analyses' => ['bac_map'],
                'type'     => 'Clone',
            }
        );
    }
    if ($checked{'acc_clones'}) {
        _add_features(
            {   'object'   => $object,
                'list'     => \@features,
                'function' => 'get_all_MiscFeatures',
                'analyses' => ['acc_bac_map'],
                'type'     => 'Clone',
            }
        );
    }
    if ($checked{'contigs'}) {
        _add_features(
            {   'object'   => $object,
                'list'     => \@features,
                'function' => 'get_all_MiscFeatures',
                'analyses' => ['superctgs'],
                'type'     => 'Contig',
            }
        );
    }
    for my $feature (sort { $a->start <=> $b->start } @features) {
        $panel->print(
            _feature(
                {   'feature' => $feature,
                    'format'  => $format,
                }
            )
        );
    }

    $panel->print("</pre>");
}

=pod

=head2 _add_features
    Add features

=cut

sub _add_features {
    my ($params) = @_;
    my $object   = $params->{'object'};
    my $function = $params->{'function'};
    my @analyses = @{ $params->{'analyses'} };
    my $type     = $params->{'type'};
    my $features = $params->{'list'};
    if (scalar @analyses == 0) {
        push @analyses, undef;
    }
    for my $analysis (@analyses) {
        for my $feature (@{ $object->slice->$function($analysis) }) {
            $feature->{__feature_type} = $type;
            if (defined $feature->analysis) {
                $feature->{__feature_subtype} = $feature->analysis->logic_name;
            } elsif (defined $feature->get_scalar_attribute('embl_acc')) {
                $feature->{__accession}
                    = $feature->get_scalar_attribute('embl_acc');
                $feature->{__seqstatus}
                    = $feature->get_scalar_attribute('external') eq 'true'
                    ? 'EXTERNAL'
                    : $feature->get_scalar_attribute('seqstatus');
            }
            push @$features, $feature;
        }
    }
}

sub _feature {
    my ($params) = @_;
    my $feature  = $params->{'feature'};
    my $format   = $params->{'format'};

    my $feature_attributes = +{
        'seqname' => $feature->display_id || 'SEQ',
        'feature'    => $feature->{__feature_type},
        'subtype'    => $feature->{__feature_subtype},
        'start'      => $feature->start,
        'end'        => $feature->end,
        'source'     => 'Ensembl',
        'strand'     => '+',
        'frame'      => '.',
        'score'      => '.',
        'refseq'     => $feature->slice->desc,
        'attributes' => 'RefSeq "' . $feature->slice->desc . '"',
        'status'     => $feature->{__seqstatus},
        'accession'  => $feature->{__accession},
    };

    if ($format eq 'gff') {
        $feature_attributes->{'attributes'} = join('; ',
            map      {"$_=\"@{[$feature_attributes->{$_}]}\""}
                grep { defined $feature_attributes->{$_} }
                qw(refseq subtype accession status));
    }

    return (
        join(
            $FORMATS{$format}{'delim'},
            map { $feature_attributes->{$_} } @{ $FORMATS{$format}{'fields'} }
        ),
        "\n"
    );
}

1;
