package EnsEMBL::DB::Core;

# Things we need for the web site. A sort of grab bag at the moment

use strict;

#############################
# Load the species defs file
#############################

use DBI;
use Data::Dumper;
use Carp;

use EnsEMBL::Web::SpeciesDefs;
use Bio::EnsEMBL::DBLoader;
use Bio::EnsEMBL::DBSQL::DBAdaptor;
use Bio::EnsEMBL::Lite::DBAdaptor;
use Bio::EnsEMBL::ExternalData::FASTA::FASTAAdaptor;
use Bio::EnsEMBL::ExternalData::SNPSQL::DBAdaptor;
use Bio::EnsEMBL::ExternalData::SNPSQL::WebAdaptor;
use Bio::EnsEMBL::Compara::DBSQL::DBAdaptor;
use Bio::EnsEMBL::ExternalData::Expression::LibraryAdaptor;
use Bio::EnsEMBL::ExternalData::DAS::DASAdaptor;
use Bio::EnsEMBL::ExternalData::DAS::DAS;
use Bio::EnsEMBL::ExternalData::GO::GOAdaptor;
use Bio::EnsEMBL::ExternalData::Glovar::DBAdaptor;
use Bio::EnsEMBL::ExternalData::Haplotype::DBAdaptor;
use Bio::EnsEMBL::External::BlastAdaptor;
use Bio::EnsEMBL::External::ExternalFeatureAdaptor;
use DiseaseHandler;
use Bio::EnsEMBL::Registry;
my $reg = "Bio::EnsEMBL::Registry";
 

use vars qw( $SPECIES_DEFS );
BEGIN{ 
       $SPECIES_DEFS = EnsEMBL::Web::SpeciesDefs->new();
     }

my @additional_dbs;
if ($SPECIES_DEFS->ENSEMBL_SITETYPE eq "Vega") {
    eval "require Bio::Otter::DBSQL::DBAdaptor";
    if ($@) {
        warn(qq(Failed to require Bio::Otter::DBSQL::DBAdaptor: $@\n));
    } else {
        Bio::Otter::DBSQL::DBAdaptor->import();
    }
    push @additional_dbs, 'vega';
}

sub get_databases {
  my $species = $ENV{ENSEMBL_SPECIES};
  return &get_databases_common( $species, @_ );
}

sub get_databases_species {
  my $species = shift || die( "Need a species!" );
  return &get_databases_common( $species, @_ );
}

## Parameters -> 'core'       = core ensemble datbase 
##               'SNP'        = SNP database 
##               'compara'    = Compara db
##               'exp'        = Expression databae 
##		 'embl'	      = EMBL database + adds as "EFF"
##		 'disease     = disease database
##		 'family'     = family database
##		 'archive'    = archive db - old identifiers

## Result => Hash ref containing the following keys
##    'core'              => Handle to core database
##    'error'             => Error message if fails to connect to core databases
##    'SNP','exp','map','embl','disease','family' => Handles to respective databases 
##    'non_fatal_error'   => Error message if fails to connect to any ancillary database

sub get_databases_common {
#    my $db_details = shift;
#    my $ms_db_details = shift;
  my $species = shift || die( "Need a species" );
  ( $species ) = $SPECIES_DEFS->valid_species( $species );
  if( ! $species ){ die("Invalid species: $species") }

  my $hashref = { 'non_fatal_error' => '' };
  my %databases = map {($_,1)} @_, @additional_dbs;
  # Get core DB first
  if( $databases{'core'} ) {
    eval{ $hashref->{'core'} = get_core_database($species); };
    if( $@ || !$hashref->{'core'} ){ 
	$hashref->{'error'} = qq(Unable to connect to core database: );
        $hashref->{'error'} .= $@ || "database not found";
	return $hashref;
    }
    delete $databases{'core'};
  }

  # Simple DB's; no dependence on core
  if( $databases{'go'} ) {
    eval{ $hashref->{'go'} = get_go_database($species); };
    if( $@ || !$hashref->{'go'} ){ 
      $hashref->{'go'} = 0;
      $hashref->{'non_fatal_error'} .= "\ngo database: ";
      $hashref->{'non_fatal_error'} .= $@ || "database not found";
    }
    delete $databases{'go'};
  } 
 
  if( $databases{'mart'} ) {
    eval{ $hashref->{'mart'} = get_mart_database($species); };
    if( $@ || !$hashref->{'mart'} ){ 
      $hashref->{'mart'} = 0;
      $hashref->{'non_fatal_error'} .= "\nmart database: ";
      $hashref->{'non_fatal_error'} .= $@ || "database not found";
    }
    delete $databases{'mart'};
  } 
  
  if( $databases{'help'} ) {
    eval{ $hashref->{'help'} = get_help_database($species); };
    if( $@ || !$hashref->{'help'} ){ 
      $hashref->{'help'} = 0;
      $hashref->{'non_fatal_error'} .= "\nhelp database: ";
      $hashref->{'non_fatal_error'} .= $@ || "database not found";
    }
    delete $databases{'help'};
  } 
  
  if($databases{'fasta'}) {
    eval{ $hashref->{'fasta'} = get_fasta_database($species); };
    if( $@ || !$hashref->{'fasta'} ){ 
      $hashref->{'fasta'} = 0;
      $hashref->{'non_fatal_error'} .= "\nfasta database: ";
      $hashref->{'non_fatal_error'} .= $@ || "database not found";
    }
    delete $databases{'fasta'};
  }
  
  if($databases{'cdna'}) {
    eval{ $hashref->{'cdna'} = get_cdna_database($species); };
    if( $@ || !$hashref->{'cdna'} ){ 
      $hashref->{'cdna'} = 0;
      $hashref->{'non_fatal_error'} .= "\ncdna database: ";
      $hashref->{'non_fatal_error'} .= $@ || "database not found";
    } elsif($hashref->{'core'}) {
      $hashref->{'cdna'}->dnadb( $hashref->{'core'} );
      $hashref->{'core'}->add_db_adaptor( 'cdna', $hashref->{'cdna'} );
      $hashref->{'cdna'}->add_db_adaptor( 'core', $hashref->{'core'} );
    }
  
    delete $databases{'cdna'};
  }
    
  if($databases{'family'}) {
    eval{ $hashref->{'family'} = get_family_database($species); };
    if( $@ || !$hashref->{'family'} ){ 
      $hashref->{'family'} = 0;
      $hashref->{'non_fatal_error'} .= "\nfamily database: ";
      $hashref->{'non_fatal_error'} .= $@ || "database not found";
    }
    delete $databases{'family'};
  }
  

  # Complex databases: Attach to core if available
  
  if( $databases{'SNP'} ) {
    eval{ $hashref->{'SNP'} = get_snp_database($species); };
    if( $@ || !$hashref->{'SNP'} ){ 
      $hashref->{'SNP'} = 0;
      $hashref->{'non_fatal_error'} .= "\nSNP database: ";
      $hashref->{'non_fatal_error'} .= $@ || "database not found";
    } elsif( $hashref->{'core'} ){
      $hashref->{'core'}->add_db_adaptor('SNP', $hashref->{'SNP'});
    }
    delete $databases{'SNP'};
  }
  
  if($databases{'disease'}) {
    eval{ $hashref->{'disease'} = get_disease_database($species); };
    if( $@ || !$hashref->{'disease'} ){ 
      $hashref->{'disease'} = 0;
      $hashref->{'non_fatal_error'} .= "\ndisease database: ";
      $hashref->{'non_fatal_error'} .= $@ || "database not found";
    } elsif( my $core_db = $hashref->{'core'}) {
      my $dis_db = $hashref->{'disease'};
      $dis_db->add_db_adaptor($core_db);
    }
    delete $databases{'disease'};
  }

  if( $databases{'blast'} ) {
    eval{ $hashref->{'blast'} = get_blast_database($species); };
    if( $@ || !$hashref->{'blast'} ){ 
      $hashref->{'blast'} = 0;
      $hashref->{'non_fatal_error'} .= "\nblast database: ";
      $hashref->{'non_fatal_error'} .= $@ || "database not found";
    } elsif( my $core_db = $hashref->{'core'}) {
      $core_db->add_db_adaptor('blast', $hashref->{'blast'});
    }
    delete $databases{'blast'};
  }
  
  if($databases{'haplotype'}) {
    eval{ $hashref->{'haplotype'} = get_haplotype_database($species); };
    if( $@ || !$hashref->{'haplotype'} ){ 
      $hashref->{'haplotype'} = 0;
      $hashref->{'non_fatal_error'} .= "\nhaplotype database: ";
      $hashref->{'non_fatal_error'} .= $@ || "database not found";
    } elsif( my $core_db = $hashref->{'core'} ){
      my $hapl_db = $hashref->{'haplotype'};
      $core_db->add_db_adaptor('haplotype', $hapl_db );
      $hapl_db->add_db_adaptor('core',      $core_db );
    } 
    delete $databases{'haplotype'};
  }

  if($databases{'variation'}) {
    eval{ $hashref->{'variation'} = get_variation_database($species); };
    if( $@ ){ 
      $hashref->{'variation'} = 0;
      $hashref->{'non_fatal_error'} .= "\nvariation database: $@";
    } elsif( my $core_db = $hashref->{'core'} ){
      my $vari_db = $hashref->{'variation'};
      $core_db->add_db_adaptor('variation', $vari_db );
      $vari_db->add_db_adaptor('core',      $core_db );
    } 
    delete $databases{'variation'};
  }

  if($databases{'glovar'}) {
      eval{ $hashref->{'glovar'} = get_glovar_database($species); };
      if( $@ || !$hashref->{'glovar'} ){ 
          $hashref->{'glovar'} = 0;
          $hashref->{'non_fatal_error'} .= "\nglovar database: ";
          $hashref->{'non_fatal_error'} .= $@ || "database not found";
      }
      elsif( my $core_db = $hashref->{'core'} ){
          ## Glovar can serve several track sources via 
          ## a subclassed version of the same DB adaptor
          ## which share the same DB connection
          my $glovar_db = $hashref->{'glovar'};
          my $glovar_snp_adaptor = $glovar_db->get_GlovarSNPAdaptor;
          $glovar_snp_adaptor->consequence_exp($SPECIES_DEFS->GLOVAR_SNP_CONSEQUENCE_EXP);
          $core_db->add_ExternalFeatureAdaptor($glovar_snp_adaptor);
          $core_db->add_ExternalFeatureAdaptor($glovar_db->get_GlovarSTSAdaptor);
          #$core_db->add_ExternalFeatureAdaptor($glovar_db->get_GlovarTraceAdaptor);
          $core_db->add_ExternalFeatureAdaptor($glovar_db->get_GlovarHaplotypeAdaptor);
          #$core_db->add_ExternalFeatureAdaptor($glovar_db->get_GlovarBaseCompAdaptor);
      }
      
      if (!$ENV{'ORACLE_HOME'}){
            my $ora_home = $SPECIES_DEFS->ENSEMBL_ORACLE_HOME;
            warn("Trying to initialize and Oracle DBI driver but no ORACLE_HOME environment found!\n") if ($ora_home eq ''); 
            $ENV{'ORACLE_HOME'} = $ora_home;
            $ENV{'LD_LIBRARY_PATH'} = $SPECIES_DEFS->LD_LIBRARY_PATH;
      }

      delete $databases{'glovar'};
  }
  
  if($databases{'est'}) {
    eval{ $hashref->{'est'} = get_est_database($species); };
    if( $@ || !$hashref->{'est'} ){ 
      $hashref->{'est'} = 0;
      $hashref->{'non_fatal_error'} .= "\nest database: ";
      $hashref->{'non_fatal_error'} .= $@ || "database not found";
    }
    elsif( my $cor_db = $hashref->{'core'} ){
      my $est_db  = $hashref->{'est'};
      $cor_db->add_db_adaptor('est',  $est_db);
      $est_db->add_db_adaptor('core', $cor_db);
    }
    delete $databases{'est'};
  }
  
  if($databases{'lite'}) {
    eval{ $hashref->{'lite'} = get_lite_database($species); };
    if( $@ || !$hashref->{'lite'} ){ 
      $hashref->{'lite'} = 0;
      $hashref->{'non_fatal_error'} .= "\nlite database: ";
      $hashref->{'non_fatal_error'} .= $@ || "database not found";
    }
    elsif( my $core_db = $hashref->{'core'} ){
      my $lite_db  = $hashref->{'lite'};
      warn ref $lite_db;
      $core_db->add_db_adaptor('lite', $lite_db);
      $lite_db->add_db_adaptor('core', $core_db);
    }      
    delete $databases{'lite'};
  }
  
  if($databases{'vega'}) {
    eval{ $hashref->{'vega'} = get_vega_database($species); };
    if( $@ || !$hashref->{'vega'} ){ 
      $hashref->{'vega'} = 0;
      $hashref->{'non_fatal_error'} .= "\nvega database: ";
      $hashref->{'non_fatal_error'} .= $@ || "database not found";
    }
    elsif($hashref->{'core'}) {
      $hashref->{'vega'}->dnadb( $hashref->{'core'} );
      $hashref->{'core'}->add_db_adaptor( 'vega', $hashref->{'vega'} );
      $hashref->{'vega'}->add_db_adaptor( 'core', $hashref->{'core'} );
    }
    delete $databases{'vega'};
  }
  
  # Very complex DB: compara!
  
  if( $databases{'compara'} ) {
    eval{ $hashref->{'compara'} = get_compara_database($species); };
    if( $@ || !$hashref->{'compara'} ){ 
      $hashref->{'compara'} = 0;
      $hashref->{'non_fatal_error'} .= "\ncompara database: ";
      $hashref->{'non_fatal_error'} .= $@ || "database not found";
    } else {
      if( $hashref->{'core'}) {
	my $comp_db = $hashref->{compara};
	$hashref->{'core'}->add_db_adaptor( 'compara', $comp_db );
      }
    } 
    delete $databases{'compara'};
  } 
  
  # Check all requested db's exist
  if(%databases) {
    $hashref->{'non_fatal_error'} = 
      "You have specified unknown database(s): ".
	(join ", ",keys %databases);
  }
  return $hashref;
}

#--------------------
# CORE DB 
sub get_core_database{
  my $species = shift || undef();
  return $reg->get_DBAdaptor($species, 'core');
}

#--------------------
# Single species DBs

sub get_snp_database{
  my $species = shift || undef();
  return $reg->get_DBAdaptor($species,'SNP');
}

sub get_haplotype_database{
  my $species = shift || undef();
  return $reg->get_DBAdaptor($species,'haplotype');
}

sub get_variation_database{
  my $species = shift || undef();
  return $reg->get_DBAdaptor($species,'variation');
}

sub get_fasta_database{
  my $db_info = _get_database_info( shift, 'ENSEMBL_FASTA' ) ||
    die( "No fasta database for this species" );
  my $adpt = _get_database( $db_info,
			    'Bio::EnsEMBL::DBSQL::DBAdaptor' );
  return Bio::EnsEMBL::ExternalData::FASTA::FASTAAdaptor->new($adpt);
}

## vega
sub get_vega_database{
  my $species = shift || undef();
  return $reg->get_DBAdaptor($species,'vega');
}

sub get_cdna_database{
  my $species = shift || undef();
  return $reg->get_DBAdaptor($species,'cdna');
}

sub get_disease_database{
  my $db_info = _get_database_info( shift, 'ENSEMBL_DISEASE' ) ||
    die( "No disease database for this species" );
  return _get_database( $db_info,
			'DiseaseHandler' );
}

sub get_est_database{
  my $species = shift || undef();
  return $reg->get_DBAdaptor($species,'est');
}

sub get_lite_database{
  my $species = shift || undef();
  return $reg->get_DBAdaptor($species,'lite');
}

sub get_glovar_database{
  my $species = shift || undef();
  my $dba = $reg->get_DBAdaptor($species, 'glovar');
  $reg->add_DNAAdaptor($species, 'glovar', $species, 'core');
  return $dba;
}

#--------------------
# Multispecies DBs

sub get_compara_database{
 my $dba =  $reg->get_DBAdaptor('Multi','compara');
 if(defined($dba)){
   return $dba;
 }
 my $db_info = $SPECIES_DEFS->multidb->{ENSEMBL_COMPARA} ||
   die( "No compara database for this species" );
 return _get_database( $db_info,
		       'Bio::EnsEMBL::Compara::DBSQL::DBAdaptor' );
}

sub get_go_database{
  my $dba =  $reg->get_DBAdaptor('Multi','go');
  if(defined($dba)){
    return $dba;
  }
  my $db_info = $SPECIES_DEFS->multidb->{ENSEMBL_GO} ||
    die( "No go database for this species" );
  return _get_database( $db_info,
			'Bio::EnsEMBL::ExternalData::GO::GOAdaptor' );
}

sub get_mart_database{
  my $db_info = $SPECIES_DEFS->multidb->{ENSEMBL_MART} ||
    die( "No mart database in MULTI" );

  my $driver = $db_info->{DRIVER};
  ## This probably won't work for Oracle...
  my $dsn = ( "DBI:$driver:".
	      join( ';',
		    "database=".$db_info->{NAME},
		    "host="    .$db_info->{HOST},
		    "port="    .$db_info->{PORT} ) );

  my $dbh = DBI->connect($dsn,
			 $db_info->{USER},
			 $db_info->{PASS} ) || die( $DBI::errstr );
  return $dbh;
}

sub get_help_database {
  my $db_info = $SPECIES_DEFS->multidb->{ENSEMBL_HELP} ||
    die( "No help database in MULTI" );

  my $driver = $db_info->{DRIVER};
  my $dsn = ( "DBI:$driver:".
	      join( ';',
		    "database=".$db_info->{NAME},
		    "host="    .$db_info->{HOST},
		    "port="    .$db_info->{PORT} ) );

  my $dbh = DBI->connect($dsn,
			 $db_info->{USER},
			 $db_info->{PASS},
                         { PrintError => 1, RaiseError => 1 }
                         ) || die( $DBI::errstr );
  return $dbh;
}

sub get_blast_database{
  my $dba =  $reg->get_DBAdaptor('Multi','blast');
  if(defined($dba)){
    return $dba;
  }
  my $db_info = $SPECIES_DEFS->multidb->{ENSEMBL_BLAST} ||
    die( "No blast database in MULTI" );
  # Write DB, so update user and pass
  return _get_database( $db_info,
			'Bio::EnsEMBL::External::BlastAdaptor' );
}

sub _get_database_info{
  my $species  = shift || $ENV{ENSEMBL_SPECIES};
  my $conf_key = shift || die( "Need a DB conf key" );
  my $conf = $SPECIES_DEFS->get_config( $species, 'databases' ) || 
    return undef();
  return $conf->{$conf_key} || undef();
}

sub _get_database{
  my $db_info = shift || die( "Need DB info" );
  my $a_class = shift || die( "Need an adaptor class" );

  return $a_class->new(
		       -dbname => $db_info->{NAME},
		       -user   => $db_info->{USER},
		       -pass   => $db_info->{PASS},
		       -host   => $db_info->{HOST},
		       -port   => $db_info->{PORT},
		       -driver => $db_info->{DRIVER},
		      );
}


sub add_track_das_sources {
   &_add_internal_das_sources( 'track', @_ );
}
sub add_internal_das_sources {
   &_add_internal_das_sources( 'internal', @_ );
}

sub proxy {
  my $URL = shift;
  my $PROXY = 1;
  return 0 unless $URL=~/^https?:\/\/([^:\/]+)/;
  return 0 unless defined $SPECIES_DEFS->ENSEMBL_WWW_PROXY;
  return 1 unless defined $SPECIES_DEFS->ENSEMBL_NO_PROXY;
  my $DOMAIN = $1 ;
  foreach my $suffix ( @{$SPECIES_DEFS->ENSEMBL_NO_PROXY||[]} ) {
    my $suf2 = '.*'.quotemeta($suffix).'$';
    return 0 if $DOMAIN=~/$suf2/;
  }
  return 1;
}

sub _add_internal_das_sources {
   my $source    = shift;
   my $databases = shift;
   return unless $databases->{'core'};
   my $das_sources = $source eq 'track' ? $SPECIES_DEFS->ENSEMBL_TRACK_DAS_SOURCES : $SPECIES_DEFS->ENSEMBL_INTERNAL_DAS_SOURCES;

   foreach (@_) {
      s/^managed_//;
      my $dbname = $das_sources->{$_};
      next unless $dbname;
      next unless $dbname->{'retrieve_features'}==1;
      my $adaptor = undef;
      eval {
	 my $URL = $dbname->{'url'};
         $URL = "http://$URL" unless $URL =~ /https?:\/\//i;
         $adaptor = Bio::EnsEMBL::ExternalData::DAS::DASAdaptor->new(
            -url        => $URL,
            -dsn        => $dbname->{'dsn'},
				-type  => $dbname->{'type'} || 'ensembl_location',
				-name  => $dbname->{'name'},
            -types => $dbname->{'types'} || [], 
	    -ens        => $databases->{'core'},
	    proxy($URL) ? ( '-proxy_url' => $SPECIES_DEFS->ENSEMBL_WWW_PROXY ) : ()
         );
      };
      if($@) {
          warn("DAS error >> $@ <<") 
      } else {
         $databases->{'core'}->add_DASFeatureFactory(
            Bio::EnsEMBL::ExternalData::DAS::DAS->new( $adaptor )
         );
      }
   }
}

sub add_external_das_sources {
   my $databases = shift;
   my $das_data  = shift;
   return unless $databases->{'core'};

   foreach (@_) {
      s/^managed_//;
      s/^extdas_//;
      my $dbname = $das_data->{$_};
      next unless $dbname;
      my $adaptor = undef;
      eval {
	 my $URL = $dbname->{'URL'};
         $URL = "http://$URL" unless $URL =~ /https?:\/\//i;
         $adaptor = Bio::EnsEMBL::ExternalData::DAS::DASAdaptor->new(
            -url   => "$URL/das",
            -dsn   => $dbname->{'dsn'},
				-type  => $dbname->{'type'} || 'ensembl_location',
				-name  => $dbname->{'name'},
            -ens   => $databases->{'core'},
            proxy($URL) ? ( '-proxy_url' => $SPECIES_DEFS->ENSEMBL_WWW_PROXY ) : ()
         );
      };
      if($@) {
          warn("DAS error >> $@ <<") 
      } else {
         $databases->{'core'}->add_DASFeatureFactory(
            Bio::EnsEMBL::ExternalData::DAS::DAS->new( $adaptor )
         );
      }
   }
}

1;
