package GrameneBuilder;

=head1 NAME

GrameneBuilder

=head1 DESCRIPTION

This is the builder/installer script for Gramene.  It subclasses
Module::Build.

=head1 METHODS

All methods are invoked via the "Build" script (which is created by
"Build.PL").  Only custom methods are documented here.  Please read
the POD for Module::Build for full documentation on the targets for
the "Build" script.

=cut

# ----------------------------------------------------

use strict;
use Config::General;
use Cwd;
use DBI;
use File::Find;
use File::Path 'mkpath';
use File::Spec::Functions;
use Module::Build;
use Readonly;

Readonly my $MYSQLDUMP => '/usr/local/mysql/bin/mysqldump';

use base 'Module::Build';

# ----------------------------------------------------
sub ACTION_build {

=pod

=head2 build

We override the default "build" action of Module::Build and do nothing
because we don't want to create the "blib" directory and install from
there.  We're only going to copy our desired directories into the
"destdir" (default is "/usr/local/gramene").

=cut

    my $self = shift;
    return 1;
}

# ----------------------------------------------------
#sub ACTION_dist {
#
#=pod
#
#=head2 dist
#
#Here we can insert the building of other tarballs, e.g., to package
#bulky tutorials or auxillary data files as separate packages.  Maybe
#this gets defined in a "build config"?  For now we just let the basic
#"dist" action happen.
#
#=cut
#
#    my $self = shift;
#
#    $self->ACTION_schemas;
#    $self->SUPER::ACTION_dist;
#
#    return 1;
#}

# ----------------------------------------------------
#
# Hijack the installation process to just copy stuff to destdir.
#
sub ACTION_install {
    my $self = shift;
    my $dest = $self->destdir || '/usr/local/gramene/';
    my $cwd  = cwd();
    my @dirs = qw[ conf cgi-perl html lib scripts sql templates ];

    unless ( -d $dest ) {
        mkpath( $dest, 1, 0755 );
    }

    for my $dir ( @dirs ) {
        my $from  = catdir( $cwd, $dir );
        next unless -d $from;
        my @files = $self->read_dir( $from );
        for my $file ( @files ) {
            my $to = catdir( $dest, $dir );
            if ( $file =~ /($from)(.*)/ ) {
                $to = catdir( $to, $2 );
            }

            if ( $dir eq 'conf' && -e $to ) {
                next if $self->y_n( "'$to' exists.  Overwrite?", 'n' );
            }

            $self->copy_if_modified(
                from    => $file,
                to      => $to,
                flatten => 0,
            );

            if ( $dir eq 'cgi-perl' ) {
                chmod 0755, $to or die "Cannot make '$to' executable: $!\n";
            }
        } 
    }

    print "Gramene installation complete.\n";

    return 1;
}

# ----------------------------------------------------
sub ACTION_schemas {

=pod

=head2 ACTION_schemas

  ./Build schemas

Dumps the SQL DDLs directly from MySQL.  Uses the "modules"
section of "gramene.conf" to determine which dbs to dump.  Schemas
DDLs go into "schemas/<module_name>-<build_number>.sql" and data dumps
go into "data/<module_name>-<build_number>.sql."  Won't overwrite any
existing files without confirmation.

=cut

    my $self = shift;
    return unless $self->y_n( 'OK to build SQL?', 'y' );

    eval { require Gramene::Config; };
    if ( $@ ) {
        die "Must install Gramene before running 'schemas' build.\n";
    }

    my $gconfig       = Gramene::Config->new;
    my $modules_conf  = $gconfig->get('modules');
    my @modules       = ref $modules_conf->{'module'} eq 'ARRAY'
                        ? @{ $modules_conf->{'module'} }
                        : ( $modules_conf->{'module'} );
    my $gramene_root = '/usr/local/gramene';
    my $schema_dir   = catfile( $gramene_root, 'schemas' );

    mkpath( $schema_dir, 1, 0755 ) unless -e $schema_dir;

    #
    # Dump schemas for defined modules
    #
    for my $module ( @modules ) {
        my $section    = $gconfig->get( $module )
                         or die "No Gramene conf section '$module'";
        my $user       = $section->{'db_user'};
        my $pass       = $section->{'db_pass'};
        my $dsn        = $section->{'db_dsn'};
        (my $db_name    = (DBI->parse_dsn( $dsn ))[-1] ) =~ s/^\w+=//;
        my $data_dir   = catfile( $gramene_root, 'data' );
        my $sql_dest   = catfile( $schema_dir, "${db_name}.sql" );
        my $opts       = "--skip-opt -u$user -p$pass";


        my $write_sql = 1;
        if ( -e $sql_dest ) {
            $write_sql = $self->y_n( 
                "'$sql_dest' exists.  OK to overwrite?", 'n'
            );
        }

        if ( $write_sql ) {
            print "Dumping schema for $module to '$sql_dest'\n";
            `$MYSQLDUMP --no-data $opts $db_name > $sql_dest`;
        }
    }

    $self->SUPER::ACTION_manifest;

    return 1;
}

# ----------------------------------------------------
sub read_dir { 
    my $self = shift;
    my $dir  = shift;
    die "Directory '$dir' does not exist\n" unless -d $dir;
    
    my @files;
    find( 
        sub {
            push @files, $File::Find::name
                if -f $_ && $File::Find::name !~ /CVS/
        },
        $dir
    );
    
    return @files;
}

1;

# ----------------------------------------------------

=pod

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.orgE<gt>.

=cut
