#!/usr/local/bin/perl

# vim: tw=78: sw=4: ts=4: et: 

# $Id: mk-cdbi.pl,v 1.5 2006/08/16 14:55:39 kycl4rk Exp $

use strict;
use warnings;
use English qw( -no_match_vars );
use File::Basename;
use Getopt::Long;
use Graph::Directed;
use Pod::Usage;
use Readonly;
use SQL::Translator;
use Template;

Readonly my $VERSION => sprintf '%d.%02d', 
                        qq$Revision: 1.5 $ =~ /(\d+)\.(\d+)/;

my ( $help, $man_page, $show_version, $template );
GetOptions(
    't|template=s' => \$template,
    'help'         => \$help,
    'man'          => \$man_page,
    'version'      => \$show_version,
) or pod2usage(2);

if ( $help || $man_page ) {
    pod2usage({
        -exitval => 0,
        -verbose => $man_page ? 2 : 1
    });
}; 

if ( $show_version ) {
    my $prog = basename( $PROGRAM_NAME );
    print "$prog v$VERSION\n";
    exit 0;
}

my $file   = shift or pod2usage('No input file');
my $sqlt   = SQL::Translator->new;

$sqlt->parser( $file =~ /\.ya?ml$/ ? 'YAML' : 'MySQL' );
$sqlt->producer('YAML'); # not really our destination

my $junk   = $sqlt->translate( file => $file );
my $schema = $sqlt->schema;
my $g      = Graph::Directed->new;

# Reduce table info to names, fields, and foriegn key relations
# Build a graph of the relationships
my %tables;
for my $table ( $schema->get_tables ) {
    my $tname  = $table->name;
    $g->add_vertex( $tname );

    $tables{ $tname }{'name'} = $tname;

    my %fks    = ();
    my @fields = ();
    my @unique = ();
    my $name_field;
    for my $field ( $table->get_fields ) {
        if ( $field->is_primary_key ) {
            $tables{ $tname }{'pk'} = $field->name;
        }
        else {
            push @fields, $field->name;
        }

        if ( 
             !$name_field 
          && !$field->is_primary_key
          && $field->name !~ /_id$/
        ) {
            $name_field = $field->name;
        }

        if ( $field->is_foreign_key ) {
            my $fkref    = $field->foreign_key_reference;
            my $fktable  = $fkref->reference_table;
            my $fkfields = $fkref->reference_fields;

            push @{ $fks{ $fktable } }, $field->name;
            $tables{ $fktable }{'links'}{ $tname } = 1;

            $tables{ $fktable }{'path_to'}{ $tname } 
                = [ $fkfields->[0], $field->name ];
            $tables{ $tname }{'path_to'}{ $fktable }
                = [ $field->name, $fkfields->[0] ];

            $g->add_edge( $fktable, $tname );
        }

        if ( $field->is_unique ) {
            push @unique, $field->name;
        }
    }

    $tables{ $tname }{'fks'}              = \%fks;
    $tables{ $tname }{'fields'}           = \@fields;
    $tables{ $tname }{'unique'}           = \@unique;
    $tables{ $tname }{'essential_fields'} = [ 
        grep {!/^(comments|ref_seq)$/} @fields 
    ];
    $tables{ $tname }{'stringify'} = $name_field || $fields[0];
}

# Use a depth-first search to make Class::DBI happy about 
# the order of class declarations
my $d = Graph::Traversal::DFS->new( $g, next_alphabetic => 1 );
$d->preorder;

# Because TT may complain about relative file paths
open my $fh, '<', $template or die "Can't read template '$template': $!\n";
my $tt = do{ local $/; <$fh> };
close $fh;

my $t = Template->new;
my $output;
$t->process( 
    \$tt, 
    {
        graph  => $d,
        tables => \%tables
    },
    \$output
) or die $t->error;

print $output;

__END__

# ----------------------------------------------------
=head1 NAME

mk-cdbi.pl - make Class::DBI module for GDPDM

=head1 VERSION

This documentation refers to mk-cdbi.pl version $Revision: 1.5 $

=head1 SYNOPSIS

  mk-cdbi.pl --template ../templates/class-dbi.tmpl \
    ../schema/gdpdm.sql > ../lib/GDPDM/CDBI.pm

Options:

  -t|--template  A Template Toolkit template

  --help         Show brief help and exit
  --man          Show full documentation
  --version      Show version and exit

=head1 DESCRIPTION

This script parses the GPDMD schema and creates the Class::DBI
object-relational model (GDPDM::CDBI) to mediate database requests.
The version of the GDPDM schema used to create this module is included
in the "schema" directory as "gdpdm.sql," but you can regenerate this
layer yourself using whatever schema you desire (e.g., if you add
extra tables or fields).  The template needed by this script is
included in the "templates" directory.

=head1 SEE ALSO

SQL::Translator, Template, Class::DBI.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
