#!/usr/local/bin/perl

# vim: tw=78: sw=4: ts=4: et: 

# $Id: gdpdm-search-index.pl,v 1.4 2006/08/31 18:24:36 kycl4rk Exp $

use strict;
use warnings;
use Data::Dumper;
use DBI;
use IO::Prompt;
use English qw( -no_match_vars );
use File::Basename;
use Getopt::Long;
use Pod::Usage;
use Readonly;

use GDPDM::CDBI;
use GDPDM::Config qw( get_config );
use GDPDM::Utils qw( table_name_to_class );

Readonly my $VERSION => sprintf '%d.%02d', 
                        qq$Revision: 1.4 $ =~ /(\d+)\.(\d+)/;

my ( $help, $man_page, $show_version, $db_name );
GetOptions(
    'd|db:s'  => \$db_name,
    'help'    => \$help,
    'man'     => \$man_page,
    'version' => \$show_version,
) or pod2usage(2);

if ( $help || $man_page ) {
    pod2usage({
        -exitval => 0,
        -verbose => $man_page ? 2 : 1
    });
}; 

if ( $show_version ) {
    my $prog = basename( $PROGRAM_NAME );
    print "$prog v$VERSION\n";
    exit 0;
}

if ( !$db_name ) {
    my @dbs = map { $_->{'key'} } GDPDM::Config->available_databases();
    $db_name = prompt( -menu => \@dbs, "Which database? ");
}

my $ok = prompt -yn, "OK to recache in $db_name? ";

exit 0 unless $ok;

my $db = GDPDM::CDBI->db_Main( $db_name );

$db->do('truncate gdpdm_search');

my $insert_sql = q[
    insert
    into   gdpdm_search (table_name, record_id, record_text)
    values (?, ?, ?)
];

my $i = 0;
#for my $table ( GDPDM::CDBI->represented_tables ) {
for my $table ( qw[ div_passport div_synonym cdv_marker div_stock ] ) {
    print "Processing $table\n";

    my $class    = table_name_to_class( $table );
    my @data     = $class->retrieve_all;
    my $id_field = $class->columns('Primary');
    my @has_a    = keys %{ $class->meta_info('has_a') || {} };
    my $skip     = join '|', $id_field, @has_a;
    my @columns  = grep { !/($skip)/ } $class->columns('All');

    for my $rec ( @data ) {
        $i++;
        my $text = join ' ', map { defined $_ ? $_ : () } $rec->get(@columns);
        next if !$text;
        $text =~ s/^\s+|\s+$//g; # trim
        $text =~ s/\s+/ /g;      # collapse spaces
        $db->do( $insert_sql, {}, ( $table, $rec->id, $text ) );
    }
}

print "Done, processed $i records.\n";

__END__

# ----------------------------------------------------
=head1 NAME

gdpdm-search-index.pl - rebuild the MySQL table for text searches

=head1 VERSION

This documentation refers to gdpdm-search-index.pl version $Revision: 1.4 $

=head1 SYNOPSIS

  gdpdm-search-index.pl [database]

Options:

  --help        Show brief help and exit
  --man         Show full documentation
  --version     Show version and exit

=head1 DESCRIPTION

This table truncates the "gdpdm_search" table in your GDPDM schema 
and recreates the data needed for FULLTEXT searching.

=head1 SEE ALSO

MySQL's FULLTEXT indexes.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
