#!/usr/local/bin/perl

# vim: tw=78: sw=4: ts=4: et: 

# $Id: gdpdm-import.pl,v 1.6 2007/05/03 15:24:07 kycl4rk Exp $

use strict;
use DBI;
use Getopt::Long;
use GDPDM::Import;
use IO::Prompt qw( prompt );
use Pod::Usage;
use Readonly;

Readonly my $VERSION => sprintf "%d.%02d", 
                        q$Revision: 1.6 $ =~ /(\d+)\.(\d+)/;

my ( $help, $show_version, $man_page, $update, $verify_only, 
    $db_name, $debug );
GetOptions(
    'd|database:s' => \$db_name,
    'u|update'     => \$update,
    'v|verify'     => \$verify_only,
    'debug'        => \$debug,
    'help'         => \$help,
    'man'          => \$man_page,
    'version'      => \$show_version,
) or pod2usage(2);

if ( $help || $man_page ) {
    pod2usage({
        -exitval => 0,
        -verbose => $man_page ? 2 : 1
    });
};

if ( $show_version ) {
    my $prog = basename( $0 );
    print "$prog v$VERSION\n";
    exit(0);
}

if ( !@ARGV ) {
    pod2usage( 'No input files' );
}

my @files = @ARGV;
@ARGV = ();

if ( !$db_name ) {
    my @dbs;
    for my $db ( GDPDM::Config->available_databases() ) {
        my $key = $db->{'key'};
        my $dsn = (DBI->parse_dsn( $db->{'dsn'} ))[-1];
        if ( $dsn =~ /database=(\w+)/ ) {
            $dsn = $1;
        }

        push @dbs, "$key ($dsn)";
    }

    $db_name = prompt( -menu => \@dbs, "Which database? ");
}

my $files = join(', ', map { qq['$_'] } @files);

prompt( "OK to import $files into db '$db_name'? ", -yn ) or exit 0;

$db_name =~ s/\s+(.*)$//;

for my $file ( @files ) {
    print "Importing '$file'\n";
    GDPDM::Import->import_data( 
        $file, 
        { 
            update  => $update,
            verify  => $verify_only,
            db_name => $db_name,
            debug   => $debug,
        }
    );
}

print "Done.\n";

__END__

# ----------------------------------------------------

=head1 NAME

gdpdm-import.pl - import (S-expression, XML) data into a GDPDM schema

=head1 SYNOPSIS

  gdpdm-import.pl [options] file [file2 ...]

Options:

  -d|--database  The database name to import into
  -u|--update    Use primary key IDs to update existing db records
  -v|--verify    Only verify the integrity of the data, don't import

  --help         Show brief help and exit
  --man          Show all documentation
  --version      Show version and exit

=head1 DESCRIPTION

This script parses input files and imports the data into a GDPDM
schema.  For information on the GDPDM schema, please see the project
homepage:

  http://www.maizegenetics.net/gdpdm/

The "--update" flag indicates that the primary key (PK) identifiers in
the input file match those in the database.  With this flag, only the
PK will be used to find records to update.  When a PK can't be found,
the record will be created.  Without this flag, the default action
will be to use the fields indicated as "lookup"s or, if there are none 
defined, to use all fields which aren't comments or the PK.  If a
record is found matching, then it will be updated with any new
information in the file, otherwise the record will be created.

=head1 SEE ALSO

GDPDM::Import.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
