#!/usr/local/bin/perl

# vim: tw=78: sw=4: ts=4: et: 

# $Id: gdpdm-export.pl,v 1.3 2006/04/19 15:33:15 kycl4rk Exp $

use strict;
use Cwd qw( cwd );
use IO::Prompt;
use File::Spec::Functions;
use Getopt::Long;
use GDPDM::Export;
use Pod::Usage;
use Readonly;

Readonly my $VERSION => sprintf "%d.%02d", 
                        q$Revision: 1.3 $ =~ /(\d+)\.(\d+)/;

my ( $help, $show_version, $man_page, $output_file, $db_name );
GetOptions(
    'o|output=s' => \$output_file,
    'd|db:s'     => \$db_name,
    'help'       => \$help,
    'man'        => \$man_page,
    'version'    => \$show_version,
) or pod2usage(2);


if ( $help || $man_page ) {
    pod2usage({
        -exitval => 0,
        -verbose => $man_page ? 2 : 1
    });
};

if ( $show_version ) {
    my $prog = basename( $0 );
    print "$prog v$VERSION\n";
    exit(0);
}

if ( !$db_name ) {
    my @dbs = map { $_->{'key'} } GDPDM::Config->available_databases();
    $db_name = prompt( -menu => \@dbs, "Which database? ");
}

my $ok;
if ( $output_file ) {
    if ( -e $output_file ) {
        my $overwrite = prompt -yn, "OK to overwrite '$output_file'? ";
        exit 0 unless $overwrite;
        $ok = $overwrite;
    }
    elsif ( $output_file !~ /\.xls$/ ) {
        die "'$output_file' must end in '.xls' (Excel file)\n";
    }
}
else {
    $output_file ||= catfile( cwd(), $db_name . '.xls' );
}

$ok ||= prompt -yn, "OK to export data in '$db_name' to '$output_file'? ";

exit 0 unless $ok;

print "Exporting to '$output_file'\n";
GDPDM::Export->export_data( 
    { 
        output_file => $output_file, 
        output_type => 'xls',
        db_name     => $db_name,
        object      => '',
        object_id   => '',
        log_level   => 'debug',
    }
);

print "Done.\n";

__END__

# ----------------------------------------------------

=head1 NAME

gdpdm-export.pl - export data from a GDPDM schema

=head1 SYNOPSIS

  gdpdm-export.pl [options] -o file

Options:

  -o|--output  File name to put data

  -d|--db      Database name

  --help       Show brief help and exit
  --man        Show all documentation
  --version    Show version and exit

=head1 DESCRIPTION

Foo.

=head1 SEE ALSO

GDPDM::Export.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
