package GDPDM::Export;

# vim: set ts=4 sw=4 et tw=78:

# $Id: Export.pm,v 1.5 2007/06/19 15:38:15 kycl4rk Exp $

use strict;
use Carp qw( croak );
use Data::Stag;
use Data::Dumper;
use GDPDM::Config qw( get_config );
use GDPDM::CDBI;
use GDPDM::Utils qw( table_name_to_class get_logger );
use Graph::Directed;
use Graph::Traversal::DFS;
use Readonly;
use Spreadsheet::WriteExcel;

Readonly my $EMPTY_STR => q{};

# --------------------------------------------------------
sub export_data {
    my $class       = shift;
    my $opts        = shift || {};
    my $output_file = $opts->{'output_file'} or croak 'No output file';
    my $output_type = $opts->{'output_type'} || 'sxp';
    my $object      = $opts->{'object'}      || '';
    my $object_id   = $opts->{'object_id'}   || 0;
    my $log_level   = $opts->{'log_level'}   || 'info';
    my $db_name     = $opts->{'db_name'}     || '';
    my $logger      = get_logger();
    my $config      = get_config();
            
    if ( $db_name ) {
        $logger->info("Setting db to '$db_name'");
        GDPDM::CDBI->db_Main( $db_name );
    }

    my $g = Graph::Traversal::DFS->new( GDPDM::CDBI->as_graph );

    my $ss = Spreadsheet::WriteExcel->new( $output_file );
    my $stag;
    if ( $object ) {
        my %valid = map { $_, 1 } GDPDM::CDBI->represented_tables;
        if ( !$valid{ $object } ) {
            croak "Invalid object ($object)";
        }

        $logger->info("Exporting object '$object'");

                
    }
    else {
        $logger->info("Exporting all data"); 
        for my $table ( GDPDM::CDBI->represented_tables ) {
            my $class   = table_name_to_class( $table );
            my $it      = $class->retrieve_all or next;
            my $ws      = $ss->add_worksheet( $table );
            my $row_num = 0;
            my @columns = $class->columns('Ordered');
            my %fk_col  = map { $_, 1 } $class->columns('Foreign');

            $ws->write_row($row_num, 0, \@columns);

            while ( my $item = $it->next ) {
                my @record;
                for my $col ( @columns ) {
                    if ( $fk_col{ $col } ) {
                        if ( my $fk = $item->get($col) ) {
                            push @record, $fk->id;
                        }
                        else {
                            push @record, $EMPTY_STR;
                        }
                    } 
                    else {
                        push @record, $item->get($col);
                    } 
                }

                $ws->write_row( ++$row_num, 0, \@record );
            }
        }
    }


    if ( $output_type eq 'xls' ) {
        $ss->close; 
    }
    else {
        $stag->write( $output_file, $output_type );
    }

    $logger->info('Done with export');

    return 1;
}

1;

# --------------------------------------------------------

=pod

=head1 NAME

GDPDM::Export - import data file into a GDPDM schema

=head1 VERSION

This documentation refers to version $Revision: 1.5 $.

=head1 SYNOPSIS

  use GDPDM::Export;

=head1 DESCRIPTION

Exports a data file.

=head1 METHODS

=head2 export_data

  GDPDM::Export->export_data( $file );

Exports a structured file (S-expression, XML, indented).

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
