package GDPDM::Config;

# vim: set ts=4 sw=4 et tw=78:

# $Id: Config.pm,v 1.5 2007/06/19 15:38:15 kycl4rk Exp $

use strict;
use Carp qw( croak );
use Config::Tiny;
use Readonly;
use Exporter 'import';

Readonly my $DEFAULT_CONF => '/usr/local/gramene/conf/gdpdm.conf';

use vars qw( @EXPORT_OK );

@EXPORT_OK = qw( get_config available_databases );

# --------------------------------------------------------
sub get_config {
    my $file   = shift || $DEFAULT_CONF;
    my $config = Config::Tiny->read( $file ) or croak $Config::Tiny::errstr;

    return wantarray ? %$config : $config;
}

# --------------------------------------------------------
sub available_databases {
    my %config = get_config();

    my @dbs;
    for my $db ( grep { /^database(_\w+)?$/ } keys %config ) {
        push @dbs, { 
            key  => $db, 
            name => $db, 
            %{ $config{ $db } }
        };
    }

    return sort { $a->{'name'} cmp $b->{'name'} } @dbs;
}

1;

# --------------------------------------------------------

=pod

=head1 NAME

GDPDM::Config - configuration for GDPDM 

=head1 VERSION

This documentation refers to version $Revision: 1.5 $.

=head1 SYNOPSIS

  use GDPDM::Config qw( get_config );

=head1 DESCRIPTION

Interface to configuration for GDPDM.

=head1 METHODS

=head2 get_config

  my %config = get_config();

A simple wrapper to point Config::Std to the config file.

=head2 available_databases

  my @dbs = GDPDM::Config->available_databases();

Returns an array of hashrefs with the config info for every "database"
defined in the configuration file.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
