package GDPDM::CDBI;

# vim: set ts=4 sw=4 et tw=78:

# $Id: CDBI.pm,v 1.12 2007/06/19 15:38:14 kycl4rk Exp $

use strict;

use base qw( Class::DBI::Sweet );

use Carp qw( croak );
use Class::DBI;
use Class::DBI::AsForm;
use Class::DBI::FromForm;
use Class::DBI::Plugin::AbstractCount;
use Class::DBI::Plugin::RetrieveAll;
use Class::DBI::Pager;
use Data::Page;
use Graph::Directed;
use GDPDM::Config qw( available_databases get_config );
use GDPDM::Utils qw( table_name_to_class );
use Readonly;

Readonly my $COLON    => q{:};
Readonly my $COMMA    => q{,};
Readonly my @WEAK_FKS => (
    [ 'div_stock', 'div_allele_assay' ],
    [ 'div_stock', 'cdv_marker' ],
    [ 'div_allele_assay', 'div_allele_assay' ],
);

my ( $CURRENT_DB_NAME, $DBH );

# ------------------------------------------------
sub current_db_name {
    return $CURRENT_DB_NAME || '';
}

# ------------------------------------------------
sub db_Main {
    my $class   = shift;
    my $db_name = shift || $CURRENT_DB_NAME || '';
    my $config  = get_config();
    my @dbs     = available_databases();

    if ( !$db_name ) {
        for my $db ( available_databases() ) {
            if ( $db->{'is_default'} ) {
                $db_name = $db->{'key'};
                last;
            }
        }

        if ( !$db_name ) {
            $db_name = $dbs[0]->{'key'};
        }
    }

    if ( defined $DBH && $CURRENT_DB_NAME eq $db_name ) {
        return $DBH;
    }
    else {
        if ( defined $config->{ $db_name } ) {
            $DBH = DBI->connect(
                $config->{ $db_name }{'dsn'},
                $config->{ $db_name }{'user'},
                $config->{ $db_name }{'password'},
                { __PACKAGE__->_default_attributes },
            );

            $CURRENT_DB_NAME = $db_name;
        }
        else {
            croak( "Bad db name '$db_name'");
        }
    }

    return $DBH;
}

# ------------------------------------------------
sub represented_tables {
    qw/
        cdv_allele_curated_allele
        cdv_curated_allele
        cdv_curation
        cdv_map_feature
        cdv_map_feature_annotation
        cdv_map_feature_annotation_type
        cdv_marker
        cdv_marker_annotation
        cdv_marker_annotation_type
        cdv_passport_group
        cdv_passport_set
        cdv_reason
        cdv_source
        div_aa_annotation
        div_accession_collecting
        div_allele
        div_allele_assay
        div_annotation_type
        div_experiment
        div_generation
        div_locality
        div_obs_unit
        div_obs_unit_sample
        div_passport
        div_poly_type
        div_scoring_tech_type
        div_statistic_type
        div_stock
        div_stock_parent
        div_synonym
        div_taxonomy
        div_trait
        div_trait_uom
        div_treatment
        div_treatment_uom
        div_unit_of_measure
        qtl_analysis_method
        qtl_analysis_result
        qtl_marker
        qtl_obs_unit
        qtl_obs_unit_set
        qtl_qtl
        qtl_study
        qtl_trait
        xref
        xref_type

    /
}

# ------------------------------------------------
sub as_graph {
    my $g = Graph::Directed->new;

    my %weak;
    for my $w ( @WEAK_FKS ) {
        $weak{ $w->[0] }{ $w->[1] } = 1;
        $weak{ $w->[1] }{ $w->[0] } = 1;
    }

    for my $table ( represented_tables() ) {
        my $class   = table_name_to_class( $table );
        my $has_a   = $class->meta_info('has_a') || {};
        my %valid   = map { $_, 1 } $class->columns('All');
        my $pk_name = $class->columns('Primary');

        # Check foreign key references.
        if ( %$has_a ) {
            while ( my ( $fk, $fk_info ) = each %$has_a ) {
                my $fk_class = $fk_info->{'foreign_class'};

                next if exists $weak{ $table }{ $fk_class->table }
                     || exists $weak{ $fk_class->table }{ $table };

                $g->add_edge( $table, $fk_class->table );
            }
        }

        if ( !$g->has_vertex( $table ) ) {
            $g->add_vertex( $table );
        }
    }

    return $g;
}

# ------------------------------------------------
sub get_related {
    my $self       = shift;
    my $dest_table = shift or croak 'No destination table';
    my $opts_ref   = shift || {};
    my $this_table = $self->table;
    my $graph      = $self->as_graph;
    my $config     = get_config();

    if ( !$graph->has_vertex( $dest_table ) ) {
        croak "Invalid destination table ($dest_table)";
    }

    # make edges bidirectional
    my @edges = $graph->edges;
    for my $edge ( @edges ) {
        $graph->add_edge( reverse @$edge );
    }

    my %defined_path;
    while ( my ( $key, $val ) = each %{ $config->{'path'} } ) {
        my ( $t1, $t2 ) = sort split $COLON, $key;
        $defined_path{ join $COLON, $t1, $t2 } = [ split $COMMA, $val ];
    }

    my $these_tables = join( $COLON, sort $this_table, $dest_table );

    my @path;
    if ( my $path = $defined_path{ $these_tables } ) {
        if ( $path->[0] eq $this_table ) {
            @path = @{ $path };
        }
        else {
            @path = reverse @{ $path };
        }
    }
    else {
        @path = $graph->SP_Dijkstra( $this_table, $dest_table );
    }

    if ( @path ) {
        my $start_table    = $path[0];
        my $start_class    = table_name_to_class( $start_table );
        my $dest_class     = table_name_to_class( $dest_table );
        my $start_pk_field = $start_class->columns('Primary');
        my $dest_pk_field  = $dest_class->columns('Primary');
        my $sql
            = "select distinct $dest_table.$dest_pk_field from "
            . join( ', ', @path )
            . " where $start_table.$start_pk_field="
            . $self->id;

        my @joins;
        for my $table_num ( 0 .. $#path - 1 ) {
            my $t1                  = $path[ $table_num ];
            my $t2                  = $path[ $table_num + 1 ];
            my $class               = table_name_to_class( $t1 );
            my ( $t1_fld, $t2_fld ) = $class->path_to( $t2 );
            push @joins, "$t1.$t1_fld=$t2.$t2_fld";
        }

        if ( @joins ) {
            $sql .= join(' ', map { " and $_ " } @joins);
        }

        if ( my $order_by = $opts_ref->{'order_by'} ) {
            $sql .= " order by $order_by";
        }

        my $db  = $self->db_Main;
        my $ids = $db->selectcol_arrayref( $sql );

        if ( defined $opts_ref->{'pager'} && ref $opts_ref->{'pager'} ) {
            my $pager = $opts_ref->{'pager'};

            $$pager = Data::Page->new(
                scalar @$ids, 
                $opts_ref->{'items_per_page'} || 25, 
                $opts_ref->{'page_num'} || $opts_ref->{'page'} || 1,
            );

            @$ids = $pager->splice( $ids );
        }

        my @data;
        for my $id ( @$ids ) {
            push @data, $dest_class->retrieve( $id );
        }

        if ( @data ) {
            return wantarray ? @data : $data[0];
        }
        else {
            return wantarray ? () : undef;
        }

        if ( @data ) {
            return wantarray ? @data : $data[0];
        }
        else {
            return wantarray ? () : undef;
        }
    }
    else {
        my $this_table = $self->table;
        croak "No path from $this_table to $dest_table";
    }

    return;
}

# ------------------------------------------------
sub object_type {
    my $self  = shift;
    my $table = $self->table;
    $table =~ s/^.*?_//xms;
    return join ' ', map { ucfirst $_ } split /_/, $table;
}

# ------------------------------------------------

package GDPDM::CDBI::CdvAlleleCuratedAllele;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::CdvAlleleCuratedAllele->table('cdv_allele_curated_allele');

sub cdv_curated_allele {
    my $self = shift;
    return $self->cdv_curated_allele_id;
}

sub cdv_curation {
    my $self = shift;
    return $self->cdv_curation_id;
}

sub div_allele {
    my $self = shift;
    return $self->div_allele_id;
}

GDPDM::CDBI::CdvAlleleCuratedAllele->columns( 
    All => qw/ 
        cdv_allele_curated_allele_id 
        cdv_curated_allele_id
        div_allele_id
        cdv_curation_id 
    /
);

GDPDM::CDBI::CdvAlleleCuratedAllele->columns( 
    Essential => qw/ 
        cdv_allele_curated_allele_id 
        cdv_curated_allele_id
        div_allele_id
        cdv_curation_id 
    /
);

GDPDM::CDBI::CdvAlleleCuratedAllele->columns( 
    Ordered => qw/ 
        cdv_allele_curated_allele_id 
        cdv_curated_allele_id
        div_allele_id
        cdv_curation_id 
    /
);

GDPDM::CDBI::CdvAlleleCuratedAllele->columns( 
    Foreign => qw/ 
        cdv_curated_allele_id
        cdv_curation_id
        div_allele_id 
    /
);

GDPDM::CDBI::CdvAlleleCuratedAllele->columns( 
    Stringify => 'cdv_curated_allele_id' 
);


GDPDM::CDBI::CdvAlleleCuratedAllele->has_a(
    cdv_curated_allele_id => 'GDPDM::CDBI::CdvCuratedAllele'
);

GDPDM::CDBI::CdvAlleleCuratedAllele->has_a(
    cdv_curation_id => 'GDPDM::CDBI::CdvCuration'
);

GDPDM::CDBI::CdvAlleleCuratedAllele->has_a(
    div_allele_id => 'GDPDM::CDBI::DivAllele'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_curated_allele => [ 'cdv_curated_allele_id', 'cdv_curated_allele_id' ],
            cdv_curation => [ 'cdv_curation_id', 'cdv_curation_id' ],
            div_allele => [ 'div_allele_id', 'div_allele_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::CdvCuratedAllele;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::CdvCuratedAllele->table('cdv_curated_allele');

GDPDM::CDBI::CdvCuratedAllele->columns( 
    All => qw/ 
        cdv_curated_allele_id 
        name 
    /
);

GDPDM::CDBI::CdvCuratedAllele->columns( 
    Essential => qw/ 
        cdv_curated_allele_id 
        name 
    /
);

GDPDM::CDBI::CdvCuratedAllele->columns( 
    Ordered => qw/ 
        cdv_curated_allele_id 
        name 
    /
);
GDPDM::CDBI::CdvCuratedAllele->columns( 
    Stringify => 'name' 
);


GDPDM::CDBI::CdvCuratedAllele->has_many( 
    cdv_allele_curated_allele => 'GDPDM::CDBI::CdvAlleleCuratedAllele' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_allele_curated_allele => [ 'cdv_curated_allele_id', 'cdv_curated_allele_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::CdvCuration;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::CdvCuration->table('cdv_curation');

sub cdv_reason {
    my $self = shift;
    return $self->cdv_reason_id;
}

GDPDM::CDBI::CdvCuration->columns( 
    All => qw/ 
        cdv_curation_id 
        cdv_reason_id
        curator
        curation_date
        comments 
    /
);

GDPDM::CDBI::CdvCuration->columns( 
    Essential => qw/ 
        cdv_curation_id 
        cdv_reason_id
        curator
        curation_date 
    /
);

GDPDM::CDBI::CdvCuration->columns( 
    Ordered => qw/ 
        cdv_curation_id 
        cdv_reason_id
        curator
        curation_date
        comments 
    /
);

GDPDM::CDBI::CdvCuration->columns( 
    Foreign => qw/ 
        cdv_reason_id 
    /
);

GDPDM::CDBI::CdvCuration->columns( 
    Stringify => 'curator' 
);


GDPDM::CDBI::CdvCuration->has_a(
    cdv_reason_id => 'GDPDM::CDBI::CdvReason'
);

GDPDM::CDBI::CdvCuration->has_many( 
    cdv_allele_curated_allele => 'GDPDM::CDBI::CdvAlleleCuratedAllele' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_allele_curated_allele => [ 'cdv_curation_id', 'cdv_curation_id' ],
            cdv_reason => [ 'cdv_reason_id', 'cdv_reason_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::CdvMapFeatureAnnotation;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::CdvMapFeatureAnnotation->table('cdv_map_feature_annotation');

sub cdv_map_feature {
    my $self = shift;
    return $self->cdv_map_feature_id;
}

sub cdv_map_feature_annotation_type {
    my $self = shift;
    return $self->cdv_map_feature_annotation_type_id;
}

GDPDM::CDBI::CdvMapFeatureAnnotation->columns( 
    All => qw/ 
        cdv_map_feature_annotation_id 
        cdv_map_feature_id
        cdv_map_feature_annotation_type_id
        annotation_value 
    /
);

GDPDM::CDBI::CdvMapFeatureAnnotation->columns( 
    Essential => qw/ 
        cdv_map_feature_annotation_id 
        cdv_map_feature_id
        cdv_map_feature_annotation_type_id
        annotation_value 
    /
);

GDPDM::CDBI::CdvMapFeatureAnnotation->columns( 
    Ordered => qw/ 
        cdv_map_feature_annotation_id 
        cdv_map_feature_id
        cdv_map_feature_annotation_type_id
        annotation_value 
    /
);

GDPDM::CDBI::CdvMapFeatureAnnotation->columns( 
    Foreign => qw/ 
        cdv_map_feature_id
        cdv_map_feature_annotation_type_id 
    /
);

GDPDM::CDBI::CdvMapFeatureAnnotation->columns( 
    Stringify => 'annotation_value' 
);


GDPDM::CDBI::CdvMapFeatureAnnotation->has_a(
    cdv_map_feature_id => 'GDPDM::CDBI::CdvMapFeature'
);

GDPDM::CDBI::CdvMapFeatureAnnotation->has_a(
    cdv_map_feature_annotation_type_id => 'GDPDM::CDBI::CdvMapFeatureAnnotationType'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_map_feature => [ 'cdv_map_feature_id', 'cdv_map_feature_id' ],
            cdv_map_feature_annotation_type => [ 'cdv_map_feature_annotation_type_id', 'cdv_map_feature_annotation_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::CdvMarkerAnnotation;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::CdvMarkerAnnotation->table('cdv_marker_annotation');

sub cdv_marker {
    my $self = shift;
    return $self->cdv_marker_id;
}

sub cdv_marker_annotation_type {
    my $self = shift;
    return $self->cdv_marker_annotation_type_id;
}

GDPDM::CDBI::CdvMarkerAnnotation->columns( 
    All => qw/ 
        cdv_marker_annotation_id 
        cdv_marker_id
        cdv_marker_annotation_type_id
        annotation_value 
    /
);

GDPDM::CDBI::CdvMarkerAnnotation->columns( 
    Essential => qw/ 
        cdv_marker_annotation_id 
        cdv_marker_id
        cdv_marker_annotation_type_id
        annotation_value 
    /
);

GDPDM::CDBI::CdvMarkerAnnotation->columns( 
    Ordered => qw/ 
        cdv_marker_annotation_id 
        cdv_marker_id
        cdv_marker_annotation_type_id
        annotation_value 
    /
);

GDPDM::CDBI::CdvMarkerAnnotation->columns( 
    Foreign => qw/ 
        cdv_marker_id
        cdv_marker_annotation_type_id 
    /
);

GDPDM::CDBI::CdvMarkerAnnotation->columns( 
    Stringify => 'annotation_value' 
);


GDPDM::CDBI::CdvMarkerAnnotation->has_a(
    cdv_marker_id => 'GDPDM::CDBI::CdvMarker'
);

GDPDM::CDBI::CdvMarkerAnnotation->has_a(
    cdv_marker_annotation_type_id => 'GDPDM::CDBI::CdvMarkerAnnotationType'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_marker => [ 'cdv_marker_id', 'cdv_marker_id' ],
            cdv_marker_annotation_type => [ 'cdv_marker_annotation_type_id', 'cdv_marker_annotation_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivAaAnnotation;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivAaAnnotation->table('div_aa_annotation');

sub div_allele_assay {
    my $self = shift;
    return $self->div_allele_assay_id;
}

sub div_annotation_type {
    my $self = shift;
    return $self->div_annotation_type_id;
}

GDPDM::CDBI::DivAaAnnotation->columns( 
    All => qw/ 
        div_aa_annotation_id 
        div_annotation_type_id
        div_allele_assay_id
        annotation_value 
    /
);

GDPDM::CDBI::DivAaAnnotation->columns( 
    Essential => qw/ 
        div_aa_annotation_id 
        div_annotation_type_id
        div_allele_assay_id
        annotation_value 
    /
);

GDPDM::CDBI::DivAaAnnotation->columns( 
    Ordered => qw/ 
        div_aa_annotation_id 
        div_annotation_type_id
        div_allele_assay_id
        annotation_value 
    /
);

GDPDM::CDBI::DivAaAnnotation->columns( 
    Foreign => qw/ 
        div_allele_assay_id
        div_annotation_type_id 
    /
);

GDPDM::CDBI::DivAaAnnotation->columns( 
    Stringify => 'annotation_value' 
);


GDPDM::CDBI::DivAaAnnotation->has_a(
    div_allele_assay_id => 'GDPDM::CDBI::DivAlleleAssay'
);

GDPDM::CDBI::DivAaAnnotation->has_a(
    div_annotation_type_id => 'GDPDM::CDBI::DivAnnotationType'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_allele_assay => [ 'div_allele_assay_id', 'div_allele_assay_id' ],
            div_annotation_type => [ 'div_annotation_type_id', 'div_annotation_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivAllele;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivAllele->table('div_allele');

sub div_allele_assay {
    my $self = shift;
    return $self->div_allele_assay_id;
}

sub div_obs_unit_sample {
    my $self = shift;
    return $self->div_obs_unit_sample_id;
}

GDPDM::CDBI::DivAllele->columns( 
    All => qw/ 
        div_allele_id 
        div_obs_unit_sample_id
        div_allele_assay_id
        accession
        referencedb
        allele_num
        quality
        value
        proportion
        total_n 
    /
);

GDPDM::CDBI::DivAllele->columns( 
    Essential => qw/ 
        div_allele_id 
        div_obs_unit_sample_id
        div_allele_assay_id
        accession
        referencedb
        allele_num
        quality
        value
        proportion
        total_n 
    /
);

GDPDM::CDBI::DivAllele->columns( 
    Ordered => qw/ 
        div_allele_id 
        div_obs_unit_sample_id
        div_allele_assay_id
        accession
        referencedb
        allele_num
        quality
        value
        proportion
        total_n 
    /
);

GDPDM::CDBI::DivAllele->columns( 
    Foreign => qw/ 
        div_allele_assay_id
        div_obs_unit_sample_id 
    /
);

GDPDM::CDBI::DivAllele->columns( 
    Stringify => 'accession' 
);


GDPDM::CDBI::DivAllele->has_a(
    div_allele_assay_id => 'GDPDM::CDBI::DivAlleleAssay'
);

GDPDM::CDBI::DivAllele->has_a(
    div_obs_unit_sample_id => 'GDPDM::CDBI::DivObsUnitSample'
);

GDPDM::CDBI::DivAllele->has_many( 
    cdv_allele_curated_allele => 'GDPDM::CDBI::CdvAlleleCuratedAllele' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_allele_curated_allele => [ 'div_allele_id', 'div_allele_id' ],
            div_allele_assay => [ 'div_allele_assay_id', 'div_allele_assay_id' ],
            div_obs_unit_sample => [ 'div_obs_unit_sample_id', 'div_obs_unit_sample_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivAlleleAssay;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivAlleleAssay->table('div_allele_assay');

sub cdv_marker {
    my $self = shift;
    return $self->cdv_marker_id;
}

sub div_source_assay {
    my $self = shift;
    return $self->div_source_assay_id;
}

sub div_poly_type {
    my $self = shift;
    return $self->div_poly_type_id;
}

sub div_scoring_tech_type {
    my $self = shift;
    return $self->div_scoring_tech_type_id;
}

sub div_ref_stock {
    my $self = shift;
    return $self->div_ref_stock_id;
}

GDPDM::CDBI::DivAlleleAssay->columns( 
    All => qw/ 
        div_allele_assay_id 
        div_source_assay_id
        div_poly_type_id
        div_scoring_tech_type_id
        div_ref_stock_id
        cdv_marker_id
        name
        assay_date
        producer
        position
        ref_seq
        length
        comments 
    /
);

GDPDM::CDBI::DivAlleleAssay->columns( 
    Essential => qw/ 
        div_allele_assay_id 
        div_source_assay_id
        div_poly_type_id
        div_scoring_tech_type_id
        div_ref_stock_id
        cdv_marker_id
        name
        assay_date
        producer
        position
        length 
    /
);

GDPDM::CDBI::DivAlleleAssay->columns( 
    Ordered => qw/ 
        div_allele_assay_id 
        div_source_assay_id
        div_poly_type_id
        div_scoring_tech_type_id
        div_ref_stock_id
        cdv_marker_id
        name
        assay_date
        producer
        position
        ref_seq
        length
        comments 
    /
);

GDPDM::CDBI::DivAlleleAssay->columns( 
    Foreign => qw/ 
        cdv_marker_id
        div_source_assay_id
        div_poly_type_id
        div_scoring_tech_type_id
        div_ref_stock_id 
    /
);

GDPDM::CDBI::DivAlleleAssay->columns( 
    Stringify => 'name' 
);


GDPDM::CDBI::DivAlleleAssay->has_a(
    cdv_marker_id => 'GDPDM::CDBI::CdvMarker'
);

GDPDM::CDBI::DivAlleleAssay->has_a(
    div_source_assay_id => 'GDPDM::CDBI::DivAlleleAssay'
);

GDPDM::CDBI::DivAlleleAssay->has_a(
    div_poly_type_id => 'GDPDM::CDBI::DivPolyType'
);

GDPDM::CDBI::DivAlleleAssay->has_a(
    div_scoring_tech_type_id => 'GDPDM::CDBI::DivScoringTechType'
);

GDPDM::CDBI::DivAlleleAssay->has_a(
    div_ref_stock_id => 'GDPDM::CDBI::DivStock'
);

GDPDM::CDBI::DivAlleleAssay->has_many( 
    div_aa_annotation => 'GDPDM::CDBI::DivAaAnnotation' 
);

GDPDM::CDBI::DivAlleleAssay->has_many( 
    div_allele => 'GDPDM::CDBI::DivAllele' 
);

GDPDM::CDBI::DivAlleleAssay->has_many( 
    div_allele_assay => 'GDPDM::CDBI::DivAlleleAssay' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_marker => [ 'cdv_marker_id', 'cdv_marker_id' ],
            div_aa_annotation => [ 'div_allele_assay_id', 'div_allele_assay_id' ],
            div_allele => [ 'div_allele_assay_id', 'div_allele_assay_id' ],
            div_allele_assay => [ 'div_source_assay_id', 'div_allele_assay_id' ],
            div_poly_type => [ 'div_poly_type_id', 'div_poly_type_id' ],
            div_scoring_tech_type => [ 'div_scoring_tech_type_id', 'div_scoring_tech_type_id' ],
            div_stock => [ 'div_ref_stock_id', 'div_stock_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::QtlMarker;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::QtlMarker->table('qtl_marker');

sub cdv_marker {
    my $self = shift;
    return $self->cdv_marker_id;
}

sub qtl_qtl {
    my $self = shift;
    return $self->qtl_qtl_id;
}

GDPDM::CDBI::QtlMarker->columns( 
    All => qw/ 
        qtl_marker_id 
        qtl_qtl_id
        cdv_marker_id
        genetic_relationship
        comments 
    /
);

GDPDM::CDBI::QtlMarker->columns( 
    Essential => qw/ 
        qtl_marker_id 
        qtl_qtl_id
        cdv_marker_id
        genetic_relationship 
    /
);

GDPDM::CDBI::QtlMarker->columns( 
    Ordered => qw/ 
        qtl_marker_id 
        qtl_qtl_id
        cdv_marker_id
        genetic_relationship
        comments 
    /
);

GDPDM::CDBI::QtlMarker->columns( 
    Foreign => qw/ 
        cdv_marker_id
        qtl_qtl_id 
    /
);

GDPDM::CDBI::QtlMarker->columns( 
    Stringify => 'genetic_relationship' 
);


GDPDM::CDBI::QtlMarker->has_a(
    cdv_marker_id => 'GDPDM::CDBI::CdvMarker'
);

GDPDM::CDBI::QtlMarker->has_a(
    qtl_qtl_id => 'GDPDM::CDBI::QtlQtl'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_marker => [ 'cdv_marker_id', 'cdv_marker_id' ],
            qtl_qtl => [ 'qtl_qtl_id', 'qtl_qtl_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::CdvMarker;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::CdvMarker->table('cdv_marker');

sub cdv_map_feature {
    my $self = shift;
    return $self->cdv_map_feature_id;
}

sub div_ref_stock {
    my $self = shift;
    return $self->div_ref_stock_id;
}

GDPDM::CDBI::CdvMarker->columns( 
    All => qw/ 
        cdv_marker_id 
        cdv_map_feature_id
        div_ref_stock_id
        name
        position
        length
        ref_seq
        marker_aid 
    /
);

GDPDM::CDBI::CdvMarker->columns( 
    Essential => qw/ 
        cdv_marker_id 
        cdv_map_feature_id
        div_ref_stock_id
        name
        position
        length
        marker_aid 
    /
);

GDPDM::CDBI::CdvMarker->columns( 
    Ordered => qw/ 
        cdv_marker_id 
        cdv_map_feature_id
        div_ref_stock_id
        name
        position
        length
        ref_seq
        marker_aid 
    /
);

GDPDM::CDBI::CdvMarker->columns( 
    Foreign => qw/ 
        cdv_map_feature_id
        div_ref_stock_id 
    /
);

GDPDM::CDBI::CdvMarker->columns( 
    Stringify => 'name' 
);


GDPDM::CDBI::CdvMarker->has_a(
    cdv_map_feature_id => 'GDPDM::CDBI::CdvMapFeature'
);

GDPDM::CDBI::CdvMarker->has_a(
    div_ref_stock_id => 'GDPDM::CDBI::DivStock'
);

GDPDM::CDBI::CdvMarker->has_many( 
    cdv_marker_annotation => 'GDPDM::CDBI::CdvMarkerAnnotation' 
);

GDPDM::CDBI::CdvMarker->has_many( 
    div_allele_assay => 'GDPDM::CDBI::DivAlleleAssay' 
);

GDPDM::CDBI::CdvMarker->has_many( 
    qtl_marker => 'GDPDM::CDBI::QtlMarker' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_map_feature => [ 'cdv_map_feature_id', 'cdv_map_feature_id' ],
            cdv_marker_annotation => [ 'cdv_marker_id', 'cdv_marker_id' ],
            div_allele_assay => [ 'cdv_marker_id', 'cdv_marker_id' ],
            div_stock => [ 'div_ref_stock_id', 'div_stock_id' ],
            qtl_marker => [ 'cdv_marker_id', 'cdv_marker_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::CdvMapFeature;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::CdvMapFeature->table('cdv_map_feature');

GDPDM::CDBI::CdvMapFeature->columns( 
    All => qw/ 
        cdv_map_feature_id 
        name
        chromosome_name
        genetic_bin
        xref_map_id
        genetic_map
        genetic_position
        locus_type
        physical_position
        comments 
    /
);

GDPDM::CDBI::CdvMapFeature->columns( 
    Essential => qw/ 
        cdv_map_feature_id 
        name
        chromosome_name
        genetic_bin
        xref_map_id
        genetic_map
        genetic_position
        locus_type
        physical_position 
    /
);

GDPDM::CDBI::CdvMapFeature->columns( 
    Ordered => qw/ 
        cdv_map_feature_id 
        name
        chromosome_name
        genetic_bin
        xref_map_id
        genetic_map
        genetic_position
        locus_type
        physical_position
        comments 
    /
);
GDPDM::CDBI::CdvMapFeature->columns( 
    Stringify => 'name' 
);


GDPDM::CDBI::CdvMapFeature->has_many( 
    cdv_map_feature_annotation => 'GDPDM::CDBI::CdvMapFeatureAnnotation' 
);

GDPDM::CDBI::CdvMapFeature->has_many( 
    cdv_marker => 'GDPDM::CDBI::CdvMarker' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_map_feature_annotation => [ 'cdv_map_feature_id', 'cdv_map_feature_id' ],
            cdv_marker => [ 'cdv_map_feature_id', 'cdv_map_feature_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::CdvMapFeatureAnnotationType;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::CdvMapFeatureAnnotationType->table('cdv_map_feature_annotation_type');

GDPDM::CDBI::CdvMapFeatureAnnotationType->columns( 
    All => qw/ 
        cdv_map_feature_annotation_type_id 
        anno_type 
    /
);

GDPDM::CDBI::CdvMapFeatureAnnotationType->columns( 
    Essential => qw/ 
        cdv_map_feature_annotation_type_id 
        anno_type 
    /
);

GDPDM::CDBI::CdvMapFeatureAnnotationType->columns( 
    Ordered => qw/ 
        cdv_map_feature_annotation_type_id 
        anno_type 
    /
);
GDPDM::CDBI::CdvMapFeatureAnnotationType->columns( 
    Stringify => 'anno_type' 
);


GDPDM::CDBI::CdvMapFeatureAnnotationType->has_many( 
    cdv_map_feature_annotation => 'GDPDM::CDBI::CdvMapFeatureAnnotation' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_map_feature_annotation => [ 'cdv_map_feature_annotation_type_id', 'cdv_map_feature_annotation_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::CdvMarkerAnnotationType;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::CdvMarkerAnnotationType->table('cdv_marker_annotation_type');

GDPDM::CDBI::CdvMarkerAnnotationType->columns( 
    All => qw/ 
        cdv_marker_annotation_type_id 
        anno_type 
    /
);

GDPDM::CDBI::CdvMarkerAnnotationType->columns( 
    Essential => qw/ 
        cdv_marker_annotation_type_id 
        anno_type 
    /
);

GDPDM::CDBI::CdvMarkerAnnotationType->columns( 
    Ordered => qw/ 
        cdv_marker_annotation_type_id 
        anno_type 
    /
);
GDPDM::CDBI::CdvMarkerAnnotationType->columns( 
    Stringify => 'anno_type' 
);


GDPDM::CDBI::CdvMarkerAnnotationType->has_many( 
    cdv_marker_annotation => 'GDPDM::CDBI::CdvMarkerAnnotation' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_marker_annotation => [ 'cdv_marker_annotation_type_id', 'cdv_marker_annotation_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::CdvPassportSet;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::CdvPassportSet->table('cdv_passport_set');

sub cdv_passport_group {
    my $self = shift;
    return $self->cdv_passport_group_id;
}

sub div_passport {
    my $self = shift;
    return $self->div_passport_id;
}

GDPDM::CDBI::CdvPassportSet->columns( 
    All => qw/ 
        cdv_passport_set_id 
        div_passport_id
        cdv_passport_group_id 
    /
);

GDPDM::CDBI::CdvPassportSet->columns( 
    Essential => qw/ 
        cdv_passport_set_id 
        div_passport_id
        cdv_passport_group_id 
    /
);

GDPDM::CDBI::CdvPassportSet->columns( 
    Ordered => qw/ 
        cdv_passport_set_id 
        div_passport_id
        cdv_passport_group_id 
    /
);

GDPDM::CDBI::CdvPassportSet->columns( 
    Foreign => qw/ 
        cdv_passport_group_id
        div_passport_id 
    /
);

GDPDM::CDBI::CdvPassportSet->columns( 
    Stringify => 'div_passport_id' 
);


GDPDM::CDBI::CdvPassportSet->has_a(
    cdv_passport_group_id => 'GDPDM::CDBI::CdvPassportGroup'
);

GDPDM::CDBI::CdvPassportSet->has_a(
    div_passport_id => 'GDPDM::CDBI::DivPassport'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_passport_group => [ 'cdv_passport_group_id', 'cdv_passport_group_id' ],
            div_passport => [ 'div_passport_id', 'div_passport_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::CdvPassportGroup;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::CdvPassportGroup->table('cdv_passport_group');

GDPDM::CDBI::CdvPassportGroup->columns( 
    All => qw/ 
        cdv_passport_group_id 
        group_name 
    /
);

GDPDM::CDBI::CdvPassportGroup->columns( 
    Essential => qw/ 
        cdv_passport_group_id 
        group_name 
    /
);

GDPDM::CDBI::CdvPassportGroup->columns( 
    Ordered => qw/ 
        cdv_passport_group_id 
        group_name 
    /
);
GDPDM::CDBI::CdvPassportGroup->columns( 
    Stringify => 'group_name' 
);


GDPDM::CDBI::CdvPassportGroup->has_many( 
    cdv_passport_set => 'GDPDM::CDBI::CdvPassportSet' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_passport_set => [ 'cdv_passport_group_id', 'cdv_passport_group_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::CdvReason;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::CdvReason->table('cdv_reason');

GDPDM::CDBI::CdvReason->columns( 
    All => qw/ 
        cdv_reason_id 
        description 
    /
);

GDPDM::CDBI::CdvReason->columns( 
    Essential => qw/ 
        cdv_reason_id 
        description 
    /
);

GDPDM::CDBI::CdvReason->columns( 
    Ordered => qw/ 
        cdv_reason_id 
        description 
    /
);
GDPDM::CDBI::CdvReason->columns( 
    Stringify => 'description' 
);


GDPDM::CDBI::CdvReason->has_many( 
    cdv_curation => 'GDPDM::CDBI::CdvCuration' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_curation => [ 'cdv_reason_id', 'cdv_reason_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivObsUnitSample;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivObsUnitSample->table('div_obs_unit_sample');

sub div_obs_unit {
    my $self = shift;
    return $self->div_obs_unit_id;
}

GDPDM::CDBI::DivObsUnitSample->columns( 
    All => qw/ 
        div_obs_unit_sample_id 
        div_obs_unit_id
        name
        sample_date
        producer
        comments 
    /
);

GDPDM::CDBI::DivObsUnitSample->columns( 
    Essential => qw/ 
        div_obs_unit_sample_id 
        div_obs_unit_id
        name
        sample_date
        producer 
    /
);

GDPDM::CDBI::DivObsUnitSample->columns( 
    Ordered => qw/ 
        div_obs_unit_sample_id 
        div_obs_unit_id
        name
        sample_date
        producer
        comments 
    /
);

GDPDM::CDBI::DivObsUnitSample->columns( 
    Foreign => qw/ 
        div_obs_unit_id 
    /
);

GDPDM::CDBI::DivObsUnitSample->columns( 
    Stringify => 'name' 
);


GDPDM::CDBI::DivObsUnitSample->has_a(
    div_obs_unit_id => 'GDPDM::CDBI::DivObsUnit'
);

GDPDM::CDBI::DivObsUnitSample->has_many( 
    div_allele => 'GDPDM::CDBI::DivAllele' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_allele => [ 'div_obs_unit_sample_id', 'div_obs_unit_sample_id' ],
            div_obs_unit => [ 'div_obs_unit_id', 'div_obs_unit_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivTrait;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivTrait->table('div_trait');

sub div_obs_unit {
    my $self = shift;
    return $self->div_obs_unit_id;
}

sub div_statistic_type {
    my $self = shift;
    return $self->div_statistic_type_id;
}

sub div_trait_uom {
    my $self = shift;
    return $self->div_trait_uom_id;
}

GDPDM::CDBI::DivTrait->columns( 
    All => qw/ 
        div_trait_id 
        div_trait_uom_id
        div_statistic_type_id
        div_obs_unit_id
        value
        date_measured 
    /
);

GDPDM::CDBI::DivTrait->columns( 
    Essential => qw/ 
        div_trait_id 
        div_trait_uom_id
        div_statistic_type_id
        div_obs_unit_id
        value
        date_measured 
    /
);

GDPDM::CDBI::DivTrait->columns( 
    Ordered => qw/ 
        div_trait_id 
        div_trait_uom_id
        div_statistic_type_id
        div_obs_unit_id
        value
        date_measured 
    /
);

GDPDM::CDBI::DivTrait->columns( 
    Foreign => qw/ 
        div_obs_unit_id
        div_statistic_type_id
        div_trait_uom_id 
    /
);

GDPDM::CDBI::DivTrait->columns( 
    Stringify => 'value' 
);


GDPDM::CDBI::DivTrait->has_a(
    div_obs_unit_id => 'GDPDM::CDBI::DivObsUnit'
);

GDPDM::CDBI::DivTrait->has_a(
    div_statistic_type_id => 'GDPDM::CDBI::DivStatisticType'
);

GDPDM::CDBI::DivTrait->has_a(
    div_trait_uom_id => 'GDPDM::CDBI::DivTraitUom'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_obs_unit => [ 'div_obs_unit_id', 'div_obs_unit_id' ],
            div_statistic_type => [ 'div_statistic_type_id', 'div_statistic_type_id' ],
            div_trait_uom => [ 'div_trait_uom_id', 'div_trait_uom_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivTreatment;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivTreatment->table('div_treatment');

sub div_obs_unit {
    my $self = shift;
    return $self->div_obs_unit_id;
}

sub div_treatment_uom {
    my $self = shift;
    return $self->div_treatment_uom_id;
}

GDPDM::CDBI::DivTreatment->columns( 
    All => qw/ 
        div_treatment_id 
        div_treatment_uom_id
        div_obs_unit_id
        value 
    /
);

GDPDM::CDBI::DivTreatment->columns( 
    Essential => qw/ 
        div_treatment_id 
        div_treatment_uom_id
        div_obs_unit_id
        value 
    /
);

GDPDM::CDBI::DivTreatment->columns( 
    Ordered => qw/ 
        div_treatment_id 
        div_treatment_uom_id
        div_obs_unit_id
        value 
    /
);

GDPDM::CDBI::DivTreatment->columns( 
    Foreign => qw/ 
        div_obs_unit_id
        div_treatment_uom_id 
    /
);

GDPDM::CDBI::DivTreatment->columns( 
    Stringify => 'value' 
);


GDPDM::CDBI::DivTreatment->has_a(
    div_obs_unit_id => 'GDPDM::CDBI::DivObsUnit'
);

GDPDM::CDBI::DivTreatment->has_a(
    div_treatment_uom_id => 'GDPDM::CDBI::DivTreatmentUom'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_obs_unit => [ 'div_obs_unit_id', 'div_obs_unit_id' ],
            div_treatment_uom => [ 'div_treatment_uom_id', 'div_treatment_uom_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivObsUnit;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivObsUnit->table('div_obs_unit');

sub div_experiment {
    my $self = shift;
    return $self->div_experiment_id;
}

sub div_locality {
    my $self = shift;
    return $self->div_locality_id;
}

sub div_stock {
    my $self = shift;
    return $self->div_stock_id;
}

GDPDM::CDBI::DivObsUnit->columns( 
    All => qw/ 
        div_obs_unit_id 
        div_experiment_id
        div_stock_id
        div_locality_id
        name
        coord_x
        coord_y
        rep
        block
        plot
        season
        plant
        planting_date
        harvest_date
        comments 
    /
);

GDPDM::CDBI::DivObsUnit->columns( 
    Essential => qw/ 
        div_obs_unit_id 
        div_experiment_id
        div_stock_id
        div_locality_id
        name
        coord_x
        coord_y
        rep
        block
        plot
        season
        plant
        planting_date
        harvest_date 
    /
);

GDPDM::CDBI::DivObsUnit->columns( 
    Ordered => qw/ 
        div_obs_unit_id 
        div_experiment_id
        div_stock_id
        div_locality_id
        name
        coord_x
        coord_y
        rep
        block
        plot
        season
        plant
        planting_date
        harvest_date
        comments 
    /
);

GDPDM::CDBI::DivObsUnit->columns( 
    Foreign => qw/ 
        div_experiment_id
        div_locality_id
        div_stock_id 
    /
);

GDPDM::CDBI::DivObsUnit->columns( 
    Stringify => 'name' 
);


GDPDM::CDBI::DivObsUnit->has_a(
    div_experiment_id => 'GDPDM::CDBI::DivExperiment'
);

GDPDM::CDBI::DivObsUnit->has_a(
    div_locality_id => 'GDPDM::CDBI::DivLocality'
);

GDPDM::CDBI::DivObsUnit->has_a(
    div_stock_id => 'GDPDM::CDBI::DivStock'
);

GDPDM::CDBI::DivObsUnit->has_many( 
    div_obs_unit_sample => 'GDPDM::CDBI::DivObsUnitSample' 
);

GDPDM::CDBI::DivObsUnit->has_many( 
    div_trait => 'GDPDM::CDBI::DivTrait' 
);

GDPDM::CDBI::DivObsUnit->has_many( 
    div_treatment => 'GDPDM::CDBI::DivTreatment' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_experiment => [ 'div_experiment_id', 'div_experiment_id' ],
            div_locality => [ 'div_locality_id', 'div_locality_id' ],
            div_obs_unit_sample => [ 'div_obs_unit_id', 'div_obs_unit_id' ],
            div_stock => [ 'div_stock_id', 'div_stock_id' ],
            div_trait => [ 'div_obs_unit_id', 'div_obs_unit_id' ],
            div_treatment => [ 'div_obs_unit_id', 'div_obs_unit_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::QtlQtl;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::QtlQtl->table('qtl_qtl');

sub qtl_analysis_result {
    my $self = shift;
    return $self->qtl_analysis_result_id;
}

GDPDM::CDBI::QtlQtl->columns( 
    All => qw/ 
        qtl_qtl_id 
        qtl_analysis_result_id
        qtl_accession_id
        published_symbol
        linkage_group
        start_position
        peak_position
        stop_position
        additive
        dominance
        lod
        f_value
        t_value
        p_value
        r_square
        comments 
    /
);

GDPDM::CDBI::QtlQtl->columns( 
    Essential => qw/ 
        qtl_qtl_id 
        qtl_analysis_result_id
        qtl_accession_id
        published_symbol
        linkage_group
        start_position
        peak_position
        stop_position
        additive
        dominance
        lod
        f_value
        t_value
        p_value
        r_square 
    /
);

GDPDM::CDBI::QtlQtl->columns( 
    Ordered => qw/ 
        qtl_qtl_id 
        qtl_analysis_result_id
        qtl_accession_id
        published_symbol
        linkage_group
        start_position
        peak_position
        stop_position
        additive
        dominance
        lod
        f_value
        t_value
        p_value
        r_square
        comments 
    /
);

GDPDM::CDBI::QtlQtl->columns( 
    Foreign => qw/ 
        qtl_analysis_result_id 
    /
);


GDPDM::CDBI::QtlQtl->columns( 
    Unique => qw/ 
        qtl_accession_id 
    /
);

GDPDM::CDBI::QtlQtl->columns( 
    Stringify => 'published_symbol' 
);


GDPDM::CDBI::QtlQtl->has_a(
    qtl_analysis_result_id => 'GDPDM::CDBI::QtlAnalysisResult'
);

GDPDM::CDBI::QtlQtl->has_many( 
    qtl_marker => 'GDPDM::CDBI::QtlMarker' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            qtl_analysis_result => [ 'qtl_analysis_result_id', 'qtl_analysis_result_id' ],
            qtl_marker => [ 'qtl_qtl_id', 'qtl_qtl_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::QtlAnalysisResult;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::QtlAnalysisResult->table('qtl_analysis_result');

sub div_trait_uom {
    my $self = shift;
    return $self->div_trait_uom_id;
}

sub qtl_analysis_method {
    my $self = shift;
    return $self->qtl_analysis_method_id;
}

sub qtl_obs_unit {
    my $self = shift;
    return $self->qtl_obs_unit_id;
}

GDPDM::CDBI::QtlAnalysisResult->columns( 
    All => qw/ 
        qtl_analysis_result_id 
        qtl_obs_unit_id
        qtl_analysis_method_id
        div_trait_uom_id
        r_square
        genetic_var
        gxe_var
        environment_var 
    /
);

GDPDM::CDBI::QtlAnalysisResult->columns( 
    Essential => qw/ 
        qtl_analysis_result_id 
        qtl_obs_unit_id
        qtl_analysis_method_id
        div_trait_uom_id
        r_square
        genetic_var
        gxe_var
        environment_var 
    /
);

GDPDM::CDBI::QtlAnalysisResult->columns( 
    Ordered => qw/ 
        qtl_analysis_result_id 
        qtl_obs_unit_id
        qtl_analysis_method_id
        div_trait_uom_id
        r_square
        genetic_var
        gxe_var
        environment_var 
    /
);

GDPDM::CDBI::QtlAnalysisResult->columns( 
    Foreign => qw/ 
        div_trait_uom_id
        qtl_analysis_method_id
        qtl_obs_unit_id 
    /
);

GDPDM::CDBI::QtlAnalysisResult->columns( 
    Stringify => 'r_square' 
);


GDPDM::CDBI::QtlAnalysisResult->has_a(
    div_trait_uom_id => 'GDPDM::CDBI::DivTraitUom'
);

GDPDM::CDBI::QtlAnalysisResult->has_a(
    qtl_analysis_method_id => 'GDPDM::CDBI::QtlAnalysisMethod'
);

GDPDM::CDBI::QtlAnalysisResult->has_a(
    qtl_obs_unit_id => 'GDPDM::CDBI::QtlObsUnit'
);

GDPDM::CDBI::QtlAnalysisResult->has_many( 
    qtl_qtl => 'GDPDM::CDBI::QtlQtl' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_trait_uom => [ 'div_trait_uom_id', 'div_trait_uom_id' ],
            qtl_analysis_method => [ 'qtl_analysis_method_id', 'qtl_analysis_method_id' ],
            qtl_obs_unit => [ 'qtl_obs_unit_id', 'qtl_obs_unit_id' ],
            qtl_qtl => [ 'qtl_analysis_result_id', 'qtl_analysis_result_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::QtlAnalysisMethod;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::QtlAnalysisMethod->table('qtl_analysis_method');

sub positive_parent_stock {
    my $self = shift;
    return $self->positive_parent_stock_id;
}

sub qtl_study {
    my $self = shift;
    return $self->qtl_study_id;
}

GDPDM::CDBI::QtlAnalysisMethod->columns( 
    All => qw/ 
        qtl_analysis_method_id 
        qtl_study_id
        threshold_method
        permutation_number
        statistical_model
        positive_parent_stock_id
        comments 
    /
);

GDPDM::CDBI::QtlAnalysisMethod->columns( 
    Essential => qw/ 
        qtl_analysis_method_id 
        qtl_study_id
        threshold_method
        permutation_number
        statistical_model
        positive_parent_stock_id 
    /
);

GDPDM::CDBI::QtlAnalysisMethod->columns( 
    Ordered => qw/ 
        qtl_analysis_method_id 
        qtl_study_id
        threshold_method
        permutation_number
        statistical_model
        positive_parent_stock_id
        comments 
    /
);

GDPDM::CDBI::QtlAnalysisMethod->columns( 
    Foreign => qw/ 
        positive_parent_stock_id
        qtl_study_id 
    /
);

GDPDM::CDBI::QtlAnalysisMethod->columns( 
    Stringify => 'threshold_method' 
);


GDPDM::CDBI::QtlAnalysisMethod->has_a(
    positive_parent_stock_id => 'GDPDM::CDBI::DivStockParent'
);

GDPDM::CDBI::QtlAnalysisMethod->has_a(
    qtl_study_id => 'GDPDM::CDBI::QtlStudy'
);

GDPDM::CDBI::QtlAnalysisMethod->has_many( 
    qtl_analysis_result => 'GDPDM::CDBI::QtlAnalysisResult' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_stock_parent => [ 'positive_parent_stock_id', 'div_stock_parent_id' ],
            qtl_analysis_result => [ 'qtl_analysis_method_id', 'qtl_analysis_method_id' ],
            qtl_study => [ 'qtl_study_id', 'qtl_study_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivStockParent;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivStockParent->table('div_stock_parent');

sub div_stock {
    my $self = shift;
    return $self->div_stock_id;
}

GDPDM::CDBI::DivStockParent->columns( 
    All => qw/ 
        div_stock_parent_id 
        div_stock_id
        div_parent_id
        role
        recurrent 
    /
);

GDPDM::CDBI::DivStockParent->columns( 
    Essential => qw/ 
        div_stock_parent_id 
        div_stock_id
        div_parent_id
        role
        recurrent 
    /
);

GDPDM::CDBI::DivStockParent->columns( 
    Ordered => qw/ 
        div_stock_parent_id 
        div_stock_id
        div_parent_id
        role
        recurrent 
    /
);

GDPDM::CDBI::DivStockParent->columns( 
    Foreign => qw/ 
        div_stock_id 
    /
);

GDPDM::CDBI::DivStockParent->columns( 
    Stringify => 'role' 
);


GDPDM::CDBI::DivStockParent->has_a(
    div_stock_id => 'GDPDM::CDBI::DivStock'
);

GDPDM::CDBI::DivStockParent->has_many( 
    qtl_analysis_method => 'GDPDM::CDBI::QtlAnalysisMethod' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_stock => [ 'div_parent_id', 'div_stock_id' ],
            qtl_analysis_method => [ 'div_stock_parent_id', 'positive_parent_stock_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivStock;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivStock->table('div_stock');

sub div_generation {
    my $self = shift;
    return $self->div_generation_id;
}

sub div_passport {
    my $self = shift;
    return $self->div_passport_id;
}

GDPDM::CDBI::DivStock->columns( 
    All => qw/ 
        div_stock_id 
        div_generation_id
        div_passport_id
        seed_lot
        stock_source
        comments 
    /
);

GDPDM::CDBI::DivStock->columns( 
    Essential => qw/ 
        div_stock_id 
        div_generation_id
        div_passport_id
        seed_lot
        stock_source 
    /
);

GDPDM::CDBI::DivStock->columns( 
    Ordered => qw/ 
        div_stock_id 
        div_generation_id
        div_passport_id
        seed_lot
        stock_source
        comments 
    /
);

GDPDM::CDBI::DivStock->columns( 
    Foreign => qw/ 
        div_generation_id
        div_passport_id 
    /
);

GDPDM::CDBI::DivStock->columns( 
    Stringify => 'seed_lot' 
);


GDPDM::CDBI::DivStock->has_a(
    div_generation_id => 'GDPDM::CDBI::DivGeneration'
);

GDPDM::CDBI::DivStock->has_a(
    div_passport_id => 'GDPDM::CDBI::DivPassport'
);

GDPDM::CDBI::DivStock->has_many( 
    cdv_marker => 'GDPDM::CDBI::CdvMarker' 
);

GDPDM::CDBI::DivStock->has_many( 
    div_allele_assay => 'GDPDM::CDBI::DivAlleleAssay' 
);

GDPDM::CDBI::DivStock->has_many( 
    div_obs_unit => 'GDPDM::CDBI::DivObsUnit' 
);

GDPDM::CDBI::DivStock->has_many( 
    div_stock_parent => 'GDPDM::CDBI::DivStockParent' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_marker => [ 'div_stock_id', 'div_ref_stock_id' ],
            div_allele_assay => [ 'div_stock_id', 'div_ref_stock_id' ],
            div_generation => [ 'div_generation_id', 'div_generation_id' ],
            div_obs_unit => [ 'div_stock_id', 'div_stock_id' ],
            div_passport => [ 'div_passport_id', 'div_passport_id' ],
            div_stock_parent => [ 'div_stock_id', 'div_parent_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivSynonym;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivSynonym->table('div_synonym');

sub div_passport {
    my $self = shift;
    return $self->div_passport_id;
}

GDPDM::CDBI::DivSynonym->columns( 
    All => qw/ 
        div_synonym_id 
        div_passport_id
        synonym
        comments 
    /
);

GDPDM::CDBI::DivSynonym->columns( 
    Essential => qw/ 
        div_synonym_id 
        div_passport_id
        synonym 
    /
);

GDPDM::CDBI::DivSynonym->columns( 
    Ordered => qw/ 
        div_synonym_id 
        div_passport_id
        synonym
        comments 
    /
);

GDPDM::CDBI::DivSynonym->columns( 
    Foreign => qw/ 
        div_passport_id 
    /
);

GDPDM::CDBI::DivSynonym->columns( 
    Stringify => 'synonym' 
);


GDPDM::CDBI::DivSynonym->has_a(
    div_passport_id => 'GDPDM::CDBI::DivPassport'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_passport => [ 'div_passport_id', 'div_passport_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivPassport;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivPassport->table('div_passport');

sub cdv_source {
    my $self = shift;
    return $self->cdv_source_id;
}

sub div_accession_collecting {
    my $self = shift;
    return $self->div_accession_collecting_id;
}

sub div_taxonomy {
    my $self = shift;
    return $self->div_taxonomy_id;
}

GDPDM::CDBI::DivPassport->columns( 
    All => qw/ 
        div_passport_id 
        div_taxonomy_id
        div_accession_collecting_id
        cdv_source_id
        accename
        germplasm_type
        source
        accenumb
        sampstat
        comments 
    /
);

GDPDM::CDBI::DivPassport->columns( 
    Essential => qw/ 
        div_passport_id 
        div_taxonomy_id
        div_accession_collecting_id
        cdv_source_id
        accename
        germplasm_type
        source
        accenumb
        sampstat 
    /
);

GDPDM::CDBI::DivPassport->columns( 
    Ordered => qw/ 
        div_passport_id 
        div_taxonomy_id
        div_accession_collecting_id
        cdv_source_id
        accename
        germplasm_type
        source
        accenumb
        sampstat
        comments 
    /
);

GDPDM::CDBI::DivPassport->columns( 
    Foreign => qw/ 
        cdv_source_id
        div_accession_collecting_id
        div_taxonomy_id 
    /
);


GDPDM::CDBI::DivPassport->columns( 
    Unique => qw/ 
        accename 
    /
);

GDPDM::CDBI::DivPassport->columns( 
    Stringify => 'accename' 
);


GDPDM::CDBI::DivPassport->has_a(
    cdv_source_id => 'GDPDM::CDBI::CdvSource'
);

GDPDM::CDBI::DivPassport->has_a(
    div_accession_collecting_id => 'GDPDM::CDBI::DivAccessionCollecting'
);

GDPDM::CDBI::DivPassport->has_a(
    div_taxonomy_id => 'GDPDM::CDBI::DivTaxonomy'
);

GDPDM::CDBI::DivPassport->has_many( 
    cdv_passport_set => 'GDPDM::CDBI::CdvPassportSet' 
);

GDPDM::CDBI::DivPassport->has_many( 
    div_stock => 'GDPDM::CDBI::DivStock' 
);

GDPDM::CDBI::DivPassport->has_many( 
    div_synonym => 'GDPDM::CDBI::DivSynonym' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            cdv_passport_set => [ 'div_passport_id', 'div_passport_id' ],
            cdv_source => [ 'cdv_source_id', 'cdv_source_id' ],
            div_accession_collecting => [ 'div_accession_collecting_id', 'div_accession_collecting_id' ],
            div_stock => [ 'div_passport_id', 'div_passport_id' ],
            div_synonym => [ 'div_passport_id', 'div_passport_id' ],
            div_taxonomy => [ 'div_taxonomy_id', 'div_taxonomy_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::CdvSource;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::CdvSource->table('cdv_source');

GDPDM::CDBI::CdvSource->columns( 
    All => qw/ 
        cdv_source_id 
        source
        contact
        institute
        department
        address
        city
        state_province
        country
        phone
        fax
        email
        url
        comments 
    /
);

GDPDM::CDBI::CdvSource->columns( 
    Essential => qw/ 
        cdv_source_id 
        source
        contact
        institute
        department
        address
        city
        state_province
        country
        phone
        fax
        email
        url 
    /
);

GDPDM::CDBI::CdvSource->columns( 
    Ordered => qw/ 
        cdv_source_id 
        source
        contact
        institute
        department
        address
        city
        state_province
        country
        phone
        fax
        email
        url
        comments 
    /
);
GDPDM::CDBI::CdvSource->columns( 
    Stringify => 'source' 
);


GDPDM::CDBI::CdvSource->has_many( 
    div_passport => 'GDPDM::CDBI::DivPassport' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_passport => [ 'cdv_source_id', 'cdv_source_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivAccessionCollecting;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivAccessionCollecting->table('div_accession_collecting');

sub div_locality {
    my $self = shift;
    return $self->div_locality_id;
}

GDPDM::CDBI::DivAccessionCollecting->columns( 
    All => qw/ 
        div_accession_collecting_id 
        div_locality_id
        collector
        collnumb
        collsrc
        collcode
        col_date 
    /
);

GDPDM::CDBI::DivAccessionCollecting->columns( 
    Essential => qw/ 
        div_accession_collecting_id 
        div_locality_id
        collector
        collnumb
        collsrc
        collcode
        col_date 
    /
);

GDPDM::CDBI::DivAccessionCollecting->columns( 
    Ordered => qw/ 
        div_accession_collecting_id 
        div_locality_id
        collector
        collnumb
        collsrc
        collcode
        col_date 
    /
);

GDPDM::CDBI::DivAccessionCollecting->columns( 
    Foreign => qw/ 
        div_locality_id 
    /
);

GDPDM::CDBI::DivAccessionCollecting->columns( 
    Stringify => 'collector' 
);


GDPDM::CDBI::DivAccessionCollecting->has_a(
    div_locality_id => 'GDPDM::CDBI::DivLocality'
);

GDPDM::CDBI::DivAccessionCollecting->has_many( 
    div_passport => 'GDPDM::CDBI::DivPassport' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_locality => [ 'div_locality_id', 'div_locality_id' ],
            div_passport => [ 'div_accession_collecting_id', 'div_accession_collecting_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivAnnotationType;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivAnnotationType->table('div_annotation_type');

GDPDM::CDBI::DivAnnotationType->columns( 
    All => qw/ 
        div_annotation_type_id 
        anno_type 
    /
);

GDPDM::CDBI::DivAnnotationType->columns( 
    Essential => qw/ 
        div_annotation_type_id 
        anno_type 
    /
);

GDPDM::CDBI::DivAnnotationType->columns( 
    Ordered => qw/ 
        div_annotation_type_id 
        anno_type 
    /
);
GDPDM::CDBI::DivAnnotationType->columns( 
    Stringify => 'anno_type' 
);


GDPDM::CDBI::DivAnnotationType->has_many( 
    div_aa_annotation => 'GDPDM::CDBI::DivAaAnnotation' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_aa_annotation => [ 'div_annotation_type_id', 'div_annotation_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::QtlObsUnitSet;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::QtlObsUnitSet->table('qtl_obs_unit_set');

sub qtl_obs_unit {
    my $self = shift;
    return $self->qtl_obs_unit_id;
}

GDPDM::CDBI::QtlObsUnitSet->columns( 
    All => qw/ 
        qtl_obs_unit_set_id 
        qtl_obs_unit_id
        name
        comments 
    /
);

GDPDM::CDBI::QtlObsUnitSet->columns( 
    Essential => qw/ 
        qtl_obs_unit_set_id 
        qtl_obs_unit_id
        name 
    /
);

GDPDM::CDBI::QtlObsUnitSet->columns( 
    Ordered => qw/ 
        qtl_obs_unit_set_id 
        qtl_obs_unit_id
        name
        comments 
    /
);

GDPDM::CDBI::QtlObsUnitSet->columns( 
    Foreign => qw/ 
        qtl_obs_unit_id 
    /
);


GDPDM::CDBI::QtlObsUnitSet->columns( 
    Unique => qw/ 
        name 
    /
);

GDPDM::CDBI::QtlObsUnitSet->columns( 
    Stringify => 'name' 
);


GDPDM::CDBI::QtlObsUnitSet->has_a(
    qtl_obs_unit_id => 'GDPDM::CDBI::QtlObsUnit'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            qtl_obs_unit => [ 'qtl_obs_unit_id', 'qtl_obs_unit_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::QtlTrait;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::QtlTrait->table('qtl_trait');

sub div_statistic_type {
    my $self = shift;
    return $self->div_statistic_type_id;
}

sub div_trait_uom {
    my $self = shift;
    return $self->div_trait_uom_id;
}

sub qtl_obs_unit {
    my $self = shift;
    return $self->qtl_obs_unit_id;
}

GDPDM::CDBI::QtlTrait->columns( 
    All => qw/ 
        qtl_trait_id 
        div_trait_uom_id
        div_statistic_type_id
        qtl_obs_unit_id
        date_measured
        value 
    /
);

GDPDM::CDBI::QtlTrait->columns( 
    Essential => qw/ 
        qtl_trait_id 
        div_trait_uom_id
        div_statistic_type_id
        qtl_obs_unit_id
        date_measured
        value 
    /
);

GDPDM::CDBI::QtlTrait->columns( 
    Ordered => qw/ 
        qtl_trait_id 
        div_trait_uom_id
        div_statistic_type_id
        qtl_obs_unit_id
        date_measured
        value 
    /
);

GDPDM::CDBI::QtlTrait->columns( 
    Foreign => qw/ 
        div_statistic_type_id
        div_trait_uom_id
        qtl_obs_unit_id 
    /
);

GDPDM::CDBI::QtlTrait->columns( 
    Stringify => 'date_measured' 
);


GDPDM::CDBI::QtlTrait->has_a(
    div_statistic_type_id => 'GDPDM::CDBI::DivStatisticType'
);

GDPDM::CDBI::QtlTrait->has_a(
    div_trait_uom_id => 'GDPDM::CDBI::DivTraitUom'
);

GDPDM::CDBI::QtlTrait->has_a(
    qtl_obs_unit_id => 'GDPDM::CDBI::QtlObsUnit'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_statistic_type => [ 'div_statistic_type_id', 'div_statistic_type_id' ],
            div_trait_uom => [ 'div_trait_uom_id', 'div_trait_uom_id' ],
            qtl_obs_unit => [ 'qtl_obs_unit_id', 'qtl_obs_unit_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::QtlObsUnit;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::QtlObsUnit->table('qtl_obs_unit');

sub qtl_study {
    my $self = shift;
    return $self->qtl_study_id;
}

GDPDM::CDBI::QtlObsUnit->columns( 
    All => qw/ 
        qtl_obs_unit_id 
        qtl_study_id
        name
        comments 
    /
);

GDPDM::CDBI::QtlObsUnit->columns( 
    Essential => qw/ 
        qtl_obs_unit_id 
        qtl_study_id
        name 
    /
);

GDPDM::CDBI::QtlObsUnit->columns( 
    Ordered => qw/ 
        qtl_obs_unit_id 
        qtl_study_id
        name
        comments 
    /
);

GDPDM::CDBI::QtlObsUnit->columns( 
    Foreign => qw/ 
        qtl_study_id 
    /
);


GDPDM::CDBI::QtlObsUnit->columns( 
    Unique => qw/ 
        name 
    /
);

GDPDM::CDBI::QtlObsUnit->columns( 
    Stringify => 'name' 
);


GDPDM::CDBI::QtlObsUnit->has_a(
    qtl_study_id => 'GDPDM::CDBI::QtlStudy'
);

GDPDM::CDBI::QtlObsUnit->has_many( 
    qtl_analysis_result => 'GDPDM::CDBI::QtlAnalysisResult' 
);

GDPDM::CDBI::QtlObsUnit->has_many( 
    qtl_obs_unit_set => 'GDPDM::CDBI::QtlObsUnitSet' 
);

GDPDM::CDBI::QtlObsUnit->has_many( 
    qtl_trait => 'GDPDM::CDBI::QtlTrait' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            qtl_analysis_result => [ 'qtl_obs_unit_id', 'qtl_obs_unit_id' ],
            qtl_obs_unit_set => [ 'qtl_obs_unit_id', 'qtl_obs_unit_id' ],
            qtl_study => [ 'qtl_study_id', 'qtl_study_id' ],
            qtl_trait => [ 'qtl_obs_unit_id', 'qtl_obs_unit_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::QtlStudy;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::QtlStudy->table('qtl_study');

sub div_experiment {
    my $self = shift;
    return $self->div_experiment_id;
}

GDPDM::CDBI::QtlStudy->columns( 
    All => qw/ 
        qtl_study_id 
        name
        reference_id
        div_experiment_id
        cmap_map_set_acc 
    /
);

GDPDM::CDBI::QtlStudy->columns( 
    Essential => qw/ 
        qtl_study_id 
        name
        reference_id
        div_experiment_id
        cmap_map_set_acc 
    /
);

GDPDM::CDBI::QtlStudy->columns( 
    Ordered => qw/ 
        qtl_study_id 
        name
        reference_id
        div_experiment_id
        cmap_map_set_acc 
    /
);

GDPDM::CDBI::QtlStudy->columns( 
    Foreign => qw/ 
        div_experiment_id 
    /
);

GDPDM::CDBI::QtlStudy->columns( 
    Stringify => 'name' 
);


GDPDM::CDBI::QtlStudy->has_a(
    div_experiment_id => 'GDPDM::CDBI::DivExperiment'
);

GDPDM::CDBI::QtlStudy->has_many( 
    qtl_analysis_method => 'GDPDM::CDBI::QtlAnalysisMethod' 
);

GDPDM::CDBI::QtlStudy->has_many( 
    qtl_obs_unit => 'GDPDM::CDBI::QtlObsUnit' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_experiment => [ 'div_experiment_id', 'div_experiment_id' ],
            qtl_analysis_method => [ 'qtl_study_id', 'qtl_study_id' ],
            qtl_obs_unit => [ 'qtl_study_id', 'qtl_study_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivExperiment;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivExperiment->table('div_experiment');

GDPDM::CDBI::DivExperiment->columns( 
    All => qw/ 
        div_experiment_id 
        name
        design
        originator
        comments 
    /
);

GDPDM::CDBI::DivExperiment->columns( 
    Essential => qw/ 
        div_experiment_id 
        name
        design
        originator 
    /
);

GDPDM::CDBI::DivExperiment->columns( 
    Ordered => qw/ 
        div_experiment_id 
        name
        design
        originator
        comments 
    /
);
GDPDM::CDBI::DivExperiment->columns( 
    Stringify => 'name' 
);


GDPDM::CDBI::DivExperiment->has_many( 
    div_obs_unit => 'GDPDM::CDBI::DivObsUnit' 
);

GDPDM::CDBI::DivExperiment->has_many( 
    qtl_study => 'GDPDM::CDBI::QtlStudy' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_obs_unit => [ 'div_experiment_id', 'div_experiment_id' ],
            qtl_study => [ 'div_experiment_id', 'div_experiment_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivGeneration;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivGeneration->table('div_generation');

GDPDM::CDBI::DivGeneration->columns( 
    All => qw/ 
        div_generation_id 
        value
        icis_id
        comments
        selfing_number
        sibbing_number 
    /
);

GDPDM::CDBI::DivGeneration->columns( 
    Essential => qw/ 
        div_generation_id 
        value
        icis_id
        selfing_number
        sibbing_number 
    /
);

GDPDM::CDBI::DivGeneration->columns( 
    Ordered => qw/ 
        div_generation_id 
        value
        icis_id
        comments
        selfing_number
        sibbing_number 
    /
);
GDPDM::CDBI::DivGeneration->columns( 
    Stringify => 'value' 
);


GDPDM::CDBI::DivGeneration->has_many( 
    div_stock => 'GDPDM::CDBI::DivStock' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_stock => [ 'div_generation_id', 'div_generation_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivLocality;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivLocality->table('div_locality');

GDPDM::CDBI::DivLocality->columns( 
    All => qw/ 
        div_locality_id 
        elevation
        city
        country
        origcty
        latitude
        longitude
        locality_name
        state_province 
    /
);

GDPDM::CDBI::DivLocality->columns( 
    Essential => qw/ 
        div_locality_id 
        elevation
        city
        country
        origcty
        latitude
        longitude
        locality_name
        state_province 
    /
);

GDPDM::CDBI::DivLocality->columns( 
    Ordered => qw/ 
        div_locality_id 
        elevation
        city
        country
        origcty
        latitude
        longitude
        locality_name
        state_province 
    /
);
sub stringify_self {
    my $self   = shift;
    my $string = join ', ', map { $self->get($_) || () } qw[ city country ];
    return $string || $self->id;
}


GDPDM::CDBI::DivLocality->has_many( 
    div_accession_collecting => 'GDPDM::CDBI::DivAccessionCollecting' 
);

GDPDM::CDBI::DivLocality->has_many( 
    div_obs_unit => 'GDPDM::CDBI::DivObsUnit' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_accession_collecting => [ 'div_locality_id', 'div_locality_id' ],
            div_obs_unit => [ 'div_locality_id', 'div_locality_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivPolyType;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivPolyType->table('div_poly_type');

GDPDM::CDBI::DivPolyType->columns( 
    All => qw/ 
        div_poly_type_id 
        poly_type 
    /
);

GDPDM::CDBI::DivPolyType->columns( 
    Essential => qw/ 
        div_poly_type_id 
        poly_type 
    /
);

GDPDM::CDBI::DivPolyType->columns( 
    Ordered => qw/ 
        div_poly_type_id 
        poly_type 
    /
);
GDPDM::CDBI::DivPolyType->columns( 
    Stringify => 'poly_type' 
);


GDPDM::CDBI::DivPolyType->has_many( 
    div_allele_assay => 'GDPDM::CDBI::DivAlleleAssay' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_allele_assay => [ 'div_poly_type_id', 'div_poly_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivScoringTechType;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivScoringTechType->table('div_scoring_tech_type');

GDPDM::CDBI::DivScoringTechType->columns( 
    All => qw/ 
        div_scoring_tech_type_id 
        scoring_tech_group
        scoring_tech_type 
    /
);

GDPDM::CDBI::DivScoringTechType->columns( 
    Essential => qw/ 
        div_scoring_tech_type_id 
        scoring_tech_group
        scoring_tech_type 
    /
);

GDPDM::CDBI::DivScoringTechType->columns( 
    Ordered => qw/ 
        div_scoring_tech_type_id 
        scoring_tech_group
        scoring_tech_type 
    /
);
GDPDM::CDBI::DivScoringTechType->columns( 
    Stringify => 'scoring_tech_group' 
);


GDPDM::CDBI::DivScoringTechType->has_many( 
    div_allele_assay => 'GDPDM::CDBI::DivAlleleAssay' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_allele_assay => [ 'div_scoring_tech_type_id', 'div_scoring_tech_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivStatisticType;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivStatisticType->table('div_statistic_type');

GDPDM::CDBI::DivStatisticType->columns( 
    All => qw/ 
        div_statistic_type_id 
        stat_type 
    /
);

GDPDM::CDBI::DivStatisticType->columns( 
    Essential => qw/ 
        div_statistic_type_id 
        stat_type 
    /
);

GDPDM::CDBI::DivStatisticType->columns( 
    Ordered => qw/ 
        div_statistic_type_id 
        stat_type 
    /
);
GDPDM::CDBI::DivStatisticType->columns( 
    Stringify => 'stat_type' 
);


GDPDM::CDBI::DivStatisticType->has_many( 
    div_trait => 'GDPDM::CDBI::DivTrait' 
);

GDPDM::CDBI::DivStatisticType->has_many( 
    qtl_trait => 'GDPDM::CDBI::QtlTrait' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_trait => [ 'div_statistic_type_id', 'div_statistic_type_id' ],
            qtl_trait => [ 'div_statistic_type_id', 'div_statistic_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivTaxonomy;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivTaxonomy->table('div_taxonomy');

GDPDM::CDBI::DivTaxonomy->columns( 
    All => qw/ 
        div_taxonomy_id 
        genus
        species
        subspecies
        subtaxa
        race
        population
        common_name
        term_accession 
    /
);

GDPDM::CDBI::DivTaxonomy->columns( 
    Essential => qw/ 
        div_taxonomy_id 
        genus
        species
        subspecies
        subtaxa
        race
        population
        common_name
        term_accession 
    /
);

GDPDM::CDBI::DivTaxonomy->columns( 
    Ordered => qw/ 
        div_taxonomy_id 
        genus
        species
        subspecies
        subtaxa
        race
        population
        common_name
        term_accession 
    /
);
sub stringify_self {
    my $self = shift;
    return join ' ', map { $self->get($_) || '' } 
        qw[ genus species subspecies ];
}


GDPDM::CDBI::DivTaxonomy->has_many( 
    div_passport => 'GDPDM::CDBI::DivPassport' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_passport => [ 'div_taxonomy_id', 'div_taxonomy_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivTraitUom;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivTraitUom->table('div_trait_uom');

sub div_unit_of_measure {
    my $self = shift;
    return $self->div_unit_of_measure_id;
}

GDPDM::CDBI::DivTraitUom->columns( 
    All => qw/ 
        div_trait_uom_id 
        div_unit_of_measure_id
        local_trait_name
        trait_protocol
        to_accession
        eo_accession 
    /
);

GDPDM::CDBI::DivTraitUom->columns( 
    Essential => qw/ 
        div_trait_uom_id 
        div_unit_of_measure_id
        local_trait_name
        trait_protocol
        to_accession
        eo_accession 
    /
);

GDPDM::CDBI::DivTraitUom->columns( 
    Ordered => qw/ 
        div_trait_uom_id 
        div_unit_of_measure_id
        local_trait_name
        trait_protocol
        to_accession
        eo_accession 
    /
);

GDPDM::CDBI::DivTraitUom->columns( 
    Foreign => qw/ 
        div_unit_of_measure_id 
    /
);

GDPDM::CDBI::DivTraitUom->columns( 
    Stringify => 'local_trait_name' 
);


GDPDM::CDBI::DivTraitUom->has_a(
    div_unit_of_measure_id => 'GDPDM::CDBI::DivUnitOfMeasure'
);

GDPDM::CDBI::DivTraitUom->has_many( 
    div_trait => 'GDPDM::CDBI::DivTrait' 
);

GDPDM::CDBI::DivTraitUom->has_many( 
    qtl_analysis_result => 'GDPDM::CDBI::QtlAnalysisResult' 
);

GDPDM::CDBI::DivTraitUom->has_many( 
    qtl_trait => 'GDPDM::CDBI::QtlTrait' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_trait => [ 'div_trait_uom_id', 'div_trait_uom_id' ],
            div_unit_of_measure => [ 'div_unit_of_measure_id', 'div_unit_of_measure_id' ],
            qtl_analysis_result => [ 'div_trait_uom_id', 'div_trait_uom_id' ],
            qtl_trait => [ 'div_trait_uom_id', 'div_trait_uom_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivTreatmentUom;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivTreatmentUom->table('div_treatment_uom');

sub div_unit_of_measure {
    my $self = shift;
    return $self->div_unit_of_measure_id;
}

GDPDM::CDBI::DivTreatmentUom->columns( 
    All => qw/ 
        div_treatment_uom_id 
        div_unit_of_measure_id
        local_treatment_name
        treatment_protocol
        eo_accession 
    /
);

GDPDM::CDBI::DivTreatmentUom->columns( 
    Essential => qw/ 
        div_treatment_uom_id 
        div_unit_of_measure_id
        local_treatment_name
        treatment_protocol
        eo_accession 
    /
);

GDPDM::CDBI::DivTreatmentUom->columns( 
    Ordered => qw/ 
        div_treatment_uom_id 
        div_unit_of_measure_id
        local_treatment_name
        treatment_protocol
        eo_accession 
    /
);

GDPDM::CDBI::DivTreatmentUom->columns( 
    Foreign => qw/ 
        div_unit_of_measure_id 
    /
);

GDPDM::CDBI::DivTreatmentUom->columns( 
    Stringify => 'local_treatment_name' 
);


GDPDM::CDBI::DivTreatmentUom->has_a(
    div_unit_of_measure_id => 'GDPDM::CDBI::DivUnitOfMeasure'
);

GDPDM::CDBI::DivTreatmentUom->has_many( 
    div_treatment => 'GDPDM::CDBI::DivTreatment' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_treatment => [ 'div_treatment_uom_id', 'div_treatment_uom_id' ],
            div_unit_of_measure => [ 'div_unit_of_measure_id', 'div_unit_of_measure_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::DivUnitOfMeasure;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::DivUnitOfMeasure->table('div_unit_of_measure');

GDPDM::CDBI::DivUnitOfMeasure->columns( 
    All => qw/ 
        div_unit_of_measure_id 
        unit_type 
    /
);

GDPDM::CDBI::DivUnitOfMeasure->columns( 
    Essential => qw/ 
        div_unit_of_measure_id 
        unit_type 
    /
);

GDPDM::CDBI::DivUnitOfMeasure->columns( 
    Ordered => qw/ 
        div_unit_of_measure_id 
        unit_type 
    /
);
GDPDM::CDBI::DivUnitOfMeasure->columns( 
    Stringify => 'unit_type' 
);


GDPDM::CDBI::DivUnitOfMeasure->has_many( 
    div_trait_uom => 'GDPDM::CDBI::DivTraitUom' 
);

GDPDM::CDBI::DivUnitOfMeasure->has_many( 
    div_treatment_uom => 'GDPDM::CDBI::DivTreatmentUom' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            div_trait_uom => [ 'div_unit_of_measure_id', 'div_unit_of_measure_id' ],
            div_treatment_uom => [ 'div_unit_of_measure_id', 'div_unit_of_measure_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::Xref;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::Xref->table('xref');

sub xref_type {
    my $self = shift;
    return $self->xref_type_id;
}

GDPDM::CDBI::Xref->columns( 
    All => qw/ 
        xref_id 
        table_name
        record_id
        xref_type_id
        xref_value
        comments 
    /
);

GDPDM::CDBI::Xref->columns( 
    Essential => qw/ 
        xref_id 
        table_name
        record_id
        xref_type_id
        xref_value 
    /
);

GDPDM::CDBI::Xref->columns( 
    Ordered => qw/ 
        xref_id 
        table_name
        record_id
        xref_type_id
        xref_value
        comments 
    /
);

GDPDM::CDBI::Xref->columns( 
    Foreign => qw/ 
        xref_type_id 
    /
);

GDPDM::CDBI::Xref->columns( 
    Stringify => 'table_name' 
);


GDPDM::CDBI::Xref->has_a(
    xref_type_id => 'GDPDM::CDBI::XrefType'
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            xref_type => [ 'xref_type_id', 'xref_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}
# ------------------------------------------------

package GDPDM::CDBI::XrefType;

use base qw( GDPDM::CDBI );

GDPDM::CDBI::XrefType->table('xref_type');

GDPDM::CDBI::XrefType->columns( 
    All => qw/ 
        xref_type_id 
        xref_type
        url_template 
    /
);

GDPDM::CDBI::XrefType->columns( 
    Essential => qw/ 
        xref_type_id 
        xref_type
        url_template 
    /
);

GDPDM::CDBI::XrefType->columns( 
    Ordered => qw/ 
        xref_type_id 
        xref_type
        url_template 
    /
);

GDPDM::CDBI::XrefType->columns( 
    Unique => qw/ 
        xref_type 
    /
);

GDPDM::CDBI::XrefType->columns( 
    Stringify => 'xref_type' 
);


GDPDM::CDBI::XrefType->has_many( 
    xref => 'GDPDM::CDBI::Xref' 
);


sub path_to {
    my $self     = shift;
    my $fk_table = shift or return;
    my %path_to  = (
            xref => [ 'xref_type_id', 'xref_type_id' ],
    );

    if ( my $fields = $path_to{ $fk_table } ) {
        return @$fields;
    }
    else {
        return;
    }
    return;
}


1;

__END__

# ----------------------------------------------------

=pod

=head1 NAME

GDPDM::CDBI - Class::DBI classes for GDPDM schema

=head1 VERSION

This documentation refers to version $Revision: 1.12 $.

=head1 SYNOPSIS

  use GDPDM::CDBI;

  my @passport = GDPDM::CDBI::DivPassport->retrieve_all;
  my $tax      = GDPDM::CDBI::DivTaxonomy->retrieve( 42 );

=head1 DESCRIPTION

This module contains all the Class::DBI classes for the GDPDM schema.

=head1 METHODS

=head2 db_Main

  GDPDM::CDBI->db_Main('alternate_database');

Use this to indicate an alternate database as defined by the string
indicating the section of the config file with the connection info.

=head2 current_db_name

  my $db_name = GDPDM::CDBI->current_db_name;

Returns the configuration key name of the current database.

=head2 represented_tables

  my @table_names = GDPDM::CDBI->represented_tables;

Returns an array of the tables represented in the classes.  You can
use GDPDM::Utils::table_name_to_class to convert to the Class::DBI
class name.

=head2 as_graph

Returns the schema represented as a Graph::Directed object where the
tables are vertices and their relationships are the edges.  The tables
must be listed in the L<represented_tables> method.

=head2 get_related

  my @objects = $cdbi->get_related('foo');

This method finds the shortest path through the schema (using
L<as_graph>) to find the objects related to the current object.  That
is, to get the objects related immediately proximate to the current
object (e.g., a foreign key one table away), you would just use the
object's normal accessor.  Say your "foo" object has many "bar"
objects, you would do this:

  my @bars = $foo->bar;

But if the "quux" objects related to "foo" have to go through "bar"
and "baz" first, then this is easier:

  my @quux = $foo->get_related('quux');

=head2 object_type

  my $type = $cdbi->object_type;

Prints a prettied up version of the table name.  Splits on underscores
and then uppercases the first letter of each resulting word, then
joins them on spaces, so "marker_type" would print as "Marker Type."

=head1 CLASSES

=head2 GDPDM::CDBI::CdvAlleleCuratedAllele

Corresponds to the table "cdv_allele_curated_allele."

Fields:

=over 4

=item * cdv_curated_allele_id

=item * div_allele_id

=item * cdv_curation_id

=back


Has A:

=over 4

=item * cdv_curated_allele => GDPDM::CDBI::CdvCuratedAllele

=item * cdv_curation => GDPDM::CDBI::CdvCuration

=item * div_allele => GDPDM::CDBI::DivAllele

=back



=head2 GDPDM::CDBI::CdvCuratedAllele

Corresponds to the table "cdv_curated_allele."

Fields:

=over 4

=item * name

=back



Has Many:

=over 4

=item * cdv_allele_curated_allele => GDPDM::CDBI::CdvAlleleCuratedAllele

=back


=head2 GDPDM::CDBI::CdvCuration

Corresponds to the table "cdv_curation."

Fields:

=over 4

=item * cdv_reason_id

=item * curator

=item * curation_date

=item * comments

=back


Has A:

=over 4

=item * cdv_reason => GDPDM::CDBI::CdvReason

=back


Has Many:

=over 4

=item * cdv_allele_curated_allele => GDPDM::CDBI::CdvAlleleCuratedAllele

=back


=head2 GDPDM::CDBI::CdvMapFeature

Corresponds to the table "cdv_map_feature."

Fields:

=over 4

=item * name

=item * chromosome_name

=item * genetic_bin

=item * xref_map_id

=item * genetic_map

=item * genetic_position

=item * locus_type

=item * physical_position

=item * comments

=back



Has Many:

=over 4

=item * cdv_map_feature_annotation => GDPDM::CDBI::CdvMapFeatureAnnotation

=item * cdv_marker => GDPDM::CDBI::CdvMarker

=back


=head2 GDPDM::CDBI::CdvMapFeatureAnnotation

Corresponds to the table "cdv_map_feature_annotation."

Fields:

=over 4

=item * cdv_map_feature_id

=item * cdv_map_feature_annotation_type_id

=item * annotation_value

=back


Has A:

=over 4

=item * cdv_map_feature => GDPDM::CDBI::CdvMapFeature

=item * cdv_map_feature_annotation_type => GDPDM::CDBI::CdvMapFeatureAnnotationType

=back



=head2 GDPDM::CDBI::CdvMapFeatureAnnotationType

Corresponds to the table "cdv_map_feature_annotation_type."

Fields:

=over 4

=item * anno_type

=back



Has Many:

=over 4

=item * cdv_map_feature_annotation => GDPDM::CDBI::CdvMapFeatureAnnotation

=back


=head2 GDPDM::CDBI::CdvMarker

Corresponds to the table "cdv_marker."

Fields:

=over 4

=item * cdv_map_feature_id

=item * div_ref_stock_id

=item * name

=item * position

=item * length

=item * ref_seq

=item * marker_aid

=back


Has A:

=over 4

=item * cdv_map_feature => GDPDM::CDBI::CdvMapFeature

=item * div_ref_stock => GDPDM::CDBI::DivStock

=back


Has Many:

=over 4

=item * cdv_marker_annotation => GDPDM::CDBI::CdvMarkerAnnotation

=item * div_allele_assay => GDPDM::CDBI::DivAlleleAssay

=item * qtl_marker => GDPDM::CDBI::QtlMarker

=back


=head2 GDPDM::CDBI::CdvMarkerAnnotation

Corresponds to the table "cdv_marker_annotation."

Fields:

=over 4

=item * cdv_marker_id

=item * cdv_marker_annotation_type_id

=item * annotation_value

=back


Has A:

=over 4

=item * cdv_marker => GDPDM::CDBI::CdvMarker

=item * cdv_marker_annotation_type => GDPDM::CDBI::CdvMarkerAnnotationType

=back



=head2 GDPDM::CDBI::CdvMarkerAnnotationType

Corresponds to the table "cdv_marker_annotation_type."

Fields:

=over 4

=item * anno_type

=back



Has Many:

=over 4

=item * cdv_marker_annotation => GDPDM::CDBI::CdvMarkerAnnotation

=back


=head2 GDPDM::CDBI::CdvPassportGroup

Corresponds to the table "cdv_passport_group."

Fields:

=over 4

=item * group_name

=back



Has Many:

=over 4

=item * cdv_passport_set => GDPDM::CDBI::CdvPassportSet

=back


=head2 GDPDM::CDBI::CdvPassportSet

Corresponds to the table "cdv_passport_set."

Fields:

=over 4

=item * div_passport_id

=item * cdv_passport_group_id

=back


Has A:

=over 4

=item * cdv_passport_group => GDPDM::CDBI::CdvPassportGroup

=item * div_passport => GDPDM::CDBI::DivPassport

=back



=head2 GDPDM::CDBI::CdvReason

Corresponds to the table "cdv_reason."

Fields:

=over 4

=item * description

=back



Has Many:

=over 4

=item * cdv_curation => GDPDM::CDBI::CdvCuration

=back


=head2 GDPDM::CDBI::CdvSource

Corresponds to the table "cdv_source."

Fields:

=over 4

=item * source

=item * contact

=item * institute

=item * department

=item * address

=item * city

=item * state_province

=item * country

=item * phone

=item * fax

=item * email

=item * url

=item * comments

=back



Has Many:

=over 4

=item * div_passport => GDPDM::CDBI::DivPassport

=back


=head2 GDPDM::CDBI::DivAaAnnotation

Corresponds to the table "div_aa_annotation."

Fields:

=over 4

=item * div_annotation_type_id

=item * div_allele_assay_id

=item * annotation_value

=back


Has A:

=over 4

=item * div_allele_assay => GDPDM::CDBI::DivAlleleAssay

=item * div_annotation_type => GDPDM::CDBI::DivAnnotationType

=back



=head2 GDPDM::CDBI::DivAccessionCollecting

Corresponds to the table "div_accession_collecting."

Fields:

=over 4

=item * div_locality_id

=item * collector

=item * collnumb

=item * collsrc

=item * collcode

=item * col_date

=back


Has A:

=over 4

=item * div_locality => GDPDM::CDBI::DivLocality

=back


Has Many:

=over 4

=item * div_passport => GDPDM::CDBI::DivPassport

=back


=head2 GDPDM::CDBI::DivAllele

Corresponds to the table "div_allele."

Fields:

=over 4

=item * div_obs_unit_sample_id

=item * div_allele_assay_id

=item * accession

=item * referencedb

=item * allele_num

=item * quality

=item * value

=item * proportion

=item * total_n

=back


Has A:

=over 4

=item * div_allele_assay => GDPDM::CDBI::DivAlleleAssay

=item * div_obs_unit_sample => GDPDM::CDBI::DivObsUnitSample

=back


Has Many:

=over 4

=item * cdv_allele_curated_allele => GDPDM::CDBI::CdvAlleleCuratedAllele

=back


=head2 GDPDM::CDBI::DivAlleleAssay

Corresponds to the table "div_allele_assay."

Fields:

=over 4

=item * div_source_assay_id

=item * div_poly_type_id

=item * div_scoring_tech_type_id

=item * div_ref_stock_id

=item * cdv_marker_id

=item * name

=item * assay_date

=item * producer

=item * position

=item * ref_seq

=item * length

=item * comments

=back


Has A:

=over 4

=item * cdv_marker => GDPDM::CDBI::CdvMarker

=item * div_source_assay => GDPDM::CDBI::DivAlleleAssay

=item * div_poly_type => GDPDM::CDBI::DivPolyType

=item * div_scoring_tech_type => GDPDM::CDBI::DivScoringTechType

=item * div_ref_stock => GDPDM::CDBI::DivStock

=back


Has Many:

=over 4

=item * div_aa_annotation => GDPDM::CDBI::DivAaAnnotation

=item * div_allele => GDPDM::CDBI::DivAllele

=item * div_allele_assay => GDPDM::CDBI::DivAlleleAssay

=back


=head2 GDPDM::CDBI::DivAnnotationType

Corresponds to the table "div_annotation_type."

Fields:

=over 4

=item * anno_type

=back



Has Many:

=over 4

=item * div_aa_annotation => GDPDM::CDBI::DivAaAnnotation

=back


=head2 GDPDM::CDBI::DivExperiment

Corresponds to the table "div_experiment."

Fields:

=over 4

=item * name

=item * design

=item * originator

=item * comments

=back



Has Many:

=over 4

=item * div_obs_unit => GDPDM::CDBI::DivObsUnit

=item * qtl_study => GDPDM::CDBI::QtlStudy

=back


=head2 GDPDM::CDBI::DivGeneration

Corresponds to the table "div_generation."

Fields:

=over 4

=item * value

=item * icis_id

=item * comments

=item * selfing_number

=item * sibbing_number

=back



Has Many:

=over 4

=item * div_stock => GDPDM::CDBI::DivStock

=back


=head2 GDPDM::CDBI::DivLocality

Corresponds to the table "div_locality."

Fields:

=over 4

=item * elevation

=item * city

=item * country

=item * origcty

=item * latitude

=item * longitude

=item * locality_name

=item * state_province

=back



Has Many:

=over 4

=item * div_accession_collecting => GDPDM::CDBI::DivAccessionCollecting

=item * div_obs_unit => GDPDM::CDBI::DivObsUnit

=back


=head2 GDPDM::CDBI::DivObsUnit

Corresponds to the table "div_obs_unit."

Fields:

=over 4

=item * div_experiment_id

=item * div_stock_id

=item * div_locality_id

=item * name

=item * coord_x

=item * coord_y

=item * rep

=item * block

=item * plot

=item * season

=item * plant

=item * planting_date

=item * harvest_date

=item * comments

=back


Has A:

=over 4

=item * div_experiment => GDPDM::CDBI::DivExperiment

=item * div_locality => GDPDM::CDBI::DivLocality

=item * div_stock => GDPDM::CDBI::DivStock

=back


Has Many:

=over 4

=item * div_obs_unit_sample => GDPDM::CDBI::DivObsUnitSample

=item * div_trait => GDPDM::CDBI::DivTrait

=item * div_treatment => GDPDM::CDBI::DivTreatment

=back


=head2 GDPDM::CDBI::DivObsUnitSample

Corresponds to the table "div_obs_unit_sample."

Fields:

=over 4

=item * div_obs_unit_id

=item * name

=item * sample_date

=item * producer

=item * comments

=back


Has A:

=over 4

=item * div_obs_unit => GDPDM::CDBI::DivObsUnit

=back


Has Many:

=over 4

=item * div_allele => GDPDM::CDBI::DivAllele

=back


=head2 GDPDM::CDBI::DivPassport

Corresponds to the table "div_passport."

Fields:

=over 4

=item * div_taxonomy_id

=item * div_accession_collecting_id

=item * cdv_source_id

=item * accename

=item * germplasm_type

=item * source

=item * accenumb

=item * sampstat

=item * comments

=back


Has A:

=over 4

=item * cdv_source => GDPDM::CDBI::CdvSource

=item * div_accession_collecting => GDPDM::CDBI::DivAccessionCollecting

=item * div_taxonomy => GDPDM::CDBI::DivTaxonomy

=back


Has Many:

=over 4

=item * cdv_passport_set => GDPDM::CDBI::CdvPassportSet

=item * div_stock => GDPDM::CDBI::DivStock

=item * div_synonym => GDPDM::CDBI::DivSynonym

=back


=head2 GDPDM::CDBI::DivPolyType

Corresponds to the table "div_poly_type."

Fields:

=over 4

=item * poly_type

=back



Has Many:

=over 4

=item * div_allele_assay => GDPDM::CDBI::DivAlleleAssay

=back


=head2 GDPDM::CDBI::DivScoringTechType

Corresponds to the table "div_scoring_tech_type."

Fields:

=over 4

=item * scoring_tech_group

=item * scoring_tech_type

=back



Has Many:

=over 4

=item * div_allele_assay => GDPDM::CDBI::DivAlleleAssay

=back


=head2 GDPDM::CDBI::DivStatisticType

Corresponds to the table "div_statistic_type."

Fields:

=over 4

=item * stat_type

=back



Has Many:

=over 4

=item * div_trait => GDPDM::CDBI::DivTrait

=item * qtl_trait => GDPDM::CDBI::QtlTrait

=back


=head2 GDPDM::CDBI::DivStock

Corresponds to the table "div_stock."

Fields:

=over 4

=item * div_generation_id

=item * div_passport_id

=item * seed_lot

=item * stock_source

=item * comments

=back


Has A:

=over 4

=item * div_generation => GDPDM::CDBI::DivGeneration

=item * div_passport => GDPDM::CDBI::DivPassport

=back


Has Many:

=over 4

=item * cdv_marker => GDPDM::CDBI::CdvMarker

=item * div_allele_assay => GDPDM::CDBI::DivAlleleAssay

=item * div_obs_unit => GDPDM::CDBI::DivObsUnit

=item * div_stock_parent => GDPDM::CDBI::DivStockParent

=back


=head2 GDPDM::CDBI::DivStockParent

Corresponds to the table "div_stock_parent."

Fields:

=over 4

=item * div_stock_id

=item * div_parent_id

=item * role

=item * recurrent

=back


Has A:

=over 4

=item * div_stock => GDPDM::CDBI::DivStock

=back


Has Many:

=over 4

=item * qtl_analysis_method => GDPDM::CDBI::QtlAnalysisMethod

=back


=head2 GDPDM::CDBI::DivSynonym

Corresponds to the table "div_synonym."

Fields:

=over 4

=item * div_passport_id

=item * synonym

=item * comments

=back


Has A:

=over 4

=item * div_passport => GDPDM::CDBI::DivPassport

=back



=head2 GDPDM::CDBI::DivTaxonomy

Corresponds to the table "div_taxonomy."

Fields:

=over 4

=item * genus

=item * species

=item * subspecies

=item * subtaxa

=item * race

=item * population

=item * common_name

=item * term_accession

=back



Has Many:

=over 4

=item * div_passport => GDPDM::CDBI::DivPassport

=back


=head2 GDPDM::CDBI::DivTrait

Corresponds to the table "div_trait."

Fields:

=over 4

=item * div_trait_uom_id

=item * div_statistic_type_id

=item * div_obs_unit_id

=item * value

=item * date_measured

=back


Has A:

=over 4

=item * div_obs_unit => GDPDM::CDBI::DivObsUnit

=item * div_statistic_type => GDPDM::CDBI::DivStatisticType

=item * div_trait_uom => GDPDM::CDBI::DivTraitUom

=back



=head2 GDPDM::CDBI::DivTraitUom

Corresponds to the table "div_trait_uom."

Fields:

=over 4

=item * div_unit_of_measure_id

=item * local_trait_name

=item * trait_protocol

=item * to_accession

=item * eo_accession

=back


Has A:

=over 4

=item * div_unit_of_measure => GDPDM::CDBI::DivUnitOfMeasure

=back


Has Many:

=over 4

=item * div_trait => GDPDM::CDBI::DivTrait

=item * qtl_analysis_result => GDPDM::CDBI::QtlAnalysisResult

=item * qtl_trait => GDPDM::CDBI::QtlTrait

=back


=head2 GDPDM::CDBI::DivTreatment

Corresponds to the table "div_treatment."

Fields:

=over 4

=item * div_treatment_uom_id

=item * div_obs_unit_id

=item * value

=back


Has A:

=over 4

=item * div_obs_unit => GDPDM::CDBI::DivObsUnit

=item * div_treatment_uom => GDPDM::CDBI::DivTreatmentUom

=back



=head2 GDPDM::CDBI::DivTreatmentUom

Corresponds to the table "div_treatment_uom."

Fields:

=over 4

=item * div_unit_of_measure_id

=item * local_treatment_name

=item * treatment_protocol

=item * eo_accession

=back


Has A:

=over 4

=item * div_unit_of_measure => GDPDM::CDBI::DivUnitOfMeasure

=back


Has Many:

=over 4

=item * div_treatment => GDPDM::CDBI::DivTreatment

=back


=head2 GDPDM::CDBI::DivUnitOfMeasure

Corresponds to the table "div_unit_of_measure."

Fields:

=over 4

=item * unit_type

=back



Has Many:

=over 4

=item * div_trait_uom => GDPDM::CDBI::DivTraitUom

=item * div_treatment_uom => GDPDM::CDBI::DivTreatmentUom

=back


=head2 GDPDM::CDBI::QtlAnalysisMethod

Corresponds to the table "qtl_analysis_method."

Fields:

=over 4

=item * qtl_study_id

=item * threshold_method

=item * permutation_number

=item * statistical_model

=item * positive_parent_stock_id

=item * comments

=back


Has A:

=over 4

=item * positive_parent_stock => GDPDM::CDBI::DivStockParent

=item * qtl_study => GDPDM::CDBI::QtlStudy

=back


Has Many:

=over 4

=item * qtl_analysis_result => GDPDM::CDBI::QtlAnalysisResult

=back


=head2 GDPDM::CDBI::QtlAnalysisResult

Corresponds to the table "qtl_analysis_result."

Fields:

=over 4

=item * qtl_obs_unit_id

=item * qtl_analysis_method_id

=item * div_trait_uom_id

=item * r_square

=item * genetic_var

=item * gxe_var

=item * environment_var

=back


Has A:

=over 4

=item * div_trait_uom => GDPDM::CDBI::DivTraitUom

=item * qtl_analysis_method => GDPDM::CDBI::QtlAnalysisMethod

=item * qtl_obs_unit => GDPDM::CDBI::QtlObsUnit

=back


Has Many:

=over 4

=item * qtl_qtl => GDPDM::CDBI::QtlQtl

=back


=head2 GDPDM::CDBI::QtlMarker

Corresponds to the table "qtl_marker."

Fields:

=over 4

=item * qtl_qtl_id

=item * cdv_marker_id

=item * genetic_relationship

=item * comments

=back


Has A:

=over 4

=item * cdv_marker => GDPDM::CDBI::CdvMarker

=item * qtl_qtl => GDPDM::CDBI::QtlQtl

=back



=head2 GDPDM::CDBI::QtlObsUnit

Corresponds to the table "qtl_obs_unit."

Fields:

=over 4

=item * qtl_study_id

=item * name

=item * comments

=back


Has A:

=over 4

=item * qtl_study => GDPDM::CDBI::QtlStudy

=back


Has Many:

=over 4

=item * qtl_analysis_result => GDPDM::CDBI::QtlAnalysisResult

=item * qtl_obs_unit_set => GDPDM::CDBI::QtlObsUnitSet

=item * qtl_trait => GDPDM::CDBI::QtlTrait

=back


=head2 GDPDM::CDBI::QtlObsUnitSet

Corresponds to the table "qtl_obs_unit_set."

Fields:

=over 4

=item * qtl_obs_unit_id

=item * name

=item * comments

=back


Has A:

=over 4

=item * qtl_obs_unit => GDPDM::CDBI::QtlObsUnit

=back



=head2 GDPDM::CDBI::QtlQtl

Corresponds to the table "qtl_qtl."

Fields:

=over 4

=item * qtl_analysis_result_id

=item * qtl_accession_id

=item * published_symbol

=item * linkage_group

=item * start_position

=item * peak_position

=item * stop_position

=item * additive

=item * dominance

=item * lod

=item * f_value

=item * t_value

=item * p_value

=item * r_square

=item * comments

=back


Has A:

=over 4

=item * qtl_analysis_result => GDPDM::CDBI::QtlAnalysisResult

=back


Has Many:

=over 4

=item * qtl_marker => GDPDM::CDBI::QtlMarker

=back


=head2 GDPDM::CDBI::QtlStudy

Corresponds to the table "qtl_study."

Fields:

=over 4

=item * name

=item * reference_id

=item * div_experiment_id

=item * cmap_map_set_acc

=back


Has A:

=over 4

=item * div_experiment => GDPDM::CDBI::DivExperiment

=back


Has Many:

=over 4

=item * qtl_analysis_method => GDPDM::CDBI::QtlAnalysisMethod

=item * qtl_obs_unit => GDPDM::CDBI::QtlObsUnit

=back


=head2 GDPDM::CDBI::QtlTrait

Corresponds to the table "qtl_trait."

Fields:

=over 4

=item * div_trait_uom_id

=item * div_statistic_type_id

=item * qtl_obs_unit_id

=item * date_measured

=item * value

=back


Has A:

=over 4

=item * div_statistic_type => GDPDM::CDBI::DivStatisticType

=item * div_trait_uom => GDPDM::CDBI::DivTraitUom

=item * qtl_obs_unit => GDPDM::CDBI::QtlObsUnit

=back



=head2 GDPDM::CDBI::Xref

Corresponds to the table "xref."

Fields:

=over 4

=item * table_name

=item * record_id

=item * xref_type_id

=item * xref_value

=item * comments

=back


Has A:

=over 4

=item * xref_type => GDPDM::CDBI::XrefType

=back



=head2 GDPDM::CDBI::XrefType

Corresponds to the table "xref_type."

Fields:

=over 4

=item * xref_type

=item * url_template

=back



Has Many:

=over 4

=item * xref => GDPDM::CDBI::Xref

=back


=head1 SEE ALSO

Class::DBI, Class::DBI::Pager.

=head1 AUTHOR

Ken Youens-Clark E<lt>kclark@cshl.eduE<gt>.

=head1 COPYRIGHT

Copyright (c) 2006 Cold Spring Harbor Laboratory

This library is free software;  you can redistribute it and/or modify 
it under the same terms as Perl itself.

=cut
