=head1 NAME

Bio::EnsEMBL::Pipeline::Config::GeneDupl

=head1 SYNOPSIS

    use Bio::EnsEMBL::Pipeline::Config::GeneDupl;
    use Bio::EnsEMBL::Pipeline::Config::GeneDupl qw( GD_HIT_COVERAGE );

=head1 DESCRIPTION

=cut

package Bio::EnsEMBL::Pipeline::Config::GeneDupl;

use strict;
use vars qw( %GeneDupl );

# Hash containing config info
%GeneDupl = 
  (

   # Species specific options:
   #
   # To allow multiple species to be included in the same running pipeline
   # the species-specific config options are held in hashes keyed by the
   # prefix of gene stable ids.  Code that uses this config can auto-detect
   # which set of options to use.  A generic set of options could
   # also be included, keyed on '.*' or some other regex that would
   # match all possible ids.
   #
   # Explanation of these config options:
   #
   # GD_SPECIES - 
   #   Species name of sequences matching to id regex.  For Compara
   #   database output this must be the same as that stored in the
   #   Compara database genome table.
   # GD_BLASTDB_FILE -
   #   Location of the _formatted_ blast database - the formatted fasta file.
   # GD_INGROUP_REGEX -
   #   Regex that will match ids from ingroup species (eg. ENSG for Homo sapiens).
   # GD_OUTGROUP_REGEXES -
   #   Listref of regexes that will match ids from outgroup species.
   # GD_TRANSL_REGEX
   #   Ingroup stable translation id regex
   # GD_HIT_COVERAGE -
   #   Minimum permitted coverage of match sequences.  Defaults to 0.80.
   # GD_HIT_IDENTITY -
   #   Mimimum permitted identity for homology matches.  Defaults to 0.80.
   # GD_DISTANCE_CUTOFF -
   #  Arbritrary genetic distance (Ks) cutoff to use when an outgroup match
   #  has not been found.  Also used when outgroup matches are more 
   #  distant than this cutoff value.  Defaults to 1.0.
   # GD_GENETIC_CODE -
   #  Genetic code to use when translating sequences.  Codes are exactly the
   #  same as those used within BioPerl.  Universal genetic code is 1.  Vertebrate
   #  mitochondrial genetic code is 2.  See the Bio::Tools::CodonTable method 
   #  for a full list of options.
   # GD_OUTPUT_DIR
   #  Location of directory where output files should be written.

   GD_OPTIONS => {'ENSG0'    => {GD_SPECIES          => 'Homo sapiens',
				 GD_BLASTDB_FILE     => '/path/to/blastdb/file',
				 GD_INGROUP_REGEX    => 'ENSG0',
				 GD_OUTGROUP_REGEXES => ['ENSRNOG0', 'ENSMUSG0'],
				 GD_TRANSL_REGEX     => 'ENSP',
				 GD_HIT_COVERAGE     => '80',
				 GD_HIT_IDENTITY     => '50',
				 GD_DISTANCE_CUTOFF  => '0.6',
				 GD_GENETIC_CODE     => '1',
				 GD_OUTPUT_DIR       => '/output/directory',
			       },
		  'ENSRNOG0' => {GD_SPECIES          => 'Rattus norvegicus',
				 GD_BLASTDB_FILE     => '/path/to/blastdb/file',
				 GD_INGROUP_REGEX    => 'ENSRNOG0',
				 GD_OUTGROUP_REGEXES => ['ENSG0'],
				 GD_TRANSL_REGEX     => 'ENSRNOP',
				 GD_HIT_COVERAGE     => '80',
				 GD_HIT_IDENTITY     => '50',
				 GD_DISTANCE_CUTOFF  => '0.6',
				 GD_GENETIC_CODE     => '1',
				 GD_OUTPUT_DIR       => '',
				},
		  'ENSMUSG0' => {GD_SPECIES          => 'Mus musculus',
				 GD_BLASTDB_FILE     => '/path/to/blastdb/file',
				 GD_INGROUP_REGEX    => 'ENSMUSG0',
				 GD_OUTGROUP_REGEXES => ['ENSG0'],
				 GD_TRANSL_REGEX     => 'ENSMUSP',
				 GD_HIT_COVERAGE     => '80',
				 GD_HIT_IDENTITY     => '50',
				 GD_DISTANCE_CUTOFF  => '0.6',
				 GD_GENETIC_CODE     => '1',
				 GD_OUTPUT_DIR       => '',
				},
		  'SINFRUG0' => {GD_SPECIES          => 'Fugu rubripes',
				 GD_BLASTDB_FILE     => '/path/to/blastdb/file',
				 GD_INGROUP_REGEX    => 'SINFRUG0',
				 GD_OUTGROUP_REGEXES => [''],
				 GD_TRANSL_REGEX     => 'SINFRUP0',
				 GD_HIT_COVERAGE     => '80',
				 GD_HIT_IDENTITY     => '50',
				 GD_DISTANCE_CUTOFF  => '1.0',
				 GD_GENETIC_CODE     => '1',
				 GD_OUTPUT_DIR       => '',
			       },
		  'ENSGALG0' => {GD_SPECIES          => 'Gallus gallus',
				 GD_BLASTDB_FILE     => '/path/to/blastdb/file',
				 GD_INGROUP_REGEX    => 'ENSGALG0',
				 GD_OUTGROUP_REGEXES => [''],
				 GD_TRANSL_REGEX     => 'ENSGALP0',
				 GD_HIT_COVERAGE     => '80',
				 GD_HIT_IDENTITY     => '50',
				 GD_DISTANCE_CUTOFF  => '1.0',
				 GD_GENETIC_CODE     => '1',
				 GD_OUTPUT_DIR       => '',
				},
		  'ENSDARG0' => {GD_SPECIES          => 'Danio rerio',
				 GD_BLASTDB_FILE     => '/path/to/blastdb/file',
				 GD_INGROUP_REGEX    => 'ENSDARG0',
				 GD_OUTGROUP_REGEXES => [''],
				 GD_TRANSL_REGEX     => 'ENSDARP0',
				 GD_HIT_COVERAGE     => '80',
				 GD_HIT_IDENTITY     => '50',
				 GD_DISTANCE_CUTOFF  => '1.0',
				 GD_GENETIC_CODE     => '1',
				 GD_OUTPUT_DIR       => '',
				},
		  'CG'       => {GD_SPECIES          => 'Drosophila melanogaster',
				 GD_BLASTDB_FILE     => '/path/to/blastdb/file',
				 GD_INGROUP_REGEX    => 'CG',
				 GD_OUTGROUP_REGEXES => [''],
				 GD_TRANSL_REGEX     => 'CG\d+-PA',
				 GD_HIT_COVERAGE     => '80',
				 GD_HIT_IDENTITY     => '50',
				 GD_DISTANCE_CUTOFF  => '1.0',
				 GD_GENETIC_CODE     => '1',
				 GD_OUTPUT_DIR       => '',
				},
		  'GSTENG|^AY|^HOX|^EVX'  => {
				 GD_SPECIES          => 'Tetrodon nigroviridis',
				 GD_BLASTDB_FILE     => '/path/to/blastdb/file',
				 GD_INGROUP_REGEX    => 'GSTENG',
				 GD_OUTGROUP_REGEXES => [''],
				 GD_TRANSL_REGEX     => 'GSTENP0',
				 GD_HIT_COVERAGE     => '80',
				 GD_HIT_IDENTITY     => '50',
				 GD_DISTANCE_CUTOFF  => '1.0',
				 GD_GENETIC_CODE     => '1',
				 GD_OUTPUT_DIR       => '',
				},
		  'ENSCAFG0' => {GD_SPECIES          => 'Canis familiaris',
				 GD_BLASTDB_FILE     => '/path/to/blastdb/file',
				 GD_INGROUP_REGEX    => 'ENSCAFG0',
				 GD_OUTGROUP_REGEXES => ['ENSG', 'ENSMUSG', 'ENSRNOG'],
				 GD_TRANSL_REGEX     => 'ENSCAFP0',
				 GD_HIT_COVERAGE     => '80',
				 GD_HIT_IDENTITY     => '50',
				 GD_DISTANCE_CUTOFF  => '0.60',
				 GD_GENETIC_CODE     => '1',
				 GD_OUTPUT_DIR       => '',
				},
		  'ENSANGG0' => {GD_SPECIES          => 'Anopheles gambiae',
				 GD_BLASTDB_FILE     => '/path/to/blastdb/file',
				 GD_INGROUP_REGEX    => 'ENSANGG0',
				 GD_OUTGROUP_REGEXES => [''],
				 GD_TRANSL_REGEX     => 'ENSANGP0',
				 GD_HIT_COVERAGE     => '80',
				 GD_HIT_IDENTITY     => '50',
				 GD_DISTANCE_CUTOFF  => '1.0',
				 GD_GENETIC_CODE     => '1',
				 GD_OUTPUT_DIR       => '',
				},
		  'ENSAPMG0' => {GD_SPECIES          => 'Apis mellifera',
				 GD_BLASTDB_FILE     => '/path/to/blastdb/file',
				 GD_INGROUP_REGEX    => 'ENSAPMG0',
				 GD_OUTGROUP_REGEXES => [''],
				 GD_TRANSL_REGEX     => 'ENSAPMP0',
				 GD_HIT_COVERAGE     => '80',
				 GD_HIT_IDENTITY     => '50',
				 GD_DISTANCE_CUTOFF  => '1.0',
				 GD_GENETIC_CODE     => '1',
				 GD_OUTPUT_DIR       => '',
				},
		  'ENSPTRG0' => {GD_SPECIES          => 'Pan troglodytes',
				 GD_BLASTDB_FILE     => '/path/to/blastdb/file',
				 GD_INGROUP_REGEX    => 'ENSPTRG0',
				 GD_OUTGROUP_REGEXES => [''],
				 GD_TRANSL_REGEX     => 'ENSPTRP0',
				 GD_HIT_COVERAGE     => '80',
				 GD_HIT_IDENTITY     => '50',
				 GD_DISTANCE_CUTOFF  => '0.6',
				 GD_GENETIC_CODE     => '1',
				 GD_OUTPUT_DIR       => '',
			       },
		  'CELG'     => {GD_SPECIES          => 'Caenorhabitis elegans',
				 GD_BLASTDB_FILE     => '/path/to/blastdb/file',
				 GD_INGROUP_REGEX    => 'CELG',
				 GD_OUTGROUP_REGEXES => [''],
				 GD_TRANSL_REGEX     => 'CELP\S+',
				 GD_HIT_COVERAGE     => '80',
				 GD_HIT_IDENTITY     => '50',
				 GD_DISTANCE_CUTOFF  => '1.0',
				 GD_GENETIC_CODE     => '1',
				 GD_OUTPUT_DIR       => '',
				},
		  'ENSCING'  => {GD_SPECIES          => 'Ciona intestinalis',
				 GD_BLASTDB_FILE     => '/path/to/blastdb/file',
				 GD_INGROUP_REGEX    => 'ENSCING',
				 GD_OUTGROUP_REGEXES => [''],
				 GD_TRANSL_REGEX     => 'ENSCINP0',
				 GD_HIT_COVERAGE     => '80',
				 GD_HIT_IDENTITY     => '50',
				 GD_DISTANCE_CUTOFF  => '1.0',
				 GD_GENETIC_CODE     => '1',
				 GD_OUTPUT_DIR       => '',
				},
		  'ENSXETG0' => {GD_SPECIES          => 'Xenopus tropicalis',
				 GD_BLASTDB_FILE     => '/path/to/blastdb/file',
				 GD_INGROUP_REGEX    => 'ENSXETG0',
				 GD_OUTGROUP_REGEXES => [''],
				 GD_TRANSL_REGEX     => 'ENSXETP0',
				 GD_HIT_COVERAGE     => '80',
				 GD_HIT_IDENTITY     => '50',
				 GD_DISTANCE_CUTOFF  => '1.0',
				 GD_GENETIC_CODE     => '1',
				 GD_OUTPUT_DIR       => '',
				},
		  '^Y|^Q'    => {GD_SPECIES          => 'Saccharomyces cerevisiae',
				 GD_BLASTDB_FILE     => '/path/to/blastdb/file',
				 GD_INGROUP_REGEX    => 'Y\w+',
				 GD_OUTGROUP_REGEXES => [''],
				 GD_TRANSL_REGEX     => '\w+',
				 GD_HIT_COVERAGE     => '90',
				 GD_HIT_IDENTITY     => '60',
				 GD_DISTANCE_CUTOFF  => '1.0',
				 GD_GENETIC_CODE     => '1',
				 GD_OUTPUT_DIR       => '',
				},
		 },


   # Location of blastn executable to use.  Defaults to 'wublastn'.  This
   # blast flavour should match the index type specified in GB_BLAST_VARIANT.
   GD_BLAST_EXE            => '',

   # Variant of BLAST used to format the database.  Options are :
   #
   #   String     Description           Database format command-stub
   #   ------     -----------           ----------------------------
   #   'ncbi'     original NCBI BLAST   formatdb -o T -p F
   #   'wu_new'   newer WashU BLAST     xdformat -n -I
   #   'wu_old'   older WashU BLAST     pressdb
   GD_BLAST_VARIANT        => 'wu_new',

   # Location of codeml executable (from PAML package).
   GD_CODEML_EXE           => '',

   # Location of temporary working disk space.  Defaults to /tmp.
   GD_WORK_DIR             => '/tmp',

   # Method to use for calculating genetic distance.  Options are : 
   #   'NeiGojobori'  - Nei & Gojobori (1986) Mol. Bio. Evol. 3:418-26.
   #   'ML'           - Ad-hoc pairwise method implemented within PAML.
   # Default method is 'NeiGojobori'
   GD_DISTANCE_METHOD      => 'NeiGojobori',

   # How output is to be stored.  Options are :
   #   'files'  - output written to text file.
   #   'db'     - output written directly to a Compara database.
   GD_OUTPUT_METHOD        => 'db',

   # Mode for output generation.  Compara alignments are stored in amino acid
   # coordinates, however it may also be desirable to produce results in
   # nucleotide coordinates.  Options are :
   #   'nucleotide'  - alignments as nucleotide sequences
   #   'aminoacid'   - alignments as amino acid sequences
   # Defaults to 'nucleotide'.  If GD_OUTPUT_METHOD above is 'db' then 
   # GD_OUTPUT_TYPE be automatically be over-ridden to 'aminoacid'.
   GD_OUTPUT_TYPE          => 'aminoacid',

   # (Database output only) Full path to the registry config file.
   GD_REGISTRY_CONF        => '/path/to/ensembl_init.example',

   # (Database output only) Name of compara database from registry.
   GD_COMPARA_DBNAME       => 'compara31',

   # (Database output only) Compara method_link type.
   GD_COMPARA_TYPE          => 'ENSEMBL_PARALOGUES',

   # (Database output only) Compara method_link description.
   GD_COMPARA_DESCRIPTION   => 'YoungParalogues',

   # (Database output only) Compara sequence source names.
   GD_COMPARA_PEP_SOURCE    => 'ENSEMBLPEP',
   GD_COMPARA_GENE_SOURCE   => 'ENSEMBLGENE',

  );

sub import {
  my ($callpack) = caller(0); # Name of the calling package
  my $pack = shift; # Need to move package off @_

  # Get list of variables supplied, or else everything
  my @vars = @_ ? @_ : keys( %GeneDupl );
  return unless @vars;

    # Predeclare global variables in calling package
  eval "package $callpack; use vars qw("
    . join(' ', map { '$'.$_ } @vars) . ")";
    die $@ if $@;


    foreach (@vars) {
	if ( defined $GeneDupl{ $_ } ) {
            no strict 'refs';
	    # Exporter does a similar job to the following
	    # statement, but for function names, not
	    # scalar variables:
	    *{"${callpack}::$_"} = \$GeneDupl{ $_ };
	} else {
	    die "Error: GeneDupl: $_ not known\n";
	}
    }
}

1;
