#!/usr/local/bin/perl -w
###############################################################################
#
#   Name:           ExtIndex.pm
#
#   Description:    Handler for external indexers.  The indexers are defined in
#                   SiteDefs, and this module passes on get_seq_by_[id|acc] to
#                   the appropriate indexer.
#
#                   Indexers are very simple modules that offer the two get_seq
#                   functions and nothing else.
#
#   History:        2000-10-26  jws:    original version
#
###############################################################################

package ExtIndex;

use strict;
use SiteDefs qw(:INDEX);
my $INDEXER_PREFIX="ExtIndex_";

########################################################
# Hash of valid options - initialised with a hash slice
########################################################
my %options;
@options{qw(id acc seq desc all)}=();



sub new {
    my $self=shift;
    my %databases = %{$ENSEMBL_EXTERNAL_DATABASES};
    my %indexers = %{$ENSEMBL_EXTERNAL_INDEXERS};

    ##########################################################################
    # Check that each indexer in the db hash has been defined in indexer hash
    ##########################################################################
    foreach my $database(keys %databases){
        my $indexer=$databases{$database};
        if(!exists $indexers{$indexer}){
            warn "The indexer $indexer which has been specified for $database databases is not listed in the ENSEMBL_EXTERNAL_INDEXERS section of SiteDefs\n";
            return undef;
        }
    }
    
    ##############################
    # Build object data structure
    ##############################
    my %obj;
    
    foreach my $database(keys %databases){
        my $indexer = $databases{$database};
        $obj{$database} = {
                            'indexer'   =>  $indexer,
                            'exe'       =>  $indexers{$indexer},
                            };
    }
    
    ###########################
    # Load the indexer modules
    ###########################
    foreach my $indexer(keys %indexers){
        $indexer=$INDEXER_PREFIX.$indexer.'.pm';
        eval{require ("ExtIndex/$indexer");};
        if ($@){
            warn "Cannot use $indexer index module: $!\n";
            return undef;
        }
    }
    
    return bless \%obj, $self;
}


sub get_seq_by_id{
    my ($self, $arghashref)=@_;
    return $self->_get_seq('ID',$arghashref);
}



sub get_seq_by_acc{
    my ($self, $arghashref)=@_;
    return $self->_get_seq('ACC',$arghashref);

}


sub _get_seq{
    my ($self,$type,$arghashref)=@_;
    my %args=%$arghashref;
    
    ###############################################
    # Check for valid options and fix if necessary
    ###############################################
    $args{'OPTIONS'}='all' unless (exists $args{'OPTIONS'} && exists $options{$args{'OPTIONS'}});
    
    ############################################
    # retrieve the indexer and executable names
    ############################################
    my ($indexer, $exe)=$self->_indexer($args{'DB'});
    
    
    if (defined $indexer && defined $args{$type}){
        $indexer=$INDEXER_PREFIX.$indexer;
        my $function='get_seq_by_'.lc($type);
        
        return $indexer->$function({
                                    'EXE'       => $exe,
                                    'DB'        => $args{'DB'},
                                    $type       => $args{$type},
                                    'FORMAT'    => $args{'FORMAT'},
                                    'OPTIONS'   => $args{'OPTIONS'},
                                    });
    }
    return undef;
}   

    
sub _indexer{
    my ($self, $db)=@_;
    my $indexer=$self->{$db}{'indexer'};
    my $exe = $self->{$db}{'exe'};
    return ($indexer, $exe);
}
1;

