--- GRAMENE ONTOLOGY
--- based on GENE ONTOLOGY

CREATE TABLE term 
(
	term_id		number(11) not null	CONSTRAINT pk_term PRIMARY KEY,
	term_name	varchar2(255) not null,
	term_type	number(11) not null,
	term_accession	number(11) not null,
	is_obsolete	number(1) not null 	CHECK (is_obsolete = 0 OR is_obsolete = 1),
	is_root		number(1) not null 	CHECK (is_root = 0 OR is_root = 1),
	UNIQUE(term_type, term_accession)
);
--- COMMENT ON TABLE term IS
--- term_id: internal key for each term
--- term_type: root, process, component, function, trait 
--- term_accession: used to be unique independently
--- is_obsolete: 1 if this term is obsolete; 0 otherwise
--- is_root: 1 if this term is the root term; 0 otherwise 
--- UNIQUE(term_type, term_accession) is very loose; be careful


CREATE TABLE modification 
(
	modification_id		number(11) not null 	CONSTRAINT pk_modification PRIMARY KEY,
	contributor_id		number(11) not null     references contributor,
	mod_type		varchar2(32) not null,
	mod_time		number(11) not null,
	rank			number(11) not null
);
COMMENT ON modification.contributor_id is
"use contributor_id instead of person".

CREATE TABLE term_comment 
(
	term_id		number(11) not null 	REFERENCES term,
	comment_text	varchar2(2048) not null
);

CREATE TABLE category 
(
	category_id	number(11) not null 	CONSTRAINT pk_category PRIMARY KEY,
	category_name	varchar2(32) not null,
	category_description	varchar2(2048)
);

CREATE TABLE term_category
(
	term_id		number(11) not null	REFERENCES term,
	category_id	number(11) not null 	REFERENCES category,
	UNIQUE(term_id, category_id)
);

CREATE TABLE relationship_type
(
	relationship_type_id	number(11) 	CONSTRAINT pk_relationship_type PRIMARY KEY,
	type_name		varchar2(40) not null,
	type_desc		varchar2(255) not null
);

CREATE TABLE term_to_term 
(
	term_to_term_id	number(11) not null	CONSTRAINT pk_term2term PRIMARY KEY,
	is_inheritance	number(1) not null	CHECK (is_inheritance = 0 OR is_inheritance = 1),
	relationship_type_id number(11) not null	REFERENCES
relationship_type,
	term1_id	number(11) not null	REFERENCES term,
	term2_id	number(11) not null	REFERENCES term,	 
	is_obsolete	number(1) not null	CHECK (is_obsolete = 0 OR is_obsolete = 1),
	modification_id	number(11) not null	REFERENCES modification,
	CONSTRAINT unique_term1_term2 UNIQUE (term1_id, term2_id)
);
---   back since 2/1/2002
---   is_inheritance	number(1) not null	CHECK (is_inheritance = 0 OR is_inheritance = 1),
---   relationship_type is added in 02/2002, but it seems to be a dummy.


CREATE TABLE term_definition 
(
	term_id		number(11) not null	REFERENCES term ON DELETE CASCADE,
	definition	varchar2(4000) not null,
	UNIQUE(term_id)
);
/* no definition no term */
--- definition	long not null

CREATE TABLE term_synonym 
(
	term_id		number(11) not null 	REFERENCES term,
	synonym_name	varchar2(128) not null,
	synonym_acc	number(11),
	UNIQUE (term_id, synonym_name)
);

CREATE TABLE dbxref
(
	dbxref_id	number(11) 		CONSTRAINT pk_dbxref PRIMARY KEY,
	xref_key  	varchar2(255) not null,
	xref_keytype  	varchar2(32),
	xref_dbname 	varchar2(32) not null,
	xref_desc	varchar2(255),
	UNIQUE(xref_key, xref_keytype, xref_dbname)
);
--- 	xref_extra	varchar2(255),

CREATE TABLE term_dbxref
(
	term_id	    	number(11) not null	REFERENCES  term,
	dbxref_id   	number(11) not null	REFERENCES  dbxref,
	refers_to	varchar2(16),
	UNIQUE(term_id, dbxref_id)
);
--- a dbxref can (optionally) refer to a synonym of a term
--- rather than the whole term
--- anatomy, term, definition, synonym

CREATE TABLE species 
(
	species_id 	 	number(11) not null CONSTRAINT pk_species PRIMARY KEY,
	ncbi_taxa_id 		number(11),
	common_name 		varchar2(64),
	lineage_string 		varchar2(2048),
	genus 			varchar2(32),
	species 		varchar2(32)
);

CREATE TABLE gene_product 
(
	gene_product_id		number(11) CONSTRAINT pk_gene_product PRIMARY KEY,
	gene_product_symbol	varchar2(128) not null,
	organism_dbxref_id	number(11) not null	REFERENCES  dbxref(dbxref_id),
	species_id		number(11),
	gene_product_full_name	varchar2(1024),
	UNIQUE(gene_product_symbol, organism_dbxref_id)
);
--- NOTICE: Ideally it should have
--- 	species_id		number(11) not null	REFERENCES species(id),

CREATE TABLE gene_product_synonym 
(
	gene_product_id		number(11) not null 	REFERENCES gene_product,
	gene_product_synonym_symbol	varchar2(128) not null,
	UNIQUE(gene_product_id, gene_product_synonym_symbol)
);

CREATE TABLE seq 
(
	seq_id 			number(11) 		CONSTRAINT pk_seq PRIMARY KEY,
	display_id 		varchar2(64),
	description 		varchar2(255),
	seq 			clob,
	seq_len 		number(11),
	md5checksum 		varchar2(32),
	moltype 		varchar2(16),
	timestamp 		number(11),
	UNIQUE(display_id, md5checksum)
);

CREATE TABLE seq_dbxref
(
	seq_id		number(11) not null	REFERENCES seq,
	dbxref_id	number(11) not null,
	UNIQUE(seq_id, dbxref_id)
);

CREATE TABLE association 
(
	association_id  	number(11) 		CONSTRAINT pk_association PRIMARY KEY,
	term_id	        	number(11) not null	REFERENCES term,
	gene_product_id		number(11) not null,	
	object_class 		varchar2(32) not null,
	is_not 			number(1)		CHECK (is_not = 0 OR is_not = 1),
	role_group		number(11),
	UNIQUE (term_id, gene_product_id)
);
COMMENT on column association.gene_product_id is
'The internal id of whatever object is associated, not necessarily a gene_product';
COMMENT on column association.object_class is
'the class of the "gene_product_id": "gene_product", "mutant", "translation", ... ';
--- boolean, indicating if this is negated
--- boolean, indicating if this is a primary association
--- NOTICE: Ideally it should have 
--- gene_product_id             number(11) not null     REFERENCES gene_product 
--- The data does not keep is_primary any more.         
--- is_primary          number(1)               CHECK (is_primary = 0 OR is_primary = 1)
--- role group is added in 02/2002  (GO copies Gramene without giving credits)


CREATE TABLE evidence 
(
	evidence_id  		number(11) 		CONSTRAINT pk_evidence PRIMARY KEY,
	evidence_code		varchar2(8) not null,
	association_id		number(11) not null,
	object_class 		varchar2(32) not null,
	dbxref_id		number(11) not null	REFERENCES dbxref,
	seq_acc			varchar2(48)
);
COMMENT on column evidence.association_id is
'The internal id of whatever object the evidence is for, not necessarily an association';
COMMENT on column evidence.object_class is
'the class of the "association_id": "association", "candidate_gene", ... ';
--- NOTICE: Ideally it should have
--- association_id              number(11) not null     REFERENCES association
--- UNIQUE (association_id, dbxref_id, evidence_code),
--- old: seq_acc                varchar2(16)

-------------------------------------------------
--- the tables below are for edit tracking
-------------------------------------------------

CREATE TABLE edit_session 
(
	edit_sessio_id  	number(11) not null 	CONSTRAINT pk_edit_session PRIMARY KEY,
	edit_session_person	varchar2(32) not null,
	mod_time 		date not null,
	edit_comment		varchar2(2048)
);
/* mod_time		number(11) not null, */
--- seconds since 1970
--- mod_time		date not null,

CREATE TABLE operation 
(
	operation_id		number(11) not null 	CONSTRAINT pk_operation PRIMARY KEY,
	operation_type		varchar2(64),
	term_id	        	number(11) not null	REFERENCES term,
	edit_session_id		number(11) not null	REFERENCES edit_session
);
--- new table name: node_operation

CREATE TABLE term_dbxref_edit 
(
	operation_id		number(11) not null	REFERENCES operation,
	xref_key 		varchar2(32) not null,
	xref_dbname 		varchar2(32) not null,		
	xref_type		varchar2(32),
	new_xref_key 		varchar2(32),
	new_xref_dbname		varchar2(32),
	new_xref_type		varchar2(32),
	edit_type		varchar2(4) not null
);

CREATE TABLE term_text_edit 
(
	operation_id		number(11) not null	REFERENCES operation,
	name			varchar2(128),
	definition		varchar2(2048),
	term_text_edit_comment	varchar2(4000)
);

CREATE TABLE synonym_edit 
(
	operation_id		number(11) not null	REFERENCES operation(operation_id),
	synonym_text		varchar2(128),
	acc			number(11),
	new_synonym		varchar2(255),
	new_acc			number(11),
	edit_type		varchar2(4) not null	
);

CREATE TABLE term_category_edit 
(
	operation_id		number(11) not null	REFERENCES operation,
	category_id 		number(11) not null	REFERENCES category,
	is_del			number(11) not null
);

CREATE TABLE structure_edit 
(
	structure_edit_id  	number(11) not null	CONSTRAINT pk_structure_edit PRIMARY KEY,
	operation_id		number(11) not null	REFERENCES operation,
	term_id	        	number(11) not null	REFERENCES term(term_id),
	term_output_id		number(11)		REFERENCES term(term_id)	
);

CREATE TABLE structure_edit_src 
(
	structure_edit_id	number(11) not null	REFERENCES structure_edit(structure_edit_id),
	source_parent_id	number(11)		REFERENCES term(term_id),
	source_term_id		number(11) not null	REFERENCES term(term_id),
	source_type		number(11),
	implied			number(11) not null	
);

--------------------------------------------------------
--- new table since October: path, path_to_term, and ...
--------------------------------------------------------

CREATE TABLE graph_path
(
	graph_path_id		number(11) not null		CONSTRAINT pk_path PRIMARY KEY,
	term1_id	number(11) not null		REFERENCES term,
	term2_id	number(11) not null		REFERENCES term,
	distance	number(11) not null
);

CREATE TABLE graph_path_to_term
(
	graph_path_id			number(11) not null	REFERENCES graph_path(graph_path_id),	
	term_id			number(11) not null	REFERENCES term(term_id),
	rank			number(11) not null	
);

CREATE TABLE term_history 
(
	term1_id	number(11) not null	REFERENCES term(term_id),
	term2_id	number(11),
	modification_id	number(11) not null 	REFERENCES modification(modification_id)
);

--- 10/31/2001 modified
CREATE TABLE gene_product_count
(
	term_id			number(11) not null	REFERENCES term(term_id),
	code			varchar2(8),
	species_db_name		varchar2(32) not null,
	product_count		number(11),
	UNIQUE(term_id, code, species_db_name)
);

CREATE TABLE gene_product_seq 
(
  	gene_product_id 	number(11) not null,
  	seq_id 			number(11) not null,
  	is_primary_seq 		number(11),
  	UNIQUE (gene_product_id,seq_id)
);

CREATE TABLE word_unit_to_term
(
	term_id			number(11) not null	REFERENCES term(term_id),
	is_synonym		number(11) not null,
	is_definition		number(11) not null,
	word_type		varchar2(32),
	word_sound		varchar2(32),
	word_unit		varchar2(128),
	UNIQUE(word_unit)	
);

CREATE TABLE mod_comment
(
	modification_id number(11) not null REFERENCES modification(modification_id),
	mod_comment varchar2(2048),
	UNIQUE(modification_id)
);

CREATE TABLE godb_meta_data 
(
	date_created 		number(11) not null,
	creation_comment 	varchar2(2048),
	version 		varchar2(64) not null,
	created_by 		varchar2(64) not null,
	current_id 		number(11) not null
);

CREATE TABLE seq_property
(
	seq_property_id		number(11)	CONSTRAINT pk_seq_property PRIMARY KEY,
	seq_id			number(11) NOT NULL REFERENCES seq(seq_id),
	property_key		varchar2(64) NOT NULL,
	property_val		varchar2(255) NOT NULL				
);

select * from tab;
-----------------------------------------
--- ONTOLOGY END 
-----------------------------------------

-----------------------------------------
--- Performance Enhancement Start
-----------------------------------------

CREATE TABLE quick_evidence_codes
(
	association_id	number(11) not null	CONSTRAINT pk_quick_evidence_codes PRIMARY KEY,
	evidence_codes 	varchar2(128) not null
);
COMMENT ON quick_evidence_codes is
"improve performance from association_id to evidence_codes".

-----------------------------------------
--- Performance Enhancement End
-----------------------------------------



CREATE TABLE sequence_feature 
(
	feature_id	            NUMBER                       NOT NULL,
	key_name	            VARCHAR2(127)                NOT NULL,
	description                 VARCHAR2(255),
        PRIMARY KEY ( feature_id )
);

CREATE TABLE sequence_to_feature 
(
	sequence_to_feature_id	    NUMBER NOT NULL	 	 CONSTRAINT pk_seq_to_feature PRIMARY KEY,	
	gene_product_id	            NUMBER                       NOT NULL,
	feature_id	            NUMBER                       NOT NULL,
	from_endpoint               NUMBER,
        to_endpoint                 NUMBER,
        UNIQUE(gene_product_id, feature_id),
	FOREIGN KEY ( gene_product_id ) REFERENCES gene_product,
	FOREIGN KEY ( feature_id ) REFERENCES sequence_feature
);

CREATE TABLE data_base (
    data_base_id                     NUMBER(11)                      NOT NULL ,
    xref_dbname                      VARCHAR2(32)                    NOT NULL,
    data_base_name                   VARCHAR2(80)                    NOT NULL,
    release                          VARCHAR2(40),
    url_pattern                      VARCHAR2(255),
    table_name                       VARCHAR2(40),
    key_column                       VARCHAR2(40),
    PRIMARY KEY (data_base_id)
);
COMMENT on column data_base.xref_dbname is 
'matches dbxref.xref_dbname - GO Consortium database abbreviation (lower case)';


CREATE TABLE objectxref (
    objectxref_id                       NUMBER(11)                  NOT NULL,
    table_name                          VARCHAR2(60)                NOT NULL,
    row_id                              NUMBER(11)                  NOT NULL,
    dbxref_id                           NUMBER(11)                  NOT NULL,
    PRIMARY KEY (objectxref_id),
    FOREIGN KEY (dbxref_id) REFERENCES dbxref
);




CREATE TABLE subspecies (
    subspecies_id	           NUMBER(11)                       NOT NULL,
    species_id	                   NUMBER(11)                       NOT NULL,
    subspecies_name                VARCHAR2(100)                    NOT NULL,
    PRIMARY KEY (subspecies_id),
    FOREIGN KEY (species_id) REFERENCES species
);
COMMENT on table subspecies is
 'Includes a row for "Unknown" -for cultivar';



CREATE TABLE cultivar (
    cultivar_id	                   NUMBER(11)                        NOT NULL,
    subspecies_id	           NUMBER(11)                        NOT NULL,
    species_id	                   NUMBER(11)                        NOT NULL,
    cultivar_name                  VARCHAR2(100)                     NOT NULL,
    PRIMARY KEY (cultivar_id),
    FOREIGN KEY (species_id) REFERENCES species,
    FOREIGN KEY (subspecies_id) REFERENCES subspecies
);
COMMENT on column cultivar.subspecies_id is
 ' may point to the "Unknown" row';

CREATE TABLE gene_product_to_cultivar (
    gene_product_id	           NUMBER(11)                         NOT NULL,
    cultivar_id	                   NUMBER(11)                         NOT NULL,
    PRIMARY KEY (gene_product_id, cultivar_id),
    FOREIGN KEY (gene_product_id) REFERENCES gene_product, 
    FOREIGN KEY (cultivar_id) REFERENCES cultivar
);

CREATE TABLE gene_product_to_subspecies ( 
    gene_product_id	           NUMBER(11)                         NOT NULL,
    subspecies_id	           NUMBER(11)                         NOT NULL,
    PRIMARY KEY (gene_product_id, subspecies_id),
    FOREIGN KEY (gene_product_id) REFERENCES gene_product, 
    FOREIGN KEY (subspecies_id) REFERENCES subspecies
);
--- COMMENT on table GENE_PRODUCT_TO_SUBSPECIES IS
---    'Necessary because don't always know cultivar';



CREATE TABLE gene_product_to_species (
    gene_product_id	           NUMBER(11)                         NOT NULL,
    species_id	                   NUMBER(11)                         NOT NULL,
    PRIMARY KEY (gene_product_id, species_id),
    FOREIGN KEY (gene_product_id) REFERENCES gene_product, 
    FOREIGN KEY (species_id) REFERENCES species
);
---COMMENT on table GENE_PRODUCT_TO_SPECIES IS
---    'Necessary because don't always know subspecies or cultivar';


CREATE TABLE expression (
    expression_id                   NUMBER(11)                        NOT NULL,
    gene_product_id                 NUMBER(11)                        NOT NULL,
    expression_annotation           VARCHAR2(4000),
    PRIMARY KEY (expression_id),
    FOREIGN KEY (gene_product_id) REFERENCES gene_product 
);

CREATE TABLE curation (
   table_name	                    VARCHAR2(40)                      NOT NULL,
   row_id	                    NUMBER(11)                        NOT NULL,
   curation_time	            DATE                              NOT NULL,
   contributor_id	            NUMBER(11)                        NOT NULL,
   curation_comment                 VARCHAR2(4000),
   FOREIGN KEY (contributor_id) REFERENCES contributor
);


--- CREATE TABLE PROTEIN_CLEAVING (
---   protein_cleaving_id  NUMBER(11) NOT NULL,
---   from_gene_product_id      NUMBER(11) NOT NULL,
---   N_end_gene_product_id  NUMBER(11),
---   C_end_gene_product_id  NUMBER(11),
---   by_gene_product_id      NUMBER(11) NOT NULL,
---   cleaving_association_id NUMBER(11),
---   peptide_position  NUMBER(7),
---  PRIMARY KEY (protein_cleaving_id),
---   FOREIGN KEY (from_gene_product_id) REFERENCES GENE_PRODUCT (gene_product_id),
---   FOREIGN KEY (N_end_gene_product_id) REFERENCES GENE_PRODUCT (gene_product_id),
---   FOREIGN KEY (C_end_gene_product_id) REFERENCES GENE_PRODUCT (gene_product_id)
--- );

--- CREATE TABLE PROTEIN_COMPLEX (
---   complex_gene_product_id      NUMBER(11) NOT NULL,
---   part_gene_product_id      NUMBER(11) NOT NULL,
---   PRIMARY KEY (complex_gene_product_id,part_gene_product_id),
---   FOREIGN KEY (part_gene_product_id) REFERENCES GENE_PRODUCT (gene_product_id),
---   FOREIGN KEY (complex_gene_product_id) REFERENCES GENE_PRODUCT (gene_product_id) 
--- );


CREATE TABLE gene_product_to_keyword 
(
   gene_product_id  NUMBER(11) REFERENCES GENE_PRODUCT (gene_product_id),
   keyword_id       NUMBER(11) REFERENCES KEYWORD (keyword_id)
);


CREATE TABLE gene_product_gene_name
(
	gene_product_id	       number(11)          not null,
	gene_name  	       varchar2(255)       not null,
	UNIQUE (gene_product_id, gene_name), 
	FOREIGN KEY (gene_product_id) REFERENCES gene_product
);


CREATE TABLE gene_product_type
(
	gene_product_id	       number(11)          not null,
	gene_product_type      varchar2(255)       not null,
	PRIMARY KEY (gene_product_id),
	FOREIGN KEY (gene_product_id) REFERENCES gene_product
);
---For data from Gene Ontology, gene_product_type is "GO".
---For data from Swissprot, gene_product_type is "SPTrembl". (Including data from Trembl).



CREATE TABLE gene_product_trembl_new 
(
	gene_product_id		number(11)         not null,
	swissprot_acc	        varchar2(128)      not null,
	FOREIGN KEY (gene_product_id) REFERENCES gene_product
);
---Proteins in this table are from Trembl-New.
---Should use gene_product_type table?


CREATE TABLE gene_product_to_gi_number 
(
	gene_product_id		number(11)         not null,
	gi_number	        varchar2(30)       not null,
	FOREIGN KEY (gene_product_id) REFERENCES gene_product
);
---Should use dbxref table?


CREATE TABLE gene_product_tissue
(
	gene_product_id	       number(11)          not null,
	tissue  	       varchar2(255)       not null,
	UNIQUE (gene_product_id, tissue), 
	FOREIGN KEY (gene_product_id) REFERENCES gene_product
);


CREATE TABLE gene_product_extra 
(
	gene_product_id		number(11)         not null,
	gene_product_name	varchar2(1024)     not null,
	swissprot_acc	        varchar2(128)      not null,
	swissprot_id	        varchar2(128)      not null,
	FOREIGN KEY (gene_product_id) REFERENCES gene_product
);
---For protein search only. This is actually a redundant table, but it improves the performance. 


CREATE TABLE gene_product_helper 
(
	gene_product_id				number(11)         not null,
	swissprot_acc	       			varchar2(128)      not null,
	swissprot_id	        		varchar2(128)      not null,
	gene_product_name			varchar2(2048)     not null,
	gene_product_gene_name  	       	varchar2(255)      not null,
	organism				varchar2(511)      not null,
	is_curated				number(1)          not null,
	FOREIGN KEY (gene_product_id) REFERENCES gene_product
);
---For protein search only. This is actually a redundant table, but it improves the performance. 


