package Bio::EnsemblViewer::Sequence::Contig;

use Bio::Root::Object;
use vars qw(@ISA);
use strict;

use Bio::EnsEMBL::Transcript;
use Bio::EnsemblViewer::DrawableElement;
use Bio::EnsemblViewer::Sequence::SuperLinearElement;
use Bio::EnsemblViewer::LinearElement;
use Bio::EnsemblViewer::BasicElement;
use Bio::EnsemblViewer::JSTools;
use POSIX;
@ISA = qw(Bio::Root::Object DrawableElement);


use Bio::EnsEMBL::Utils::Eprof qw( eprof_start eprof_end eprof_dump);


sub _initialize 
{
    my($self,@args) = @_;      

 
    my($contig,$image_par)= $self->_rearrange([qw(CONTIG IMAGE_PAR)],@args);   
    my $make = $self->SUPER::_initialize;   
	
	$self->_get_contig($contig);
    $self->_get_image_par($image_par);
    
	my $image_param_ref=$self->_get_image_par;
	my $img_offset=$image_param_ref->{left_margin}
					+$image_param_ref->{legend_margin};

	my	$img_len=$image_param_ref->{x_img_len} 
					-$image_param_ref->{left_margin} 
					-$image_param_ref->{legend_margin} 
					-$image_param_ref->{right_margin};
	
	my $img_pixel_bases=$img_len/$self->_get_contig->length;
	
	$image_param_ref->{img_offset}      = $img_offset;
	$image_param_ref->{img_pixel_bases} = $img_pixel_bases;

    return $make;     
}



sub contig
{    
    my ($self)=@_;
           
    my ($y_coord_end_plus,$y_coord_end_minus)=$self->_draw_sequence;

    my $plus=undef;
    my $minus=undef;

    &eprof_start('Contig.pm - transcript_draw');
    my ($plus,$minus)=$self->_draw_transcripts($y_coord_end_plus,$y_coord_end_minus); 
    if (defined $plus){$y_coord_end_plus=$plus;}
    if (defined $minus){$y_coord_end_minus=$minus;}   
    &eprof_end('Contig.pm - transcript_draw');


    &eprof_start('Contig.pm - feature_draw');
    my ($plus,$minus)=$self->_draw_features($y_coord_end_plus,$y_coord_end_minus);
    if (defined $plus){$y_coord_end_plus=$plus;}
    if (defined $minus){$y_coord_end_minus=$minus;}
    &eprof_end('Contig.pm - feature_draw');    

    ($y_coord_end_plus,$y_coord_end_minus)=$self->_draw_scale($y_coord_end_plus,$y_coord_end_minus);

    $self->_get_image_par->{y_img_len}=$y_coord_end_minus-$y_coord_end_plus+
	$self->_get_image_par->{bottom_margin}+$self->_get_image_par->{top_margin};
    $self->_set_start(-$y_coord_end_plus+$self->_get_image_par->{top_margin});

   

}




sub _draw_sequence
{
    my ($self)=@_;

    my $y_coord_end_plus;
    my $y_coord_end_minus;
    my $height=$self->gene_feature_height;

    my $superlinear=SuperLinearElement->new($self->_get_image_par);
    my $linear=LinearElement->new($self->_get_image_par);   



    my $y_cent=0;    
    my $y_start=$y_cent-0.5*$height;	
    my $y_end=$y_cent+0.5*$height;
  

    $y_coord_end_plus=$y_cent;
    $y_coord_end_minus=$y_cent;

    my $strand=0;
    my ($y_start,$y_end)=$self->_calc_y_coord($strand,$y_cent,0);

    my %args=(name=>"sequence",type=>'printable',subtype=>'small',
	      x_start=>$self->_get_image_par->{left_margin},y_start=>$y_start+0.5*$height+$self-> _legend_adjustment,color=>"black");
    
    $linear->_add_Element(BasicElement->new(%args));
    
    $superlinear->_add_Element($linear);  

    my $linear=LinearElement->new($self->_get_image_par);   
  
   # my ($y_start,$y_end)=$self->_calc_y_coord($strand,$y_cent,$height,0);
    my %args= (type=>'drawable', subtype=>'filledrect',color=>'red',
	       x_start=>$self->_get_image_par->{left_margin}+$self->_get_image_par->{legend_margin},
	       x_end=>$self->_get_image_par->{x_img_len}-$self->_get_image_par->{right_margin},
	       y_start=>$y_start,y_end=>$y_end);
    
    $linear->_add_Element(BasicElement->new(%args));

    $superlinear->_add_Element($linear);    

    $self->_add_Element($superlinear);      


    return ($y_coord_end_plus, $y_coord_end_minus);

}




sub _draw_transcripts
{

    my ($self,$plus,$minus)=@_;
  
    my $last_plus;
    my $last_minus;

    my $contig=$self->_get_contig;

    my @genes = $contig->get_all_Genes(); 
    my $type='EnsEMBL trans';
    ($plus,$minus)=$self->_build_genes($plus,$minus,$type,$contig,@genes);
  
  
    my @genes = $contig->get_all_ExternalGenes(); 
	my $type='EMBL trans';
    my ($new_plus,$new_minus)=$self->_build_genes($plus,$minus,$type,$contig,@genes);

    if(defined $new_plus){$last_plus=$new_plus;}
    else {$last_plus=$plus;}
    if(defined $new_minus){$last_minus=$new_minus;}
    else {$last_minus=$minus;}



    return ($last_plus,$last_minus);
          
}


sub _build_genes
{
    my ($self,$plus,$minus,$type,$contig,@genes)=@_;

   my $bmp_transcripts;
    my @transcripts;
    my @all_transcripts;
        
    foreach my $gene(@genes)
    {		
	my @new_transcripts;
	@transcripts=$gene->each_Transcript;
	
	if( $gene->is_known ) {
	   foreach my $temp_trans ( @transcripts ) {
	        # sort of hacky
		$temp_trans->gene_is_known(1);
		my @temp_geneDBlinks = $gene->each_DBLink;
		my $gene_name;
		
		foreach my $DB_link ( @temp_geneDBlinks ){
		    $gene_name = $DB_link->primary_id;
		    last if $DB_link->database eq 'HUGO';
		}
		$temp_trans->gene_name($gene_name);

	    }		
	}	 

	@new_transcripts= $self->_exclude_exons_from_diffrent_contigs($contig,@transcripts);

	push @all_transcripts,@new_transcripts;
    }


    my @transcripts_plus;
    my @transcripts_minus;
 
    foreach my $transcript (@all_transcripts)
    {	

	if (($transcript->each_Exon)[0]->strand ==1){print STDERR $transcript->id," strand ",($transcript->each_Exon)[0]->strand,"\n";push @transcripts_plus,$transcript;}
	if (($transcript->each_Exon)[0]->strand ==-1){push @transcripts_minus,$transcript;}
    }
	
    my @transcripts_minus=$self->_order_transcripts_acc2coord(@transcripts_minus);   
    my @transcripts_plus=$self->_order_transcripts_acc2coord(@transcripts_plus);   

    my $bmp_transcripts_plus=$self->_bumpable_transcripts(@transcripts_plus);
    my $bmp_transcripts_minus=$self->_bumpable_transcripts(@transcripts_minus);


    my $strand=1;
    my $y_coord_end_plus=$self->_create_transcripts($contig,$strand,$plus,$type,$bmp_transcripts_plus);

    my $strand=-1;
    my $y_coord_end_minus=$self->_create_transcripts($contig,$strand,$minus,$type,$bmp_transcripts_minus);

    #&eprof_end('Contig.pm - gene_get');

    return ($y_coord_end_plus, $y_coord_end_minus);





}



sub _draw_features
{
    my ($self,$plus,$minus)=@_;

    my @super_feature;
    my $contig=$self->_get_contig; 

   if(0) { print STDERR 'Contig.pm - draw features';
   my $i=0;
     while(my($package,$filename,$line,$subroutine)=caller($i++)) {
         print STDERR " >  $subroutine $filename line $line\n";
     }
   }

  
    &eprof_start('Contig.pm - fd - external_feat_get');

    my @features=$contig->get_all_ExternalFeatures();

    &eprof_end('Contig.pm - fd - external_feat_get');

	## need to sort external features into SNPs or traces and treat them differently
	my @snp;
	my %TR;
	my @exonerate;
	foreach my $f (@features){

		# We don't do Tcores any more....
		if ($f->isa("Bio::EnsEMBL::ExternalData::Variation")) {
			# A SNP
			push(@snp, $f);
		} elsif ($f->isa("Bio::EnsEMBL::FeaturePair")) {
			# An Exonerate trace match
			if (! defined $TR{'Mouse'}){
				$TR{'Mouse'} = [];
			}
			if ($f->analysis->dbID == 7) { # its an exonerate mouse trace match
				push (@{$TR{'Mouse'}}, $f);
			}
		}
	}

    my @tracelist;
    my $tracelink;
    my $label;
    foreach my $key (sort keys %TR){
    	$label = $key. "_link";
		$label =~ s/ /_/g;					#Mus musculus_link => Mus_musculus_link
		#print STDERR "Key: ", $label, "\n";
    	$tracelink=$self->trace_link;
    	@tracelist=($key,$tracelink,\@{$TR{$key}});    
    	push @super_feature,\@tracelist;
	
    }

    my $snplink=$self->snp_link;
    my @snplist=('snp',$snplink,\@snp);    
    push @super_feature,\@snplist;

    &eprof_start('Contig.pm - fd - marker_feat_get');
    
    my $link=$self->marker_link;
    my @features=$contig->get_landmark_MarkerFeatures;   

    my @list=('marker',$link,\@features);
    push @super_feature,\@list;
    &eprof_end('Contig.pm - fd - marker_feat_get');

    &eprof_start('Contig.pm - fd - genscan_get');
    my $link=$self->genscan_link;
    my @features;
    foreach my $seq_feat ($contig->get_all_PredictionFeatures)
    {
	push @features,$seq_feat->sub_SeqFeature();
    }


    my @list=('genscan',$link,\@features);
    push @super_feature,\@list;
    &eprof_end('Contig.pm - fd - genscan_get');

    &eprof_start('Contig.pm - fd - simil_get');

    my $glob_bp = 100;
    my @features=$contig->get_all_SimilarityFeatures_above_score("unigene.seq",80,$glob_bp);  

    my $link=$self->est_link;
    my @list=('unigene',$link,\@features);
    push @super_feature,\@list;

    my @vert=$contig->get_all_SimilarityFeatures_above_score("embl_vertrna",80,$glob_bp);  
    my $link=$self->vert_link;
    my @list=('mRNA',$link,\@vert);
    push @super_feature,\@list;

    my @features=$contig->get_all_SimilarityFeatures_above_score("sptr",80,$glob_bp);  
    my $link=$self->swiss_link;
    my @list=('sptr',$link,\@features);
    push @super_feature,\@list;


    my @features=$contig->get_all_SimilarityFeatures_above_score("pfam",25,$glob_bp);      
    my $link=$self->pfam_link;
    my @list=('pfam',$link,\@features);
    push @super_feature,\@list;


    my @features=$contig->get_all_SimilarityFeatures_above_score("cpg",25,$glob_bp);      
    my $link=$self->cpg_link;
    my @list=('cpg',$link,\@features);
    push @super_feature,\@list;


    my @features=$contig->get_all_SimilarityFeatures_above_score("tRNA",25,$glob_bp);      
    my $link=$self->tRNA_link;
    my @list=('tRNA',$link,\@features);
    push @super_feature,\@list;





	

    &eprof_end('Contig.pm - fd - simil_get');
    
    &eprof_start('Contig.pm - fd - repeat_get');
    my $link=$self->repeat_link;
    my @features=$contig->get_all_RepeatFeatures;   
    my @list=('repeats',$link,\@features);
    push @super_feature,\@list;
    &eprof_end('Contig.pm - fd - repeat_get');


    &eprof_start('Contig.pm - fd - split_and_create');	
    foreach my $list_ref (@super_feature)
    {
	my ($one,$two)=
	    $self->_split_features_acc2strand_and_create($contig,$plus,$minus,$list_ref->[0],$list_ref->[1],
							 @{$list_ref->[2]});    
	if (defined $one){$plus=$one;}
	if (defined $two){$minus=$two;}

    }
    &eprof_end('Contig.pm - fd - split_and_create');	


    return ($plus,$minus);
 
}


sub _draw_scale
{

    my ($self,$plus,$minus)=@_;

    my $y_coord_end_plus;
    my $y_coord_end_minus;

    my @elements=$self->_calculate_scale_elements_coord;
   
    my $hash;

    $hash->{-1}=$minus;
    $hash->{1}=$plus;
  
    foreach my $strand (keys %{$hash})
    {
	
	my $end=$self->_create_scale($strand,$hash->{$strand},@elements);
	if ($strand==1){$y_coord_end_plus=$end;}
	if ($strand==-1){$y_coord_end_minus=$end;}

    }  

    return ($y_coord_end_plus, $y_coord_end_minus);

}



sub _create_scale
{

    my ($self,$strand,$start,@elements)=@_;

    my $contig=$self->_get_contig;
    my $height=$self->gene_feature_height;
    my $end;


    my ($y_start,$y_end)=$self->_calc_y_coord($strand,$start,0);
   

    my $superlinear=SuperLinearElement->new($self->_get_image_par);
    my $linear=LinearElement->new($self->_get_image_par); 


	my %args=(name=>"scale (bp)",type=>'printable',subtype=>'small',
		  x_start=>$self->_get_image_par->{left_margin},y_start=>$y_start+0.5*$height+$self-> _legend_adjustment,color=>"black");
	
	$linear->_add_Element(BasicElement->new(%args));


    my %args= (type=>'drawable', subtype=>'line',color=>'black',
	       x_start=>$self->_get_image_par->{left_margin}+$self->_get_image_par->{legend_margin},
	       x_end=>$self->_get_image_par->{x_img_len}-$self->_get_image_par->{right_margin},
	       y_start=>$y_start,y_end=>$y_end);
    
    $linear->_add_Element(BasicElement->new(%args));



    my $counter;
    my $print_start;

    foreach my $el (@elements){
	my ($x_start,$x_end)=$self->_calc_x_coord($el,$el);
	my ($y_start,$y_end)=$self->_calc_y_coord($strand,$start,$height);	
	$counter++;

	$print_start=$x_start-12;
	if ($counter==1){$print_start=$x_start-2;}

	my $number_start;

	if ($strand==1){$number_start=$y_start-$strand*15;}
	if ($strand==-1){$number_start=$y_start-$strand*10;}

	if ($self->_get_image_par->{x_img_len}-$self->_get_image_par->{right_margin}-$x_start>100 || $counter==$#elements+1){	
	    my %args=(name=>$el+$self->set_offset,type=>'printable',subtype=>'small',
		      x_start=>$print_start,y_start=>$number_start,color=>"black");
	    
	    $linear->_add_Element(BasicElement->new(%args));
	    
	    
	    my %args= (name=> $el,type=>'drawable', subtype=>'filledrect',color=>'black',
		       x_start=>$x_start,x_end=>$x_end,y_start=>$y_start,y_end=>$y_end);
	    
	    $linear->_add_Element(BasicElement->new(%args));
	}
	if ($strand==-1){$end=$y_end;}
	if ($strand==1){$end=$y_start;}
	

    }

    $superlinear->_add_Element($linear);
    $self->_add_Element($superlinear);


  return $end;


}




sub _calculate_scale_elements_coord
{
    my ($self)=@_;

    my $contig=$self->_get_contig;
    my $repeats=8;
    my $factor=1;
    my $counter;

    while ($repeats*$factor<$contig->length){$factor=$factor*100;}
    while($factor*$counter<$contig->length){$counter++;}

    my $i;
    my @elements;
    for($i=0;$i<$counter;$i++)
    {
	if ($i==0 && !defined $self->set_offset){ push @elements,1;}
	else {push @elements,$i*$factor;}
    }
    push @elements,$contig->length;

    return @elements;

}





sub _exclude_exons_from_diffrent_contigs
{
    my ($self,$contig,@transcripts)=@_;

    my @new_transcripts;

    my $hash=$self->_first_last_exon;


	foreach my $transcript(@transcripts)
	{
	    my ($first,$last);
	    my $f_exon=$transcript->start_exon->id;
	    my $l_exon=$transcript->end_exon->id;
	  

	    my @new_exons;	    
	    my @exons = $transcript->each_Exon();
	    
	    foreach my $exon(@exons) 
	    {

		if ($exon->seqname eq $contig->id){push @new_exons,$exon;}
	        if ($exon->seqname eq $contig->id && $f_exon eq $exon->id){$first=1;}
		if ($exon->seqname eq $contig->id && $l_exon eq $exon->id){$last=1;}

	    }
	 
	   
	    my @array=($first,$last);
	    $hash->{$transcript->id}=\@array; 
	  
	    if (scalar @new_exons !=0){ 
		my $new_transcript=Bio::EnsEMBL::Transcript->new(@new_exons);
		$new_transcript->id($transcript->id);
		$new_transcript->gene_is_known($transcript->gene_is_known);
		if ($transcript->gene_name){
		    $new_transcript->gene_name($transcript->gene_name);	
		}
		else {
		    $new_transcript->gene_name($transcript->id);	
		} 
		
		push @new_transcripts,$new_transcript;
	    }
	    
	}

    $self->_first_last_exon($hash);
    
    
    return @new_transcripts;


}







sub _bumpable_transcripts
{

    my ($self,@transcripts)=@_;

    my $contig=$self->_get_contig;
    my $level_count=1;
    my $level;
    my $transcript_counter;  
    my $hash;
  

    foreach my $transcript (@transcripts)
    {			
	$transcript_counter++;
	
	my @exons = sort { $a->start <=> $b->start } $transcript->each_Exon();
	my $exon_counter;
	foreach my $exon ( @exons ){$exon_counter++;}	

	my $transcript_start  = $exons[0]->start;
	my $transcript_end    = $exons[$exon_counter-1]->end;
	my $transcript_strand = $exons[0]->strand;

	while (int $transcript_start<= int $level->{$level_count}){$level_count++;}
	
	$level->{$level_count}=$transcript_end;	
	push @{$hash->{$level_count}},$transcript;
	$level_count=1;	
    }    
    return  $hash;
}


sub _order_transcripts_acc2coord
{

    my ($self,@transcripts)=@_;

    my @plus;
    my @minus;
    my @all;

    foreach my $transcript (@transcripts)
    {	
	if (($transcript->each_Exon)[0]->strand ==1){push @plus,$transcript;}
	if (($transcript->each_Exon)[0]->strand ==-1){push @minus,$transcript;}
    }
 
    my @minus_reversed = sort { $a->end_exon->start <=> $b->end_exon->start } @minus;
    my @plus_reversed = sort { $a->start_exon->start <=> $b->start_exon->start } @plus;

    push @all,@plus_reversed,@minus_reversed;

    return @all;
}




sub _create_transcripts
{

    my ($self,$contig,$strand,$start,$type,$trans)=@_;

    my $y_coord_end;
    my $height=$self->gene_feature_height;
    my $intron_height=1;
    my $hashref=$self->_first_last_exon;

    foreach my $level( sort {$a<=>$b} keys %{$trans})
    { 

		my @transcripts=@{$trans->{$level}};
		my $keep_exon_end;
		my $superlinear=SuperLinearElement->new($self->_get_image_par);
		my $linear=LinearElement->new($self->_get_image_par);   
		my ($y_start,$y_end)=$self->_calc_y_coord($strand,$start,0,$level);
		my %args=(	name=>$type,
					type=>'printable',
					subtype=>'small',
			  		x_start=>$self->_get_image_par->{left_margin},
			  		y_start=>$y_start+0.5*$height+$self->_legend_adjustment,
					color=>'black'
		);

		$linear->_add_Element(BasicElement->new(%args));
		$superlinear->_add_Element($linear);    

		foreach my $transcript(@transcripts)
		{

	    	my $new_transcript_status=1;
	    	my $transcript_start;
	    	my $transcript_end;  
	    	my $transcript_y_start;
	    	my $transcript_y_end;
	    	my $trans_color;	
	    	my $linear=LinearElement->new($self->_get_image_par); 
	    	my @exons = sort { $a->start <=> $b->start } $transcript->each_Exon();

	    	foreach my $exon ( @exons )
	    	{	
				my ($x_start,$x_end)= $self->_calc_x_coord($exon->start,$exon->end);
				my ($y_start,$y_end)=$self->_calc_y_coord($strand,$start,$height,$level);
				my $last_end=$x_end;
				if ($new_transcript_status==1)
				{
		    		$transcript_start=$x_start;
		    		$transcript_y_start=$y_start;
				}
				$transcript_end=$x_end;
				$transcript_y_end=$y_end;

				$y_coord_end=$y_start+0.5*$height;
				my $color;

				if( $transcript->gene_is_known) {
		    		$color = "rust"; $trans_color="rust";
				} else {
		    		$color = 'black'; $trans_color=$self->transcript_color;
		    		if ($type eq "EMBL trans"){
						$trans_color=$self->embl_transcript_color;
		    			$color=$self->embl_transcript_color;
		    		}
				}
				my %args= (	name=> $exon->id,
							type=>'drawable', 
							subtype=>'filledrect',
							color=>$color,
			   				x_start=>$x_start,
							x_end=>$x_end,
							y_start=>$y_start,
							y_end=>$y_end
				);

				$linear->_add_Element(BasicElement->new(%args));

				if ($new_transcript_status!=1)
				{		 
		    		my ($x_start,$x_end)= $self->_calc_x_coord($keep_exon_end,$exon->start);
		    		my $new_y_start=$y_start+0.5*($y_end-$y_start)-0.5*$intron_height;
		    		my $y_end=$new_y_start+$intron_height;

		    		my %args= (	type=>'drawable', 
								subtype=>'filledrect',
								color=>$trans_color,
			    		   		x_start=>$x_start,
								x_end=>$x_end,
								y_start=>$new_y_start,
								y_end=>$y_end
					);
		    		$linear->_add_Element(BasicElement->new(%args));
				}
				$keep_exon_end=$exon->end;  
				$new_transcript_status=0;

				my $left_end=$self->_get_image_par->{left_margin}+$self->_get_image_par->{legend_margin};
				my $right_end=$self->_get_image_par->{x_img_len}-$self->_get_image_par->{right_margin};

				if (($hashref->{$transcript->id}->[0] !=1 && $strand ==1)||($hashref->{$transcript->id}->[1] !=1 && $strand ==-1))
				{
		    		#print STDERR "need to add the start ",$transcript->id," first ",
		    		#$hashref->{$transcript->id}->[0] ," last ",$hashref->{$transcript->id}->[1],"\n";

		    		my $new_y_start=$y_start+0.5*($y_end-$y_start)-0.5*$intron_height;
		    		my $y_end=$new_y_start+$intron_height;
		    		my %args= (	type=>'drawable', 
								subtype=>'filledrect',
								color=>$trans_color,
			    		   		x_start=>$left_end,
								x_end=>$x_end,
								y_start=>$new_y_start,
								y_end=>$y_end
					);
		    		$linear->_add_Element(BasicElement->new(%args));
	    		}

				if (($hashref->{$transcript->id}->[1] !=1 && $strand ==1)||($hashref->{$transcript->id}->[0] !=1 && $strand ==-1) )
				{
		    		#print STDERR "need to add the end ",$transcript->id," first ",
		    		#$hashref->{$transcript->id}->[0] ," last ",$hashref->{$transcript->id}->[1],"\n";
		    		my $new_y_start=$y_start+0.5*($y_end-$y_start)-0.5*$intron_height;
		    		my $y_end=$new_y_start+$intron_height;
		    		my %args= (	type=>'drawable', 
								subtype=>'filledrect',
								color=>$trans_color,
			    		   		x_start=>$last_end,
								x_end=>$right_end,
								y_start=>$new_y_start,
								y_end=>$y_end
					);
		    		$linear->_add_Element(BasicElement->new(%args));
				}
	    	}	

	    	$superlinear->_add_Element($linear);
	    	my $linear=LinearElement->new($self->_get_image_par); 
	    	my $mouseover=$transcript->id;     
	    	my %new_link;	
			if ($transcript->id =~ /(.*)\.trans\.\d+/){
		    	%new_link=%{$self->embltrans_link};	
			} else {
	    		%new_link=%{$self->transcript_link};	
			}
	    	my $url;
	    	my $mac;
			## This is where we add menus for the transcripts (as opposed to the features!)
	    	foreach my $key ( keys %new_link){
				if ($transcript->id =~ /(.*)\.trans\.\d+/){
					#print STDERR "EMBL: $type\n";
					my $transname = $1;
					%new_link->{$key}=%new_link->{$key}.$1;
					if ($key eq 'caption'){%new_link->{$key}='EMBL: '. $1;}
					if (defined $ENV{'NO_JS_MENUS'}){$url='http://www.ebi.ac.uk/cgi-bin/emblfetch?'.$1;$mac=1;}
				} else {
					#print STDERR "NON EMBL: $type\n";
					%new_link->{$key}=%new_link->{$key}.$transcript->id;
					if ($key eq 'caption'){%new_link->{$key}=$transcript->gene_name;}
				}
				if ($key eq 'mac'){$url=%new_link->{$key}; $mac=1;}
	    	}
	    	if ($mac!=1){$url = &JSTools::js_menu(\%new_link);}
			#print STDERR "jsURL - $url\n";
	    	my %args=(	name=>$transcript->id,
						type=>'IMAP',
						x_start=>$transcript_start,
						x_end=>$transcript_end,
		      			y_start=>$transcript_y_start,
						y_end=>$transcript_y_end,
						url=>$url,
						mouseover=>$mouseover
			);

	    	$linear->_add_Element(BasicElement->new(%args));
	    	$superlinear->_add_Element($linear);
		}    
		$self->_add_Element($superlinear);    

    }
    
    return $y_coord_end;
}





sub _split_features_acc2strand_and_create
{

    my ($self,$contig,$plus,$minus,$name,$link,@features)=@_;

    my @features_plus;
    my @features_minus; 
    foreach my $feature(@features)
    {

	if ($feature->strand==1){push @features_plus,$feature;}
	if ($feature->strand==-1){push @features_minus,$feature;}
	if ($feature->strand==0){push @features_plus,$feature;}

    }

    &eprof_start('Contig.pm - create_features');  
    my $plus=$self->_create_features($contig,$plus,1,$name,$link,@features_plus);
    my $minus=$self->_create_features($contig,$minus,-1,$name,$link,@features_minus);
    &eprof_end('Contig.pm - create_features');  

    return ($plus,$minus);

}


sub _create_features
{

    my ($self,$contig,$start,$strand,$name,$link,@features)=@_;
    my $y_coord_end;
    my $prev;
    my $prevend;   
    my @processed;
    my $linear=LinearElement->new($self->_get_image_par);  
    my $height=$self->gene_feature_height;
    @features = sort { $a->start <=> $b->start } @features;
    
   	# Calculate the image offset from left, and the scaling multiplier to 
	# convert from coordinates to pixels.
	my $image_param_ref=$self->_get_image_par;
	my $img_offset=$image_param_ref->{left_margin} 
					+$image_param_ref->{legend_margin};

	my	$img_len=$image_param_ref->{x_img_len} 
					-$image_param_ref->{left_margin} 
					-$image_param_ref->{legend_margin} 
					-$image_param_ref->{right_margin};
	
	my $img_pixel_bases=$img_len/$contig->length;
	
    foreach my $feature ( @features ) {
     	my ($x_start, $x_end)=$self->_calc_x_coord($feature->start, $feature->end);
	 	$x_start = floor($x_start)-1;
        $x_end   = ceil($x_end)+1;

		if( defined $prev ) {
	    	if( $x_start <= $prevend && $x_end >= $prevend) {
				if( $feature->start < $prev->end ) { 
		    		$prev->end($feature->end); # 
				} else {
		    		if( $x_end < $prevend +5 ) {
						# discard, but don't merge
						next;
		    		} else {
						# make new block 
						push(@processed,$prev);
						$prev = undef;
						$prevend = $x_end;               
		    		}
				}
	    	} else {
				push(@processed,$prev);
				$prev = undef;
                $prevend = $x_end;               
	    	}
		} else {
	    	$prev = $feature;
		}
    }			 
    



    # bypass globbing (we are loosing single features), temp fix
    my @processed = @features;

    foreach my $feature (@processed){

		my ($y_start,$y_end)=$self->_calc_y_coord($strand,$start,0);
		my %args=(	name=>$name,
					type=>'printable',
					subtype=>'small',
			  		x_start=>$self->_get_image_par->{left_margin},
					y_start=>$y_start+0.5*$height+$self-> _legend_adjustment,
					color=>"black"
		);

		$linear->_add_Element(BasicElement->new(%args));
		my ($y_start,$y_end)=$self->_calc_y_coord($strand,$start,$height);
		my ($x_start,$x_end)= $self->_calc_x_coord($feature->start,$feature->end);

		if ($strand==-1){$y_coord_end=$y_end;}
		if ($strand==1){$y_coord_end=$y_start;}
		$y_coord_end=$y_start+0.5*$height;

		my $color='blue';
		if ($name eq 'trace'){$color=$self->trace_color;}
		if ($name eq 'Mus musculus'){$color=$self->Mus_musculus_color;}
		if ($name eq 'Mouse'){$color=$self->Mouse_color;}
		if ($name eq 'snp'){$color=$self->snp_color;}
		if ($name eq 'pfam'){$color=$self->pfam_color;}
		if ($name eq 'sptr'){$color=$self->swiss_color;}
		if ($name eq 'unigene'){$color=$self->est_color;}
		if ($name eq 'genscan'){$color=$self->genscan_color;}
		if ($name eq 'repeats'){$color=$self->repeat_color;}
		if ($name eq 'mRNA'){$color=$self->vert_color;}
		if ($name eq 'marker'){$color=$self->marker_color;}
		if ($name eq 'tRNA'){$color=$self->tRNA_color;}
		if ($name eq 'cpg'){$color=$self->cpg_color;}

		my %args= (	name=> $feature->id,
					type=>'drawable', 
					subtype=>'filledrect',
					color=>$color,
			   		x_start=>$x_start,
					x_end=>$x_end,
					y_start=>$y_start,
					y_end=>$y_end
		);

		$linear->_add_Element(BasicElement->new(%args));
		my $mouseover=$feature->id;
   		unless ($feature->isa ("Bio::EnsEMBL::ExternalData::Variation") 
							|| $feature->isa("Bio::EnsEMBL::SeqFeature") 
							|| $feature->isa("Bio::EnsEMBL::ExternalData::Tcore")
							|| $feature->isa("Bio::EnsEMBL::FeaturePairI")){

				$mouseover=$mouseover." start ".$feature->hstart." end ".$feature->hend;     
		}
		if (defined $link){ 
			my %new_link=%{$link};	
			my $mac;
			my $url;
			foreach my $key ( keys %new_link){
	    		unless ( %new_link->{$key} eq " "){
			    # JWS NASTY HACK FOR UNIGENE
			    if ($name eq 'unigene'){
				my $unigeneid = $feature->id;
				$unigeneid =~ s/\./&CID=/;
				%new_link->{$key}=%new_link->{$key}.$unigeneid;
			    }
			    else {
				%new_link->{$key}=%new_link->{$key}.$feature->id;
			    }
	    		}
	    		if ($key eq 'caption'){%new_link->{$key}=$feature->id;}
	    		if ($key eq 'mac'){$url=%new_link->{$key}; $mac=1;}
			}
			#print STDERR "feature type: $name\n";
			#print STDERR "feature id:" . $feature->id . "\n";
			if ($name eq 'cpg'){
				$new_link{'caption'} = 'options';
				$new_link{'CPG Island'} = '';
			}
			if ($name eq 'tRNA'){
				$new_link{'caption'} = 'options';
				$new_link{'tRNA'} = '';
			}
			if ($mac !=1){
				#if($name eq 'repeats'){
				#	$url = &JSTools::js_tooltip('Repeat');
				#} else {
					$url = &JSTools::js_menu(\%new_link);
				#}
			}		
			if ($name eq 'snp'){$x_start=$x_start-1;$x_end=$x_end+1;}
			my %args=(	name=>$feature->id,
						type=>'IMAP',
						x_start=>$x_start,
						x_end=>$x_end,
		  				y_start=>$y_start,
						y_end=>$y_end,
						url=>$url,
						mouseover=>$mouseover
			);
			$linear->_add_Element(BasicElement->new(%args));
    	}	
    }
    $self->_add_Element($linear);    
    return $y_coord_end;    
}


sub _calc_x_coord {
    my ($self,$coord_start,$coord_end)=@_;
    my $image_param_ref=$self->_get_image_par;	
    my $img_offset      = $image_param_ref->{'img_offset'};
    my $img_pixel_bases = $image_param_ref->{'img_pixel_bases'};    
    my $feature_len=($coord_end-$coord_start) * $img_pixel_bases;
    my $x_start = ($coord_start*$img_pixel_bases) + $img_offset;   
    my $x_end   = $x_start + $feature_len;  
    return ($x_start,$x_end);
}




sub _calc_y_coord
{

    my ($self,$strand,$start,$height,$level)=@_;

    if (! defined $level){$level=1;}
    my $y_ct=$start-$strand*$level*$self->set_gap;
    my $y_start=$y_ct-0.5*$height;		
    my $y_end=$y_ct+0.5*$height;

    return ($y_start,$y_end);


}




sub _get_contig
{
    my ($self,$contig)=@_;
    
    if ($contig){$self->{'contig'}=$contig;}
    return $self->{'contig'};
   
}


sub _get_image_par
{
    my ($self,$image_par)=@_;
    
    if ($image_par){$self->{'image_par'}=$image_par;}

	
    return $self->{'image_par'};
   
}




sub get_GIF {
    my ($self,$fh)=@_;
    my @element_list;
    my $var=$self;
    foreach my $el($var->_each_Element){
		if ($el->isa("LinearElement")){
	    	push @element_list,$el->_each_Element;
		   # print STDERR "I am linear 1 ", $el->name,"\n";
		}    
		else {
	    	#print "not a linear 1 element\n";
	    	$var=$el;
	    	foreach my $el($var->_each_Element){
				if ($el->isa("LinearElement")){
		    		push @element_list,$el->_each_Element;
		    		#print "I am linear 2 ", $el->name,"\n";
				}    
				else {
		    		#print "not a linear 2 element\n";
		    		$var=$el;
		    		foreach my $el($var->_each_Element){
						if ($el->isa("LinearElement")){
			    			push @element_list,$el->_each_Element;
			    			#print "I am linear 3\n";
						}    
						else {
			    			#print "not a linear 3 element\n";
			    			$var=$el;
						}
		    		}
				}
	    	}
		}
    }
    foreach my $element (@element_list){
		$element->y_start($element->y_start+$self->_set_start);
		$element->y_end($element->y_end+$self->_set_start);	
    }
    my $var=GIF->new(\@element_list,$self->_get_image_par,$fh);
    my $gif=$var->get_GIF($fh);
    
}






sub set_gap
{
    my ($self,$gap)=@_;
    
    if ($gap){$self->{'gap'}=$gap;}
    if (! defined $self->{'gap'}){$self->{'gap'}=11;}
    return $self->{'gap'};
   
}


sub set_offset
{
    my ($self,$value)=@_;
    
    if ($value){$self->{'offset'}=$value;}
    return $self->{'offset'};
   
}






############################################################3
# Ext. DB colour info
############################################################3

sub snp_color
{
    my ($self,$color)=@_;
    
    if ($color){$self->{'snp_color'}=$color;}
    if (! defined $self->{'snp_color'}){$self->{'snp_color'}='blue';}
    return $self->{'snp_color'};
   
}


sub pfam_color
{
    my ($self,$color)=@_;
    
    if ($color){$self->{'pfam_color'}=$color;}
   if (! defined $self->{'pfam_color'}){$self->{'pfam_color'}='flora';} 
    return $self->{'pfam_color'};
   
}

sub repeat_color
{
    my ($self,$color)=@_;
    
    if ($color){$self->{'repeat_color'}=$color;}
    if (! defined $self->{'repeat_color'}){$self->{'repeat_color'}='darkgreen';}
    return $self->{'repeat_color'};
   
}


sub swiss_color
{
    my ($self,$color)=@_;
    
    if ($color){$self->{'swiss_color'}=$color;}
    if (! defined $self->{'swiss_color'}){$self->{'swiss_color'}='darkorange';}
    return $self->{'swiss_color'};
   
}

sub est_color
{
    my ($self,$color)=@_;
    
    if ($color){$self->{'est_color'}=$color;}
    if (! defined $self->{'est_color'}){$self->{'est_color'}='red';}
    return $self->{'est_color'};
   
}


sub genscan_color
{
    my ($self,$color)=@_;
    
    if ($color){$self->{'genscan_color'}=$color;}
   if (! defined $self->{'genscan_color'}){$self->{'genscan_color'}='light_sea_green';} 
    return $self->{'genscan_color'};
   
}

sub transcript_color
{
    my ($self,$color)=@_;
    
    if ($color){$self->{'transcript_color'}=$color;}
  if (! defined $self->{'transcript_color'}){$self->{'transcript_color'}='black';}  
    return $self->{'transcript_color'};
   
}


sub embl_transcript_color
{
    my ($self,$color)=@_;
    
    if ($color){$self->{'embl_transcript_color'}=$color;}
  if (! defined $self->{'embl_transcript_color'}){$self->{'embl_transcript_color'}='ebigreen';}  
    return $self->{'embl_transcript_color'};
   
}

sub vert_color
{
    my ($self,$color)=@_;
    
    if ($color){$self->{'vert_color'}=$color;}
  if (! defined $self->{'vert_color'}){$self->{'vert_color'}='pine';}  
    return $self->{'vert_color'};
   
}


sub marker_color
{
    my ($self,$color)=@_;
    
    if ($color){$self->{'marker_color'}=$color;}
    if (! defined $self->{'marker_color'}){$self->{'marker_color'}='pine';}  
    return $self->{'marker_color'};
   
}

sub trace_color
{
    my ($self,$color)=@_;
    
    if ($color){$self->{'trace_color'}=$color;}
    if (! defined $self->{'trace_color'}){$self->{'trace_color'}='sangerblue';}  
    return $self->{'trace_color'};
   
}

sub Mus_musculus_color
{
    my ($self,$color)=@_;
    
    if ($color){$self->{'trace_color'}=$color;}
    if (! defined $self->{'trace_color'}){$self->{'trace_color'}='deep_pink';}  
    return $self->{'trace_color'};
   
}

sub Mouse_color
{
    my ($self,$color)=@_;
    
    if ($color){$self->{'trace_color'}=$color;}
    if (! defined $self->{'trace_color'}){$self->{'trace_color'}='deep_pink';}  
    return $self->{'trace_color'};
   
}

sub tRNA_color
{
    my ($self,$color)=@_;
    
    if ($color){$self->{'tRNA_color'}=$color;}
    if (! defined $self->{'tRNA_color'}){$self->{'tRNA_color'}='gold';}  
    return $self->{'tRNA_color'};
   
}
sub cpg_color
{
    my ($self,$color)=@_;
    
    if ($color){$self->{'cpg_color'}=$color;}
    if (! defined $self->{'cpg_color'}){$self->{'cpg_color'}='violet1';}  
    return $self->{'cpg_color'};
   
}

############################################################3
# Ext. DB link info
############################################################3

sub embltrans_link
{
    my ($self,$link)=@_;
    
    if ($link){$self->{'embltrans_link'}=$link;}
    return $self->{'embltrans_link'};
   
}


sub snp_link
{
    my ($self,$link)=@_;
    
    if ($link){$self->{'snp_link'}=$link;}
    return $self->{'snp_link'};
   
}


sub pfam_link
{
    my ($self,$link)=@_;
    
    if ($link){$self->{'pfam_link'}=$link;}
    return $self->{'pfam_link'};
   
}

sub repeat_link
{
    my ($self,$link)=@_;
    
    if ($link){$self->{'repeat_link'}=$link;}
    return $self->{'repeat_link'};
   
}


sub swiss_link
{
    my ($self,$link)=@_;
    
    if ($link){$self->{'swiss_link'}=$link;}
    return $self->{'swiss_link'};
   
}

sub est_link
{
    my ($self,$link)=@_;
    
    if ($link){$self->{'est_link'}=$link;}
    return $self->{'est_link'};
   
}


sub genscan_link
{
    my ($self,$link)=@_;
    
    if ($link){$self->{'genscan_link'}=$link;}
    return $self->{'genscan_link'};
   
}

sub transcript_link
{
    my ($self,$link)=@_;
    
    if ($link){$self->{'transcript_link'}=$link;}
    return $self->{'transcript_link'};
   
}

sub vert_link
{
    my ($self,$link)=@_;
    
    if ($link){$self->{'vert_link'}=$link;}
    return $self->{'vert_link'};
   
}


sub marker_link
{
    my ($self,$link)=@_;
    
    if ($link){$self->{'marker_link'}=$link;}
    return $self->{'marker_link'};
   
}

sub Mus_musculus_link
{
    my ($self,$link)=@_;
    
    if ($link){$self->{'trace_link'}=$link;}
    return $self->{'trace_link'};
   
}

sub Mouse_link
{
    my ($self,$link)=@_;
    
    if ($link){$self->{'trace_link'}=$link;}
    return $self->{'trace_link'};
   
}

sub trace_link
{
    my ($self,$link)=@_;
    
    if ($link){$self->{'trace_link'}=$link;}
    return $self->{'trace_link'};
   
}


sub cpg_link
{
    my ($self,$link)=@_;
    
    if ($link){$self->{'cpg_link'}=$link;}
    return $self->{'cpg_link'};
   
}



sub tRNA_link
{
    my ($self,$link)=@_;
    
    if ($link){$self->{'tRNA_link'}=$link;}
    return $self->{'tRNA_link'};
   
}






sub gene_feature_height
{
    my ($self,$value)=@_;
    
    if ($value){$self->{'gene_feature_height'}=$value;}

    if (! defined $self->{'gene_feature_height'}){$self->{'gene_feature_height'}=8;}
    return $self->{'gene_feature_height'};
   
}



sub _set_start
{
    my ($self,$start)=@_;
    
    if ($start){$self->{'start'}=$start;}
    return $self->{'start'};
   
}

sub _first_last_exon
{
    my ($self,$hashref)=@_;
    
    if ($hashref){$self->{'first_last_exon'}=$hashref;}

    return $self->{'first_last_exon'};
   
}






sub _legend_adjustment
{
    my ($self,$value)=@_;
    
    if ($value){$self->{'legend_adjustment'}=$value;}
    if (  ! defined $self->{'legend_adjustment'}){$self->{'legend_adjustment'}=-12;}
    return $self->{'legend_adjustment'};
   
}







1;





