package Chromosome;

use strict;
use Bio::EnsemblViewer::LinearElement;
use Bio::EnsemblViewer::DrawableElement;
use Bio::EnsemblViewer::BasicElement;
use Bio::Root::Object;
use vars qw(@ISA);

@ISA = qw( Bio::Root::Object DrawableElement);



sub _initialize 
{
    my($self,@args) = @_;      
    my ($bands,$x_start,$x_end,$y_start,$y_end,$print,$arc_height,$mapbands)= 
    $self->_rearrange([qw(BANDS X_START X_END Y_START Y_END PRINT ARC_HEIGHT MAP_BANDS)],@args);

     my $make = $self->SUPER::_initialize;   

    $self->_x_start($x_start);
    $self->_x_end($x_end);  
    $self->_y_start($y_start);
    $self->_y_end($y_end);
    $self->_print($print);
    $self->_mapbands($mapbands);
    $self->_arc_height($arc_height);

    $self->_create_chromosome($bands);
    return $make;     
}


sub _create_chromosome

{
    my ($self,$bands)=@_;
    
    my @bands=@$bands;

    my $x_start=$self->_x_start;
    my $x_end=$self->_x_end;  
    my $y_start=$self->_y_start;
    my $y_end=$self->_y_end;

    my $arc_height=$self->_arc_height;	# height of curved endcaps

    if ($y_end<=$y_start){$self->throw('cant draw with such y margins');}
 
    my $count=scalar(@bands);
    my $canvas_factor=($y_end-$y_start-($arc_height*2))/$bands[-1]->end;   
    my $band_counter;   
  
    my @arr;
    my $print_margin=0;

    if ($self->_print){$print_margin=40;}
    
    my $colors = {
		    'outline'	=> 'darkgrey',
		    'gpos100'	=> 'lightgrey',
		    'gpos75'	=> 'lightgrey1',
		    'gpos50'	=> 'lightgrey3',
		    'gpos25'	=> 'lightgrey4',
		    'gneg'	=> 'white',
		    'gvar'	=> 'lightgrey2',
		};
	
    my $last_label_bottom = -1;

    foreach my $band (@bands){

        $band_counter++;
                
        my $subtype;
        my $end=($band->end)*$canvas_factor+$y_start+$arc_height-1;
        my $start=($band->start)*$canvas_factor+$y_start+$arc_height;
        
        my @args=($x_start,$x_end,$start,$end,$print_margin,$colors,$arc_height);
        
        if ($band_counter==1){
            $self->_create_first_band($band,@args);
        }
        elsif ($band_counter==$count){
            $self->_create_last_band($band,@args);
        }
        elsif ($band->stain eq 'stalk'){
            $self->_create_s2_band($band,@args);
        }
        elsif ($band->stain eq 'acen'){
            $self->_create_triangle_band($band,@args);
        }
        else {
            $self->_create_band($band,@args);
        }
    
	#####################
    	# Label, if required
    	#####################
    	if ($self->_print){
	    my $label_top=($end+$start) * 0.5 ;
	    if ($label_top > $last_label_bottom) {
            	my %args =(name=>$band->name,type=>'printable',subtype=>'tiny',
               	x_start=>$x_start,y_start=>$label_top,y_end=>$end,color=>'black');
	    	my $linear=LinearElement->new();
	    	$linear->_add_Element(BasicElement->new(%args));
    	    	$self->_add_Element($linear);
	   	
		$last_label_bottom=$label_top + 7;
	    }
		
    	}

	####################################
	# Imagemap the band into contigview
	####################################
    	if ($self->_mapbands){
	    my $url = "/perl/contigview?chr=".$band->chromosome."&vc_start=".($band->start - 100000)."&vc_end=".($band->end + 100000);
	    my $mouseover = "Band : ".$band->name;
	    my %args=(type=>'IMAP',x_start=>$x_start,x_end=>$x_end, y_start=>$start,y_end=>$end,url=>$url,mouseover=>$mouseover); 
	    my $linear=LinearElement->new();
	    $linear->_add_Element(BasicElement->new(%args));
    	    $self->_add_Element($linear);
	}
    }
}


#   First band made up of following sections:
#
#    -------
#   /       \
#   |       |   filled rectangle - 1, with outline 2
#   |       |
#   |       |
#
sub _create_first_band
{

    my ($self,$band,$x_start,$x_end,$start,$end,$print_margin,$colors,$arc_height)=@_;
    
    my $linear=LinearElement->new();
    $linear->name($band->name);

    my $outline_col=$colors->{'outline'};  
    my $fill_col=$colors->{$band->stain};
    
    #####################################
    # Filled rectangle - 1 in diag above
    #####################################
    my $subtype='filledrect';
    my %args=(  name        => $band->name,
                type        => 'drawable',subtype=>$subtype,
                x_start     => $x_start+$print_margin,
                x_end       => $x_end,
                y_start     => $start,
                y_end       => $end,
                color       => $fill_col,
                fillcolor   => $fill_col
            );
    $linear->_add_Element(BasicElement->new(%args));


    #################################################
    # Outline of filled rectangle  - 2 in diag above
    #################################################
    my $subtype='rect';
    my %args=(  name        => $band->name,
                type        => 'drawable',
                subtype     => $subtype,
                x_start     => $x_start+$print_margin,
                x_end       => $x_end,
                y_start     => $start,
                y_end       => $end,
                color       => $outline_col,
                fillcolor   => $fill_col
            );
   
    $linear->_add_Element(BasicElement->new(%args));
  
    ###################################################################
    # Overdraw the top lines of the rectangle we just drew to remove it
    ###################################################################
    my $subtype='line';
    my %args=(  
                name        => $band->name,
                type        => 'drawable',
                subtype     => $subtype,
                x_start     => $x_start+$print_margin+1,
                x_end       => $x_end-1,
                y_start     => $start,
                y_end       => $start,
                color       => $fill_col,
                fillcolor   => $fill_col
            );
   
    $linear->_add_Element(BasicElement->new(%args));


    my %args=(  
                name        => $band->name,
                type        => 'drawable',
                subtype     => $subtype,
                x_start     => $x_start+$print_margin+1,
                x_end       => $x_end-1,
                y_start     => $end,
                y_end       => $end,
                color       => $fill_col,
                fillcolor   => $fill_col
            );
   
    $linear->_add_Element(BasicElement->new(%args));

    #######################################
    # Draw top arc.  This is a filled arc.
    #######################################
    my $subtype='arc_up';
    my  %args=(
                name        => $band->name,
                type        => 'drawable',
                subtype     => $subtype,
                arc_height  => $arc_height,
                x_start     => $x_start+$print_margin,
                x_end       => $x_end,
                y_start     => $start-$arc_height,
                y_end       => $end,
                color       => $outline_col,
                fillcolor   => $fill_col
            );
   
    
    $linear->_add_Element(BasicElement->new(%args));  

    $self->_add_Element($linear);

}



#   last band made up of following sections:
#
#    -------
#   /       \
#   |       |   filled rectangle - 1, with outline 2
#   |       |
#   |       |
#

sub _create_last_band
{

    my ($self,$band,$x_start,$x_end,$start,$end,$print_margin,$colors,$arc_height)=@_;
    
    my $linear=LinearElement->new();
    $linear->name($band->name);
    
    my $outline_col=$colors->{'outline'};  
    my $fill_col=$colors->{$band->stain};

    #####################################
    # Filled rectangle - 1 in diag above
    #####################################
    my $subtype='filledrect';
    my %args=(  name        => $band->name,
                type        => 'drawable',
                subtype     => $subtype,
                x_start     => $x_start+$print_margin,
                x_end       => $x_end,
                y_start     => $start,
                y_end       => $end,
                color       => $fill_col,
                fillcolor   => $fill_col
            );

    $linear->_add_Element(BasicElement->new(%args));


    #################################################
    # Outline of filled rectangle  - 2 in diag above
    #################################################
    my $subtype='rect';
    my %args=(  name        => $band->name,
                type        => 'drawable',
                subtype     => $subtype,
                x_start     => $x_start+$print_margin,
                x_end       => $x_end,
                y_start     => $start,
                y_end       => $end,
                color       => $outline_col,
                fillcolor   => $fill_col
            );
   
    $linear->_add_Element(BasicElement->new(%args));
  
    ###################################################################
    # Overdraw the top line of the rectangle we just drew to remove it
    ###################################################################
    my $subtype='line';
    my %args=(  name        => $band->name,
                type        => 'drawable',
                subtype     => $subtype,
                x_start     => $x_start+$print_margin+1,
                x_end       => $x_end-1,
                y_start     => $end,
                y_end       => $end,
                color       => $fill_col,
                fillcolor   => $fill_col
            ); 
   
    $linear->_add_Element(BasicElement->new(%args));

    
    ###################################################################
    # Overdraw the bottom line of the rectangle we drew to remove it
    ###################################################################
    my $subtype='line';
    my %args=(  name        => $band->name,
                type        => 'drawable',
                subtype     => $subtype,
                x_start     => $x_start+$print_margin+1,
                x_end       => $x_end-1,
                y_start     => $start,
                y_end       => $start,
                color       => $fill_col,
                fillcolor   => $fill_col
            ); 
   
    $linear->_add_Element(BasicElement->new(%args));

    #########################################
    # Draw bottom arc.  This is a filled arc.
    #########################################
    my $subtype='arc_down';
    my  %args=( name        => $band->name,
                type        => 'drawable',
                subtype     => $subtype,
                arc_height  => $arc_height,
                x_start     => $x_start+$print_margin,
                x_end       => $x_end,
                y_start     => $end,
                y_end       => $end+$arc_height,
                color       => $outline_col,
                fillcolor   => $fill_col
            );

    
    $linear->_add_Element(BasicElement->new(%args));  

    $self->_add_Element($linear);


}



sub _create_s2_band
{
    my ($self,$band,$x_start,$x_end,$start,$end,$print_margin,$colors,$arc_height)=@_;
    
    my $linear=LinearElement->new();
    $linear->name($band->name);
    my $ind=($x_end-$x_start-$print_margin)*0.2;  
    
    my $outline_col=$colors->{'outline'};  
    my $fill_col=$colors->{'gvar'};
    
    my $subtype='filledpoly';

    ###################################
    # Create "hourglass" poly"
    ###################################
    my %args=(  name    => $band->name,
                type    => 'drawable',
                subtype => $subtype,
                x_y     =>  [
                                {
                                    'x' =>  $x_start+$print_margin,
                                    'y' =>  $start
                                },
                                {
                                    'x' =>  $x_start+$print_margin+$ind,
                                    'y' =>  $start+($end-$start)*0.3
                                },
                                {
                                    'x' =>  $x_start+$print_margin+$ind,
                                    'y' =>  $start+($end-$start)*0.7
                                },
                                {
                                    'x' =>  $x_start+$print_margin,
                                    'y' =>  $end
                                },
                                {
                                    'x' =>  $x_end,
                                    'y' =>  $end
                                },
                                {
                                    'x' =>  $x_end-$ind,
                                    'y' =>  $start+($end-$start)*0.7
                                },
                                {
                                    'x' =>  $x_end-$ind,
                                    'y' =>  $start+($end-$start)*0.3    
                                },
                                {
                                    'x' =>  $x_end,
                                    'y' =>  $start
                                },

                            ],
                color       =>  $outline_col,
                fillcolor   =>  $fill_col,
            );   
    
    $linear->_add_Element(BasicElement->new(%args));


    ###################################
    # Now draw the two horizontal bars 
    ###################################

    $subtype='filledrect';

    my  %args=(name=>$band->name,type=>'drawable',subtype=>$subtype,
           x_start=>$x_start+$print_margin+$ind,x_end=>$x_end-$ind,y_start=>$start+($end-$start)*0.3,
           y_end=>$start+($end-$start)*0.4,color=>$outline_col);
   
    
    $linear->_add_Element(BasicElement->new(%args));

    my  %args=(name=>$band->name,type=>'drawable',subtype=>$subtype,
           x_start=>$x_start+$print_margin+$ind,x_end=>$x_end-$ind,y_start=>$start+($end-$start)*0.6,
           y_end=>$start+($end-$start)*0.7,color=>$outline_col);
   
    
    $linear->_add_Element(BasicElement->new(%args));
    
    $self->_add_Element($linear);

}



sub _create_band
{
    my ($self,$band,$x_start,$x_end,$start,$end,$print_margin,$colors,$arc_height)=@_;

    my $linear=LinearElement->new();        
    $linear->name($band->name);

    my $outline_col=$colors->{'outline'};  
    my $fill_col=$colors->{$band->stain};

    my %args=(	name=>$band->name,
		type=>'drawable',
		subtype=>'filledrect',
		x_start=>$x_start+$print_margin,
		x_end=>$x_end,
		y_start=>$start,
		y_end=>$end,
		color=>$fill_col
	      );   
    
    $linear->_add_Element(BasicElement->new(%args));

    %args=(	name=>$band->name,
		type=>'drawable',
		subtype=>'line',
		x_start=>$x_start+$print_margin,
		x_end=>$x_start+$print_margin,
		y_start=>$start,
		y_end=>$end,
		color=>$outline_col
	    );   
    
    $linear->_add_Element(BasicElement->new(%args));

    %args=(	name=>$band->name,
		type=>'drawable',
		subtype=>'line',
		x_start=>$x_end,
		x_end=>$x_end,
		y_start=>$start,
		y_end=>$end,
		color=>$outline_col
	    );   
    
    $linear->_add_Element(BasicElement->new(%args));
        
    $self->_add_Element($linear);
    
}




sub _create_triangle_band
{
    my ($self,$band,$x_start,$x_end,$start,$end,$print_margin,$colors,$arc_height)=@_;

   
    my $outline_col=$colors->{'outline'};  
    my $linear=LinearElement->new();        
    $linear->name($band->name);
    
    my $subtype='triangle_up';
    
    if ($band->name =~ m/p/i ){
        $subtype="triangle_down";
    }
    

   my %args=(name=>$band->name,type=>'drawable',subtype=>$subtype,
          x_start=>$x_start+$print_margin,x_end=>$x_end,y_start=>$start,y_end=>$end,color=>$outline_col);   
    
    $linear->_add_Element(BasicElement->new(%args));
      
    $self->_add_Element($linear);
    
}




sub _print
{
    my ($self,$value)=@_;
    
    if (defined $value){$self->{'print'}=$value;}
    return $self->{'print'};
   
}



sub _mapbands
{
    my ($self,$value)=@_;
    
    if (defined $value){$self->{'mapbands'}=$value;}
    return $self->{'mapbands'};
   
}



sub _arc_height
{
    my ($self,$value)=@_;
    
    if (defined $value){$self->{'arc_height'}=$value;}
    return $self->{'arc_height'};
}   


sub name 
{
    my ($self,$value) = @_;
    if( defined $value ) {
    $self->{'name'} = $value;}
    return $self->{'name'};
}




1;







