
#
# Ensembl module for Bio::EnsEMBL::ExternalData::Tcore
#
# Cared for by Ewan Birney <birney@ebi.ac.uk>
#
# Copyright GRL and EBI
#
# You may distribute this module under the same terms as perl itself

# POD documentation - main docs before the code

=head1 NAME

Bio::EnsEMBL::ExternalData::Tcore - DESCRIPTION of Object

=head1 SYNOPSIS

Give standard usage here

=head1 DESCRIPTION

Describe the object here

=head1 CONTACT

Ensembl - ensembl-dev@ebi.ac.uk

=head1 APPENDIX

The rest of the documentation details each of the object methods. Internal methods are usually preceded with a _

=cut


# Let the code begin...


package Bio::EnsEMBL::ExternalData::Tcore;
use vars qw(@ISA);
use strict;

# Object preamble - inheriets from Bio::Root::RootI

use Bio::Root::RootI;
use Bio::SeqFeatureI;
use Bio::DBLinkContainerI;
use Bio::Annotation::DBLink;

@ISA = qw(Bio::SeqFeatureI  Bio::DBLinkContainerI Bio::Root::RootI );


sub new {
    my($class,@args) = @_;
    
    my $self = {};
    bless $self,$class;
   
    return $self;
}



sub create_from_array {
    my ($class,@array)=@_;
    my ($tcore_id,$contig_id,$start,$end,$adaptor)=@array;

    my $self= Bio::EnsEMBL::ExternalData::Tcore->new();

    $self->throw ("I need id") unless defined $tcore_id;
    $self->throw ("I need contig id") unless defined $contig_id;
    $self->throw ("start") unless defined $start;
    $self->throw ("I need end") unless defined $end;
    $self->throw ("I need adaptor") unless defined $adaptor;

    $self->id($tcore_id);
    $self->contig_id($contig_id);
    $self->start($start);
    $self->end($end);
    $self->source("Mus musculus");
    $self->adaptor($adaptor);

    return $self;
}


# return a list of hits, this tcore is made from

=head2 get_all_hits

  Title   : get_all_hits
  Usage   : $self->get_all_hits
  Function: get an arrayref of references to
          
  Returns : readname, raw_contig_id startbp, length, strand 
  Args    : 

=cut

sub get_all_hits {
  my $self = shift;
  my @hits; 

  if( ! defined $self->adaptor ) {
    return ();
  }
  @hits = $self->adaptor->list_hits_by_tcoreId( $self->id );
  # recalc beginning and ends
  foreach my $hit ( @hits ) {
    $hit->[2] -= $self->start-1;
  }
  return @hits;
}
  


sub sub_SeqFeature{
   my ($self,@args) = @_;
   return;
}





=head2 id

 Title   : id
 Usage   : $self->id($newval)
 Function: 
 Returns : value of id
 Args    : newvalue (optional)


=cut

sub id{
   my $self = shift;
   if( @_ ) {
      my $value = shift;
      $self->{'id'} = $value;
    }
    return $self->{'id'};

}

sub dbID {
   my $self = shift;
   if( @_ ) {
      my $value = shift;
      $self->{'id'} = $value;
    }
    return $self->{'id'};
}






=head2 start

 Title   : start
 Usage   : $self->start($newval)
 Function: 
 Returns : value of start
 Args    : newvalue (optional)


=cut

sub start{
   my $self = shift;
   if( @_ ) {
      my $value = shift;
      $self->{'start'} = $value;
    }
    return $self->{'start'};

}

=head2 end

 Title   : end
 Usage   : $self->end($newval)
 Function: 
 Returns : value of end
 Args    : newvalue (optional)


=cut

sub end{
   my $self = shift;
   if( @_ ) {
      my $value = shift;
      $self->{'end'} = $value;
    }
    return $self->{'end'};

}

=head2 strand

 Title   : strand
 Usage   : $self->strand($newval)
 Function: returns 1. Double strand similarity is not stranded
           will return 0 soon. 
 Returns : value of strand
 Args    : newvalue (optional)


=cut

sub strand{
   my $self = shift;
   return 1;
}

=head2 seqname

 Title   : seqname
 Usage   : $self->seqname($newval)
 Function: 
 Returns : value of seqname
 Args    : newvalue (optional)


=cut

sub seqname{
   my $self = shift;
   if( @_ ) {
      my $value = shift;
      $self->{'seqname'} = $value;
    }
    return $self->{'seqname'};

}



=head2 contig_id

 Title   : contig_id
 Usage   : $self->contig_id($newval)
 Function: 
 Returns : value of contig_id
 Args    : newvalue (optional)


=cut

sub contig_id{
   my ($self,$value)=@_;
   if( $value ) {
      $self->{'contig_id'} = $value;
    }
    return $self->{'contig_id'};

}




=head2 primary_tag

 Title   : primary_tag
 Usage   :
 Function:
 Example :
 Returns : 
 Args    :


=cut

sub primary_tag{
   my ($self,@args) = @_;

   return 'tcore';
}


=head2 source_tag

 Title   : source_tag
 Usage   :
 Function:
 Example :
 Returns : 
 Args    :


=cut

sub source_tag{
   my ($self,@args) = @_;

   return 'ensembl';
}






=head2 source

 Title   : source
 Usage   : $obj->source;
 Function:
           Returns the source organism for the trace match

 Returns : A string
 Args    : A string (optional, for setting)

=cut

sub source {
   my ($obj,$value) = @_;
   if( defined $value) {
      $obj->{'source'} = $value;
    }
   if( ! exists $obj->{'source'} ) {
       return undef;
   }
   return $obj->{'source'};

}





=head2 add_DBLink

 Title   : add_DBLink
 Usage   : $self->add_DBLink($ref)
 Function: adds a link object
 Example :
 Returns :
 Args    :


=cut

sub add_DBLink{
   my ($self,$com) = @_;
   if( ! $com->isa('Bio::Annotation::DBLink') ) {
       $self->throw("Is not a link object but a  [$com]");
   }
   push(@{$self->{'link'}},$com);
}

=head2 each_DBLink

 Title   : each_DBLink
 Usage   : foreach $ref ( $self->each_DBlink() )
 Function: gets an array of DBlink of objects
 Example :
 Returns :
 Args    :


=cut

sub each_DBLink{
    my ($self) = @_;
    if (defined $self->{'link'}){
	return @{$self->{'link'}};
    } else{return;}   
}



=head2 adaptor

 Title   : adaptor
 Usage   : $obj->adaptor($newval)
 Function:
 Example :
 Returns : The db adaptor
 Args    : newvalue (optional)


=cut

sub adaptor {
   my ($self,$value) = @_;
   if( defined $value) {
      $self->{'_adaptor'} = $value;
    }
    return $self->{'_adaptor'};

}

