# BioPerl module for Bio::EnsEMBL::Map::DBSQL::RHMap
#
# Creator: Arne Stabenau <stabenau@ebi.ac.uk>
# Date of creation: 21.01.2000
# Last modified : 27.04.2000 by Arne Stabenau
#
# Copyright EMBL-EBI 2000
#
# You may distribute this module under the same terms as perl itself

# POD documentation - main docs before the code

=head1 NAME

Bio::EnsEMBL::Map::DBSQL::RHMap 

    subclass of Map. When using EnsEMBL::Map
    you should never be too much aware of what Map you use.

=head1 SYNOPSIS

$map = Bio::EnsEMBL::Map::DBSQL::Obj->get_Map( $id );
@maps = Bio::EnsEMBL::Map::DBSQL::Obj->get_Maps();

=head1 DESCRIPTION

    Design detail, look in Map

=head1 CONTACT

Contact Arne Stabenau on implemetation/design detail: stabenau@ebi.ac.uk
  Contact Ewan Birney on EnsEMBL in general: birney@sanger.ac.uk


=head1 APPENDIX

The rest of the documentation details each of the object methods. Internal methods are usually preceded with a _

=cut


# Let the code begin...


package Bio::EnsEMBL::Map::DBSQL::RHMap;
use vars qw(@ISA);
use strict;

require Bio::EnsEMBL::Map::DBSQL::Obj;
use Bio::EnsEMBL::Map::DBSQL::Map;
use Bio::EnsEMBL::Map::DBSQL::MapMarker;
use Bio::EnsEMBL::Map::DBSQL::Marker;
use Bio::EnsEMBL::Map::PositionI;
use Bio::EnsEMBL::Map::DBSQL::LinearPosition;

@ISA = qw( Bio::EnsEMBL::Map::DBSQL::Map );


# Object preamble - inheriets from Bio::Root::Object

# new() is inherited from Bio::EnsEMBL::Map::DBSQL::Map





=head2 get_MapMarkers

    Title   : get_MapMarkers
    Usage   : ...::Map->get_MapMarkers
    Function: Returns all MapMarker objects accociated with this Map. These
              are quite a few, better fetch only needed ones! 
    Example : -
    Returns : Reference to an array
    Args    : - 


=cut

sub get_MapMarkers {
    my ( $self ) = @_;
    my $dbobj = $self->{'_dbsqlobj'};
    my $result = [];
    my ($markerId, $mapMarker,$dbRowHash );

    my $command = "select marker, map_position,chromosome from ".
      $self->tablename." where map=\"".$self->mapname."\"";

    my $sth = $dbobj->prepare( $command );
    $sth->execute;
    while( $dbRowHash = $sth->fetchrow_hashref ) {
	$markerId = $dbRowHash->{'marker'};
	$mapMarker = $self->_create_MapMarker( $markerId, $dbRowHash );
	push( @$result, $mapMarker );
    }

    $result;
}

=head2 get_MarkerIds

    Title   : get_MarkerIds
    Usage   : ...::Map->get_MarkerIds
    Function: Returns all Markernames in this Map.
    Example : -
    Returns : Reference to an array
    Args    : - 


=cut

sub get_MarkerIds {
  my ( $self ) = @_;
  my $dbobj = $self->{'_dbsqlobj'};
  my $result = [];
  my ($markerId, $mapMarker,$dbRowHash );

  my $command = "select marker from ".
    $self->tablename." where map=\"".$self->mapname."\"";

  my $sth = $dbobj->prepare( $command );
  $sth->execute;
  while( $dbRowHash = $sth->fetchrow_hashref ) {
    $markerId = $dbRowHash->{'marker'};
    push( @$result, $markerId );
  }

  $result;
}

=head2 get_MapMarker

    Title   : get_MapMarker
    Usage   : ...::Map->get_MapMarker( $id )
    Function: Returns MapMarker object with  given id or undef if  there isnt
    Example : -
    Returns : -
    Args    : - 


=cut

sub get_MapMarker {
  my $self  =  shift;
  my $id = shift;
  my $dbobj = $self->{'_dbsqlobj'};
  return $self->_create_MapMarker( $id );
}

=head2 _get_MapMarkers_Chromosome

    Title   : _get_MapMarkers_Chromosome
    Usage   : ...::ChromosomMap->get_MapMarkers
    ( [ $startMapMarker, $endMapMarker ] );
    Function: Returns MapMarker objects specific to a chromosomeMap. It is
              assumed that the function is called from a ChromosomeMap object!
    Example : -
    Returns : Reference to an array
    Args    : - 


=cut

sub _get_MapMarkers_Chromosome {
  my $self = shift;
  my ( $chrName, $mapMarkerStart, $mapMarkerEnd ) = @_;
  my $dbobj = $self->{'_dbsqlobj'};
  my $result = [];
  my ($markerId, $mapMarker, $dbRowHash );

  my $desc = "";

  my $startMarkerSeen;
  if( $mapMarkerStart && $mapMarkerEnd ) {
    $startMarkerSeen = 0;
    if( $mapMarkerStart->position->coord > 
	$mapMarkerEnd->position->coord ) {
      $desc = "desc ";
    }
  } else {
    $startMarkerSeen = 1;
  }

  my $command = "select marker, chromosome, map_position from ".
    $self->tablename." where map=\"".$self->mapname.
      "\" and chromosome=\"$chrName\" order by map_position $desc,marker $desc";

  my $sth = $dbobj->prepare( $command );
  $sth->execute;

  while( $dbRowHash = $sth->fetchrow_hashref ) {
    $markerId = $dbRowHash->{'marker'};
    if( $mapMarkerStart && ( $mapMarkerStart->marker->id eq $markerId )) {
      $startMarkerSeen = 1;
    }
    if( $startMarkerSeen ) {
      $mapMarker = $self->_create_MapMarker( $markerId, $dbRowHash );
      push( @$result, $mapMarker );
      #	    print $mapMarker->marker->id," ", $mapMarker->position->id,"\n";
    }
    if( $mapMarkerEnd && ( $mapMarkerEnd->marker->id eq $markerId )) {
      last;
    }
  }
  $result;
}

=head2 _get_MapMarkers_Chromosome_partly

    Title   : _get_MapMarkers_Chromosome_partly
    Usage   : ...::ChromosomMap->get_MapMarkers_partly
    ( $where_in_%_in_the_chromosome, $how_many );
    Function: Returns MapMarker objects specific to a chromosomeMap. It is
              assumed that the function is called from a ChromosomeMap object!
    Example : -
    Returns : Reference to an array
    Args    : - 


=cut

sub _get_MapMarkers_Chromosome_partly {
  my $self = shift;
  my ( $chrName, $where, $count ) = @_;
  my $dbobj = $self->{'_dbsqlobj'};
  my $result = [];
  my ($markerId, $mapMarker, $dbRowHash );
  my ($markername, $position );
  my %seen;

  if( !$self->{_maxChromHash}) {
    $self->_load_maxpositions;
  }

  my $pos = $self->{_maxChromHash}->{$chrName} * $where;

  my $downlist = "select marker, map_position from ".
    $self->tablename." where map=\"".$self->mapname.
      "\" and chromosome=\"$chrName\" and map_position <= $pos ".
	"order by map_position desc,marker desc".
	" limit $count";
  my $uplist = "select marker, map_position from ".
    $self->tablename." where map=\"".$self->mapname.
      "\" and chromosome=\"$chrName\" and map_position >= $pos ".
	"order by map_position,marker".
	" limit $count";

  # getting markers which could be in the endlist.
  my $sth = $dbobj->prepare( $downlist );
  $sth->execute;
  $sth->bind_columns( \$markername, \$position );
  while( $sth->fetch ) {
    $seen{$markername} = $position;
  }

  $sth = $dbobj->prepare( $uplist );
  $sth->execute;
  $sth->bind_columns( \$markername, \$position );
  while( $sth->fetch ) {
    $seen{$markername} = $position;
  }

  my @markers = keys %seen;
  my @sortMarkers = sort {
    $seen{$a} <=> $seen{$b} ||
      $a cmp $b
    } @markers;

  foreach $markername (@sortMarkers) {
    $dbRowHash->{chromosome} = $chrName;
    $dbRowHash->{map_position} = $seen{$markername};
    $dbRowHash->{marker} = $markername;

    $mapMarker = $self->_create_MapMarker( $markername, $dbRowHash );
    push( @{$result}, $mapMarker ); 
  }

  return $result;
}


sub _next_MapMarker_Chromosome {
  my ( $self, $mapMarker, $chrName ) = @_;

  my $id = $mapMarker->marker->id;
  my $pos = $mapMarker->position->coord - 0.001;
  my $idx = 0;
  my $thisIdx = -1;
  my ( $command, $sth, $rowRef );
  my $dbh = $self->{'_dbsqlobj'};

  while (1) {
    my $command = "select * from ".$self->tablename." where ".
      "map='".$self->mapname."' and chromosome='$chrName' and".
	" map_position >=$pos ".
	  "order by map_position, marker ".
	    "limit $idx,".($idx+5);
    $sth = $dbh->prepare( $command );
    $sth->execute;
    if( $sth->rows == 0 ) {
      die( "No next Marker" );
    }
    while( $rowRef = $sth->fetchrow_hashref ) {
      #	    print( "Got row: ",%$rowRef,"\n" );
      if( $thisIdx != -1 ) {
	return $self->_create_MapMarker( $rowRef->{'marker'}, $rowRef );
      }
      if( $rowRef->{'marker'} eq $id ) {
	$thisIdx = $idx;
      }
      if( $rowRef->{'map_position'} > $pos+0.002 ) {
	die( "No next Marker" );
      }
      $idx++;
    }
  }
}

sub _prev_MapMarker_Chromosome {
    my ( $self, $mapMarker, $chrName ) = @_;
    my $id = $mapMarker->marker->id;
    my $pos = $mapMarker->position->coord + 0.001;
    my $idx = 0;
    my $thisIdx = -1;
    my ( $command, $sth, $rowRef );
    my $dbh = $self->{'_dbsqlobj'};

    while (1) {
      my $command = "select * from ".$self->tablename.
        " where map='".$self->mapname."' and chromosome='$chrName'".
        " and map_position <=$pos ".
        "order by map_position DESC, marker DESC limit $idx,".($idx+5);
	$sth = $dbh->prepare( $command );
	$sth->execute;
	if( $sth->rows == 0 ) {
	    die( "No previous Marker" );
	}
	while( $rowRef = $sth->fetchrow_hashref ) {
#	    print( "Got row: ",%$rowRef,"\n" );
	    if( $thisIdx != -1 ) {
		return $self->_create_MapMarker( $rowRef->{'marker'}, $rowRef );
	    }
	    if( $rowRef->{'marker'} eq $id ) {
		$thisIdx = $idx;
	    }
	    if( $rowRef->{'map_position'} < $pos-0.002 ) {
		die( "No previous Marker" );
	    }
	    $idx++;
	}
    }
}




=head2 _create_MapMarker 

    Title : _create_MapMarker 
    Usage : ...::Map->_create_MapMarker 
    Function: MapMarker objects are kept unique in the application. Every
    reference to them is generated by this function and it caches already
    generated MapMarkers. 
    Example : - 
    Returns : undef if given markerId not in this map.
    Args : 1: $markerId, 2: (optional)
    a hash reference with named attributes of the MapMarker.

=cut


sub _create_MapMarker {
    my $self = shift;
    my ( $markerId, $dbRowHash ) = @_;
    my ( $sth, $chrName );

    if( $self->{'_MapMarker_cache'}->{$markerId} ) {
	return $self->{'_MapMarker_cache'}->{$markerId};
    }

    my ( $dbobj, $marker, $chrMap );
    my ( $mapMarker, $position, $command );
    $dbobj = $self->{'_dbsqlobj'};

    if( !$dbRowHash ) {

	# Genetic and radiationhybrids are in the same table at the moment
	# and have the same data...
	$command = "select map_position,chromosome from ".$self->tablename.
	    " where marker=\"$markerId\" and map=\"".$self->mapname.
		"\"";

	$sth = $dbobj->prepare( $command );
	$sth->execute;
	$dbRowHash = $sth->fetchrow_hashref;
	if( ! $dbRowHash ) { return undef; }
    }

    $chrName = $dbRowHash->{'chromosome'};

    $marker = Bio::EnsEMBL::Map::DBSQL::Marker->new( $dbobj, $markerId );
    $chrMap = $self->get_ChromosomeMap( $chrName );

    # change on final implementation
    $position = Bio::EnsEMBL::Map::DBSQL::LinearPosition->new
	( $dbRowHash->{'map_position'} );
    #print STDERR "Got $marker, $chrMap, $position \n";

    $mapMarker = Bio::EnsEMBL::Map::DBSQL::MapMarker->new
	( $dbobj, $marker, $chrMap, $position );
    $self->{'_MapMarker_cache'}->{$markerId} = $mapMarker;
    $mapMarker;
}

=head2 _positionGuess 

    Title : _positionGuess
    Usage : dont, is used by MapMarker.
    Function: returns ($position, $length) in basepairs
    Example : - 
    Returns : 
    Args : 1: $position object, 2: $chromosomeMap

=cut

sub _positionGuess {
    my ($self, $position, $chromosomeMap ) = @_;
    my ( $pos, $length );

    # difficult, what is the max pos?

    if( !$self->{_maxChromHash}) {
      $self->_load_maxpositions;
    }
    my $max = $self->{_maxChromHash}->{$chromosomeMap->name};
    if( ! defined( $max)) {
	return( undef, undef );
    }

    my $fract = $position->coord / $max;
    $pos = int( $fract * $chromosomeMap->chromosome->length * 1000 );
    # arbitrary length just to indicate its not exact
    $length = 100000;

    return ( $pos, $length );
}

sub _load_maxpositions {
  my $self = shift;
  my $dbobj = $self->{'_dbsqlobj'};
  my $arrRef;
  my $sth = $dbobj->prepare( "select chromosome, max(map_position) from ".
			     $self->tablename." where map=\"".
			     $self->mapname."\" group by chromosome" );
  $sth->execute;
  while( $arrRef = $sth->fetchrow_arrayref ) {
    $self->{_maxChromHash}->{$arrRef->[0]} = $arrRef->[1];
  }
  $sth->finish;
}


# compiled succesfully
1;

