# Perl module for Bio::EnsEMBL::Map::DBSQL::Obj
#
# Creator: Arne Stabenau <stabenau@ebi.ac.uk>
# Date of creation: 18.01.2000
# Last modified : 26.05.2000 by Arne Stabenau
#
# Copyright EMBL-EBI 2000
#
# You may distribute this module under the same terms as perl itself

# POD documentation - main docs before the code

=head1 NAME

   Bio::EnsEMBL::Map::DBSQL::Obj - Object representing an instance of an
	EnsEMBL-Map DB.

=head1 SYNOPSIS

    $db = new Bio::EnsEMBL::Map::DBSQL::Obj(
        -user => 'root',
        -dbname => 'Map',
        -host => 'caldy',
        -driver => 'mysql' );

=head1 DESCRIPTION

	Most of the basic DB-functionality of the objects in Bio::EnsEMBL::Map will
	be implemented in this object. It opens and closes the connection and
	gets most of the collections of data out, whereas the single objects
	representatives will just fetch the data belonging to them.

=head1 CONTACT

	Contact Arne Stabenau on implemetation/design detail: stabenau@ebi.ac.uk
	Contact Ewan Birney on EnsEMBL in general: birney@sanger.ac.uk

=head1 APPENDIX

    The rest of the documentation details each of the object methods. 
    Internal methods are usually preceded with a _.

=cut


# Let the code begin... 

package Bio::EnsEMBL::Map::DBSQL::Obj;
use vars qw(@ISA);
use strict;

# we only allow one mapdb object in one interpreter.
# my $singleton=undef;

# Object preamble - inheriets from Bio::Root::Object

use Bio::Root::Object;
use Bio::EnsEMBL::Map::DBSQL::Marker;
use Bio::EnsEMBL::Map::DBSQL::RHMap;
use Bio::EnsEMBL::Map::DBSQL::GeneticMap;
use Bio::EnsEMBL::Map::DBSQL::CGMap;
use Bio::EnsEMBL::Map::DBSQL::Chromosome;
use Bio::EnsEMBL::Map::DBSQL::ChromosomeBand;
use Bio::EnsEMBL::Map::DBSQL::FPC_Map;
use DBI;

use Carp('cluck');

@ISA = qw( Bio::Root::RootI );
# new() is inherited from Bio::Root::Object

# _initialize is where the heavy stuff will happen when new is called


sub new {
  my($class,@args) = @_;
  
#  if( defined( $singleton )) {
#    return $singleton;
#  }
  
  my $self = bless {}, $class;

#  print "Got",join(',',@args),"\n";
  my ($db,$host,$port,$driver,$user,$password,$debug,$ensdb) = 
      $self->_rearrange([qw(DBNAME
			    HOST
			    PORT
			    DRIVER
			    USER
			    PASS
			    DEBUG
			    ENSDB
			    )],@args);

  $db || $self->throw("Database object must have a database name");
#  $user || $self->throw("Database object must have a user");

  $driver || ( $driver = 'mysql' );
  $host   || ( $host   = 'localhost' );
  # $db     || ( $db     = 'maps' );
  $ensdb  || ( $ensdb  = 'ensembl' );

  my $dsn;
  if($driver =~ /Oracle/i ) {
      $dsn="DBI:$driver:$db"; 
  } else {
      $port  || ($port = 3306);

      $dsn = "DBI:$driver:database=$db;host=$host;port=$port";
  }
  $user   || ( $user   = 'nobody' );
  $self->ensdbname( $ensdb );
  if( $debug && $debug > 10 ) {
      $self->_db_handle("dummy dbh handle in debug mode $debug");
  } else {
      #my $dbh = DBI->connect("$dsn","$user",$password, {RaiseError => 1});
      my $dbh = Bio::EnsEMBL::DBSQL::SQL->new("$dsn","$user",$password);
      		#NB this has RaiseError => 0

      $dbh || cluck "connect($dsn,$user,$password) failed";

      $dbh || $self->throw("Could not connect to database $db user $user using [$dsn] as a locator");
      $self->_db_handle($dbh);

        {
            my($sth)=$dbh->prepare("select table_name from user_tables");
            if($sth) {
                $sth->execute();
                print STDERR "tables:\n";
                my $table;
                print STDERR  "$table\n" while( ($table)=$sth->fetchrow_array());
            } else {
                print STDERR "Can't do table query\n";
            }
        }
  }

  print STDERR "Connected to ensembl maps database \"$db\" on server \"$host\" as user \"$user\".\n";

  $self->get_Maps;

#  $singleton = $self;

  return $self; # success - we hope!
}



=head2 get_Marker_Synonym

    Title   : get_Marker_Synonym
    Usage   : $markerArrRef =  ...::DBSQL::Obj->
               get_Marker_Synonym( $synonymName )
    Function: Retrieve all Marker objects for a given synonym, which may be
               ambigious.
    Example : -
    Returns : -
    Args    : -


=cut

sub get_Marker_Synonym {
    my ($self, $synonym) = @_;
    my $result = [];

    my ($sth, $row_ref );

#    print ( $dbobj, $self->id, "\n" );
    my $command = "select distinct marker from MarkerSynonym where name=\"$synonym\"";
#    print( $command,"\n" );
    $sth = $self->prepare( $command );
    $sth->execute;

    while($row_ref = $sth->fetchrow_arrayref) {
#	print $row_ref->[0]."\n";
	my $markerObj = Bio::EnsEMBL::Map::DBSQL::Marker->new
	    ( $self, $row_ref->[0] );
	push( @$result, $markerObj );
    }
    $sth->finish;
    return $result;
}    




=head2 get_Markers

    Title   : get_Markers
    Usage   : $markerArrRef =  ...::DBSQL::Obj->
               get_Markers
    Function: Retrieve all Marker objects in this database, a HUGE list,
    dont do it.

    Example : -
    Returns : -
    Args    : -


=cut

sub get_Markers {
    my ($self) = @_;
    my $result = [];

    my ($sth, $row_ref );
    my $refMarkerIds;
    my @uniqueMarkerIds;

    my ( $maps, $tableName, $mapName );
    my %saw;

    $maps = $self->get_Maps;
    foreach my $map (@$maps) {
	$refMarkerIds = $map->get_MarkerIds;
	foreach my $markerId (@$refMarkerIds) {
	    $saw{$markerId} = 1;
	}
    }

    @uniqueMarkerIds = keys %saw;

    foreach  my $markerId (@uniqueMarkerIds) {
	push( @$result, Bio::EnsEMBL::Map::DBSQL::Marker->new
	     ( $self, $markerId ));
    }
    return $result;
}

=head2 get_Markers_Chromosome

    Title   : get_Markers_Chromosome
    Usage   : $markerArrRef =  ...::DBSQL::Obj->
    get_Markers_Chromosome( $chromosomeName );
    Function: Retrieve all Marker objects for a chromosome.

    Example : -
    Returns : -
    Args    : -


=cut

sub get_Markers_Chromosome {
    my ($self, $chromosomeName ) = @_;
    my $result = [];

    my ($sth, $row_ref );
    my @mapNames = ();
    my @markerIds = ();
    my %saw;
    undef %saw;
    my ($tableName, $mapName );
# searches Meta Table for all valid Map-Tables
# these are searched for the Markers
# 
    $self->throw( "In get_Markers_Chromosome( ) no name as".
		 " first argument." ) unless ( $chromosomeName );

    my $command = "select tablename,name from Map";
    $sth = $self->prepare( $command );
    $sth->execute;
    while( $row_ref = $sth->fetchrow_arrayref) {
	push( @mapNames, $row_ref->[0] );
	push( @mapNames, $row_ref->[1] );
    }
    $sth->finish;

    while ( $#mapNames >= 0 ) {
	$tableName = shift @mapNames;
	$mapName = shift @mapNames;

	if( $mapName ) {
	    $command = "select marker from $tableName ".
		"where map=\"$mapName\" and chromosome=\"$chromosomeName\"";
	} else {
	    $command = "select marker from $tableName".
		" where chromosome=\"$chromosomeName\"";
	}

	$sth = $self->prepare( $command );
	$sth->execute;

	while( $row_ref = $sth->fetchrow_arrayref ) {
	    $saw{$row_ref->[0]} = 1;
	}
	$sth->finish;
    }

    @markerIds = keys %saw;

    foreach my $markerName (@markerIds) {
	my $markerObj = Bio::EnsEMBL::Map::DBSQL::Marker->new
	    ( $self, $markerName );
	unshift( @$result, $markerObj );
    }
    return $result;
}


=head2 get_Maps

    Title   : get_Maps
    Usage   : $mapArrRef =  ...::DBSQL::Obj->get_Maps;
    Function: Retrieve all Map objects in this db. Handcoded in here.
    Example : -
    Returns : -
    Args    : -

=cut

sub get_Maps {
    my ($self) = @_;
    my $result = [];
    my $mapRef;

    if( ! ( $mapRef = $self->{'_Map_RH_int'} )) {
	$mapRef = ( $self->{'_Map_RH_int'} = Bio::EnsEMBL::Map::DBSQL::RHMap->new( $self, 'SG3-GB4' ));
	$mapRef->tablename( "RHMaps" );
	$mapRef->mapname( "RHint" );
    }
    push( @$result, $mapRef );
    if( ! ( $mapRef = $self->{'_Map_RH_GB4'} )) {
	$mapRef = ( $self->{'_Map_RH_GB4'} = Bio::EnsEMBL::Map::DBSQL::RHMap->new( $self, 'GB4' ));
	$mapRef->tablename( "RHMaps" );
	$mapRef->mapname( "GB4" );
    }
    push( @$result, $mapRef );
    if( ! ( $mapRef = $self->{'_Map_RH_SG3'} )) {
	$mapRef = ( $self->{'_Map_RH_SG3'} = Bio::EnsEMBL::Map::DBSQL::RHMap->new( $self, 'SG3' ));
	$mapRef->tablename( "RHMaps" );
	$mapRef->mapname( "SG3" );
    }
    push( @$result, $mapRef );

#    if( ! ( $mapRef = $self->{'_Map_Genetic'} )) {
#	$mapRef = ( $self->{'_Map_Genetic'} = Bio::EnsEMBL::Map::DBSQL::GeneticMap->new( $self, 'Genetic' ));
#	$mapRef->tablename( "MapMarker" );
#	$mapRef->mapname( "" );
#    }
#    push( @$result, $mapRef );

    if( ! ( $mapRef = $self->{'_Map_FISH'} )) {
	$mapRef = ( $self->{'_Map_FISH'} = Bio::EnsEMBL::Map::DBSQL::CGMap->new
	    ( $self, 'FISH' ));
	$mapRef->tablename( "CytogeneticMap" );
    }
    push( @$result, $mapRef );

    $result;
}

=head2 get_Map

    Title   : get_Map
    Usage   : $map =  ...::DBSQL::Obj->get_Map( $id );
 Function: Retrieve given Map object from this db. They are hashed on
    construction  
    Example : -
    Returns : A Map object or exception if id string has no map object. 
    Args    : They id string of the object. Currently in the db are:
    ( "RH", "GB4", "SG3", "FISH", "FPC" )

=cut

sub get_Map {
    my $self = shift;
    my $id = shift;
    my $mapRef;

    if(( $id cmp 'SG3-GB4' ) == 0 ) {
	$mapRef = $self->{'_Map_RH_int'};
    } elsif (( $id cmp 'GB4' ) == 0 ) {
	$mapRef = $self->{'_Map_RH_GB4'};
    } elsif (( $id cmp 'SG3' ) == 0 ) {
	$mapRef = $self->{'_Map_RH_SG3'};
    } elsif (( $id cmp 'FISH' ) == 0 ) {
	$mapRef = $self->{'_Map_FISH'};
    } elsif (( $id cmp 'FPC' ) == 0 ) {
	$mapRef = Bio::EnsEMBL::Map::DBSQL::FPC_Map->new( $self );
	$mapRef->cloneTable( 'Fpc_Clone' );
	$mapRef->contigTable( 'Fpc_Contig' );
    } else {
	$self->throw( "Map id $id unknown in Obj::get_Map call." );
    }
    $mapRef;
}


=head2 get_Chromosome 

    Title   : get_Chromosome
    Usage   : $chromosome = $obj->get_Chromosome( $name );
 Function: Return a chromosome object for the given name or undef if there isnt.
    Example : 
    Returns : 
    Args    : A name for a chromosome

=cut

sub get_Chromosome {
    my $self = shift;
    my $name = shift;

    $self->throw( "No name given in Obj::get_Chromosome!" ) unless $name;

    my $sth = $self->prepare( "select colour, major_band, minor_band, length from ChromosomeBands where chromosome = \"$name\" order by tag" );
    $sth->execute;
    my ( $chr, $arrayRef, $band, $colour, $length );

    if( $sth->rows == 0 ) {
	return undef;
    } else {
	$chr = Bio::EnsEMBL::Map::DBSQL::Chromosome->new( $name );
    }

    while( $arrayRef = $sth->fetchrow_arrayref ) {
	if( $arrayRef->[2] ) {
	    $band = $arrayRef->[1].".".$arrayRef->[2];
	} else {
	    $band = $arrayRef->[1];
	}

	my $chrBand = Bio::EnsEMBL::Map::DBSQL::ChromosomeBand->new
	    ( $band, $arrayRef->[3], $arrayRef->[0] );
	$chr->add_Band( $chrBand );
    }
    return $chr;
}


# get/set the name of the ensembl database, colocated on the same 
# server as Map database
sub ensdbname {
  my $self = shift;
  my $ensdb = shift;
  $ensdb && 
    ( $self->{_ensdb} = $ensdb );
  $self->{_ensdb};
}



sub prepare{
  my ($self,$string) = @_;
  
  if( ! $string ) {
     $self->throw("Attempting to prepare an empty SQL query!");
   }

#    if( $self->_debug > 10 ) {
#        print STDERR "Prepared statement $string\n";
#        my $st = Bio::EnsEMBL::DBSQL::DummyStatement->new();
#        $st->_fileh(\*STDERR);
#        $st->_statement($string);
#        return $st;
#    }

   if($self->_debug>3) {
   print STDERR "preparing <$string> on ",ref($self->_db_handle),"\n";
   { my $i=0;
     while(my($package,$filename,$line,$subroutine)=caller($i++)) {
         print STDERR " >  $package",'::',"$subroutine $filename line $line\n";
     }
   }
   }

  # should we try to verify the string?
    my( $sth );
    eval {
        $sth = $self->_db_handle->prepare($string);
    };

    $self->throw("Error preparing $string\n$@") if $@;
    return $sth;

###############################################################################
#    if(!$@){
#	return $sth;
#    }
#    else {
#	######################################################################
#	# OK, this is bad, but... we are having problems with the _db_handle
#	# not existing at this point.  Let's try to reconnect, if this is the
#	# case.
#	######################################################################
#	print STDERR "** Lost _db_handle ** \n";
#      	my $dbh = DBI->connect("DBI:mysql:database=maps074;host=localhost","ensro",'', {RaiseError => 1});
#      	$dbh || $self->throw("Could not reconnect to maps database as a locator");
#      	$self->_db_handle($dbh);
#    	eval {
#            $sth = $self->_db_handle->prepare($string);
#    	};
#        $self->throw("Error preparing $string\n$@") if $@;
#	print STDERR "** Regained _db_handle **\n";
#        return $sth;
#    }
###############################################################################
}


sub _db_handle {
  my ($self,$value) = @_;
  if( defined $value) {
    $self->{'_db_handle'} = $value;
  }
  
  return $self->{'_db_handle'};
}


=head2 deleteObj

    Title   : deleteObj
    Usage   : $dbObj->deleteObj
    Function: Call when you are done with this object. Breaks links between
    objects. Necessary to clean up memory.
    Example : -
    Returns : -
    Args    : -


=cut

sub deleteObj {
  my  $self=shift;
  my $dummy;
  $self->DESTROY;
  #$singleton = undef;
  foreach my $name ( keys %{$self} ) {
    eval '$dummy = $self->{$name}; 
          $self->{$name} = undef;
          $dummy->deleteObj;';
  }
}


=head2 DESTROY

 Title   : DESTROY
 Usage   :
 Function:
 Example :
  Returns : 
 Args    :


=cut

  sub DESTROY{
    my ($obj) = @_;
    
    if( $obj->{'_db_handle'} ) {
      $obj->{'_db_handle'}->disconnect;
       $obj->{'_db_handle'} = undef;
    }
  }
  


1;
