# BioPerl module for Bio::EnsEMBL::Map::DBSQL::Marker
#
# Creator: Arne Stabenau <stabenau@ebi.ac.uk>
# Date of creation: 18.01.2000
# Last modified : 23.01.2000 by Arne Stabenau
#
# Copyright EMBL-EBI 2000
#
# You may distribute this module under the same terms as perl itself

# POD documentation - main docs before the code

=head1 NAME

Bio::EnsEMBL::Map::DBSQL::Marker 

=head1 SYNOPSIS
Bio::EnsEMBL::Map::DBSQL::Marker->new( $dbobj, $id );

=head1 DESCRIPTION
    All Marker objects have to be given a reference to a DBSQL::Obj object and
    an id. By that they can fetch themselves all information they contain
    from db. They implement the ...Map::MarkerI object. Markers usually
    appear in different kinds of maps. They have names and synonymous names 
    under which they are known. Some are microsattelite DNA, some STS,
    Genes or whatever. The object describing the location of a Marker in a
    special Map is the MapMarker. 

=head1 CONTACT

    Contact Arne Stabenau on implemetation/design detail: stabenau@ebi.ac.uk
    Contact Ewan Birney on EnsEMBL in general: birney@sanger.ac.uk

=head1 APPENDIX

The rest of the documentation details each of the object methods. Internal methods are usually preceded with a _

=cut


# Let the code begin...


package Bio::EnsEMBL::Map::DBSQL::Marker;
use vars qw(@ISA);
use strict;

use Bio::EnsEMBL::Map::MarkerI;
use Bio::EnsEMBL::SeqFeature;
use Bio::EnsEMBL::FeaturePair;
use Bio::EnsEMBL::Analysis;
use Bio::EnsEMBL::Map::MarkerFeature;

@ISA = qw( Bio::EnsEMBL::Map::MarkerI );

# Object preamble - inheriets from Bio::Root::Object

# new() is inherited from Bio::Root::Object
# _initialize is where the heavy stuff will happen when new is called
=head2 Constructor

    Title   : _initialize
    Usage   : ...::Marker->new( DBSQL::Obj, string id ) ; 
 Function: 
    Example :
    Returns : a Marker object 
    Args    : DBSQL::Obj, string id


=cut


sub new {
    my ($class,@args) = @_;

    my $self = {};
    bless $self,$class;


# check args
    $self->{'_dbsqlobj'} = shift @args;
    $self->throw
	("new Marker(..) called with no DBSQL::Obj as first argument." ) 
	    unless ref($self->{'_dbsqlobj'}) && 
		$self->{'_dbsqlobj'}->isa("Bio::EnsEMBL::Map::DBSQL::Obj");

    $self->{'_id'} = shift @args;
    $self->throw
	("new Marker(..) called with no second argument (id-string)!")
	    unless $self->{'_id'};


    return $self;
}



=head2 FunctionTitle

    Title   : id
    Usage   : ...::Marker->id;
 Function: returns an id string for the Marker. Its the name by which  
    it is most commonly known. It is known on generation of Marker object
    and used to pull more information out of db.
    Example :
    Returns : 
    Args    : -


=cut

sub id {
    my ($self) = @_;
    return $self->{'_id'};
}

=head2 synonyms

    Title   : synonyms
    Usage   : $syn = ...::Marker->synonyms;

    Function: get the reference to an array of string-scalars, containing
              synonyms of this marker, followed by a name for the database,
              where this synonyms are kept.

    Example : $synArrRef = $marker->synonyms;

    Returns : The reference or an exception if something with the db goes
    wrong.
    Args    : -

=cut

sub synonyms {
    my ($self) = @_;
    my $result = [];
    my ($dbobj, $sth, $row_ref );

    $self->{'_synonyms'} && ( return $self->{'_synonyms'} );
#    print "db-query\n";
    $dbobj = $self->{'_dbsqlobj'};
#    print ( $dbobj, $self->id, "\n" );
    my $command = "select name, db from MarkerSynonym where marker=\"".$self->id."\"";
#    print( $command,"\n" );
    $sth = $dbobj->prepare( $command );
    $sth->execute;

    while($row_ref = $sth->fetchrow_arrayref) {
	push( @$result, $row_ref->[0] );
	push( @$result, $row_ref->[1] );
    }
    $sth->finish;
    $self->{'_synonyms'} = $result;
    return $result;
}

=head2 get_MapMarkers

    Title   : get_MapMarkers
    Usage   : $mapMarkerArr = $marker->get_MapMarkers;

    Function: get the reference to an array of string-scalars, containing
    alternative names for this Marker.         

    Example : $synArrRef = $marker->get_MapMarkers;

    Returns : Return all the MapMarker objects that are attached to this Marker.
    Args    : -

=cut

sub get_MapMarkers {
    my ($self) = @_;
    my $result = {};
    my ($dbobj, $maps, $mapMarker);

    $dbobj = $self->{'_dbsqlobj'};

    $maps = $dbobj->get_Maps;
    foreach my $map (@$maps) {
	$mapMarker = $map->get_MapMarker( $self->{'_id'});
	(defined $mapMarker) &&
	    ( $result->{$map->id} = $mapMarker );
    }
    return $result;
}

=head2 maps

    Title   : maps
    Usage   : $mapArr = $marker->maps;

    Function: get the reference to an array of Map objects, where this
    marker is in. 

    Example : 
    Returns : 
    Args    : -

=cut

sub maps {
    my ($self) = @_;
    my $result = [];
    my ($dbobj, $maps, $mapMarker);

    $dbobj = $self->{'_dbsqlobj'};

    $maps = $dbobj->get_Maps;
    foreach my $map (@$maps) {
	$mapMarker = $map->get_MapMarker( $self->{'_id'});
	(defined $mapMarker) &&
	    push( @$result, $map );
    }
    return $result;
}

=head2 chromosomeMaps

    Title   : chromosomeMaps
    Usage   : $mapArr = $marker->chromosomeMaps;

    Function: get the reference to an array of ChromosomeMap objects, where
    this marker is in. 

    Example : 
    Returns : 
    Args    : -

=cut

sub chromosomeMaps {
    my ($self) = @_;
    my $result = [];
    my ($dbobj, $maps, $mapMarker);

    $dbobj = $self->{'_dbsqlobj'};

    $maps = $dbobj->get_Maps;
    foreach my $map (@$maps) {
	$mapMarker = $map->get_MapMarker( $self->{'_id'});
	(defined $mapMarker) &&
	    push( @$result, $mapMarker->chromosomeMap );
    }
    return $result;
}

=head2 get_Marker_SeqFeatures

    Title   : get_Marker_SeqFeatures
    Usage   : $seqFeatures = ...::MarkerI->get_Marker_SeqFeatures;
    Function: Maybe this Marker was detected somewhere in the sequences
    within EnsEMBL. Return SeqFeature where this is true.
    Example : -
    Returns : Reference to an array of FeaturePair objects. Empty array if it has none.
    Args    : -

=cut

sub get_Marker_SeqFeatures {
    my ($self) = @_;
    my $result = [];

    # get embl-id
    # assume db-ensembl to get with the same database handle
    my $dbobj = $self->{'_dbsqlobj'};
    my $ensname = $dbobj->ensdbname;

    my $sth = $dbobj->prepare
      ( "select a.db,a.db_version,a.program,a.program_version,".
	"a.gff_source,a.gff_feature,c.id, e.seq_start, e.seq_end,".
	" e.strand, e.score, e.name, e.hstart, e.hend, e.hid ".
	"from $ensname.feature as e, $ensname.analysis as a, ".
	"$ensname.contig as c ".
	"where e.hid=\"".$self->id."\"" .
	"and a.id = e.analysis and e.contig = c.internal_id" );

    $sth->execute;
    my $hashRef;

    # make FeaturePairs
    if( $sth->rows == 0 ) {
	# print STDERR ("No Marker locations found for ", $self->id,"\n" );
    }

    while( $hashRef = $sth->fetchrow_hashref ) {
	# print STDERR ("Found feature for ", $self->id, "\n" );

 	my $anal = Bio::EnsEMBL::Analysis->new();

	$anal->db($hashRef->{'db'});
	$anal->db_version($hashRef->{'db_version'});

 	$anal->program($hashRef->{'program'});
 	$anal->program_version($hashRef->{'program_version'});
 	$anal->gff_source($hashRef->{'gff_source'});
 	$anal->gff_feature($hashRef->{'gff_feature'});
	my $f1 = new Bio::EnsEMBL::SeqFeature(-seqname => $hashRef->{'id'},
					      -start   => $hashRef->{'seq_start'},
					      -end     => $hashRef->{'seq_end'},
					      -score   => 1,
					      -source_tag  => $hashRef->{'name'},
					      -primary_tag =>'similarity',
					      -strand      => $hashRef->{'strand'},
					      -analysis    => $anal,
					      );

	my $f2 = new Bio::EnsEMBL::SeqFeature(-seqname => $hashRef->{'hid'},
					      -start   => $hashRef->{'hstart'},
					      -end     => $hashRef->{'hend'},
					      -score   => 1,
					      -source_tag  => $hashRef->{'name'},
					      -primary_tag =>'similarity',
					      -strand      => 1,
					      -analysis    => $anal,
					      );

	my $fp = new Bio::EnsEMBL::Map::MarkerFeature(-feature1 => $f1,
						      -feature2 => $f2);
	$fp->mapdb( $dbobj );
	push( @$result, $fp );
    }
    return $result;
}

=head2 get_primer

    Title   : get_primer
    Usage   : @primerPairList = $marker->get_primer;
    Function: If this Marker has primers associated with it, return them in 
              a list. If not, the list is empty.
    Example : -
    Returns : a possible empty list of primer sequences.
    Args    : -

=cut

sub get_primer {

  my $self = shift;
  my @result;
  my ( $seq_left, $seq_right );
  my $dbobj = $self->{'_dbsqlobj'};
  
  my $sth = $dbobj->prepare( "select seq_left, seq_right from Marker where ".
			     "marker = \"".$self->id."\"" );
  $sth->execute;
  ($seq_left,$seq_right)=$sth->fetchrow_array;
  if( $seq_left && $seq_right ) {
    return( $seq_left, $seq_right );
  } else {
    return ();
  }
}

=head2 get_type

    Title   : get_type
    Usage   : $type = $marker->get_type;
    Function: The Marker could be of type msat or cdna. If neither undef is returned.
              More types may be added in the future.
    Example : -
    Returns : "msat", "cdna", undef
    Args    : -

=cut

sub get_type {
  my $self = shift;
  my $dbobj = $self->{'_dbsqlobj'};
  my $type;

  my $sth= $dbobj->prepare( "select marker_type from Marker where marker = \"".
			    $self->id."\"" );
  $sth->execute;
  $sth->bind_columns( \$type );
  if( $sth->fetch ) {
    if( !$type ) {
      return undef;
    } else {
      return $type;
    }
  } else {
    return undef;
  }
}


# compiled succesfully
1;


