# BioPerl module for Bio::EnsEMBL::Map::DBSQL::MapMarker
#
# Creator: Arne Stabenau <stabenau@ebi.ac.uk>
# Date of creation: 22.01.2000
# Last modified : 23.01.2000 by Arne Stabenau
#
# Copyright EMBL-EBI 2000
#
# You may distribute this module under the same terms as perl itself

# POD documentation - main docs before the code

=head1 NAME

Bio::EnsEMBL::Map::DBSQL::MapMarker

=head1 SYNOPSIS

Bio::EnsEMBL::Map::DBSQL::Marker->get_MapMarkers()
Bio::EnsEMBL::Map::DBSQL::ChromosomeMap->get_MapMarkers()
Bio::EnsEMBL::Map::DBSQL::ChromosomeMap->get_MapMarkers
    (MapMarker start, MapMarker end)


Bio::EnsEMBL::Map::DBSQL::MapMarker->new ???

=head1 DESCRIPTION
    MapMarker objects define a position of an interesting Marker object on
    a linear piece of DNA. Depending on the type of DNA representation 
    (say Map) the position information might be different. It could be band 
    information, rads or men??. The MapMarker object is connected to the
    Marker andd the ChromosomeMap where it is in.

=head1 CONTACT

    Contact Arne Stabenau on implemetation/design detail: stabenau@ebi.ac.uk
    Contact Ewan Birney on EnsEMBL in general: birney@sanger.ac.uk

=head1 APPENDIX

The rest of the documentation details each of the object methods. Internal methods are usually preceded with a _

=cut


# Let the code begin...


package Bio::EnsEMBL::Map::DBSQL::MapMarker;
use vars qw(@ISA);
use strict;

use Bio::EnsEMBL::Map::MapMarkerI;
use Bio::Root::RootI;


@ISA = qw( Bio::EnsEMBL::Map::MapMarkerI Bio::Root::RootI );

# Object preamble - inheriets from Bio::Root::Object

# new() is inherited from Bio::Root::Object
# _initialize is where the heavy stuff will happen when new is called

# -- dont know how to initialize the thing
# -- probably dbobj, chromosomeMap, marker, positionObject


# _initialize( this, Obj, Marker, ChromosomeMap, PositionI ) 
sub new {
    my ($class,@args) = @_;

    my $self = {};
    bless $self,$class;

    my ($dbobj, $marker, $chrMap, $posI ) = @args;
    #print STDERR "Marker init: $dbobj, $marker, $chrMap, $posI \n";
    
# check args
    $self->throw
	("new MapMarker(..) called with no DBSQL::Obj as first argument." ) 
	    unless ref($dbobj) && 
		$dbobj->isa("Bio::EnsEMBL::Map::DBSQL::Obj");
    $self->{'_dbsqlobj'} = $dbobj;

    $self->throw
	("new MapMarker(..) called with no second argument DBSQL::Marker!")
	    unless ref($marker) && $marker->isa
		( "Bio::EnsEMBL::Map::DBSQL::Marker" );
    $self->{'_marker'} = $marker;

    $self->throw
	("new MapMarker(..) called with no third".
	 " argument DBSQL::ChromosomeMap!")
	    unless ref($chrMap) && $chrMap->isa
		( "Bio::EnsEMBL::Map::DBSQL::ChromosomeMap" );
    $self->{'_chromosomeMap'} = $chrMap;

    $self->throw
	("new MapMarker(..) called with no fourth".
	 " argument PositionI!")
	    unless ref($posI) && $posI->isa
		( "Bio::EnsEMBL::Map::PositionI" );
    $self->{'_position'} = $posI;

    return $self;
}


=head2 map

    Title   : map
    Usage   : $map = ...::MapMarker->map;

    Function: Give the MapI object this MapMarker belongs
    Example : $map = $mapMarker->map;

    Returns : -
    Args    : -

=cut


sub map {
    my $self = shift;
    $self->{'_chromosomeMap'}->get_Map;
}

=head2 chromosomeMap

    Title   : chromosomeMap
    Usage   : $chrMap = ...::MapMarker->chromosomeMap;

    Function: Give the ChromosomeMapI object this MapMarker belongs to.
    Example : -

    Returns : -
    wrong.
    Args    : -

=cut

sub chromosomeMap {
    my $self = shift;
    $self->{'_chromosomeMap'};
}


=head2 position

    Title   : position
    Usage   : $posI = ...::MapMarker->position;

    Function: Give the PositionI object, describing the position of the MapMarker in the Map.

    Example : -
    Returns : -
    Args    : -

=cut

sub position {
    my $self = shift;
    $self->{'_position'};
}


=head2 marker

    Title   : marker
    Usage   : $marker = ...::MapMarker->marker;

    Function: Get the Marker object for this MapMarker. It contains the
    name and other nice functions for Markers.

    Example : -
    Returns : -
    Args    : -

=cut

sub marker {
    my $self = shift;
    $self->{'_marker'}
}

=head2 position_guess

    Title   : position_guess
    Usage   : ($start,$lenght) = $mapMarker->position_guess;

    Function: Gives an estimation where the marker starts and ends in
              basepairs on the DNA. Could be quite wrong. Understand it as
              the common coordinate system, where we dont really know yet,
              how to transform the coords we have into. 
    Example : -

    Returns : a 2 element list with huge integer numbers.
    Args    : -
    NB      : As this is type dependent, transfered to Map ...
=cut

sub position_guess {
    my $self = shift;
    $self->map->_positionGuess( $self->position, $self->chromosomeMap );
}



# compiled succesfully
1;

