# BioPerl module for Bio::EnsEMBL::Map::DBSQL::Map
#
# Creator: Arne Stabenau <stabenau@ebi.ac.uk>
# Date of creation: 21.01.2000
# Last modified : 18.02.2000 by Arne Stabenau
#
# Copyright EMBL-EBI 2000
#
# You may distribute this module under the same terms as perl itself

# POD documentation - main docs before the code

=head1 NAME

Bio::EnsEMBL::Map::DBSQL::Map 

=head1 SYNOPSIS
    $map = Bio::EnsEMBL::Map::DBSQL::Obj->get_Map( $id );
    @maps = Bio::EnsEMBL::Map::DBSQL::Obj->get_Maps();

=head1 DESCRIPTION
    A Map object contains all Markers on a certain type and provider of a
    map. The recommended  way to get a map object is either to know its
    identifier and fetch it from the Obj in the database or ask the
    database Obj for a list of map objects. There should be few.

    Map objects in turn contain a serious of linear DNA representing
    submaps which are seperate objects (ChromosomeMap).

=head1 CONTACT

    Contact Arne Stabenau on implemetation/design detail: stabenau@ebi.ac.uk
    Contact Ewan Birney on EnsEMBL in general: birney@sanger.ac.uk


=head1 APPENDIX

The rest of the documentation details each of the object methods. Internal methods are usually preceded with a _

=cut


# Let the code begin...


package Bio::EnsEMBL::Map::DBSQL::Map;
use vars qw(@ISA);
use strict;

use Bio::EnsEMBL::Map::MapI;
use Bio::EnsEMBL::Map::DBSQL::ChromosomeMap;
require Bio::EnsEMBL::Map::DBSQL::Obj;

@ISA = qw( Bio::EnsEMBL::Map::MapI );

# Object preamble - inheriets from Bio::Root::Object

# new() is inherited from Bio::Root::Object
# _initialize is where the heavy stuff will happen when new is called
=head2 Constructor

    Title   : _initialize
    Usage   : ...::Map->new( $dbsqlobj, $mapID );
 Function: Map is only created from Obj. 
    Example : -
    Returns : -
    Args    : First is the DBSQL::Obj, second an id string for the map. Its 
    used to find more things on the map in the db.

=cut


sub new {
    my ($class,@args) = @_;

    my $self = {};
    bless $self,$class;

    my ( $dbobj, $idString ) = @args;

# check args
    $self->throw
	("new Map(..) called with no DBSQL::Obj as first argument." ) 
	    unless ref($dbobj) && 
		( $dbobj->isa("Bio::EnsEMBL::Map::DBSQL::Obj"));

    $self->{'_dbsqlobj'} = $dbobj;

    $self->throw
	("new Map(..) called with no second argument (id-string)!")
	    unless $idString;
    $self->{'_id'} = $idString;


    #print STDERR "Oh yes... we are making a real map object $self with $dbobj";
    return $self;
}

=head2 id

    Title   : id
    Usage   : ...::Map->id
  Function: Return the id with which this Map was constructed.
    Example : -
    Returns : -
    Args    : - 


=cut

sub id {
    my $self = shift;
    $self->{'_id'};
}

=head2 origin

    Title   : origin
    Usage   : ...::Map->origin( [ $setValue ] )
  Function: Get/Set value for origin
    Example : -
    Returns : -
    Args    : - 


=cut

sub origin {
    my ( $self, $origin ) = @_;

    $origin &&( $self->{'_origin'} = $origin );
    $self->{'_origin'};
}

=head2 last_updated

    Title   : last_updated
    Usage   : ...::Map->last_updated( [ $setValue ] )
  Function: Get/Set value for last_updated. Its the time when you updated last the data in the underlying database.
    Example : -
    Returns : -
    Args    : - 


=cut

sub last_updated {
    my ( $self, $last_updated ) = @_;

    $last_updated && ( $self->{'_last_updated'} = $last_updated );
    $self->{'_last_updated'};
}

=head2 version

    Title   : version
    Usage   : ...::Map->version( [ $setValue ] )
  Function: Get/Set value for version. Its the time when you updated last the data in the underlying database.
    Example : -
    Returns : -
    Args    : - 


=cut

sub version {
    my ( $self, $version ) = @_;

    $version && ( $self->{'_version'} = $version );
    $self->{'_version'};
}

=head2 type

    Title   : type
    Usage   : ...::Map->type( [ $setValue ] )
  Function: Get/Set value for type. Describes the type of the Map, like
    radiohybrid,cytogenetic, physical .... Should be suitable to decide
    about the type of Positions the MapMarkers are on.
    Example : -
    Returns : -
    Args    : - 


=cut

sub type {
    my ( $self, $type ) = @_;

    $type && ( $self->{'_type'} = $type );
    $self->{'_type'};
}


=head2 tablename

    Title   : tabelname
    Usage   : ...::Map->tablename( [give tablename] )
  Function: Sets the tablename where Map gets its information. Is
    neccessary for operation, but not for generation of the object. 
    Example : -
    Returns : -
    Args    : - 


=cut

sub tablename {
    my ( $self, $tablename ) = @_;
    $tablename && ( $self->{'tablename'} = $tablename );
    $self->{'tablename'};
}

=head2 mapname

    Title   : mapname
    Usage   : ...::Map->mapname( [give mapname] )
  Function: Sets the mapname where Map gets its information. Some maps may
    share the same table, but not the same name. There is expected to be a
    map column in the table.
    Example : -
    Returns : -
    Args    : - 


=cut

sub mapname {
    my ( $self, $mapname ) = @_;
    $mapname && ( $self->{'mapname'} = $mapname );
    $self->{'mapname'};
}

=head2 get_ChromosomeMap

    Title   : get_ChromosomeMap
    Usage   : ...::Map->get_ChromosomeMap( $name );
  Function: Returns a ChromosomMap object. Its cached here, so the next
    request for this ChromosomMap gets the same.
    Example : -
    Returns : A ChromosomeMap object.
    Args    : 1: the name of the Chromosome


=cut

sub get_ChromosomeMap {
    my $self = shift;
    my $name = shift;

    $self->throw
	("get_ChromosomeMap(..) called with no first argument name-string!")
	    unless $name;
    if( $self->{'_ChromosomeMap_cache'}->{$name} ) {
	return $self->{'_ChromosomeMap_cache'}->{$name};
    }

    my $dbobj = $self->{'_dbsqlobj'};
    if( !defined $dbobj) {
       $self->throw("No idea how we did this ... no dbobj $self");
    }


    # check if this chromosome exists ...
    my $command = "select chromosome from ".$self->tablename.
	" where chromosome=\"$name\" limit 1";
    my $sth = $dbobj->prepare( $command );
    $sth->execute;
    if( !$sth->fetchrow_hashref ) {
	$self->throw( "Unknown Chromosome name in get_ChromosomeMap." );
    }

    my $chrMapObj = Bio::EnsEMBL::Map::DBSQL::ChromosomeMap->new
	( $dbobj, $self, $name );
    my $chromosome = $dbobj->get_Chromosome( $name );
    $chrMapObj->chromosome( $chromosome );
    $self->{'_ChromosomeMap_cache'}->{$name} = $chrMapObj; 
    $chrMapObj;
  }

=head2 get_ChromosomeNames

    Title   : get_ChromosomeNames
    Usage   : ...::Map->get_ChromosomeNames;
  Function: returns a list of chromosome names known in this Map.
    Example : -
    Returns : Reference to array of string-scalars
    Args    : -


=cut

  sub get_ChromosomeNames {
    my $self = shift;
    my $result = [];
    my $dbRowArray;

    my $dbobj = $self->{'_dbsqlobj'};
    # check if this chromosome exists ...
    my $command = "select chromosome from ".$self->tablename.
	" group by chromosome";
    my $sth = $dbobj->prepare( $command );
    $sth->execute;
    while( $dbRowArray = $sth->fetchrow_arrayref ) {
      push( @$result, $dbRowArray->[0] );
    }
    $result;
  }

=head2 deleteObj

    Title   : deleteObj
    Usage   : $dbObj->deleteObj
    Function: Call when you are done with this object. Breaks links between
    objects. Necessary to clean up memory.
    Example : -
    Returns : -
    Args    : -


=cut

sub deleteObj {
  my  $self=shift;
  my $dummy;
  foreach my $name ( keys %{$self} ) {
    eval '$dummy = $self->{$name}; 
          $self->{$name} = undef;
          $dummy->deleteObj;';
  }
}


# compiled succesfully
1;

