# BioPerl module for Bio::EnsEMBL::Map::DBSQL::GeneticMap
#
# Creator: Arne Stabenau <stabenau@ebi.ac.uk>
# Date of creation: 31.01.2000
# Last modified : 05.02.2000 by Arne Stabenau
#
# Copyright EMBL-EBI 2000
#
# You may distribute this module under the same terms as perl itself

# POD documentation - main docs before the code

=head1 NAME

Bio::EnsEMBL::Map::DBSQL::GeneticMap 

=head1 SYNOPSIS
    $map = Bio::EnsEMBL::Map::DBSQL::Obj->get_Map( $id );
    @maps = Bio::EnsEMBL::Map::DBSQL::Obj->get_Maps();

=head1 DESCRIPTION

    GeneticMap is the subtype of Map. Specially for generating MapMarker
    objects, which have different representations in each map. 

=head1 CONTACT

    Contact Arne Stabenau on implemetation/design detail: stabenau@ebi.ac.uk
    Contact Ewan Birney on EnsEMBL in general: birney@sanger.ac.uk


=head1 APPENDIX

The rest of the documentation details each of the object methods. Internal methods are usually preceded with a _

=cut


# Let the code begin...


package Bio::EnsEMBL::Map::DBSQL::GeneticMap;
use vars qw(@ISA);
use strict;

use Bio::EnsEMBL::Map::DBSQL::Obj;
use Bio::EnsEMBL::Map::DBSQL::Map;

@ISA = qw( Bio::EnsEMBL::Map::DBSQL::Map );

# Object preamble - inheriets from Bio::Root::Object

# new() is inherited from Bio::Root::Object
# _initialize is where the heavy stuff will happen when new is called
=head2 Constructor

    Title   : _initialize
    Usage   : ...::GeneticMap->new( $dbsqlobj, $mapID );
 Function: Map is only created from Obj. 
    Example : -
    Returns : -
    Args    : First is the DBSQL::Obj, second an id string for the map. Its 
    used to find more things on the map in the db.

=cut

sub _initialize {
    my $self = shift;
    $self->SUPER::_initialize( @_ );

    $self;
}


=head2 get_MapMarker

    Title   : get_MapMarker
    Usage   : ...::Map->get_MapMarker( $id )
  Function: Returns MapMarker object with  given id or undef if  there isnt
    Example : -
    Returns : -
    Args    : - 


=cut

sub get_MapMarker {
    my $self  =  shift;
    my $id = shift;
    return $self->_create_MapMarker( $id );
}

=head2 get_MarkerIds

    Title   : get_MarkerIds
    Usage   : ...::Map->get_MarkerIds
  Function: Returns all Markernames in this Map.
    Example : -
    Returns : Reference to an array
    Args    : - 


=cut

sub get_MarkerIds {
    my ( $self ) = @_;
    my $dbobj = $self->{'_dbsqlobj'};
    my $result = [];
    my ($markerId, $mapMarker,$dbRowHash );

    my $command = "select marker from ".
	$self->tablename." where map=\"".$self->mapname."\"";

    my $sth = $dbobj->prepare( $command );
    $sth->execute;
    while( $dbRowHash = $sth->fetchrow_hashref ) {
	$markerId = $dbRowHash->{'marker'};
	push( @$result, $markerId );
    }

    $result;
}

=head2 get_MapMarkers

    Title   : get_MapMarkers
    Usage   : ...::Map->get_MapMarkers
  Function: Returns all MapMarker objects accociated with this Map. These
    are quite a few, better fetch only needed ones! 
    Example : -
    Returns : -
    Args    : - 


=cut

sub get_MapMarkers {
    my ( $self ) = @_;
    my $dbobj = $self->{'_dbsqlobj'};
    my $result = [];
    my ($markerId, $mapMarker, $dbRowHash );

    my $command = "select marker, map_position,chromosome from ".
	$self->tablename." where map=\"".$self->mapname."\"";

    my $sth = $dbobj->prepare( $command );
    $sth->execute;
    while( $dbRowHash = $sth->fetchrow_hashref ) {
	$markerId = $dbRowHash->{'marker'};
	$mapMarker = $self->_create_MapMarker( $markerId, $dbRowHash );
	push( @$result, $mapMarker );
    }

    $result;
}


=head2 _get_MapMarkers_Chromosome

    Title   : _get_MapMarkers_Chromosome
    Usage   : ...::ChromosomMap->get_MapMarkers
    ( [ $startMapMarker, $endMapMarker ] );
 Function: Returns MapMarker objects specific to a chromosomeMap. It is
    assuemd that the function is called from a ChromosomeMap object!
    Example : -
    Returns : Reference to an array
    Args    : - 


=cut

sub _get_MapMarkers_Chromosome {
    my $self = shift;
    my ( $chrName, $mapMarkerStart, $mapMarkerEnd ) = @_;
    my $dbobj = $self->{'_dbsqlobj'};
    my $result = [];
    my ($markerId, $mapMarker, $dbRowHash );

    my $command = "select marker, chromosome, map_position from ".
	$self->tablename." where map=\"".$self->mapname.
	    "\" and chromosome=\"$chrName\" order by map_position";

    my $sth = $dbobj->prepare( $command );
    $sth->execute;
    while( $dbRowHash = $sth->fetchrow_hashref ) {
	$markerId = $dbRowHash->{'marker'};
	$mapMarker = $self->_create_MapMarker( $markerId, $dbRowHash );
	push( @$result, $mapMarker );
    }

    $result;
}

=head2 _create_MapMarker 
    Title : _create_MapMarker 
    Usage : ...::Map->_create_MapMarker 
    Function: MapMarker objects are kept unique in the application. Every
    reference to them is generated by this function and it caches already
    generated MapMarkers. 
    Example : - 
    Returns : - 
    Args : 1: $markerId, 2: (optional)
    a hash reference with named attributes of the MapMarker.


=cut


sub _create_MapMarker {
    my $self = shift;
    my ( $markerId, $dbRowHash ) = @_;
    my ( $sth, $chrName );

    if( $self->{'_MapMarker_cache'}->{$markerId} ) {
	return $self->{'_MapMarker_cache'}->{$markerId};
    }

    my ( $dbobj, $marker, $chrMap );
    my ( $mapMarker, $position, $command );
    $dbobj = $self->{'_dbsqlobj'};

    if( !$dbRowHash ) {

	# Genetic and radiationhybrids are in the same table at the moment
	# and have the same data...
	$command = "select map_position,chromosome from ".$self->tablename.
	    " where marker=\"$markerId\" and map=\"".$self->mapname.
		"\"";

	$sth = $dbobj->prepare( $command );
	$sth->execute;
	$dbRowHash = $sth->fetchrow_hashref;
	if( ! $dbRowHash ) { return undef; }

    }

    $chrName = $dbRowHash->{'chromosome'};

    $marker = Bio::EnsEMBL::Map::DBSQL::Marker->new( $dbobj, $markerId );
    $chrMap = $self->get_ChromosomeMap( $chrName );

    # change on final implementation
    $position = Bio::EnsEMBL::Map::PositionI->new
	( $dbRowHash->{'map_position'}." cR" );
    $mapMarker = Bio::EnsEMBL::Map::DBSQL::MapMarker->new
	( $dbobj, $marker, $chrMap, $position );
    $self->{'_MapMarker_cache'}->{$markerId} = $mapMarker;
    $mapMarker;
}



# compiled succesfully
1;

