# BioPerl module for Bio::EnsEMBL::Map::DBSQL::FPC_Map
#
# Creator: Arne Stabenau <stabenau@ebi.ac.uk>
# Date of creation: 22.05.2000
# Last modified : 22.05.2000 by Arne Stabenau
#
# Copyright EMBL-EBI 2000
#
# You may distribute this module under the same terms as perl itself

# POD documentation - main docs before the code

=head1 NAME

Bio::EnsEMBL::Map::DBSQL::FPC_Map 

=head1 SYNOPSIS

   $fpc_map = Bio::EnsEMBL::Map::DBSQL::Obj->get_Map( 'FPC' );
   May change to something like ...->get_Map( 'human_FPC' );


=head1 DESCRIPTION
   
     

=head1 CONTACT

    Contact Arne Stabenau on implemetation/design detail: stabenau@ebi.ac.uk
    Contact Ewan Birney on EnsEMBL in general: birney@sanger.ac.uk

=head1 APPENDIX

The rest of the documentation details each of the object methods. Internal methods are usually preceded with a _

=cut


# Let the code begin...


package Bio::EnsEMBL::Map::DBSQL::FPC_Map;
use vars qw(@ISA);
use strict;

use Bio::EnsEMBL::Map::DBSQL::FPC_ChromosomeMap;
use Bio::EnsEMBL::Map::DBSQL::FPC_Clone;

@ISA = qw( Bio::Root::Object );


# Object preamble - inheriets from Bio::Root::Object

# new() is inherited from Bio::Root::Object
# _initialize is where the heavy stuff will happen when new is called


=head2 Constructor

    Title   : _initialize
    Usage   : 
 Function: 
    Example :
    Returns : 
    Args    : 


=cut

sub _initialize {
  my $self = shift;
  my $dbobj = shift;

  $self->db( $dbobj );
  $self;
}


sub get_Clone_by_name {
  my $self = shift;
  my $clonename = shift;
  
  my $db = $self->db;
  my $cloneTable = $self->cloneTable;
  my $contigTable = $self->contigTable;

  my $sth = $db->prepare
    ( "select clone_name, embl_id, start_guess, fpc_size, ".
      "contig_name, start, length, chromosome from ".
      "$cloneTable a, $contigTable b where a.embl_id = \"".
      $clonename."\" and a.contig_id = b.contig_id"
    );

  my ( $clone_name, $embl_acc, $start_guess, $fpc_size,
       $contig_name, $start, $length, $chromosome );

  $sth->execute;
  $sth->bind_columns( undef,\$clone_name, \$embl_acc, \$start_guess,
		      \$fpc_size, \$contig_name, \$start, \$length,
		      \$chromosome );
  if( $sth->fetch ) {
    my $chromosomeMap = $self->get_ChromosomeMap
      ( $chromosome );

    my $contig  = Bio::EnsEMBL::Map::DBSQL::FPC_Contig->new
      ( $self, $contig_name, $chromosomeMap, $start, $length );

    my $clone = Bio::EnsEMBL::Map::DBSQL::FPC_Clone->new 
      ( $clone_name, $contig, $start_guess, $fpc_size );
    
    $clone->embl_acc( $embl_acc );
    return $clone;
  }

  return undef;
}

sub get_Contig_by_name {
  my $self = shift;
  my $contigname = shift;
  
  my $sth = $self->db->prepare
    ( "select contig_name, start, length, chromosome".
      " from Fpc_Contig where contig_name = \"".
      $contigname."\"" );
  $sth->execute;

  my $rowHash = $sth->fetchrow_hashref;
  if( ! defined $rowHash ) {
    return undef;
  }
  
  my $chromosomeMap = $self->get_ChromosomeMap
    ( $rowHash->{chromosome} );

  my $contig = Bio::EnsEMBL::Map::DBSQL::FPC_Contig->new
    ( $self, $rowHash->{contig_name},$chromosomeMap, 
      $rowHash->{start}, $rowHash->{'length'} );
  
  return $contig;
}


sub get_ChromosomeMap {
  my $self = shift;
  my $name = shift;

  if( $name eq 'unknown' ) {
    $name = '?';
  }
  
  if( defined $self->{_chrMaps}{$name} ) {
    return $self->{_chrMaps}{$name};
  }

  my $chrMap = Bio::EnsEMBL::Map::DBSQL::FPC_ChromosomeMap->new
    ( $self, $name );
  $self->{_chrMaps}{$name} = $chrMap;

  return $chrMap;
}

sub get_ChromosomeNames {
  my $self = shift;
  my @result;
  my $chr;

  my $sth = $self->db->prepare
    ( "select chromosome from ".$self->contigTable.
      " group by chromosome" );
  $sth->execute;
  $sth->bind_col( 1, \$chr );
  while( $sth->fetch ) {
    my $chrName = $chr;
    ( $chrName eq '?' ) &&
      ( $chrName = 'unknown' );
    push( @result, $chrName );
  }
  
  return @result;
}

# get set SQL table information

sub cloneTable {
  my $self = shift;
  my $cloneTable = shift;
  $cloneTable && 
    ( $self->{_cloneTable} = $cloneTable );
  $self->{_cloneTable};
}


sub contigTable {
  my $self = shift;
  my $contigTable = shift;
  $contigTable && 
    ( $self->{_contigTable} = $contigTable );

  $self->{_contigTable};
}

sub db {
  my $self = shift;
  my $dbobj = shift;
  
  $dbobj && 
    ( $self->{'_dbobj'} = $dbobj );
  
  $self->{'_dbobj'};
}

sub _get_FpcClone {
  my $self = shift;
  my $rowHash = shift;
  
  # is hashed?
  # hash overflow ?

  my $clone = Bio::EnsEMBL::Map::DBSQL::Fpc_Clone->new
    ( $self->db, $rowHash->clone_name, $rowHash->embl_id );
  $clone->_contig_name( $rowHash->contig_name );
}

 

sub deleteObj {
  my $self = shift;
  # do some deletion
}



  
