# BioPerl module for Bio::EnsEMBL::Map::DBSQL::FPC_Contig
#
# Creator: Arne Stabenau <stabenau@ebi.ac.uk>
# Date of creation: 23.05.2000
# Last modified : 31.05.2000 by Arne Stabenau
#
# Copyright EMBL-EBI 2000
#
# You may distribute this module under the same terms as perl itself

# POD documentation - main docs before the code

=head1 NAME

Bio::EnsEMBL::Map::DBSQL::FPC_Contig 

=head1 SYNOPSIS

   $fpc_contig = ($fpc_chrmap->get_Contig_by_fraction( 0.0 ))[0];
   ( Should give you the first FPC_Contig in a chromosome )

=head1 DESCRIPTION
   
     

=head1 CONTACT

    Contact Arne Stabenau on implemetation/design detail: stabenau@ebi.ac.uk
    Contact Ewan Birney on EnsEMBL in general: birney@sanger.ac.uk

=head1 APPENDIX

The rest of the documentation details each of the object methods. Internal methods are usually preceded with a _

=cut


# Let the code begin...


package Bio::EnsEMBL::Map::DBSQL::FPC_Contig;
use vars qw(@ISA);
use strict;

use Bio::EnsEMBL::Map::DBSQL::FPC_Clone;

@ISA = qw( Bio::Root::Object );


# Object preamble - inheriets from Bio::Root::Object

# new() is inherited from Bio::Root::Object
# _initialize is where the heavy stuff will happen when new is called


=head2 Constructor

    Title   : _initialize
    Usage   : 
 Function: 
    Example :
    Returns : 
    Args    : 


=cut

sub _initialize {
  my $self = shift;
  my $map = shift;
  my $name = shift;
  my $chromosome = shift;
  my $start =shift;
  my $length = shift;

  $self->fpcmap( $map );
  $self->{_name} = $name;
  $self->{_chromosome} = $chromosome;
  $self->{_start} = $start;
  $self->{_length} = $length;

  $self;

}


sub get_Clones {
  my $self = shift;
  my @result = ( 'Clone1', 'Clone2', 'Clone3' );

  $self->throw( "Not implemented yet, no data." );
}


sub get_sequenced_Clones {
  my $self = shift;

  my $db = $self->fpcmap->db;
  my $cloneTable = $self->fpcmap->cloneTable;
  my $contigTable = $self->fpcmap->contigTable;
  my @result;

  my $sth = $db->prepare
    ( "select clone_name, embl_id, start_guess, fpc_size from ".
      "$cloneTable a, $contigTable b where a.contig_id = b.contig_id and ".
      "b.contig_name = \"".$self->name."\" order by start_guess" );
  my ( $clone_name, $embl_acc, $start_guess, $fpc_size );
 
  $sth->execute;
  $sth->bind_columns( undef, \$clone_name, \$embl_acc, \$start_guess,
		      \$fpc_size );
  while( $sth->fetch ) {
    my $clone = Bio::EnsEMBL::Map::DBSQL::FPC_Clone->new 
      ( $clone_name, $self, $start_guess, $fpc_size );
    $clone->embl_acc( $embl_acc );
    push( @result, $clone );
  }
  return @result;
}

sub _get_ensembl_Clones {
  my $self = shift;

  my $db = $self->fpcmap->db;
  my $cloneTable = $self->fpcmap->cloneTable;
  my $contigTable = $self->fpcmap->contigTable;
  my @result;

  if( !$db->ensdbname ) {
    $self->throw( "No connection to EnsEMBL db. Set ensdbname on construction of map_db_obj." );
  }

  my $ensdbname = $db->ensdbname;

  my $sth = $db->prepare
    ( "select clone_name, a.embl_id, start_guess, fpc_size from ".
      "$cloneTable a, $contigTable b, $ensdbname.clone c where a.contig_id = b.contig_id and ".
      "b.contig_name = \"".$self->name."\" and a.embl_id = c.id order by start_guess" );
  my ( $clone_name, $embl_acc, $start_guess, $fpc_size );
 
  $sth->execute;
  $sth->bind_columns( undef, \$clone_name, \$embl_acc, \$start_guess,
		      \$fpc_size );
  while( $sth->fetch ) {
    my $clone = Bio::EnsEMBL::Map::DBSQL::FPC_Clone->new 
      ( $clone_name, $self, $start_guess, $fpc_size );
    $clone->embl_acc( $embl_acc );
    push( @result, $clone );
  }
  return @result;
  
}



sub name {
  my $self = shift;
  $self->{_name};
}

sub position_estimate {
  my $self = shift;
  if( $self->{_start} == -1 ) {
    return ();
  }
  return( $self->{_start}, $self->{_length});
}


# deprecated
sub virtual_contig {
  my $self = shift;
  $self->throw( "Not supported any more!" );
}


sub sister_Contig_up {
  my $self = shift;
  my $db = $self->fpcmap->db;
  my $contigTable = $self->fpcmap->contigTable;
  my $chromosomeMap = $self->{_chromosome};
  my $chromosome = $chromosomeMap->_sql_name;
  my $start = $self->{_start};

  if( $chromosome eq "unknown" ) {
    $chromosome = '?';
  }

  # max 3 should have the same start coordinate though
  # should not happen anyway
  my $sth = $db->prepare
    ( "select contig_name, start, length from ".
      $contigTable." where chromosome = $chromosome".
      " and start <= $start order by start desc ".
      "limit 3" );
  my ($name, $startval, $length );
  $sth->execute;
  $sth->bind_columns( \$name, \$startval, \$length );
  my $seen = 0;
  while( $sth->fetch ) {
    if( $startval == -1 ) {
      return undef;
    }

    if( $seen ) {
      my $contig = Bio::EnsEMBL::Map::DBSQL::FPC_Contig->new
	( $self->fpcmap, $name, $chromosomeMap, $startval, $length );
      return $contig;
    }
    if( $name eq $self->name ) {
      $seen = 1;
    }
  }
  return undef;
}

sub sister_Contig_down {
  my $self = shift;
  my $db = $self->fpcmap->db;
  my $contigTable = $self->fpcmap->contigTable;
  my $chromosomeMap = $self->{_chromosome};
  my $chromosome = $chromosomeMap->_sql_name;
  my $start = $self->{_start};

  if( $chromosome eq "unknown" ) {
    $chromosome = '?';
  }
  # max 3 should have the same start coordinate though
  # should not happen anyway
  my $sth = $db->prepare
    ( "select contig_name, start, length from ".
      $contigTable." where chromosome = $chromosome".
      " and start >= $start order by start ".
      "limit 3" );
  my ($name, $startval, $length );

  $sth->execute;

  $sth->bind_columns( \$name, \$startval, \$length );
  my $seen = 0;
  while( $sth->fetch ) {
    if( $seen ) {
      my $contig = Bio::EnsEMBL::Map::DBSQL::FPC_Contig->new
	( $self->fpcmap, $name, $chromosomeMap, $startval, $length );
      return $contig;
    }
    if( $name eq $self->name ) {
      $seen = 1;
    }
  }
  return undef;
  
}

sub get_ChromosomeMap {
  my $self = shift;
  return $self->{_chromosome};
}


sub fpcmap {
  my $self = shift;
  my $fpcmap = shift;

  ( defined $fpcmap ) &&
    ( $self->{_fpcmap} = $fpcmap );
  
  $self->{_fpcmap};
}
  
