# BioPerl module for Bio::EnsEMBL::Map::DBSQL::FPC_ChromosomeMap
#
# Creator: Arne Stabenau <stabenau@ebi.ac.uk>
# Date of creation: 22.05.2000
# Last modified : 31.05.2000 by Arne Stabenau
#
# Copyright EMBL-EBI 2000
#
# You may distribute this module under the same terms as perl itself

# POD documentation - main docs before the code

=head1 NAME

Bio::EnsEMBL::Map::DBSQL::FPC_ChromosomeMap 

=head1 SYNOPSIS

   $fpc_chrmap = $fpc_map->get_ChromosomeMap( '22' );

=head1 DESCRIPTION
   
     

=head1 CONTACT

    Contact Arne Stabenau on implemetation/design detail: stabenau@ebi.ac.uk
    Contact Ewan Birney on EnsEMBL in general: birney@sanger.ac.uk

=head1 APPENDIX

The rest of the documentation details each of the object methods. Internal methods are usually preceded with a _

=cut


# Let the code begin...


package Bio::EnsEMBL::Map::DBSQL::FPC_ChromosomeMap;
use vars qw(@ISA);
use strict;

use Bio::EnsEMBL::Map::DBSQL::FPC_Contig;

@ISA = qw( Bio::Root::Object );


# Object preamble - inheriets from Bio::Root::Object

# new() is inherited from Bio::Root::Object
# _initialize is where the heavy stuff will happen when new is called


=head2 Constructor

    Title   : 
    Usage   : 
 Function: 
    Example :
    Returns : 
    Args    : 


=cut

sub _initialize {
  my $self = shift;
  my $fpcMap = shift;
  my $name = shift;

  $self->fpcmap( $fpcMap );
  $self->chromosome_name( $name );

  $self;
}


sub get_ordered_Contigs {
  my $self = shift;
  my @result;

  my $contigTable = $self->fpcmap->contigTable;
  my $db = $self->fpcmap->db;

  my $sth = $db->prepare
    ( "select contig_name, start, length from $contigTable".
      " where chromosome = \"".$self->_sql_name."\"".
      " and start >= 0 order by start" );
  
  my ( $name, $start, $length );
  $sth->execute;

  $sth->bind_columns( undef,\$name, \$start, \$length );
  while( $sth->fetch ) {
 
    if( $start == -1 ) {
      next;
    }
    
    my $contig = Bio::EnsEMBL::Map::DBSQL::FPC_Contig->new
      ( $self->fpcmap, $name, $self, $start, $length );
    push( @result, $contig );
  }
  
  return @result;
}

sub get_unordered_Contigs {
  my $self = shift;
  my @result;



  my $contigTable = $self->fpcmap->contigTable;
  my $db = $self->fpcmap->db;

  my $sth = $db->prepare
    ( "select contig_name, start, length from $contigTable".
      " where chromosome = \"".$self->_sql_name."\"" );
  
  my ( $name, $start, $length );
  $sth->execute;

  $sth->bind_columns( undef,\$name, \$start, \$length );
  while( $sth->fetch ) {
 
    if( $start == -1 ) {
      my $contig = Bio::EnsEMBL::Map::DBSQL::FPC_Contig->new
	( $self->fpcmap, $name, $self, $start, $length );
      push( @result, $contig );
    }
  }
  
  return @result;
}

sub get_Contig_by_chromosome_fraction {
  my $self = shift;
  my $startnumber = shift; 
  # optional endnumber
  my $endnumber = shift;
  
  if( $startnumber < 0.0 || $startnumber > 1.0 ) {
    $self->throw( "Bad start fraction!" );
  }
  if( defined $endnumber && 
      ( $endnumber < 0.0 || $endnumber > 1.0 )) {
    $self->throw( "Bad end fraction!" );
  }
	
  my $length = $self->length;
  if( !defined $length ) {
    $self->throw( "Contig by fraction not supported!" );
  }

  $startnumber *= $length;
  if( defined $endnumber ) {
    $endnumber *= $length;
  }

  $self->get_Contig_by_base_position
    ( $startnumber, $endnumber );
}


sub get_Contig_by_base_position {
  my $self = shift;
  my $startnumber = shift;

  #optional endnumber
  my $endnumber = shift;
  my @result;
  my $contigTable = $self->fpcmap->contigTable;
  my $db = $self->fpcmap->db;
  my $takeIt = 0;

  my $sth = $db->prepare
    ( "select contig_name, start, length from $contigTable".
      " where chromosome = \"".$self->_sql_name."\"" );

  my ( $name, $start, $length );
  $sth->execute;

  $sth->bind_columns( \$name, \$start, \$length );
  while( $sth->fetch ) {
 
    if( $start == -1 ) {
      next;
    }
    
    if( $startnumber >= $start &&
	$startnumber <= ( $start + $length ) ) {
      $takeIt = 1;
    }
      
    if( defined $endnumber ) {
      if( $endnumber >= $start &&
	  $endnumber <= ( $start + $length ) ) {
	$takeIt = 1;
      }
    }

    if( $takeIt ) {
      my $contig = Bio::EnsEMBL::Map::DBSQL::FPC_Contig->new
	( $self->fpcmap, $name, $self->chromosome_name, $start, $length );
      push( @result, $contig );
    }
    $takeIt = 0;
  }
    
  return @result;
}


sub get_Clones_by_start_end {
    my $self = shift;
    my $chr_start = shift;
    my $chr_end = shift;
    my $db = $self->fpcmap->db;
    

    my $cloneTable = $self->fpcmap->cloneTable;
    my $contigTable = $self->fpcmap->contigTable;
    my @result;
    my $chr = $self->chromosome_name;

    my $sth = $db->prepare
	( "select clone_name, embl_id, start_guess, fpc_size, contig_name,
          start, length, start+start_guess+fpc_size as global_end,
          start+start_guess as global_start from ".
         "$cloneTable a, $contigTable b where a.contig_id = b.contig_id ".
	  "and chromosome='$chr'".
	  "and start+start_guess+fpc_size >= $chr_start
       and start+start_guess < $chr_end order by global_start" );

    my ( $clone_name, $embl_acc, $start_guess, $fpc_size );
    my ( $contig_name, $start, $length, $contigstart, $end );

    $sth->execute;
    $sth->bind_columns( undef, \$clone_name, \$embl_acc, \$start_guess,
			\$fpc_size, \$contig_name, \$contigstart, \$length, \$end, \$start );
    my ( $contig, $lastContig );
    $lastContig = undef;

    while( $sth->fetch ) {
	if( $contig_name ne $lastContig ) {
	    $contig = Bio::EnsEMBL::Map::DBSQL::FPC_Contig->new
		( $self->fpcmap, $contig_name, $self->chromosome_name, $contigstart, $length );
	    $lastContig = $contig_name;
	}

	my $clone = Bio::EnsEMBL::Map::DBSQL::FPC_Clone->new 
	    ( $clone_name, $contig, $start, $fpc_size );
	$clone->embl_acc( $embl_acc );
	push( @result, $clone );
    }
    return @result;
}



# return 100Mb
sub chromosome_length {
  my $self = shift;
  my $length;

  if( defined $self->{_length}) {
    return $self->{_length};
  }

  if( $self->chromosome_name eq 'unknown' ) {
    return undef;
  }
  
  my $db = $self->fpcmap->db;
  my $sth = $db->prepare( "select sum(length) from ChromosomeBands where chromosome = \"".$self->_sql_name."\"" );
  $sth->execute;
  $sth->bind_columns( undef,\$length);
  if( $sth->fetch ) {
    $self->{_length} = $length;
    return $length;
  } else {
    return undef;
  }
}

sub chromosome_name {
  my $self = shift;
  my $name = shift;

  if( defined $name ) {
    if( $name eq '?' ) {
      $name = 'unknown';
    }
    $self->{_name} = $name;
  }
  
  $self->{_name};
}

sub _sql_name {
  my $self = shift;
  my $name = $self->chromosome_name;
  if( $name eq 'unknown' ) {
    return '?';
  } else {
    return $name;
  }
}

sub fpcmap {
  my $self = shift;
  my $fpcmap = shift;

  ( defined $fpcmap ) &&
    ( $self->{_fpcmap} = $fpcmap );
  
  $self->{_fpcmap};
}
  

