# BioPerl module for Bio::EnsEMBL::Map::DBSQL::ChromosomeMap
#
# Creator: Arne Stabenau <stabenau@ebi.ac.uk>
# Date of creation: 21.01.2000
# Last modified : 28.04.2000 by Arne Stabenau
#
# Copyright EMBL-EBI 2000
#
# You may distribute this module under the same terms as perl itself

# POD documentation - main docs before the code

=head1 NAME

Bio::EnsEMBL::Map::DBSQL::ChromosomeMap 

=head1 SYNOPSIS
    $chromosomeMap = Bio::EnsEMBL::Map::DBSQL::Map->get_ChromosomeMap( $name );

=head1 DESCRIPTION

    As many chromosomes make a
    genome, many ChromosomeMap s make a Map. A ChromosomeMap contains
    merely a name and a refernce to which Map it belongs. It
    provides functions to extract MapMarkers which are on this chromosome
    in this map. The functions itself are provided by the Map object, as it
    knows how to prepare MapMarker objects.

=head1 CONTACT

    Contact Arne Stabenau on implemetation/design detail: stabenau@ebi.ac.uk
    Contact Ewan Birney on EnsEMBL in general: birney@sanger.ac.uk


=head1 APPENDIX

The rest of the documentation details each of the object methods. Internal methods are usually preceded with a _

=cut


# Let the code begin...


package Bio::EnsEMBL::Map::DBSQL::ChromosomeMap;
use vars qw(@ISA);
use strict;

use Bio::EnsEMBL::Map::ChromosomeMapI;
use Bio::EnsEMBL::Map::DBSQL::Obj;
require Bio::EnsEMBL::Map::DBSQL::Map;

@ISA = qw( Bio::EnsEMBL::Map::ChromosomeMapI );

# Object preamble - inheriets from Bio::Root::Object

# new() is inherited from Bio::Root::Object
# _initialize is where the heavy stuff will happen when new is called
=head2 Constructor

    Title   : _initialize
    Usage   : ...::ChromosomeMap->new( $dbsqlobj, $mapObject, $chrName );
 Function: ChromosomeMap is only created from Map. 
    Example : -
    Returns : -
    Args    : First is the DBSQL::Obj, second a Map object, third a  name
    string for the chromosome. Its should be the same for any Map object
    this chromosome appears. 

=cut

sub _initialize {
  my $self  = shift;
  my ( $dbobj, $mapObj, $chrName ) = @_;
  $self->SUPER::_initialize( @_ );

  # check args
  $self->throw
	("new ChromosomeMap(..) called with no ".
	 "DBSQL::Obj as first argument." ) 
	    unless ref($dbobj) && 
		$dbobj->isa("Bio::EnsEMBL::Map::DBSQL::Obj");
    $self->{'_dbsqlobj'} = $dbobj;

    $self->throw
	("new ChromosomeMap(..) called with no ".
	 "DBSQL::Map as second argument.")
	    unless ref($mapObj) && 
		$mapObj->isa("Bio::EnsEMBL::Map::DBSQL::Map");

    $self->{'_Map'} = $mapObj;

    $self->throw
	("new ChromosomeMap(..) called with no ".
	 "name as third argument.")
	    unless $chrName;

    $self->{'_name'} = $chrName;
    $self;
}

=head2 name

    Title   : name
    Usage   : ...::ChromosomeMap->name;
  Function: Gets the name of the ChromosomeMap object, which should be the
    name of the chromosome.
    Example : -
    Returns : A scalar with the name.
    Args    : - 


=cut

sub name {
    my ( $self ) = @_;
    $self->{'_name'};
}

=head2 get_Map

    Title   : get_Map
    Usage   : ...::ChromosomeMap->get_Map;
  Function: Gets the Map object this ChromosomeMap belongs to.
    Example : -
    Returns : A DBSQL::Map objects, ie a subclass of it.
    Args    : - 


=cut

sub get_Map {
    my ( $self ) = @_;
    $self->{'_Map'};
}

=head2 get_MapMarkers

    Title   : get_MapMarkers( [MapMarker start, MapMarker end] )
    Usage   : ...::ChromosomeMap->get_MapMarkers( $MapMarker1, $MapMarker2 );
  Function: Gets all the MapMarkers in this chromosome. Is put through to
    the Map object. Optionally give only thos between given (including them).
    Example : -
    Returns : A reference to an array of MapMarker objects.
    Args    : - 


=cut

sub get_MapMarkers {
  my $self = shift;
  my ( $mapMarkerStart, $mapMarkerEnd ) = @_;

  if( $mapMarkerEnd ){
    $self->throw( "Argument 1 of get_MapMarkers is not".
		  " a MapMarker" ) 
      unless ref( $mapMarkerStart ) && $mapMarkerStart->isa
	( "Bio::EnsEMBL::Map::DBSQL::MapMarker" );
    $self->throw( "Argument 2 of get_MapMarkers is not".
		  " a MapMarker" ) 
      unless ref( $mapMarkerEnd ) && $mapMarkerEnd->isa
	( "Bio::EnsEMBL::Map::DBSQL::MapMarker" );
    return $self->{'_Map'}->_get_MapMarkers_Chromosome
      ( $self->{'_name'}, $mapMarkerStart, $mapMarkerEnd );
  } else {
    return $self->{'_Map'}->_get_MapMarkers_Chromosome
      ( $self->{'_name'});
  }

}

=head2 get_MapMarkers_partly

    Title   : get_MapMarkers_partly
    Usage   : ...::ChromosomeMap->
                     get_MapMarkers_partly( $where_in_%, $minimum_number);
    Function: Gets all MapMarkers in this chromosome. Its tried to get 
              $minimum_number of MapMarker up and downwards of the position. 
              There is no guarantee on the returned number of markers.
    Example : -
    Returns : A reference to an array of MapMarker objects.
    Args    : $where is between 0.0 and 1.0

=cut


sub get_MapMarkers_partly {
   my $self = shift; 
  my ( $where, $num ) = @_;

  return $self->{'_Map'}->_get_MapMarkers_Chromosome_partly
	    ( $self->{'_name'}, $where, $num);
}


=head2 next_MapMarker

  Title   : next_MapMarker( $oneMarker )
  Usage   : $chrMap->next_MapMarker( $MapMarker1 );
  Function: Given one mapMarker, returns the following.
  Example : -
  Returns : A MapMarker object
  Args    : - 

=cut

sub next_MapMarker {
  my $self = shift;
  my $mapMarker = shift;

# argument check, map and chromosome correct?
    $self->throw( "Argument 1 of next_MapMarker is not".
		 " a MapMarker" ) 
	unless ref( $mapMarker ) && $mapMarker->isa
	    ( "Bio::EnsEMBL::Map::DBSQL::MapMarker" );

    $self->{'_Map'}->_next_MapMarker_Chromosome
	( $mapMarker, $self->{'_name'} );
}

=head2 prev_MapMarker

    Title   : prev_MapMarker( $oneMarker )
    Usage   : $chrMap->prev_MapMarker( $MapMarker1 );
 Function: Given one mapMarker, returns the previous.
    Example : -
    Returns : A MapMarker object
    Args    : - 


=cut

sub prev_MapMarker {
    my $self = shift;
    my $mapMarker = shift;

# argument check, map and chromosome correct?
    $self->throw( "Argument 1 of prev_MapMarker is not".
		 " a MapMarker" ) 
	unless ref( $mapMarker ) && $mapMarker->isa
	    ( "Bio::EnsEMBL::Map::DBSQL::MapMarker" );

    $self->{'_Map'}->_prev_MapMarker_Chromosome
	( $mapMarker, $self->{'_name'} );
}

=head2 chromosome

    Title   : chromosome
    Usage   : $chrMap->chromosome( [$chrmosome] );
 Function: Get/Set the chromosome object, which will provide a length for
    the chromosome and access to the bands. 
    Example : -
    Returns : -
    Args    : - 


=cut

sub chromosome {
    my $self = shift;
    my $chromosome = shift;

    $chromosome && ( $self->{'_chromosome'} = $chromosome );
    $self->{'_chromosome'};
}


# compiled succesfully
1;

