# BioPerl module for Bio::EnsEMBL::Map::DBSQL::Chromosome
#
# Creator: Arne Stabenau <stabenau@ebi.ac.uk>
# Date of creation: 16.03.2000
# Last modified : 16.03.2000 by Arne Stabenau
#
# Copyright EMBL-EBI 2000
#
# You may distribute this module under the same terms as perl itself

# POD documentation - main docs before the code

=head1 NAME

Bio::EnsEMBL::Map::DBSQL::Chromosome

=head1 SYNOPSIS


=head1 DESCRIPTION

    Implements ChromosomeI. Uses table ??.

=head1 CONTACT


    Contact Arne Stabenau on implemetation/design detail: stabenau@ebi.ac.uk
    Contact Ewan Birney on EnsEMBL in general: birney@sanger.ac.uk


=head1 APPENDIX

The rest of the documentation details each of the object methods. Internal methods are usually preceded with a _

=cut


# Let the code begin...


package Bio::EnsEMBL::Map::DBSQL::Chromosome;
use vars qw(@ISA);
use strict;
use Bio::Root::Object;
use Bio::EnsEMBL::Map::ChromosomeI;
use Bio::EnsEMBL::Map::DBSQL::ChromosomeBand;

# Object preamble - inheriets from Bio::Root::Object
@ISA = qw( Bio::EnsEMBL::Map::ChromosomeI );

=head2 _initialize

  Title   : _initialize
  Usage   : $chromosome = Bio::EnsEMBL::Map::DBSQL::Chromosome->new
    ( $name );
  Function: Constructing the Chromosome object.
  Example :
  Returns :
  Args    :

=cut

sub _initialize {
    my $self = shift;
    $self->SUPER::_initialize( @_ );

    $self->{'_name'} = shift;
    $self->{'_bands'} = [];

    $self;
}

=head2 name

    Title   : name
    Usage   : $name = $chromosomeI->name;
    Function: returns an name string for a Chromosome
    Example : 
    Returns : 
    Args    : -

=cut

sub name {
    my ($self) = @_;
    $self->{'_name'};
}

=head2 length

    Title   : length
    Usage   : $length = $chromosomeI->length 
    Function: Returns a length guess for the Chromosome in kB.
    Example : -
    Returns : -
    Args    : -

=cut

sub length {
    my $self = shift;
    my $len = shift;
    if( defined $len ) { 
	if( $len == 0 ) { 
	    #$self->warn("Setting length... to $len\n"); # 
	} 
	$self->{'_length'} = $len;
    }

   $self->{'_length'};
}

=head2 band_Count

    Title   : band_Count
    Usage   : $num = $chrMap->band_Count;
    Function: Returns the number of visible Bands in the Chromosome.
    Example : -
    Returns : -
    Args    : -

=cut

sub band_Count {
    my ($self) = @_;
    my $count = $#{$self->{'_bands'}} + 1;
#    print "Count is $count.\n";
    return $count;
}

=head2 get_Band

    Title   : get_Band
    Usage   : $chromomsomeBandI = $chromosomeI->get_Band( $num );
    Function: Returns the n-th band/subband as ChromosomeBandI object.
    Example : -
    Returns : undef, if not possible.
    Args    : -

=cut

sub get_Band {
    my $self = shift;
    my $num = shift;
    ($self->{'_bands'})->[$num-1];
}

sub get_Band_by_name {
    my $self = shift;
    my $name = shift;
# if the given name is finer than the stored bands
# give just a band
# if its coarser, collect bands and return artificial band 
    my ( $start, $length, $bandname, $band, $position, $colour );
    foreach $band (@{$self->{'_bands'}}) {
	$bandname = $band->name;

	if( $name =~ /^$bandname/ ) {
	    return $band;
	}
	if( $bandname =~ /^$name/ ) {
	    if( defined( $position )) {
		$length += $band->length;
	    } else {
		$position = $band->position;
		$colour = $band->colour;
		$length = $band->length;
	    }
	} else {
	    if( defined( $position )) {
		last;
	    }
	}
    }

    if( !defined( $position )) {
	return undef;
    }

    $band = Bio::EnsEMBL::Map::DBSQL::ChromosomeBand->new
	( $name, $length, $colour );
    $band->position( $position );
    return $band;
}

sub get_Bands {
    my $self = shift;
    $self->{'_bands'};
}

=head2 add_Band

  Title   : add_Band
  Usage   : $chromosome->add_Band( $chromosomeBand );
  Function: Add a band at the down (q) end of a chromosome.
  Example : -
  Returns : -
  Args    : a ChromosomeBand

=cut

sub add_Band {
    my $self = shift;
    my $band = shift;
    $self->throw( "add_Band not called with ChromosomeBandI object!" )
	unless $band->isa( "Bio::EnsEMBL::Map::ChromosomeBandI" );
    my $count = $self->band_Count;
    push( @{$self->{'_bands'}}, $band );
    if( $count == 0 ) {
	$band->position( 0 );
	$self->length( $band->length );
    } else {
	my $prevBand = $self->get_Band( $count );
	$band->position( $prevBand->position + $prevBand->length );
	$self->length( $self->length + $band->length );
    }
}

# compiled successfull

1;




