# BioPerl module for Bio::EnsEMBL::Map::DBSQL::CGMap
#
# Creator: Arne Stabenau <stabenau@ebi.ac.uk>
# Date of creation: 08.02.2000
# Last modified : 08.02.2000 by Arne Stabenau
#
# Copyright EMBL-EBI 2000
#
# You may distribute this module under the same terms as perl itself

# POD documentation - main docs before the code

=head1 NAME

Bio::EnsEMBL::Map::DBSQL::CGMap subclass of Map. When using EnsEMBL::Map
    you should never be too much aware of what Map you use.

=head1 SYNOPSIS

    $map = Bio::EnsEMBL::Map::DBSQL::Obj->get_Map( $id );
    @maps = Bio::EnsEMBL::Map::DBSQL::Obj->get_Maps();

=head1 DESCRIPTION

    Design detail, look in Map

=head1 CONTACT

    Contact Arne Stabenau on implemetation/design detail: stabenau@ebi.ac.uk
    Contact Ewan Birney on EnsEMBL in general: birney@sanger.ac.uk


=head1 APPENDIX

The rest of the documentation details each of the object methods. Internal methods are usually preceded with a _

=cut


# Let the code begin...


package Bio::EnsEMBL::Map::DBSQL::CGMap;
use vars qw(@ISA);
use strict;

use Bio::EnsEMBL::Map::DBSQL::Obj;
use Bio::EnsEMBL::Map::DBSQL::Map;
use Bio::EnsEMBL::Map::DBSQL::MapMarker;
use Bio::EnsEMBL::Map::DBSQL::Marker;
use Bio::EnsEMBL::Map::DBSQL::BandedPosition;


@ISA = qw( Bio::EnsEMBL::Map::DBSQL::Map );


# Object preamble - inheriets from Bio::Root::Object

# new() is inherited from Bio::Root::Object
# _initialize is where the heavy stuff will happen when new is called
=head2 Constructor

    Title   : _initialize
    Usage   : ...::RHMap->new( $dbsqlobj, $mapID );
 Function: Map is only created from Obj. 
    Example : -
    Returns : -
    Args    : First is the DBSQL::Obj, second an id string for the map. Its 
    used to find more things on the map in the db.

=cut

sub _initialize {
    my $self = shift;
    $self->SUPER::_initialize( @_ );

    $self;
}

=head2 get_MapMarkers

    Title   : get_MapMarkers
    Usage   : ...::Map->get_MapMarkers
  Function: Returns all MapMarker objects accociated with this Map. These
    are quite a few, better fetch only needed ones! 
    Example : -
    Returns : Reference to an array
    Args    : - 


=cut

sub get_MapMarkers {
    my ( $self ) = @_;
    my $dbobj = $self->{'_dbsqlobj'};
    my $result = [];
    my ($markerId, $mapMarker,$dbRowHash );

    my $command = "select marker, chromosome, start_band, end_band, start_sub_band, end_sub_band from ".$self->tablename."\"";

    my $sth = $dbobj->prepare( $command );
    $sth->execute;
    while( $dbRowHash = $sth->fetchrow_hashref ) {
	$markerId = $dbRowHash->{'marker'};
	$mapMarker = $self->_create_MapMarker( $markerId, $dbRowHash );
	push( @$result, $mapMarker );
    }

    $result;
}

=head2 get_MapMarker

    Title   : get_MapMarker
    Usage   : ...::Map->get_MapMarker( $id )
  Function: Returns MapMarker object with  given id or undef if  there isnt
    Example : -
    Returns : -
    Args    : - 


=cut

sub get_MapMarker {
    my $self  =  shift;
    my $id = shift;
    return $self->_create_MapMarker( $id );
}

=head2 _get_MapMarkers_Chromosome

    Title   : _get_MapMarkers_Chromosome
    Usage   : ...::ChromosomMap->get_MapMarkers
    ( [ $startMapMarker, $endMapMarker ] );
 Function: Returns MapMarker objects specific to a chromosomeMap. It is
    assuemd that the function is called from a ChromosomeMap object!
    Example : -
    Returns : Reference to an array
    Args    : - 


=cut

sub _get_MapMarkers_Chromosome {
    my $self = shift;
    my ( $chrName, $mapMarkerStart, $mapMarkerEnd ) = @_;
    my $dbobj = $self->{'_dbsqlobj'};
    my @result ;
    my @unsortedResult;

    my ( $markerId, $mapMarker, $dbRowHash );

    my $desc = "";

    my $startMarkerSeen;
    if( $mapMarkerStart && $mapMarkerEnd ) {
	$startMarkerSeen = 0;
	if( $mapMarkerStart->position->is_lower
	   ( $mapMarkerEnd->position )) {
	    $desc = "desc ";
	}
    } else {
	$startMarkerSeen = 1;
    }

    my $command = "select marker, chromosome, start_band, end_band, start_sub_band, end_sub_band,position_tag  from ".$self->tablename." where chromosome=\"$chrName\" order by position_tag $desc, marker $desc";

    my $sth = $dbobj->prepare( $command );
    $sth->execute;

    while( $dbRowHash = $sth->fetchrow_hashref ) {
	$markerId = $dbRowHash->{'marker'};

	if( $mapMarkerStart && ( $mapMarkerStart->marker->id eq $markerId )) {
	    $startMarkerSeen = 1;
	}
	if( $startMarkerSeen ) {
	    $mapMarker = $self->_create_MapMarker( $markerId, $dbRowHash );
	    push( @unsortedResult, $mapMarker );
	}
	if( $mapMarkerEnd && ( $mapMarkerEnd->marker->id eq $markerId )) {
	    last;
	}
    }
    
    @result = sort {
	if( $a->position->is_higher( $b->position )) {
	    -1;
	} elsif( $a->position->is_lower( $b->position )) {
	    1;
	} else {
	    $a->marker->id cmp $b->marker->id;
	}
    } @unsortedResult;

    if( $desc eq "desc " ) {
	@result = reverse @result;
    }
    \@result;
}

=head2 get_MarkerIds

    Title   : get_MarkerIds
    Usage   : ...::Map->get_MarkerIds
  Function: Returns all Markernames in this Map.
    Example : -
    Returns : Reference to an array
    Args    : - 


=cut

sub get_MarkerIds {
    my ( $self ) = @_;
    my $dbobj = $self->{'_dbsqlobj'};
    my $result = [];
    my ($markerId, $mapMarker,$dbRowHash );

    my $command = "select marker from ".
	$self->tablename;

    my $sth = $dbobj->prepare( $command );
    $sth->execute;
    while( $dbRowHash = $sth->fetchrow_hashref ) {
	$markerId = $dbRowHash->{'marker'};
	push( @$result, $markerId );
    }

    $result;
}



=head2 _create_MapMarker 
    Title : _create_MapMarker 
    Usage : ...::Map->_create_MapMarker 
    Function: MapMarker objects are kept unique in the application. Every
    reference to them is generated by this function and it caches already
    generated MapMarkers. 
    Example : - 
    Returns : undef if given markerId not in this map.
    Args : 1: $markerId, 2: (optional)
    a hash reference with named attributes of the MapMarker.


=cut


sub _create_MapMarker {
    my $self = shift;
    my ( $markerId, $dbRowHash ) = @_;
    my ( $sth, $chrName );

    if( $self->{'_MapMarker_cache'}->{$markerId} ) {
	return $self->{'_MapMarker_cache'}->{$markerId};
    }

    my ( $dbobj, $marker, $chrMap );
    my ( $mapMarker, $position, $command );
    $dbobj = $self->{'_dbsqlobj'};

    if( !$dbRowHash ) {

	# Genetic and radiationhybrids are in the same table at the moment
	# and have the same data...
   	$command = "select chromosome, start_band, end_band, start_sub_band, end_sub_band,position_tag from ".$self->tablename." where marker=\"$markerId\" "; 

	$sth = $dbobj->prepare( $command );
	$sth->execute;
	$dbRowHash = $sth->fetchrow_hashref;
	if( ! $dbRowHash ) { return undef; }
    } 
    $chrName = $dbRowHash->{'chromosome'};

    $marker = Bio::EnsEMBL::Map::DBSQL::Marker->new( $dbobj, $markerId );
    $chrMap = $self->get_ChromosomeMap( $chrName );

    # change on final implementation
    $position = Bio::EnsEMBL::Map::DBSQL::BandedPosition->new
	( $dbRowHash->{'start_band'}, $dbRowHash->{'end_band'},
	 $dbRowHash->{'start_sub_band'}, $dbRowHash->{'end_sub_band'});
    $position->tag( $dbRowHash->{'position_tag'});
    $mapMarker = Bio::EnsEMBL::Map::DBSQL::MapMarker->new
	( $dbobj, $marker, $chrMap, $position );
    $self->{'_MapMarker_cache'}->{$markerId} = $mapMarker;
    $mapMarker;
}


sub _next_MapMarker_Chromosome {
    my ( $self, $mapMarker, $chrName ) = @_;

    my $id = $mapMarker->marker->id;
    my $pos = $mapMarker->position->tag;
    my $idx = 0;
    my $thisIdx = -1;
    my ( $command, $sth, $rowRef );
    my $dbh = $self->{'_dbsqlobj'};

    while (1) {
	my $command = "select * from ".$self->tablename." where chromosome='$chrName' and position_tag >=$pos order by position_tag, marker limit $idx,".($idx+5);
	$sth = $dbh->prepare( $command );
	$sth->execute;
	if( $sth->rows == 0 ) {
	    die( "No next Marker" );
	}
	while( $rowRef = $sth->fetchrow_hashref ) {
	    if( $thisIdx != -1 ) {
		return $self->_create_MapMarker( $rowRef->{'marker'}, $rowRef );
	    }
	    if( $rowRef->{'marker'} eq $id ) {
		$thisIdx = $idx;
	    }
	    if( $rowRef->{'position_tag'} > $pos ) {
		die( "No next Marker" );
	    }
	    $idx++;
	}
    }
}

sub _prev_MapMarker_Chromosome {
    my ( $self, $mapMarker, $chrName ) = @_;
    my $id = $mapMarker->marker->id;
    my $pos = $mapMarker->position->tag;
    my $idx = 0;
    my $thisIdx = -1;
    my ( $command, $sth, $rowRef );
    my $dbh = $self->{'_dbsqlobj'};

    while (1) {
	my $command = "select chromosome, start_band, end_band, start_sub_band, end_sub_band,position_tag,marker from ".$self->tablename." where chromosome='$chrName' and position_tag <= $pos order by position_tag DESC, marker DESC limit $idx,".($idx+5);
	$sth = $dbh->prepare( $command );
	$sth->execute;
#	print STDERR ($command,"\n");
	if( $sth->rows == 0 ) {
	    die( "No previous Marker" );
	}
	while( $rowRef = $sth->fetchrow_hashref ) {
#	    print STDERR  ( "Got row: ",%$rowRef,"\n" );
	    if( $thisIdx != -1 ) {
		return $self->_create_MapMarker( $rowRef->{'marker'}, $rowRef );
	    }
	    if( $rowRef->{'marker'} eq $id ) {
		$thisIdx = $idx;
	    }
	    if( $rowRef->{'position_tag'} < $pos ) {
		die( "No previous Marker" );
	    }
	    $idx++;
	}
    }
}

=head2 _positionGuess 
    Title : _positionGuess
    Usage : dont, is used by MapMarker.
    Function: returns ($position, $length) in basepairs
    Example : - 
    Returns : 
    Args : 1: $position object, 2: $chromosomeMap

=cut

sub _positionGuess {
    my ($self, $position, $chromosomeMap ) = @_;


    # get the band from start position
    # get the band from end position
    my ( $startPosition, $endPosition );
    my ( $startGuess, $lengthGuess, $band );

    $startPosition = $position->start_band;
    if( $position->start_sub_band ) {
	$startPosition .= ".".$position->start_sub_band;
    }

    $band = $chromosomeMap->chromosome->
	get_Band_by_name( $startPosition );

    if( !defined( $band )) {
	return ( undef, undef );
    }

    $startGuess = $band->position;
    $lengthGuess =  $band->length;
    if( $position->end_band ) {
	if( $position->end_sub_band ) {
	    $endPosition = $position->end_band.".".
		$position->end_sub_band;
	} else {
	    $endPosition = $position->end_band;
	}
    }

    if( defined( $endPosition )) {
	$band = $chromosomeMap->chromosome->
	    get_Band_by_name( $endPosition );
	$lengthGuess = $band->position - $startGuess +
	    $band->length;
    }

    $startGuess = int( $startGuess*1000 );
    $lengthGuess = int( $lengthGuess*1000 );

    return ( $startGuess, $lengthGuess );
}

# compiled succesfully
1;

