# BioPerl module for Bio::EnsEMBL::Map::DBSQL::BandedPosition
#
# Creator: Arne Stabenau <stabenau@ebi.ac.uk>
# Date of creation: 25.01.2000
# Last modified : 25.01.2000 by Arne Stabenau
#
# Copyright EMBL-EBI 2000
#
# You may distribute this module under the same terms as perl itself

# POD documentation - main docs before the code

=head1 NAME

Bio::EnsEMBL::Map::DBSQL::BandPostion

=head1 SYNOPSIS


=head1 DESCRIPTION

    An implementation of the PositionI interface that provides some
    functionality for banded map-positions.

=head1 CONTACT

    Contact Arne Stabenau on implemetation/design detail: stabenau@ebi.ac.uk
    Contact Ewan Birney on EnsEMBL in general: birney@sanger.ac.uk

=head1 APPENDIX

The rest of the documentation details each of the object methods. Internal methods are usually preceded with a _

=cut


# Let the code begin...


package Bio::EnsEMBL::Map::DBSQL::BandedPosition;
use vars qw(@ISA);
use strict;
use Bio::EnsEMBL::Map::BandedPositionI;

@ISA = qw( Bio::EnsEMBL::Map::BandedPositionI );

# Object preamble - inheriets from Bio::Root::Object

# new() is inherited from Bio::Root::Object
# _initialize is where the heavy stuff will happen when new is called
sub _initialize {
    my $self = shift;
    $self->SUPER::_initialize( @_ );
    my ( $start_band, $end_band, $start_sub_band, $end_sub_band ) = @_;

    defined( $start_band ) || 
	$self->throw( "One band required in BandedPosition initializer.");
    $self->{'_start_band'} = $start_band;
    $self->{'_end_band'} = $end_band;
    $self->{'_start_sub_band'} = $start_sub_band;
    $self->{'_end_sub_band'} = $end_sub_band;
}


=head2 

    Title   : id  
    Usage   : ...::BandedPosition->id;
    Function: returns a string that describes this positon
    Example : -
    Returns : -
    Args    : -

=cut

sub id {
    my $self = shift;
    if( $self->{'_start_sub_band'} ) {
	return $self->{'_start_band'}.".".$self->{'_start_sub_band'};
    } else {
	return $self->{'_start_band'};
    }
}

=head2 tag

    Title   : tag
    Usage   : $bandedPosition->tag;
    Function: The tag is an integer, that help getting the BandedMarker out 
    of the sql table in a sorted way. It increases when
    start_band/start_sub_band goes closer to q-end. Stays same when start
    position stays same.
    Example : -
    Returns : -
    Args    : -

=cut

sub tag {
    my ($self,$tag) = @_;
    $tag && ( $self->{'_tag'} = $tag );
    $self->{'_tag'};
}


=head2 Access functions for the attributes

    Title   : start_band, start_sub_band, end_band, end_sub_band
    Usage   : $bandedPosition->xxxx;
    Function: returns a string that describes this positon
    Example : -
    Returns : -
    Args    : -

=cut

sub start_band {
    my $self = shift;
    $self->{'_start_band'};
}

sub start_sub_band {
    my $self = shift;
    $self->{'_start_sub_band'};
}

sub end_band {
    my $self = shift;
    $self->{'_end_band'};
}

sub end_sub_band {
    my $self = shift;
    $self->{'_end_sub_band'};
}

=head2 is_higher

    Title   : is_higher  
    Usage   : ...::$bandedPositionObj->is_higher( $otherBandPos ) ;
    Function: returns a boolean if this position is closer to p end than
    the given one. Only start position is considered.
    Example : -
    Returns : -
    Args    : -

=cut

sub is_higher {
    my $self = shift;
    my $otherBand = shift;

    $self->throw
	("is_higher(..) called with no argument BandedPositionI!")
	    unless ref($otherBand) &&
		$otherBand->isa( "Bio::EnsEMBL::Map::BandedPositionI");
    ( $self->_cmp( $otherBand ) == 1 );
}

=head2 is_lower

    Title   : is_lower  
    Usage   : ...::$bandedPositionObj->is_lower( $otherBandPos ) ;
    Function: returns a boolean if this position is farer to p end than
    the given one. Only start position is considered 
    Example : -
    Returns : -
    Args    : -

=cut

sub is_lower {
    my $self = shift;
    my $otherBand = shift;

    $self->throw
	("is_higher(..) called with no argument BandedPositionI!")
	    unless ref($otherBand) &&
		$otherBand->isa( "Bio::EnsEMBL::Map::BandedPositionI");
    ( $self->_cmp( $otherBand ) == -1 );
}


sub _cmp {
    my ( $self, $other ) = @_;
    my $reverse;

    if( $self->start_band =~ /^[pP]/ ) {
	if( $other->start_band =~ /^[pP]/ ) {
	    $reverse = 1;
	} else {
	    return 1;
	}
    } else {
	if( $other->start_band =~ /^[pP]/ ) {
	    return -1;
	} else {
	    $reverse = -1;
	}
    }

    my ($myNum, $otherNum );
    ($myNum) = ( $self->start_band =~ /^[pPqQ]([0-9]+)/ );
    ($otherNum) = ( $other->start_band =~ /^[pPqQ]([0-9]+)/ );
    if( $otherNum > $myNum ) { return -1*$reverse; }
    if( $myNum > $otherNum ) { return $reverse; }

   # major band seems equal

    if( !$self->start_sub_band ) {
	$myNum = 0;
    } else {
	$myNum = $self->start_sub_band;
    }

    if( !$other->start_sub_band ) {
	$otherNum = 0;
    } else {
	$otherNum = $other->start_sub_band;
    }
    if( $myNum > $otherNum ) {
	return $reverse;
    }

    if( $myNum < $otherNum ) {
	return -1*$reverse;
    }

    # they are equal 

    return 0;

}
# compiled successfull

1;
