#
# BioPerl module for EnsEMBL::ExternalData::GMAPAdaptor
#

# POD documentation - main docs before the code

=head1 NAME

Bio::EnsEMBL::ExternalData::Obj - Object representing an instance of an EnsEMBL DB

=head1 SYNOPSIS

    $db = Bio::EnsEMBL::ExternalData::GMAPAdaptor->new(
        -user   => 'root',
        -dbname => 'pog',
        -host   => 'caldy',
        -driver => 'mysql',
        );

    $clone  = $db->get_clone('X45667');



    

=head1 DESCRIPTION

This object represents a database that is implemented somehow (you shouldn\'t
care much as long as you can get the object). From the object you can pull
out other objects by their stable identifier, such as Clone (accession number),
The clone gives you a DB::Clone object, from
which you can pull out associated genes and features. 

=head1 CONTACT

Describe contact details here

=head1 APPENDIX

The rest of the documentation details each of the object methods. Internal methods are usually preceded with a _

=cut


# Let the code begin...


package Bio::EnsEMBL::ExternalData::GMAPAdaptor;

use vars qw(@ISA $AUTOLOAD);
use strict;

# Object preamble 

use Bio::Root::RootI;

use Bio::EnsEMBL::DB::ObjI;
use Bio::EnsEMBL::DBSQL::Update_Obj;
use Bio::EnsEMBL::DBSQL::Feature_Obj;

use Bio::EnsEMBL::Clone;
use Bio::EnsEMBL::DBSQL::AnalysisAdaptor;
use Bio::EnsEMBL::FeatureFactory;

use Bio::EnsEMBL::DB::ExternalFeatureFactoryI;
use Bio::EnsEMBL::FeaturePair;
use Bio::EnsEMBL::SeqFeature;

use Carp qw(cluck); 
use DBI; 
use Bio::EnsEMBL::DBSQL::SQL;


@ISA = qw(Bio::Root::RootI Bio::EnsEMBL::DB::ExternalFeatureFactoryI);
			#Bio::EnsEMBL::DB::ObjI 

sub new {
  my($pkg, @args) = @_;

  my $self = bless {}, $pkg;

    my (
        $db,
        $host,
        $driver,
        $user,
        $password,
        $debug,
        $perl,
        $perlonlysequences,
        $external,
        $port,
        $contig_overlap_source,
        $overlap_distance_cutoff,
        ) = $self->_rearrange([qw(
            DBNAME
	    HOST
	    DRIVER
	    USER
	    PASS
	    DEBUG
	    PERLONLYFEATURES
	    PERLONLYSEQUENCES
	    EXTERNAL
	    PORT
            CONTIG_OVERLAP_SOURCE
            OVERLAP_DISTANCE_CUTOFF
	    )],@args);
    $db   || $self->throw("Database object must have a database name");
    $user || $self->throw("Database object must have a user");

    #
    # This needs to be rethought. We are caching sequences
    # here to allow multiple exons to be retrieved fine
    # And now more cache's. I think cache's might be a fact of life...
    # 

    $self->{'_lock_table_hash'} = {};
    $self->_analysis_cache({});
    $self->{'_external_ff'} = [];

    if( ! defined $perlonlysequences ) {
        $perlonlysequences = 0;
    }


    if( $debug ) {
	print STDERR "GMAP D $debug U $user P $password by $$\n"; 
	$self->_debug($debug);
    } else {
	$self->_debug(0);
    }
    if( ! $driver ) {
	$driver = 'mysql';
    }
    if( ! $host ) {
	$host = 'localhost';
    }

    my $dsn;
    if($driver =~ /Oracle/i ) {
        # $dsn="DBI:$driver:$db"; #worked on gruyere, not on chevre
	$dsn="dbi:$driver:";
	$user=qq{$user/$password@(DESCRIPTION=
	      (ADDRESS=(PROTOCOL=TCP)(HOST=$host)(PORT=$port))
	      (CONNECT_DATA=(SID=$db)))};
        $password="";
    } else {
	if ( ! $port ) {
	    $port = 3306;
	}
	$dsn = "DBI:$driver:database=$db;host=$host;port=$port";
    }

    print STDERR "trying $dsn\n" if $debug;
	
    if( $debug && $debug > 10 ) {
        $self->_db_handle("dummy dbh handle in debug mode $debug");
    } else {

        #my $dbh = DBI->connect("$dsn","$user",$password, {RaiseError => 1});
        my $dbh = Bio::EnsEMBL::DBSQL::SQL->new($dsn,$user,$password );
	    #NB: this has RaiseError=>0

	$dbh || cluck "connect($dsn,$user,$password) failed";

        $dbh || $self->throw("Could not connect to database $db user $user using [$dsn] as a locator");

        if( $self->_debug > 3 ) {
	    $self->warn("Using connection $dbh");
        }

        $self->_db_handle($dbh);
	if($self->_debug >4)
	{
	    my $sth=$dbh->prepare("select table_name from user_tables");
	    if($sth) {
		$sth->execute();
		print STDERR "tables:\n";
		my $table;
		print STDERR  "$table\n" while( ($table)=$sth->fetchrow_array());
	    } else {
		print STDERR "Can't do table query\n";
	    }
	}
    }
    
    $self->driver ( $driver );
    $self->username( $user );
    $self->host( $host );
    $self->dbname( $db );
    $self->password( $password);



    if ($perl && $perl == 1) {
        $Bio::EnsEMBL::FeatureFactory::USE_PERL_ONLY = 1;
    }

    $self->perl_only_sequences($perlonlysequences);

    if( defined $external ){
        foreach my $external_f ( @{$external} ) {
	    $self->add_ExternalFeatureFactory($external_f);
        }
    }


  

    return $self; # success - we hope!
}

=head2 get_Update_Obj

 Title   : get_Update_Obj
 Usage   :
 Function:
 Example :
 Returns : 
 Args    :


=cut

sub get_Update_Obj {
    my ($self) = @_;
    
    my $update_obj = Bio::EnsEMBL::DBSQL::Update_Obj->new($self);
 
    return $update_obj;
  }

# only the get part of the 3 functions should be considered public

sub dbname {
  my ($self, $arg ) = @_;
  ( defined $arg ) &&
    ( $self->{_dbname} = $arg );
  $self->{_dbname};
}

sub driver {
  my ($self, $arg ) = @_;
  ( defined $arg ) &&
    ( $self->{_driver} = $arg );
  $self->{_driver};
}

sub username {
  my ($self, $arg ) = @_;
  ( defined $arg ) &&
    ( $self->{_username} = $arg );
  $self->{_username};
}

sub host {
  my ($self, $arg ) = @_;
  ( defined $arg ) &&
    ( $self->{_host} = $arg );
  $self->{_host};
}

sub password {
  my ($self, $arg ) = @_;
  ( defined $arg ) &&
    ( $self->{_password} = $arg );
  $self->{_password};
}


=head2 get_Feature_Obj

 Title   : get_Feature_Obj
 Usage   :
 Function:
 Example :
 Returns : 
 Args    :


=cut

sub get_Feature_Obj {
    my ($self) = @_;
    
    my $update_obj = Bio::EnsEMBL::DBSQL::Feature_Obj->new($self);
 
    return $update_obj;
}




=head2 get_MetaContainer

 Title   : get_Meta_Container
 Usage   :
 Function:
 Example :
 Returns : 
 Args    :


=cut

sub get_MetaContainer {
    my( $self ) = @_;
    
    my( $mc );
    unless ($mc = $self->{'_meta_container'}) {
        require Bio::EnsEMBL::DBSQL::MetaContainer;
        $mc = Bio::EnsEMBL::DBSQL::MetaContainer->new($self);
        $self->{'_meta_container'} = $mc;
    }
    return $mc;
}




=head2 get_object_by_wildcard

 Title   : get_object_by_wildcard
 Usage   :
 Function:
 Example :
 Returns : 
 Args    :


=cut

sub get_object_by_wildcard{
   my ($self,$type,$string) = @_;

   #print STDERR "Got type: $type and string: $string\n";
   my @ids;
   my $sth = $self->prepare("select id from $type where id like \'$string\'");
   #print STDERR "mysql: select id from $type where id like \'$string\'\n";
   my $res = $sth->execute || $self->throw("Could not get any ids!");
   while( my $rowhash = $sth->fetchrow_hashref('NAME_lc')) {
       push(@ids,$rowhash->{'id'});
   }
   
   if ($type eq 'gene') {
       return $self->gene_Obj->get_array_supporting('without',@ids);
   }
   if ($type eq 'transcript') {
       my @trans;
       foreach my $id (@ids) {
	   push @trans, $self->gene_Obj->get_Transcript($id);
       }
       return @trans;
   }
   if ($type eq 'exon') {
       my @exons;
       foreach my $id (@ids) {
	   push @exons, $self->gene_Obj->get_Exon($id);
       }
       return @exons;
   }
   if ($type eq 'clone') {
       my @clones;
       foreach my $id (@ids) {
	   push @clones, $self->get_Cone($id);
       }
       return @clones;
   }
   else {
       $self->throw("Type $type not supported, only gene, transcript, exon and clone\n");
   }
   return;
}







=head2 prepare

 Title   : prepare
 Usage   : $sth = $dbobj->prepare("select seq_start,seq_end from feature where analysis = \" \" ");
 Function: prepares a SQL statement on the DBI handle

           If the debug level is greater than 10, provides information into the
           DummyStatement object
 Example :
 Returns : A DBI statement handle object
 Args    : a SQL string


=cut

sub prepare {
   my ($self,$string) = @_;

   if( ! $string ) {
       $self->throw("Attempting to prepare an empty SQL query!");
   }
   if( !defined $self->_db_handle ) {
      $self->throw("Database object has lost its database handle! getting otta here!");
   }
      
   if ($self->diffdump) {
       my $fh=$self->diff_fh;
       open (FILE,">>$fh");
       if ($string =~ /insert|delete|replace/i) {
	   print FILE "$string\n";
       }
       
   }
   

   # should we try to verify the string?
   if($self->_debug) {
       print STDERR "preparing <$string> on ",ref($self->_db_handle),"\n";
       #cluck "preparing <$string> on ".ref($self->_db_handle);
       #eval {
       #    cluck "at";
       #};
       #Cluck sometimes craps out, so
       { my $i=0;
	 while(my($package,$filename,$line,$subroutine)=caller($i++)) {
	     print STDERR " >  $subroutine $filename line $line\n";
	 }
       }
   }

   return $self->_db_handle->prepare($string);
}




=head2 add_ExternalFeatureFactory

 Title   : add_ExternalFeatureFactory
 Usage   :
 Function:
 Example :
 Returns : 
 Args    :


=cut

sub add_ExternalFeatureFactory{
   my ($self,$value) = @_;

   unless( ref $value && $value->isa('Bio::EnsEMBL::DB::ExternalFeatureFactoryI') ) {
       $self->throw("[$value] is not a Bio::EnsEMBL::DB::ExternalFeatureFactoryI but it should be!");
   }

   push(@{$self->{'_external_ff'}},$value);
}

=head2 _each_ExternalFeatureFactory

 Title   : _each_ExternalFeatureFactory
 Usage   :
 Function:
 Example :
 Returns : 
 Args    :


=cut

sub _each_ExternalFeatureFactory{
   my ($self) = @_;

   return @{$self->{'_external_ff'}}
}


=head2 _analysis_cache

 Title   : _analysis_cache
 Usage   : $obj->_analysis_cache()
 Function: 
 Returns : reference to a hash
 Args    : newvalue (optional)


=cut

sub _analysis_cache{
   my $obj = shift;
   if( @_ ) {
      my $value = shift;
      $obj->{'_analysis_cache'} = $value;
    }
    return $obj->{'_analysis_cache'};

}




=head2 _debug

 Title   : _debug
 Usage   : $obj->_debug($newval)
 Function: 
 Example : 
 Returns : value of _debug
 Args    : newvalue (optional)


=cut

sub _debug{
    my ($self,$value) = @_;
    if( defined $value) {
	$self->{'_debug'} = $value;
    }
    return $self->{'_debug'};
    
}


=head2 _db_handle

 Title   : _db_handle
 Usage   : $obj->_db_handle($newval)
 Function: 
 Example : 
 Returns : value of _db_handle
 Args    : newvalue (optional)


=cut

sub _db_handle{
   my ($self,$value) = @_;
   if( defined $value) {
      $self->{'_db_handle'} = $value;
    }
    return $self->{'_db_handle'};

}

=head2 _lock_tables

 Title   : _lock_tables
 Usage   :
 Function:
 Example :
 Returns : 
 Args    :


=cut

sub _lock_tables{
   my ($self,@tables) = @_;
   
   my $state;
   foreach my $table ( @tables ) {
       if( $self->{'_lock_table_hash'}->{$table} == 1 ) {
	   $self->warn("$table already locked. Relock request ignored");
       } else {
	   if( $state ) { $state .= ","; } 
	   $state .= "$table write";
	   $self->{'_lock_table_hash'}->{$table} = 1;
       }
   }

   my $sth = $self->prepare("lock tables $state");
   my $rv = $sth->execute();
   $self->throw("Failed to lock tables $state") unless $rv;

}

=head2 _unlock_tables

 Title   : _unlock_tables
 Usage   :
 Function:
 Example :
 Returns : 
 Args    :


=cut

sub _unlock_tables{
   my ($self,@tables) = @_;

   my $sth = $self->prepare("unlock tables");
   my $rv  = $sth->execute();
   $self->throw("Failed to unlock tables") unless $rv;
   %{$self->{'_lock_table_hash'}} = ();
}


=head2 DESTROY

 Title   : DESTROY
 Usage   :
 Function:
 Example :
 Returns : 
 Args    :


=cut

#sub DESTROY {
#   my ($obj) = @_;
#
#   #$obj->_unlock_tables();
#
#   if( $obj->{'_db_handle'} ) {
#       $obj->{'_db_handle'}->disconnect;	#no, since it's parasitic
#       $obj->{'_db_handle'} = undef;
#   }
#}

=head2 get_Clone

 Title   : get_Clone
 Usage   :
 Function: retrieve latest version of a clone from the database
 Example :
 Returns : 
 Args    :


=cut

sub get_Clone { 
    my ($self,$accession) = @_;

    my $ca= Bio::EnsEMBL::DBSQL::CloneAdaptor->new($self);

    return $ca->fetch_by_accession($accession);
}
  
=head2 get_Clone_by_version

 Title   : get_Clone_by_version
 Usage   :
 Function: retrieve specific version of a clone from the database
 Example :
 Returns : 
 Args    :


=cut

sub get_Clone_by_version { 
    my ($self,$accession,$ver) = @_;

    my $ca= Bio::EnsEMBL::DBSQL::CloneAdaptor->new($self);

    return $ca->fetch_by_accession_version($accession,$ver);
}
  

=head2 get_all_Clone_id

 Title   : get_all_Clone_id
 Usage   : @cloneid = $obj->get_all_Clone_id
 Function: returns all the valid (live) Clone ids in the database
 Example :
 Returns : 
 Args    :


=cut

sub get_all_Clone_id{
   my ($self) = @_;
   my @out;

   my $sth = $self->prepare("select id from clone");
   my $res = $sth->execute;

   while( my $rowhash = $sth->fetchrow_hashref('NAME_lc')) {
       push(@out,$rowhash->{'id'});
   }

   return @out;
}



=head2 perl_only_sequences

 Title   : perl_only_sequences
 Usage   : $obj->perl_only_sequences($newval)
 Function: 
 Returns : value of perl_only_sequences
 Args    : newvalue (optional)


=cut

sub perl_only_sequences{
   my $obj = shift;
   if( @_ ) {
      my $value = shift;
      $obj->{'perl_only_sequences'} = $value;
    }
    return $obj->{'perl_only_sequences'};

}

=head2 perl_only_contigs

 Title   : perl_only_contigs
 Usage   : $obj->perl_only_contigs($newval)
 Function: 
 Returns : value of perl_only_contigs
 Args    : newvalue (optional)


=cut

sub perl_only_contigs{
   my $obj = shift;
   if( @_ ) {
      my $value = shift;
      $obj->{'perl_only_contigs'} = $value;
    }
    return $obj->{'perl_only_contigs'};

}



=head2 gene_Obj
    
 Title   : gene_Obj
 Usage   : my $geneobj = $db->gene_Obj
 Function: Returns the gene object database handle
 Example : 
 Returns : Bio::EnsEMBL::DB::Gene_ObjI
 Args    : 

=cut


sub get_FeatureAdaptor {
    my ($self) = @_;

    unless (defined($self->{_featureAdaptor})) {
	$self->{_featureAdaptor} = Bio::EnsEMBL::DBSQL::FeatureAdaptor->new($self);    
    }

    return $self->{_featureAdaptor};
}




=head2 feature_Obj
    
 Title   : feature_Obj
 Usage   : my $featureobj = $db->feature_Obj
 Function: Returns the feature object database handle
 Example : 
 Returns : Bio::EnsEMBL::DB::Feature_ObjI
 Args    : 

=cut

sub feature_Obj {
    my ($self) = @_;

    unless (defined($self->{_feature_obj})) {
	$self->{_feature_obj} = Bio::EnsEMBL::DBSQL::Feature_Obj->new($self);    
    }

    return $self->{_feature_obj};

}

=head2 get_AnalysisAdaptor

 Title   : get_AnalysisAdaptor
 Usage   : $analysisAdaptor = $dbObj->get_AnalysisAdaptor;
 Function: gives the adaptor to fetch/store Analysis objects.
 Example :
 Returns : the adaptor
 Args    :


=cut

sub get_AnalysisAdaptor {
   my ($self) = @_;
   if( ! defined $self->{_analysisAdaptor} ) {
     $self->{_analysisAdaptor} = 
       Bio::EnsEMBL::DBSQL::AnalysisAdaptor->new($self);
   }
   return $self->{_analysisAdaptor};
}


sub get_DBEntryAdaptor {
   my ($self) = @_;
   if( ! defined $self->{_dBEntryAdaptor} ) {
     $self->{_dBEntryAdaptor} = 
       Bio::EnsEMBL::DBSQL::DBEntryAdaptor->new($self);
   }
   return $self->{_dBEntryAdaptor};
}



=head2 get_ChromosomeAdaptor

 Title   : get_ChromosomeAdaptor
 Usage   :
 Function:
 Example :
 Returns : 
 Args    :


=cut

sub get_ChromosomeAdaptor{
   my ($self,@args) = @_;

   return Bio::EnsEMBL::DBSQL::ChromosomeAdaptor->new($self);
}



=head2 find_GenomeHits
    
 Title   : find_GenomeHits
 Usage   : my @features = $self->find_GenomeHits($hid)
 Function: Finds all features in the db that
           are hits to a sequence with id $hid
 Example : 
 Returns : @ Bio::EnsEMBL::FeaturePair
 Args    : string

=cut
 
sub find_GenomeHits {
    my ($self,$arg) = @_;

    return $self->feature_Obj->find_GenomeHits($arg);
}
			     

=head2 deleteObj

    Title   : deleteObj
    Usage   : $dbObj->deleteObj
    Function: Call when you are done with this object. Breaks links between objects. Necessary to clean up memory.
    Example : -
    Returns : -
    Args    : -


=cut

sub deleteObj {

  my  $self=shift;
  my $dummy;

  print STDERR "Destroying DB Obj!\n";       
  $self->DESTROY;
  
  foreach my $name ( keys %{$self} ) {
    eval {
      $dummy = $self->{$name}; 
      $self->{$name}  = undef;
      $dummy->deleteObj;
    };
  }
}



=head2 diff_fh

 Title   : diff_fh
 Usage   : $obj->diff_fh($newval)
 Function: path and name of the file to use for writing the mysql diff dump
 Example : 
 Returns : value of diff_fh
 Args    : newvalue (optional)


=cut

sub diff_fh{
    my ($self,$value) = @_;
    if( defined $value) {
	$self->{'_diff_fh'} = $value;
    }
    return $self->{'_diff_fh'};
    
}


=head2 diffdump

 Title   : diffdump
 Usage   : $obj->diffdump($newval)
 Function: If set to 1 sets $self->_prepare to print the diff sql 
           statementents to the filehandle specified by $self->diff_fh
 Example : 
 Returns : value of diffdump
 Args    : newvalue (optional)


=cut

sub diffdump{
    my ($self,$value) = @_;
    if( defined $value) {
	$self->{'_diffdump'} = $value;
    }
    return $self->{'_diffdump'};
    
}


=head2 extension_tables

 Title   : extension_tables
 Usage   : $obj->extension_tables($newval)
 Function: 
 Returns : value of extension_tables
 Args    : newvalue (optional)


=cut

sub extension_tables{
   my $obj = shift;
   if( @_ ) {
      my $value = shift;
      $obj->{'extension_tables'} = $value;
    }
    return $obj->{'extension_tables'};

}


=head2 _crossdb

 Title   : _crossdb
 Usage   : $obj->_crossdb($newval)
 Function: 
 Returns : value of _crossdb
 Args    : newvalue (optional)


=cut

sub _crossdb {
   my $obj = shift;
   if( @_ ) {
      my $value = shift;
      $obj->{'_crossdb'} = $value;
    }
    return $obj->{'_crossdb'};

}

## internal stuff for external adaptors
=head2 _ext_adaptor

 Title   : _ext_adaptor
 Usage   : $obj->_ext_adaptor('family' [, $famAdaptorObj] )
 Function: 
 Returns : an adaptor or undef
 Args    : a name and a adaptor object. 

=cut

sub _ext_adaptor {
    my ($self, $adtor_name, $adtor_obj) = @_;
    if( $adtor_obj ) {
        if ($adtor_obj eq 'DELETE') { 
            delete $adtor_obj->{'_ext_adaptors'}{$adtor_name};
        } else {
            $self->{'_ext_adaptors'}{$adtor_name} = $adtor_obj;
        }
    }
    return $self->{'_ext_adaptors'}{$adtor_name};
}

## support for external adaptors
=head2 list_ExternalAdaptors

 Title   : list_ExternalAdaptors
 Usage   : $obj->list_ExternalAdaptors
 Function: returns all the names of installed external adaptors
 Returns : a (possibly empty) list of name of external adaptors
 Args    : none

=cut

sub list_ExternalAdaptors {
    my ($self) = @_;
    return keys % {$self->{_ext_adaptors}};
}

=head2 add_ExternalAdaptor

 Title   : add_ExternalAdaptor
 Usage   : $obj->add_ExternalAdaptor('family', $famAdaptorObj);
 Function: adds the external adaptor the internal hash of known 
           external adaptors. If an adaptor of the same name is installed, 
           it will be overwritten.
 Returns : undef
 Args    : a name and a adaptor object. 

=cut

sub add_ExternalAdaptor {
    my ($self, $adtor_name, $adtor_obj) = @_;
    $self->_ext_adaptor($adtor_name, $adtor_obj);
    undef;
}

=head2 get_ExternalAdaptor

 Title   : get_ExternalAdaptor
 Usage   : $obj->get_ExternalAdaptor('family');
 Function: retrieve external adaptor by name
 Returns : an adaptor (sub-type of BaseAdaptor) or undef
 Args    : the name 

=cut

sub get_ExternalAdaptor {
    my ($self, $adtor_name) = @_;
    $self->_ext_adaptor($adtor_name);
}


=head2 remove_ExternalAdaptor

 Title   : remove_ExternalAdaptor
 Usage   : $obj->remove_ExternalAdaptor('family')
 Function: removes the named external adaptor from the internal hash of known 
           external adaptors. If the adaptor name is not known, nothing 
           happens. 
 Returns : undef
 Args    : a name

=cut

sub remove_ExternalAdaptor {
    my ($self, $adtor_name) = @_;
    $self->_ext_adaptor($adtor_name, 'DELETE');
    undef;
}

=head2 AUTOLOAD

 Assume misc functions are meant to use the database handle
 Example : selectall_arrayref


=cut

sub AUTOLOAD {
   my ($self,@args) = @_;

   #print STDERR __PACKAGE__," AUTOLOAD ",$args[0],"\n";
   if( !defined $self->_db_handle ) {
      $self->throw("Database object has lost its database handle! getting otta here!");
   }

   if($args[0] && ! ref $args[0] ) {
       print STDERR "to handle: $AUTOLOAD($args[0],...) on ",ref($self->_db_handle),"\n";
       { my $i=0;
	 while(my($package,$filename,$line,$subroutine)=caller($i++)) {
	     print STDERR " >  $subroutine $filename line $line\n";
	 }
       }
   }
   (my $sub=$AUTOLOAD) =~ s/.*:://;
   #print STDERR "$sub (",join(',',@args),")\n";

   $self->_db_handle->can($sub) or $self->throw("$AUTOLOAD does not exist and the database handle can't $sub either!");

   return $self->_db_handle->$sub(@args);
}



=head2 get_Ensembl_SeqFeatures_clone

 Title   : get_Ensembl_SeqFeatures_clone
 Usage   : get_Ensembl_SeqFeatures_clone($accession_number,
                                          $sequence_version,$start,$stop);


=cut

sub get_Ensembl_SeqFeatures_clone{
   my ($self,$clone,$version,$start,$stop)  = @_;

   if (!defined($clone)) {
       $self->throw("No clone entered for get_Ensembl_SeqFeatures_clone");
   }
   if (!defined($version)) {
       $self->throw("No version entered for get_Ensembl_SeqFeatures_clone");
   }
   if (defined $start) {
        $start = 1 if $start eq "";
        if ( $start !~ /^\d+$/  and $start > 0) {
            $self->throw("$start is not a valid start");
        }
    }
    if (defined $stop) {
        $start = 1 if not defined $start;
        if ( $stop !~ /^\d+$/ and $stop > 0 ) {
            $self->throw("$stop is not a valid stop");
        }
    }
    if (defined $start and defined $stop) {
        if ($stop < $start) {
            $self->throw("$stop is smaller than $start not a valid start");
        }
    }

   my @array;

   my $statement = "SELECT id, clone_start, clone_end, strand, score, analysis, program_name, marker_start, marker_end, marker, evalue, perc_id, marker_accession " . 
           "FROM marker_clone " .
           "WHERE clone='$clone' " .
	   "AND clone_version=$version " .
	   ( defined($stop) ? "AND clone_start<=$stop " : "" ).
	   ( defined($start) ? "AND clone_end>=$start" : "" );

   my $sth = $self->_db_handle->prepare($statement);
   my $res = $sth->execute;

   my %analhash;

   # bind the columns
   my ($fid,$end,$strand,$f_score,$analysisid,$name,$hstart,$hend,$hid,$evalue,$perc_id,$marker_acc);
   $sth->bind_columns(undef,\$fid,\$start,\$end,\$strand,\$f_score,\$analysisid,\$name,\$hstart,\$hend,\$hid,\$evalue,\$perc_id,\$marker_acc);
   
   while($sth->fetch) {

       my $out;
       my $analysis;
              
       if (!$analhash{$analysisid}) {
	   
	   my $feature_obj=Bio::EnsEMBL::DBSQL::Feature_Obj->new($self->_db_handle);
	   $analysis = $feature_obj->get_Analysis($analysisid);
	   $analhash{$analysisid} = $analysis;
	   
       } else {
	   $analysis = $analhash{$analysisid};
       }
       #print STDERR "A $analysisid,";
       #print STDERR " db=",$analysis->db if $analysis->db;
       #print STDERR " db=",$analysis->db if $analysis->db;
       #print STDERR " dbID=",$analysis->dbID if $analysis->dbID;
       #print STDERR "\n";
       
       if( !defined $name ) {
	   $name = 'no_source';
       }
       
       if( $hid ne '__NONE__' ) {
	   # is a paired feature
	   # build EnsEMBL features and make the FeaturePair
	 
	   $out = Bio::EnsEMBL::FeatureFactory->new_feature_pair();


	   $out->set_all_fields($start,$end,$strand,$f_score,$name,'similarity',$clone,
				$hstart,$hend,1,$f_score,$name,'similarity',$marker_acc||$hid);

	   $out->analysis    ($analysis);
	   $out->id          ($hid);              # MC This is for Arek - but I don't
	                                          #    really know where this method has come from.
       } else {
	   $out = new Bio::EnsEMBL::SeqFeature;
	   $out->seqname    ($clone);
	   $out->start      ($start);
	   $out->end        ($end);
	   $out->strand     ($strand);
	   $out->source_tag ($name);
	   $out->primary_tag('similarity');
	   $out->id         ($fid);
	   $out->p_value    ($evalue)    if (defined $evalue);
	   $out->percent_id ($perc_id)   if (defined $perc_id); 

	   if( defined $f_score ) {
	       $out->score($f_score);
	   }
	   $out->analysis($analysis);
       }
       # Final check that everything is ok.
       $out->validate();
       
      push(@array,$out);
      
   }
   
   return @array;


}

=head2 get_Ensembl_SeqFeatures_contig

 Title   : get_Ensembl_SeqFeatures_contig (Abstract)
 Usage   :
 Function:
 Example :
 Returns : 
 Args    :


=cut

sub get_Ensembl_SeqFeatures_contig{
   my ($self) = @_;
   
   return ();
}

=head2 get_Ensembl_Genes_clone

 Title   : get_Ensembl_Genes_clone
 Function: returns Gene objects in clone coordinates from a gene id
 Returns : An array of Gene objects
 Args    : clone id

=cut

sub get_Ensembl_Genes_clone {
    my $self = @_;

   return ();
}

=head2 get_SeqFeature_by_id

 Title   : get_SeqFeature_by_id (Abstract)
 Usage   : 
 Function: Return SeqFeature object for any valid unique id  
 Example :
 Returns : 
 Args    : id as determined by the External Database


=cut

       
sub get_SeqFeature_by_id {
   my ($self,$id) = @_;

   $self->throw("get_SeqFeature_by_id not implmented for GMAP database");

}



=head2 get_SeqFeatures_by_Marker

 Title   : get_SeqFeature_by_Marker
 Usage   : @feature_array=get_SeqFeature_by_Marker($marker_name);


=cut

sub get_SeqFeatures_by_Marker{
   my ($self,$marker)  = @_;

   if (!defined($marker)) {
       $self->throw("No marker entered for get_SeqFeature_by_Marker");
   }

   my @array;

   my $statement = "SELECT id, clone, clone_start, clone_end, strand, score, analysis, program_name, evalue, perc_id, marker_start, marker_end, marker_accession " . 
           "FROM marker_clone " .
           "WHERE marker='$marker' " ;

   #my $sth = $self->_db_handle->prepare($statement);
   my $sth = $self->prepare($statement);
   my $res = $sth->execute;

   my %analhash;

   # bind the columns
   my ($fid,$clone,$start,$end,$strand,$f_score,$analysisid,$name,$evalue,$perc_id,$hstart,$hend,$marker_acc);
   $sth->bind_columns(undef,\$fid,\$clone,\$start,\$end,\$strand,\$f_score,\$analysisid,\$name,\$evalue,\$perc_id,\$hstart,\$hend,\$marker_acc);
   
   while($sth->fetch) {

       my $out;
       my $analysis;
              
       if (!$analhash{$analysisid}) {
	   
	   my $feature_obj=Bio::EnsEMBL::DBSQL::Feature_Obj->new($self->_db_handle);
	   $analysis = $feature_obj->get_Analysis($analysisid);
	   $analhash{$analysisid} = $analysis;
	   
       } else {
	   $analysis = $analhash{$analysisid};
       }
       #print STDERR "A $analysisid,";
       #print STDERR " db=",$analysis->db if $analysis->db;
       #print STDERR " db=",$analysis->db if $analysis->db;
       #print STDERR " dbID=",$analysis->dbID if $analysis->dbID;
       #print STDERR "\n";
       
       # is a paired feature
       # build EnsEMBL features and make the FeaturePair
     
       $out = Bio::EnsEMBL::FeatureFactory->new_feature_pair();


       $out->set_all_fields($start,$end,$strand,$f_score,$name,'similarity',
       		$clone,$hstart,$hend,1,$f_score,$name,'similarity',$marker_acc||$marker);

       $out->analysis    ($analysis);
       $out->id          ($marker);              

       $out->p_value    ($evalue)    if (defined $evalue);
       $out->percent_id ($perc_id)   if (defined $perc_id); 

       
       # Final check that everything is ok.
       $out->validate();
       
      push(@array,$out);
      
   }
   
   return @array;


}

sub cmap_links_by_marker {
   my ($self,$marker)  = @_;
   my $statement = "SELECT feature_id FROM cmap.feature WHERE feature_name=?";
   my $sth = $self->prepare($statement) or print STDERR "$statement:$self->errstr\n" and return ();
   $sth->execute($marker) or print STDERR "$statement:$sth->errstr\n" and return ();
   my @links;
   while (my($mid)=$sth->fetchrow_array()) {
       #print STDERR "$marker to $mid\n";
       push @links,qq{<a href="/maps/marker?marker_id=$mid">Maps</a>};
   }
   @links;
}

1;



