package CSHL::mutant::MutantFormMailer;

######################################################
# File       : MutantFormMailer.pm
# Programmer : Kuan Y. Chang
# Created    : 2001/12/08
# Purpose    : mail form contents
######################################################

=head1 NAME

CSHL::mutant::MutantFormMailer.pm - mail form contents

=head1 SYNOPSIS

  <Location /foo>
    SetHandler  perl-script
    PerlHandler CSHL::mutant::MutantFormMailer
  </Location>

=head1 DESCRIPTION

This module is a mod_perl handler responsible for validating
and e-mailing the contents of the "Mutant Submission Form" form on 
the Gramene Website.  If the data is successfully
sent, then the user is shown a page confirming his submission and is
given the option to submit again with his basic contact information
(name, organization, address, etc.) already seeded.

=head1 AUTHOR

Kuan Y. Chang

=head1 SEE ALSO

perl(1).

=cut

use strict;
use vars qw( $VERSION );
$VERSION = (qw$Revision: 1.2 $)[-1];

use Apache::Constants;
use Apache::File;
use Apache::Reload;  # comment this out on development servers!
use Apache::Request;
use Email::Valid;
use Template;

use GramenePage;

use constant TEMPLATE_DIR       => '/usr/local/gramene/templates/mutant';
use constant CONFIRM_TEMPLATE   => 'confirm_mutant.tmpl';
use constant FORM_TEMPLATE      => 'submit_mutant.tmpl';
use constant EMAIL_FROM         => 'mutant_submission@gramene.org';
use constant EMAIL_SUBJECT      => 'Mutant Submission from gramene.org';
use constant OTHER_FIELD_SUFFIX => '_other';
use constant SENDMAIL           => '/usr/bin/sendmail';
use constant SENDMAIL_FLAGS     => ' -t -oi';

my %DISPATCH = (
    default  => \&show_form,
    submit   => \&process_form,
    resubmit => \&show_form,
);

my @EMAIL_RECIPIENTS = (
"Junjian Ni <jn66\@cornell.edu>",
"Kuan Y. Chang <changku\@cshl.org>",
);

my %FIELDS = (
    mutant_name             => { required => 1 },
    phenotypic_description  => { required => 1 },
    other_description       => { },
    reference_category      => { },
    reference_curator       => { },
    reference_source        => { },
    references              => { required => 1},
    map_study               => { },
    chromosome_number       => { },
    chromosome_location     => { },
    markers                 => { },
    first_name              => { required => 1 },
    middle_initial          => { },
    last_name               => { required => 1 },
    organization            => { required => 1 },
    address                 => { required => 1 },
    city                    => { required => 1 },
    state                   => { },
    postal_code             => { },
    country                 => { default => 'USA' },
    telephone               => { },
    fax                     => { },
    email_address           => { required => 1, is_email_address => 1 },
    lab_name                => { },
    lab_email               => { },
);

#-----------------------------------------------------
#
# Main entry for all subroutines; also, notice that 
# the "eval" section catches and handles all errors 
# thrown by "die" statements in called subroutines
#
sub handler {
    my $r      = shift;
    my $apr    = Apache::Request->new( $r->is_main ? $r : $r->main );
    my $return;
    
    eval {
        my $action = $apr->param( 'action' ) || '';
        $action    = 'default' unless exists $DISPATCH{ $action };
        $return    = $DISPATCH{ $action }->( $apr );
    };

    return ( $@ ) ? handle_error( $apr, $@ ) : $return || OK;
}

#-----------------------------------------------------
#
# Handle errors
#
sub handle_error {
    my ( $apr, $error ) = @_;

    unless ( $apr->status_line ) {
        $apr->content_type('text/html');
        $apr->send_http_header;
    }

    $apr->print( $error );
    return OK;
} 

#-----------------------------------------------------
#
# Mail form contents to recipients;
# Show user the result, allow them to resubmit.
#
sub process_form {
    my $apr     = shift;
    my @fields  = keys %FIELDS;

    #
    # Data Validation
    #
    my @errors = ();
    for my $field_name ( @fields ) {
        my $field = $FIELDS{ $field_name };
        #
        # If given no value, check the "other" field 
        # If nothing there, then use the default value (if any)
        #
        unless ( $apr->param( $field_name ) ) {{
            if ( $field->{'allows_other'} ) {
                my $other_field_name = $field_name.OTHER_FIELD_SUFFIX;
                my $other = $apr->param( $other_field_name ) || '';
                $apr->param( $field_name, $other ), last if $other; 
            } 

            my $default = $field->{'default'} || '';
            $apr->param( $field_name, $default ) if $default; 
        }}

        if ( $field->{'is_email_address'} ) {
            my $address = $apr->param( $field_name ) || '';
            push @errors, "'$address' is not a valid e-mail address"
                unless Email::Valid->address( $address );
        }

        next unless $field->{'required'};
        next if $apr->param( $field_name );

        #
        # At this point, we have an error because:
        # 1) There was no value given.
        # 2) There was no "other" field where the user could
        #    type a value other than one supplied by the form.
        # 3) There was no default value specified in the code.
        # 4) The field is required.
        #
        my $value   = $apr->param($field_name) || '';
        $field_name = join ' ', map { ucfirst } split( /_/, $field_name );
        push @errors, "$field_name ($value) is a required field";
    }
    return show_form( $apr, \@errors ) if @errors;

    #
    # Send the message if everything looks good.
    #
    my $message = join( "\t", @fields ) . "\n";
    $message   .= join( "\t", 
                      map{ s/\"/\\\"/g; "\"$_\"" } # escape quotes
                      map { $apr->param( $_ ) } @fields 
                  ) . "\n";

    die "sendmail binary not present"    unless -e SENDMAIL;
    die "sendmail binary not executable" unless -x _;
    for my $address ( @EMAIL_RECIPIENTS ) {
        my $fh = Apache::File->new( '|'.SENDMAIL.SENDMAIL_FLAGS ) 
                 or die "Can't open pipe to sendmail (".SENDMAIL."): $!";
        print $fh 'Subject: ', EMAIL_SUBJECT, "\n";
        print $fh "To: $address\n";
        print $fh 'From: ', EMAIL_FROM, "\n";
        print $fh "Content-type: text/plain\n\n";
        print $fh $message;
        $fh->close or die "Can't close sendmail pipe: $!";
    }

    my $page = GramenePage->new( $apr ) or die 
               "Can't open Gramene page configurator";

    my $t = Template->new({
        INCLUDE_PATH => TEMPLATE_DIR,
        OUTPUT       => $apr,
    });

    $apr->content_type('text/html');
    $apr->send_http_header;
    $t->process( 
        CONFIRM_TEMPLATE, 
        { 
            apr         => $apr, 
            message     => $message,
            GramenePage => $page,
        } 
    ) or die $t->error;
    return OK;
} 

#-----------------------------------------------------
# 
# Show the form; uses GramenePage for CSS, navigation, etc.
# 
sub show_form {
    my ( $apr, $errors ) = @_;
    my $page = GramenePage->new( $apr ) or die 
               "Can't open Gramene page configurator";

    my $t = Template->new({
        INCLUDE_PATH => TEMPLATE_DIR,
        OUTPUT       => $apr,
    });

    $apr->content_type('text/html');
    $apr->send_http_header;
    $t->process( 
        FORM_TEMPLATE, 
        { 
            apr         => $apr, 
            errors      => $errors,
            GramenePage => $page,
        } 
    ) or die $t->error;
    return OK;
}

1;

#-----------------------------------------------------
# If the river was whiskey I would've dived in there,
# Dive to the bottom, never would I come up.
# Muddy Waters
#-----------------------------------------------------
