package CSHL::SOAP::ComparativeMapData;

#-----------------------------------------------------
# $Id: ComparativeMapData.pm,v 1.33 2002/04/17 02:13:16 kclark Exp $
#
# File       : MarkerData.pm
# Programmer : Ken Y. Clark, kclark@logsoft.com
# Created    : 2001/08/07
# Purpose    : retrieve marker data
#-----------------------------------------------------

use strict;
use vars qw( $VERSION );
$VERSION = (qw$Revision: 1.33 $)[-1];

use CSHL::Config;
use CSHL::AppConfig qw[ :argcount ];
use CSHL::ComparativeMaps::Constants;
use Data::Dumper;

#-----------------------------------------------------
#
# Instantiates a new object (not needed if actually used with SOAP).
#
sub new {
    my $class = shift;
    my %args  = @_;
    my $self  = { %args };
    return bless $self, $class;
}

#-----------------------------------------------------
#
# Creates a database handle, stores it within the object.
#
sub db { 
    my $self = shift;
    unless ( defined $self->{'db'} ) {
        $self->{'db'} = DBI->connect( 
            MapDataSource,
            MapDBUser,
            MapDBPassword,
            MapDBOptions,
        );

#        my $config = CSHL::AppConfig->new;
#        if ( my $apr = $self->{'apr'} ) { 
##            warn "conf = ", $apr->server_root_relative( CONF_FILE ), "\n";
#            $config->file( 
#                $apr->server_root_relative( CONF_FILE )
#            );
#        }
#        $config->define('DBOptions', { ARGCOUNT => ARGCOUNT_HASH });
#        $self->{'db'} = DBI->connect( 
#            $config->get('MapDataSource'),
#            $config->get('MapDBUser'),
#            $config->get('MapDBPassword'),
#            $config->get('DBOptions'),
#        );
    }
    return $self->{'db'};
}

#-----------------------------------------------------
# Private subs
#-----------------------------------------------------
sub _extract_numbers {
#
# Returns just the numbers in a string
#
    my $arg = shift;
    $arg =~ s/[^\d]//g;
    return $arg;
}

#-----------------------------------------------------
sub _commify {
#
# Turns "12345" into "12,345"
#
    my $arg = shift;
    1 while $arg =~ s/^(-?\d+)(\d{3})/$1,$2/;
    return $arg;
}

#-----------------------------------------------------
# Public methods
#-----------------------------------------------------
sub marker_search {
#
# Used by the MarkerSearch module.  
# Expects:
#   markers:  A delimited string containing feature or 
#             locus names.  Delimiters can be just 
#             about anything.
#   order by: The output field by which to sort the results
#
# Returns:
#   An arrayref of hashrefs showing each feature's name and 
#   the locus name (map_position_name) at the position on the map
#   that feature appears.   Also included is the map study info
#   (species, type, etc.).
#
    my ( $self, %args ) = @_;
    my $db              = $self->db;
    my $markers         = $args{'markers'} or return;
    my $string          = join( ', ', 
        map { qq['\U$_'] } split( /[,:;\s+]/, $markers ) 
    );

    my $order_by = $args{'order_by'} || 
        'feature_name, map_study_name, linkage_group, position_start';

    my @feature_ids = @{ $db->selectcol_arrayref(
        qq[
            select distinct f.feature_id
            from   cmap_feature f, 
                   cmap_feature_type ft,
                   cmap_genetic_map_position mp, 
                   cmap_genetic_map map
            where  (
                       upper(f.feature_name)       in ($string)
                       or 
                       upper(mp.map_position_name) in ($string)
                   )
            and    f.feature_type_id=ft.feature_type_id
            and    upper(ft.feature_type)=?
            and    f.feature_id=mp.feature_id
            and    mp.genetic_map_id=map.genetic_map_id
        ],
        {},
        ( MARKER )
    ) };

    return unless @feature_ids; 

    #
    # Add to the feature ids all the corresponding feature ids.
    #
    push @feature_ids, @{
        $db->selectcol_arrayref(
            q[
                select distinct feature_id2
                from   cmap_feature_correspondence
                where  feature_id1 in (].
                       join( ',', @feature_ids ).q[)
            ]
        )
    };

    return $db->selectall_arrayref(
        q[
            select   f.feature_name, mp.map_position_name, mp.position_start,
                     map.genetic_map_id, map.linkage_group, map.map_study_id,
                     ms.map_study_id, ms.short_name as map_study_name,
                     s.common_name as species_common_name
            from     cmap_feature f, 
                     cmap_genetic_map_position mp, 
                     cmap_genetic_map map,
                     cmap_map_study ms, 
                     cmap_species s
            where    f.feature_id in (].
            join( ',', @feature_ids ).qq[)
            and      f.feature_id=mp.feature_id
            and      mp.genetic_map_id=map.genetic_map_id
            and      map.map_study_id=ms.map_study_id
            and      ms.species_id=s.species_id
            order by $order_by
        ], { Columns => {} }
    );
}

#-----------------------------------------------------
sub marker_position_by_name {
#
# Given a feature or map position (locus) name and a map id, 
# returns the marker's position on the map.  Useful when 
# given start and stop coordinates for the reference map 
# with marker or locus names.  Returns undef if not found.
#
    my ( $self, %args ) = @_;
    my $map_id          = $args{'map_id'};
    my $marker_name     = uc $args{'marker_name'};
    my $db              = $self->db;

    #
    # First, get the map type (physical, genetic, etc.)
    #
    my $map_type = 
        $db->selectrow_array(
            q[
                select count(genetic_map_id)
                from   cmap_genetic_map
                where  genetic_map_id=?
            ],
            {},
            ( $map_id )
        )
        ? GENETIC
        : PHYSICAL
    ;

    my $position;
    if ( $map_type eq GENETIC ) {
        $position = $db->selectrow_array(
            q[
                select mp.position_start
                from   cmap_genetic_map_position mp, 
                       cmap_feature f
                where  mp.feature_id=f.feature_id
                and    mp.genetic_map_id=?
                and    upper(f.feature_name)=?
            ], {}, ( $map_id, "$marker_name" ) 
        );

        unless ( defined $position ) {
            $position = $db->selectrow_array(
                q[
                    select position_start
                    from   cmap_genetic_map_position
                    where  genetic_map_id=?
                    and    upper(map_position_name)=?
                ], {}, ( $map_id, "$marker_name" ) 
            );
        }
    }
    else {
        $position = $db->selectrow_array(
            q[
                select mp.position_start
                from   cmap_physical_map_position mp, 
                       cmap_feature f
                where  mp.feature_id=f.feature_id
                and    mp.physical_map_id=?
                and    upper(f.feature_name)=?
            ], {}, ( $map_id, "$marker_name" ) 
        );

        unless ( defined $position ) {
            $position = $db->selectrow_array(
                q[
                    select position_start
                    from   cmap_physical_map_position
                    where  physical_map_id=?
                    and    upper(map_position_name)=?
                ], {}, ( $map_id, "$marker_name" ) 
            );
        }
    }

    return $position;
}

#-----------------------------------------------------
sub map_viewer_form_data {
#
# Called by MapViewer.  Returns the data used on the HTML form 
# to select and view comparative maps.
#
# Expects:
#   reference_map_study_id
#   reference_map_id      
#   start (of the reference map)
#   end   (of the reference map)
#   comparative_maps (an arrayref of "map_study_id" and "map_id")
#
# Returns:
#   A big hashref ($form_data) that has lots o' goodies about
#   the reference map and all it's comparative maps.
#

    my ( $self, %args ) = @_;
#    warn "args =\n", Dumper( %args ), "\n";
    my $db = $self->db;
    my $form_data;

    #
    # See if there's a map_study_id.  If not, get it for
    # whatever map_id is given (if any).  The reference
    # map can only be a genetic map.
    #
    if ( $args{'reference_map_id'} && !$args{'reference_map_study_id'} ) {
        $args{'reference_map_study_id'} = $db->selectrow_array(
            q[
                select map_study_id
                from   cmap_genetic_map
                where  genetic_map_id=?
            ],
            {}, ( $args{'reference_map_id'} )
        );
    }
    $form_data->{'reference_map_study_id'} = $args{'reference_map_study_id'};
    $form_data->{'reference_map_id'}       = $args{'reference_map_id'};

    #
    # Go get all the map studies
    #
    my $map_studies = $db->selectall_arrayref(
        q[
            select   ms.map_study_id, ms.short_name as map_study_name,
                     s.species_id, s.common_name as species_name,
                     mt.map_type
            from     cmap_map_study ms, 
                     cmap_species s, 
                     cmap_map_type mt
            where    ms.species_id=s.species_id
            and      ms.map_type_id=mt.map_type_id
            order by species_name, map_study_name
        ],
        { Columns => {} }
    );

    # 
    # Reference Map Studies
    # Exclude all the physical maps for the reference map studies
    #
    $form_data->{'reference_map_studies'} = [
        grep { uc $_->{'map_type'} eq GENETIC }
        @$map_studies
    ];

    # 
    # Reference Maps
    #
    if ( $args{'reference_map_study_id'} ) {
        my @maps = @{ $db->selectall_arrayref(
            q[
                select   genetic_map_id, linkage_group
                from     cmap_genetic_map
                where    map_study_id=?
                order by linkage_group
            ],
            { Columns => {} }, ( $args{'reference_map_study_id'} )
        ) };

        my $all_numbers = grep { $_->{'linkage_group'} =~ m/^[0-9]/ } @maps;
        if ( $all_numbers == scalar @maps ) {
            @maps = 
                map  { $_->[0] }
                sort { $a->[1] <=> $b->[1] }
                map  { [$_, _extract_numbers( $_->{'linkage_group'} )] }
                @maps
            ;
        }

        $form_data->{'reference_maps'} = \@maps;
        $form_data->{'map_study_id1'}  = $args{'reference_map_study_id'};

        for my $map_study ( @$map_studies ) {
            if ($map_study->{'map_study_id'}==$args{'reference_map_study_id'}){
                $form_data->{'map_title1'} = $map_study->{'map_study_name'},
                last;
            }
        }

        for my $map ( @maps ) {
            if ( $map->{'genetic_map_id'} == $args{'reference_map_id'} ) {
                $form_data->{'map_title1'} .= '-'.$map->{'linkage_group'},
                last;
            }
        }
    }

    #
    # Comparative Maps
    #
    my $map_region_lookup;
    if ( $args{'reference_map_id'} ) {
        my $data = $self->get_comparative_maps(
            start                 => $args{'start'},
            end                   => $args{'end'},
            reference_map_id      => $args{'reference_map_id'},
            include_physical_maps => 1,
        );

        $form_data->{'comparative_map_studies'} = [@{ $data->{'maps'} || [] }];
        $form_data->{'start'}                   = $data->{'start'};
        $form_data->{'end'}                     = $data->{'end'};
        $map_region_lookup                      = $data->{'map_region_lookup'};
    }

    #
    # More comparative maps
    #
    my @comparative_maps = @{ $args{'comparative_maps'} || [] };
    my $map_no           = 2;
#    warn "SOAP: form_data comparative maps =\n", Dumper(@comparative_maps),"\n";
    for my $i ( 0 .. $#comparative_maps ) {
        my $map            = $comparative_maps[ $i ];
        my $map_study_id   = $map->{'map_study_id'};
        my $genetic_map_id = $map->{'genetic_map_id'};

        unless ( $map_study_id ) {
            $map_study_id = $db->selectrow_array(
                q[
                    select map_study_id
                    from   cmap_genetic_map
                    where  genetic_map_id=?
                ],
                {}, ( $genetic_map_id ) 
            );
        }

#        warn "comp map id = $map_id, start = ", 
#            $map_region_lookup->{ $map_id }{'start'}, ", ",
#            "end = ", $map_region_lookup->{ $map_id }{'end'}, "\n";

        my $data = $self->get_comparative_maps( 
            reference_map_id => $genetic_map_id,
            start            => $map_region_lookup->{$genetic_map_id}{'start'},
            end              => $map_region_lookup->{$genetic_map_id}{'end'},
            exclude_map_ids  => [ $args{'reference_map_id'} ],
            include_physical_maps => 0,
        );

        $form_data->{"comparative_maps$map_no"} = [@{ $data->{'maps'} || [] }];

        $form_data->{"map_study_id$map_no"}  = $map_study_id;

        for my $map_study ( @$map_studies ) {
            if ( $map_study->{'map_study_id'} == $map_study_id ){
                $form_data->{"map_title$map_no"} = 
                    $map_study->{'map_study_name'};
                last;
            }
        }

        $map_no++;
    }

    return $form_data; 
}

#-----------------------------------------------------
sub get_comparative_maps {
#
# Finds all the maps related to a given map.
#
# Expects:
#   reference_map_id
#   exclude_map_ids (an arrayref of map ids to exclude in the search)
#   include_physical_maps ("True/False" values determining whether 
#       or not to include physical maps in the search)
#
#
#
#
    my ( $self, %args )       = @_;
#    warn "get_comparative_maps() args =\n", Dumper( %args ), "\n";
    my $db                    = $args{'db'} || $self->db;
    my $reference_map_id      = $args{'reference_map_id'} or return;
    my @exclude_map_ids       = @{ $args{'exclude_map_ids'} || [] };
    my $include_physical_maps = defined $args{'include_physical_maps'} 
        ? $args{'include_physical_maps'} : 1
    ;

    #
    # To find the region we're looking, first see if the coordinates
    # are given in marker or locus name.  If so, find their positions.
    #
    for my $position ( qw[ start end ] ) {
        next unless defined $args{ $position };
        next unless $args{ $position } =~ m/[a-zA-Z]/;
        $args{ $position } = $self->marker_position_by_name( 
            map_id      => $args{'reference_map_id'},
            marker_name => $args{ $position },
        ); 
    } 

    #
    # Find the extreme ends of the map.
    #
    my ( $map_begin, $map_end ) = $db->selectrow_array(
        q[
            select min(position_start), max(position_start)
            from   cmap_genetic_map_position
            where  genetic_map_id=?
        ],
        {}, ( $args{'reference_map_id'} )
    );
        
    #
    # Make sure that what we have for the start and stop are valid
    # positions.  If not, set them to the beginning and end of the
    # map.  (This means that, if no positions are given, the default
    # region is the entire reference map.)
    #
    my $start = $args{'start'} =~ FLOAT_REGEX ? $args{'start'} : $map_begin;
    my $end   = $args{'end'}   =~ FLOAT_REGEX ? $args{'end'}   : $map_end  ;
       $end   = $map_end if $end > $map_end;

    #
    # Flip the start and end if they need to be.
    #
    ( $start, $end ) = ( $end, $start ) if $start > $end;

    #
    # Find all the features on the reference map in the given region.
    # Keep expanding the search region until we find some features.
    #
    my @feature_ids = ();
    while ( !@feature_ids ) {
        @feature_ids = @{
            $db->selectcol_arrayref(
                q[
                    select distinct feature_id
                    from   cmap_genetic_map_position
                    where  genetic_map_id=?
                    and    position_start>=? 
                    and    position_start<=? 
                ],
                {}, ( $args{'reference_map_id'}, $start, $end )
            )
        };
        last if @feature_ids;

        $start = $start - 1 > $map_begin ? $start - 1 : $map_begin;
        $end   = $end   + 1 < $map_end   ? $end   + 1 : $map_end;
    }

    #
    # Find all other maps on which occur any of our found features or
    # their correspondences.
    #
    my ( @comparative_map_studies, %map_region_lookup );
    if ( @feature_ids ) {
        #
        # Add to the feature ids all the corresponding feature ids.
        #
        push @feature_ids, @{
            $db->selectcol_arrayref(
                q[
                    select distinct feature_id2
                    from   cmap_feature_correspondence
                    where  feature_id1 in (].
                           join( ',', @feature_ids ).q[)
                ]
            )
        };

        my $genetic_comparative_maps = $db->selectall_arrayref(
            q[
                select   mp.genetic_map_id, 
                         min(mp.position_start) as map_start,
                         max(mp.position_start) as map_end,
                         map.linkage_group, 
                         ms.map_study_id, ms.short_name as map_study_name,
                         s.common_name as species_name,
                         mt.map_type
                from     cmap_genetic_map_position mp, 
                         cmap_genetic_map map, 
                         cmap_map_study ms, 
                         cmap_species s, 
                         cmap_map_type mt
                where    mp.feature_id in (].
                         join( ',', @feature_ids ).q[)
                and      mp.genetic_map_id not in (].
                         join( ',', @exclude_map_ids, $reference_map_id ).q[)
                and      mp.genetic_map_id=map.genetic_map_id
                and      map.map_study_id=ms.map_study_id
                and      ms.species_id=s.species_id
                and      ms.map_type_id=mt.map_type_id
                group by mp.genetic_map_id, map.linkage_group, ms.map_study_id,
                         ms.short_name, s.common_name, mt.map_type
                order by linkage_group
            ],
            { Columns => {} } 
        );

        my $physical_comparative_maps = 
            $include_physical_maps
            ? $db->selectall_arrayref(
                q[
                    select   distinct 
                             ms.map_study_id, ms.short_name as map_study_name,
                             s.common_name as species_name,
                             mt.map_type
                    from     cmap_physical_map_position mp, 
                             cmap_physical_map map, 
                             cmap_map_study ms, 
                             cmap_species s, 
                             cmap_map_type mt
                    where    mp.feature_id in (].
                             join( ',', @feature_ids ).q[)
                    and      mp.physical_map_id=map.physical_map_id
                    and      map.map_study_id=ms.map_study_id
                    and      ms.species_id=s.species_id
                    and      ms.map_type_id=mt.map_type_id
                    order by map_study_name
                ],
                { Columns => {} } 
            )
            : undef
        ;

        #
        # Arrange the data for display
        #
        my %sorted_maps;
        for my $map ( 
            @$genetic_comparative_maps, @$physical_comparative_maps 
        ) {
            $map_region_lookup{ $map->{'genetic_map_id'} } = {
                start => $map->{'map_start'},
                end   => $map->{'map_end'},
            };

            next unless uc $map->{'map_type'} eq GENETIC;

            push @{ 
                $sorted_maps { 
                    join(':',
                        $map->{'species_name'},
                        $map->{'map_study_name'},
                        $map->{'map_study_id'},
                    )
                } 
            }, $map;
        }

        my @genetic;
        for my $map_study ( sort keys %sorted_maps ) {
            my ( $species_name, $map_study_name, $map_study_id ) = 
                split( /:/, $map_study );

            #
            # Skip physical maps because we don't want to show
            # individual contigs as map selections.
            #
            my @maps;
            for my $hr ( @{ $sorted_maps{ $map_study } } ) {
                push @maps, {
                    linkage_group  => $hr->{'linkage_group'},
                    genetic_map_id => $hr->{'genetic_map_id'},
                };
            }

            my $all_numbers = grep {$_->{'linkage_group'}=~m/^[0-9]/} @maps;
            if ( $all_numbers == scalar @maps ) {
                @maps = 
                    map  { $_->[0] }
                    sort { $a->[1] <=> $b->[1] }
                    map  { [$_, _extract_numbers( $_->{'linkage_group'} )] }
                    @maps
                ;
            }

            push @genetic, {
                species_name   => $species_name,
                map_study_name => $map_study_name,
                map_study_id   => $map_study_id,
                maps           => \@maps,
            };
        }

#        push @comparative_map_studies, {
#            map_study_name => $map_study_name,
#            map_study_id   => $map_study_id,
#            species_name   => $species_name,
#            maps           => {
#            }
#        };

        push @comparative_map_studies, {
            type        => PHYSICAL,
            map_studies => $physical_comparative_maps
        } if @$physical_comparative_maps;

        push @comparative_map_studies, {
            type        => GENETIC,
            map_studies => \@genetic,
        } if @genetic;
    }

    #
    # Return the data.
    #
    return {
        start             => $start,
        end               => $end,
        maps              => \@comparative_map_studies,
        map_region_lookup => \%map_region_lookup,
    };
}

#-----------------------------------------------------
sub map_data {
#
# Called by Drawer::Map for the data for drawing the maps.
#
# Expects:
#   reference_map (a hashref detailing the map_id, start, and stop)
#   comparative_maps (an arrayref of the maps to compare)
#
# Returns:
#   A hashref containing each of the maps numbered (starting at "1"
#   for the reference map), all the feature coorespondences for all
#   the found features, and a list of the related features across all
#   maps.
#
    my ( $self, %args )  = @_;
    my @comparative_maps = @{ $args{'comparative_maps'} || [] };
    my %correspondences  = ();
    my %related_features = ();
    my $reference_map    = $args{'reference_map'};
    my $comparative_map  = shift( @comparative_maps );

    my $result = $self->get_relationships(
        reference_map                => $reference_map,
        comparative_map              => $comparative_map,
        correspondences              => \%correspondences,
        related_features             => \%related_features,
        show_only_concordant_contigs => $args{'show_only_concordant_contigs'},
    ); 

    my @finished_comparative_maps = ( $comparative_map );
    my $prev_comparative_map      = $comparative_map;
    for $comparative_map ( @comparative_maps ) {
        my $result = $self->get_relationships(
            reference_map                => $prev_comparative_map,
            comparative_map              => $comparative_map,
            correspondences              => \%correspondences,
            related_features             => \%related_features,
            show_only_concordant_contigs =>
                $args{'show_only_concordant_contigs'},
        ); 

        push @finished_comparative_maps, $comparative_map;
        $prev_comparative_map = $comparative_map;
    }

    while ( my ( $feature_id, $correspondences ) = each %correspondences ) {
        my %unique_ids = map { $_, undef } @$correspondences;
        if ( %unique_ids ) {
            $correspondences{ $feature_id } = [ keys %unique_ids ];
        }
        else {
            delete $correspondences{ $feature_id };
        }
    }

#    warn "corr =\n";
#    for my $feature ( sort { $a<=>$b } keys %correspondences ) {
#        my $corr = $correspondences{ $feature };
#        warn "$feature = ", join( ', ', @$corr ), "\n";
#    }

    my %data = (
        1                       => $reference_map,
        feature_correspondences => \%correspondences,
        related_features        => \%related_features,
    );

    my $map_no = 2;
    for my $map ( @finished_comparative_maps ) {
        $data{ $map_no } = $map;
        $map_no++;
    }

#    warn "Data = \n", Dumper( %data ), "\n";

    return \%data;
}

#-----------------------------------------------------
sub build_contig {
#
# Builds a minimal version of the contig -- basically figures out the
# contig's name and gets the features, then figures out how many
# features are there, how many are discordant, percent discordant,
# etc.  There's a lot more we could know about a contig, but this
# routine only gets the basics because the extra stuff isn't needed
# all the time.
#
# Expects:
#   physical_map_id (the current contig)
#   genetic_map_id (the current reference map)
#   a database handle
#
# Returns:
#   A hashref containing all the goodies.
#
    my ( $self, %args ) = @_;
    my $physical_map_id = $args{'physical_map_id'} or return;
    my $genetic_map_id  = $args{'genetic_map_id'}  or return;
    my $db              = $args{'db'}              || $self->db;

    #
    # Find the basic contig info
    #
    my $sth = $db->prepare(q[
        select map.physical_map_id, 
               map.map_name, 
               map.position_start as map_begin,
               map.position_stop as map_end, 
               ms.map_study_id, 
               ms.map_study_name, 
               ms.remarks, 
               ms.short_name as map_study_short_name, 
               mt.map_type, 
               s.species_id, 
               s.full_name as species_full_name, 
               s.common_name as species_common_name,
               ptg.genetic_position_start,
               ptg.genetic_position_stop,
               ptg.no_concordant_features,
               ptg.is_discordant, ptg.is_transposed
        from   cmap_physical_map map, 
               cmap_map_study ms, 
               cmap_map_type mt, 
               cmap_species s,
               cmap_physical_to_genetic_map ptg
        where  map.physical_map_id=?
        and    map.map_study_id=ms.map_study_id
        and    ms.map_type_id=mt.map_type_id
        and    ms.species_id=s.species_id
        and    map.physical_map_id=ptg.physical_map_id
        and    ptg.genetic_map_id=?
    ]);
    $sth->execute( $physical_map_id, $genetic_map_id );
    my $contig = $sth->fetchrow_hashref;

    $contig->{'position_start_bp'} = _commify( 5000*$contig->{'map_begin'} );
    $contig->{'position_stop_bp'}  = _commify( 5000*$contig->{'map_end'}  );

    my $markers = $db->selectall_arrayref(
        q[
            select   f.feature_id, f.feature_name,
                     mp.map_position_name,
                     mp.bac_hits,
                     mp.position_start as start_band,
                     mp.position_stop as stop_band,
                     pf.genetic_position, pf.is_discordant, pf.is_transposed
            from     cmap_physical_feature pf, 
                     cmap_physical_map_position mp, 
                     cmap_feature f
            where    mp.physical_map_id=?
            and      mp.physical_map_id=pf.physical_map_id
            and      pf.genetic_map_id=?
            and      pf.feature_id=mp.feature_id
            and      pf.feature_id=f.feature_id
            order by start_band
        ],
        { Columns => {} }, 
        ( $physical_map_id, $genetic_map_id )
    );

    # 
    # Set the 'link_out' field
    # 
    my $species = lc $contig->{'species_common_name'};
       $species = 'default' 
                  unless defined LINK_OUT_FIELDS->{ MARKER }{ $species };
    my $link_out_field = LINK_OUT_FIELDS->{ MARKER }{ $species };
    for my $marker ( @$markers ) {
        $marker->{'feature_link_out'} = $marker->{ $link_out_field };
    }
    $contig->{'features'} = $markers;

    return $contig;
}

#-----------------------------------------------------
sub contig_data {
#
# Called by ContigViewer (well, really by Drawer::Contig).  Gets all
# the detailed info on a contig.
#
# Expects:
#   physical_map_id
#   genetic_map_id (the current reference map)
#   a database handle
#
# Returns:
#   A hashref containing all the goodies.
#
    my ( $self, %args ) = @_;
    my $contig = $self->build_contig( %args );

    #
    # Go through all the features and find the lowest and highest
    # feature positions so we can later get the features on the 
    # genetic map just before and after them.  Also, start creating
    # the data structure needed to draw the contig detail map by 
    # pushing all these features onto the "physical" features
    # structure which is a hashref keyed by the position on the
    # contig.
    #
    my $physical_features; # The features on the contig (hashref)
    my %related_features;  # The features found on the contig and genetic map
    my %table_features;    # What will go in the big table on the left
    my $reference_map_id   = $args{'genetic_map_id'};
    my $db                 = $self->db;
    my $i                  = 0;
    my %feature_correspondences;

    #
    # Find NO_CONTIG_REF_POSITIONS on the genetic map just before 
    # and after the ones found on the contig.
    #
    my $start_cM           = $contig->{'genetic_position_start'};
    my @previous_positions = @{ 
        $db->selectcol_arrayref(
            q[
                select distinct position_start
                from   cmap_genetic_map_position
                where  position_start<?
                and    genetic_map_id=?
            ],
            {}, 
            ( $start_cM, $reference_map_id )
        )
    };
    my $previous_position = 
        @previous_positions 
        ? scalar @previous_positions <= NO_CONTIG_REF_POSITIONS 
            ? $previous_positions[ 0 ]
            : $previous_positions[ 
                $#previous_positions - NO_CONTIG_REF_POSITIONS 
            ]
        : $start_cM
    ;

    my $end_cM         = $contig->{'genetic_position_stop'};
    my @next_positions = @{
        $db->selectcol_arrayref(
            q[
                select distinct position_start
                from   cmap_genetic_map_position
                where  position_start>?
                and    genetic_map_id=?
            ],
            {}, 
            ( $end_cM, $reference_map_id )
        )
    };
    my $next_position = 
        @next_positions 
        ? scalar @next_positions <= NO_CONTIG_REF_POSITIONS 
            ? $next_positions[ -1 ]
            : $next_positions[ NO_CONTIG_REF_POSITIONS - 1 ]
        : $end_cM
    ;

    #
    # Using the positions found above, find all the features on the
    # genetic map, including those not found on the contig.
    #
    my $genetic_features = $db->selectall_arrayref( 
        q[
            select f.feature_id, f.feature_name, 
                   mp.map_position_name, mp.position_start,
                   ft.feature_type
            from   cmap_feature f, 
                   cmap_feature_type ft,
                   cmap_genetic_map_position mp
            where  mp.position_start>=?
            and    mp.position_start<=?
            and    mp.genetic_map_id=?
            and    mp.feature_id=f.feature_id
            and    f.feature_type_id=ft.feature_type_id
            and    upper(ft.feature_type)=?
        ],
        { Columns => {} }, 
        ( $previous_position, $next_position, $reference_map_id, MARKER )
    );

    #
    # Find all the correspondences to the genetic features
    #
    if ( 
        my @genetic_feature_ids = map { $_->{'feature_id'} } @$genetic_features
    ) {
        my $correspondences = $db->selectall_arrayref(
            q[
                select feature_id1, feature_id2
                from   cmap_feature_correspondence
                where  feature_id1 in (].
                       join(',', @genetic_feature_ids).q[)
                and    feature_id1<>feature_id2
            ],
            { Columns => {} }
        );

        for my $hr ( @$correspondences ) {
            push @{ $feature_correspondences{ $hr->{'feature_id1'} } },
                $hr->{'feature_id2'};
            $related_features{ $_ }++ 
                for $hr->{'feature_id1'}, $hr->{'feature_id2'};
        }
    }

    #
    # Now go through the contig's features and figure out good stuff
    # on them.
    #
    for my $feature ( @{ $contig->{'features'} } ) {
        my $feature_id = $feature->{'feature_id'};
        $related_features{ $feature_id }++;

        #
        # Record all the correspondences to this feature.
        #
        my @correspondences = @{
            $db->selectcol_arrayref(
                q[
                    select distinct feature_id2
                    from   cmap_feature_correspondence
                    where  feature_id1=?
                ],
                {}, ( $feature_id )
            )
        };

        push @{ $feature_correspondences{ $feature_id } }, $_
            for @correspondences;

        #
        # Get every position at which this feature (and any of its
        # correspondences) occurs on a genetic map.
        #
        my $positions = $db->selectall_arrayref(
            q[
                select   distinct 
                         f.feature_name, 
                         ft.feature_type, 
                         mp.feature_id, 
                         mp.map_position_name,
                         mp.position_start, 
                         mp.position_stop,
                         map.genetic_map_id, 
                         map.linkage_group,
                         ms.map_study_id, 
                         ms.map_study_name,
                         ms.short_name as map_study_short_name,
                         s.common_name as species_common_name
                from     cmap_genetic_map_position mp, 
                         cmap_genetic_map map,
                         cmap_map_study ms, 
                         cmap_species s,
                         cmap_feature f,
                         cmap_feature_type ft
                where    mp.feature_id in (].
                         join( ',', $feature_id, @correspondences ).q[)
                and      mp.feature_id=f.feature_id
                and      f.feature_type_id=ft.feature_type_id
                and      upper(ft.feature_type)=?
                and      mp.genetic_map_id=map.genetic_map_id
                and      map.map_study_id=ms.map_study_id
                and      ms.species_id=s.species_id
                order by linkage_group, position_start
            ], 
            { Columns => {} },
            ( MARKER )
        );

        #
        # Try to sort out the positions for the marker putting the
        # position for the reference map first, all other later.
        #
        my %position_lookup = map { $_->{'genetic_map_id'}, $_ } @$positions;
        my $reference_position;

        for my $map_id ( keys %position_lookup ) {
            if ( $map_id == $reference_map_id ) {
                $reference_position = $position_lookup{ $map_id };
                delete $position_lookup{ $map_id };
                last;
            }
        }

        $positions = [
            map { $_ || () } ( $reference_position, values %position_lookup )
        ];

        $feature->{'genetic_positions'} = $positions;

        #
        # Figure out the physical position of this feature on the
        # contig.  If the contig "is_transposed" (meaning all of its
        # features occur in a different order on the genetic map than
        # they do on the physical), then measure the physical position
        # of the feature by subtracting the start_band from the end of
        # the contig.
        #
        my $physical_position = $contig->{'is_transposed'}
            ? $contig->{'map_end'} - $feature->{'start_band'}
            : $feature->{'start_band'}
        ;

        #
        # Many features can occur at the same physical position, so
        # push this feature onto our "physical_features" hashref.
        #
        push @{ $physical_features->{ $physical_position } }, $feature;

        #
        # Record this feature in the data structure used to fill in
        # the big information table on the contig viewer page.
        #
        $table_features{ $feature->{'feature_id'} } = { 
            order  => $i++, 
            feature => $feature 
        };
    }

    #
    # Get rid of repeats in the correspondences.
    #
    while ( 
        my ( $feature_id, $correspondences ) = each %feature_correspondences 
    ) {
        my %unique_ids = map { $_, undef } @$correspondences;
        if ( %unique_ids ) {
            $feature_correspondences{ $feature_id } = [ keys %unique_ids ];
        }
        else {
            delete $feature_correspondences{ $feature_id };
        }
    }

#    warn "Correspondences =\n", Dumper( %feature_correspondences ), "\n";

    #
    # Sort out the info we'll use to draw the genetic (reference) map.
    #
    my ( $genetic_map_study_id, $genetic_map_study_name, 
         $genetic_map_name, $genetic_species_id, $genetic_species ) = 
        $db->selectrow_array(
            q[
                select ms.map_study_id, ms.short_name, 
                       map.linkage_group, 
                       s.species_id, s.common_name
                from   cmap_genetic_map map, 
                       cmap_map_study ms, 
                       cmap_species s
                where  map.genetic_map_id=?
                and    map.map_study_id=ms.map_study_id
                and    ms.species_id=s.species_id
            ],
            {}, ( $reference_map_id )
        )
    ; 

    my $lc_species = lc $genetic_species;
    $lc_species    = 'default' unless 
        defined LINK_OUT_FIELDS->{ MARKER }{ $lc_species };
    my $link_out_field = LINK_OUT_FIELDS->{ MARKER }{$lc_species};

    my $reference_map_features;
    for my $feature ( @$genetic_features ) {
#        warn "$feature->{'feature_name'}, $feature->{'feature_id'}, ", 
#            "$feature->{'is_related'}\n";
        push @{ $reference_map_features->{ $feature->{'position_start'} } }, {
            feature_id        => $feature->{'feature_id'},
            feature_name      => $feature->{'feature_name'},
            feature_type      => $feature->{'feature_type'},
            map_position_name => $feature->{'map_position_name'},
            is_related        => $related_features{ $feature->{'feature_id'} },
            feature_link_out  => $feature->{$link_out_field},
        };
    }
#    warn "genetic features = \n", Dumper( $reference_map_features ), "\n";
#    warn "related genetic features = ", join ("\n", join( ':', 
#        map { $_->{'feature_name'}, $_->{'feature_id'}, $_->{'is_related'} }
##        grep { $_->{'is_related'} }
#        @$genetic_features 
#    )), "\n";

    #
    # Find any clones associated with this contig.
    #
    $contig->{'clones'} = $db->selectall_arrayref(
        q[
            select   f.feature_id, f.feature_name as clone_name,
                     mp.position_start, mp.position_stop
            from     cmap_physical_map_position mp, 
                     cmap_feature f,
                     cmap_feature_type ft
            where    mp.physical_map_id=?
            and      mp.feature_id=f.feature_id
            and      f.feature_type_id=ft.feature_type_id
            and      upper(ft.feature_type)=?
            order by position_start, clone_name
        ],
        { Columns => {} }, 
        ( $args{'physical_map_id'}, CLONE )
    );
    
    #
    # Now sum it all up and send it back.
    #
    my $reference_map  =  {
        map_study_id   => $genetic_map_study_id,
        map_id         => $reference_map_id,
        map_name       => $genetic_map_name,
        map_label      => join('-', $genetic_species, $genetic_map_study_name,
                              $genetic_map_name),
        features       => $reference_map_features,
        species        => $genetic_species,
        map_begin      => defined $previous_position 
                          ? $previous_position : $start_cM,
        map_end        => defined $next_position     
                          ? $next_position     :   $end_cM,
    };

#    warn "features\n", Dumper( $physical_features ), "\n";

    $start_cM--, $end_cM++ while $end_cM - $start_cM < 2;
    $contig->{'map_id'}          = $args{'physical_map_id'};
    $contig->{'features'}        = $physical_features || {};
    $contig->{'species'}         = $contig->{'species_common_name'};
    $contig->{'map_label'}       = join('-',
                                       $contig->{'species_common_name'},
                                       $contig->{'map_study_short_name'},
                                       $contig->{'map_name'}
                                   );
    $contig->{'map_study_id'}    = $contig->{'map_study_id'};
    $contig->{'table_features'}  = [
        map    { $_->{'feature'} }
        sort   { 
            $contig->{'is_transposed'}
            ? $b->{'order'} <=> $a->{'order'} 
            : $a->{'order'} <=> $b->{'order'} 
        }
        values %table_features
    ];

#    warn "SOAP contig =\n", Dumper( $contig ), "\n";
#    warn "feature_correspondences =>\n", Dumper(%feature_correspondences), "\n";

    return {
        1                       => $reference_map,
        2                       => $contig,
        feature_correspondences => \%feature_correspondences,
    };
}

#-----------------------------------------------------
sub map_study_info {
#
# Returns all the basic info on one or all map studies.
#
    my ( $self, %args ) = @_;
    my @map_study_ids   = @{ $args{'map_study_ids'} || [] };

    my $db = $self->db;
    my $sql = q[
        select   ms.map_study_id, ms.map_study_name, ms.short_name,
                 ms.map_type_id, ms.species_id, ms.remarks,
                 mt.map_type, mt.map_units, s.common_name, s.full_name
        from     cmap_map_study ms, 
                 cmap_map_type mt, 
                 cmap_species s
        where    ms.map_type_id=mt.map_type_id
        and      ms.species_id=s.species_id
    ]; 

    if ( @map_study_ids ) {
        $sql .= 'and ms.map_study_id in ('.join(',',@map_study_ids).')';
    }
    $sql .= 'order by short_name';

    my $map_studies = $db->selectall_arrayref( $sql, { Columns => {} } );

    for my $map_study ( @$map_studies ) {
        next if uc $map_study->{'map_type'} eq PHYSICAL;
        my @linkage_groups =  @{
            $db->selectall_arrayref(
                q[
                    select   genetic_map_id, linkage_group
                    from     cmap_genetic_map
                    where    map_study_id=?
                    order by linkage_group
                ],
                { Columns => {} }, ( $map_study->{'map_study_id'} )
            )
        } or next;

        my $all_numbers = 
            grep { $_->{'linkage_group'} =~ m/^[0-9]/ } @linkage_groups;

        if ( $all_numbers == scalar @linkage_groups ) {
            @linkage_groups = 
                map  { $_->[0] }
                sort { $a->[1] <=> $b->[1] }
                map  { [$_, _extract_numbers( $_->{'linkage_group'} )] }
                @linkage_groups
            ;
        }

        $map_study->{'linkage_groups'} = \@linkage_groups;
    }

    return $map_studies;
}

#-----------------------------------------------------
# commented out KYC 2002-02-14 -- probably obsolete
#sub map_type {
#    my ( $self, %args ) = @_;
#    my $map_study_id    = $args{'map_study_id'} or return;
#    my $db              = $self->db;
#    my $map_type        = $db->selectrow_array(q[
#        select mt.map_type
#        from   map_type mt, map_study ms
#        where  ms.map_type_id=mt.map_type_id
#        and    ms.map_study_id=?
#    ], {}, ( $map_study_id ) );
#
#    return $map_type;
#}

#-----------------------------------------------------
sub map_end {
    my ( $self, %args ) = @_;
    my $db = $self->db;

    my $position = $db->selectrow_array( 
        q[
            select   max(position_start)
            from     cmap_genetic_map_position
            where    genetic_map_id=?
        ], 
        {}, ( $args{'map_id'} ) 
    );

    unless ( defined $position ) {
        $position = $db->selectrow_array( 
            q[
                select   max(position_start)
                from     cmap_physical_map_position
                where    physical_map_id=?
            ], 
            {}, ( $args{'map_id'} ) 
        );
    }

    return $position;
}

#-----------------------------------------------------
# commented out KYC 2002-02-14 -- probably obsolete
#sub map_start {
#    my ( $self, %args ) = @_;
#
#    return $self->db->selectrow_array( 
#        q[
#            select   min(position_start)
#            from     map_position
#            where    map_id=?
#        ], 
#        {}, ( $args{'map_id'} ) 
#    ) || 0;
#}

#-----------------------------------------------------
# commented out KYC 2002-02-14 -- probably obsolete
#sub marker_id_by_map_position {
#    my ( $self, %args ) = @_;
#    return $self->db->selectrow_array(
#        q[
#            select marker_id
#            from   map_position
#            where  map_position_id=?
#        ], 
#        {}, ( $args{'map_position_id'} ) 
#    ) || 0; 
#}

#-----------------------------------------------------
sub map_label {
    my ( $self, %args ) = @_;
    my $map_study_id    = $args{'map_study_id'} || 0;
    my $map_id          = $args{'map_id'}       || 0;
    my $db              = $self->db;
    my $label;
    if ( $map_id ) {
        my @fields = 
            $db->selectrow_array(
                q[
                    select count(genetic_map_id)
                    from   cmap_genetic_map
                    where  genetic_map_id=?
                ], {}, ( $map_id ) 
            )
            ? $db->selectrow_array(
                q[
                    select s.common_name, ms.short_name, 
                           map.linkage_group
                    from   cmap_genetic_map map, 
                           cmap_map_study ms, 
                           cmap_species s
                    where  map.genetic_map_id=?
                    and    map.map_study_id=ms.map_study_id
                    and    ms.species_id=s.species_id
                ], {}, ( $map_id ) 
            )
            : $db->selectrow_array(
                q[
                    select s.common_name, ms.short_name, map.map_name
                    from   cmap_physical_map map, 
                           cmap_map_study ms, 
                           cmap_species s
                    where  map.physical_map_id=?
                    and    map.map_study_id=ms.map_study_id
                    and    ms.species_id=s.species_id
                ], {}, ( $map_id ) 
            )
        ;
        $label = @fields ? join( '-', @fields ) : '';
    }
    else {
        my @fields = $db->selectrow_array(q[
            select s.common_name, ms.short_name
            from   cmap_map_study ms, cmap_species s
            where  ms.map_study_id=?
            and    ms.species_id=s.species_id
        ], {}, ( $map_study_id ) );
        $label = @fields ? join( '-', @fields ) : '';
    }

    return $label || '';
}

#-----------------------------------------------------
sub get_relationships {
#
# Returns the data for drawing maps
#
    my ( $self, %args )  = @_;
    my $db               = $self->db;
    my $reference_map    = $args{'reference_map'}    || {};
    my $comparative_map  = $args{'comparative_map'}  || {};
    my $correspondences  = $args{'correspondences'}  || {};
    my $related_features = $args{'related_features'} || {};

    #
    # Collect general info on the reference map
    #
    my $sth = $db->prepare(
        q[
            select   map.linkage_group,
                     s.common_name, 
                     mt.map_type, 
                     ms.map_study_id,
                     min(mp.position_start) as map_begin, 
                     max(mp.position_start) as map_end
            from     cmap_genetic_map_position mp, 
                     cmap_genetic_map map, 
                     cmap_map_study ms, 
                     cmap_species s, 
                     cmap_map_type mt
            where    mp.genetic_map_id=?
            and      mp.genetic_map_id=map.genetic_map_id
            and      map.map_study_id=ms.map_study_id
            and      ms.species_id=s.species_id
            and      ms.map_type_id=mt.map_type_id
            group by map.linkage_group, ms.map_study_id, 
                     mt.map_type, s.common_name
        ]
    );
    $sth->execute( $reference_map->{'genetic_map_id'} );
    my $hr = $sth->fetchrow_hashref;

    $reference_map->{ $_ }       =  $hr->{ $_ } for keys %$hr;
    $reference_map->{'start'}    =  $hr->{'map_begin'} 
        unless $reference_map->{'start'} =~ FLOAT_REGEX;
    $reference_map->{'end'}      =  $hr->{'map_end'}   
        unless $reference_map->{'end'}   =~ FLOAT_REGEX;
    $reference_map->{'map_id'}   = $reference_map->{'genetic_map_id'};
    $reference_map->{'map_name'} = $hr->{'linkage_group'};
    $reference_map->{'species'}  = $hr->{'common_name'};

    $reference_map->{'map_label'} ||=  $self->map_label(
        map_study_id  => $reference_map->{'map_study_id'},
        map_id        => $reference_map->{'genetic_map_id'},
    );

    #
    # Get all the features on the reference map
    #
    my $reference_map_features = $db->selectall_arrayref(
        q[
            select   mp.map_position_name,
                     mp.position_start, mp.position_stop,
                     f.feature_id, f.feature_name, 
                     ft.feature_type, map.linkage_group
            from     cmap_genetic_map_position mp,
                     cmap_feature f,
                     cmap_feature_type ft,
                     cmap_genetic_map map
            where    mp.genetic_map_id=?
            and      mp.feature_id=f.feature_id
            and      f.feature_type_id=ft.feature_type_id
            and      mp.genetic_map_id=map.genetic_map_id
            and      (
                (
                    mp.position_start>=?
                    and
                    mp.position_start<=?
                    and
                    mp.position_stop is null
                )
                or (
                    mp.position_start<=?
                    and
                    mp.position_stop>=?
                )
                or ( 
                    mp.position_start>=?
                    and      
                    mp.position_stop<=?
                )
            )
        ],
        { Columns => {} }, 
        ( $reference_map->{'genetic_map_id'}, 
          $reference_map->{'start'}, 
          $reference_map->{'end'},
          $reference_map->{'start'}, 
          $reference_map->{'end'},
          $reference_map->{'start'}, 
          $reference_map->{'end'},
        )
    );

    my %reference_feature_positions =
        map { $_->{ 'feature_id' }, $_->{ 'position_start' } } 
        @$reference_map_features;

    my $reference_feature_string = 
        join( ',', keys %reference_feature_positions ) || '';
#    warn "ref feature str = $reference_feature_string\n";

    #
    # Comparative maps could be physical or genetic, whole map
    # studies or individual maps.
    #
    my %corr_lookup;
    if ( %reference_feature_positions ) {
        #
        # Find all the aliases for these features
        #
        my $feature_correspondences = $db->selectall_arrayref(
            qq[
                select feature_id1, feature_id2
                from   cmap_feature_correspondence
                where  feature_id1 in ( $reference_feature_string )
            ]
        );

        #
        # Use the data to populate the correspondence lookups
        #
        for my $ar ( @$feature_correspondences ) {
            my ( $f1, $f2 ) = @$ar;
            push @{ $correspondences->{ $f1 } }, $f2;
            push @{ $corr_lookup      { $f1 } }, $f2;
            push @{ $corr_lookup      { $f2 } }, $f1;
        }

        #
        # Use the correspondences to say that they also occur at the 
        # same positions as the original markers.
        # 
        while ( 
            my ( $feature_id, $position ) = each %reference_feature_positions 
        ) {
            for my $corr_id ( @{ $correspondences->{ $feature_id } } ) {
                $reference_feature_positions{ $corr_id } = $position;
            }
        }
    }

    #
    # Update $reference_feature_string to include the correspondences.
    #
    my %all_feature_ids = map { $_, 1 }
        keys %reference_feature_positions, keys %corr_lookup;
    $reference_feature_string = join( ',', keys %all_feature_ids );

    my %comparative_feature_ids = ();
    if ( %reference_feature_positions and $comparative_map ) {
        my $map_study_id = $comparative_map->{'map_study_id'}   || 0;
        my $map_id       = $comparative_map->{'genetic_map_id'} || 0;

        #
        # Make sure we have a map_study_id...
        #
        unless ( $map_study_id ) {
            $map_study_id = $db->selectrow_array( 
                q[
                    select map_study_id
                    from   cmap_genetic_map map
                    where  genetic_map_id=?
                ],
                {}, ( $map_id )
            );
        }

        unless ( $map_study_id ) {
            $map_study_id = $db->selectrow_array( 
                q[
                    select map_study_id
                    from   cmap_physical_map map
                    where  physical_map_id=?
                ],
                {}, ( $map_id )
            );
        }

        $comparative_map->{'map_study_id'} = $map_study_id;

        #
        # And a map type & species...
        #
        my ( $map_type, $species ) = $db->selectrow_array(
            q[
                select mt.map_type, s.common_name
                from   cmap_map_study ms, 
                       cmap_map_type mt, 
                       cmap_species s
                where  ms.map_type_id=mt.map_type_id
                and    ms.species_id=s.species_id
                and    ms.map_study_id=?
            ],
            {}, ( $map_study_id )
        );

        $comparative_map->{'map_type'} = $map_type;
        $comparative_map->{'species'}  = $species;

        #
        # For genetic maps, see if we're looking for just one map or all
        # the maps in a study.  For each one, find the min and max
        # positions on the comparative map of the markers found on the
        # reference map, then move back/forward one position in order to
        # gain context), then find all the markers on the comparative in
        # that region.
        #
        if ( uc $map_type eq GENETIC ) {
            my @map_ids = ( $map_id ) ? ( $map_id ) :
                map { $_->[0] } @{ 
                    $db->selectall_arrayref(
                        qq[
                            select   distinct map.genetic_map_id
                            from     cmap_genetic_map map, 
                                     cmap_genetic_map_position mp
                            where    mp.feature_id in 
                                     ($reference_feature_string)
                            and      mp.genetic_map_id<>?
                            and      mp.genetic_map_id=map.genetic_map_id
                            and      map.map_study_id=?
                        ],
                        {}, 
                        ( $reference_map->{'genetic_map_id'}, $map_study_id )
                    )
                }
            ;
            if ( scalar @map_ids == 1 ) {
                $map_id = $map_ids[0];
                $comparative_map->{'map_id'}   = $map_id;
                $comparative_map->{'map_name'} = $db->selectrow_array(
                    q[
                        select linkage_group
                        from   cmap_genetic_map
                        where  genetic_map_id=?
                    ],
                    {},
                    ( $map_id )
                );
            }

            my ( $start, $end );
            for my $comparative_map_id ( @map_ids ) {
                my ( $min, $max ) = $db->selectrow_array(
                    qq[
                        select min(mp.position_start), 
                               max(mp.position_start)
                        from   cmap_genetic_map_position mp,
                               cmap_feature f,
                               cmap_feature_type ft
                        where  mp.feature_id in
                               ($reference_feature_string)
                        and    mp.genetic_map_id=?
                        and    mp.feature_id=f.feature_id
                        and    f.feature_type_id=ft.feature_type_id
                        and    upper(ft.feature_type)=?
                    ],
                    {}, ( $comparative_map_id, MARKER )
                );
                next unless defined $min and defined $max;

                my $next_min = $db->selectrow_array(
                    q[
                        select max(mp.position_start) 
                        from   cmap_genetic_map_position mp,
                               cmap_feature f,
                               cmap_feature_type ft
                        where  mp.position_start<?
                        and    mp.genetic_map_id=?
                        and    mp.feature_id=f.feature_id
                        and    f.feature_type_id=ft.feature_type_id
                        and    upper(ft.feature_type)=?
                    ],
                    {}, ( $min, $comparative_map_id, MARKER )
                ) || $min;

                my $next_max = $db->selectrow_array(
                    q[
                        select min(mp.position_start) 
                        from   cmap_genetic_map_position mp,
                               cmap_feature f,
                               cmap_feature_type ft
                        where  mp.position_start>?
                        and    mp.genetic_map_id=?
                        and    mp.feature_id=f.feature_id
                        and    f.feature_type_id=ft.feature_type_id
                        and    upper(ft.feature_type)=?
                    ],
                    {}, ( $max, $comparative_map_id, MARKER )
                ) || $max;
                
                if ( scalar @map_ids == 1 ) {
                    $start = $next_min;
                    $end   = $next_max;
                }

                my $comparative_map_features = $db->selectall_arrayref(
                    q[
                        select   mp.position_start, 
                                 mp.genetic_map_position_id,
                                 mp.map_position_name,
                                 f.feature_id, f.feature_name,
                                 ft.feature_type, map.linkage_group
                        from     cmap_genetic_map_position mp, 
                                 cmap_feature f, 
                                 cmap_feature_type ft, 
                                 cmap_genetic_map map
                        where    mp.genetic_map_id=?
                        and      mp.position_start>=?
                        and      mp.position_start<=?
                        and      mp.feature_id=f.feature_id
                        and      f.feature_type_id=ft.feature_type_id
                        and      mp.genetic_map_id=map.genetic_map_id
                        order by position_start, feature_name
                    ],
                    { Columns => {} }, 
                    ( $comparative_map_id, $next_min, $next_max )
                );

                # 
                # Set the 'link_out' field
                # 
                my $lc_species = lc $species;
                $lc_species    = 'default' unless 
                    defined LINK_OUT_FIELDS->{ MARKER }{ $lc_species };
                my $link_out_field = LINK_OUT_FIELDS->{ MARKER }{$lc_species};

                my ( $map_data, $no_anchored_features );
                for my $feature ( @$comparative_map_features ) {
                    $all_feature_ids{ $feature->{'feature_id'} }++;

                    $feature->{'feature_link_out'} = 
                        $feature->{$link_out_field};

                    $feature->{'reference_map_position'} = 
                        $reference_feature_positions{$feature->{'feature_id'}};

                    if ( $feature->{'reference_map_position'} ) {
                        $related_features->{ $feature->{'feature_id'} }++;
                        $no_anchored_features++;
                    }

                    $comparative_feature_ids{ $_ }++
                        for $feature->{'feature_id'}, 
                            @{ $corr_lookup{ $feature->{'feature_id'} } };

                    push @{ $map_data->{ $feature->{'position_start'} } }, 
                        $feature;
                }

                $comparative_map->{'maps'}{$comparative_map_id}{'features'}=
                    $map_data;

                $comparative_map->{'maps'}{$comparative_map_id}{'map_id'} = 
                    $comparative_map_id;

                $comparative_map->{'maps'}{$comparative_map_id}
                    {'no_anchored_features'} =
                    $no_anchored_features;

                #
                # We also need the map's name
                #
                $comparative_map->{'maps'}{$comparative_map_id}{'map_name'}=
                    $db->selectrow_array(
                        q[
                            select linkage_group
                            from   cmap_genetic_map map
                            where  genetic_map_id=?
                        ],
                        {}, ( $comparative_map_id )
                    )
                ;
            }

            $comparative_map->{'start'} = $start;
            $comparative_map->{'end'}   = $end;
        }
        #
        # Else, this is a physical map...
        #
        else {
            my $sql = qq[
                select   distinct map.physical_map_id, map.map_name,
                         map.position_start as contig_start,
                         map.position_stop as contig_stop,
                         ptg.no_concordant_features,
                         ptg.is_discordant, ptg.is_transposed
                from     cmap_physical_map_position mp, 
                         cmap_physical_map map,
                         cmap_physical_to_genetic_map ptg
                where    mp.feature_id in 
                         ($reference_feature_string)
                and      mp.physical_map_id=ptg.physical_map_id
                and      ptg.genetic_map_id=?
                and      mp.physical_map_id=map.physical_map_id
                and      map.map_study_id=?
            ];

            $sql .= q[ and ptg.is_discordant=0 ]
                if $args{'show_only_concordant_contigs'};

            my $contigs = $db->selectall_arrayref(
                $sql, 
                { Columns => {} }, 
                ( $reference_map->{'genetic_map_id'}, $map_study_id )
            );

            my %feature_hash = ();
            if ( @$contigs ) {
                my $features = $db->selectall_arrayref( 
                    q[
                        select   distinct f.feature_id, f.feature_name, 
                                 mp.position_start as start_band,
                                 mp.position_stop as stop_band, 
                                 mp.bac_hits, mp.physical_map_id,
                                 pf.genetic_map_id, pf.genetic_position, 
                                 pf.is_discordant, pf.is_transposed
                        from     cmap_physical_map_position mp,
                                 cmap_feature f, 
				                 cmap_physical_feature pf
                        where    mp.physical_map_id in (].
                        join(',',map{$_->{'physical_map_id'}}@$contigs).q[)
                        and      mp.physical_map_id=pf.physical_map_id
                        and      mp.feature_id=pf.feature_id
                        and      pf.feature_id=f.feature_id
                        and      pf.genetic_map_id=?
                    ],
                    { Columns => {} }, ( $reference_map->{'genetic_map_id'} )
                );

                for my $feature ( @$features ) {
                    push @{ $feature_hash { $feature->{'physical_map_id'} } }, 
                        $feature;
                    $all_feature_ids{ $feature->{'feature_id'} }++;
                }
            }

            for my $contig ( @$contigs ) {
                my $contig_id = $contig->{'physical_map_id'};
                my $features  = $feature_hash{ $contig_id };
                for my $feature ( @$features ) {
                    $comparative_feature_ids{ $_ }++
                        for $feature->{'feature_id'}, 
                            @{ $corr_lookup{ $feature->{'feature_id'} } };
                }
                $contig->{'features'} = $features;

                $comparative_map->{'maps'}{ $contig_id } = $contig;
            }
        }

        #
        # Don't forget to set the map label
        #
        $comparative_map->{'map_label'} = $self->map_label(
            map_study_id => $map_study_id,
            map_id       => $map_id,
        );
    }

    #
    # Get all the correspondences
    #
    my @correspondences = ();
    if ( my @all_ids = keys %all_feature_ids ) {
        my $number   = scalar @all_ids;
        $number      = 500 if $number > 500;
        my @slice    = splice @all_ids, 0, $number; 
        push @correspondences, @{ 
            $db->selectall_arrayref(
                qq[
                    select feature_id1, feature_id2
                    from   cmap_feature_correspondence
                    where  feature_id1 in (].
                           join( ',', @slice ).q[)
                ]
            )
        };
    }

    for my $ar ( @correspondences ) {
        push @{ $correspondences->{ $ar->[0] } }, $ar->[1];
    }

    unless ( $reference_map->{'maps'} ) {
        my $species = lc $reference_map->{'species'};
           $species = 'default' 
                unless defined LINK_OUT_FIELDS->{ MARKER }{$species};
        my $link_out_field = LINK_OUT_FIELDS->{ MARKER }{$species};

        my $map_data;
        for my $feature ( @$reference_map_features ) {
            $feature->{'is_related'}       = 
                $comparative_feature_ids{ $feature->{'feature_id'} };
            $feature->{'feature_link_out'} = $feature->{ $link_out_field };

            push @{ $map_data->{ $feature->{'position_start'} } }, $feature;
        }

        $reference_map->{'maps'} = $map_data;
    }

    return 1;
}

1;

#-----------------------------------------------------
# Perfection is achieved not when there is nothing more
# to add, but when there is nothing more to take away.
# Antoine de Saint-Exupery
#-----------------------------------------------------

=pod

=head1 NAME

CSHL::SOAP::ComparativeMapData

=head1 SYNOPSIS

As a stand-alone object:

  use CSHL::SOAP::ComparativeMapData
  my $soap_lib = CSHL::SOAP::ComparativeMapData->new;
  return $soap_lib->method( { foo => bar } ); 

As a SOAP object:

    package SOAP::Apache;
    use SOAP::Transport::HTTP;

    my $server = SOAP::Transport::HTTP::Apache->dispatch_to(
        '/usr/local/gramene/lib/perl/CSHL/SOAP/ComparativeMapData'
    );

    sub handler { $server->handler(@_) }

    1;

=head1 DESCRIPTION

This module is intended to encapsulate all database intereactions.
Originally, this was because we intended to use SOAP as the method for
retrieving data.  When this was (temporarily) abandoned, it was found
that it was just as easy to use this module as a regular object.  In
addition, I found it was very nice to have only one place where I
created a database handle, only one file to edit when tables changed,
etc.  

Even though it's not necessarily a SOAP module, it could be used as
such.  All you need to do is to change the
CSHL::ComparativeMaps::Constants::DATA_SOURCE_LOCATION from "local" to
"remote."

=head1 AUTHOR

Ken Y. Clark, kclark@logsoft.com

=head1 SEE ALSO

perl(1), SOAP::Lite, DBI, DBD::(mysql|Oracle).

=cut
