package CSHL::ComparativeMaps::UserPreferences;

#-----------------------------------------------------
# $Id: UserPreferences.pm,v 1.5 2002/04/18 19:08:32 kclark Exp $
#
# File       : UserPreferences.pm
# Programmer : Ken Y. Clark, kclark@cshl.org
# Created    : 2002/04/02
# Purpose    : set/retrieve user preferences
#-----------------------------------------------------

use strict;
use vars qw( $VERSION );
$VERSION = (qw$Revision: 1.5 $)[-1];

use Apache::Constants;
use Apache::Cookie;
use Apache::Request;

use CSHL::ComparativeMaps::Constants;

use constant FIELD_SEP  => '=';
use constant RECORD_SEP => ';';
use Data::Dumper;

sub handler {
#
# Get the current preferences and any existing cookie.  Always take
# current settings over cookie settings.  End by always setting
# cookie with current settings.
#
    my $r                 = shift;
    my @preference_fields = @{ +PREFERENCE_FIELDS };
    my $apr               = Apache::Request->new( $r );
    my $cookie_name       = $r->dir_config('USER_PREF_COOKIE_NAME') ||
                            DEFAULT->{'user_pref_cookie_name'};
    my %preferences       = ();

    #
    # Try to fetch the cookie and read it.
    #
    if ( my %cookies = Apache::Cookie->new($r)->fetch ) {{
        my $cookie        = $cookies{$cookie_name} or last;
        my $cookie_string = $cookie->value         ||   '';
        my @cookie_fields = split RECORD_SEP, $cookie_string;

        foreach ( @cookie_fields ) {
            my ( $name, $value )  = split FIELD_SEP;
            $preferences{ $name } = $value if $value;
        }
    }}

    #
    # This updates the preferences with whatever is in the latest
    # request from the user.  If the preference isn't defined in this
    # request, then we'll leave whatever's there.  If nothing is
    # defined, then we'll set it with the default value.
    #
    for my $pref ( @preference_fields ) {
        $preferences{ $pref } = 
            defined $apr->param( $pref )
                ? $apr->param( $pref ) 
                : defined $preferences{ $pref }
                    ? $preferences{ $pref } 
                    : DEFAULT->{ $pref }
        ;
    }

    #
    # Place the preferences into pnotes for use further downstream.
    #
    $apr->pnotes( PREFERENCES => \%preferences );

    #
    # Set a new cookie with the latest preferences.
    #
    my $cookie_domain = $apr->dir_config('COOKIE_DOMAIN') ||
                        DEFAULT->{'cookie_domain'};
    my $cookie_value  = join( RECORD_SEP,
        map { join( FIELD_SEP, $_, $preferences{ $_ } ) } @preference_fields 
    );

    my $cookie   = Apache::Cookie->new(
        $apr,
        -name    => $cookie_name,
        -value   => $cookie_value,
        -expires => '+1y',
        -domain  => $cookie_domain,
        -path    => '/'
    );
    $cookie->bake;

    return OK;
}

1;

#-----------------------------------------------------
# We should seek not to define ourselves
# by what we can do but rather
# by the things we have the capacity to enjoy.
# Jim Willis
#-----------------------------------------------------

=head1 NAME

CSHL::ComparativeMaps::UserPreferences - set/retrieve user preferences

=head1 SYNOPSIS

  <Location /maps/viewer>
      PerlInitHandler CSHL::ComparativeMaps::UserPreferences
      SetHandler      perl-script
      PerlHandler     CSHL::ComparativeMaps::MapViewer
  </Location>

=head1 DESCRIPTION

Blah blah blah.

=head1 AUTHOR

Ken Y. Clark, kclark@cshl.org

=head1 SEE ALSO

perl(1).

=cut
