package CSHL::ComparativeMaps::Drawer::Map;

#-----------------------------------------------------
# $Id: Map.pm,v 1.15 2002/04/17 01:47:42 kclark Exp $
#
# File       : CSHL/ComparativeMaps/Drawer/Map.pm
# Programmer : Ken Y. Clark, kclark@logsoft.com
# Created    : 2001/08/10
# Purpose    : draw comparative maps
#-----------------------------------------------------

use strict;
use vars qw( $VERSION );
$VERSION = (qw$Revision: 1.15 $)[-1];

use Data::Dumper;

use CSHL::DB;
use CSHL::ComparativeMaps::Constants;
use CSHL::ComparativeMaps::Drawer;
use base qw( CSHL::ComparativeMaps::Drawer );

#-----------------------------------------------------
sub _init {
    my $self = shift;
    $self->SUPER::_init();

    #
    # Figure out which maps to flip
    #
    if ( my @map_nos = split( /,/, $self->{'flip'} ) ) {
        $self->{'flip_map'}{ $_ }++ for @map_nos;
    }

    $self->{'data'} = $self->soap_call(
        method => 'map_data',
        params => { 
            reference_map      => {
                map_study_id   => $self->map_study_id( 1 ),
                genetic_map_id => $self->map_id      ( 1 ),
                start          => $self->start,
                end            => $self->end,
            },
            comparative_maps => $self->{'comparative_maps'},
            show_only_concordant_contigs => 
                $self->{'show_only_concordant_contigs'},
        },
    );
#    warn "self->data =\n", Dumper($self->{'data'}),"\n";

    #
    # Find the longest cM distance in order to determine the scale
    #
    my $longest_cM = $self->end - $self->start || 0;
    for my $map_no ( 2 .. $self->no_maps ) {
        next unless uc $self->map_type( $map_no ) eq GENETIC;
        my $data = $self->features( $map_no ) or next;

        for my $map_id ( keys %$data ) {
            my @positions = sort { $a <=> $b } 
                            keys %{ $data->{$map_id}{'features'} };
            my $length    = abs( $positions[ 0 ] - $positions[ -1 ] ) || 0;
            $longest_cM   = $length if $length > $longest_cM;
        }
    }
    $longest_cM = MIN_MAP_CM_LENGTH if $longest_cM < MIN_MAP_CM_LENGTH;

    my $image_height = $self->{'image_height'} || DEFAULT->{'image_height'};
    $self->scale( $image_height / $longest_cM );

    return $self;
}

#-----------------------------------------------------
sub data {
    my $self   = shift;
    my $map_no = shift || 1;
    return undef if $map_no > $self->no_maps;
    return undef unless $self->{'data'};
    return $self->{'data'}{$map_no};
}

#-----------------------------------------------------
sub features {
    my $self   = shift;
    my $map_no = shift || 1;
    return undef if $map_no > $self->no_maps;
    return undef unless $self->{'data'};
    return $self->{'data'}{$map_no}{'maps'};
}

#-----------------------------------------------------
sub end {
    my $self = shift;
    my $arg  = shift;
    $self->{'end'} = $arg if defined $arg;
    return $self->{'end'};
}

#-----------------------------------------------------
sub flip {
    my $self   = shift;
    if ( my $map_no = shift ) {
        return $self->{'flip_map'}{ $map_no } || 0;
    }
    else {
        return join(',', keys %{ $self->{'flip_map'} } );
    }
}

#-----------------------------------------------------
sub map_end {
    my $self = shift;

    unless ( defined $self->{'map_end'} ) {
        $self->{'map_end'} = $self->soap_call(
            method => 'map_end',
            params => { map_id => shift },
        );
    }

    return $self->{'map_end'};
}

#sub map_relationships {
#    my $self = shift;
#    return unless $self->no_maps == 2;
#    return [ { foo => 'bar' } ];
#}

#-----------------------------------------------------
sub map_id {
    my $self   = shift;
    my $map_no = shift || 1;
    return undef if $map_no > $self->no_maps;
    return $self->{'map_id'.$map_no};
}

#-----------------------------------------------------
sub map_label {
    my $self   = shift;
    my $map_no = shift || 1;
    return undef if $map_no > $self->no_maps;
    return $self->{'data'}{$map_no}{'map_label'} || '';
}

#-----------------------------------------------------
sub map_study_id {
    my $self   = shift;
    my $map_no = shift || 1;
    return undef if $map_no > $self->no_maps;
    my $field  = 'map_study_id'.$map_no;
    unless ( $self->{ $field } ) {
        $self->{ $field } = $self->{'data'}{$map_no}{'map_study_id'};
    }

    return $self->{ $field };
}

#-----------------------------------------------------
sub map_type {
    my $self   = shift;
    my $map_no = shift || 1;
#    return undef if $map_no > $self->no_maps;
    return $self->{'data'}{$map_no}{'map_type'} || '';
}

#-----------------------------------------------------
sub component_list { 
    # menu
    qw[ reference_map comparative_maps ] 
}

#-----------------------------------------------------
sub menu_fields { 
    qw[ reference_map_study_id reference_map_id
        comparative_map_study_id comparative_map_id
        start end highlight show_only_concordant_contigs image_type 
    ]
}

#-----------------------------------------------------
sub menu_uri { URIS->{'map_viewer'} }

#-----------------------------------------------------
sub menu_query_string {
    my $self = shift;
    my $uri  = $self->menu_uri;

    my $comparative_map_field1 = $self->map_id( 2 )
        ? 'genetic_map_id%3D'.$self->map_id( 2 )
        : 'map_study_id%3D'  .$self->map_study_id( 2 )
    ;

    my $qs = join( ';',
        'reference_map_study_id='      .$self->map_study_id( 1 ),
        'reference_map_id='            .$self->map_id( 1 ),
        'comparative_map1='            .$comparative_map_field1,
        'highlight='                   .$self->{'highlight'},
        'show_only_concordant_contigs='.
            $self->{'show_only_concordant_contigs'},
        'flip='                        .$self->flip,
        'font_size='                   .$self->{'font_size'},
    );

    return $qs;
}

#-----------------------------------------------------
sub layout_genetic_map {
    my ( $self, $map_no )   = @_;
    my $data              = $self->features( $map_no );
    my $scale             = $self->scale;
    my $y                 = $self->{'y'} ||  OFFSET;
    my $tick_font         = $self->tick_font;
    my $tick_width        = $self->tick_width;
    my $data_label_offset = $self->data_label_offset;
    my $highlight         = $self->highlight;
    my $x                 = $self->max_x + 60;
    my $start             = $self->start;
    my $end               = $self->end;
    my $prev_x            = $x;
    my @columns           = (); # for laying out the maps
    my $column_width      = $tick_width + $tick_font->height + 10;
    my $no_comp_maps      = scalar keys %$data;
    my $map_is_flipped    = $self->flip( $map_no );
    my $map_max_x;
    my $marker_uri        = $self->marker_viewer_uri($self->species($map_no));
    my ( @map_data, @marker_data ); # to return

    #
    # For the same-map features
    #
    my %all_feature_ids = ();
    for my $map_id ( keys %$data ) {
        my $map = $data->{ $map_id };
        $all_feature_ids{ $_->{'feature_id'} } = undef for
            map { @{ $map->{'features'}{ $_ } } } 
            keys %{ $map->{'features'} }
        ;
    }

    for my $map ( 
        sort { $b->{'no_anchored_features'} <=> $a->{'no_anchored_features'} }
        values %$data 
    ) {
        my @sorted_positions = sort { $a <=> $b } keys %{ $map->{'features'} };
           @sorted_positions = reverse @sorted_positions if $map_is_flipped;
        my $first_position   = $sorted_positions[  0 ];
        my $last_position    = $sorted_positions[ -1 ];

        my @sorted_reference_positions = 
            sort { $a <=> $b } 
            map  { $_->{'reference_map_position'} }
            grep { defined $_->{'reference_map_position'} }
            map  { @$_ } 
            values %{ $map->{'features'} }
        ;
        my $first_ref_position = $sorted_reference_positions[  0 ];
        my $last_ref_position  = $sorted_reference_positions[ -1 ];
        
        my $real_length     = abs( $last_position - $first_position );
        my $cm_length       = $real_length < MIN_MAP_CM_LENGTH ?
                              MIN_MAP_CM_LENGTH : $real_length ;
        my $min_y           = $self->reference_y_position(
                                  $first_ref_position, $map_no - 1
                              );
        my $max_y           = $self->reference_y_position(
                                  $last_ref_position, $map_no - 1
                              );
        my $half_way        = ( $min_y + $max_y ) / 2;
        my $line_len        = $cm_length * $scale;
        my $line_start      = $half_way - ( $line_len / 2 );
        my $line_end        = $half_way + ( $line_len / 2 );

        #
        # Place the tag centered on the line
        #
        my $tag             = $map->{'map_name'} or next;
        my $tag_y           = $half_way + 
                              $tick_font->width * ( length( $tag ) / 2 ); 
        my $tag_top         = $tag_y - 
                              ( $tick_font->width * length( $tag ) ); 
        my $furthest_south  = $tag_y   > $line_end   ? $tag_y   : $line_end;
        my $furthest_north  = $tag_top < $line_start ? $tag_top : $line_start;
        
        #
        # Figure out which column (the x) to place this map in
        #
        my $cur_x;
        my $offset       = 0;
        my $buffer       = 4;
        my $max_distance = 100 * $self->scale;

        for my $i ( 0..$#columns ) {
            my $column       = $columns[$i];
            my $col_x        = $column->{'x'} || $x;
            my $column_is_ok = 1;

            my @taken = sort { $a->[0] <=> $b->[0] } @{$column->{'taken'}};
            for my $i ( 0..$#taken ) {
                my ( $furthest_north_col, $furthest_south_col ) = 
                    @{ $taken[ $i ] };

                #
                # See if we should skip this interval
                #
                if ( ( $furthest_south_col + $buffer ) < $furthest_north ) {
                    next;
                }

                #
                # See if the line will fit directly across
                #
                if ( ( $furthest_north_col - $buffer ) > $furthest_south or
                     ( $furthest_south_col + $buffer ) < $furthest_north 
                ) {
                    last;
                }
                else {
                    $column_is_ok = 0;
                }

                if ( ( $furthest_north_col - $buffer ) > $furthest_south ) {
                    next;
                }

                # 
                # Check the areas above and below
                #
                my $map_length = $furthest_south - $furthest_north;
                my ( $offset_above, $offset_below );
                unless ( $column_is_ok ) {
                    my ( $prev_north_col, $prev_south_col ) = $i > 0 
                        ? @{ $taken[ $i - 1 ] } 
                        : ( $self->min_y, $self->min_y );#( $y, $y );
                    my $space_above = $furthest_north_col - $prev_south_col;

                    if ( $space_above > ( $map_length + $buffer ) ) {
                        $offset_above = ($furthest_north_col - $buffer) -
                                         $furthest_south; 
                        $offset_above = undef
                            unless abs ( $offset_above ) <= $max_distance;

                    }

                    my ( $next_north_col, $next_south_col ) = $i < $#taken 
                        ? @{ $taken[ $i + 1 ] } 
                        : ($self->max_y, $self->max_y);
                    my $space_below = $next_north_col - $furthest_south_col;

                    if ( $space_below > ( $map_length + $buffer ) ) {
                        $offset_below = ($furthest_south_col + $buffer) -
                                        $furthest_north; 
                        $offset_below = undef
                            unless abs ( $offset_below ) <= $max_distance;
                    }

                    # 
                    # Take the closest open region (above or below)
                    #
                    if ( defined $offset_above && defined $offset_below ) {
                        $offset = abs($offset_above) < abs($offset_below)
                                  ? $offset_above : $offset_below;
                    }
                    elsif ( defined $offset_above ) {
                        $offset = $offset_above;
                    }
                    elsif ( defined $offset_below ) {
                        $offset = $offset_below;
                    }
                    else {
                        $offset = 0;
                    }

                    if ( $offset ) {
                        $furthest_north += $offset;
                        $furthest_south += $offset;
                        $column_is_ok    = 1;
                        last;
                    }
                }

                last if $column_is_ok;
            }

            if ( $column_is_ok ) {
                $cur_x = $col_x;
                push @{ $column->{'taken'} }, [
                    $furthest_north, $furthest_south,
                ];
                last;
            }
            else {
                next;
            }
        }

        #
        # If nothing suitable was found, go to the next column
        #
        unless ( $cur_x ) {
            my $next_col   = $#columns + 1;
            $cur_x         = $x + ( $column_width * $next_col );
            push @columns, {
                x     => $cur_x,
                taken => [ [ $furthest_north, $furthest_south ] ],
            };
        }

        #
        # Draw the map
        #
        my $x_diff = 2;
        my $furthest_point_east = $cur_x + $tick_width * .75;

        if ( $no_comp_maps == 1 ) {
            #
            # Put the species and linkage group at the top.
            #
            my $species       = $self->species( $map_no );
            my $linkage_group = $self->map_name( $map_no );

            for my $string ( $species, $linkage_group ) {
                push @marker_data, [
                    STRING,
                    $tick_font,
                    $x - ( $tick_font->width * length($string) ) / 2,
                    $line_start,
                    $string,
                    'black',
                ];

                #
                # Push the map down a bit.
                #
                $_ += $tick_font->height + 2 for $line_start, $line_end;
                $furthest_south  = $line_end if $line_end > $furthest_south;
            }
        }

        push @marker_data, [
            RECTANGLE, 
            $cur_x - $x_diff,
            $line_start + $offset, 
            $cur_x + $x_diff,
            $line_end + $offset, 
            'black'
        ];

        push @marker_data, [
            FILLED_RECT, 
            $cur_x - $x_diff,
            $line_start + $offset, 
            $cur_x + $x_diff,
            $line_end + $offset, 
            $self->map_color
        ];

        #
        # If there are many maps, place a label beside each.
        #
        unless ( $no_comp_maps == 1 ) {
            my $tag_x            = $cur_x + $tick_width * .75;
            $furthest_point_east = $tag_x + $tick_font->height;
            push @marker_data, [
                STRING_UP, 
                $tick_font, 
                $tag_x, 
                $tag_y + $offset, 
                $tag, 
                'blue'
            ];
            my $furthest_east = $tag_x + $tick_font->height;
            $map_max_x = $furthest_east if $furthest_east > $map_max_x;
        }
        else {
            #
            # If there's only one map, then place a "Flip" tag underneath
            #
            my $flip_tag   = 'Flip';
            my $tag_start_x = $cur_x - 
                ( ( length( $flip_tag ) * $tick_font->width ) / 2 );
            my $tag_start_y = $furthest_south + $buffer * 4;

            push @marker_data, [
                STRING, 
                $tick_font,
                $tag_start_x,
                $tag_start_y,
                $flip_tag,
                'green'
            ];

            my $bottom = $tag_start_y +  $tick_font->height + $buffer;
            $self->max_y( $bottom );
            my @coords = (
                $tag_start_x - $buffer,
                $tag_start_y - $buffer,
                $tag_start_x + (length($flip_tag)*$tick_font->width) + $buffer,
                $bottom,
            );

            push @marker_data, [
                RECTANGLE, 
                @coords,
                'grey'
            ];

            #
            # If the map is currently flipped, then take 
            # it out of the flip string.
            #
            my %flipped_maps = map{ $_, 1 } $self->flip, $map_no;
            delete $flipped_maps{ $map_no } if $map_is_flipped;
            my $flip_string  = join( ',', map{ $_ || () } keys %flipped_maps );
            push @map_data, {
                coords => [ @coords ],
                uri    => URIS->{'map_viewer'}.'?'.
                    join( ';', 
                        'reference_map_study_id='   .$self->map_study_id( 1 ),
                        'reference_map_id='         .$self->map_id( 1 ),
                        'start='                    .$self->start,
                        'end='                      .$self->end,
                        'comparative_map1=genetic_map_id%3d'.$map->{'map_id'},
                        'highlight='                .$self->highlight_string,
                        'flip='                     .$flip_string,
                        'font_size='                .$self->{'font_size'},
                    ),
                alt    => 'Flip map',
            };
        }

        #
        # Make the map clickable
        #
        my @bounding_coords = (
            $cur_x - $x_diff * 2,
            $furthest_north,
            $furthest_point_east,
            $furthest_south,
        );

        #
        # When the user is viewing an entire map study as 
        # the comparative map, then clicking on a map will
        # show just the one map in relation to the reference 
        # map.  When only a single map is the second map,
        # then take the user to a full view of that map.
        #
        my $map_id = $map->{'map_id'};
        my $uri    = ( $no_comp_maps == 1 ) 
            ?
                URIS->{'map_viewer'}.'?'.
                join( ';',
                    'reference_map_study_id='   .$self->map_study_id( 2 ),
                    'reference_map_id='         .$map_id,
                    'highlight='                .$self->highlight_string,
                )
            :
                URIS->{'map_viewer'}.'?'.
                join( ';', 
                    'reference_map_study_id='           .$self->map_study_id(1),
                    'reference_map_id='                 .$self->map_id(1),
                    'start='                            .$first_ref_position,
                    'end='                              .$last_ref_position,
                    'comparative_map1=genetic_map_id%3d'.$map_id,
                    'highlight='                .$self->highlight_string,
                )
        ;

        push @map_data, {
            coords => [ @bounding_coords ],
            uri    => $uri,
            alt    => "View: $tag",
        };
        $prev_x = $furthest_point_east + 10;

        #
        # Optional highlighting of the map itself (kinda silly)
        #
        my $lc_name = lc $map->{'map_name'};
        if ( exists $highlight->{$lc_name} ) {
            push @marker_data, [
                FILLED_RECT,
                @bounding_coords,
                DEFAULT->{'highlight_color'}
            ];
        }

        #
        # If there's only one map, show some intervals (in cm)
        #
        if ( $no_comp_maps == 1 ) {
            my $tick_pos_interval = $self->tick_mark_interval( $cm_length );
            my $tick_x1           = $x - $tick_width * .5;
            my $tick_x2           = $x + $tick_width * .125;

            my @tick_positions = $map_is_flipped
                ? reverse int $last_position .. int $first_position
                : int $first_position .. int $last_position
            ;

            for my $tick_pos ( @tick_positions ) {
                if ( $map_is_flipped ) {
                    next if $tick_pos < $last_position || 
                            $tick_pos > $first_position;
                }
                else {
                    next if $tick_pos < $first_position || 
                            $tick_pos > $last_position;
                }

                next unless $tick_pos % $tick_pos_interval == 0;
                my $relative_distance = 
                    (abs($tick_pos-$first_position)/$real_length) * $cm_length;
                my $cur_y = $line_start + ( $relative_distance * $scale );
                my $tag_x = $cur_x - $tick_width - 5;
                my $tag_y = $cur_y + (($tick_font->width*length($tick_pos))/2); 

                push @marker_data, [
                    STRING_UP, 
                    $tick_font, 
                    $tag_x, 
                    $tag_y, 
                    $tick_pos, 
                    DEFAULT->{'interval_tick_color'},
                ];

                push @marker_data, [
                    LINE, 
                    $tick_x1,
                    $cur_y, 
                    $tick_x2,
                    $cur_y, 
                    DEFAULT->{'interval_tick_color'},
                ];
            }
        }

        #
        # Place a tick mark for each feature on the map
        #
        my @keys              = sort { $a <=> $b } keys %{ $map->{'features'} };
        @keys                 = reverse @keys if $map_is_flipped;
        my $midpoint          = int $#keys / 2;
        my $midpoint_value    = $keys[ $midpoint ];
        my @ordered_keys      = map { $keys[$_] } 
                                reverse( 0..$midpoint ), 
                                $midpoint + 1 .. $#keys;
        my $max_keys          = $#ordered_keys;
        my ( $prev_y, $mid_y );

        for my $i ( 0 .. $max_keys ) {
            my $position          = $ordered_keys[ $i ];
            my $relative_distance = $real_length > 0
                                    ?abs($position-$first_position)/$real_length
                                    : .5;
            my $cm_pos            = $cm_length * $relative_distance;
            my $x1                = $cur_x - $tick_width*.4;
            my $x2                = $cur_x + $tick_width*.4;
            my $tick_y            = $line_start + ( $cm_pos * $scale );
            $prev_y               = $tick_y unless defined $prev_y;
            $prev_y               = $tick_y if $position == $midpoint_value;

            push @marker_data, [
                LINE, 
                $x1, 
                $tick_y + $offset, 
                $x2, 
                $tick_y + $offset, 
                'black'
            ];

            $self->reference_y_position( $position, $map_no, $tick_y );

            for my $feature ( 
                sort { 
                    $b->{'reference_map_position'} 
                    <=> 
                    $a->{'reference_map_position'} 
                }
                @{ $map->{'features'}{$position} } 
            ) {
                my $feature_id = $feature->{'feature_id'};

                #
                # If there's only one map to display, then show all the
                # features
                #
                unless ( $no_comp_maps == 1 ) { 
                    $self->feature_position(
                        feature_id => $feature_id,
                        map_no     => $map_no,
                        position   => {
                            tag_coords => [ ],
                            link_line  => [ 
                                $cur_x - $tick_width*.75, 
                                $tick_y + $offset 
                            ],
                        }
                    );
                    next;
                }

                #
                # We'll skip any tags when they are too
                # far from their tick mark, so figure out the y
                # coordinate and skip -- unless the feature has
                # a relationship on the reference map.
                #
                my $tag_y = $tick_y;
#                warn "pos = $position, prev_y = $prev_y\n";
#                if ( $position <= $midpoint_value ) {
#                    $tag_y-- while $tag_y >= $prev_y;
#                }
#                else {
#                    $tag_y++ while $tag_y <= $prev_y;
#                    $tag_y++ while $tag_y - $tick_font->height <= $mid_y;
#                }


                if ( $map_is_flipped ) {
                    if ( $position >= $midpoint_value ) {
                        $tag_y-- while $tag_y >= $prev_y;
                    }
                    else {
                        $tag_y++ while $tag_y <= $prev_y;
                        $tag_y++ while $tag_y - $tick_font->height <= $mid_y;
                    }
                }
                else {
                    if ( $position <= $midpoint_value ) {
                        $tag_y-- while $tag_y >= $prev_y;
                    }
                    else {
                        $tag_y++ while $tag_y <= $prev_y;
                        $tag_y++ while $tag_y - $tick_font->height <= $mid_y;
                    }
                }

                $mid_y = $tag_y if $position == $midpoint_value
                    && !defined $mid_y;
                my $name    = $feature->{'map_position_name'};
                my $lc_name = lc $name;

                #
                # Figure out where the tag starts
                #
                my $tag     = $name;
                my $tag_x   = $x + $data_label_offset;

                #
                # Remember these for future use
                #
                my $tag_end = $tag_x + ( $tick_font->width * length( $tag ) );
                my @tag_coords = (
                    $tag_x - 3, 
                    $tag_y - ( $tick_font->height/2 ),
                    $tag_end,
                    $tag_y + ( $tick_font->height/2 ),
                );

                #
                # Remember the position of the tick
                # for linking those with corresponding features
                #
                my $link_line_x = $x - $tick_width*.75;

                #
                # Use the correspondences to see if this feature is 
                # related to anything else on this same map.
                #
                my @correspondences = 
                    $self->feature_correspondences( $feature_id );
                for my $id ( @correspondences ) {
                    next unless exists $all_feature_ids{ $id };
                    $feature->{'is_related'} = 1;
                    $all_feature_ids{ $feature_id } = [ $tag_end, $tag_y ];
                }

#                unless ( $feature->{'is_related'} ||
#                         $feature->{'reference_map_position'} ||
#                         exists $highlight->{$lc_name} ||
#                         $self->feature_is_related( $feature->{'feature_id'} )
#                ) {
#                    next if abs( $tag_y - $tick_y ) >
#                    MAX_PIXELS_FROM_TICK;
#                }

                #
                # Highlighting the user's choices
                #
                my ( $color, $line_color );
                if ( exists $highlight->{$lc_name} ) {
                    $color = $self->feature_color( 
                        $feature_id, DEFAULT->{'highlight_color'} 
                    );
                    $line_color = DEFAULT->{'highlight_box_color'};
                    $feature->{'is_highlighted'} = 1;
                }
                elsif ( 
                    $feature->{'is_related'} ||
                    $feature->{'reference_map_position'} ||
                    $self->feature_is_related( $feature->{'feature_id'} )
                ) {
#                    $color = $self->feature_color( $feature_id );
                    $color = '';
                }

                #
                # Criteria for dropping tags:
                # 1) Never drop out a highlighted marker
                # 2) Drop a highlighted tag that's too far away
                # 3) Drop any non-colored tag that's a very small distance away
                #    (to make room for the others)
                #
                my $distance_from_tag = abs( $tag_y - $tick_y );
                my $skip_tag;
                if ( 
                    $feature->{'is_highlighted'} ||
                    ( 
                        defined $color && 
                        $distance_from_tag < MAX_PIXELS_FROM_TICK 
                    )
                ) {
                    push @marker_data, [
                        FILLED_RECT,
                        @tag_coords,
                        $color
                    ] if $color;

                    push @marker_data, [
                        RECTANGLE,
                        @tag_coords,
                        $line_color
                    ] if $line_color;

                    $skip_tag = 0;
                } 
                elsif ( !$color && $distance_from_tag < 10 ) { 
                    $skip_tag = 0;
                }
                else { 
                    $skip_tag = 1;
                }

                $self->feature_position(
                    feature_id      => $feature_id,
                    map_no          => $map_no,
                    position        => {
                        tag_coords      => $skip_tag ? [] : [@tag_coords],
                        tag_end         => $skip_tag ? [] : [$tag_end, $tag_y],
                        link_line       => [ $link_line_x, $tick_y ],
                    },
                );

                next if $skip_tag;

                $map_max_x = $tag_end if $tag_end > $map_max_x;

                #
                # The feature tag
                #
                push @marker_data, [
                    STRING, 
                    $tick_font, 
                    $tag_x,
                    $tag_y - ( $tick_font->height/2 ), 
                    $tag, 
                    'blue'
                ];

                #
                # The map coords
                #
                push @map_data, {
                    coords => [ @tag_coords ],
                    uri    => $marker_uri.$feature->{'feature_link_out'},
                    alt    => "View: $name ($position)",
                };

                #
                # The line linking the tag and the tick mark
                #
                $color ||= 'gray';
                push @marker_data, $self->_indicator_line(
                    $x2 + 5,
                    $tick_y, 
                    $tag_x - 5,
                    $tag_y, 
                    $line_color || $color
                );

                #
                # Remember the last y position of a feature tag
                #
#                $prev_y = $position <= $midpoint_value
#                    ? $tag_y - $tick_font->height
#                    : $tag_y + $tick_font->height;

                if ( $map_is_flipped ) {
                    $prev_y = $position >= $midpoint_value
                        ? $tag_y - $tick_font->height
                        : $tag_y + $tick_font->height;
                }
                else {
                    $prev_y = $position <= $midpoint_value
                        ? $tag_y - $tick_font->height
                        : $tag_y + $tick_font->height;
                }

                $furthest_north = $prev_y if $prev_y < $furthest_north;
                $furthest_south = $prev_y if $prev_y > $furthest_south;
            }

            $self->max_y( $furthest_south );
            $self->min_y( $furthest_north );
        }
    }

    #
    # For each feature that has a correspondence,
    # draw a line connecting it to its cousin
#    my $genetic_positions = $self->positions;
#    for my $feature_id ( keys %$genetic_positions ) {
#        my @gpositions = @{ $genetic_positions->{ $feature_id } };
#        
#        for my $data1 ( @gpositions ) {
#            #next unless $data1->{'tag_is_present'};
#            my ($x1, $y1) = @{ $data1->{'link_line'} };
#            my $map_no1   = $data1->{'map_no'};
#
#            for my $data2 ( @{ $positions{$feature_id} } ) {
#                #
#                # This data draws the lines between related feature.
#                #
#                my ($x2, $y2) = @{ $data2->{'link_line'} };
#                my $map_no2   = $data2->{'map_no'} || 2;
#                my $map_no    = $map_no1 == $map_no2 ? $map_no1 : 0;
#                my $color     = $self->feature_color( $feature_id );
#
#                push @marker_data, $self->_indicator_line(
#                    $x1, $y1, $x2, $y2, $color, $map_no
#                );
#
#                #
#                # This highlights the tags between related markers.
#                #
#                push @marker_data, [
#                    FILLED_RECT,
#                    @{ $data1->{'tag_coords'} },
#                    $color
#                ];
#            }
#        }
#    }
    
    my $reference_map_no = $map_no - 1;
    my $reference_link   = $map_no == 2 ? 'link_line' : 'tag_end';
    for my $feature_id ( keys %all_feature_ids ) {
        my $color            = 'lightblue';#$self->feature_color( $feature_id );
        my @correspondences  = $self->feature_correspondences( $feature_id );

        my @current_map_positions   = $self->feature_position(
            feature_id => $feature_id,
            map_no     => $map_no,
        );

        for my $id ( $feature_id, @correspondences ) {
            my @reference_map_positions = $self->feature_position(
                feature_id => $id,
                map_no     => $reference_map_no,
            ) or next;

            for my $reference_map_position ( @reference_map_positions ) {
                my ($x1, $y1) = @{$reference_map_position->{ $reference_link }}
                    or next;

                for my $current_map_position ( @current_map_positions ) {
                    my ($x2, $y2) = @{ $current_map_position->{'link_line'} }
                        or next;

                    push @marker_data, $self->_indicator_line(
                        $x1, $y1, $x2, $y2, $color, $map_no
                    );
                }
            }
        }
    }

    #
    # Same-map relationships
    #
    @columns                = ();
    $column_width           = 4;
    my %done                = ();
    my %col_x_by_feature_id = ();
    my $max_x               = $map_max_x + $column_width;

    for my $feature_id ( keys %all_feature_ids ) {
        my @correspondences = 
            grep { defined $all_feature_ids{ $_ } }
            $self->feature_correspondences( $feature_id );

        next unless @correspondences;

        my @position1 = $self->feature_position( 
            feature_id => $feature_id,
            map_no     => $map_no,
        ) or next;
#        warn "pos1 =\n", Dumper( @position1 ), "\n";

        for my $position1 ( @position1 ) {
            my @tag_end1 = @{ $position1->{'tag_end'} } or next;
            my $start_x  = $tag_end1[0] + 1;
            my $start_y  = $tag_end1[1];

            for my $id ( @correspondences ) {
#                next if $done{ $feature_id }{ $id };
                my @position2 = $self->feature_position( 
                    feature_id => $id,
                    map_no     => $map_no,
                ) or next;
#                warn "pos2 =\n", Dumper( @position2 ), "\n";

                for my $position2 ( @position2 ) {
                    my @tag_end2       = @{ $position2->{'tag_end'} } or next;
                    my $end_x          = $tag_end2[0] + 1;
                    my $end_y          = $tag_end2[1];
                    my $color          = 'gray';
                    my $furthest_south = $start_y > $end_y ? $start_y : $end_y;
                    my $furthest_north = $start_y < $end_y ? $start_y : $end_y;

                    #
                    # figure out which "column" to put the vertical line in
                    #
                    my ( $column_index, $furthest_north_col, $furthest_south_col );
                    my $cur_x = $col_x_by_feature_id{ $feature_id };
                    unless ( $cur_x ) {
                        for my $i ( 0..$#columns ) {
                            my $column          = $columns[ $i ];
                            my $col_x           = $column->{'x'}     || $max_x;
                            $furthest_north_col = $column->{'min_y'} ||  undef;
                            $furthest_south_col = $column->{'max_y'} ||  undef;

                            #
                            # first, see if there's anything in this column
                            # if not, just use it.
                            #
                            unless ( defined $furthest_north_col && 
                                     defined $furthest_south_col
                            ) {
                                $cur_x        = $col_x;
                                $column_index = $i;
                                last;
                            }

                            #
                            # now see if our line will fit above
                            # if not, see if our line will fit below
                            #
                            if ( $furthest_north_col >= $furthest_south or
                                 $furthest_south_col <= $furthest_north ) {
                                $cur_x        = $col_x;
                                $column_index = $i;
                                last;
                            }

                            $column_index = $i;
                        }

                        #
                        # if nothing suitable was found, go to the next column
                        #
                        unless ( $cur_x ) {
                            $column_index += 1;
                            $cur_x = $max_x + ( $column_width * $column_index );
                        }
                    }

                    push @marker_data, [ 
                        LINE, 
                        $start_x, 
                        $start_y, 
                        $cur_x,
                        $start_y, 
                        $color 
                    ];
                    push @marker_data, [ 
                        LINE, 
                        $cur_x,
                        $start_y, 
                        $cur_x,
                        $end_y, 
                        $color 
                    ];
                    push @marker_data, [ 
                        LINE, 
                        $cur_x,
                        $end_y, 
                        $end_x, 
                        $end_y, 
                        $color 
                    ];

                    #
                    # figure out the furthest point north and
                    # south that we just drew.
                    #
                    if ( defined $furthest_north_col ) {
                        $furthest_north_col = $furthest_north if
                            $furthest_north < $furthest_north_col;
                    }
                    else {
                        $furthest_north_col = $furthest_north;
                    }

                    if ( defined $furthest_south_col ) {
                        $furthest_south_col = $furthest_south if
                            $furthest_south > $furthest_south_col;
                    }
                    else {
                        $furthest_south_col = $furthest_south;
                    }

                    #
                    # update the @columns array
                    #
                    $columns[ $column_index ] = {
                        x     => $cur_x,
                        min_y => $furthest_north_col,
                        max_y => $furthest_south_col,
                    };

#                    $done{ $feature_id }{ $id }++;
#                    $done{ $id }{ $feature_id }++;
                    $col_x_by_feature_id{ $feature_id } = $cur_x;
                    $col_x_by_feature_id{ $id }         = $cur_x;
                    $map_max_x = $cur_x if $cur_x > $map_max_x;
                }
            }
        }
    }

    $self->max_x( $map_max_x );

    return { data => \@marker_data, map => \@map_data };
}

#-----------------------------------------------------
sub layout_physical_map {
    my ( $self, $map_no )   = @_;
    my $data              = $self->features( $map_no ) or return;
    my $scale             = $self->scale;
    my $y                 = $self->{'y'}      ||  OFFSET;
    my $tick_font         = $self->tick_font;
    my $tick_width        = $self->tick_width;
    my $data_label_offset = $self->data_label_offset;
    my $highlight         = $self->highlight;
    my $x                 = $self->max_x + 30;
    my $start             = $self->start;
    my $end               = $self->end;
    my $prev_x            = $x;
    my %positions         = (); # for the positions of markers
    my ( @map_data, @concordant_contigs, @discordant_contigs ); # to return

#    warn "data =\n", Dumper( $data ), "\n";

    my $comparative_map_field1 = 'map_study_id%3D'.$self->map_study_id( 2 );

    my $qs = 'highlight='.$self->{'highlight'}.
             ';image_height='.$self->{'image_height'}.
             ';font_size='.$self->{'font_size'};

    #
    # Sort the contigs by position of first marker
    # Scratch that -- Must be sure that the contigs
    # are ALWAYS sorted in the same order, otherwise
    # the map coords get way off on the two separate
    # calls (one for the graphing data, the other for
    # the map coordinates that were drawn).
    #
    my ( @columns, $last_concordant_column );
    my $no_concordant_contigs = grep { !$_->{'is_discordant'} } values %$data;
    my $column_width = $tick_width + $tick_font->height + 10;
    for my $contig ( 
        sort { 
            $a->{'is_discordant'}          <=> $b->{'is_discordant'} 
            ||
            $b->{'no_concordant_features'} <=> $a->{'no_concordant_features'}
        }
        values %$data 
    ) {
        #
        # First have to figure the y-coords of where the 
        # line should be drawn, placing the contig near to the 
        # features anchored on it.  Skip the discordant features
        # (those found on the contig but not from the genetic map 
        # to which we're currently comparing this).
        #
        my @sorted_positions =
            sort { $a <=> $b }
            map  { $_ >= $start && $_ <= $end  ? $_ : () }
            map  { defined $_->{'genetic_position'}
                   ? $_->{'genetic_position'} : ()
            }
            grep { !$_->{'is_discordant'} }
            @{ $contig->{'features'} };

        #
        # If there are no concordant features, then just use
        # the ones anchored to the genetic map.
        #
        unless ( @sorted_positions ) {
            @sorted_positions =
                sort { $a <=> $b }
                map  { $_ >= $start && $_ <= $end  ? $_ : () }
                map  { defined $_->{'genetic_position'}
                       ? $_->{'genetic_position'} : ()
                }
                @{ $contig->{'features'} };
        }

        #
        # Position the line half-way b/w the first and last markers
        #
        my $is_discordant       = $contig->{'is_discordant'};
        my $first_position      = $sorted_positions[  0 ];
        my $last_position       = $sorted_positions[ -1 ];
        my $cM_length           = $is_discordant
                                  ? MIN_MAP_PIXEL_LENGTH / $scale
                                  : ($last_position-$first_position)*$scale < 
                                    MIN_MAP_PIXEL_LENGTH
                                    ? MIN_MAP_PIXEL_LENGTH / $scale
                                    : ( $last_position - $first_position ) >= 
                                       MIN_MAP_CM_LENGTH 
                                       ? $last_position - $first_position
                                       : MIN_MAP_CM_LENGTH;
        my $min_y               = $self->reference_y_position(
                                      $first_position, $map_no - 1 
                                  );
        my $max_y               = $self->reference_y_position(
                                      $last_position,  $map_no - 1
                                  );
        my $line_len            = $cM_length * $scale;
        my $half_way            = ( $min_y + $max_y ) / 2;
        my $line_start          = $half_way - ( $line_len / 2 );
        my $line_end            = $half_way + ( $line_len / 2 );

        #
        # Place the tag centered on the line
        #
        my $tag            = $contig->{'map_name'} || 'n/a';
        my $tag_y          = $half_way + 
                             $tick_font->width * ( length( $tag ) / 2 ); 
        my $tag_top        = $tag_y - ( $tick_font->width * length( $tag ) ); 
        my $furthest_south = $tag_y   > $line_end   ? $tag_y   : $line_end;
        my $furthest_north = $tag_top < $line_start ? $tag_top : $line_start;

        #
        # Figure out which column (the X) to place this contig in
        #
        my $cur_x;
        my $offset = 0;
        my $start_col = $is_discordant ? $last_concordant_column : 0;
        my $buffer = 4;
        my $max_distance = 100 * $self->scale;

        for my $i ( $start_col..$#columns ) {
            my $column       = $columns[$i];
            my $col_x        = $column->{'x'}     || $x;
            my $column_is_ok = 1;

            my @taken = sort { $a->[0] <=> $b->[0] } @{$column->{'taken'}};
            for my $i ( 0..$#taken ) {
                my ( $furthest_north_col, $furthest_south_col ) = 
                    @{ $taken[ $i ] };

                #
                # See if we should skip this interval
                #
                if ( ( $furthest_south_col + $buffer ) < $furthest_north ) {
                    next;
                }

                #
                # See if the line will fit directly across
                #
                if ( ( $furthest_north_col - $buffer ) > $furthest_south or
                     ( $furthest_south_col + $buffer ) < $furthest_north 
                ) {
                    last;
                }
                else {
                    $column_is_ok = 0;
                }

                if ( ( $furthest_north_col - $buffer ) > $furthest_south ) {
                    next;
                }

                # 
                # Check the areas above and below
                #
                my $contig_length = $furthest_south - $furthest_north;
                my ( $offset_above, $offset_below );
                unless ( $column_is_ok ) {
                    my ( $prev_north_col, $prev_south_col ) = $i > 0 
                        ? @{ $taken[ $i - 1 ] } 
                        : ( $self->min_y, $self->min_y );#( $y, $y );
                    my $space_above = $furthest_north_col - $prev_south_col;

                    if ( $space_above > ( $contig_length + $buffer ) ) {
                        $offset_above = ($furthest_north_col - $buffer) -
                                         $furthest_south; 
                        $offset_above = undef
                            unless abs ( $offset_above ) <= $max_distance;

                    }

                    my ( $next_north_col, $next_south_col ) = $i < $#taken 
                        ? @{ $taken[ $i + 1 ] } 
                        : ($self->max_y, $self->max_y);
                    my $space_below = $next_north_col - $furthest_south_col;

                    if ( $space_below > ( $contig_length + $buffer ) ) {
                        $offset_below = ($furthest_south_col + $buffer) -
                                        $furthest_north; 
                        $offset_below = undef
                            unless abs ( $offset_below ) <= $max_distance;
                    }

                    # 
                    # Take the closest open region (above or below)
                    #
                    if ( defined $offset_above && defined $offset_below ) {
                        $offset = abs($offset_above) < abs($offset_below)
                                  ? $offset_above : $offset_below;
                    }
                    elsif ( defined $offset_above ) {
                        $offset = $offset_above;
                    }
                    elsif ( defined $offset_below ) {
                        $offset = $offset_below;
                    }
                    else {
                        $offset = 0;
                    }

                    if ( $offset ) {
                        $furthest_north += $offset;
                        $furthest_south += $offset;
                        $column_is_ok    = 1;
                        last;
                    }
                }

                last if $column_is_ok;
            }

            if ( $column_is_ok ) {
                $cur_x = $col_x;
                push @{ $column->{'taken'} }, [
                    $furthest_north, $furthest_south,
                ];
                last;
            }
            else {
                next;
            }

            $last_concordant_column = $i unless $is_discordant;
        }

        #
        # If nothing suitable was found, go to the next column
        #
        unless ( $cur_x ) {
#            my $next_col = 
#                $is_discordant && $no_concordant_contigs > 0
#                    ? $#columns + 4
#                        : $last_concordant_column > 0
#                            ? $last_concordant_column + 1 
#                                : $#columns + 1
#            ;
            my $next_col = $is_discordant && $no_concordant_contigs > 0
                ? $#columns + 2 : $#columns + 1
            ;
            $cur_x         = $x + ( $column_width * $next_col );
            push @columns, {
                x     => $cur_x,
                taken => [ [ $furthest_north, $furthest_south ] ],
            };
            $last_concordant_column++ unless $is_discordant;
        }

        #
        # Make a line for the contig and write its name beside it
        #
        my $tag_x        = $cur_x + $tick_width * .75;
        my $contig_color = $is_discordant 
            ? DEFAULT->{'discordant_contig_box_color'}
            : DEFAULT->{'contig_box_color'}
        ;
        my $tag_color   = $is_discordant
            ? DEFAULT->{'discordant_contig_tag_color'}
            : DEFAULT->{'tag_color'}
        ;
        my $contig_data = $is_discordant 
            ? \@discordant_contigs 
            : \@concordant_contigs
        ;

        push @$contig_data, [
            STRING_UP, 
            $tick_font, 
            $tag_x, 
            $tag_y + $offset, 
            $tag, 
            $tag_color,
        ];

        push @$contig_data, [
            RECTANGLE, 
            $cur_x - $tick_width/8, 
            $line_start + $offset, 
            $cur_x + $tick_width/8, 
            $line_end + $offset, 
            $contig_color,
        ];

        push @$contig_data, [
            FILLED_RECT, 
            $cur_x - $tick_width/8, 
            $line_start + $offset, 
            $cur_x + $tick_width/8, 
            $line_end + $offset, 
            DEFAULT->{'contig_fill_color'},
        ];

        my @bounding_coords = (
            $cur_x - $tick_width / 2,
            $furthest_north,
            $tag_x + $tick_font->height,
            $furthest_south,
        );

        my $lc_name = lc $contig->{'map_name'};
        if ( exists $highlight->{$lc_name} ) {
            push @$contig_data, [
                FILLED_RECT,
                @bounding_coords,
                DEFAULT->{'highlight_color'}
            ];
        }

        #
        # For the clickable area coordinates, figure out 
        # the area that covers the line and the tag
        #
        my $map_id = $contig->{'physical_map_id'};
        push @map_data, {
            coords => [ @bounding_coords ],
            uri    => URIS->{'contig_viewer'}.
                      "?physical_map_id=$map_id".
                      ";genetic_map_id=".$self->map_id( 1 ).
                      ";$qs",
            alt    => "View: $tag",
        };
        $prev_x = $tag_x + $tick_font->height + 10;


        #
        # Used to draw the contigs the actual physical
        # distance from start to end.  Now we're going to draw
        # them the cM length covered on the genetic map.
        #
        my $physical_length = 
            $contig->{'contig_stop'} - $contig->{'contig_start'};

        #
        # Place a tick mark for each feature on the contig
        #
        for my $feature ( @{ $contig->{'features'} } ) {
            my $position          = $feature->{'start_band'};
            my $relative_distance = ( $position - $contig->{'contig_start'} )
                                    / $physical_length;
            my $feature_id        = $feature->{'feature_id'};
            my $color             = $feature->{'is_discordant'}
                ? DEFAULT->{'discordant_contig_feature_color'}
                : DEFAULT->{'concordant_contig_feature_color'}
            ;
            my $cm_pos            = $cM_length * $relative_distance;
            my $x1                = $cur_x - $tick_width * .4;
            my $x2                = $cur_x + $tick_width * .4;
            my $y1                = $contig->{'is_transposed'}
                ? $line_end   - ( $cm_pos * $scale )
                : $line_start + ( $cm_pos * $scale );

            push @$contig_data, [
                LINE, 
                $x1, 
                $y1 + $offset, 
                $x2, 
                $y1 + $offset, 
                $color
            ];

            push @{ $positions{ $feature_id } }, {
                map_no     => 2,
                tag_coords => [ ],
                link_line  => [ $cur_x - $tick_width * .75, $y1 + $offset ],
            };
        }

        $self->max_x( $tag_x + $tick_font->height );
        $self->max_y( $furthest_south );
        $self->min_y( $furthest_north );
    }

    #
    # For each feature that has a correspondence,
    # draw a line connecting it to its cousin
    #
    for my $feature_id ( keys %positions ) {
        my @physical_positions = @{ $positions{ $feature_id } };
        my @genetic_positions  = $self->feature_position( 
            feature_id => $feature_id, 
            map_no     => $map_no - 1 
        );
        
        for my $data1 ( @physical_positions ) {
            my ($x1, $y1) = @{ $data1->{'link_line'} } or next;
            my $map_no1   = $data1->{'map_no'};

            for my $data2 ( @genetic_positions ) {
                #
                # This data draws the lines between related features.
                #
                my ($x2, $y2) = @{ $data2->{'link_line'} } or next;
                my $map_no2   = $data2->{'map_no'} || 2;
                my $map_no    = $map_no1 == $map_no2 ? $map_no1 : 0;
#                my $color     = $self->feature_color( $feature_id );
                my $color     = 'lightblue';
                push @concordant_contigs, $self->_indicator_line(
                    $x1, 
                    $y1, 
                    $x2, 
                    $y2, 
                    $color, 
                    $map_no
                );
            }
        }
    }

    return { 
        data => [ @discordant_contigs, @concordant_contigs ], 
        map  => \@map_data 
    };
}

#-----------------------------------------------------
sub comparative_maps {
    my $self = shift;
    
    my ( @map, @data );
    my $map_no = 2;
    for my $comparative_map ( @{ $self->{'comparative_maps'} } ) {
        my $action   = ( uc $self->map_type( $map_no ) eq GENETIC ) 
                       ? 'layout_genetic_map' : 'layout_physical_map';
        my $returned = $self->$action( $map_no ) or next;
        push @data, @{ $returned->{'data'} };
        push @map , @{ $returned->{'map'}  };
        $map_no++;
    }
    
    return { data => \@data, map => \@map };
}

#-----------------------------------------------------
sub reference_map_id {
    my $self = shift;
    return $self->{'reference_map_id'} || 0;
}

#-----------------------------------------------------
sub map_name {
    my $self   = shift;
    my $map_no = shift || 1;
    my $data   = $self->data( $map_no );
    return $data->{'map_name'};
}

#-----------------------------------------------------
sub start {
    my $self = shift;
    my $arg  = shift;
    $self->{'start'} = $arg if defined $arg;
    return $self->{'start'};
}

1;

#-----------------------------------------------------
# Something there is that doesn't love a wall.
# Robert Frost
#-----------------------------------------------------

=pod

=head1 NAME

CSHL::ComparativeMaps::Drawer::Map - draw comparative maps

=head1 SYNOPSIS

  use CSHL::ComparativeMaps::Drawer::Map;
  my $drawer    = CSHL::ComparativeMaps::Drawer::Map->new( %args );
  $drawer->layout;
  my $data      = $drawer->data;
  my $map_areas = $drawer->image_map_areas;

=head1 DESCRIPTION

Inherits from CSHL::ComparativeMaps::Drawer.  This module draws 
genetic and physical maps.

=head1 AUTHOR

Ken Y. Clark, kclark@logsoft.com

=head1 SEE ALSO

perl(1).

=cut
