package CSHL::ComparativeMaps::Drawer::Contig;

#-----------------------------------------------------
# $Id: Contig.pm,v 1.16 2002/04/17 01:47:42 kclark Exp $
#
# File       : Contig.pm
# Programmer : Ken Y. Clark, kclark@logsoft.com
# Created    : 2001/09/06
# Purpose    : draw a contig next to a genetic map
#-----------------------------------------------------

use strict;
use vars qw( $VERSION );
$VERSION = (qw$Revision: 1.16 $)[-1];

use Data::Dumper;

use CSHL::DB;
use CSHL::ComparativeMaps::Constants;
use CSHL::ComparativeMaps::Drawer;
use base qw( CSHL::ComparativeMaps::Drawer );

#-----------------------------------------------------
sub _init {
    my $self = shift;
    $self->SUPER::_init();

    $self->{'data'} = $self->soap_call(
        method => 'contig_data',
        params => {
            genetic_map_id  => $self->{'genetic_map_id'},
            physical_map_id => $self->{'physical_map_id'},
        }
    );

#    warn "DATA =\n",Dumper($self->{'data'}{'feature_correspondences'}),"\n";

    #
    # Find the longest cM distance in order to determine the scale
    #
    my $genetic_cM_length  = ( $self->map_end(1) - $self->map_begin(1) ) || 0;
    my $start_band         = $self->map_begin(2);
    my $end_band           = $self->map_end(2);
    my $physical_cM_length = $self->bands_to_cM( $end_band - $start_band );
    my $longest_cM         = $physical_cM_length > $genetic_cM_length ?
                             $physical_cM_length : $genetic_cM_length ;

    $longest_cM = MIN_MAP_CM_LENGTH if $longest_cM < MIN_MAP_CM_LENGTH;

    my $image_height = $self->{'image_height'} || DEFAULT->{'image_height'};

    $self->scale( $image_height / $longest_cM );

    $self->no_maps( 2 );

    return $self;
}

#-----------------------------------------------------
sub is_transposed {
    my $self   = shift;
    my $map_no = 2;
    return $self->{'data'}{ $map_no }{'is_transposed'} || 0;
}

#-----------------------------------------------------
sub map_name {
    my $self   = shift;
    my $map_no = shift || 1;
    return $self->{'data'}{ $map_no }{'map_name'} || '';
}

#-----------------------------------------------------
sub map_begin {
    my $self   = shift;
    my $map_no = shift || 1;
    return $self->{'data'}{ $map_no }{'map_begin'} || '';
}

#-----------------------------------------------------
sub map_end {
    my $self   = shift;
    my $map_no = shift || 1;
    return $self->{'data'}{ $map_no }{'map_end'} || '';
}

#-----------------------------------------------------
sub map_label {
    my $self   = shift;
    my $map_no = shift || 1;
    return $self->{'data'}{ $map_no }{'map_label'} || '';
}

#-----------------------------------------------------
sub map_study_id {
    my $self   = shift;
    my $map_no = shift || 1;
    return $self->{'data'}{ $map_no }{'map_study_id'} || '';
}

#-----------------------------------------------------
sub map_type {
    my $self   = shift;
    my $map_no = shift || 1;
    return $self->{'data'}{ $map_no }{'map_type'} || '';
}

#-----------------------------------------------------
sub contig {
#
# Draws the individual contig.
#
    my $self       = shift;
    my $map_no     = 2;
    my $data       = $self->features( $map_no );
    my $scale      = $self->scale;
    my $y          = $self->{'y'} ||  OFFSET;
    my $x          = $self->{'x'} + 100;
    my $species    = $self->species( $map_no );
    my $marker_uri = $self->marker_viewer_uri( lc $species );

    #
    # Figure out the range covered by concordant features.
    #
    my @sorted_genetic_positions =
        sort { $a <=> $b }
        map  { 
            !$_->{'is_discordant'} ? 
                defined $_->{'genetic_position'} ?
                    $_->{'genetic_position'} : ()
                : ()
        }
        map { @$_ }
        values %$data
    ;

    #
    # If no concordant features, then just figure out the span.
    #
    unless ( @sorted_genetic_positions ) {
        @sorted_genetic_positions =
            sort { $a <=> $b }
            map  { 
                defined $_->{'genetic_position'} 
                ? $_->{'genetic_position'} : ()
            }
            map { @$_ }
            values %$data
        ;
    }

    #
    # Figure out the first and last genetic positions (relative to the map's
    # beginning).
    #
    my $start                   = $self->map_begin( $map_no - 1 );
    my $first_genetic_position  = $sorted_genetic_positions[ 0];# - $start;
    my $last_genetic_position   = $sorted_genetic_positions[-1];# - $start;
       $first_genetic_position -= $start;
       $last_genetic_position  -= $start;

    #
    # Find the pixel distance to midway between those two points.
    #
    my $halfway    = ( $last_genetic_position + $first_genetic_position ) / 2;
       $halfway   *= $scale;                     
       $halfway   += $y;                     

    #
    # Here we try to center the contigs features (not just its
    # length) on the genetic map features which are shared.
    #
    my $start_band   = $self->map_begin( $map_no );
    my $end_band     = $self->map_end  ( $map_no );
    my $cM_length    = $self->bands_to_cM( $end_band - $start_band );
    my $length       = $cM_length * $scale;
    my $contig_width = 4;
    my $tick_width   = 5;
    my $tick_font    = $self->tick_font;
    my $label_font   = $self->label_font;

    #
    # Find the physical features which are anchored to the genetic
    # (reference) map.
    #
    my @sorted_physical_positions =
        sort { $a <=> $b }
        map  { $_->{'genetic_position'} ? $_->{'start_band'} : () }
        map { @$_ }
        values %$data
    ;
    if ( $self->is_transposed ) {
        @sorted_physical_positions = reverse @sorted_physical_positions;
    }

#    #
#    # If all the features are discordant, we'll just use the ones 
#    # which are on the same linkage group.
#    #
#    unless ( @sorted_physical_positions ) {
#        @sorted_physical_positions =
#            sort { $a <=> $b }
#            map  { $_->{'genetic_position'} ? $_->{'start_band'} : ()
#                 }
#            map { @$_ }
#            values %$data
#    }

#    my $first_physical_position = $self->bands_to_cM(
#        $sorted_physical_positions[0] - $start_band
#    );
#
#    my $last_physical_position  = $self->bands_to_cM(
#        $sorted_physical_positions[-1] - $start_band
#    );

    my $first_physical_position = $sorted_physical_positions[ 0];
    my $last_physical_position  = $sorted_physical_positions[-1];

#    warn "1st = $first_physical_position, ",
#         "last = $last_physical_position\n";
    for ( $first_physical_position, $last_physical_position ) {
        if ( $self->is_transposed ) {
            $_ = $end_band - $_;
        }
        else {
            $_ -= $start_band;
        }
        $_ = $self->bands_to_cM( $_ );
    }
    my $midpoint_of_physical_features = $first_physical_position +
        abs($last_physical_position - $first_physical_position)/2;
    $midpoint_of_physical_features *= $scale; # convert to pixels

#    warn "1st = $first_physical_position, ",
#         "last = $last_physical_position, ",
#         "mid = $midpoint_of_physical_features\n";

#    warn "length = $length\n";

    my $line_start        = $halfway - $midpoint_of_physical_features;
    my $line_end          = $halfway + ($length-$midpoint_of_physical_features);
    my $dot_size          = 4;
    my $data_label_offset = $self->data_label_offset;
    my $color             = $data->{'is_discordant'} ? 'red' : 'black';
    $self->max_y( $line_end );

    #
    # What we will return.
    #
    my ( @data, @map );

    #
    # Put the species and contig name at the top.
    #
    my ( $title_leftmost, $title_bottom );
    $self->min_y( $line_start );
    my $map_name = $self->map_name($map_no);
    for my $string ( $species, $map_name ) {
        my $cur_x = $x - ( $tick_font->width * length($string) ) / 2;
        my $cur_y = $line_start;

        push @data, [
            STRING,
            $tick_font,
            $cur_x,
            $cur_y,
            $string,
            'black',
        ];

        my $diff = $tick_font->height + 4;
        $_ += $diff for $line_start, $line_end;

        $title_leftmost = $cur_x unless defined $title_leftmost;
        $title_leftmost = $cur_x if $cur_x < $title_leftmost;
        $title_bottom   = $cur_y if $cur_y > $title_bottom;
#        warn "title bottom = $title_bottom, cur y = $cur_y\n";
    }

    #
    # The box representing the contig.
    #
    my @coords = (
        $x - $contig_width / 2, 
        $line_start, 
        $x + $contig_width / 2, 
        $line_end, 
    );

    push @data, [ 
        RECTANGLE, 
        @coords,
        $color 
    ];

    push @data, [ 
        FILLED_RECT, 
        @coords,
        DEFAULT->{'contig_fill_color'}
    ];

    my @keys           = sort { $a <=> $b } keys %$data;
    my $midpoint       = int $#keys / 2;
    my $midpoint_value = $keys[ $midpoint ];
    my @ordered_keys   = 
        map { $keys[$_] } reverse(0..$midpoint), $midpoint+1..$#keys;
    my ( $prev_y, $furthest_north, $furthest_south );
#    my $genetic_positions = $self->positions;
#    warn "gen. positions = ", Dumper( $genetic_positions ), "\n";

    my %colors    = (); # Track the feature colors by feature_id
    my %done      = (); # the physical features already drawn
    my $highlight = $self->highlight; # A hash of the highlighted features
    for my $tick_pos ( @ordered_keys ) {
        #
        # Place a tick on the physical map
        #
        my $x1          = $x - $tick_width;
        my $x2          = $x + $tick_width;
        my $cM_position = $self->bands_to_cM( $tick_pos - $start_band );
        my $tick_y      = $line_start + ( $cM_position * $scale );
        my @features    = @{ $data->{$tick_pos} || [] } or next;
        $prev_y         = $tick_y unless defined $prev_y;
        $prev_y         = $tick_y if $tick_pos == $midpoint_value;

        for my $feature ( 
            map  { $_->[0] }
            sort { $a->[1] cmp $b->[1] }
            map  { [ $_, $_->{'feature_name'} ] }
            @features
        ) {
            my $feature_id  = $feature->{'feature_id'};
            my @tag_coords = ();
            my @genetic_positions = $self->feature_position(
                feature_id => $feature_id,
                map_no     => 1,
            );
#            warn "genetic positions for feature id $feature_id=\n",
#                Dumper( @genetic_positions ), "\n";

            unless ( $done{ $feature_id } ) {
                #
                # I know I'm running the risk of drawing the tick mark
                # more than once, but I'd like to have the line drawn in 
                # red for a discordant feature.
                #
                my $is_discordant = $feature->{'is_discordant'} || 0;
                my $tick_color    = $is_discordant
                    ? DEFAULT->{'discordant_contig_feature_color'}
                    : DEFAULT->{'concordant_contig_feature_color'}
                ;
                my $tag_color     = $is_discordant
                    ? 'red' #DEFAULT->{'discordant_contig_feature_color'}
                    : DEFAULT->{'tag_color'}
                ;
                my $line_color    = $is_discordant ? 'red' : 'gray';
#                                        @genetic_positions
#                                        ? $self->feature_color( $feature_id )
#                                        : 'grey';

                push @data, [
                    LINE, 
                    $x1, 
                    $tick_y, 
                    $x2, 
                    $tick_y, 
                    $tick_color,
                ];

                my $tag         = $feature->{'feature_name'};
                my $tag_x       = $x + $data_label_offset;
                my $tag_y       = $tick_y;
                $furthest_north = $tick_y unless defined $furthest_north;
                $furthest_south = $tick_y unless defined $furthest_south;

                if ( $tick_pos <= $midpoint_value ) {
                    $tag_y-- while $tag_y >= $prev_y;
                    $tag_y-- while $tag_y >= $furthest_north;
                }
                else {
                    $tag_y++ while $tag_y <= $prev_y;
                    $tag_y++ while $tag_y <= $furthest_south;
                }

                my $tag_top     = $tag_y - $tick_font->height;
                my $tag_bottom  = $tag_y + $tick_font->height;
                $furthest_north = $tag_top    if $tag_top    < $furthest_north;
                $furthest_south = $tag_bottom if $tag_bottom > $furthest_south;

                #
                # The feature's tag
                #
                push @data, [
                    STRING, 
                    $tick_font, 
                    $tag_x, 
                    $tag_y - ( $tick_font->height/2 ), 
                    $tag, 
                    $tag_color,
                ];
                $self->max_x( $tag_x + $tick_font->width*length($tag) );

                #
                # Make the area clickable
                # 
                @tag_coords = map { int } (
                    $tag_x - 2,
                    $tag_y - ( $tick_font->height / 2 ) - 1,
                    $tag_x + $tick_font->width * length( $tag ) + 1,
                    $tag_y + ( $tick_font->height / 2 ),
                );

                #
                # See if this feature is highlighted.
                #
                if ( exists $highlight->{lc $feature->{'feature_name'} } ) {
                    $color      = DEFAULT->{'highlight_color'};
                    $line_color = DEFAULT->{'highlight_box_color'};
                    $colors{ $feature_id } = $color;
                    push @data, [
                        RECTANGLE,
                        @tag_coords,
                        $line_color,
                    ];

                    push @data, [
                        FILLED_RECT,
                        @tag_coords,
                        $color
                    ];
                }

                #
                # A line connecting the tag to the tick
                #
                push @data, $self->_indicator_line(
                    $x + $tick_width + 3,
                    $tick_y, 
                    $tag_x - $tick_width * .25, 
                    $tag_y, 
                    $line_color, 
                    2
                );
                
                push @map, {
                    coords => [ @tag_coords ],
                    uri    => $marker_uri.$feature->{'feature_name'},
                    alt    => "View: $tag",
                };

                my $buffer = 2;
                $prev_y = $tick_pos <= $midpoint_value
                    ? $tag_y - $tick_font->height - $buffer
                    : $tag_y + $tick_font->height + $buffer;
            }
            
            #
            # Connect the feature on the physical map to it's
            # corresponding feature on the genetic map (unless 
            # it's discordant).
            #
            for my $genetic_feature ( @genetic_positions ) {
                my ($gx, $gy)   = @{ $genetic_feature->{'link_line'} };
                my ($px, $py)   = ( $x - $tick_width - 3, $tick_y );
                my @gtag_coords = @{ $genetic_feature->{'tag_coords'} }
                    or next;

#                my $color = $self->feature_color( $feature_id );
#                $colors{ $feature_id } = $color;
                my $color = 'lightblue';

                push @data, $self->_indicator_line(
                    $gx, $gy, $px, $py, $color, 2
                );

#                push @data, [
#                    FILLED_RECT,
#                    @gtag_coords,
#                    $color
#                ];
#
#                push @data, [
#                    FILLED_RECT,
#                    @tag_coords,
#                    $color
#                ] if @tag_coords;
            }

            $done{ $feature_id }++;
        }
    }

    $furthest_north = $line_start if $line_start < $furthest_north;
    $furthest_south = $line_end   if $line_end   > $furthest_south;

    #
    # Carry the features colors into the data used by the form.
    #
    my $contig_data = $self->data( 2 );
    for my $feature ( @{ $contig_data->{'table_features'} } ) {
        $feature->{'color'} = $self->color_hex_value(
            $colors{ $feature->{'feature_id'} }
        );
    }

    #
    # Draw the clones.
    #
    my $genome_viewer_uri = URIS->{'genome_viewer'};
    my @columns           = ();
    my $i                 = 1; # Number each clone for ease of identification.
    my $buffer            = 2; # The pixels between the clones.
    my $clones            = $self->data( $map_no )->{'clones'};
    my $width             = $tick_font->height + 4;
    my ( $topmost, $leftmost, $rightmost );

    for my $clone ( @$clones ) { 
        $clone->{'number'} = $i;
        my $clone_name     = $clone->{'clone_name'};
        my $start          = $clone->{'position_start'};
        my $stop           = $clone->{'position_stop'};
        next unless defined $start && defined $stop;

        for ( $start, $stop ) {
            $_  = $line_start + ($self->bands_to_cM($_-$start_band) * $scale);
        }

        my $column_index;
        if ( @columns ) { 
            for my $i ( 0..$#columns ) {
                my $column_is_ok = 1;
                my $column       = $columns[ $i ];
                my @sections     = sort { $a->[0] <=> $b->[0] } @{ $column };

                for my $section ( @sections ) {
                    my ( $furthest_north, $furthest_south ) = @{ $section };

                    if ( $furthest_south + $buffer >= $start ) {
                        $column_is_ok = 0;
                        last;
                    }
                }

                if ( $column_is_ok ) {
                    $column_index = $i;
                    push @{ $column }, [ $start, $stop ];
                    last;
                }
            }
    
            unless ( defined $column_index ) {
                push @columns, [ [ $start, $stop ] ];
                $column_index = $#columns;
            }
        }
        else {
            $column_index = 0;
            push @columns, [ [ $start, $stop ] ];
        }

        my $column_width = $width * ($column_index+1);
        my $color        = 'gray';
        my $cur_x        = $x - $column_width - 5;
        my $half_tag     = ( $tick_font->width * length( $i ) ) / 2;
        my $left         = $cur_x - $tick_font->height - 2;
        my $right        = $cur_x + 2;
        my @coords       = map { int } (
            $left,
            $start,
            $right,
            $stop,
        );

        $leftmost       = $left  unless defined $leftmost;
        $leftmost       = $left  if $left  < $leftmost;
        $rightmost      = $right if $right > $rightmost;
        $topmost        = $start unless defined $topmost;
        $topmost        = $start if $start < $topmost;
        $furthest_north = $start if $start < $furthest_north;
        $furthest_south = $stop  if $stop  > $furthest_south;

        push @map, {
            coords => [ @coords ],
            uri    => $genome_viewer_uri.$clone_name,
            alt    => $clone_name,
        };

        push @data, [
            STRING_UP,
            $tick_font,
            $cur_x - $tick_font->height - 2,
            $start + ( ( $stop - $start ) / 2 ) + 
                ( ( $tick_font->width * length( $i ) ) / 2 ),
            $i,
            $color,
        ];

        push @data, [
            LINE,
            $cur_x,
            $start,
            $cur_x,
            $stop,
            $color,
        ];

        push @data, [
            LINE,
            $cur_x - 1,
            $start,
            $cur_x + 1,
            $start,
            $color,
        ];

        push @data, [
            LINE,
            $cur_x - 1,
            $stop,
            $cur_x + 1,
            $stop,
            $color,
        ];

        $i++;
    }

    #
    # Label all the clones, if necessary.
    #
    if ( @$clones ) {
        my @tags   = qw[ Seq. Clones ];
        my $buffer = 10; 
        $leftmost  = $title_leftmost if $title_leftmost < $leftmost;
        $topmost  -= $buffer;
        $leftmost -= $buffer * 2;
        $topmost   = $title_bottom   if $title_bottom   < $topmost;
        my $midway = $leftmost + ( ( $rightmost - $leftmost ) / 2 );
        my $cur_y  = $topmost;

        #
        # Push the midpoint over while a tag will overwrite the leftmost.
        #
#        for my $tag ( @tags ) { 
#            my $halflength = ( $tick_font->width * length( $tag ) ) / 2;
##            warn "midway ($midway) + half ($halflength) = ", 
##                $midway + $halflength, " ?? $leftmost\n";
#            $midway-- while $midway + $halflength > $leftmost;
##            warn "midway now = $midway\n";
#
#            push @data, [
#                LINE,
#                $midway,
#                $cur_y - 20,
#                $midway,
#                $cur_y - 10,
#                'yellow'
#            ];
#        }
#
#        push @data, [
#            LINE,
#            $midway,
#            $cur_y - 20,
#            $midway,
#            $cur_y - 10,
#            'red'
#        ];

        for my $tag ( reverse @tags ) { 
            push @data, [
                STRING,
                $tick_font,
                $midway - ( $tick_font->width * length( $tag ) ) / 2,
                $cur_y,
                $tag,
                $color,
            ];

            $cur_y          -= $tick_font->height;
            $furthest_north  = $cur_y if $cur_y < $furthest_north;
        }
    }

    $self->min_y( $furthest_north );
    $self->max_y( $furthest_south );
    return { data => \@data, map => \@map };
}

#-----------------------------------------------------
sub data {
    my $self   = shift;
    my $map_no = shift || 1;
    return $self->{'data'}{ $map_no };
}

#-----------------------------------------------------
sub data_dump {
#
# Returns tab-delimited data suitable for download.
#
    my $self       = shift;
    my $field_sep  = "\t";
    my $record_sep = "\n";

    my $return .= join( $field_sep, qw[
        genetic_feature_name
        genetic_map_position_name
        genetic_species
        genetic_map_study_name
        genetic_linkage_group
        genetic_position_start
        genetic_position_stop
        physical_feature_name
        physical_map_position_name
        physical_position_start
        physical_position_stop
        bac_hits
    ]) . $record_sep;

    for my $marker ( @{ $self->{'data'}{ 2 }{'table_features'} } ) {
        for my $position ( @{ $marker->{'genetic_positions'} } ) {
            $return .= join( $field_sep, 
                $position->{'feature_name'},
                $position->{'map_position_name'},
                $position->{'species_common_name'},
                $position->{'map_study_name'},
                $position->{'linkage_group'},
                $position->{'position_start'},
                $position->{'position_stop'},
                $marker->{'feature_name'},
                $marker->{'map_position_name'},
                $marker->{'start_band'},
                $marker->{'stop_band'},
                $marker->{'bac_hits'},
            ) . $record_sep;
        }
    }

    return $return;
}

#-----------------------------------------------------
sub features {
    my $self   = shift;
    my $map_no = shift || 1;
    return $self->{'data'}{ $map_no }{'features'};
}

#-----------------------------------------------------
sub max_pixels_from_tick {
    my $self     = shift;
    return MAX_PIXELS_FROM_TICK * 1.5;
}

#-----------------------------------------------------
sub component_list { qw[ reference_map contig ] };

#-----------------------------------------------------
#sub reference_map_foo {
#    my ( $self, %args ) = @_;
#    my $data       = $self->data( 1 );
##    warn "Contig reference_map data =\n", Dumper( $data ), "\n";
#    my $start      = $data->{'start_cM'}  ||  0;
#    my $end        = $data->{'end_cM'}    ||  0;
#    my $cM_length  = $end - $start;
#    my $scale      = $self->scale;
#    my $x          = $self->{'x'}      ||  0;
#    my $y          = $self->{'y'}      ||  0; #OFFSET;
#    my $length     = ( $end - $start ) * $scale;
#    my $tick_font  = $self->tick_font;
#    my $tick_width = $self->tick_width;
#    my $marker_uri = $self->marker_viewer_uri($data->{'reference_species'});
#    my ( @data, @map ); # What we will return
#
##    warn "data =\n", Dumper( $data ), "\n";
#
#    #
#    # We'll use this to center the contig
#    #
#    $self->{'halfway'} = ($length/2) + $y;
#
#    #
#    # The vertical line
#    #
#    my $max_y = $y + $length;
#    $self->max_y( $max_y );
#
#    push @data, [ 
#        LINE, 
#        $x - $tick_width / 4, 
#        $y, 
#        $x - $tick_width / 4, 
#        $max_y, 
#        'black' 
#    ];
#
#    push @data, [ 
#        LINE, 
#        $x + $tick_width / 4, 
#        $y, 
#        $x + $tick_width / 4, 
#        $max_y, 
#        'black' 
#    ];
#
#    my $map_color = $self->map_color;
#    my @coords    = map { int } (
#        $x - $tick_width / 4, 
#        $y, 
#        $x + $tick_width / 4, 
#        $max_y, 
#    );
#
#    push @data, [ 
#        FILLED_RECT, 
#        @coords,
#        $map_color,
#    ];
#    $self->min_x( $x - $tick_width / 4 );
#
#    push @map, {
#        coords => [ @coords ],
#        uri    => URIS->{'map_viewer'}.
#                  "?reference_map_id=".$self->{'genetic_map_id'}.
#                  ";highlight=".$self->highlight_string,
#        alt    => $data->{'map_name'},
#    };
#
#
#    # 
#    # The tick marks
#    # 
#    my $diff = $end - $start;
#    my $tick_pos_interval = 
#        $diff <= 5 ? 1
#            : $diff <= 10 ? 5
#                : 10;
#    my $tick_x1 = $x - $tick_width * .25;
#    my $tick_x2 = $x + $tick_width * .75;
#    for my $tick_pos ( int $start.. int $end ) {
#        next if     $tick_pos <= $start || $tick_pos >= $end;
#        next unless $tick_pos % $tick_pos_interval == 0;
#        my $cur_y = $y + ( ($tick_pos-$start) * $scale);
#        my $tag_x = $x + $tick_width;
#        my $tag_y = $cur_y + (($tick_font->width*length($tick_pos))/2); 
#
#        push @data, [
#            STRING_UP, 
#            $tick_font, 
#            $tag_x, 
#            $tag_y, 
#            $tick_pos, 
#            'gray'
#        ];
#
#        push @data, [
#            LINE, 
#            $tick_x1,
#            $cur_y, 
#            $tick_x2,
#            $cur_y, 
#            'gray'
#        ];
#    }
#
#    #
#    # The features
#    #
#    my $data_label_offset = $self->data_label_offset;
#    my $prev_y            = undef; 
#    my $highlight         = $self->highlight;
##    warn "highlight = ", Dumper( $highlight ), "\n";
#
#    #
#    # Used to build the query string for scrolling through the map
#    #
#    my $map_end        = $self->map_end;
#    my $features        = $data->{'genetic_features'};
#    my @keys           = sort { $a <=> $b } keys %$features;
#    my $midpoint       = int $#keys / 2;
#    my $midpoint_value = $keys[ $midpoint ];
#    my @ordered_keys   = 
#        map { $keys[$_] } reverse(0..$midpoint), $midpoint+1..$#keys;
#
##    my %positions = ();
#    my ( $furthest_north, $furthest_south ) = (0, 0);
#    for my $i ( 0..$#ordered_keys ) {
#        my $tick_pos    = $ordered_keys[$i];
#        my $tick_y      = $y + ( ( $tick_pos - $start ) * $scale );
#        my $cur_y       = $tick_pos > 0
#                          ? $tick_y # $y + ( ($tick_pos - $start ) * $scale)
#                          : defined $prev_y ? $prev_y : $y;
#        $prev_y         = $cur_y unless defined $prev_y;
#        $prev_y         = $cur_y if $tick_pos == $midpoint_value;
#        my @tick_coords = (
#            $x - $tick_width/2, 
#            $tick_y, 
#            $x + $tick_width/2, 
#            $tick_y
#        );
#
#        #
#        # Place a tick mark at the location and make it
#        # a hyperlink to recenter and zoom in on a region.
#        #
#        push @data, [ LINE, @tick_coords, 'black' ];
#
#        #
#        # Each cM position can have several features.
#        # Sort them by their names (which should be unique).
#        #
#        for my $feature ( 
#            sort { 
#                $b->{'is_related'} <=> $a->{'is_related'} 
#                ||
#                $a->{'name'} cmp $b->{'name'}
#            }
#            @{ $features->{$tick_pos} } 
#        ) {
#
#            #
#            # The feature tag
#            #
#            my $map_position_name = $feature->{'map_position_name'};
#            my $feature_name      = $feature->{'feature_name'};
#            my $tag               = $map_position_name;
#            
#            my $start_x = $x - $tick_width/2;
#            my $tag_x = $start_x - $data_label_offset -
#                      ( $tick_font->width * length( $tag ) );
#            my $tag_y = int $tick_y;
#            if ( $tick_pos <= $midpoint_value ) {
#                $tag_y-- while $tag_y >= $prev_y;
#            }
#            else {
#                $tag_y++ while $tag_y <= $prev_y;
#            }
#
#            #
#            # Remember these for future use
#            #
#            my @tag_coords = map { int } (
#                $tag_x - 3, 
#                $tag_y - ( $tick_font->height/2 ),
#                $tag_x + ( $tick_font->width * length( $tag ) ), 
#                $tag_y + ( $tick_font->height/2 ),
#            );
#            $self->min_x( $tag_x );
#
#            #
#            # See if we need to highlight this feature's tag
#            # If not, see if we should we skip it because it 
#            # would be too far away from the tick mark.
#            #
#            my $feature_id = $feature->{'feature_id'};
#            my $ok_to_skip = 1;
#            my ( $color, $line_color );
#
#            for my $name ( $map_position_name, $feature_name ) {
#                my $lc_name = lc $name;
#                if ( exists $highlight->{$lc_name} ) {
#                    $color      = DEFAULT->{'highlight_color'};
#                    $line_color = DEFAULT->{'highlight_box_color'};
#                    $self->feature_color( $feature_id, $color ) if $color;
#                    $ok_to_skip = 0;
#                }
#            }
#
##            unless ( $ok_to_skip ) {
##                next if abs( $tag_y - $tick_y ) > MAX_PIXELS_FROM_TICK;
##            }
#
#            #
#            # The line connecting the tag and the tick mark
#            #
#            unless ( $color ) {
#                if ( 
#                    $feature->{'is_related'} ||
#                    $self->feature_is_related( $feature_id )
#                ) {
#                    $color = $self->feature_color( $feature_id );
#                    $ok_to_skip = 0; 
#                }
#            }
#
#            my $skip = $ok_to_skip && abs($tag_y-$tick_y)>MAX_PIXELS_FROM_TICK
#                ? 1 : 0;
#
#            #
#            # Record the position of the tick to link to later
#            #
#            for my $id ( 
#                $feature->{'feature_id'}, 
#                $self->feature_correspondences( $feature->{'feature_id'} )
#            ) {
#                $self->feature_position(
#                    feature_id => $id,
#                    map_no     => 1,
#                    position   => {
#                        tag_coords => $skip ? [] : [ @tag_coords ],
#                        link_line  => [ 
#                            $x + $tick_width*.75, 
#                            $tick_y,
#                        ],
#                    },
#                ); 
#            }
#
#            next if $skip;
#
#            my $x1    = $x - $tick_width * .75;
#            my $y1    = $tick_y;
#            my $x2    = $x - $data_label_offset - 4;
#            my $y2    = $tag_y;
#            push @data, $self->_indicator_line(
#                $x1, 
#                $y1, 
#                $x2, 
#                $y2, 
#                $line_color || $color || 'grey'
#            );
#
#            #
#            # Remember the last y position of a feature tag
#            #
#            $prev_y = $tick_pos <= $midpoint_value
#                ? $tag_y - $tick_font->height
#                : $tag_y + $tick_font->height;
#
#            $furthest_north = $prev_y if $prev_y < $furthest_north;
#            $furthest_south = $prev_y if $prev_y > $furthest_south;
#
#
#            push @data, [
#                STRING, 
#                $tick_font, 
#                $tag_x,
#                $tag_y - ( $tick_font->height / 2 ), 
#                $tag, 
#                'blue'
#            ];
#
#            push @data, [
#                FILLED_RECT, 
#                @tag_coords,
#                $color
#            ] if $color;
#
#            push @data, [
#                RECTANGLE,
#                @tag_coords,
#                $line_color
#            ] if $line_color;
#
#            #
#            # The map coords
#            #
#            push @map, {
#                coords => [ @tag_coords ],
#                uri    => $marker_uri.$map_position_name,
#                alt    => "View: $map_position_name ($tick_pos)",
#            };
#        }
#    }
#
#    $self->max_y( $furthest_south );
#    $self->min_y( $furthest_north );
#
#    return { data => \@data, map => \@map };
#}

#-----------------------------------------------------
sub start {
    my $self = shift;
    return $self->map_begin( 1 );
#    return $self->{'data'}{ 2 }{'start_cM'};
}

#-----------------------------------------------------
sub end {
    my $self = shift;
    return $self->map_end( 1 );
#    return $self->{'data'}{ 2 }{'end_cM'};
}

##-----------------------------------------------------
#sub feature_correspondences {
#    my $self       = shift;
#    warn "called feature_corr in Contig\n";
#    my $feature_id = shift or return ();
#    return @{ 
#        $self->{'data'}{'feature_correspondences'}{$feature_id} || [] 
#    };
#}

##-----------------------------------------------------
#sub map_id {
#    my $self   = shift;
#    my $map_no = shift || 1;
#    return $self->{'data'}{ $map_no }{'map_id'};
#}

##-----------------------------------------------------
#sub map_end {
#    my $self = shift;
#
#    unless ( defined $self->{'map_end'} ) {
#        $self->{'map_end'} = $self->soap_call(
#            method => 'map_end',
#            params => { map_id => $self->{'genetic_map_id'} },
#        );
#    }
#    
#    return $self->{'map_end'};
#}

#-----------------------------------------------------
sub menu_fields   { qw[ contig_map_id genetic_map_id image_type ] }

#-----------------------------------------------------
sub menu_uri      { URIS->{'contig_viewer'} }

#-----------------------------------------------------
sub menu_commands { return } 

#-----------------------------------------------------
sub menu_query_string {
    my $self = shift;

    my $qs = join( ';',
        'genetic_map_id=' .$self->{'genetic_map_id'},
        'physical_map_id='.$self->{'physical_map_id'},
    );

    return $qs;
}

##-----------------------------------------------------
#sub reference_map_begin {
#    my $self     = shift;
#    return $self->{'start_cM'};
#}
#
##-----------------------------------------------------
#sub reference_map_end {
#    my $self     = shift;
#    return $self->{'end_cM'};
#}
#
##-----------------------------------------------------
#sub species {
#    my $self   = shift;
#    my $map_no = shift or return;
#    return $map_no == 1 
#        ? $self->{'reference_species'} 
#        : $self->{'species'}
#    ;
#}

1;

#-----------------------------------------------------
# For thy sweet love remembered such wealth brings
# That then I scorn to change my state with kings.
# William Shakespeare 
#-----------------------------------------------------

=pod

=head1 NAME

CSHL::ComparativeMaps::Drawer::Contig - draw a contig next to a genetic map

=head1 SYNOPSIS

  use CSHL::ComparativeMaps::Drawer::Contig;
  my $drawer    = CSHL::ComparativeMaps::Drawer::Contig->new( %args );
  $drawer->layout;
  my $data      = $drawer->data;
  my $map_areas = $drawer->image_map_areas;

=head1 DESCRIPTION

Inherits from CSHL::ComparativeMaps::Drawer.  This module draws an
individual contig beside a reference (genetic) map.

=head1 AUTHOR

Ken Y. Clark, kclark@logsoft.com

=head1 SEE ALSO

perl(1).

=cut
