package CSHL::ComparativeMaps::Drawer;

#-----------------------------------------------------
# $Id: Drawer.pm,v 1.20 2002/04/17 01:47:14 kclark Exp $
#
# File       : Drawer.pm
# Programmer : Ken Y. Clark, kclark@logsoft.com
# Created    : 2001/07/17
# Purpose    : base object for drawing maps
#-----------------------------------------------------

use strict;
use vars qw( $VERSION );
$VERSION = (qw$Revision: 1.20 $)[-1];

use GD;
use Data::Dumper;
use File::MkTemp;

use CSHL::ComparativeMaps::Constants;

my $ERRMSG;

#-----------------------------------------------------
sub new {
#
# Create a new Drawer object
#
    my $class = shift;
    my %args  = @_;
    my $self  = { %args };
    bless $self, $class;
    return $self->_init();
}

#-----------------------------------------------------
sub _init { 
    my $self = shift;

    $self->{'tick_pos_interval'} ||= 10;

    $self->{'map_study_id1'} = $self->{'reference_map_study_id'};
    $self->{'map_id1'}       = $self->{'reference_map_id'};

    my $i = 1;
    for my $comparative_map ( @{ $self->{'comparative_maps'} } ) {
        $i++;
        $self->{'map_study_id'.$i} = $comparative_map->{'map_study_id'};
        $self->{'map_id'.$i}       = $comparative_map->{'genetic_map_id'};
    }
    $self->no_maps( $i );

    if ( $self->{'font_size'} ) {
        $self->tick_font ( $self->{'font_size'} );
        $self->label_font( $self->{'font_size'} );
    }

    return $self;
}

#-----------------------------------------------------
sub _indicator_line {
    my ($self, $x1, $y1, $x2, $y2, $color, $map_no) = @_;
    my @return;

    {
        # 
        # See if the y axes are the same
        # 
        if ( $y1 == $y2 ) {
            # 
            # If the x axes are also the same, then draw a 
            # triangle, like so:
            # 
            #  |\        /|
            #  | >  or  < |
            #  |/        \|
            # 
            # (Use $map_no to determine the orientation)
            # 
            if ( $x1 == $x2 ) {
                my $tick_width = $map_no == 1 ? 5 : -5; 
                push @return, 
                [ LINE, $x1, $y1, $x1+$tick_width, $y1-$tick_width, $color ];
                push @return, 
                [ LINE, $x1+$tick_width, $y1-$tick_width,
                    $x1+$tick_width, $y1+$tick_width, $color ];
                push @return, 
                [ LINE, $x1+$tick_width, $y2+$tick_width, $x1, $y1, $color ];
                last;
            }
            else {
                # 
                # It's a straight line...
                # 
                push @return, [ LINE, $x1, $y1, $x2, $y2, $color ];
                last;
            }
        }

        
        # 
        # If the x axis is the same for both points, draw it like this:
        #   ,--        --.
        #   |     or     |
        #   |            |
        #   `--        --'
        #
        # (Use $map_no to determine the orientation) 
        # 
        if ( $x1 == $x2 ) {
            my $tick_width = $map_no == 1 ? 5 : -5; 
            push @return, 
                [ LINE, $x1            , $y1, $x1+$tick_width, $y1, $color ];
            push @return, 
                [ LINE, $x1+$tick_width, $y1, $x2+$tick_width, $y2, $color ];
            push @return, 
                [ LINE, $x2+$tick_width, $y2, $x2            , $y2, $color ];
            last;
        }

        #
        # Else, draw it like this
        # __               __
        #   \      or     /
        #    \__       __/
        #
        if ( $x1 < $x2 ) {
            push @return, [ LINE, $x1  , $y1, $x1+5, $y1, $color ];
            push @return, [ LINE, $x1+5, $y1, $x2-5, $y2, $color ];
            push @return, [ LINE, $x2-5, $y2, $x2  , $y2, $color ];
        } 
        else {
            push @return, [ LINE, $x1  , $y1, $x1-5, $y1, $color ];
            push @return, [ LINE, $x1-5, $y1, $x2+5, $y2, $color ];
            push @return, [ LINE, $x2+5, $y2, $x2  , $y2, $color ];
        }
    }

    return @return;
}

#-----------------------------------------------------
sub color_hex_value {
    my $self  = shift;
    my $color = shift or return;
    return join '', @{ COLORS->{$color} };
}

#-----------------------------------------------------
sub data_label_offset {
    my $self = shift;
    $self->{'data_label_offset'} = shift if @_;
    return $self->{'data_label_offset'} || DEFAULT->{ 'data_label_offset' };
}

#-----------------------------------------------------
sub bands_to_cM {
    my ( $self, $bands ) = @_;
    return ($bands*5000)/280_000;
}

#-----------------------------------------------------
sub map_id {
    my $self   = shift;
    my $map_no = shift || 1;
    return $self->{'data'}{ $map_no }{'map_id'};
}

#-----------------------------------------------------
sub no_maps {
    my $self = shift;
    $self->{'no_maps'} = shift if @_;
    return $self->{'no_maps'};
}

#-----------------------------------------------------
sub draw {
    my $self       = shift;
    my @data       = $self->drawing_data;
    my $image_type = $self->image_type;
#    warn "width = ", $self->width, "\n";
    my $gd         = GD::Image->new( $self->width, $self->height );
    my %colors     =
        map { $_, $gd->colorAllocate( map { hex $_ } @{ +COLORS->{$_} } ) } 
        keys %{+COLORS}
    ;
    $gd->interlaced( 'true' );
    $gd->fill( 0, 0, $colors{'beige'} );

    for my $obj ( @data ) {
        my $method = shift @$obj;
        my $color  = pop   @$obj;
        $gd->$method( @$obj, $colors{$color} );
    }

    my ( $fh, $filename ) = mkstempt( 'X' x 9, CACHE_DIRECTORY );
    print $fh $gd->$image_type();
    $fh->close;

    $self->image_name( $filename );
    return 1;
}

#-----------------------------------------------------
sub drawing_data {
    my $self = shift;
    $self->layout unless $self->{'drawing_data'};
    return @{ $self->{'drawing_data'}{'data'} };
}

#-----------------------------------------------------
sub image_name {
    my $self = shift;
    $self->{'image_name'} = shift if @_;
    return $self->{'image_name'} || '';
}

#-----------------------------------------------------
sub image_type {
    my $self = shift;
    $self->{'image_type'} = shift if @_;
    return $self->{'image_type'} || DEFAULT->{'image_type'};
}

#-----------------------------------------------------
sub image_map_areas {
    my $self   = shift;
    $self->layout unless $self->{'drawing_data'};
    return $self->{'drawing_data'}{'map'};
}

#-----------------------------------------------------
sub increment_scale {
    my $self = shift;
    my $scale = shift || $self->{'scale'};
    $self->{'scale'} = $self->next_scale( $scale );
    return $self->{'scale'};
}

#-----------------------------------------------------
sub feature_is_highlighted {
    my $self         = shift;
    my $feature_name = lc shift or return;
    my $highlight    = $self->highlight;
    return exists $highlight->{$feature_name};
}

#-----------------------------------------------------
sub feature_is_related {
    my $self       = shift;
    my $feature_id = shift or return;
    return defined $self->{'data'}{'related_features'}{ $feature_id };
}

#-----------------------------------------------------
sub feature_correspondences {
    my $self       = shift;
    my $feature_id = shift or return;
    return @{ $self->{'data'}{'feature_correspondences'}{$feature_id} || [] };
}

#-----------------------------------------------------
sub feature_position {
    my ( $self, %args ) = @_;
    my $feature_id      = $args{'feature_id'} or return;
    my $map_no          = $args{'map_no'} || 1;

    if ( my $arg = $args{ 'position' } ) {
         push @{ $self->{'position'}{ $map_no }{ $feature_id } }, $arg;
    }

    return @{ $self->{'position'}{ $map_no }{ $feature_id } || [] };
}

#-----------------------------------------------------
sub marker_position {
    my ( $self, $marker_name, $map_id ) = @_;
    $map_id ||= $self->{'map_id'};

    my $position = $self->soap_call(
        method => 'marker_position_by_name',
        params => { 
            map_id => $map_id,
            marker_name => $marker_name,
        }
    );
}

#-----------------------------------------------------
sub marker_viewer_uri {
    my $self    = shift;
    my $species = lc shift || '';
       $species = 'default' unless defined URIS->{'marker_viewer'}{$species};
    return URIS->{'marker_viewer'}{$species} || '';
}

#-----------------------------------------------------
sub legend {
    my $self = shift;
    my $x    = OFFSET;
    my $y    = $self->{'y'};
    
    my @map_data = ();
    return { data => \@map_data, map => [] };
}

#-----------------------------------------------------
sub menu_height {
    my $self = shift;
    $self->{'menu_height'} = shift if @_;
    return $self->{'menu_height'};
}

#-----------------------------------------------------
sub menu_commands {
    return ( 'Up', 'Down', 'Zoom In', 'Zoom Out', 'Full View' );
}

#-----------------------------------------------------
sub size { return }

#-----------------------------------------------------
sub menu {
    my $self       = shift;
    my $buffer     = 4;
    my $tick_font  = $self->tick_font;
    my $label_font = $self->label_font;
    my $label_y    = OFFSET; # Where the label will go
    my $y          = $label_y + ( $label_font->height * $self->no_maps );
    my $x          = OFFSET + $buffer;
    my ( @map_data, @coord_data );

    #
    # The commands
    #
    if ( $self->menu_commands ) {
        my $start       = $self->start;
        my $end         = $self->end;
        my $map_end     = $self->map_end( $self->map_id( 1 ) );
        my $tag_y       = $y + $buffer*2;
        my $rect_height = $tick_font->height + ( $buffer*2 );
        my $uri         = $self->menu_uri;

        for my $tag ( $self->menu_commands ) {
            my $qs          = $self->menu_query_string;
            my $tag_width   = $tick_font->width * length( $tag );
            my $rect_width  = $tag_width + ( $buffer*2 );
            my @rect_coords = ( map { int }
                $x,
                $tag_y,
                $x + $rect_width,
                $tag_y + $rect_height,
            );

            #
            # Check to see if we should make the command active
            #
            my $map_begin = $self->reference_map_begin;
            my $make_link = 
            $tag eq 'Up'        && $start > $map_begin                    ? 1 :
            $tag eq 'Down'      && $end   < $map_end                      ? 1 :
            $tag eq 'Zoom In'   && ( $end - $start ) > 2                  ? 1 :
            $tag eq 'Zoom Out'  && ($start!=$map_begin || $end!=$map_end) ? 1 :
            $tag eq 'Full View' && ($start >$map_begin || $end <$map_end) ? 1 :
            0
            ;

            my $box_color  = $make_link ? 'blue'  : 'gray';
            my $text_color = $make_link ? 'black' : 'gray';
            push @map_data, [ 
                STRING, 
                $tick_font, 
                $x + $buffer, 
                $tag_y + $buffer, 
                $tag, 
                $text_color 
            ];

            push @map_data, [ 
                RECTANGLE, 
                @rect_coords, 
                $box_color 
            ];
            $x += $rect_width + $buffer;

            #
            # If necessary, figure the zoom and make it clickable
            #
            next unless $make_link;
            my $diff = $end - $start;
            my ( $next_start, $next_end );

            if ( $tag eq 'Up' ) {
                $next_start = $start - $diff > $map_begin ?
                              $start - $diff : $map_begin ;
                $next_end   = $end   - $diff > $map_begin ?
                              $end   - $diff : $map_end   ;
            }
            elsif ( $tag eq 'Down' ) {
                $next_start = $start + $diff < $map_end ?
                              $start + $diff : $map_end ;
                $next_end   = $end   + $diff < $map_end ?
                              $end   + $diff : $map_end ;
            }
            elsif ( $tag eq 'Zoom In' ) {
#                $diff /= 2;
#                $diff++ while $diff < DEFAULT->{'scroll_interval'};
                $diff = DEFAULT->{'scroll_interval'};
                $next_start = $start + $diff < $map_end ?
                              $start + $diff : $map_end ;
                $next_end   = $end   - $diff < $map_end ?
                              $end   - $diff : $map_end ;
            }
            elsif ( $tag eq 'Zoom Out' ) {
                $diff *= 2;
                $next_start = $start - $diff > $map_begin ? 
                              $start - $diff : $map_begin ;
                $next_end   = $end   + $diff < $map_end   ?
                              $end   + $diff : $map_end   ;
            }
            elsif ( $tag eq 'Full View' ) {
                ; # nothing to do
            }
            else {
                $qs .= ";size=$tag";
            }

            #
            # Make sure there's at least 2 cM
            #
            my $link = "$uri?$qs";
            if ( defined $next_start && defined $next_end ) {
                $_ = sprintf( "%.1f", $_ ) for $next_start, $next_end;
                unless ( $next_end - $next_start >= 2 ) {
                    $next_start-- if $next_start - 1 > 0;
                    $next_end++   if $next_end   + 1 < $map_end;
                }
                $link .= ";start=$next_start;end=$next_end";
            }

            push @coord_data, {
                coords => join( ',', @rect_coords ),
                uri    => $link,
                alt    => $tag,
            }
        }

        $y = $tag_y + $rect_height;
    }

    #
    # The map title
    #
    my $no_maps = $self->no_maps;
    my @labels  = map { $self->map_label( $_ ) || () } 1..$no_maps;
    my $cur_y   = $label_y;
    my $map_no  = 1;
    my $max_x;
    for my $label ( @labels ) {
        my $map_name       = $label;
        $label            .= ' vs. ' unless $map_no == $no_maps;
        my $label_width    = $label_font->width * length( $label );
        my $cur_max_x      = $self->width + abs( $self->min_x ) > $x ?
                             $self->width + abs( $self->min_x ) : $x ;
        my $width          = $cur_max_x > $label_width + $buffer * 8 ? 
                             $cur_max_x : $label_width + $buffer * 8 ;
        my $img_midpoint   = $width / 2;
        my $label_midpoint = $label_width / 2;
        my $label_x        = $img_midpoint - $label_midpoint;
        my $label_y        = $cur_y + $buffer / 2;

        push @map_data, [
            STRING,
            $label_font,
            $label_x,
            $label_y,
            $label,
            'black'
        ];

        push @coord_data, {
            coords => join( ',', 
                map { int }
                $label_x,
                $label_y,
                $label_x + $label_width,
                $label_y + $label_font->height,
            ),
            uri    => URIS->{'map_study_info'}.
                      "?map_study_id=".$self->map_study_id( $map_no ),
            alt    => "Info on $map_name",
        };

        $max_x  = $width if $width > $max_x;
        $cur_y += $label_font->height + $buffer;
        $map_no++;
    }

#    my $width = $max_x > $label_width ? $max_x : $label_width;
#    $self->max_x( $width );
    $self->map_title( join( '', @labels ) );
    $self->max_x( $max_x );

    $y = $cur_y if $cur_y > $y;

    #
    # Put the menu in white with a black border
    #
    my @menu_coords = (
        OFFSET,
        OFFSET,
        $max_x,
        $y + $buffer,
    );

    unshift @map_data, [
        FILLED_RECT,
        @menu_coords,
        'white'
    ];

    push @map_data, [
        RECTANGLE,
        @menu_coords,
        'black'
    ];

    $self->max_x( $x );
    $self->menu_height( $y );

    return { data => \@map_data, map => \@coord_data };
}

#-----------------------------------------------------
sub end {
    my $self   = shift;
    return $self->err_out( 'end() not defined' );
}

#-----------------------------------------------------
sub err_out {
    my ( $self, $err ) = @_;
    $ERRMSG = $err if $err;
    return wantarray ? () : undef;
}

#-----------------------------------------------------
sub errstr {
    return $ERRMSG || 'Last error not defined';
}

#-----------------------------------------------------
sub height {
    my $self = shift;
    return $self->max_y + OFFSET;
}

#-----------------------------------------------------
sub highlight_color {
    my $self = shift;
    my $hl_index = shift || $self->{'hl_color_index'} || 0;
#       $hl_index = 0 unless exists HL_COLORS->[ $hl_index ];
       $hl_index = 0 unless defined HL_COLORS->[ $hl_index ];
    $self->{'hl_color_index'} = $hl_index + 1;
    return HL_COLORS->[ $hl_index ];
}

#sub highlight {
#    my $self = shift;
#    unless ( defined $self->{'highlight_hash'} ) {
#        $self->{'highlight_hash'} = { 
#            map  { lc $_, 0 }
##            map  { s/\s+//g; $_ }
#            split( /[,:;\s+]/, $self->{'highlight'} )
#        };
#    }
#
#    return $self->{'highlight_hash'};
#}

#-----------------------------------------------------
sub highlight {
    my $self = shift;

    if ( my $arg = shift ) {
        $self->{'highlight'}      = $arg;
        $self->{'highlight_hash'} = undef;
    }

    unless ( defined $self->{'highlight_hash'} ) {
        $self->{'highlight_hash'} = {
            map{ lc $_, 0 }
            split( /[,:;\s+]/, $self->{'highlight'} )
        };
    }

    return $self->{'highlight_hash'};
}

#-----------------------------------------------------
sub highlight_string {
    return shift->{'highlight'} || '';
}

#-----------------------------------------------------
sub map_title {
    my $self = shift;
    $self->{'map_title'} = shift if @_;
    return $self->{'map_title'} || '';
}

#-----------------------------------------------------
sub map_color {
    my $self = shift;
    my $map_index = shift || $self->{'map_color_index'} || 0;
       $map_index = 0 unless defined MAP_COLORS->[ $map_index ];
    $self->{'map_color_index'} = $map_index + 1;
    return MAP_COLORS->[ $map_index ];
}

#-----------------------------------------------------
sub map_y_length {
    my $self    = shift;
    my $map_no  = shift || 1;
    my $field   = $map_no == 1 ? 'map1_y_length' : 'map2_y_length';
    if ( @_ ) { $self->{$field} = shift } 

    return $self->{$field} || 0;
}

#-----------------------------------------------------
sub map_coord_type {
    my $self    = shift;
    my $field   = 'map_coord_type';

    unless ( $self->{$field} ) {
        my $start       = $self->start;
        my $end         = $self->end;
        my $pattern     = '^\d+\.?\d+?$';
        $self->{$field} = ( $start =~ m/$pattern/ || $end =~ m/$pattern/ ) 
            ? 'cM' : 'marker';
    }

    return $self->{$field};
}

#-----------------------------------------------------
sub width {
    my $self  = shift;
    my $max_x = $self->max_x || DEFAULT->{'width'};
    return $max_x + OFFSET;
}

#-----------------------------------------------------
sub layout {
    my ( $self, %args ) = @_;
    unless ( $args{'force'} ) {
        return if $self->{'drawing_data'};
    }

    my @shapes = ();
    for my $component ( $self->component_list ) {
        my $data = $self->$component() or next;
        push @shapes, @{ $data->{'data'} };
#        push @{ $self->{'drawing_data'}{'data'} }, @{ $data->{'data'} };
        push @{ $self->{'drawing_data'}{'map'}  }, @{ $data->{'map'}  };
    }

    #
    # The data for drawing the maps must be
    # returned in alphabetical order, which
    # is basically a hack right now.  We don't 
    # want 'filledRectangle's to be drawn over
    # 'line's and 'string[Up]'s, and it just happens
    # that drawing them in alphabetical order
    # happens to avoid this.  This could break, of
    # course, if a new shape falls later in the alphabet
    # and will overlay an existing component.
    #
    # Note the structure of the data is something like
    # [ LINE, x1, y1, x2, y2, 'color' ], so we're doing
    # the Schwartzian xform to sort on that first field.
    #
    my ( @filled_rectangles, @lines, @other_shapes );
    for my $shape ( @shapes ) {
        if ( $shape->[0] =~ m/^filled/o ) {
            push @filled_rectangles, $shape; 
        }
        elsif ( $shape->[0] =~ m/^line$/o ) {
            push @lines, $shape; 
        }
        else {
            push @other_shapes, $shape;
        }
    }
    push @{ $self->{'drawing_data'}{'data'} },
        @filled_rectangles, @lines, @other_shapes
    ;

    #
    # After everything else has been drawn, now add the menu.
    #
    my $menu    = $self->menu;
    my $y_diff  = int (abs( $self->min_y ) + OFFSET*2 + $self->{'menu_height'});
    my $x_diff  = $self->min_x < 0 ? int( abs( $self->min_x ) + OFFSET ) : 0;
    my %y_table = (
        FILL       , [ 2 ],
        ARC        , [ 2 ],
        DASHED_LINE, [ 2, 4 ],
        FILLED_RECT, [ 2, 4 ],
        LINE       , [ 2, 4 ],
        RECTANGLE  , [ 2, 4 ],
        STRING     , [ 3 ],
        STRING_UP  , [ 3 ],
    );
    my %x_table = (
        FILL       , [ 1 ],
        ARC        , [ 1 ],
        DASHED_LINE, [ 1, 3 ],
        FILLED_RECT, [ 1, 3 ],
        LINE       , [ 1, 3 ],
        RECTANGLE  , [ 1, 3 ],
        STRING     , [ 2 ],
        STRING_UP  , [ 2 ],
    );

    for my $rec ( @{ $self->{'drawing_data'}{'data'} } ) {
        for my $y_field ( @{ $y_table{ $rec->[0] } } ) {
            $rec->[ $y_field ] += $y_diff;
        }
        for my $x_field ( @{ $x_table{ $rec->[0] } } ) {
            $rec->[ $x_field ] += $x_diff;
        }
    }

    for my $rec ( @{ $self->{'drawing_data'}{'map'} } ) {
        my @coords       = @{ $rec->{'coords'} };
        $coords[$_]     += $y_diff for ( 1, 3 );
        $coords[$_]     += $x_diff for ( 0, 2 );
        $rec->{'coords'} = join( ',', map { int } @coords );
    }

    $self->{'max_y'} += $y_diff;
    $self->{'min_x'}  = 0;

    if ( $menu ) {
        push @{ $self->{'drawing_data'}{'data'} }, @{ $menu->{'data'} };
        push @{ $self->{'drawing_data'}{'map'}  }, @{ $menu->{'map'}  };
    }

    #
    # Put black border around the whole thing
    #
    push @{ $self->{'drawing_data'}{'data'} }, [
        RECTANGLE,
        0,
        0,
        $self->width-1,
        $self->height-1,
        'black'
    ];

    return $self->draw;
}

#-----------------------------------------------------
sub feature_color {
    my $self            = shift;
    my $feature_id      = shift or return;
    my $color           = shift || '';
    my @correspondences = $self->feature_correspondences( $feature_id );

    unless ( $color ) {
        #
        # See if there's already a color associated with 
        # the feature or any of its corrspondences.
        #
        for my $id ( $feature_id, @correspondences ) {
            $color = $self->{'feature_colors'}{$id} || '';
            last if $color;
        }

        #
        # If not, take the next highlight color.
        #
        $color ||= $self->highlight_color;
    }

    #
    # Assign the color to the feature and all correspondences
    #
    $self->{'feature_colors'}{$_} = $color for $feature_id, @correspondences;

    return $color;
}

#-----------------------------------------------------
sub max_y {
    my $self = shift;
    if ( my $new_y = shift ) {
        $self->{'max_y'} = $new_y if $new_y > $self->{'max_y'};
    }
    return $self->{'max_y'};
}

#-----------------------------------------------------
sub min_y {
    my ( $self, $new_y ) = @_;

    if ( defined $new_y ) {
        if ( defined $self->{'min_y'} ) {
            $self->{'min_y'} = $new_y 
            if $new_y < $self->{'min_y'};
        }
        else {
            $self->{'min_y'} = $new_y;
        }
    }
    return $self->{'min_y'};
}

#-----------------------------------------------------
sub min_x {
    my ( $self, $new_x ) = @_;

    if ( defined $new_x ) {
        if ( defined $self->{'min_x'} ) {
            $self->{'min_x'} = $new_x 
            if $new_x < $self->{'min_x'};
        }
        else {
            $self->{'min_x'} = $new_x;
        }
    }
    return $self->{'min_x'};
}

#-----------------------------------------------------
sub max_x { my $self = shift;
    if ( my $new_x = shift ) {
        $self->{'max_x'} = $new_x if $new_x > $self->{'max_x'};
    }
    return $self->{'max_x'};
}

#-----------------------------------------------------
sub reference_map {
    my $self         = shift;
    my $map_no       = 1;
    my $data         = $self->features( $map_no ) or return;
    my $start        = $self->start;
    my $end          = $self->end;
    my $x            = $self->{'x'} ||  0;
    my $y            = $self->{'y'} ||  0;
    my $real_length  = $end - $start;
    my $cM_length    = $end - $start;
       $cM_length    = MIN_MAP_CM_LENGTH if $cM_length < MIN_MAP_CM_LENGTH;
    my $scale        = $self->scale;
    my $pixel_length = $cM_length * $scale;
    my $tick_font    = $self->tick_font;
    my $label_font   = $self->label_font;
    my $tick_width   = $self->tick_width;
    my $marker_uri   = $self->marker_viewer_uri( $self->species( $map_no ) );

#    warn "ref data = \n", Dumper( $data ), "\n";

    #
    # Used to build the query string for scrolling through the map
    #
    my $comparative_map_field1;
    if ( my $map_id = $self->map_id(2) ) {
        $comparative_map_field1 = 'genetic_map_id%3D'.$self->map_id(2);
    }
    else {
        $comparative_map_field1 = 'map_study_id%3D'.$self->map_study_id(2);
    }

    my $scroll_qs = join( ';', 
        'reference_map_study_id='      .$self->map_study_id( 1 ),
        'reference_map_id='            .$self->map_id( 1 ),
        'comparative_map1='            .$comparative_map_field1,
        'highlight='                   .$self->{'highlight'},
        'show_only_concordant_contigs='.$self->{'show_only_concordant_contigs'},
        'flip='                        .$self->{'flip'},
        'font_size='                   .$self->{'font_size'},
    );

    my ( @data, @map ); # What we will return
    my $map_color = $self->map_color;

    #
    # Put the species and linkage group at the top.
    #
    my $species       = $self->species(1);
    my $linkage_group = $self->map_name(1);
    for my $string ( $species, $linkage_group ) {
        push @data, [
            STRING,
            $tick_font,
            $x - ( $tick_font->width * length($string) ) / 2,
            $y,
            $string,
            'black',
        ];

        #
        # Push the map down a bit.
        #
        $y += $tick_font->height + 2;
    }

    #
    # The top "arrow" (scrolling region)
    #
    my $x_difference = $tick_width / 4; 
    my $scroll       = $end - $start > DEFAULT->{'scroll_interval'}
                       ? DEFAULT->{'scroll_interval'}
                       : $end - $start;
    my $map_begin    = $self->reference_map_begin;
    if ( $start > $map_begin ) {
        #
        # Push the map down a bit.
        #
        $y += $tick_width;

        #
        # Vertical lines
        #
        push @data, [
            LINE,
            $x - $x_difference,
            $y - $tick_width, 
            $x - $x_difference,
            $y,
            'black'
        ];

        push @data, [
            LINE,
            $x + $x_difference,
            $y - $tick_width, 
            $x + $x_difference,
            $y,
            'black'
        ];

        # 
        # Left arrow
        # 
        push @data, [
            LINE,
            $x - $x_difference,
            $y - $tick_width, 
            $x - $x_difference - 3,
            $y - 6,
            'black'
        ];

        push @data, [
            LINE,
            $x - $x_difference - 3,
            $y - 6,
            $x - $x_difference,
            $y - 6,
            'black'
        ];

        push @data, [
            FILL,
            $x - $x_difference - 2,
            $y - 7,
            'black'
        ];

        #
        # Right arrow
        #
        push @data, [
            LINE,
            $x + $x_difference,
            $y - $tick_width, 
            $x + $x_difference + 3,
            $y - 6,
            'black'
        ];

        push @data, [
            LINE,
            $x + $x_difference + 3,
            $y - 6,
            $x + $x_difference,
            $y - 6,
            'black'
        ];

        push @data, [
            FILL,
            $x + $x_difference + 2,
            $y - 7,
            'black'
        ];

        my @bounding_coords = (
            $x - $x_difference,
            $y - $tick_width, 
            $x + $x_difference,
            $y, 
        );

        push @data, [ 
            FILLED_RECT, 
            @bounding_coords,
            $map_color,
        ];

        $self->min_y( $y - $tick_width / 4 );

        my $scroll_start = $start - $scroll <  $map_begin 
                           ? $map_begin    : $start - $scroll;            
        my $scroll_end   = $end   - $scroll <= $map_begin 
                           ? $end : $end   - $scroll;

        push @map, {
            coords => [ @bounding_coords ],
            uri    => URIS->{'map_viewer'}."?$scroll_qs;".  
                      "start=$scroll_start;end=$scroll_end;",
            alt    => "Move Back",
        };
    }
    else {
        push @data, [ 
            LINE, 
            $x - $tick_width / 2, 
            $y, 
            $x + $tick_width / 2, 
            $y, 
            'black' 
        ];
    }

    #
    # The vertical lines for the reference map
    #
    my $max_y  = $pixel_length + $y;
    push @data, [ 
        LINE, 
        $x - $tick_width / 4, 
        $y, 
        $x - $tick_width / 4, 
        $max_y, 
        'black' 
    ];

    push @data, [ 
        LINE, 
        $x + $tick_width / 4, 
        $y, 
        $x + $tick_width / 4, 
        $max_y, 
        'black' 
    ];

    push @data, [ 
        FILLED_RECT, 
        $x - $tick_width / 4, 
        $y, 
        $x + $tick_width / 4, 
        $max_y, 
        $map_color,
    ];

    #
    # The bottom "arrow" (scrolling region)
    #
    my $map_end = $self->reference_map_end;
    unless ( $end >= $map_end ) {
        push @data, [
            LINE,
            $x - $x_difference,
            $max_y,
            $x - $x_difference,
            $max_y + $tick_width, 
            'black'
        ];

        push @data, [
            LINE,
            $x + $x_difference,
            $max_y,
            $x + $x_difference,
            $max_y + $tick_width, 
            'black'
        ];

        # 
        # Left arrow
        # 
        push @data, [
            LINE,
            $x - $x_difference,
            $max_y + $tick_width, 
            $x - $x_difference - 3,
            $max_y + 6,
            'black'
        ];

        push @data, [
            LINE,
            $x - $x_difference - 3,
            $max_y + 6,
            $x - $x_difference,
            $max_y + 6,
            'black'
        ];

        push @data, [
            FILL,
            $x - $x_difference - 1,
            $max_y + 7,
            'black'
        ];

        #
        # Right arrow
        #
        push @data, [
            LINE,
            $x + $x_difference,
            $max_y + $tick_width, 
            $x + $x_difference + 3,
            $max_y + 6,
            'black'
        ];

        push @data, [
            LINE,
            $x + $x_difference + 3,
            $max_y + 6,
            $x + $x_difference,
            $max_y + 6,
            'black'
        ];

        push @data, [
            FILL,
            $x + $x_difference + 1,
            $max_y + 7,
            'black'
        ];

        my @bounding_coords = (
            $x - $x_difference,
            $max_y, 
            $x + $x_difference,
            $max_y + $tick_width, 
        );

        push @data, [ 
            FILLED_RECT, 
            @bounding_coords,
            $map_color,
        ];
        $self->max_y( $max_y + $tick_width );

        my $scroll_start  = $start + $scroll > $map_end ?
                            $start : $start + $scroll   ;
        my $scroll_end    = $end   + $scroll > $map_end ?
                            $map_end : $end + $scroll   ;

        #
        # The area bounding the bottom arrow
        #
        push @map, {
            coords => [ @bounding_coords ],
            uri    => URIS->{'map_viewer'}."?$scroll_qs;".
                      "start=$scroll_start;end=$scroll_end;",
            alt    => "Move Forward",
        };
    }
    else {
        push @data, [ 
            LINE, 
            $x - $tick_width/2, 
            $max_y, 
            $x + $tick_width/2, 
            $max_y, 
            'black' 
        ];
    }

    # 
    # The tick marks
    # 
    my $tick_pos_interval = $self->tick_mark_interval( abs( $end - $start ) );
    my $tag_x             = $x + $tick_width;
    my $tick_x1           = $x - $tick_width * .25;
    my $tick_x2           = $x + $tick_width * .75;

    for my $tick_pos ( int $start .. int $end ) {
        next if $tick_pos < $start || $tick_pos > $end;
        next unless $tick_pos % $tick_pos_interval == 0;
        my $relative_distance = $real_length > 0
            ? ( ( $tick_pos - $start ) / $real_length ) : .5
        ;
        my $cm_pos = $cM_length * $relative_distance;
        my $cur_y  = $y + ($cm_pos*$scale);
        my $tag_y  = $cur_y + (($tick_font->width*length($tick_pos))/2); 
        $self->max_x( $tag_x + $tick_font->height );

        push @data, [
            STRING_UP, 
            $tick_font, 
            $tag_x, 
            $tag_y, 
            $tick_pos, 
            DEFAULT->{'interval_tick_color'},
        ];

        push @data, [
            LINE, 
            $tick_x1,
            $cur_y, 
            $tick_x2,
            $cur_y, 
            DEFAULT->{'interval_tick_color'},
        ];
    }

    $self->max_y( $max_y );

    my $map_min_x  = (); # for remembering how far out tags go

    #
    # Used to figure the recentering regions on the ticks
    #
    $scroll  = ($end - $start)/2 > DEFAULT->{'scroll_interval'}
               ? DEFAULT->{'scroll_interval'} 
               : ($end - $start)/2;
    $scroll /= 2;
    $scroll  = 1 if $scroll < 1;

    #
    # Take note of all the feature ids
    #
    my %all_feature_ids = 
        map { $_->{'feature_id'}, 1 } map { @{ $data->{$_} } } keys %$data;

    #
    # $data is a hash reference.  The keys are the cM 
    # positions of the markers.
    #
    my @keys              = sort { $a <=> $b } keys %$data;
    my $midpoint          = int $#keys / 2;
    my $midpoint_value    = $keys[ $midpoint ];
    my @ordered_keys      = map { $keys[$_] } 
                            reverse( 0..$midpoint ), $midpoint+1..$#keys;
    my %ordered_positions = map { $keys[$_], $_ } 0..$#keys; 
    my $max_keys          = $#ordered_keys;
    my ( $furthest_north, $furthest_south ) = (0, 0);
    my ( $prev_y, $mid_y );
#    my $highlight = $self->highlight;

    $self->min_x( $x - $tick_width/2 );

    for my $i ( 0..$max_keys ) {
        my $tick_pos     = $ordered_keys[ $i ];

        my $relative_distance = 
            $real_length > 0
                ? ($tick_pos-$start)/$real_length 
            : $start == 0
                ? 0 
            : $end == $self->map_end
                ? 1
            : .5;
        my $cm_pos       = $cM_length * $relative_distance;
        my $tick_order   = $ordered_positions{ $tick_pos };
        my $prev_tick    = $tick_order - 1 >= 0 
                           ? $keys[ $tick_order - 1 ] : $start;
        my $next_tick    = $tick_order + 1 <= $#keys
                           ? $keys[ $tick_order + 1 ] : $end;
        my $tick_y       = $y + ( $cm_pos * $scale );

        $self->reference_y_position( $tick_pos, 1, $tick_y );

        my $prev_tick_y  = defined $prev_tick && $real_length
            ? $y+((($prev_tick-$start)/$real_length)*$cM_length)*$scale
            : undef;
        my $next_tick_y  = defined $next_tick && $real_length
            ? $y+ ((($next_tick-$start)/$real_length)*$cM_length)*$scale
            : undef;
        my $cur_y        = $tick_pos > 0
                           ? $tick_y : defined $prev_y ? $prev_y : $y;
        my $scroll_start = $tick_pos - $scroll < 0 
                            ? 0 : $tick_pos - $scroll;
        my $scroll_end   = $tick_pos   + $scroll > $map_end
                            ? $map_end : $tick_pos   + $scroll;
        $prev_y          = $cur_y unless defined $prev_y;
        $prev_y          = $cur_y if $tick_pos == $midpoint_value;

        #
        # Place a tick mark at the location and hyperlink
        # the location so the user can recenter the map 
        # on the location.
        #
        push @data, [
            LINE, 
            $x - $tick_width/2, 
            $tick_y,
            $x + $tick_width/2, 
            $tick_y,
            'black' 
        ];

        my $y1 = defined $prev_tick_y
            ? $tick_y - ( ( $tick_y - $prev_tick_y ) / 2 )
            : $tick_y;
        my $y2 = defined $next_tick_y
            ? $tick_y + ( ( $next_tick_y - $tick_y ) / 2 )
            : $tick_y;

        push @map, {
            coords => [ 
                $x - $tick_width/2, 
                $y1,
                $x + $tick_width/2, 
                $y2
            ],
            uri    => URIS->{'map_viewer'}."?$scroll_qs;".
                      "start=$scroll_start;end=$scroll_end;",
            alt    => "Center on $tick_pos",
        };

        #
        # Each cM position can have several markers.
        # Sort them by their names (which should be unique).
        #
        for my $feature ( 
            sort { $b->{'is_related'} <=> $a->{'is_related'} }
            @{ $data->{$tick_pos} } 
        ) {
            #
            # Draw the centromere and move on.
            #
            if ( uc $feature->{'feature_type'} eq CENTROMERE ) {
                my $fill_color = DEFAULT->{'centromere_color'};
                my $box_color  = DEFAULT->{'centromere_box_color'};
                my $stop       = ( $feature->{'position_stop'} - $tick_pos )
                                 * $scale;

                my @coords = (
                    $x - 8,
                    $tick_y,
                    $x + 8,
                    $tick_y + $stop,
                );

                push @data, [
                    FILLED_RECT,
                    @coords,
                    $fill_color
                ];

                push @data, [
                    RECTANGLE,
                    @coords,
                    $box_color
                ];

#                my $feature_name = $feature->{'feature_name'};
#                push @data, [
#                    STRING_UP,
#                    $tick_font,
#                    $x + 10,
#                    $tick_y + ($stop - $tick_y),# - (length( $feature_name )/2),
#                    $feature_name,
#                    'blue'
#                ];

                next;
            }

            #
            # Skip QTLs (those markers which have "end" defined).
            #
            next if defined $feature->{'position_stop'};

            #
            # We'll also skip any tags when they are too
            # far from their tick mark, so figure out the Y
            # coordinate and skip -- unless the marker has
            # a relationship on another map (genetic or physical).
            #
            my $tag_y = $tick_y;

            if ( $tick_pos <= $midpoint_value ) {
                $tag_y-- while $tag_y >= $prev_y;
            }
            else {
                $tag_y++ while $tag_y <= $prev_y;
                $tag_y++ while $tag_y - $tick_font->height <= $mid_y;
            }

            #
            # Figure out where the tag starts:
            # If map 1, start to the left of X;
            # If map 2, start to the right of X.
            #
            my $start_x           = $x - ( $tick_width * .75 );
            my $name              = $feature->{'map_position_name'} or next;
            my $tag               = $name;
            my $data_label_offset = $self->data_label_offset;
            my $tag_x             = 
                $start_x - $data_label_offset - 
                ( $tick_font->width * length( $tag ) );

            #
            # Remember these for future use
            #
            my @tag_coords = (
                $tag_x - 3, 
                $tag_y - ( $tick_font->height/2 ),
                $tag_x + ( $tick_font->width * length( $tag ) ), 
                $tag_y + ( $tick_font->height/2 ),
            );

            #
            # Remember the position of the tick
            # for linking those with corresponding markers
            #
            my $tag_end         = $tag_x - 3;
            my $link_line_x     = $x + $tick_width*.75;
            my $feature_id      = $feature->{'feature_id'};
            my @correspondences = 
                $self->feature_correspondences( $feature->{'feature_id'} );

            #
            # Use the correspondences to see if this feature is 
            # related to anything else on this same map.
            #
            for my $id ( @correspondences ) {
                next unless exists $all_feature_ids{ $id };
                $feature->{'is_related'} = 1;
            }

            #
            #
            my ( $color, $line_color );
            for my $name ( 
                $feature->{'map_position_name'}, $feature->{'feature_name'}
            ) {
                if ( $self->feature_is_highlighted( $name ) ) {
                    $color = $self->feature_color( 
                        $feature->{'feature_id'}, DEFAULT->{'highlight_color'} 
                    );
                    $line_color = DEFAULT->{'highlight_box_color'};
                    $feature->{'is_highlighted'} = 1;
                    last;
                }
            }

            unless ( $color ) {
                if ( 
                    $feature->{'is_related'} ||
                    $self->feature_is_related( $feature->{'feature_id'} ) 
                ) {
#                    $color = $self->feature_color( $feature->{'feature_id'} );
                    $color = '';
                }
            }

            #
            # Criteria for dropping tags:
            # 1) Never drop out a highlighted marker
            # 2) Drop a highlighted tag that's too far away
            # 3) Drop any non-colored tag that's a very small distance away
            #    (to make room for the others)
            #
            my $distance_from_tag = abs( $tag_y - $tick_y );
            my $skip_tag;
            my $max_pixels = $self->max_pixels_from_tick;
            if ( 
                $feature->{'is_highlighted'} ||
                ( defined $color && $distance_from_tag < $max_pixels )
            ) {
                push @data, [
                    FILLED_RECT,
                    @tag_coords,
                    $color
                ] if $color;

                push @data, [
                    RECTANGLE,
                    @tag_coords,
                    $line_color
                ] if $line_color;

                $skip_tag = 0;
            } 
            elsif ( !$color && $distance_from_tag < $max_pixels / 3 ) { 
                $skip_tag = 0;
            }
            else { 
                $skip_tag = 1;
            }

            $self->feature_position(
                feature_id      => $_,
                map_no          => 1, 
                position        => {
                    tag_coords      => $skip_tag ? [] : [ @tag_coords      ],
                    tag_end         => $skip_tag ? [] : [ $tag_end, $tag_y ],
                    link_line       => [ $link_line_x, $tick_y ],
                }
            ) for $feature_id, @correspondences;

            next if $skip_tag;

            $mid_y = $tag_y if $tick_pos == $midpoint_value && !defined $mid_y;

            #
            # If a marker is related to another marker
            # on the same map, then we draw the relationship
            # on the outside.  To do that, we first need to take
            # note of the furthest X coordinate of the labels 
            # for each map.
            #
            $map_min_x = $tag_end unless defined $map_min_x;
            $map_min_x = $tag_end if $tag_end < $map_min_x;

            #
            # The marker tag
            #
            push @data, [
                STRING, 
                $tick_font, 
                $tag_x,
                $tag_y - ( $tick_font->height/2 ), 
                $tag, 
                'blue'
            ];

            #
            # The map coords
            #
            push @map, {
                coords => [ @tag_coords ],
                uri    => $marker_uri.$feature->{'feature_link_out'},
                alt    => "View: $name ($tick_pos)",
            };

            #
            # This is where the line connecting
            # the tick mark to the tag will end
            #
            my $dest_x = $x - $data_label_offset - 5;

            #
            # The line linking the tag and the tick mark
            #
            push @data, $self->_indicator_line(
                $start_x, 
                $tick_y, 
                $dest_x, 
                $tag_y, 
                $line_color || $color || 'gray'
            );

            #
            # Remember the last y position of a feature tag
            #
            $prev_y = $tick_pos <= $midpoint_value
                ? $tag_y - $tick_font->height
                : $tag_y + $tick_font->height;

            $furthest_north = $prev_y if $prev_y < $furthest_north;
            $furthest_south = $prev_y if $prev_y > $furthest_south;
        }

        $self->max_y( $furthest_south );
        $self->min_y( $furthest_north );
    }

    #
    # Same-map relationships
    #
    my $column_width            = 4;
    my $min_x                   = $map_min_x - $column_width;
    my @columns                 = ();
    my %done                    = ();
    my %col_index_by_feature_id = ();

    for my $feature_id ( keys %all_feature_ids ) {
        my @position1 = $self->feature_position( feature_id => $feature_id )
            or next;

        my @correspondences = 
            grep { defined $all_feature_ids{ $_ } }
            $self->feature_correspondences( $feature_id )
        or next;

        for my $position1 ( @position1 ) {
#            warn "position 1 = ", Dumper($position1), "\n";
            my @tag_end1 = @{ $position1->{'tag_end'} } or next;
            my $start_x  = $tag_end1[0] - 1;
            my $start_y  = $tag_end1[1];
#            warn "pos1, ", #tag_end = ", Dumper(@tag_end1), 
#                "startx = $start_x, start_y = $start_y\n";

            for my $id ( @correspondences ) {
                for my $position2 ( 
                    $self->feature_position( feature_id => $id ) 
                ) {
#                    next if $done{ $feature_id }{ $id };
                    my @tag_end2       = @{ $position2->{'tag_end'} } or next;
                    my $end_x          = $tag_end2[0] - 1;
                    my $end_y          = $tag_end2[1];
                    next if $done{ $start_y }{ $end_y };
                    next if $start_x == $end_x && $start_y == $end_y;
#                    warn "feature_id = $feature_id has two positions\n";
#                    warn "corr = ", Dumper( $self->feature_correspondences(
#                        $feature_id ) ), "\n";
#                    , pos2, endx = $end_x, end_y = $end_y\n";
#                    my $color          = $self->feature_color( $feature_id );
                    my $color          = 'gray';
                    my $furthest_south = $start_y > $end_y ? $start_y : $end_y;
                    my $furthest_north = $start_y < $end_y ? $start_y : $end_y;

                    #
                    # Figure out which "column" to put the vertical line in.
                    #
                    my ( $cur_x, $furthest_north_col, $furthest_south_col );
                    my $column_index;
                    for ( $feature_id, @correspondences ) {
                        $column_index = $col_index_by_feature_id{ $_ };
                        last if $column_index;
                    }

                    if ( defined $column_index ) {
                        my $column          = $columns[ $column_index ];
                        $cur_x              = $column->{'x'}     || $min_x;
                        $furthest_north_col = $column->{'min_y'} ||  undef;
                        $furthest_south_col = $column->{'max_y'} ||  undef;
                    }
                    else {
                        for my $i ( 0..$#columns ) {
                            my $column          = $columns[ $i ];
                            my $col_x           = $column->{'x'}     || $min_x;
                            $furthest_north_col = $column->{'min_y'} ||  undef;
                            $furthest_south_col = $column->{'max_y'} ||  undef;

                            #
                            # First, see if there's anything in this column.
                            # If not, just use it.
                            #
                            unless ( defined $furthest_north_col && 
                                     defined $furthest_south_col
                            ) {
                                $cur_x        = $col_x;
                                $column_index = $i;
                                last;
                            }

                            #
                            # Now see if our line will fit above.
                            # If not, see if our line will fit below.
                            #
                            if ( $furthest_north_col >= $furthest_south or
                                 $furthest_south_col <= $furthest_north ) {
                                $cur_x        = $col_x;
                                $column_index = $i;
                                last;
                            }

                            $column_index = $i;
                        }

                        #
                        # If nothing suitable was found, go to the next column.
                        #
                        unless ( $cur_x ) {
                            $column_index = @columns ? $#columns + 1 : 0;
                            $cur_x = $min_x - ( $column_width * $column_index );
                        }
                    }

                    push @data, [ 
                        LINE, 
                        $start_x, 
                        $start_y, 
                        $cur_x,
                        $start_y, 
                        $color 
                    ];

                    push @data, [ 
                        LINE, 
                        $cur_x,
                        $start_y, 
                        $cur_x,
                        $end_y, 
                        $color 
                    ];

                    push @data, [ 
                        LINE, 
                        $cur_x,
                        $end_y, 
                        $end_x, 
                        $end_y, 
                        $color 
                    ];

                    #
                    # Figure out the furthest point north and
                    # south that we just drew.
                    #
                    if ( defined $furthest_north_col ) {
                        $furthest_north_col = $furthest_north if
                            $furthest_north < $furthest_north_col;
                    }
                    else {
                        $furthest_north_col = $furthest_north;
                    }

                    if ( defined $furthest_south_col ) {
                        $furthest_south_col = $furthest_south if
                            $furthest_south > $furthest_south_col;
                    }
                    else {
                        $furthest_south_col = $furthest_south;
                    }

                    #
                    # Update the @columns array.
                    #
                    $columns[ $column_index ] = {
                        x     => $cur_x,
                        min_y => $furthest_north_col,
                        max_y => $furthest_south_col,
                    };

                    $done{ $start_y }{ $end_y }++;
                    $done{ $end_y }{ $start_y }++;
                    for ( $feature_id, @correspondences ) {
                        next if defined $col_index_by_feature_id{ $_ };
                        $col_index_by_feature_id{ $_ } = $column_index;
                    }

                    $map_min_x = $cur_x if $cur_x < $map_min_x;
                }
            }
        }
    }

    $self->min_x( $map_min_x );

    return { data => \@data, map => \@map };
}

#-----------------------------------------------------
sub reference_map_begin {
    my $self     = shift;
    return $self->{'data'}{ 1 }{'map_begin'};
}

#-----------------------------------------------------
sub reference_map_end {
    my $self     = shift;
    return $self->{'data'}{ 1 }{'map_end'};
}

#-----------------------------------------------------
sub max_pixels_from_tick {
    my $self     = shift;
    return MAX_PIXELS_FROM_TICK;
}

#-----------------------------------------------------
sub reference_y_position {
    my $self     = shift;
    my $position = shift or return;
    my $map_no   = shift || 1;
    if ( my $y   = shift ) {
        $self->{'reference_y_position'}{$map_no}{$position} = $y;
    }

    return defined $self->{'reference_y_position'}{$map_no}{$position}
        ? $self->{'reference_y_position'}{$map_no}{$position} 
        : undef
    ;
}

#-----------------------------------------------------
sub soap {
    my $self = shift;
    unless ( $self->{'soap'} ) {
        $self->{'soap'} = SOAP::Lite -> uri( SOAP_URI ) -> proxy( SOAP_PROXY );
    }
    return $self->{'soap'};
}

#-----------------------------------------------------
sub soap_call {
    my ( $self, %args ) = @_;
    my $method = $args{'method'};
    my $params = $args{'params'};

    if ( DATA_SOURCE_LOCATION eq 'local' ) {
        my $soap_lib = CSHL::SOAP::ComparativeMapData->new(apr=>$self->{'apr'});
        return $soap_lib->$method( %$params );
    }
    else {
        my $soap   = $self->soap;
        my $result = $soap->$method( %$params );
        if ( $result->fault() ) {
            return $self->err_out(
                join ', ', $result->faultcode, $result->faultstring
            );
        }
        else {
            return $result->result();
        }
    }
}

#-----------------------------------------------------
sub scale {
    my $self  = shift;
    $self->{'scale'} = shift if @_;
    return $self->{'scale'};
}

#-----------------------------------------------------
sub species {
    my $self   = shift;
    my $map_no = shift || 1;
    return $self->{'data'}{$map_no}{'species'};
}

#-----------------------------------------------------
sub start {
    my $self   = shift;
    return $self->err_out( 'start() not defined' );
}

#-----------------------------------------------------
sub label_font {
    my $self = shift;
    $self->set_font( 'label_font', shift ) if @_;
    return $self->{'label_font'} || DEFAULT->{'label_font'};
}

#-----------------------------------------------------
sub tick_font {
    my $self = shift;
    $self->set_font( 'tick_font', shift ) if @_;
    return $self->{'tick_font'} || DEFAULT->{'tick_font'};
}

#-----------------------------------------------------
sub set_font {
    my $self       = shift;
    my $font_field = shift; # e.g., 'tick_font' or 'label_font'
    
    if ( my $font = shift ) {
        if ( $font_field eq 'tick_font' ) {
            my %valid_fonts = (
                small  => gdTinyFont,
                medium => gdSmallFont,
                large  => gdLargeFont,
            );
            $font = 'small' unless exists $valid_fonts{ $font };
            $self->{$font_field} = $valid_fonts{ $font };
        }
        else {
            my %valid_fonts = (
                small  => gdMediumBoldFont,
                medium => gdLargeFont,
                large  => gdGiantFont,
            );
            $font = 'small' unless exists $valid_fonts{ $font };
            $self->{$font_field} = $valid_fonts{ $font };
        }
    }

    return 1;
}

#-----------------------------------------------------
sub tick_width {
    my $self = shift;
    $self->{'tick_width'} = shift if @_;
    return $self->{'tick_width'} || DEFAULT->{'tick_width'};
}

#-----------------------------------------------------
sub tick_mark_interval {
    my ( $self, $distance ) = @_;
    return $distance <= 5 ? 1
        : $distance <= 20 ? 5
            : $distance <= 50 ? 10
                : $distance <= 75 ? 25
                    : 50
    ;
}

#-----------------------------------------------------
sub data {
    my $self = shift;
    return $self->err_out( 'data() not defined' );
}

#sub draw_same_map_relatioships {
#    my ( $self, %args ) = @_;
#
#    #
#    # Same-map relationships
#    #
#    my $column_width            = 4;
#    my $min_x                   = $map_min_x - $column_width;
#    my @columns                 = ();
#    my %done                    = ();
#    my %col_index_by_feature_id = ();
#
##    warn "feature positions =\n", 
##        Dumper($self->{'position'}{ $map_no }), "\n";
##    warn "all feature ids =\n", Dumper( %all_feature_ids), "\n";
##    warn "correspondences =\n", 
##        Dumper($self->{'data'}{'feature_correspondences'}), "\n";
#
#    for my $feature_id ( keys %all_feature_ids ) {
#        my @position1 = $self->feature_position( feature_id => $feature_id )
#            or next;
#
#        my @correspondences = 
#            grep { defined $all_feature_ids{ $_ } }
#            $self->feature_correspondences( $feature_id )
#        or next;
##        warn "feature_id = $feature_id, correspondences = ", 
##            Dumper( @correspondences ), "\n";
#
##        warn "feature $feature_id has ",
##            scalar @correspondences, " correspondences (",
##            join(', ', @correspondences), ")\n";
#
#        for my $position1 ( @position1 ) {
##            warn "position 1 = ", Dumper($position1), "\n";
#            my @tag_end1 = @{ $position1->{'tag_end'} } or next;
#            my $start_x  = $tag_end1[0] - 1;
#            my $start_y  = $tag_end1[1];
##            warn "pos1, ", #tag_end = ", Dumper(@tag_end1), 
##                "startx = $start_x, start_y = $start_y\n";
#
#            for my $id ( @correspondences ) {
#                for my $position2 ( 
#                    $self->feature_position( feature_id => $id ) 
#                ) {
##                    next if $done{ $feature_id }{ $id };
#                    my @tag_end2       = @{ $position2->{'tag_end'} } or next;
#                    my $end_x          = $tag_end2[0] - 1;
#                    my $end_y          = $tag_end2[1];
#                    next if $done{ $start_y }{ $end_y };
#                    next if $start_x == $end_x && $start_y == $end_y;
##                    warn "feature_id = $feature_id has two positions\n";
##                    warn "corr = ", Dumper( $self->feature_correspondences(
##                        $feature_id ) ), "\n";
##                    , pos2, endx = $end_x, end_y = $end_y\n";
#                    my $color          = $self->feature_color( $feature_id );
#                    my $furthest_south = $start_y > $end_y ? $start_y : $end_y;
#                    my $furthest_north = $start_y < $end_y ? $start_y : $end_y;
#
#                    #
#                    # Figure out which "column" to put the vertical line in.
#                    #
#                    my ( $cur_x, $furthest_north_col, $furthest_south_col );
#                    my $column_index;
#                    for ( $feature_id, @correspondences ) {
#                        $column_index = $col_index_by_feature_id{ $_ };
#                        last if $column_index;
#                    }
#
#                    if ( defined $column_index ) {
#                        my $column          = $columns[ $column_index ];
#                        $cur_x              = $column->{'x'}     || $min_x;
#                        $furthest_north_col = $column->{'min_y'} ||  undef;
#                        $furthest_south_col = $column->{'max_y'} ||  undef;
#                    }
#                    else {
#                        for my $i ( 0..$#columns ) {
#                            my $column          = $columns[ $i ];
#                            my $col_x           = $column->{'x'}     || $min_x;
#                            $furthest_north_col = $column->{'min_y'} ||  undef;
#                            $furthest_south_col = $column->{'max_y'} ||  undef;
#
#                            #
#                            # First, see if there's anything in this column.
#                            # If not, just use it.
#                            #
#                            unless ( defined $furthest_north_col && 
#                                     defined $furthest_south_col
#                            ) {
#                                $cur_x        = $col_x;
#                                $column_index = $i;
#                                last;
#                            }
#
#                            #
#                            # Now see if our line will fit above.
#                            # If not, see if our line will fit below.
#                            #
#                            if ( $furthest_north_col >= $furthest_south or
#                                 $furthest_south_col <= $furthest_north ) {
#                                $cur_x        = $col_x;
#                                $column_index = $i;
#                                last;
#                            }
#
#                            $column_index = $i;
#                        }
#
#                        #
#                        # If nothing suitable was found, go to the next column.
#                        #
#                        unless ( $cur_x ) {
#                            $column_index = @columns ? $#columns + 1 : 0;
#                            $cur_x = $min_x - ( $column_width * $column_index );
#                        }
#                    }
#
#                    push @data, [ 
#                        LINE, 
#                        $start_x, 
#                        $start_y, 
#                        $cur_x,
#                        $start_y, 
#                        $color 
#                    ];
#
#                    push @data, [ 
#                        LINE, 
#                        $cur_x,
#                        $start_y, 
#                        $cur_x,
#                        $end_y, 
#                        $color 
#                    ];
#
#                    push @data, [ 
#                        LINE, 
#                        $cur_x,
#                        $end_y, 
#                        $end_x, 
#                        $end_y, 
#                        $color 
#                    ];
#
#                    #
#                    # Figure out the furthest point north and
#                    # south that we just drew.
#                    #
#                    if ( defined $furthest_north_col ) {
#                        $furthest_north_col = $furthest_north if
#                            $furthest_north < $furthest_north_col;
#                    }
#                    else {
#                        $furthest_north_col = $furthest_north;
#                    }
#
#                    if ( defined $furthest_south_col ) {
#                        $furthest_south_col = $furthest_south if
#                            $furthest_south > $furthest_south_col;
#                    }
#                    else {
#                        $furthest_south_col = $furthest_south;
#                    }
#
#                    #
#                    # Update the @columns array.
#                    #
#                    $columns[ $column_index ] = {
#                        x     => $cur_x,
#                        min_y => $furthest_north_col,
#                        max_y => $furthest_south_col,
#                    };
#
#                    $done{ $start_y }{ $end_y }++;
#                    $done{ $end_y }{ $start_y }++;
#                    for ( $feature_id, @correspondences ) {
#                        next if defined $col_index_by_feature_id{ $_ };
#                        $col_index_by_feature_id{ $_ } = $column_index;
#                    }
#
#                    $map_min_x = $cur_x if $cur_x < $map_min_x;
#                }
#            }
#        }
#    }
#
#    $self->min_x( $map_min_x );
#}

1;

=pod

=head1 NAME

CSHL::ComparativeMaps::Drawer - base object for drawing maps

=head1 SYNOPSIS

  use CSHL::ComparativeMaps::Drawer;
  use base( qw[ CSHL::ComparativeMaps::Drawer ]);

=head1 DESCRIPTION

At this time, there are two basic types of maps drawn:

  1) Single genetic map to genetic map study or map
     or 
     Single genetic map to physical map study

  2) Single genetic map to individual physical contig

These two instances are handled by the subclasses 
CSHL::ComparativeMaps::Drawer::Map and
CSHL::ComparativeMaps::Drawer::Contig, respectively.  This module
simply serves as a base class to hold those pieces common to both
implementations.

=head1 AUTHOR

Ken Y. Clark, kclark@logsoft.com

=head1 SEE ALSO

perl(1).

=cut

#-----------------------------------------------------
# All wholsome food is caught without a net or a trap.
# William Blake
#-----------------------------------------------------
