package CSHL::ComparativeMaps::Constants;

#-----------------------------------------------------
# $Id: Constants.pm,v 1.16 2002/04/18 16:26:48 kclark Exp $
#
# File       : Constants.pm
# Programmer : Ken Y. Clark, kclark@logsoft.com
# Created    : 2001/08/08
# Purpose    : constants module
#-----------------------------------------------------

use strict;
use GD;
use Data::Dumper;
use base qw( Exporter );
use vars qw( @EXPORT  );
require Exporter;

@EXPORT = qw[ 
    ARC
    BIN_DIR
    CACHE_DIRECTORY
    CENTROMERE
    CLONE
    COLORS
    CONF_FILE
    DASHED_LINE
    DATA_SOURCE_LOCATION
    DB_NAME
    DEFAULT
    DISCORDANT_CONTIG_PIXEL_LENGTH
    ERROR_TEMPLATE
    FILL
    FLOAT_REGEX 
    GENETIC
    get_data
    HL_COLORS
    LINE
    LINK_OUT_FIELDS 
    FILLED_RECT
    MAP_COLORS
    MARKER
    MIN_MAP_CM_LENGTH
    MIN_MAP_PIXEL_LENGTH
    MAX_CM_DISTANCE_DISCORDANT
    MAX_PIXELS_FROM_TICK
    NO_CONTIG_REF_POSITIONS
    next_number
    OFFSET
    PREFERENCE_FIELDS
    PHYSICAL
    PROTEIN
    RECTANGLE
    SOAP_URI
    SOAP_PROXY
    STRING
    STRING_UP
    TEMPLATE_DIR
    URIS
];

#
# This group represents strings used for the GD package for drawing.
# I'd rather use constants in order to get compile-time spell-checking
# rather than using plain strings (even though that would be somewhat
# faster).  These strings correspond to the methods of the GD package.
# Don't change these!
#
use constant ARC         => 'arc';
use constant DASHED_LINE => 'dashedLine';
use constant LINE        => 'line';
use constant FILLED_RECT => 'filledRectangle';
use constant FILL        => 'fill';
use constant RECTANGLE   => 'rectangle';
use constant STRING      => 'string';
use constant STRING_UP   => 'stringUp';

#
# The location of extra utility scripts.
# Default: /usr/local/apache/etc/bin
#
use constant BIN_DIR => '/usr/local/apache/etc/bin';

#
# The directory to store the images.  Note that this directory must
# actually exist and be writable by the httpd user process.  I would
# suggest making the directory permissions 700, owned by the
# user/group of the httpd user.  I would also suggest you purge this
# directory for old images so you don't fill up your disk.  Here's a
# simple cron job you can put in your root's crontab:
#
# 0 0 * * *  find /tmp/comparative_map_cache/ -type f -mtime +1 \
# -exec rm -rf {} \;
#
# Default: /tmp/comparative_map_cache
#
use constant CACHE_DIRECTORY => '/tmp/comparative_map_cache'; 

#
# My palette of colors available for drawing maps
#
use constant COLORS      => {
    white                => ['FF','FF','FF'],
    black                => ['00','00','00'],
    aliceblue            => ['F0','F8','FF'],
    antiquewhite         => ['FA','EB','D7'],
    aqua                 => ['00','FF','FF'],
    aquamarine           => ['7F','FF','D4'],
    azure                => ['F0','FF','FF'],
    beige                => ['F5','F5','DC'],
    bisque               => ['FF','E4','C4'],
    blanchedalmond       => ['FF','EB','CD'],
    blue                 => ['00','00','FF'],
    blueviolet           => ['8A','2B','E2'],
    brown                => ['A5','2A','2A'],
    burlywood            => ['DE','B8','87'],
    cadetblue            => ['5F','9E','A0'],
    chartreuse           => ['7F','FF','00'],
    chocolate            => ['D2','69','1E'],
    coral                => ['FF','7F','50'],
    cornflowerblue       => ['64','95','ED'],
    cornsilk             => ['FF','F8','DC'],
    crimson              => ['DC','14','3C'],
    cyan                 => ['00','FF','FF'],
    darkblue             => ['00','00','8B'],
    darkcyan             => ['00','8B','8B'],
    darkgoldenrod        => ['B8','86','0B'],
    darkgray             => ['A9','A9','A9'],
    darkgreen            => ['00','64','00'],
    darkkhaki            => ['BD','B7','6B'],
    darkmagenta          => ['8B','00','8B'],
    darkolivegreen       => ['55','6B','2F'],
    darkorange           => ['FF','8C','00'],
    darkorchid           => ['99','32','CC'],
    darkred              => ['8B','00','00'],
    darksalmon           => ['E9','96','7A'],
    darkseagreen         => ['8F','BC','8F'],
    darkslateblue        => ['48','3D','8B'],
    darkslategray        => ['2F','4F','4F'],
    darkturquoise        => ['00','CE','D1'],
    darkviolet           => ['94','00','D3'],
    deeppink             => ['FF','14','100'],
    deepskyblue          => ['00','BF','FF'],
    dimgray              => ['69','69','69'],
    dodgerblue           => ['1E','90','FF'],
    firebrick            => ['B2','22','22'],
    floralwhite          => ['FF','FA','F0'],
    forestgreen          => ['22','8B','22'],
    fuchsia              => ['FF','00','FF'],
    gainsboro            => ['DC','DC','DC'],
    ghostwhite           => ['F8','F8','FF'],
    gold                 => ['FF','D7','00'],
    goldenrod            => ['DA','A5','20'],
    gray                 => ['80','80','80'],
    green                => ['00','80','00'],
    greenyellow          => ['AD','FF','2F'],
    honeydew             => ['F0','FF','F0'],
    hotpink              => ['FF','69','B4'],
    indianred            => ['CD','5C','5C'],
    indigo               => ['4B','00','82'],
    ivory                => ['FF','FF','F0'],
    khaki                => ['F0','E6','8C'],
    lavender             => ['E6','E6','FA'],
    lavenderblush        => ['FF','F0','F5'],
    lawngreen            => ['7C','FC','00'],
    lemonchiffon         => ['FF','FA','CD'],
    lightblue            => ['AD','D8','E6'],
    lightcoral           => ['F0','80','80'],
    lightcyan            => ['E0','FF','FF'],
    lightgoldenrodyellow => ['FA','FA','D2'],
    lightgreen           => ['90','EE','90'],
    lightgrey            => ['D3','D3','D3'],
    lightpink            => ['FF','B6','C1'],
    lightsalmon          => ['FF','A0','7A'],
    lightseagreen        => ['20','B2','AA'],
    lightskyblue         => ['87','CE','FA'],
    lightslategray       => ['77','88','99'],
    lightsteelblue       => ['B0','C4','DE'],
    lightyellow          => ['FF','FF','E0'],
    lime                 => ['00','FF','00'],
    limegreen            => ['32','CD','32'],
    linen                => ['FA','F0','E6'],
    magenta              => ['FF','00','FF'],
    maroon               => ['80','00','00'],
    mediumaquamarine     => ['66','CD','AA'],
    mediumblue           => ['00','00','CD'],
    mediumorchid         => ['BA','55','D3'],
    mediumpurple         => ['100','70','DB'],
    mediumseagreen       => ['3C','B3','71'],
    mediumslateblue      => ['7B','68','EE'],
    mediumspringgreen    => ['00','FA','9A'],
    mediumturquoise      => ['48','D1','CC'],
    mediumvioletred      => ['C7','15','85'],
    midnightblue         => ['19','19','70'],
    mintcream            => ['F5','FF','FA'],
    mistyrose            => ['FF','E4','E1'],
    moccasin             => ['FF','E4','B5'],
    navajowhite          => ['FF','DE','AD'],
    navy                 => ['00','00','80'],
    oldlace              => ['FD','F5','E6'],
    olive                => ['80','80','00'],
    olivedrab            => ['6B','8E','23'],
    orange               => ['FF','A5','00'],
    orangered            => ['FF','45','00'],
    orchid               => ['DA','70','D6'],
    palegoldenrod        => ['EE','E8','AA'],
    palegreen            => ['98','FB','98'],
    paleturquoise        => ['AF','EE','EE'],
    palevioletred        => ['DB','70','100'],
    papayawhip           => ['FF','EF','D5'],
    peachpuff            => ['FF','DA','B9'],
    peru                 => ['CD','85','3F'],
    pink                 => ['FF','C0','CB'],
    plum                 => ['DD','A0','DD'],
    powderblue           => ['B0','E0','E6'],
    purple               => ['80','00','80'],
    red                  => ['FF','00','00'],
    rosybrown            => ['BC','8F','8F'],
    royalblue            => ['41','69','E1'],
    saddlebrown          => ['8B','45','13'],
    salmon               => ['FA','80','72'],
    sandybrown           => ['F4','A4','60'],
    seagreen             => ['2E','8B','57'],
    seashell             => ['FF','F5','EE'],
    sienna               => ['A0','52','2D'],
    silver               => ['C0','C0','C0'],
    skyblue              => ['87','CE','EB'],
    slateblue            => ['6A','5A','CD'],
    slategray            => ['70','80','90'],
    snow                 => ['FF','FA','FA'],
    springgreen          => ['00','FF','7F'],
    steelblue            => ['46','82','B4'],
    tan                  => ['D2','B4','8C'],
    teal                 => ['00','80','80'],
    thistle              => ['D8','BF','D8'],
    tomato               => ['FF','63','47'],
    turquoise            => ['40','E0','D0'],
    violet               => ['EE','82','EE'],
    wheat                => ['F5','DE','B3'],
    whitesmoke           => ['F5','F5','F5'],
    yellow               => ['FF','FF','00'],
    yellowgreen          => ['9A','CD','32'],
};

#
# The name of the local config file relative to the server root.
#
use constant CONF_FILE => 'conf/gramene.conf';

#
# Determines whether I use SOAP for fetching the data or try a 
# local database.  The acceptable values are 'local' or 'remote'.
#
use constant DATA_SOURCE_LOCATION => 'local'; 

#
# Used in string comparisons to determine a map's type.  Again, I'd
# rather do this and get compile-time spell-checking than mistype one
# of these.  Don't change these!
#
use constant GENETIC    => 'GENETIC';    # Types of 
use constant PHYSICAL   => 'PHYSICAL';   # maps.
use constant CENTROMERE => 'CENTROMERE'; # -.
use constant MARKER     => 'MARKER';     #  | Features that can 
use constant CLONE      => 'CLONE';      #  | occur on a map.
use constant PROTEIN    => 'PROTEIN';    # -'

#
# Used to determine valid floating point numbers.
#
use constant FLOAT_REGEX => qr[^\-?\d+(?:\.\d+)?$];

#
# While we're talking about where the data is, here are the locations
# for the SOAP server (if applicable):
#
use constant SOAP_URI   => 'http://foo.bar.org/CSHL/SOAP/ComparativeMapData';
use constant SOAP_PROXY => 'http://foo.bar.org/soap';

#
# We'll probably figure out something better to do with this...
#
use constant DB_NAME => 'MAPS';

#
# This is a list of default values to use for various things.
#
use constant DEFAULT  => {
    #
    # The color of the centromere.
    # Default: lightgrey
    #
    centromere_color     => 'lightgrey',
    centromere_box_color => 'gray',

    #
    # The domain of cookies (user prefs)
    # Default: '.gramene.org'
    #
    coookie_domain => '.gramene.org',

    #
    # The color of contigs in a physical map study.
    # Default: lightgrey
    #
    contig_box_color            => 'black',
    discordant_contig_box_color => 'gray',
    discordant_contig_tag_color => 'gray',
    contig_fill_color           => 'lightgrey',

    #
    # Color of contig features.
    # Default: gray.
    #
    discordant_contig_feature_color => 'gray',
    concordant_contig_feature_color => 'black',

    #
    # How far the labels are from the maps.
    # Default: 50
    #
    data_label_offset => 50,

    #
    # The color of highlighted features.
    # Default: yellow
    #
    highlight_color   => 'yellow',

    #
    # The color of the interval ticks (e.g., "5," "10," "15," etc.)
    #
    interval_tick_color => 'gray',
 
    #
    # The color of the box around highlighted features.
    # Default: yellow
    #
    highlight_box_color => 'red',

    #
    # The normal font size
    # Default: small
    #
    font_size => 'small',

    #
    # The way to deliver the image, 'png' or 'jpeg' 
    # (or whatever your compilation of GD offers, perhaps 'gif'?).
    # Default: png
    #
    image_type        => 'png',

    #
    # The height of the map image.  Note that there are options on the
    # template for the user to choose the size of the image they're
    # given.  You should make sure that the number you place here
    # occurs in the choices on the template.  The default values on
    # the template are 400, 600, and 800.
    # Default: 400
    #
    image_height      => 400,

    #
    # The font (from the GD package) to use for the labels.
    # Default: gdMediumBoldFont
    #
    label_font        => gdMediumBoldFont,

    #
    # The color of marker tags, contigs names, etc.
    # Default: blue
    #
    tag_color => 'blue',
    
    #
    # How large of an area (in cM) to show on a map.
    # Default: 20
    #
    scroll_interval   => 20, 

    #
    # The font (from the GD package) to use for the "ticks."
    # I'm sure I could have done a better job naming this, but
    # basically the "ticks" are intervals (10, 20, 30, ...) on the
    # maps showing the general position of the features.
    # Default: gdTinyFont
    #
    tick_font         => gdTinyFont,

    #
    # How wide (in pixels) the ticks on the maps are.
    # Default: 12
    #
    tick_width        => 12,

    #
    # The width of the image.
    # Default: 250
    #
    width             => 250,

    #
    # Name of the user preferences cookie.
    # Default: CMAPS_USER_PREFS
    #
    user_pref_cookie_name => 'CMAPS_USER_PREFS',
};

#
# The length in pixels of discordant contigs.
# Default: 20
#
use constant DISCORDANT_CONTIG_PIXEL_LENGTH => 15;

#
# The template to use when displaying caught errors.
# Default: 'error.tmpl'
#
use constant ERROR_TEMPLATE => 'error.tmpl';

#
# These are the colors that will be cycled through when highlighting
# things on the maps, such as like features from map to map (or on the
# same map).  The longer the list, the less likely colors will be
# repeated, but it can also be hard to find colors that go well with
# each other.  Obviously, you should pick from the COLORS list.
#
use constant HL_COLORS => [ qw(
    thistle
    lightgreen
    violet
    lightpink
    aqua
    lightskyblue
    pink
    lavender
    greenyellow
    plum
    wheat
    yellowgreen
) ];

#
# Like the HL_COLORS, this is a series of colors that will be cycled
# through to color the comparative maps.  Make sure the colors occur
# in the COLORS array.
#
use constant MAP_COLORS => [ qw(
    lightblue
    lightgreen
    lightcoral    
    lightgoldenrodyellow
    lightslategray
    lightgreen    
    lightseagreen 
    lightcyan     
    lightskyblue  
    lightsteelblue
    lightyellow   
) ];

#
# The maximum distance in centimorgans that a feature can occur on a
# genetic map before it's consider discordant.  That is, say two
# features found on a contig occur on a gentic map and they're within
# 10 cM of each other on that map, then they'd be consider concordant.
# But if a third feature on the contig was found more than 10 cM from
# those first two, then it would be discordant.
# Default: 10
#
use constant MAX_CM_DISTANCE_DISCORDANT => 10;

#
# The maximum number of pixels that a feature's tag can be place from
# it's tick mark on the map.  This keeps too many features from being
# drawn.
# Default: 25
#
use constant MAX_PIXELS_FROM_TICK => 35;


#
# When viewing a contig in detail, this is the number of positions on the
# reference map to show before and after those anchored to the contig.
# Default: 10
#
use constant NO_CONTIG_REF_POSITIONS => 10;

#
# The minimum centimorgan length of a genetic map.
# Default: 3
#
use constant MIN_MAP_CM_LENGTH => 3;

#
# The minumum length in pixels of a map.
# Default: 100
#
use constant MIN_MAP_PIXEL_LENGTH => 15;

#
# A general pixel distance for separating elements in the map.
# Default: 10
#
use constant OFFSET => 10;

use constant PREFERENCE_FIELDS => [ qw(
    highlight
    image_height
    font_size
    image_type
) ];

#
# The absolute path of the directory holding the templates.
# At run time, this can be overridden by an Apache dir_config, e.g.:
# PerlSetVar TEMPLATE_DIR '/web/templates/'
#
use constant TEMPLATE_DIR => '/usr/local/apache/templates/comparative_maps';

#
# These URL's are the locations for seeing more about features found
# on the maps.  They can either be relative links (e.g., for Gramene,
# a link to our own marker AceDB database) or absolute links (e.g., to
# MaizeDB's marker viewer).  These are used below in the URIS section.
#
use constant MARKER_URL_RICE  => '/gramene/map/marker?class=Marker&name=';

use constant MARKER_URL_WHEAT => 
    'http://ars-genome.cornell.edu/cgi-bin/WebAce/webace?db=graingenes'.
    '&class=Locus&object=';

use constant MARKER_URL_MAIZE => 
    'http://www.agron.missouri.edu:80/cgi-bin/sybgw_mdb/mdb3?'.
    '!Entity=Locus&!Name=';

use constant MARKER_URL_SORGHUM =>
    'http://sorghumgenome.tamu.edu/Mapping/Markers/lg_amarks.html#';

use constant PROTEIN_URL_RICE => '/perl/protein_search?acc=';

#
# Some of these URI's are relative, some are absolute.
#
use constant BASE_URI => '/maps';
use constant URIS     => {
    #
    # The handler to view a single contig beside the reference map
    # (CSHL::ComparativeMaps::ContigViewer).
    #
    contig_viewer     => BASE_URI.'/contig',

    #
    # The entry point to Gramene's genome viewer.  This will not
    # likely be applicable to other sites.  It is used as the link
    # from a sequenced clone on a physical map (contig).
    #
    genome_viewer     => '/perl/contigview?clone=',

    #
    # The handler to view the comparative maps
    # (CSHL::ComparativeMaps::MapViewer).
    #
    map_viewer        => BASE_URI.'/viewer',

    #
    # The handler to view the map study info 
    # (CSHL::ComparativeMaps::MapStudyInfo).
    #
    map_study_info    => BASE_URI.'/map_studies',

    #
    # The place to view more info on a marker, separated by species.
    #
    marker_viewer     => {
        barley        => MARKER_URL_WHEAT,
        default       => MARKER_URL_RICE,
        maize         => MARKER_URL_MAIZE,
        oat           => MARKER_URL_WHEAT,
        rice          => MARKER_URL_RICE,
        sorghum       => MARKER_URL_SORGHUM,
        wheat         => MARKER_URL_WHEAT,
    },

    #
    # The place to view more info on a protein, separated by species.
    #
    protein_viewer    => {
        barley        => '',
        default       => '',
        maize         => '',
        oat           => '',
        rice          => PROTEIN_URL_RICE,
        sorghum       => '',
        wheat         => '',
    },
};

#
# This is the field used in the query string in the above URL's to
# look up marker information.  Different species databases expect
# different things, usu. locus name or feature name.  Sorghum expects
# linkage group, however, and so this is actually kind of broken right
# now.
#
use constant LINK_OUT_FIELDS => {
    MARKER      => {
        barley  => 'map_position_name',
        default => 'feature_name',
        maize   => 'feature_name',
        oat     => 'feature_name',
        rice    => 'feature_name',
        sorghum => 'linkage_group',
        wheat   => 'map_position_name',
    },
    PROTEIN     => {
        barley  => 'map_position_name',
        default => 'feature_name',
        maize   => 'feature_name',
        oat     => 'feature_name',
        rice    => 'feature_name',
        sorghum => 'linkage_group',
        wheat   => 'map_position_name',
    },
};

#
# A stupid generic routine for getting data.  It's just a central
# place to determine whether to make a SOAP call or not.  This will
# likely change in the future.
#
sub get_data {
    my %args   = @_;
    my $method = $args{'method'};
    my $params = $args{'params'};

    if ( DATA_SOURCE_LOCATION eq 'local' ) {
        my $soap_lib = CSHL::SOAP::ComparativeMapData->new(apr=>$args{'apr'});
        return $soap_lib->$method( %$params );
    }
    else {
        my $soap   = SOAP::Lite -> uri( SOAP_URI ) -> proxy( SOAP_PROXY );
        my $result = $soap->$method( %$params );
        return $result->fault()
            ? ( undef, (join ', ', $result->faultcode, $result->faultstring) ) 
            : $result->result();
    }
}

#
# A generic routine for retrieving (and possibly setting) the next
# number for an ID field in a table.  Given a table "foo," the
# expected ID field would be "foo_id," but this isn't always the case.
# Therefore, "id_field" tells us what field to look at.  Basically, we
# look to see if there's an entry in the "next_number" table.  If not
# we do a MAX on the ID field given (or ascertained).  Either way, the
# "next_number" table gets told what the next number will be (on the
# next call), and we pass back what is the next number this time.
#
# So why not just use "auto_increment" (MySQL) or a "sequence"
# (Oracle)?  Just to make sure that this stays completely portable.
# By coding all this in Perl, I know that it will work with any
# database (that supports ANSI-SQL, that is).
#
sub next_number {
    my %args       = @_;
    my $db         = $args{'db'}         or return;
    my $table_name = $args{'table_name'} or return;
    my $id_field   = $args{'id_field'}   || $table_name.'_id';

    my $next_number = $db->selectrow_array(
        q[
            select next_number
            from   cmap_next_number
            where  table_name=?
        ],
        {}, ( $table_name )
    );

    unless ( $next_number ) {
        $next_number = $db->selectrow_array(
            qq[
                select max( $id_field )
                from   $table_name
            ]
        ) || 0;
        $next_number++;

        $db->do(
            q[
                insert
                into   cmap_next_number ( table_name, next_number )
                values ( ?, ? )
            ],
            {}, ( $table_name, $next_number+1 )
        );
    }
    else {
        $db->do(
            q[
                update cmap_next_number
                set    next_number=?
                where  table_name=?
            ],
            {}, ( $next_number + 1, $table_name )
        );
    }

    return $next_number;
}

1;

=pod

=head1 NAME

CSHL::ComparativeMaps::Constants

=head1 SYNOPSIS

  use CSHL::ComparativeMaps::Constants;
  my $image_size = IMAGE_SIZE;

=head1 DESCRIPTION

This module simply exports common constants and subroutines used by
other modules.

=head1 AUTHOR

Ken Y. Clark, kclark@logsoft.com

=head1 SEE ALSO

perl(1).

=cut

#-----------------------------------------------------
# Excess of sorrow laughs. Excess of joy weeps.
# William Blake
#-----------------------------------------------------
